/*
 * Copyright (C) 2009 Chase Douglas
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations
 * including the two.
 * You must obey the GNU General Public License in all respects
 * for all of the code used other than OpenSSL.  If you modify
 * file(s) with this exception, you may extend this exception to your
 * version of the file(s), but you are not obligated to do so.  If you
 * do not wish to do so, delete this exception statement from your
 * version.
 */

#ifndef CONNECTION_H
#define CONNECTION_H

#include <QSslSocket>
#include <QTimer>

class QSslConfiguration;

enum ConnectionType {
    CLIENT,
    SERVER
};

enum ConnectionState {
    BARE,
    ENCRYPTED,
    AUTHENTICATING,
    AUTHENTICATED
};

class Connection : public QObject {
    Q_OBJECT

public:
    Connection(int sock, QSslConfiguration &configuration);
    Connection(QString &hostname, quint16 port);
    ~Connection();
    virtual void startEncryption();
    virtual void startAuthentication();
    void startCommunication();
    QHostAddress peerAddress();
    qint64 bytesAvailable();
    qint64 peek(char *data, qint64 maxSize);
    qint64 read(char *data, qint64 maxSize);
    QByteArray read(qint64 maxSize);
    QByteArray readAll();
    bool write(const QByteArray &message);
    void disconnectLater();

private:
    bool sockWrite(const char *data, qint64 maxSize);
    bool sockWrite(const char *data);
    bool sockWrite(const QByteArray &byteArray);

    QSslSocket socket;
    ConnectionState state;
    bool communicating;
    ConnectionType type;
    quint8 credentialsSize;
    QByteArray buffer;

public slots:
    void checkPassResult(bool ok);
    void disconnect();

protected slots:
    void sockConnected();
    void sockDisconnected();
    void sockReadyRead();
    void sockError();
    void sslErrors(const QList<QSslError> &errors);
    virtual void sockEncrypted();
    void bytesWritten(qint64 size);

signals:
    void connected();
    void encrypted();
    void checkPass(const QByteArray &user, const QByteArray &pass);
    void authenticated();
    void readyRead();
    void disconnectSignal();
    void disconnected();
};

#endif // CONNECTION_H
