
#include "utils.h"

#include <QLabel>
#include <QTest>

class UtilsTest: public QObject {
	Q_OBJECT
private slots:
	void initTestCase();
	void cleanupTestCase();

	void getAppImageInfoTest();
	void getAutostartDirTest();
	void getUserTest();
	void makeHTMLTest();
	void makeLinkTest();
	void makeTitleTest();
private:
	UPath existingDir;
	QString oldXdgConfigHome;
	inline static const char* XCH = "XDG_CONFIG_HOME";
};

// private slots:

void UtilsTest::initTestCase() {
	existingDir = UPath(QDir::tempPath().toStdString()) / "kshutdown-EXISTS";
	std::filesystem::create_directories(existingDir);
	QVERIFY(std::filesystem::is_directory(existingDir));

	oldXdgConfigHome = qEnvironmentVariable(XCH, "");
}

void UtilsTest::cleanupTestCase() {
	if (! oldXdgConfigHome.isEmpty())
		qputenv(XCH, oldXdgConfigHome.toUtf8());

	std::filesystem::remove(existingDir);
	QVERIFY(! std::filesystem::exists(existingDir));
}

void UtilsTest::getAppImageInfoTest() {
	#ifdef Q_OS_LINUX
	QFileInfo info;

	qputenv("APPDIR", "foo.tmp");
	qputenv("APPIMAGE", "");
	info = Utils::getAppImageInfo();
	QVERIFY(! info.isFile());
	QCOMPARE(Utils::getAppDir().path(), QApplication::applicationDirPath());

	qputenv("APPDIR", "");
	qputenv("APPIMAGE", "bar.tmp");
	info = Utils::getAppImageInfo();
	QVERIFY(! info.isFile());
	QCOMPARE(Utils::getAppDir().path(), QApplication::applicationDirPath());

	qputenv("APPDIR", QDir::tempPath().toUtf8());
	qputenv("APPIMAGE", "");
	info = Utils::getAppImageInfo();
	QVERIFY(! info.isFile());
	QCOMPARE(Utils::getAppDir().path(), QApplication::applicationDirPath());

	qputenv("APPDIR", "");
	qputenv("APPIMAGE", QApplication::applicationFilePath().toUtf8());
	info = Utils::getAppImageInfo();
	QVERIFY(! info.isFile());
	QCOMPARE(Utils::getAppDir().path(), QApplication::applicationDirPath());

	qputenv("APPDIR", QApplication::applicationFilePath().toUtf8()); // not directory
	qputenv("APPIMAGE", QApplication::applicationFilePath().toUtf8());
	info = Utils::getAppImageInfo();
	QVERIFY(! info.isFile());
	QCOMPARE(Utils::getAppDir().path(), QApplication::applicationDirPath());

	qputenv("APPDIR", QDir::tempPath().toUtf8());
	qputenv("APPIMAGE", QDir::tempPath().toUtf8()); // not file
	info = Utils::getAppImageInfo();
	QVERIFY(! info.isFile());
	QCOMPARE(Utils::getAppDir().path(), QApplication::applicationDirPath());

	qputenv("APPDIR", QDir::tempPath().toUtf8()); // existing directory
	qputenv("APPIMAGE", QApplication::applicationFilePath().toUtf8()); // existing file
	info = Utils::getAppImageInfo();
	QVERIFY(info.isFile());
// TODO: proper test
	QCOMPARE(Utils::getAppDir(), info.dir());

	qunsetenv("APPDIR");
	qunsetenv("APPIMAGE");
	info = Utils::getAppImageInfo();
	QVERIFY(! info.isFile());
	QCOMPARE(Utils::getAppDir().path(), QApplication::applicationDirPath());
	#else
	QSKIP("Linux only");
	#endif // Q_OS_LINUX
}

void UtilsTest::getAutostartDirTest() {
	UPath defaultDir = UPath(QDir::home().path().toStdString()) / ".config" / "autostart";

	// test no env
	qunsetenv(XCH);
	QCOMPARE(
		Utils::getAutostartDir(false),
		defaultDir
	);

	// test trim
	qputenv(XCH, " /tmp ");
	QCOMPARE(
		Utils::getAutostartDir(false),
		UPath("/tmp") / "autostart"
	);

/* TODO: test mkdir
	qputenv(XCH, " /tmp ");
	QCOMPARE(
		Utils::getAutostartDir(false),
		UPath("/tmp") / "autostart"
	);
*/

	// not existing dir
	qputenv(XCH, "/tmp/kshutdown-NOT-EXISTS");
	QCOMPARE(
		Utils::getAutostartDir(false),
		defaultDir
	);

	// existing dir
	qputenv(XCH, KS_S(QString::fromStdString(existingDir.string())));
	QCOMPARE(
		Utils::getAutostartDir(false),
		existingDir / "autostart"
	);
}

void UtilsTest::getUserTest() {
// TODO: QCOMPARE_NE #qt6
	QVERIFY(! Utils::getUser().isEmpty());
}

void UtilsTest::makeHTMLTest() {
	QCOMPARE(Utils::makeHTML(""), "<html></html>");
	QCOMPARE(Utils::makeHTML("<br />"), "<html><br /></html>"); // test no escape
	QCOMPARE(Utils::makeHTML("foo"), Utils::HTML_START + "foo" + Utils::HTML_END);
}

void UtilsTest::makeLinkTest() {
	QCOMPARE(
		Utils::makeLink(nullptr, "https://example.com", "Foo"),
		"<a href=\"https://example.com\" style=\"color: " + QColorConstants::Svg::royalblue.name() + "; text-decoration: none\">Foo</a>"
	);

	QCOMPARE(
		Utils::makeLink(nullptr, "https://example.com/&", "Foo & Bar"),
		"<a href=\"https://example.com/&amp;\" style=\"color: " + QColorConstants::Svg::royalblue.name() + "; text-decoration: none\">Foo &amp; Bar</a>"
	);

	auto label = QLabel();

	// dark widget

	QPalette p(label.palette());
	p.setColor(QPalette::Window, Qt::black);
	label.setPalette(p);

	QCOMPARE(
		Utils::makeLink(&label, "https://example.com", "Foo"),
		"<a href=\"https://example.com\" style=\"color: " + QColorConstants::Svg::lightskyblue.name() + "; text-decoration: none\">Foo</a>"
	);

	// light widget

	p.setColor(QPalette::Window, Qt::white);
	label.setPalette(p);

	QCOMPARE(
		Utils::makeLink(&label, "https://example.com", "Foo"),
		"<a href=\"https://example.com\" style=\"color: " + QColorConstants::Svg::royalblue.name() + "; text-decoration: none\">Foo</a>"
	);
}

void UtilsTest::makeTitleTest() {
	QCOMPARE(
		Utils::makeTitle("", QString()),
		""
	);

	QCOMPARE(
		Utils::makeTitle("foo", ""),
		"foo"
	);

	QCOMPARE(
		Utils::makeTitle("foo", "foo"),
		"foo"
	);

	QCOMPARE(
		Utils::makeTitle("foo", "bar"),
		"foo" + Utils::TITLE_SEPARATOR + "bar"
	);

	QCOMPARE(
		Utils::makeTitle("", "bar"),
		"bar"
	);
}

QTEST_MAIN(UtilsTest)
#include "utilstest.moc"
