// Licensed to the .NET Foundation under one or more agreements.
// The .NET Foundation licenses this file to you under the MIT license.

/*============================================================
**
** Source: criticalsectionfunctions/test1/initializecriticalsection.c
**
** Purpose: Test Semaphore operation using classic IPC problem:
**          "Producer-Consumer Problem".
**
** Dependencies: CreateThread
**               InitializeCriticalSection
**               EnterCriticalSection
**               LeaveCriticalSection
**               DeleteCriticalSection
**               WaitForSingleObject
**               Sleep 
** 

**
**=========================================================*/

#include <palsuite.h>

#define PRODUCTION_TOTAL 26

#define _BUF_SIZE 10

DWORD dwThreadId_CriticalSectionFunctions_test1;  /* consumer thread identifier */

HANDLE hThread_CriticalSectionFunctions_test1; /* handle to consumer thread */

CRITICAL_SECTION CriticalSectionM_CriticalSectionFunctions_test1; /* Critical Section Object (used as mutex) */

typedef struct Buffer
{
    short readIndex;
    short writeIndex;
    CHAR message[_BUF_SIZE];

} BufferStructure;

CHAR producerItems_CriticalSectionFunctions_test1[PRODUCTION_TOTAL + 1];

CHAR consumerItems_CriticalSectionFunctions_test1[PRODUCTION_TOTAL + 1];

/* 
 * Read next message from the Buffer into provided pointer.
 * Returns:  0 on failure, 1 on success. 
 */
int
readBuf_CriticalSectionFunctions_test1(BufferStructure *Buffer, char *c)
{
    if( Buffer -> writeIndex == Buffer -> readIndex )
    {
	return 0;    
    }
    *c = Buffer -> message[Buffer -> readIndex++];
    Buffer -> readIndex %= _BUF_SIZE;
    return 1;
}

/* 
 * Write message generated by the producer to Buffer. 
 * Returns:  0 on failure, 1 on success.
 */
int 
writeBuf_CriticalSectionFunctions_test1(BufferStructure *Buffer, CHAR c)
{
    if( ( ((Buffer -> writeIndex) + 1) % _BUF_SIZE) ==
	(Buffer -> readIndex) )
    {
	return 0;
    }
    Buffer -> message[Buffer -> writeIndex++] = c;
    Buffer -> writeIndex %= _BUF_SIZE;
    return 1;
}

/* 
 * Sleep 500 milleseconds.
 */
VOID 
consumerSleep_CriticalSectionFunctions_test1(VOID)
{
    Sleep(500);
}

/* 
 * Sleep between 10 milleseconds.
 */
VOID 
producerSleep_CriticalSectionFunctions_test1(VOID)
{
    Sleep(10);
}

/* 
 * Produce a message and write the message to Buffer.
 */
VOID
producer_CriticalSectionFunctions_test1(BufferStructure *Buffer)
{

    int n = 0;
    char c;
    
    while (n < PRODUCTION_TOTAL) 
    {
	c = 'A' + n ;   /* Produce Item */   
	
	EnterCriticalSection(&CriticalSectionM_CriticalSectionFunctions_test1);
	
	if (writeBuf_CriticalSectionFunctions_test1(Buffer, c)) 
	{
            printf("Producer produces %c.\n", c);
	    producerItems_CriticalSectionFunctions_test1[n++] = c;
	}
	
	LeaveCriticalSection(&CriticalSectionM_CriticalSectionFunctions_test1);
	
	producerSleep_CriticalSectionFunctions_test1();
    }

    return;
}

/* 
 * Read and "Consume" the messages in Buffer. 
 */
DWORD
PALAPI 
consumer_CriticalSectionFunctions_test1( LPVOID lpParam )
{
    int n = 0;
    char c; 
    
    consumerSleep_CriticalSectionFunctions_test1();
    
    while (n < PRODUCTION_TOTAL) 
    {
	
	EnterCriticalSection(&CriticalSectionM_CriticalSectionFunctions_test1);
	
	if (readBuf_CriticalSectionFunctions_test1((BufferStructure*)lpParam, &c)) 
	{
	    printf("\tConsumer consumes %c.\n", c);
	    consumerItems_CriticalSectionFunctions_test1[n++] = c;
	}
	
	LeaveCriticalSection(&CriticalSectionM_CriticalSectionFunctions_test1);
	
	consumerSleep_CriticalSectionFunctions_test1();
    }
    
    return 0;
}

PALTEST(threading_CriticalSectionFunctions_test1_paltest_criticalsectionfunctions_test1, "threading/CriticalSectionFunctions/test1/paltest_criticalsectionfunctions_test1")
{

    BufferStructure Buffer, *pBuffer;
    
    pBuffer = &Buffer;
    
    if(0 != (PAL_Initialize(argc, argv)))
    {
	return FAIL;
    }
    
    /*
     * Create mutual exclusion mechanisms
     */
    
    InitializeCriticalSection ( &CriticalSectionM_CriticalSectionFunctions_test1 );
    
    /* 
     * Initialize Buffer
     */
    pBuffer->writeIndex = pBuffer->readIndex = 0;
    
    
    
    /* 
     * Create Consumer
     */
    hThread_CriticalSectionFunctions_test1 = CreateThread(
	NULL,         
	0,            
	consumer_CriticalSectionFunctions_test1,     
	&Buffer,     
	0,           
	&dwThreadId_CriticalSectionFunctions_test1);
    
    if ( NULL == hThread_CriticalSectionFunctions_test1 ) 
    {
	Fail ( "CreateThread() returned NULL.  Failing test.\n"
	       "GetLastError returned %d\n", GetLastError());   
    }
    
    /* 
     * Start producing 
     */
    producer_CriticalSectionFunctions_test1(pBuffer);
    
    /*
     * Wait for consumer to complete
     */
    WaitForSingleObject (hThread_CriticalSectionFunctions_test1, INFINITE);
    
    /* 
     * Compare items produced vs. items consumed
     */
    if ( 0 != strncmp (producerItems_CriticalSectionFunctions_test1, consumerItems_CriticalSectionFunctions_test1, PRODUCTION_TOTAL) )
    {
	Fail("The producerItems_CriticalSectionFunctions_test1 string %s\n and the consumerItems_CriticalSectionFunctions_test1 string "
	     "%s\ndo not match. This could be a problem with the strncmp()"
	     " function\n FailingTest\nGetLastError() returned %d\n", 
	     producerItems_CriticalSectionFunctions_test1, consumerItems_CriticalSectionFunctions_test1, GetLastError());
    }
    
    /* 
     * Clean up Critical Section object 
     */
    DeleteCriticalSection(&CriticalSectionM_CriticalSectionFunctions_test1);
    
    Trace("producerItems_CriticalSectionFunctions_test1 and consumerItems_CriticalSectionFunctions_test1 arrays match.  All %d\nitems "
	  "were produced and consumed in order.\nTest passed.\n",
	  PRODUCTION_TOTAL);
    
    PAL_Terminate();
    return (PASS);

}
