/*
 *	libetm-0.4 / str_mem.c - Copyright (C) Emmanuel Thomas-Maurin 2008-2012
 * 	<manutm007@gmail.com>
 *
 *	- A few strings and memory management functions -
 *
 * 	This program is free software: you can redistribute it and/or modify
 * 	it under the terms of the GNU General Public License as published by
 * 	the Free Software Foundation, either version 3 of the License, or
 * 	(at your option) any later version.
 *
 * 	This program is distributed in the hope that it will be useful,
 * 	but WITHOUT ANY WARRANTY; without even the implied warranty of
 * 	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * 	GNU General Public License for more details.
 *
 * 	You should have received a copy of the GNU General Public License
 * 	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <ctype.h>
#include "libetm.h"

/*
 * Copy n bytes max from src to dest then add '\0' at end of dest.
 */
char *str_n_cpy(char *dest, const char *src, size_t length)
{
	char	*dest_bak = dest;

	if (dest == NULL)
		big_error_in_lib(NULL_DEST, "str_n_cpy()");

	while (length-- > 0 && *src != '\0')
		*dest++ = *src++;

	*dest = '\0';

	return dest_bak;
}

/*
 * Concanate n bytes max of src to dest then add '\0' at end of dest.
 * Strings may not be identical and should not overlap.
 */
char *str_n_cat(char *dest, const char *src, size_t length)
{
	char	*dest_bak = dest;

	if (dest == NULL)
		big_error_in_lib(NULL_DEST, "str_n_cat()");
	else if (src == dest)
		big_error_in_lib(SRC_EQ_DEST, "str_n_cat()");

	while (*dest++ != '\0');

	str_n_cpy(dest - 1, src, length);

	return dest_bak;
}

/*
 * Create new_l_str (allocate memory) and copy str (can be NULL) to new_l_str.
 */
char *l_str_new(const char *str)
{
	char	*new_l_str;
	size_t	str_len;

	if (str == NULL)
		str_len = 0;
	else
		str_len = strlen(str);

	new_l_str = malloc2(sizeof(char) * (str_len + 1));

	return str_n_cpy(new_l_str, str, str_len);
}

/*
 * Append l_str_src (can be NULL) to l_str_dest (re-allocate memory as necessary).
 * l_str_dest must have been created by l_str_new - strings may overlap.
 */
char *l_str_cat(char *l_str_dest, const char *l_str_src)
{
	char	*new_l_str, *l_str_src2;
	size_t	src_len, dest_len;

	if (l_str_dest == NULL)
		big_error_in_lib(NULL_DEST, "l_str_cat()");
	else if (l_str_src == NULL)
		return l_str_dest;

	l_str_src2 = l_str_new(l_str_src);
	src_len = strlen(l_str_src);
	if (src_len == 0)
		return l_str_dest;
	dest_len = strlen(l_str_dest);

	new_l_str = realloc2(l_str_dest, sizeof(char) * (dest_len + src_len + 1));
	new_l_str = str_n_cat(new_l_str, l_str_src2, src_len);
	l_str_free(l_str_src2);

	return new_l_str;
}

/*
 * Free string created by l_str_new() or l_str_cat().
 */
void l_str_free(char *l_str)
{
	if (l_str != NULL)
		free(l_str);
	else
		big_error_in_lib(NULL_POINTER_FREE, "l_str_free()");
	/* CODE WITH NO EFFECT
	l_str = NULL;*/
}

/*
 * Wrappers for malloc(), realloc(), calloc() and free() which check returned value.
 */
void *malloc2(size_t size)
{
	void	*free_block = NULL;

	if (size == 0)
		big_error_in_lib(ZERO_RQ_SIZE, "malloc2()");
	/*else if (size < 0)
		big_error_in_lib(NEG_RQ_SIZE, "malloc2()");*/
	else if ((free_block = malloc(size)) == NULL)
		big_error_in_lib(OUT_OF_MEMORY, "malloc2()");

	return free_block;
}

void *realloc2(void *free_block, size_t size)
{
	void	*tmp = free_block;

	if (size == 0)
		big_error_in_lib(ZERO_RQ_SIZE, "realloc2()");
	/*else if (size < 0)
		big_error_in_lib(NEG_RQ_SIZE, "realloc2()");*/
	else if ((free_block = realloc(free_block, size)) == NULL) {
		free2(tmp);
		big_error_in_lib(OUT_OF_MEMORY, "realloc2()");
	}

	return free_block;
}

void *calloc2(size_t n_elements, size_t element_size)
{
	void	*free_block = NULL;
	size_t	size = n_elements * element_size;

	if (size == 0)
		big_error_in_lib(ZERO_RQ_SIZE, "calloc2()");
	/*else if (size < 0)
		big_error_in_lib(NEG_RQ_SIZE, "calloc2()");*/
	else if ((free_block = malloc2(size)) != NULL)
		memset(free_block, 0, size);

	return free_block;
}

void free2(void *free_block)
{
	if (free_block != NULL)
		free(free_block);
	else
		big_error_in_lib(NULL_POINTER_FREE, "free2()");

	/* CODE WITH NO EFFECT
	free_block = NULL;*/
}

/*
 * Return size in readable format (KiB, MiB, GiB, TiB).
 * *** Allow up to 16 simultaneous calls ***
 * Convention: we assume 2.5 MiB = 2 MiB + [0.5 x 1024 = 512] KiB, not 500 KiB.
 * Otherwise, there is no way to express a value in the range 1000 - 1023,
 * so x.y MiB = x MiB + 0.y MiB, that is: not y x 100 KiB but y x 102.4 KiB
 * (isn't this a bit confusing?)
 */
const char *readable_size(double size_bytes)
{
	/* We use an array of 16 strings to store the returned string in order
	 * to allow 16 simultaneous calls (should be enough in most cases?)
	 * Otherwise, sth like:
	 * printf("size1 = %s / size2 = %s\n", readable_size(size1), readable_size(size2))
	 * would produce unpredictable / false results (like readable sizes are
	 * equal whereas sizes are different).
	 */
	static char	lib_static_str[16][128];
	unsigned long	size_KiB, size_MiB, size_GiB, size_TiB;
	static int	count = -1;

	count++;
	count &= 15;

	size_KiB = (unsigned long)(size_bytes / 1024);
	size_MiB = size_KiB / 1024;
	size_GiB = size_MiB / 1024;
	size_TiB = size_GiB / 1024;

	if (size_TiB >= 1)
		snprintf(lib_static_str[count], 128, "%ld.%ld TiB",
			size_TiB, ((size_GiB - size_TiB * 1024) * 10) / 1024);
	else if (size_GiB >= 1)
		snprintf(lib_static_str[count], 128, "%ld.%ld GiB",
			size_GiB, ((size_MiB - size_GiB * 1024) * 10) / 1024);
	else if (size_MiB >= 1)
		snprintf(lib_static_str[count], 128, "%ld.%ld MiB",
			size_MiB, ((size_KiB - size_MiB * 1024) * 10) / 1024);
	else if (size_KiB >= 1)
		snprintf(lib_static_str[count], 128, "%ld.%ld KiB",
			size_KiB, (unsigned long)((size_bytes - size_KiB * 1024)  * 10) / 1024);
	else
		snprintf(lib_static_str[count], 128, "%.0f bytes", size_bytes);

	return (const char *)lib_static_str[count];
}

/*
 * itoa() is not ansi c so this one could be useful.
 * *** Allow up to 16 simultaneous calls ***
 */
const char *itoa2(long int n)
{
	/* Array of 16 strings (like for readable_size() - see above) */
	static char	lib_static_str[16][128];
	static int	count = -1;

	count++;
	count &= 15;
	snprintf(lib_static_str[count], 128, "%ld", n);
	return (const char *)lib_static_str[count];
}

/*
 * Modify string in place.
 */
char *remove_char_from_str(char *str, char c)
{
	int	i, len = strlen(str);

	for (i = 0; i < len && str[i] != '\0';) {
		if (str[i] == c)
			str_n_cpy(str + i, (const char *)(str + i + 1), len - i);
		else
			i++;
	}
	return str;
}

/*
 * Modify string in place.
 */
char *remove_trailing_whitespaces_from_str(char *str)
{
	int	i = strlen(str) - 1;

	while (isspace((int)str[i]) && i >= 0)
		i--;
	str[i + 1] = '\0';
	return str;
}

/*
 * Check that str contains only digits (at least one) and whitespaces
 * (may start and/or end with whitespaces), meaning numerical value is
 * integer >= 0.
 */
int str_is_num(const char *str2)
{
	char	*str, *s;
	int	i, len;

	if (*str2 == '\0')
		return FALSE;
	s = str = l_str_new(str2);
	len = strlen(str);
	i = 0;
	while (i < len)
		if (isspace((int)*s)) {
			i++;
			s++;
		} else
			break;
	if (i == len) {
		l_str_free(str);
		return FALSE;
	}
	while (i < len)
		if (isdigit((int)*s)) {
			i++;
			s++;
		} else
			break;
	if (i == len) {
		l_str_free(str);
		return TRUE;
	}
	while (i < len)
		if (isspace((int)*s)) {
			i++;
			s++;
		} else
			break;
	l_str_free(str);
	if (i == len)
		return TRUE;
	else
		return FALSE;
}

/*
 * Check that str contains only whitespaces or is empty.
 */
int str_is_blank(const char *str)
{
	int	i = 0, len = strlen(str);

	if (*str == '\0')
		return TRUE;
	while (i < len)
		if (isspace((int)*str)) {
			i++;
			str++;
		} else
			break;
	if (i == len)
		return TRUE;
	else
		return FALSE;
}

/*
 * Generate a random string up to 1023 chars long.
 * mode = a -> alpha / d -> digits / b ->  both
 */
const char *rnd_str(char mode, int length)
{
	static char	str[1024];
	char		str_a[65] = "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijkl";
	char		str_d[17] = "1234567890123456";
	char		str_b[65] = "1234567890abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ00";
	int		i = 0;

	if (length > 1023)
		length = 1023;
	srand(time(0));
	if (mode == 'a') {
		while (i < length)
			str[i++] = str_a[rand() & 63];
	} else if (mode == 'd') {
		while (i < length)
			str[i++] = str_d[rand() & 15];
	} else if (mode == 'b') {
		while (i < length)
			str[i++] = str_b[rand() & 63];
	} else
		big_error_in_lib(RNDSTR_UNKNOWN_MODE, "rnd_str()");
	str[i] = '\0';
	return str;
}

#ifdef LIBETM_EXPERIMENTAL_STUFF
/*
 * Very basic xor sym encryption for small strings (up to 511 chars).
 * *** Key must be ascii / if 2 chars match (in str and key) the string is cut off ***
 * *** Allow up to 16 simultaneous calls ***
 * Works with ascii strings, probably not otherwise so don't try with
 * exotic things...
 */
const char *str_crypt(const char *str, const char *key)
{
	/* Array of 16 strings (like for readable_size() - see above) */
	static char	str2[16][512];
	static int	count = -1;
	int		i, j;

	count++;
	count &= 15;
	for (i = 0, j = 0; i < 511 && str[i] != '\0'; i++) {
		if (key[j] == '\0')
			j = 0;
		str2[count][i] = str[i] ^ key[j++];
	}
	str2[count][i] = '\0';
	return (const char *)str2[count];
}

/*
 * Very basic crypto hash.
 */
unsigned long str_crypto_hash(const char *str, const char* salt)
{
	char		*str2;
	unsigned long	hash = 0;
	int		i = 0;

	str2 = (char *)str_crypt(str, salt);
	while (str2[i] != '\0') {
		hash += str2[i] * 1024 + (str2[i] + 128) * (32 + i) + str2[i] - 1;
		i ++;
	}
	return hash;
}
#endif
