#!/usr/bin/env python3
# -*- coding: utf-8 -*-

# Fenrir TTY screen reader
# By Chrys, Storm Dragon, and contributors.

import signal
import time
from multiprocessing import Process
from threading import Thread

from fenrirscreenreader.core import debug
from fenrirscreenreader.core.eventData import FenrirEventType


class ProcessManager:
    def __init__(self):
        self._Processes = []
        self._Threads = []

    def initialize(self, environment):
        self.env = environment
        self.running = self.env["runtime"]["EventManager"].get_running()
        self.add_simple_event_thread(
            FenrirEventType.heart_beat,
            self.heart_beat_timer,
            multiprocess=True,
        )

    def shutdown(self):
        self.terminate_all_processes()

    def terminate_all_processes(self):
        for proc in self._Processes:
            # try:
            #    proc.terminate()
            # except KeyboardInterrupt:
            #    pass
            # except:
            #    pass
            proc.join(timeout=5.0)  # Timeout to prevent hanging shutdown
        for t in self._Threads:
            t.join(timeout=5.0)  # Timeout to prevent hanging shutdown

    def heart_beat_timer(self, active):
        try:
            time.sleep(0.5)
        except Exception as e:
            self.env["runtime"]["DebugManager"].write_debug_out(
                "ProcessManager heart_beat_timer: Error during sleep: "
                + str(e),
                debug.DebugLevel.ERROR,
            )
        return time.time()

    def add_custom_event_thread(
        self, function, pargs=None, multiprocess=False, run_once=False
    ):
        event_queue = self.env["runtime"]["EventManager"].get_event_queue()
        original_sigint_handler = signal.signal(signal.SIGINT, signal.SIG_IGN)
        if multiprocess:
            t = Process(
                target=self.custom_event_worker_thread,
                args=(event_queue, function, pargs, run_once),
            )
            self._Processes.append(t)
        else:  # use thread instead of process
            t = Thread(
                target=self.custom_event_worker_thread,
                args=(event_queue, function, pargs, run_once),
            )
            self._Threads.append(t)
        t.start()
        signal.signal(signal.SIGINT, original_sigint_handler)

    def add_simple_event_thread(
        self, event, function, pargs=None, multiprocess=False, run_once=False
    ):
        original_sigint_handler = signal.signal(signal.SIGINT, signal.SIG_IGN)
        if multiprocess:
            t = Process(
                target=self.simple_event_worker_thread,
                args=(event, function, pargs, run_once),
            )
            self._Processes.append(t)
        else:
            t = Thread(
                target=self.simple_event_worker_thread,
                args=(event, function, pargs, run_once),
            )
            self._Threads.append(t)
        t.start()
        signal.signal(signal.SIGINT, original_sigint_handler)

    def custom_event_worker_thread(
        self, event_queue, function, pargs=None, run_once=False
    ):
        # if not isinstance(event_queue, Queue):
        #    return
        if not callable(function):
            return
        while self.running.value:
            try:
                if pargs:
                    function(self.running, event_queue, pargs)
                else:
                    function(self.running, event_queue)
            except Exception as e:
                self.env["runtime"]["DebugManager"].write_debug_out(
                    "ProcessManager:custom_event_worker_thread:function("
                    + str(function)
                    + "):"
                    + str(e),
                    debug.DebugLevel.ERROR,
                )
            if run_once:
                break

    def simple_event_worker_thread(
        self, event, function, pargs=None, run_once=False
    ):
        if not isinstance(event, FenrirEventType):
            return
        if not callable(function):
            return
        while self.running.value:
            data = None
            try:
                if pargs:
                    data = function(self.running, pargs)
                else:
                    data = function(self.running)
            except Exception as e:
                self.env["runtime"]["DebugManager"].write_debug_out(
                    "ProcessManager:simple_event_worker_thread:function("
                    + str(function)
                    + "):"
                    + str(e),
                    debug.DebugLevel.ERROR,
                )
            self.env["runtime"]["EventManager"].put_to_event_queue(event, data)
            if run_once:
                break
