/*
 * Unit tests for libifp
 * $Id: selftest.c,v 1.14 2004/12/02 01:33:33 oakhamg Exp $
 *
 * Copyright (C) Geoff Oakham, 2004; <oakhamg@users.sourceforge.net>
 */

static char const rcsid[] = "$Id: selftest.c,v 1.14 2004/12/02 01:33:33 oakhamg Exp $";

#include <stdio.h>
#include <string.h>
#include <usb.h>

#include <limits.h>
#include <errno.h>
#include <unistd.h>
#include <sys/stat.h>
#include <unistd.h>
#include <fts.h>

#include "ifp.h"

#define BUILDDIR "./"
#define TESTDATA "../testdata"
#define TEMPDIR "/tmp/ifp-selftest"
#define REMOTE_TESTDIR "\\debug"

#if 0
int t_freespace(struct ifp_device * dev) {
	int i = 0;
	long used = 0;
	long capacity;
	long freespace;
	void * tw = NULL;
	struct ifp_treewalk_entry * r = NULL;
	int e;
	int files = 0;
	int dirs = 0;

	i = ifp_treewalk_open(dev, "\\", &tw);
	ifp_err_jump(i, err, "couldn't open ifp:\\ for a treewalk");

	while(i == 0 && ((r = ifp_treewalk_next(tw)) != NULL)) {
		if (r->type == IFP_WALK_FILE) {
			used += r->filesize;
			files++;
		}
		if (r->type == IFP_WALK_DIR_PRE) {
			dirs++;
		}
	}
	e = ifp_treewalk_close(tw);
	if (e) {
		ifp_err_i(e, "error closing treewalk");
		i = i ? i : e;
	}
	if (i == 0) {
		capacity = ifp_capacity(dev);
		freespace = ifp_freespace(dev);
		if (used == (capacity - freespace)) {
			printf("space usage matches perfectly\n");
		} else {
			printf("space usage off by %ld over %d files and %d directories\n",
				(capacity-freespace-used), files, dirs);
			printf("used=%ld, freespace=%ld, capacity=%ld\n",
				used, freespace, capacity);
		}
	}
err:
	return i;
}
#endif


//#define BLOCK_SIZE 1024
#define BLOCK_SIZE 512
//compares two local files.  return 0 if equal, 1 otherwise
static int file_compare(const char * f1, const char * f2) {
	int i = 0;
	FILE * a1 = NULL;
	FILE * a2 = NULL;
	char b1[BLOCK_SIZE];
	char b2[BLOCK_SIZE];

	a1 = fopen(f1, "r");
	if (a1 == NULL) {
		ifp_err("error opening %s", f1);
		i = -errno;
		goto out0;
	}

	a2 = fopen(f2, "r");
	if (a2 == NULL) {
		ifp_err("error opening %s", f2);
		i = -errno;
		goto out1;
	}

	while(i == 0 && !feof(a1) && !feof(a2)) {
		int n1, n2;
		n1 = fread(b1, 1, sizeof(b1), a1);
		if (n1 != sizeof(b1) && ferror(a1)) {
			ifp_err("error reading %s at +%d", f1, (int)ftell(a1) - n1);
			i = errno;
			if (!i) {
				ifp_err("internal error.. how does this work?");
			}
			break;
		}

		n2 = fread(b2, 1, sizeof(b2), a2);
		if (n2 != sizeof(b2) && ferror(a2)) {
			ifp_err("error reading %s at +%d", f2, (int)ftell(a2) - n2);
			i = errno;
			if (!i) {
				ifp_err("internal error.. how does this work?");
			}
			break;
		}

		if (n1 != n2) {
			fprintf(stderr, "[file_compare] got different byte reads: %d and %d\n", n1, n2);
			i = 1;
			break;
		}

		if ((i = memcmp(b1, b2, n1)) != 0) {
			fprintf(stderr, "[file_compare] files '%s' and '%s' are different\n", f1, f2);
			i = 1;
			break;
		}
	}
	if (i==0 && (!feof(a1) || !feof(a2))) {
		i = 1;
		if (feof(a1)) {
			fprintf(stderr, "[file_compare] reached end of file one before the second.\n");
		} else {
			fprintf(stderr, "[file_compare] reached end of file two before the first.\n");
		}
	}

	fclose(a2); a2 = NULL;
out1:
	fclose(a1); a1 = NULL;
out0:
	return i;
}

//compares 'd1' with 'd2' and deletes 'd2' during the comparison
static int compare_trees(const char * d1, const char * d2) {
	int i = 0, e=0;
	FTS * tw = NULL;
	FTSENT * r = NULL;
	char * argv[2] = {(char *)d1, NULL};
	char path2[1024];
	int n=0;

	strncpy(path2, d2, sizeof(path2));

	n = strlen(path2);
	path2[n] = '/'; n++; path2[n] = '\0';

	//fprintf(stderr, "trying to rm -Rf %s\n", d);
	tw = fts_open(argv, FTS_LOGICAL | FTS_NOCHDIR, NULL);
	if (tw==NULL) {
		ifp_err("couldn't open %s", d1);
		i = -errno;
		goto err0;
	}
	while (i==0 && ((r = fts_read(tw)) != NULL) && r->fts_info != FTS_ERR) {
		switch(r->fts_info) {
		case FTS_F:
			strncpy(path2+n, r->fts_name, sizeof(path2) - n);
			//printf("f:  %s\n", path2);

			i = file_compare(r->fts_path, path2);
			ifp_err_jump(i, err1, "problem comparing %s with %s", r->fts_path, path2);
			//ifp_err_expect(i, i==1, err1, "problem comparing %s with %s", r->fts_path, path2);
			break;
		case FTS_D:
			if (r->fts_level > 0) {
				strncpy(path2+n, r->fts_name, sizeof(path2)-n);
				n += r->fts_namelen;
				path2[n] = '/'; n++; path2[n] = '\0';
				//printf("d:  %s\n", path2);
			}
			break;
		case FTS_DP:
			if (r->fts_level > 0) {
				//path2[n] = '\0';
				//printf("p:  %s\n", path2);

				n -= 1+r->fts_namelen;
				path2[n] = '\0';
				//printf("p:_ %s\n", path2);
			}
			break;
		}
	}
	if (r && r->fts_info == FTS_ERR) {
		i = r->fts_errno;
		ifp_err_i(i, "error walking tree");
	}

err1:
	e = fts_close(tw);
	if (e) {
		ifp_err_i(e, "couldn't finish fts tree walk");
		i = i ? i : e;
	}

err0:
	return i;
}

static int recursive_delete(const char * d) {
	int i = 0, e=0;
	FTS * tw = NULL;
	FTSENT * r = NULL;
	char * argv[2] = {(char *)d, NULL};

	//fprintf(stderr, "trying to rm -Rf %s\n", d);
	tw = fts_open(argv, FTS_LOGICAL | FTS_NOCHDIR, NULL);
	if (tw==NULL) {
		ifp_err("couldn't open %s", d);
		i = -errno;
		goto err0;
	}
	while (i==0 && ((r = fts_read(tw)) != NULL) && r->fts_info != FTS_ERR) {
		switch(r->fts_info) {
		case FTS_F:
			unlink(r->fts_path);
			break;
		case FTS_DP:
			rmdir(r->fts_path);
			break;
		}
	}
	if (r && r->fts_info == FTS_ERR) {
		i = r->fts_errno;
		ifp_err_i(i, "error walking tree");
	}

	e = fts_close(tw);
	if (e) {
		ifp_err_i(e, "couldn't finish fts tree walk");
		i = i ? i : e;
	}

err0:
	return i;
}

static int _noop_callback(void * context, int type, const char * name, int size)
{
	int i = 0;

	return i;
}

int t_directory_errors(struct ifp_device * dev) {
	int i = 0;
	const char * long_sample = REMOTE_TESTDIR "\\dont_be_a_mence_to_south_central_while_drinking_your_juice_in_da_hood____dont_be_a_mence_to_south_central_while_drinking";

	//----------------------------------------------------------------------
	// existing, non-existant dirnames
	i = ifp_list_dirs(dev, REMOTE_TESTDIR "\\non_existant", _noop_callback, NULL);
	if (i != -ENOENT) {
		ifp_err_i(i, "bad return code listing a non-existant directory");
		i = -1;
		goto err;
	}

	i = ifp_mkdir(dev, REMOTE_TESTDIR "\\auto_direrr4");
	ifp_err_jump(i, err, "mkdir4 failed");

	i = ifp_rename(dev, REMOTE_TESTDIR "\\auto_direrr4",
		REMOTE_TESTDIR "\\non_existant\\fooit4");
	if (i != -ENOENT) {
		ifp_err_i(i, "bad return code for moving to a non-existant directory");
		ifp_rmdir(dev, REMOTE_TESTDIR "\\auto_direrr4");
		i = -1;
		goto err;
	}
	i = ifp_rmdir(dev, REMOTE_TESTDIR "\\auto_direrr4");
	ifp_err_jump(i, err, "rmdir4 failed");

	//----------------------------------------------------------------------
	// reserved names 'VOICE' and 'RECORD'
	i = ifp_rmdir(dev, "\\VOICE");
	if (i != -EACCES) {
		ifp_err_i(i, "bad return code when attempting to delete \\VOICE");
		i = -1;
		goto err;
	}
	i = ifp_rmdir(dev, "\\RECORD");
	if (i != -EACCES) {
		ifp_err_i(i, "bad return code when attempting to delete \\RECORD");
		i = -1;
		goto err;
	}
	i = ifp_rename(dev, "\\VOICE", "\\nuVOICE");
	if (i != -EACCES) {
		ifp_err_i(i, "bad return code when attempting to rename \\VOICE");
		i = -1;
		goto err;
	}
	i = ifp_rename(dev, "\\RECORD", "\\nuRECORD");
	if (i != -EACCES) {
		ifp_err_i(i, "bad return code when attempting to rename \\RECORD");
		i = -1;
		goto err;
	}

	//----------------------------------------------------------------------
	// invalid characters in filenames "/:*?\"<>|"
	i = ifp_mkdir(dev, REMOTE_TESTDIR "\\hello?");
	if (i != IFP_ERR_BAD_FILENAME) {
		ifp_err_i(i, "bad return code when attempting to use an invalid character'?'");
		i = -1;
		goto err;
	}
	i = ifp_mkdir(dev, REMOTE_TESTDIR "\\he:llo.ddd");
	if (i != IFP_ERR_BAD_FILENAME) {
		ifp_err_i(i, "bad return code when attempting to use an invalid character':'");
		i = -1;
		goto err;
	}
	i = ifp_mkdir(dev, REMOTE_TESTDIR "\\*hello.ddd");
	if (i != IFP_ERR_BAD_FILENAME) {
		ifp_err_i(i, "bad return code when attempting to use an invalid character'*'");
		i = -1;
		goto err;
	}
	i = ifp_mkdir(dev, REMOTE_TESTDIR "\\dont_be_a_mence_to_south_central_while_drinking_your_juice_in_da_hood____dont_be_a_mence_to_south_central_while_drinking_your_ju");
	if (i != IFP_ERR_BAD_FILENAME) {
		ifp_err_i(i, "bad return code when attempting to use an invalid character'*'");
		i = -1;
		goto err;
	}
		//OK: \debug\[121 characters] or 128 bytes _total_
		//not ok: \debug\[122 characters] or 129 bytes _total_
	i = ifp_mkdir(dev, long_sample);
	ifp_err_jump(i, err, "couldn't create really long directory name ifp:\\%s", long_sample);

	i = ifp_rmdir(dev, long_sample);
	ifp_err_jump(i, err, "couldn't remove directory with really long name ifp:\\%s", long_sample);

	i=0;

err:
	return i;
}


static int ifp_touch(struct ifp_device * dev, const char * f) {
	int i = 0;

	i = ifp_write_open(dev, f, 0);
	ifp_err_expect(i, i==-ENOENT || i==-EEXIST || i==IFP_ERR_BAD_FILENAME, err,
		"error opening new file %s", f);

	i = ifp_write_close(dev);
	ifp_err_jump(i, err, "error closing new file %s", f);

err:
	return i;
}

int try_rename(struct ifp_device * dev, int filesize, const char * oldname, const char * newname) {
	int i = 0;
	int remotesize;

	i = ifp_rename(dev, oldname, newname);
	ifp_err_jump(i, err, "couldn't rename ifp:\\%s to ifp:\\%s", oldname, newname);

	i = ifp_read_open(dev, newname);
	ifp_err_jump(i, err, "couldn't open renamed file ifp:\\%s", newname);

	remotesize = ifp_read_size(dev);

	i = ifp_read_close(dev);
	ifp_err_jump(i, err, "couldn't close renamed file ifp:\\%s", newname);

	if (remotesize != filesize) {
		ifp_err("rename failed, the file sizes are different (%d instead of %d)",
			remotesize, filesize);
		i = -1;
		goto err;
	}


err:
	return i;
}

int t_rename_errors(struct ifp_device * dev) {
	int i = 0;
	const char * localfile = BUILDDIR "/selftest.o";
	const char * f1 = REMOTE_TESTDIR "\\auto_renameA1.dat";
	const char * f2 = REMOTE_TESTDIR "\\auto_renameA2.dat";
	const char * f3 = "\\root_auto_renameA3.dat";
	const char * f4 = REMOTE_TESTDIR "\\auto_renameA4.dat";
#if 0
	const char * t = TEMPDIR "/rename_err1.dat";
	const char * mp3 = REMOTE_TESTDIR "\\auto_renameA5.mp3";
#endif
	struct stat st;
	
	//big basic batch of tests
	i = stat(localfile, &st);
	if (i) { i = errno; }
	ifp_err_jump(i, err, "couldn't stat %s", localfile);

	i = ifp_upload_file(dev, localfile, f1, NULL, NULL);
	ifp_err_jump(i, err, "couldn't upload file");

	i = try_rename(dev, st.st_size, f1, f2);
	ifp_err_jump(i, err, "couldn't rename ifp:\\%s", f1);

	i = try_rename(dev, st.st_size, f2, f3);
	ifp_err_jump(i, err, "couldn't rename ifp:\\%s", f2);

	i = try_rename(dev, st.st_size, f3, f4);
	ifp_err_jump(i, err, "couldn't rename ifp:\\%s", f3);

	//apparently these extra tests leave the device in a wierd state.
#if 0
	i = ifp_download_file(dev, f4, t, NULL, NULL);
	ifp_err_jump(i, err, "couldn't download ifp:\\%s", f4);
#endif

	i = ifp_delete(dev, f4);
	ifp_err_jump(i, err, "couldn't remove renamed file");

#if 0
	i = file_compare(localfile, t);
	ifp_err_jump(i, err, "files %s and %s are different", localfile, t);
	unlink(t);

	//mp3 file reading
	i = ifp_upload_file(dev, localfile, mp3, NULL, NULL);
	ifp_err_jump(i, err, "couldn't upload ifp:\\%s", mp3);

	i = ifp_download_file(dev, mp3, t, NULL, NULL);
	ifp_err_jump(i, err, "couldn't download ifp:\\%s", mp3);

	i = ifp_delete(dev, mp3);
	ifp_err_jump(i, err, "couldn't remove mp3 file ifp:\\%s", mp3);

	i = file_compare(localfile, t);
	ifp_err_jump(i, err, "files %s and %s are different", localfile, t);
	unlink(t);
#endif

	i=0;
err:
	return i;
}



int t_file_errors(struct ifp_device * dev) {
	int i = 0;

	i = ifp_delete(dev, REMOTE_TESTDIR "\\non_existant.txt");
	if (i != -ENOENT) {
		ifp_err_i(i, "bad return code deleting non-existant file");
		i = -1;
		goto err;
	}

	i = ifp_read_open(dev, REMOTE_TESTDIR "\\non_existant.txt");
	if (i != -ENOENT) {
		ifp_err_i(i, "bad return code opening non-existant file");
		if (i == 0) {
			//ifp_read_open thinks it succeeded, so let's humour it
			//and close the file.
			ifp_read_close(dev);
		}
		i = -1;
		goto err;
	}
	i = ifp_read_open(dev, REMOTE_TESTDIR "\\non_existant\\foo.txt");
	if (i != -ENOENT) {
		ifp_err_i(i, "bad return code reading file i non-dir");
		if (i == 0) {
			//ifp_read_open thinks it succeeded, so let's humour it
			//and close the file.
			ifp_read_close(dev);
		}
		i = -1;
		goto err;
	}
	i = ifp_touch(dev, REMOTE_TESTDIR "\\non_existant\\bar.txt");
	if (i != -ENOENT) {
		ifp_err_i(i, "bad return code creating file in non-dir");
		ifp_delete(dev, REMOTE_TESTDIR "\\auto_taken.txt");
		i = -1;
		goto err;
	}

	//----------------------------------------------------------------------
	//file allready exists.
	i = ifp_touch(dev, REMOTE_TESTDIR "\\auto_taken.txt");
	ifp_err_jump(i, err, "error touching file");

	i = ifp_touch(dev, REMOTE_TESTDIR "\\auto_taken.txt");
	if (i != -EEXIST) {
		ifp_err_i(i, "bad return code creating pre-existing file");
		ifp_delete(dev, REMOTE_TESTDIR "\\auto_taken.txt");
		i = -1;
		goto err;
	}
	i = ifp_delete(dev, REMOTE_TESTDIR "\\auto_taken.txt");
	ifp_err_jump(i, err, "error removing file");
	
	//----------------------------------------------------------------------
	//special case: filename is in use by a directory.
	i = ifp_mkdir(dev, REMOTE_TESTDIR "\\auto_rmdir3");
	ifp_err_jump(i, err, "mkdir3 failed");
	i = ifp_touch(dev, REMOTE_TESTDIR "\\auto_rmdir3");
	if (i != -EEXIST) {
		ifp_err_i(i, "bad return code creating pre-existing file");
		ifp_rmdir(dev, REMOTE_TESTDIR "\\auto_rmdir3");
		i = -1;
		goto err;
	}
	i = ifp_rmdir(dev, REMOTE_TESTDIR "\\auto_rmdir3");
	ifp_err_jump(i, err, "rmdir3 failed");

	//----------------------------------------------------------------------
	//special case: dirname is in use by a file.
	i = ifp_touch(dev, REMOTE_TESTDIR "\\auto_takenB");
	ifp_err_jump(i, err, "error touching file (B)");
	i = ifp_mkdir(dev, REMOTE_TESTDIR "\\auto_takenB");
	if (i != -EEXIST) {
		ifp_err_i(i, "bad return code creating pre-existing dir");
		if (i == 0) {
			ifp_rmdir(dev, REMOTE_TESTDIR "\\auto_takenB");
		}
		ifp_delete(dev, REMOTE_TESTDIR "\\auto_takenB");
		i = -1;
		goto err;
	}
	i = ifp_delete(dev, REMOTE_TESTDIR "\\auto_takenB");
	ifp_err_jump(i, err, "delete auto_takenB failed");

	//----------------------------------------------------------------------
	// invalid characters in filenames "/:*?\"<>|"
	i = ifp_touch(dev, REMOTE_TESTDIR "\\hello?.txt");
	if (i != IFP_ERR_BAD_FILENAME) {
		ifp_err_i(i, "bad return code when attempting to use an invalid character'?'");
		i = -1;
		goto err;
	}
	i = ifp_touch(dev, REMOTE_TESTDIR "\\he\"llo.ddd");
	if (i != IFP_ERR_BAD_FILENAME) {
		ifp_err_i(i, "bad return code when attempting to use an invalid character'\"'");
		i = -1;
		goto err;
	}
	i = ifp_touch(dev, REMOTE_TESTDIR "\\<hello.ddd");
	if (i != IFP_ERR_BAD_FILENAME) {
		ifp_err_i(i, "bad return code when attempting to use an invalid character'<'");
		i = -1;
		goto err;
	}

	i=0;

err:
	return i;
}

int t_rmdir(struct ifp_device * dev) {
	int i = 0;

	//non-existant directory name
	i = ifp_rmdir(dev, REMOTE_TESTDIR "\\non_existant");
	if (i != -ENOENT) {
		ifp_err_i(i, "bad return code on removing non-existant directory");
		i = -1;
		goto err;
	}

	i = ifp_mkdir(dev, REMOTE_TESTDIR "\\non_existant\\subbed");
	if (i != -ENOENT) {
		ifp_err_i(i, "bad return code creating a sub of non-existant directory");
		i = -1;
		goto err;
	}

	//normal case
	i = ifp_mkdir(dev, REMOTE_TESTDIR "\\auto_rmdir1");
	ifp_err_jump(i, err, "mkdir1 failed");

	i = ifp_rmdir(dev, REMOTE_TESTDIR "\\auto_rmdir1");
	if (i != 0) {
		ifp_err_i(i, "bad return code removing empty directory");
		i = -1;
		goto err;
	}

	//normal case
	i = ifp_mkdir(dev, REMOTE_TESTDIR "\\auto_rmdir2");
	ifp_err_jump(i, err, "mkdir2.1 failed");
	i = ifp_mkdir(dev, REMOTE_TESTDIR "\\auto_rmdir2\\subbed");
	ifp_err_jump(i, err, "mkdir2.2 failed");

	i = ifp_rmdir(dev, REMOTE_TESTDIR "\\auto_rmdir2");
	if (i != -ENOTEMPTY) {
		ifp_err_i(i, "bad return code removing non-empty directory");
		i = -1;
		ifp_rmdir(dev, REMOTE_TESTDIR "\\auto_rmdir2\\subbed");
		ifp_rmdir(dev, REMOTE_TESTDIR "\\auto_rmdir2");
		goto err;
	}

	i = ifp_rmdir(dev, REMOTE_TESTDIR "\\auto_rmdir2\\subbed");
	ifp_err_jump(i, err, "rmdir2.2 failed");
	i = ifp_rmdir(dev, REMOTE_TESTDIR "\\auto_rmdir2");
	ifp_err_jump(i, err, "rmdir2.1 failed");

err:
	return i;
}

int dump_stations(char * b) {
	char label[7];
	int freq;
	int i = 0;
	int j;

	for (j=0; j != 20 && i == 0; j++) {
		i = ifp_get_station(j, b, label, &freq);
		//i = copy_station(b, label, fq);
		printf("%2d: %-6.6s %3d.%02dMHz\n", j+1, label, freq/100, freq%100);
	}
	
	return i;
}

int check_station(char * b) {
	char label[7];
	//char * bm = b + 6;
	int i = 0, freq;

	if (b[3] != '.') {
		ifp_err("dot not present");
		return -1;
	}
	if (b[0] < 0 || b[0] > 9) {
		ifp_err("digit 1 out of range");
		return -1;
	}
	if (b[1] < 0 || b[1] > 9) {
		ifp_err("digit 2 out of range");
		return -1;
	}
	if (b[2] < 0 || b[2] > 9) {
		ifp_err("digit 3 out of range");
		return -1;
	}
	if (b[4] < 0 || b[4] > 9) {
		ifp_err("digit 4 out of range");
		return -1;
	}
	if (b[5] < 0 || b[5] > 9) {
		ifp_err("digit 5 out of range");
		return -1;
	}

	i = ifp_get_station(0, b, label, &freq);
	ifp_err_jump(i, err, "can't read station");

	if (freq > 10800 || freq < 8750) {
		ifp_err("frequency out of range: %d", freq);
		i = -1;
	}

err:
	//printf("station: '%-6.6s'.\n",bm);
	return i;
}

int t_tuner(struct ifp_device * dev) {
	int i = 0;
	int j = 0;
	uint8_t buf[IFP_TUNER_PRESET_DATA];
	uint8_t b2 [IFP_TUNER_PRESET_DATA];
	char stn[7];
	int frq;

	//step 1, get the stations--and perform basic sanity check.
	i = ifp_get_tuner_presets(dev, buf, sizeof(buf));
	ifp_err_jump(i, err, "can't get presets");
	for (j = 0; i == 0 && j != 20; j++) {
		i = check_station(buf + 12*j);
		if (i) {
			ifp_err_i(i, "station num %d failed check", j);
		}
	}
#if 0
	i = dump_stations(buf);
	ifp_err_jump(i, err, "can't dump presets");
#endif

	//step 2, change one of the stations
	i = ifp_get_station(19, buf, stn, &frq);
	ifp_err_jump(i, err, "can't get station");

	i = ifp_set_station(19, buf, "axant",   10900);
	if (i != IFP_ERR_BAD_FREQUENCY) {
		ifp_err("bad response to a bad frequency");
		i = -1; goto err;
	}

	//i = ifp_set_station(19, buf, "",   "87.50");
	i = ifp_set_station(19, buf, "exuntus",   10780);
	ifp_err_jump(i, err, "can't change station");

#if 0
	i = dump_stations(buf);
	ifp_err_jump(i, err, "can't dump presets");
#endif

	i = ifp_set_tuner_presets(dev, buf, sizeof(buf));
	ifp_err_jump(i, err, "can't set presets");

	i = ifp_get_tuner_presets(dev, b2, sizeof(b2));
	ifp_err_jump(i, err, "can't get presets (second time)");

	i = memcmp(buf, b2, sizeof(buf));
	if (i != 0) {
		ifp_err_i(i, "data doesn't match");
		printf("buf: \n");
		dump_stations(buf);
		printf("b2: \n");
		dump_stations(b2);
		i = -1;
		goto err;
	}

	//step 3, change the station back
	i = ifp_set_station(19, buf, stn, frq);
	ifp_err_jump(i, err, "can't revert station");

	i = ifp_set_tuner_presets(dev, buf, sizeof(buf));
	ifp_err_jump(i, err, "can't revert presets");

	i = ifp_get_tuner_presets(dev, b2, sizeof(b2));
	ifp_err_jump(i, err, "can't get presets (third time)");

	i = memcmp(buf, b2, sizeof(buf));
	if (i != 0) {
		ifp_err_i(i, "data doesn't match (second time)");
		printf("buf: \n");
		dump_stations(buf);
		printf("b2: \n");
		dump_stations(b2);
		i = -1;
		goto err;
#if 0
	} else {
		printf("latest settings: \n");
		dump_stations(b2);
#endif
	}
#if 0
	i = ifp_set_station(19, buf, "normalusses",   8750);
	ifp_err_jump(i, err, "can't set it to something else ");
	i = ifp_set_tuner_presets(dev, buf, sizeof(buf));
	ifp_err_jump(i, err, "can't re-change presets");
#endif


err:
	return i;
}

int cancel_cb(void * context, struct ifp_transfer_status * ts) {
	int * c = context;
	*c += 1;
	if (*c == 2) {
		return 1;
	}
	return 0;
}

int progress_cb(void * context, struct ifp_transfer_status * ts) {
	int * c = context;
	*c += 1;
	return 0;
}

int t_transfer(struct ifp_device * dev)
{
	int i = 0;
	int ncount = 0;
	ifp_progress cp = progress_cb;
	const char * source = BUILDDIR "/selftest.o";
	const char * local = "selftest_xfer.dat";
	const char * remote = REMOTE_TESTDIR "\\selftest_xfer.dat";

	i = ifp_delete(dev, remote);
	if (i == -ENOENT) {
		//nothing
	} else if (i == 0) {
		printf("removed ifp:\\%s from previous test\n", remote);
	}


	//normal upload
	ncount = 0;
	i = ifp_upload_file(dev, source, remote, cp, &ncount);
	ifp_err_jump(i, err, "couldn't upload file");
	if (ncount < 2) {
		ifp_err("ncount is only %d", ncount);
		ifp_delete(dev, remote);
		goto err;
	}

	//Cancelled download
	ncount = 0;
	i = ifp_download_file(dev, remote, local, cancel_cb, &ncount);
	if (i != IFP_ERR_USER_CANCEL) {
		ifp_err_i(i, "bad cancel status code");
		i = i?i:-1;
		unlink(local);
		goto err;
	}
	if (ncount != 2) {
		ifp_err("ncount is only %d", ncount);
		i = -1;
		unlink(local);
		goto err;
	}

	i = unlink(local);
	if (i == 0 || errno != ENOENT) {
		int e = errno;
		ifp_err("(i=%d, errno=%d) found partial downloaded file %s", i,e, local);
		i = errno;
		goto err;
	}

	//normal download
	ncount = 0;
	i = ifp_download_file(dev, remote, local, cp, &ncount);
	ifp_err_jump(i, err, "couldn't download file");
	if (ncount < 2) {
		ifp_err("ncount is only %d", ncount);
		ifp_delete(dev, remote);
		unlink(local);
		goto err;
	}
	i = file_compare(source, local);
	ifp_err_jump(i, err, "files %s and %s are different", source, local);

	//MISSING: compare the downloaded file with the original.
	i = unlink(local);
	ifp_err_jump(i, err, "problem removing downloaded file %s", local);

	i = ifp_delete(dev, remote);
	ifp_err_jump(i, err, "couldn't remove test file");

	//Cancelled upload
	ncount = 0;
	i = ifp_upload_file(dev, source, remote, cancel_cb, &ncount);
	if (i != IFP_ERR_USER_CANCEL) {
		ifp_err_i(i, "bad cancel status code");
		i = i?i:-1;
		ifp_delete(dev, remote);
		goto err;
	}
	if (ncount != 2) {
		ifp_err("ncount is only %d", ncount);
		i = -1;
		ifp_delete(dev, remote);
		goto err;
	}
	i = ifp_delete(dev, remote);
	if (i != -ENOENT) {
		ifp_err("found partial file on device ifp:\\%s", remote);
		i = -1;
		goto err;
	}
	i = 0;

err:
	return i;
}

int t_recursive_delete(struct ifp_device * dev)
{
	int i = 0;

#if 0
	//enabled this if there's a bug causing this test to fail and 
	//leaving your device with a partially created \debug\rmdel_t
	//directory.
	i = ifp_delete_dir_recursive(dev, REMOTE_TESTDIR "\\rmdel_t");
	ifp_err_i(i, "here");
#endif

	i = ifp_mkdir(dev, REMOTE_TESTDIR "\\rmdel_t");
	ifp_err_jump(i, err, "mkdir 1");
	i = ifp_mkdir(dev, REMOTE_TESTDIR "\\rmdel_t\\a");
	ifp_err_jump(i, err, "mkdir 1.1");
	i = ifp_touch(dev, REMOTE_TESTDIR "\\rmdel_t\\a\\foo_r.txt");
	ifp_err_jump(i, err, "touch 1.1a");
	i = ifp_mkdir(dev, REMOTE_TESTDIR "\\rmdel_t\\a\\b");
	ifp_err_jump(i, err, "mkdir 1.1.1");
	i = ifp_mkdir(dev, REMOTE_TESTDIR "\\rmdel_t\\a\\b\\c");
	ifp_err_jump(i, err, "mkdir 1.1.1.1");
	i = ifp_touch(dev, REMOTE_TESTDIR "\\rmdel_t\\a\\foo_c.txt");
	ifp_err_jump(i, err, "touch 1.1.2");
	i = ifp_touch(dev, REMOTE_TESTDIR "\\rmdel_t\\a\\foo_d.txt");
	ifp_err_jump(i, err, "touch 1.1.3");

	i = ifp_mkdir(dev, REMOTE_TESTDIR "\\rmdel_t\\b");
	ifp_err_jump(i, err, "mkdir 1.2");
	i = ifp_touch(dev, REMOTE_TESTDIR "\\rmdel_t\\b\\foo_c.txt");
	ifp_err_jump(i, err, "touch 1.2.2");
	i = ifp_touch(dev, REMOTE_TESTDIR "\\rmdel_t\\b\\foo_d.txt");
	ifp_err_jump(i, err, "touch 1.2.3");
	i = ifp_mkdir(dev, REMOTE_TESTDIR "\\rmdel_t\\b\\c");
	ifp_err_jump(i, err, "mkdir 1.2.4");
	i = ifp_touch(dev, REMOTE_TESTDIR "\\rmdel_t\\b\\c\\foo_d.txt");
	ifp_err_jump(i, err, "touch 1.2.4.1");

	i = ifp_delete_dir_recursive(dev, REMOTE_TESTDIR "\\rmdel_t");
	ifp_err_jump(i, err, "delete failed");
	
	i = ifp_delete_dir_recursive(dev, REMOTE_TESTDIR "\\fake_non_exist_q");
	if (i != -ENOENT) {
		ifp_err_i(i, "bad return value for non-existant directory");
		i = -1;
		goto err;
	}
	i = 0;

err:
	return i;
}

//                          0123  1=fill1, 2==fill2, 3==fill1 and fill2
  static char * artchars = " ~_=";
//static char * artchars = " PbB";
//static char * artchars = " ^_=";
//static char * artchars = " ',=";

/** sizeof(bar) >= width + 1
 * fill >= 0 && fill <= 1*/
int fill_dualbar(char * bar, int width, double fill1, double fill2) {
	int i = 0;
	int int_fill1 = (int)(fill1*(double)width + 0.5);
	int int_fill2 = (int)(fill2*(double)width + 0.5);
	if (fill1 > 1.0 || fill1 < 0.0) {
		fprintf(stderr, "fill1=%g is out of bounds\n", fill1);
		return -1;
	}
	if (fill2 > 1.0 || fill2 < 0.0) {
		fprintf(stderr, "fill2=%g is out of bounds\n", fill2);
		return -1;
	}
	if (int_fill1 > width || int_fill1 < 0) {
		fprintf(stderr, "internal error for fill1\n");
		return -1;
	}
	if (int_fill2 > width || int_fill2 < 0) {
		fprintf(stderr, "internal error for fill2\n");
		return -1;
	}
	bar[width] = '\0';
	for (i = 0; i != width; i++) {
		int offset = 0;
		if (i < int_fill1) {
			offset |= 1;
		}
		if (i < int_fill2) {
			offset |= 2;
		}
		bar[i] = artchars[offset];
	}
	return 0;
}

int progress_bar_dual(void * context, struct ifp_transfer_status * p) {
	int i = 0;
	char bar[21];

	if (p==NULL) {
		printf("p is null\n");
		return -1;
	}
	if (p->batch_total == 0) {
		fprintf(stdout, "batch_total is 0\n");
		return -1;
	} else if (p->file_total == 0) {
		fprintf(stdout, "batch_total is 0\n");
		return -1;
	} else {
		//fprintf(stdout, "debug, in progress_test\n");
	}
	fill_dualbar(bar, sizeof(bar) - 1,
		(double)p-> file_bytes/(double)p-> file_total,
		(double)p->batch_bytes/(double)p->batch_total);
	//fillbar(bar2, 10, (double)p-> file_bytes/(double)p-> file_total);
	fprintf(stdout, "[%20.20s] %-50.50s\r", bar, p->file_name);
	fflush(stdout);

	return i;
}

/** sizeof(bar) >= width + 1
 * fill >= 0 && fill <= 1*/
int fillbar(char * bar, int width, double fill) {
	int int_fill = (int)(fill*(double)width + 0.5);
	if (fill > 1.0 || fill < 0.0) {
		fprintf(stderr, "fill=%g is out of bounds\n", fill);
		return -1;
	}
	if (int_fill > width || int_fill < 0) {
		fprintf(stderr, "internal error\n");
		return -1;
	}
	bar[width] = '\0';
	memset(bar, ' ', width);
	memset(bar, '#', int_fill);
	return 0;
}

int progress_test(void * context, struct ifp_transfer_status * p) {
	int i = 0;
	char bar1[11];
	char bar2[11];

	if (p==NULL) {
		printf("p is null\n");
		return -1;
	}
	if (p->batch_total == 0) {
		fprintf(stdout, "batch_total is 0\n");
		return -1;
	} else if (p->file_total == 0) {
		fprintf(stdout, "batch_total is 0\n");
		return -1;
	} else {
		//fprintf(stdout, "debug, in progress_test\n");
	}
	fillbar(bar1, 10, (double)p->batch_bytes/(double)p->batch_total);
	fillbar(bar2, 10, (double)p-> file_bytes/(double)p-> file_total);
	fprintf(stdout, "[%10.10s|%10.10s] %-50.50s\r", bar1, bar2, p->file_name);
	fflush(stdout);

	return i;
}

int t_recursive_download(struct ifp_device * dev)
{
	int i = 0;

	if (ifp_is_dir(dev, REMOTE_TESTDIR "\\self_recursive1") == 1) {
		i = ifp_delete_dir_recursive(dev, REMOTE_TESTDIR "\\self_recursive1");
		ifp_err_jump(i, err, "pre-cleanup failed");
	}

	//i = ifp_upload_dir(dev, TESTDATA "/d1", REMOTE_TESTDIR "\\self_recursive1", progress_test, NULL);
	i = ifp_upload_dir(dev, TESTDATA "/d1", REMOTE_TESTDIR "\\self_recursive1", progress_bar_dual, NULL);
	ifp_err_jump(i, err, "upload failed");
	fprintf(stdout, "done.%-70.70s\r", "");
	fflush(stdout);
	
	//i = ifp_download_dir(dev, REMOTE_TESTDIR "\\self_recursive1", TEMPDIR "/self_recursive", progress_test, NULL);
	i = ifp_download_dir(dev, REMOTE_TESTDIR "\\self_recursive1", TEMPDIR "/self_recursive", progress_bar_dual, NULL);
	ifp_err_jump(i, err, "download failed");
	fprintf(stdout, "done.%-70.70s\r", "");
	fflush(stdout);
	
	i = ifp_delete_dir_recursive(dev, REMOTE_TESTDIR "\\self_recursive1");
	ifp_err_jump(i, err, "remote cleanup failed");

	i = compare_trees(TESTDATA "/d1", TEMPDIR "/self_recursive");
	ifp_err_jump(i, err, "compare failed");

	i = recursive_delete(TEMPDIR "/self_recursive");
	ifp_err_jump(i, err, "local cleanup failed");

err:
	return i;
}

int local_is_dir(const char * f) {
	struct stat st;
	stat(f, &st);
	return S_ISDIR(st.st_mode);
}

#define check_test_result(i,t) if (i){ printf("test %s failed: %s\n", t, ifp_error_message(i)); } else { printf("test %s succeeded.\n", t); }

int do_something(struct ifp_device * dev) {
	int i = 0, e=0;
	int made_remote;
	int made_local;
	int has_testdata;
	int freespace;
	char buf[255];

	freespace = ifp_freespace(dev);
	if (freespace < 0) {
		ifp_err_i(freespace, "(first) freespace call failed");
		e = e ? e : freespace;
	}


	i = ifp_device_info(dev, buf, sizeof(buf));
	if (i) {
		printf("device info failed, i=%d.\n", i);
		return i;
	}
	printf("Detected: %s\n", buf);


	if (ifp_is_dir(dev, REMOTE_TESTDIR)) {
		made_remote = 0;
	} else {
		made_remote = 1;
		i = ifp_mkdir(dev, REMOTE_TESTDIR);
		if (i) {
			ifp_err_i(i, "mkdir ifp:\\%s failed", REMOTE_TESTDIR);
			return i;
		}
		//printf("I need the directory ifp:\\" REMOTE_TESTDIR " on the device to run these tests.  (But I don't want to do it without your permission.)  Please create it and try again.\n");
		//return -1;
	}
	if (local_is_dir(TEMPDIR)) {
		made_local = 0;
	} else {
		made_local = 1;
		i = mkdir(TEMPDIR,0777);
		if (i) {
			i = -errno;
			ifp_err_i(i, "mkdir i%s failed", TEMPDIR);
			if (made_remote) {
				ifp_rmdir(dev, REMOTE_TESTDIR);
			}
			return i;
		}
	}
	if (local_is_dir(TESTDATA) && local_is_dir(TESTDATA "/d1")) {
		has_testdata = 1;
	} else {
		has_testdata = 0;
	}

#if 0
	i = t_freespace(dev);
	check_test_result(i, "freespace  ");
	e = e ? e : i;
#endif

	i = t_rmdir(dev);
	check_test_result(i, "mkdir/rmdir");
	e = e ? e : i;

	i = t_file_errors(dev);
	check_test_result(i, "file_errors");
	e = e ? e : i;

	i = t_directory_errors(dev);
	check_test_result(i, "dir_errors ");
	e = e ? e : i;

	i = t_rename_errors(dev);
	check_test_result(i, "rename_err ");
	e = e ? e : i;

	i = t_tuner(dev);
	check_test_result(i, "tuner prst ");
	e = e ? e : i;

	i = t_transfer(dev);
	check_test_result(i, "transfer   ");
	e = e ? e : i;

	i = t_recursive_delete(dev);
	check_test_result(i, "rm -Rf     ");
	e = e ? e : i;

	if (has_testdata) {
		i = t_recursive_download(dev);
		check_test_result(i, "tree d/l   ");
		e = e ? e : i;
	} else {
		printf("skipping bulk transfer tests\n");
	}

	if (made_local) {
		i = rmdir(TEMPDIR);
		if (i) {
			i = -errno;
			e = e ? e : i;
			ifp_err_i(i, "local cleanup of %s failed", TEMPDIR);
		}
	}
	if (made_remote) {
		i = ifp_rmdir(dev, REMOTE_TESTDIR);
		if (i) {
			ifp_err_i(i, "remote cleanup of ifp:\\%s failed", REMOTE_TESTDIR);
			e = e ? e : i;
		}
	}

	//This 'freespace' check is supposed to catch a broken 'rename'
	//implementation that might loose discspace.  (This happened to me
	//while I was debugging rename().. this test is to make sure it's still
	//not happening.)
	i = ifp_freespace(dev);
	if (i < 0) {
		ifp_err_i(i, "(second) freespace call failed");
		e = e ? e : i;
	}
	if (i != freespace) {
		ifp_err("freespace available before and after tests are different.  (%d and %d bytes respectively)",
			freespace, i);
		i = -1;
		e = e ? e : i;
	}

	return e;
}

int main(int argc, char **argv)
{
	struct usb_device *dev = NULL;
	usb_dev_handle *dh;
	struct ifp_device ifpdev;
	int i=0, e=0;
	    
	usb_init();

	dh = ifp_find_device();
	if (dh == NULL) {
		fprintf(stderr, "A suitable iRiver iFP device couldn't be found; "
		    "perhaps it's unplugged or turned off.\n");
		i = 1;
		goto out_0;
	}

	dev = usb_device(dh);
	/* "must be called" written in the libusb documentation */
	if (usb_claim_interface(dh, dev->config->interface->altsetting->
		bInterfaceNumber))
	{
		fprintf(stderr, "Device is busy.  (I was unable to claim its"
			" interface.)\n");
		i = 1;
		goto out_1;
	}

	i = ifp_init(&ifpdev, dh);
	if (i) {
		printf("Device failed to initialize:\n%s\n (err=%d)\n",
			ifp_error_message(i), i);
		goto out_2;
	}
    
	i = do_something(&ifpdev);
	if (i) {
		printf("do_something failed, i=%d.\n%s\n", i,
			ifp_error_message(i));
		goto out_3;
	}

out_3:
	e = ifp_finalize(&ifpdev);
	if (e) { fprintf(stderr, "warning: finalize failed, i=%d\n",e); i = i?i:e; }

out_2:
	usb_release_interface(dh,
		dev->config->interface->altsetting->bInterfaceNumber);
out_1:

	e = ifp_release_device(dh);
        if (e) { fprintf(stderr, "warning: release_device failed, i=%d\n",e); i=i?i:e; }
out_0:
#if 0
	if (i) {
		fprintf(stderr, "attempting to return %d or %d\n", i, i?1:0);
	}
	return i ? 1 : 0;
#else 
	return i;
#endif
}

