/*
 * Data, and supporting functions, for PostScript-based output
 * formats.
 */

#include <assert.h>
#include "halibut.h"
#include "paper.h"

/*
 * Within the paper backends, PostScript glyph names are represented
 * by small integers.  For standard glyphs, these are indicies into
 * a table generated from the Adobe Glyph List from
 * <http://partners.adobe.com/public/developer/en/opentype/glyphlist.txt>.
 * Since all the scripts that generate fragments of code for this file
 * need that list, it's worth generating a file containing just the names.

grep '^[^#;][^;]*;[^ ][^ ][^ ][^ ]$' glyphlist.txt | sort -t\; -k1,2 | \
    cut -f1 -d\; > glyphnames.txt

< glyphnames.txt xargs printf '"%s", ' | fold -s -w68 | sed 's/^/    /'; echo

 */
static const char *const ps_glyphs_alphabetic[] = {
    "A", "AE", "AEacute", "AEmacron", "AEsmall", "Aacute", 
    "Aacutesmall", "Abreve", "Abreveacute", "Abrevecyrillic", 
    "Abrevedotbelow", "Abrevegrave", "Abrevehookabove", "Abrevetilde", 
    "Acaron", "Acircle", "Acircumflex", "Acircumflexacute", 
    "Acircumflexdotbelow", "Acircumflexgrave", "Acircumflexhookabove", 
    "Acircumflexsmall", "Acircumflextilde", "Acute", "Acutesmall", 
    "Acyrillic", "Adblgrave", "Adieresis", "Adieresiscyrillic", 
    "Adieresismacron", "Adieresissmall", "Adotbelow", "Adotmacron", 
    "Agrave", "Agravesmall", "Ahookabove", "Aiecyrillic", 
    "Ainvertedbreve", "Alpha", "Alphatonos", "Amacron", "Amonospace", 
    "Aogonek", "Aring", "Aringacute", "Aringbelow", "Aringsmall", 
    "Asmall", "Atilde", "Atildesmall", "Aybarmenian", "B", "Bcircle", 
    "Bdotaccent", "Bdotbelow", "Becyrillic", "Benarmenian", "Beta", 
    "Bhook", "Blinebelow", "Bmonospace", "Brevesmall", "Bsmall", 
    "Btopbar", "C", "Caarmenian", "Cacute", "Caron", "Caronsmall", 
    "Ccaron", "Ccedilla", "Ccedillaacute", "Ccedillasmall", "Ccircle", 
    "Ccircumflex", "Cdot", "Cdotaccent", "Cedillasmall", "Chaarmenian", 
    "Cheabkhasiancyrillic", "Checyrillic", 
    "Chedescenderabkhasiancyrillic", "Chedescendercyrillic", 
    "Chedieresiscyrillic", "Cheharmenian", "Chekhakassiancyrillic", 
    "Cheverticalstrokecyrillic", "Chi", "Chook", "Circumflexsmall", 
    "Cmonospace", "Coarmenian", "Csmall", "D", "DZ", "DZcaron", 
    "Daarmenian", "Dafrican", "Dcaron", "Dcedilla", "Dcircle", 
    "Dcircumflexbelow", "Dcroat", "Ddotaccent", "Ddotbelow", 
    "Decyrillic", "Deicoptic", "Delta", "Deltagreek", "Dhook", 
    "Dieresis", "DieresisAcute", "DieresisGrave", "Dieresissmall", 
    "Digammagreek", "Djecyrillic", "Dlinebelow", "Dmonospace", 
    "Dotaccentsmall", "Dslash", "Dsmall", "Dtopbar", "Dz", "Dzcaron", 
    "Dzeabkhasiancyrillic", "Dzecyrillic", "Dzhecyrillic", "E", 
    "Eacute", "Eacutesmall", "Ebreve", "Ecaron", "Ecedillabreve", 
    "Echarmenian", "Ecircle", "Ecircumflex", "Ecircumflexacute", 
    "Ecircumflexbelow", "Ecircumflexdotbelow", "Ecircumflexgrave", 
    "Ecircumflexhookabove", "Ecircumflexsmall", "Ecircumflextilde", 
    "Ecyrillic", "Edblgrave", "Edieresis", "Edieresissmall", "Edot", 
    "Edotaccent", "Edotbelow", "Efcyrillic", "Egrave", "Egravesmall", 
    "Eharmenian", "Ehookabove", "Eightroman", "Einvertedbreve", 
    "Eiotifiedcyrillic", "Elcyrillic", "Elevenroman", "Emacron", 
    "Emacronacute", "Emacrongrave", "Emcyrillic", "Emonospace", 
    "Encyrillic", "Endescendercyrillic", "Eng", "Enghecyrillic", 
    "Enhookcyrillic", "Eogonek", "Eopen", "Epsilon", "Epsilontonos", 
    "Ercyrillic", "Ereversed", "Ereversedcyrillic", "Escyrillic", 
    "Esdescendercyrillic", "Esh", "Esmall", "Eta", "Etarmenian", 
    "Etatonos", "Eth", "Ethsmall", "Etilde", "Etildebelow", "Euro", 
    "Ezh", "Ezhcaron", "Ezhreversed", "F", "Fcircle", "Fdotaccent", 
    "Feharmenian", "Feicoptic", "Fhook", "Fitacyrillic", "Fiveroman", 
    "Fmonospace", "Fourroman", "Fsmall", "G", "GBsquare", "Gacute", 
    "Gamma", "Gammaafrican", "Gangiacoptic", "Gbreve", "Gcaron", 
    "Gcedilla", "Gcircle", "Gcircumflex", "Gcommaaccent", "Gdot", 
    "Gdotaccent", "Gecyrillic", "Ghadarmenian", 
    "Ghemiddlehookcyrillic", "Ghestrokecyrillic", "Gheupturncyrillic", 
    "Ghook", "Gimarmenian", "Gjecyrillic", "Gmacron", "Gmonospace", 
    "Grave", "Gravesmall", "Gsmall", "Gsmallhook", "Gstroke", "H", 
    "H18533", "H18543", "H18551", "H22073", "HPsquare", 
    "Haabkhasiancyrillic", "Hadescendercyrillic", "Hardsigncyrillic", 
    "Hbar", "Hbrevebelow", "Hcedilla", "Hcircle", "Hcircumflex", 
    "Hdieresis", "Hdotaccent", "Hdotbelow", "Hmonospace", "Hoarmenian", 
    "Horicoptic", "Hsmall", "Hungarumlaut", "Hungarumlautsmall", 
    "Hzsquare", "I", "IAcyrillic", "IJ", "IUcyrillic", "Iacute", 
    "Iacutesmall", "Ibreve", "Icaron", "Icircle", "Icircumflex", 
    "Icircumflexsmall", "Icyrillic", "Idblgrave", "Idieresis", 
    "Idieresisacute", "Idieresiscyrillic", "Idieresissmall", "Idot", 
    "Idotaccent", "Idotbelow", "Iebrevecyrillic", "Iecyrillic", 
    "Ifraktur", "Igrave", "Igravesmall", "Ihookabove", "Iicyrillic", 
    "Iinvertedbreve", "Iishortcyrillic", "Imacron", "Imacroncyrillic", 
    "Imonospace", "Iniarmenian", "Iocyrillic", "Iogonek", "Iota", 
    "Iotaafrican", "Iotadieresis", "Iotatonos", "Ismall", "Istroke", 
    "Itilde", "Itildebelow", "Izhitsacyrillic", 
    "Izhitsadblgravecyrillic", "J", "Jaarmenian", "Jcircle", 
    "Jcircumflex", "Jecyrillic", "Jheharmenian", "Jmonospace", 
    "Jsmall", "K", "KBsquare", "KKsquare", "Kabashkircyrillic", 
    "Kacute", "Kacyrillic", "Kadescendercyrillic", "Kahookcyrillic", 
    "Kappa", "Kastrokecyrillic", "Kaverticalstrokecyrillic", "Kcaron", 
    "Kcedilla", "Kcircle", "Kcommaaccent", "Kdotbelow", "Keharmenian", 
    "Kenarmenian", "Khacyrillic", "Kheicoptic", "Khook", "Kjecyrillic", 
    "Klinebelow", "Kmonospace", "Koppacyrillic", "Koppagreek", 
    "Ksicyrillic", "Ksmall", "L", "LJ", "LL", "Lacute", "Lambda", 
    "Lcaron", "Lcedilla", "Lcircle", "Lcircumflexbelow", 
    "Lcommaaccent", "Ldot", "Ldotaccent", "Ldotbelow", 
    "Ldotbelowmacron", "Liwnarmenian", "Lj", "Ljecyrillic", 
    "Llinebelow", "Lmonospace", "Lslash", "Lslashsmall", "Lsmall", "M", 
    "MBsquare", "Macron", "Macronsmall", "Macute", "Mcircle", 
    "Mdotaccent", "Mdotbelow", "Menarmenian", "Mmonospace", "Msmall", 
    "Mturned", "Mu", "N", "NJ", "Nacute", "Ncaron", "Ncedilla", 
    "Ncircle", "Ncircumflexbelow", "Ncommaaccent", "Ndotaccent", 
    "Ndotbelow", "Nhookleft", "Nineroman", "Nj", "Njecyrillic", 
    "Nlinebelow", "Nmonospace", "Nowarmenian", "Nsmall", "Ntilde", 
    "Ntildesmall", "Nu", "O", "OE", "OEsmall", "Oacute", "Oacutesmall", 
    "Obarredcyrillic", "Obarreddieresiscyrillic", "Obreve", "Ocaron", 
    "Ocenteredtilde", "Ocircle", "Ocircumflex", "Ocircumflexacute", 
    "Ocircumflexdotbelow", "Ocircumflexgrave", "Ocircumflexhookabove", 
    "Ocircumflexsmall", "Ocircumflextilde", "Ocyrillic", "Odblacute", 
    "Odblgrave", "Odieresis", "Odieresiscyrillic", "Odieresissmall", 
    "Odotbelow", "Ogoneksmall", "Ograve", "Ogravesmall", "Oharmenian", 
    "Ohm", "Ohookabove", "Ohorn", "Ohornacute", "Ohorndotbelow", 
    "Ohorngrave", "Ohornhookabove", "Ohorntilde", "Ohungarumlaut", 
    "Oi", "Oinvertedbreve", "Omacron", "Omacronacute", "Omacrongrave", 
    "Omega", "Omegacyrillic", "Omegagreek", "Omegaroundcyrillic", 
    "Omegatitlocyrillic", "Omegatonos", "Omicron", "Omicrontonos", 
    "Omonospace", "Oneroman", "Oogonek", "Oogonekmacron", "Oopen", 
    "Oslash", "Oslashacute", "Oslashsmall", "Osmall", "Ostrokeacute", 
    "Otcyrillic", "Otilde", "Otildeacute", "Otildedieresis", 
    "Otildesmall", "P", "Pacute", "Pcircle", "Pdotaccent", 
    "Pecyrillic", "Peharmenian", "Pemiddlehookcyrillic", "Phi", 
    "Phook", "Pi", "Piwrarmenian", "Pmonospace", "Psi", "Psicyrillic", 
    "Psmall", "Q", "Qcircle", "Qmonospace", "Qsmall", "R", 
    "Raarmenian", "Racute", "Rcaron", "Rcedilla", "Rcircle", 
    "Rcommaaccent", "Rdblgrave", "Rdotaccent", "Rdotbelow", 
    "Rdotbelowmacron", "Reharmenian", "Rfraktur", "Rho", "Ringsmall", 
    "Rinvertedbreve", "Rlinebelow", "Rmonospace", "Rsmall", 
    "Rsmallinverted", "Rsmallinvertedsuperior", "S", "SF010000", 
    "SF020000", "SF030000", "SF040000", "SF050000", "SF060000", 
    "SF070000", "SF080000", "SF090000", "SF100000", "SF110000", 
    "SF190000", "SF200000", "SF210000", "SF220000", "SF230000", 
    "SF240000", "SF250000", "SF260000", "SF270000", "SF280000", 
    "SF360000", "SF370000", "SF380000", "SF390000", "SF400000", 
    "SF410000", "SF420000", "SF430000", "SF440000", "SF450000", 
    "SF460000", "SF470000", "SF480000", "SF490000", "SF500000", 
    "SF510000", "SF520000", "SF530000", "SF540000", "Sacute", 
    "Sacutedotaccent", "Sampigreek", "Scaron", "Scarondotaccent", 
    "Scaronsmall", "Scedilla", "Schwa", "Schwacyrillic", 
    "Schwadieresiscyrillic", "Scircle", "Scircumflex", "Scommaaccent", 
    "Sdotaccent", "Sdotbelow", "Sdotbelowdotaccent", "Seharmenian", 
    "Sevenroman", "Shaarmenian", "Shacyrillic", "Shchacyrillic", 
    "Sheicoptic", "Shhacyrillic", "Shimacoptic", "Sigma", "Sixroman", 
    "Smonospace", "Softsigncyrillic", "Ssmall", "Stigmagreek", "T", 
    "Tau", "Tbar", "Tcaron", "Tcedilla", "Tcircle", "Tcircumflexbelow", 
    "Tcommaaccent", "Tdotaccent", "Tdotbelow", "Tecyrillic", 
    "Tedescendercyrillic", "Tenroman", "Tetsecyrillic", "Theta", 
    "Thook", "Thorn", "Thornsmall", "Threeroman", "Tildesmall", 
    "Tiwnarmenian", "Tlinebelow", "Tmonospace", "Toarmenian", 
    "Tonefive", "Tonesix", "Tonetwo", "Tretroflexhook", "Tsecyrillic", 
    "Tshecyrillic", "Tsmall", "Twelveroman", "Tworoman", "U", "Uacute", 
    "Uacutesmall", "Ubreve", "Ucaron", "Ucircle", "Ucircumflex", 
    "Ucircumflexbelow", "Ucircumflexsmall", "Ucyrillic", "Udblacute", 
    "Udblgrave", "Udieresis", "Udieresisacute", "Udieresisbelow", 
    "Udieresiscaron", "Udieresiscyrillic", "Udieresisgrave", 
    "Udieresismacron", "Udieresissmall", "Udotbelow", "Ugrave", 
    "Ugravesmall", "Uhookabove", "Uhorn", "Uhornacute", 
    "Uhorndotbelow", "Uhorngrave", "Uhornhookabove", "Uhorntilde", 
    "Uhungarumlaut", "Uhungarumlautcyrillic", "Uinvertedbreve", 
    "Ukcyrillic", "Umacron", "Umacroncyrillic", "Umacrondieresis", 
    "Umonospace", "Uogonek", "Upsilon", "Upsilon1", 
    "Upsilonacutehooksymbolgreek", "Upsilonafrican", "Upsilondieresis", 
    "Upsilondieresishooksymbolgreek", "Upsilonhooksymbol", 
    "Upsilontonos", "Uring", "Ushortcyrillic", "Usmall", 
    "Ustraightcyrillic", "Ustraightstrokecyrillic", "Utilde", 
    "Utildeacute", "Utildebelow", "V", "Vcircle", "Vdotbelow", 
    "Vecyrillic", "Vewarmenian", "Vhook", "Vmonospace", "Voarmenian", 
    "Vsmall", "Vtilde", "W", "Wacute", "Wcircle", "Wcircumflex", 
    "Wdieresis", "Wdotaccent", "Wdotbelow", "Wgrave", "Wmonospace", 
    "Wsmall", "X", "Xcircle", "Xdieresis", "Xdotaccent", "Xeharmenian", 
    "Xi", "Xmonospace", "Xsmall", "Y", "Yacute", "Yacutesmall", 
    "Yatcyrillic", "Ycircle", "Ycircumflex", "Ydieresis", 
    "Ydieresissmall", "Ydotaccent", "Ydotbelow", "Yericyrillic", 
    "Yerudieresiscyrillic", "Ygrave", "Yhook", "Yhookabove", 
    "Yiarmenian", "Yicyrillic", "Yiwnarmenian", "Ymonospace", "Ysmall", 
    "Ytilde", "Yusbigcyrillic", "Yusbigiotifiedcyrillic", 
    "Yuslittlecyrillic", "Yuslittleiotifiedcyrillic", "Z", 
    "Zaarmenian", "Zacute", "Zcaron", "Zcaronsmall", "Zcircle", 
    "Zcircumflex", "Zdot", "Zdotaccent", "Zdotbelow", "Zecyrillic", 
    "Zedescendercyrillic", "Zedieresiscyrillic", "Zeta", "Zhearmenian", 
    "Zhebrevecyrillic", "Zhecyrillic", "Zhedescendercyrillic", 
    "Zhedieresiscyrillic", "Zlinebelow", "Zmonospace", "Zsmall", 
    "Zstroke", "a", "aabengali", "aacute", "aadeva", "aagujarati", 
    "aagurmukhi", "aamatragurmukhi", "aarusquare", 
    "aavowelsignbengali", "aavowelsigndeva", "aavowelsigngujarati", 
    "abbreviationmarkarmenian", "abbreviationsigndeva", "abengali", 
    "abopomofo", "abreve", "abreveacute", "abrevecyrillic", 
    "abrevedotbelow", "abrevegrave", "abrevehookabove", "abrevetilde", 
    "acaron", "acircle", "acircumflex", "acircumflexacute", 
    "acircumflexdotbelow", "acircumflexgrave", "acircumflexhookabove", 
    "acircumflextilde", "acute", "acutebelowcmb", "acutecmb", 
    "acutecomb", "acutedeva", "acutelowmod", "acutetonecmb", 
    "acyrillic", "adblgrave", "addakgurmukhi", "adeva", "adieresis", 
    "adieresiscyrillic", "adieresismacron", "adotbelow", "adotmacron", 
    "ae", "aeacute", "aekorean", "aemacron", "afii00208", "afii08941", 
    "afii10017", "afii10018", "afii10019", "afii10020", "afii10021", 
    "afii10022", "afii10023", "afii10024", "afii10025", "afii10026", 
    "afii10027", "afii10028", "afii10029", "afii10030", "afii10031", 
    "afii10032", "afii10033", "afii10034", "afii10035", "afii10036", 
    "afii10037", "afii10038", "afii10039", "afii10040", "afii10041", 
    "afii10042", "afii10043", "afii10044", "afii10045", "afii10046", 
    "afii10047", "afii10048", "afii10049", "afii10050", "afii10051", 
    "afii10052", "afii10053", "afii10054", "afii10055", "afii10056", 
    "afii10057", "afii10058", "afii10059", "afii10060", "afii10061", 
    "afii10062", "afii10063", "afii10064", "afii10065", "afii10066", 
    "afii10067", "afii10068", "afii10069", "afii10070", "afii10071", 
    "afii10072", "afii10073", "afii10074", "afii10075", "afii10076", 
    "afii10077", "afii10078", "afii10079", "afii10080", "afii10081", 
    "afii10082", "afii10083", "afii10084", "afii10085", "afii10086", 
    "afii10087", "afii10088", "afii10089", "afii10090", "afii10091", 
    "afii10092", "afii10093", "afii10094", "afii10095", "afii10096", 
    "afii10097", "afii10098", "afii10099", "afii10100", "afii10101", 
    "afii10102", "afii10103", "afii10104", "afii10105", "afii10106", 
    "afii10107", "afii10108", "afii10109", "afii10110", "afii10145", 
    "afii10146", "afii10147", "afii10148", "afii10192", "afii10193", 
    "afii10194", "afii10195", "afii10196", "afii10831", "afii10832", 
    "afii10846", "afii299", "afii300", "afii301", "afii57381", 
    "afii57388", "afii57392", "afii57393", "afii57394", "afii57395", 
    "afii57396", "afii57397", "afii57398", "afii57399", "afii57400", 
    "afii57401", "afii57403", "afii57407", "afii57409", "afii57410", 
    "afii57411", "afii57412", "afii57413", "afii57414", "afii57415", 
    "afii57416", "afii57417", "afii57418", "afii57419", "afii57420", 
    "afii57421", "afii57422", "afii57423", "afii57424", "afii57425", 
    "afii57426", "afii57427", "afii57428", "afii57429", "afii57430", 
    "afii57431", "afii57432", "afii57433", "afii57434", "afii57440", 
    "afii57441", "afii57442", "afii57443", "afii57444", "afii57445", 
    "afii57446", "afii57448", "afii57449", "afii57450", "afii57451", 
    "afii57452", "afii57453", "afii57454", "afii57455", "afii57456", 
    "afii57457", "afii57458", "afii57470", "afii57505", "afii57506", 
    "afii57507", "afii57508", "afii57509", "afii57511", "afii57512", 
    "afii57513", "afii57514", "afii57519", "afii57534", "afii57636", 
    "afii57645", "afii57658", "afii57664", "afii57665", "afii57666", 
    "afii57667", "afii57668", "afii57669", "afii57670", "afii57671", 
    "afii57672", "afii57673", "afii57674", "afii57675", "afii57676", 
    "afii57677", "afii57678", "afii57679", "afii57680", "afii57681", 
    "afii57682", "afii57683", "afii57684", "afii57685", "afii57686", 
    "afii57687", "afii57688", "afii57689", "afii57690", "afii57694", 
    "afii57695", "afii57700", "afii57705", "afii57716", "afii57717", 
    "afii57718", "afii57723", "afii57793", "afii57794", "afii57795", 
    "afii57796", "afii57797", "afii57798", "afii57799", "afii57800", 
    "afii57801", "afii57802", "afii57803", "afii57804", "afii57806", 
    "afii57807", "afii57839", "afii57841", "afii57842", "afii57929", 
    "afii61248", "afii61289", "afii61352", "afii61573", "afii61574", 
    "afii61575", "afii61664", "afii63167", "afii64937", "agrave", 
    "agujarati", "agurmukhi", "ahiragana", "ahookabove", "aibengali", 
    "aibopomofo", "aideva", "aiecyrillic", "aigujarati", "aigurmukhi", 
    "aimatragurmukhi", "ainarabic", "ainfinalarabic", 
    "aininitialarabic", "ainmedialarabic", "ainvertedbreve", 
    "aivowelsignbengali", "aivowelsigndeva", "aivowelsigngujarati", 
    "akatakana", "akatakanahalfwidth", "akorean", "alef", "alefarabic", 
    "alefdageshhebrew", "aleffinalarabic", "alefhamzaabovearabic", 
    "alefhamzaabovefinalarabic", "alefhamzabelowarabic", 
    "alefhamzabelowfinalarabic", "alefhebrew", "aleflamedhebrew", 
    "alefmaddaabovearabic", "alefmaddaabovefinalarabic", 
    "alefmaksuraarabic", "alefmaksurafinalarabic", 
    "alefmaksurainitialarabic", "alefmaksuramedialarabic", 
    "alefpatahhebrew", "alefqamatshebrew", "aleph", "allequal", 
    "alpha", "alphatonos", "amacron", "amonospace", "ampersand", 
    "ampersandmonospace", "ampersandsmall", "amsquare", "anbopomofo", 
    "angbopomofo", "angkhankhuthai", "angle", "anglebracketleft", 
    "anglebracketleftvertical", "anglebracketright", 
    "anglebracketrightvertical", "angleleft", "angleright", "angstrom", 
    "anoteleia", "anudattadeva", "anusvarabengali", "anusvaradeva", 
    "anusvaragujarati", "aogonek", "apaatosquare", "aparen", 
    "apostrophearmenian", "apostrophemod", "apple", "approaches", 
    "approxequal", "approxequalorimage", "approximatelyequal", 
    "araeaekorean", "araeakorean", "arc", "arighthalfring", "aring", 
    "aringacute", "aringbelow", "arrowboth", "arrowdashdown", 
    "arrowdashleft", "arrowdashright", "arrowdashup", "arrowdblboth", 
    "arrowdbldown", "arrowdblleft", "arrowdblright", "arrowdblup", 
    "arrowdown", "arrowdownleft", "arrowdownright", "arrowdownwhite", 
    "arrowheaddownmod", "arrowheadleftmod", "arrowheadrightmod", 
    "arrowheadupmod", "arrowhorizex", "arrowleft", "arrowleftdbl", 
    "arrowleftdblstroke", "arrowleftoverright", "arrowleftwhite", 
    "arrowright", "arrowrightdblstroke", "arrowrightheavy", 
    "arrowrightoverleft", "arrowrightwhite", "arrowtableft", 
    "arrowtabright", "arrowup", "arrowupdn", "arrowupdnbse", 
    "arrowupdownbase", "arrowupleft", "arrowupleftofdown", 
    "arrowupright", "arrowupwhite", "arrowvertex", "asciicircum", 
    "asciicircummonospace", "asciitilde", "asciitildemonospace", 
    "ascript", "ascriptturned", "asmallhiragana", "asmallkatakana", 
    "asmallkatakanahalfwidth", "asterisk", "asteriskaltonearabic", 
    "asteriskarabic", "asteriskmath", "asteriskmonospace", 
    "asterisksmall", "asterism", "asuperior", "asymptoticallyequal", 
    "at", "atilde", "atmonospace", "atsmall", "aturned", "aubengali", 
    "aubopomofo", "audeva", "augujarati", "augurmukhi", 
    "aulengthmarkbengali", "aumatragurmukhi", "auvowelsignbengali", 
    "auvowelsigndeva", "auvowelsigngujarati", "avagrahadeva", 
    "aybarmenian", "ayin", "ayinaltonehebrew", "ayinhebrew", "b", 
    "babengali", "backslash", "backslashmonospace", "badeva", 
    "bagujarati", "bagurmukhi", "bahiragana", "bahtthai", "bakatakana", 
    "bar", "barmonospace", "bbopomofo", "bcircle", "bdotaccent", 
    "bdotbelow", "beamedsixteenthnotes", "because", "becyrillic", 
    "beharabic", "behfinalarabic", "behinitialarabic", "behiragana", 
    "behmedialarabic", "behmeeminitialarabic", "behmeemisolatedarabic", 
    "behnoonfinalarabic", "bekatakana", "benarmenian", "bet", "beta", 
    "betasymbolgreek", "betdagesh", "betdageshhebrew", "bethebrew", 
    "betrafehebrew", "bhabengali", "bhadeva", "bhagujarati", 
    "bhagurmukhi", "bhook", "bihiragana", "bikatakana", 
    "bilabialclick", "bindigurmukhi", "birusquare", "blackcircle", 
    "blackdiamond", "blackdownpointingtriangle", 
    "blackleftpointingpointer", "blackleftpointingtriangle", 
    "blacklenticularbracketleft", "blacklenticularbracketleftvertical", 
    "blacklenticularbracketright", 
    "blacklenticularbracketrightvertical", "blacklowerlefttriangle", 
    "blacklowerrighttriangle", "blackrectangle", 
    "blackrightpointingpointer", "blackrightpointingtriangle", 
    "blacksmallsquare", "blacksmilingface", "blacksquare", "blackstar", 
    "blackupperlefttriangle", "blackupperrighttriangle", 
    "blackuppointingsmalltriangle", "blackuppointingtriangle", "blank", 
    "blinebelow", "block", "bmonospace", "bobaimaithai", "bohiragana", 
    "bokatakana", "bparen", "bqsquare", "braceex", "braceleft", 
    "braceleftbt", "braceleftmid", "braceleftmonospace", 
    "braceleftsmall", "bracelefttp", "braceleftvertical", "braceright", 
    "bracerightbt", "bracerightmid", "bracerightmonospace", 
    "bracerightsmall", "bracerighttp", "bracerightvertical", 
    "bracketleft", "bracketleftbt", "bracketleftex", 
    "bracketleftmonospace", "bracketlefttp", "bracketright", 
    "bracketrightbt", "bracketrightex", "bracketrightmonospace", 
    "bracketrighttp", "breve", "brevebelowcmb", "brevecmb", 
    "breveinvertedbelowcmb", "breveinvertedcmb", 
    "breveinverteddoublecmb", "bridgebelowcmb", 
    "bridgeinvertedbelowcmb", "brokenbar", "bstroke", "bsuperior", 
    "btopbar", "buhiragana", "bukatakana", "bullet", "bulletinverse", 
    "bulletoperator", "bullseye", "c", "caarmenian", "cabengali", 
    "cacute", "cadeva", "cagujarati", "cagurmukhi", "calsquare", 
    "candrabindubengali", "candrabinducmb", "candrabindudeva", 
    "candrabindugujarati", "capslock", "careof", "caron", 
    "caronbelowcmb", "caroncmb", "carriagereturn", "cbopomofo", 
    "ccaron", "ccedilla", "ccedillaacute", "ccircle", "ccircumflex", 
    "ccurl", "cdot", "cdotaccent", "cdsquare", "cedilla", "cedillacmb", 
    "cent", "centigrade", "centinferior", "centmonospace", 
    "centoldstyle", "centsuperior", "chaarmenian", "chabengali", 
    "chadeva", "chagujarati", "chagurmukhi", "chbopomofo", 
    "cheabkhasiancyrillic", "checkmark", "checyrillic", 
    "chedescenderabkhasiancyrillic", "chedescendercyrillic", 
    "chedieresiscyrillic", "cheharmenian", "chekhakassiancyrillic", 
    "cheverticalstrokecyrillic", "chi", "chieuchacirclekorean", 
    "chieuchaparenkorean", "chieuchcirclekorean", "chieuchkorean", 
    "chieuchparenkorean", "chochangthai", "chochanthai", 
    "chochingthai", "chochoethai", "chook", "cieucacirclekorean", 
    "cieucaparenkorean", "cieuccirclekorean", "cieuckorean", 
    "cieucparenkorean", "cieucuparenkorean", "circle", 
    "circlemultiply", "circleot", "circleplus", "circlepostalmark", 
    "circlewithlefthalfblack", "circlewithrighthalfblack", 
    "circumflex", "circumflexbelowcmb", "circumflexcmb", "clear", 
    "clickalveolar", "clickdental", "clicklateral", "clickretroflex", 
    "club", "clubsuitblack", "clubsuitwhite", "cmcubedsquare", 
    "cmonospace", "cmsquaredsquare", "coarmenian", "colon", 
    "colonmonetary", "colonmonospace", "colonsign", "colonsmall", 
    "colontriangularhalfmod", "colontriangularmod", "comma", 
    "commaabovecmb", "commaaboverightcmb", "commaaccent", 
    "commaarabic", "commaarmenian", "commainferior", "commamonospace", 
    "commareversedabovecmb", "commareversedmod", "commasmall", 
    "commasuperior", "commaturnedabovecmb", "commaturnedmod", 
    "compass", "congruent", "contourintegral", "control", "controlACK", 
    "controlBEL", "controlBS", "controlCAN", "controlCR", "controlDC1", 
    "controlDC2", "controlDC3", "controlDC4", "controlDEL", 
    "controlDLE", "controlEM", "controlENQ", "controlEOT", 
    "controlESC", "controlETB", "controlETX", "controlFF", "controlFS", 
    "controlGS", "controlHT", "controlLF", "controlNAK", "controlRS", 
    "controlSI", "controlSO", "controlSOT", "controlSTX", "controlSUB", 
    "controlSYN", "controlUS", "controlVT", "copyright", 
    "copyrightsans", "copyrightserif", "cornerbracketleft", 
    "cornerbracketlefthalfwidth", "cornerbracketleftvertical", 
    "cornerbracketright", "cornerbracketrighthalfwidth", 
    "cornerbracketrightvertical", "corporationsquare", "cosquare", 
    "coverkgsquare", "cparen", "cruzeiro", "cstretched", "curlyand", 
    "curlyor", "currency", "cyrBreve", "cyrFlex", "cyrbreve", 
    "cyrflex", "d", "daarmenian", "dabengali", "dadarabic", "dadeva", 
    "dadfinalarabic", "dadinitialarabic", "dadmedialarabic", "dagesh", 
    "dageshhebrew", "dagger", "daggerdbl", "dagujarati", "dagurmukhi", 
    "dahiragana", "dakatakana", "dalarabic", "dalet", "daletdagesh", 
    "daletdageshhebrew", "dalethebrew", "dalfinalarabic", 
    "dammaarabic", "dammalowarabic", "dammatanaltonearabic", 
    "dammatanarabic", "danda", "dargahebrew", "dargalefthebrew", 
    "dasiapneumatacyrilliccmb", "dblGrave", "dblanglebracketleft", 
    "dblanglebracketleftvertical", "dblanglebracketright", 
    "dblanglebracketrightvertical", "dblarchinvertedbelowcmb", 
    "dblarrowleft", "dblarrowright", "dbldanda", "dblgrave", 
    "dblgravecmb", "dblintegral", "dbllowline", "dbllowlinecmb", 
    "dbloverlinecmb", "dblprimemod", "dblverticalbar", 
    "dblverticallineabovecmb", "dbopomofo", "dbsquare", "dcaron", 
    "dcedilla", "dcircle", "dcircumflexbelow", "dcroat", "ddabengali", 
    "ddadeva", "ddagujarati", "ddagurmukhi", "ddalarabic", 
    "ddalfinalarabic", "dddhadeva", "ddhabengali", "ddhadeva", 
    "ddhagujarati", "ddhagurmukhi", "ddotaccent", "ddotbelow", 
    "decimalseparatorarabic", "decimalseparatorpersian", "decyrillic", 
    "degree", "dehihebrew", "dehiragana", "deicoptic", "dekatakana", 
    "deleteleft", "deleteright", "delta", "deltaturned", 
    "denominatorminusonenumeratorbengali", "dezh", "dhabengali", 
    "dhadeva", "dhagujarati", "dhagurmukhi", "dhook", "dialytikatonos", 
    "dialytikatonoscmb", "diamond", "diamondsuitwhite", "dieresis", 
    "dieresisacute", "dieresisbelowcmb", "dieresiscmb", 
    "dieresisgrave", "dieresistonos", "dihiragana", "dikatakana", 
    "dittomark", "divide", "divides", "divisionslash", "djecyrillic", 
    "dkshade", "dlinebelow", "dlsquare", "dmacron", "dmonospace", 
    "dnblock", "dochadathai", "dodekthai", "dohiragana", "dokatakana", 
    "dollar", "dollarinferior", "dollarmonospace", "dollaroldstyle", 
    "dollarsmall", "dollarsuperior", "dong", "dorusquare", "dotaccent", 
    "dotaccentcmb", "dotbelowcmb", "dotbelowcomb", "dotkatakana", 
    "dotlessi", "dotlessj", "dotlessjstrokehook", "dotmath", 
    "dottedcircle", "doubleyodpatah", "doubleyodpatahhebrew", 
    "downtackbelowcmb", "downtackmod", "dparen", "dsuperior", "dtail", 
    "dtopbar", "duhiragana", "dukatakana", "dz", "dzaltone", "dzcaron", 
    "dzcurl", "dzeabkhasiancyrillic", "dzecyrillic", "dzhecyrillic", 
    "e", "eacute", "earth", "ebengali", "ebopomofo", "ebreve", 
    "ecandradeva", "ecandragujarati", "ecandravowelsigndeva", 
    "ecandravowelsigngujarati", "ecaron", "ecedillabreve", 
    "echarmenian", "echyiwnarmenian", "ecircle", "ecircumflex", 
    "ecircumflexacute", "ecircumflexbelow", "ecircumflexdotbelow", 
    "ecircumflexgrave", "ecircumflexhookabove", "ecircumflextilde", 
    "ecyrillic", "edblgrave", "edeva", "edieresis", "edot", 
    "edotaccent", "edotbelow", "eegurmukhi", "eematragurmukhi", 
    "efcyrillic", "egrave", "egujarati", "eharmenian", "ehbopomofo", 
    "ehiragana", "ehookabove", "eibopomofo", "eight", "eightarabic", 
    "eightbengali", "eightcircle", "eightcircleinversesansserif", 
    "eightdeva", "eighteencircle", "eighteenparen", "eighteenperiod", 
    "eightgujarati", "eightgurmukhi", "eighthackarabic", 
    "eighthangzhou", "eighthnotebeamed", "eightideographicparen", 
    "eightinferior", "eightmonospace", "eightoldstyle", "eightparen", 
    "eightperiod", "eightpersian", "eightroman", "eightsuperior", 
    "eightthai", "einvertedbreve", "eiotifiedcyrillic", "ekatakana", 
    "ekatakanahalfwidth", "ekonkargurmukhi", "ekorean", "elcyrillic", 
    "element", "elevencircle", "elevenparen", "elevenperiod", 
    "elevenroman", "ellipsis", "ellipsisvertical", "emacron", 
    "emacronacute", "emacrongrave", "emcyrillic", "emdash", 
    "emdashvertical", "emonospace", "emphasismarkarmenian", "emptyset", 
    "enbopomofo", "encyrillic", "endash", "endashvertical", 
    "endescendercyrillic", "eng", "engbopomofo", "enghecyrillic", 
    "enhookcyrillic", "enspace", "eogonek", "eokorean", "eopen", 
    "eopenclosed", "eopenreversed", "eopenreversedclosed", 
    "eopenreversedhook", "eparen", "epsilon", "epsilontonos", "equal", 
    "equalmonospace", "equalsmall", "equalsuperior", "equivalence", 
    "erbopomofo", "ercyrillic", "ereversed", "ereversedcyrillic", 
    "escyrillic", "esdescendercyrillic", "esh", "eshcurl", 
    "eshortdeva", "eshortvowelsigndeva", "eshreversedloop", 
    "eshsquatreversed", "esmallhiragana", "esmallkatakana", 
    "esmallkatakanahalfwidth", "estimated", "esuperior", "eta", 
    "etarmenian", "etatonos", "eth", "etilde", "etildebelow", 
    "etnahtafoukhhebrew", "etnahtafoukhlefthebrew", "etnahtahebrew", 
    "etnahtalefthebrew", "eturned", "eukorean", "euro", 
    "evowelsignbengali", "evowelsigndeva", "evowelsigngujarati", 
    "exclam", "exclamarmenian", "exclamdbl", "exclamdown", 
    "exclamdownsmall", "exclammonospace", "exclamsmall", "existential", 
    "ezh", "ezhcaron", "ezhcurl", "ezhreversed", "ezhtail", "f", 
    "fadeva", "fagurmukhi", "fahrenheit", "fathaarabic", 
    "fathalowarabic", "fathatanarabic", "fbopomofo", "fcircle", 
    "fdotaccent", "feharabic", "feharmenian", "fehfinalarabic", 
    "fehinitialarabic", "fehmedialarabic", "feicoptic", "female", "ff", 
    "ffi", "ffl", "fi", "fifteencircle", "fifteenparen", 
    "fifteenperiod", "figuredash", "filledbox", "filledrect", 
    "finalkaf", "finalkafdagesh", "finalkafdageshhebrew", 
    "finalkafhebrew", "finalmem", "finalmemhebrew", "finalnun", 
    "finalnunhebrew", "finalpe", "finalpehebrew", "finaltsadi", 
    "finaltsadihebrew", "firsttonechinese", "fisheye", "fitacyrillic", 
    "five", "fivearabic", "fivebengali", "fivecircle", 
    "fivecircleinversesansserif", "fivedeva", "fiveeighths", 
    "fivegujarati", "fivegurmukhi", "fivehackarabic", "fivehangzhou", 
    "fiveideographicparen", "fiveinferior", "fivemonospace", 
    "fiveoldstyle", "fiveparen", "fiveperiod", "fivepersian", 
    "fiveroman", "fivesuperior", "fivethai", "fl", "florin", 
    "fmonospace", "fmsquare", "fofanthai", "fofathai", "fongmanthai", 
    "forall", "four", "fourarabic", "fourbengali", "fourcircle", 
    "fourcircleinversesansserif", "fourdeva", "fourgujarati", 
    "fourgurmukhi", "fourhackarabic", "fourhangzhou", 
    "fourideographicparen", "fourinferior", "fourmonospace", 
    "fournumeratorbengali", "fouroldstyle", "fourparen", "fourperiod", 
    "fourpersian", "fourroman", "foursuperior", "fourteencircle", 
    "fourteenparen", "fourteenperiod", "fourthai", "fourthtonechinese", 
    "fparen", "fraction", "franc", "g", "gabengali", "gacute", 
    "gadeva", "gafarabic", "gaffinalarabic", "gafinitialarabic", 
    "gafmedialarabic", "gagujarati", "gagurmukhi", "gahiragana", 
    "gakatakana", "gamma", "gammalatinsmall", "gammasuperior", 
    "gangiacoptic", "gbopomofo", "gbreve", "gcaron", "gcedilla", 
    "gcircle", "gcircumflex", "gcommaaccent", "gdot", "gdotaccent", 
    "gecyrillic", "gehiragana", "gekatakana", "geometricallyequal", 
    "gereshaccenthebrew", "gereshhebrew", "gereshmuqdamhebrew", 
    "germandbls", "gershayimaccenthebrew", "gershayimhebrew", 
    "getamark", "ghabengali", "ghadarmenian", "ghadeva", "ghagujarati", 
    "ghagurmukhi", "ghainarabic", "ghainfinalarabic", 
    "ghaininitialarabic", "ghainmedialarabic", "ghemiddlehookcyrillic", 
    "ghestrokecyrillic", "gheupturncyrillic", "ghhadeva", 
    "ghhagurmukhi", "ghook", "ghzsquare", "gihiragana", "gikatakana", 
    "gimarmenian", "gimel", "gimeldagesh", "gimeldageshhebrew", 
    "gimelhebrew", "gjecyrillic", "glottalinvertedstroke", 
    "glottalstop", "glottalstopinverted", "glottalstopmod", 
    "glottalstopreversed", "glottalstopreversedmod", 
    "glottalstopreversedsuperior", "glottalstopstroke", 
    "glottalstopstrokereversed", "gmacron", "gmonospace", "gohiragana", 
    "gokatakana", "gparen", "gpasquare", "gradient", "grave", 
    "gravebelowcmb", "gravecmb", "gravecomb", "gravedeva", 
    "gravelowmod", "gravemonospace", "gravetonecmb", "greater", 
    "greaterequal", "greaterequalorless", "greatermonospace", 
    "greaterorequivalent", "greaterorless", "greateroverequal", 
    "greatersmall", "gscript", "gstroke", "guhiragana", 
    "guillemotleft", "guillemotright", "guilsinglleft", 
    "guilsinglright", "gukatakana", "guramusquare", "gysquare", "h", 
    "haabkhasiancyrillic", "haaltonearabic", "habengali", 
    "hadescendercyrillic", "hadeva", "hagujarati", "hagurmukhi", 
    "haharabic", "hahfinalarabic", "hahinitialarabic", "hahiragana", 
    "hahmedialarabic", "haitusquare", "hakatakana", 
    "hakatakanahalfwidth", "halantgurmukhi", "hamzaarabic", 
    "hamzalowarabic", "hangulfiller", "hardsigncyrillic", 
    "harpoonleftbarbup", "harpoonrightbarbup", "hasquare", 
    "hatafpatah", "hatafpatah16", "hatafpatah23", "hatafpatah2f", 
    "hatafpatahhebrew", "hatafpatahnarrowhebrew", 
    "hatafpatahquarterhebrew", "hatafpatahwidehebrew", "hatafqamats", 
    "hatafqamats1b", "hatafqamats28", "hatafqamats34", 
    "hatafqamatshebrew", "hatafqamatsnarrowhebrew", 
    "hatafqamatsquarterhebrew", "hatafqamatswidehebrew", "hatafsegol", 
    "hatafsegol17", "hatafsegol24", "hatafsegol30", "hatafsegolhebrew", 
    "hatafsegolnarrowhebrew", "hatafsegolquarterhebrew", 
    "hatafsegolwidehebrew", "hbar", "hbopomofo", "hbrevebelow", 
    "hcedilla", "hcircle", "hcircumflex", "hdieresis", "hdotaccent", 
    "hdotbelow", "he", "heart", "heartsuitblack", "heartsuitwhite", 
    "hedagesh", "hedageshhebrew", "hehaltonearabic", "heharabic", 
    "hehebrew", "hehfinalaltonearabic", "hehfinalalttwoarabic", 
    "hehfinalarabic", "hehhamzaabovefinalarabic", 
    "hehhamzaaboveisolatedarabic", "hehinitialaltonearabic", 
    "hehinitialarabic", "hehiragana", "hehmedialaltonearabic", 
    "hehmedialarabic", "heiseierasquare", "hekatakana", 
    "hekatakanahalfwidth", "hekutaarusquare", "henghook", 
    "herutusquare", "het", "hethebrew", "hhook", "hhooksuperior", 
    "hieuhacirclekorean", "hieuhaparenkorean", "hieuhcirclekorean", 
    "hieuhkorean", "hieuhparenkorean", "hihiragana", "hikatakana", 
    "hikatakanahalfwidth", "hiriq", "hiriq14", "hiriq21", "hiriq2d", 
    "hiriqhebrew", "hiriqnarrowhebrew", "hiriqquarterhebrew", 
    "hiriqwidehebrew", "hlinebelow", "hmonospace", "hoarmenian", 
    "hohipthai", "hohiragana", "hokatakana", "hokatakanahalfwidth", 
    "holam", "holam19", "holam26", "holam32", "holamhebrew", 
    "holamnarrowhebrew", "holamquarterhebrew", "holamwidehebrew", 
    "honokhukthai", "hookabovecomb", "hookcmb", 
    "hookpalatalizedbelowcmb", "hookretroflexbelowcmb", "hoonsquare", 
    "horicoptic", "horizontalbar", "horncmb", "hotsprings", "house", 
    "hparen", "hsuperior", "hturned", "huhiragana", "huiitosquare", 
    "hukatakana", "hukatakanahalfwidth", "hungarumlaut", 
    "hungarumlautcmb", "hv", "hyphen", "hypheninferior", 
    "hyphenmonospace", "hyphensmall", "hyphensuperior", "hyphentwo", 
    "i", "iacute", "iacyrillic", "ibengali", "ibopomofo", "ibreve", 
    "icaron", "icircle", "icircumflex", "icyrillic", "idblgrave", 
    "ideographearthcircle", "ideographfirecircle", 
    "ideographicallianceparen", "ideographiccallparen", 
    "ideographiccentrecircle", "ideographicclose", "ideographiccomma", 
    "ideographiccommaleft", "ideographiccongratulationparen", 
    "ideographiccorrectcircle", "ideographicearthparen", 
    "ideographicenterpriseparen", "ideographicexcellentcircle", 
    "ideographicfestivalparen", "ideographicfinancialcircle", 
    "ideographicfinancialparen", "ideographicfireparen", 
    "ideographichaveparen", "ideographichighcircle", 
    "ideographiciterationmark", "ideographiclaborcircle", 
    "ideographiclaborparen", "ideographicleftcircle", 
    "ideographiclowcircle", "ideographicmedicinecircle", 
    "ideographicmetalparen", "ideographicmoonparen", 
    "ideographicnameparen", "ideographicperiod", 
    "ideographicprintcircle", "ideographicreachparen", 
    "ideographicrepresentparen", "ideographicresourceparen", 
    "ideographicrightcircle", "ideographicsecretcircle", 
    "ideographicselfparen", "ideographicsocietyparen", 
    "ideographicspace", "ideographicspecialparen", 
    "ideographicstockparen", "ideographicstudyparen", 
    "ideographicsunparen", "ideographicsuperviseparen", 
    "ideographicwaterparen", "ideographicwoodparen", "ideographiczero", 
    "ideographmetalcircle", "ideographmooncircle", 
    "ideographnamecircle", "ideographsuncircle", 
    "ideographwatercircle", "ideographwoodcircle", "ideva", 
    "idieresis", "idieresisacute", "idieresiscyrillic", "idotbelow", 
    "iebrevecyrillic", "iecyrillic", "ieungacirclekorean", 
    "ieungaparenkorean", "ieungcirclekorean", "ieungkorean", 
    "ieungparenkorean", "igrave", "igujarati", "igurmukhi", 
    "ihiragana", "ihookabove", "iibengali", "iicyrillic", "iideva", 
    "iigujarati", "iigurmukhi", "iimatragurmukhi", "iinvertedbreve", 
    "iishortcyrillic", "iivowelsignbengali", "iivowelsigndeva", 
    "iivowelsigngujarati", "ij", "ikatakana", "ikatakanahalfwidth", 
    "ikorean", "ilde", "iluyhebrew", "imacron", "imacroncyrillic", 
    "imageorapproximatelyequal", "imatragurmukhi", "imonospace", 
    "increment", "infinity", "iniarmenian", "integral", 
    "integralbottom", "integralbt", "integralex", "integraltop", 
    "integraltp", "intersection", "intisquare", "invbullet", 
    "invcircle", "invsmileface", "iocyrillic", "iogonek", "iota", 
    "iotadieresis", "iotadieresistonos", "iotalatin", "iotatonos", 
    "iparen", "irigurmukhi", "ismallhiragana", "ismallkatakana", 
    "ismallkatakanahalfwidth", "issharbengali", "istroke", "isuperior", 
    "iterationhiragana", "iterationkatakana", "itilde", "itildebelow", 
    "iubopomofo", "iucyrillic", "ivowelsignbengali", "ivowelsigndeva", 
    "ivowelsigngujarati", "izhitsacyrillic", "izhitsadblgravecyrillic", 
    "j", "jaarmenian", "jabengali", "jadeva", "jagujarati", 
    "jagurmukhi", "jbopomofo", "jcaron", "jcircle", "jcircumflex", 
    "jcrossedtail", "jdotlessstroke", "jecyrillic", "jeemarabic", 
    "jeemfinalarabic", "jeeminitialarabic", "jeemmedialarabic", 
    "jeharabic", "jehfinalarabic", "jhabengali", "jhadeva", 
    "jhagujarati", "jhagurmukhi", "jheharmenian", "jis", "jmonospace", 
    "jparen", "jsuperior", "k", "kabashkircyrillic", "kabengali", 
    "kacute", "kacyrillic", "kadescendercyrillic", "kadeva", "kaf", 
    "kafarabic", "kafdagesh", "kafdageshhebrew", "kaffinalarabic", 
    "kafhebrew", "kafinitialarabic", "kafmedialarabic", 
    "kafrafehebrew", "kagujarati", "kagurmukhi", "kahiragana", 
    "kahookcyrillic", "kakatakana", "kakatakanahalfwidth", "kappa", 
    "kappasymbolgreek", "kapyeounmieumkorean", "kapyeounphieuphkorean", 
    "kapyeounpieupkorean", "kapyeounssangpieupkorean", "karoriisquare", 
    "kashidaautoarabic", "kashidaautonosidebearingarabic", 
    "kasmallkatakana", "kasquare", "kasraarabic", "kasratanarabic", 
    "kastrokecyrillic", "katahiraprolongmarkhalfwidth", 
    "kaverticalstrokecyrillic", "kbopomofo", "kcalsquare", "kcaron", 
    "kcedilla", "kcircle", "kcommaaccent", "kdotbelow", "keharmenian", 
    "kehiragana", "kekatakana", "kekatakanahalfwidth", "kenarmenian", 
    "kesmallkatakana", "kgreenlandic", "khabengali", "khacyrillic", 
    "khadeva", "khagujarati", "khagurmukhi", "khaharabic", 
    "khahfinalarabic", "khahinitialarabic", "khahmedialarabic", 
    "kheicoptic", "khhadeva", "khhagurmukhi", "khieukhacirclekorean", 
    "khieukhaparenkorean", "khieukhcirclekorean", "khieukhkorean", 
    "khieukhparenkorean", "khokhaithai", "khokhonthai", "khokhuatthai", 
    "khokhwaithai", "khomutthai", "khook", "khorakhangthai", 
    "khzsquare", "kihiragana", "kikatakana", "kikatakanahalfwidth", 
    "kiroguramusquare", "kiromeetorusquare", "kirosquare", 
    "kiyeokacirclekorean", "kiyeokaparenkorean", "kiyeokcirclekorean", 
    "kiyeokkorean", "kiyeokparenkorean", "kiyeoksioskorean", 
    "kjecyrillic", "klinebelow", "klsquare", "kmcubedsquare", 
    "kmonospace", "kmsquaredsquare", "kohiragana", "kohmsquare", 
    "kokaithai", "kokatakana", "kokatakanahalfwidth", "kooposquare", 
    "koppacyrillic", "koreanstandardsymbol", "koroniscmb", "kparen", 
    "kpasquare", "ksicyrillic", "ktsquare", "kturned", "kuhiragana", 
    "kukatakana", "kukatakanahalfwidth", "kvsquare", "kwsquare", "l", 
    "labengali", "lacute", "ladeva", "lagujarati", "lagurmukhi", 
    "lakkhangyaothai", "lamaleffinalarabic", 
    "lamalefhamzaabovefinalarabic", "lamalefhamzaaboveisolatedarabic", 
    "lamalefhamzabelowfinalarabic", "lamalefhamzabelowisolatedarabic", 
    "lamalefisolatedarabic", "lamalefmaddaabovefinalarabic", 
    "lamalefmaddaaboveisolatedarabic", "lamarabic", "lambda", 
    "lambdastroke", "lamed", "lameddagesh", "lameddageshhebrew", 
    "lamedhebrew", "lamfinalarabic", "lamhahinitialarabic", 
    "laminitialarabic", "lamjeeminitialarabic", "lamkhahinitialarabic", 
    "lamlamhehisolatedarabic", "lammedialarabic", 
    "lammeemhahinitialarabic", "lammeeminitialarabic", "largecircle", 
    "lbar", "lbelt", "lbopomofo", "lcaron", "lcedilla", "lcircle", 
    "lcircumflexbelow", "lcommaaccent", "ldot", "ldotaccent", 
    "ldotbelow", "ldotbelowmacron", "leftangleabovecmb", 
    "lefttackbelowcmb", "less", "lessequal", "lessequalorgreater", 
    "lessmonospace", "lessorequivalent", "lessorgreater", 
    "lessoverequal", "lesssmall", "lezh", "lfblock", "lhookretroflex", 
    "lira", "liwnarmenian", "lj", "ljecyrillic", "ll", "lladeva", 
    "llagujarati", "llinebelow", "llladeva", "llvocalicbengali", 
    "llvocalicdeva", "llvocalicvowelsignbengali", 
    "llvocalicvowelsigndeva", "lmiddletilde", "lmonospace", "lmsquare", 
    "lochulathai", "logicaland", "logicalnot", "logicalnotreversed", 
    "logicalor", "lolingthai", "longs", "lowlinecenterline", 
    "lowlinecmb", "lowlinedashed", "lozenge", "lparen", "lslash", 
    "lsquare", "lsuperior", "ltshade", "luthai", "lvocalicbengali", 
    "lvocalicdeva", "lvocalicvowelsignbengali", 
    "lvocalicvowelsigndeva", "lxsquare", "m", "mabengali", "macron", 
    "macronbelowcmb", "macroncmb", "macronlowmod", "macronmonospace", 
    "macute", "madeva", "magujarati", "magurmukhi", "mahapakhhebrew", 
    "mahapakhlefthebrew", "mahiragana", "maichattawalowleftthai", 
    "maichattawalowrightthai", "maichattawathai", 
    "maichattawaupperleftthai", "maieklowleftthai", 
    "maieklowrightthai", "maiekthai", "maiekupperleftthai", 
    "maihanakatleftthai", "maihanakatthai", "maitaikhuleftthai", 
    "maitaikhuthai", "maitholowleftthai", "maitholowrightthai", 
    "maithothai", "maithoupperleftthai", "maitrilowleftthai", 
    "maitrilowrightthai", "maitrithai", "maitriupperleftthai", 
    "maiyamokthai", "makatakana", "makatakanahalfwidth", "male", 
    "mansyonsquare", "maqafhebrew", "mars", "masoracirclehebrew", 
    "masquare", "mbopomofo", "mbsquare", "mcircle", "mcubedsquare", 
    "mdotaccent", "mdotbelow", "meemarabic", "meemfinalarabic", 
    "meeminitialarabic", "meemmedialarabic", "meemmeeminitialarabic", 
    "meemmeemisolatedarabic", "meetorusquare", "mehiragana", 
    "meizierasquare", "mekatakana", "mekatakanahalfwidth", "mem", 
    "memdagesh", "memdageshhebrew", "memhebrew", "menarmenian", 
    "merkhahebrew", "merkhakefulahebrew", "merkhakefulalefthebrew", 
    "merkhalefthebrew", "mhook", "mhzsquare", 
    "middledotkatakanahalfwidth", "middot", "mieumacirclekorean", 
    "mieumaparenkorean", "mieumcirclekorean", "mieumkorean", 
    "mieumpansioskorean", "mieumparenkorean", "mieumpieupkorean", 
    "mieumsioskorean", "mihiragana", "mikatakana", 
    "mikatakanahalfwidth", "minus", "minusbelowcmb", "minuscircle", 
    "minusmod", "minusplus", "minute", "miribaarusquare", "mirisquare", 
    "mlonglegturned", "mlsquare", "mmcubedsquare", "mmonospace", 
    "mmsquaredsquare", "mohiragana", "mohmsquare", "mokatakana", 
    "mokatakanahalfwidth", "molsquare", "momathai", "moverssquare", 
    "moverssquaredsquare", "mparen", "mpasquare", "mssquare", 
    "msuperior", "mturned", "mu", "mu1", "muasquare", "muchgreater",
    "muchless", "mufsquare", "mugreek", "mugsquare", "muhiragana", 
    "mukatakana", "mukatakanahalfwidth", "mulsquare", "multiply", 
    "mumsquare", "munahhebrew", "munahlefthebrew", "musicalnote", 
    "musicalnotedbl", "musicflatsign", "musicsharpsign", "mussquare", 
    "muvsquare", "muwsquare", "mvmegasquare", "mvsquare", 
    "mwmegasquare", "mwsquare", "n", "nabengali", "nabla", "nacute", 
    "nadeva", "nagujarati", "nagurmukhi", "nahiragana", "nakatakana", 
    "nakatakanahalfwidth", "napostrophe", "nasquare", "nbopomofo", 
    "nbspace", "ncaron", "ncedilla", "ncircle", "ncircumflexbelow", 
    "ncommaaccent", "ndotaccent", "ndotbelow", "nehiragana", 
    "nekatakana", "nekatakanahalfwidth", "newsheqelsign", "nfsquare", 
    "ngabengali", "ngadeva", "ngagujarati", "ngagurmukhi", 
    "ngonguthai", "nhiragana", "nhookleft", "nhookretroflex", 
    "nieunacirclekorean", "nieunaparenkorean", "nieuncieuckorean", 
    "nieuncirclekorean", "nieunhieuhkorean", "nieunkorean", 
    "nieunpansioskorean", "nieunparenkorean", "nieunsioskorean", 
    "nieuntikeutkorean", "nihiragana", "nikatakana", 
    "nikatakanahalfwidth", "nikhahitleftthai", "nikhahitthai", "nine", 
    "ninearabic", "ninebengali", "ninecircle", 
    "ninecircleinversesansserif", "ninedeva", "ninegujarati", 
    "ninegurmukhi", "ninehackarabic", "ninehangzhou", 
    "nineideographicparen", "nineinferior", "ninemonospace", 
    "nineoldstyle", "nineparen", "nineperiod", "ninepersian", 
    "nineroman", "ninesuperior", "nineteencircle", "nineteenparen", 
    "nineteenperiod", "ninethai", "nj", "njecyrillic", "nkatakana", 
    "nkatakanahalfwidth", "nlegrightlong", "nlinebelow", "nmonospace", 
    "nmsquare", "nnabengali", "nnadeva", "nnagujarati", "nnagurmukhi", 
    "nnnadeva", "nohiragana", "nokatakana", "nokatakanahalfwidth", 
    "nonbreakingspace", "nonenthai", "nonuthai", "noonarabic", 
    "noonfinalarabic", "noonghunnaarabic", "noonghunnafinalarabic", 
    "nooninitialarabic", "noonjeeminitialarabic", 
    "noonjeemisolatedarabic", "noonmedialarabic", 
    "noonmeeminitialarabic", "noonmeemisolatedarabic", 
    "noonnoonfinalarabic", "notcontains", "notelement", "notelementof", 
    "notequal", "notgreater", "notgreaternorequal", 
    "notgreaternorless", "notidentical", "notless", "notlessnorequal", 
    "notparallel", "notprecedes", "notsubset", "notsucceeds", 
    "notsuperset", "nowarmenian", "nparen", "nssquare", "nsuperior", 
    "ntilde", "nu", "nuhiragana", "nukatakana", "nukatakanahalfwidth", 
    "nuktabengali", "nuktadeva", "nuktagujarati", "nuktagurmukhi", 
    "numbersign", "numbersignmonospace", "numbersignsmall", 
    "numeralsigngreek", "numeralsignlowergreek", "numero", "nun", 
    "nundagesh", "nundageshhebrew", "nunhebrew", "nvsquare", 
    "nwsquare", "nyabengali", "nyadeva", "nyagujarati", "nyagurmukhi", 
    "o", "oacute", "oangthai", "obarred", "obarredcyrillic", 
    "obarreddieresiscyrillic", "obengali", "obopomofo", "obreve", 
    "ocandradeva", "ocandragujarati", "ocandravowelsigndeva", 
    "ocandravowelsigngujarati", "ocaron", "ocircle", "ocircumflex", 
    "ocircumflexacute", "ocircumflexdotbelow", "ocircumflexgrave", 
    "ocircumflexhookabove", "ocircumflextilde", "ocyrillic", 
    "odblacute", "odblgrave", "odeva", "odieresis", 
    "odieresiscyrillic", "odotbelow", "oe", "oekorean", "ogonek", 
    "ogonekcmb", "ograve", "ogujarati", "oharmenian", "ohiragana", 
    "ohookabove", "ohorn", "ohornacute", "ohorndotbelow", "ohorngrave", 
    "ohornhookabove", "ohorntilde", "ohungarumlaut", "oi", 
    "oinvertedbreve", "okatakana", "okatakanahalfwidth", "okorean", 
    "olehebrew", "omacron", "omacronacute", "omacrongrave", "omdeva", 
    "omega", "omega1", "omegacyrillic", "omegalatinclosed", 
    "omegaroundcyrillic", "omegatitlocyrillic", "omegatonos", 
    "omgujarati", "omicron", "omicrontonos", "omonospace", "one", 
    "onearabic", "onebengali", "onecircle", 
    "onecircleinversesansserif", "onedeva", "onedotenleader", 
    "oneeighth", "onefitted", "onegujarati", "onegurmukhi", 
    "onehackarabic", "onehalf", "onehangzhou", "oneideographicparen", 
    "oneinferior", "onemonospace", "onenumeratorbengali", 
    "oneoldstyle", "oneparen", "oneperiod", "onepersian", "onequarter", 
    "oneroman", "onesuperior", "onethai", "onethird", "oogonek", 
    "oogonekmacron", "oogurmukhi", "oomatragurmukhi", "oopen", 
    "oparen", "openbullet", "option", "ordfeminine", "ordmasculine", 
    "orthogonal", "oshortdeva", "oshortvowelsigndeva", "oslash", 
    "oslashacute", "osmallhiragana", "osmallkatakana", 
    "osmallkatakanahalfwidth", "ostrokeacute", "osuperior", 
    "otcyrillic", "otilde", "otildeacute", "otildedieresis", 
    "oubopomofo", "overline", "overlinecenterline", "overlinecmb", 
    "overlinedashed", "overlinedblwavy", "overlinewavy", "overscore", 
    "ovowelsignbengali", "ovowelsigndeva", "ovowelsigngujarati", "p", 
    "paampssquare", "paasentosquare", "pabengali", "pacute", "padeva", 
    "pagedown", "pageup", "pagujarati", "pagurmukhi", "pahiragana", 
    "paiyannoithai", "pakatakana", "palatalizationcyrilliccmb", 
    "palochkacyrillic", "pansioskorean", "paragraph", "parallel", 
    "parenleft", "parenleftaltonearabic", "parenleftbt", "parenleftex", 
    "parenleftinferior", "parenleftmonospace", "parenleftsmall", 
    "parenleftsuperior", "parenlefttp", "parenleftvertical", 
    "parenright", "parenrightaltonearabic", "parenrightbt", 
    "parenrightex", "parenrightinferior", "parenrightmonospace", 
    "parenrightsmall", "parenrightsuperior", "parenrighttp", 
    "parenrightvertical", "partialdiff", "paseqhebrew", "pashtahebrew", 
    "pasquare", "patah", "patah11", "patah1d", "patah2a", 
    "patahhebrew", "patahnarrowhebrew", "patahquarterhebrew", 
    "patahwidehebrew", "pazerhebrew", "pbopomofo", "pcircle", 
    "pdotaccent", "pe", "pecyrillic", "pedagesh", "pedageshhebrew", 
    "peezisquare", "pefinaldageshhebrew", "peharabic", "peharmenian", 
    "pehebrew", "pehfinalarabic", "pehinitialarabic", "pehiragana", 
    "pehmedialarabic", "pekatakana", "pemiddlehookcyrillic", 
    "perafehebrew", "percent", "percentarabic", "percentmonospace", 
    "percentsmall", "period", "periodarmenian", "periodcentered", 
    "periodhalfwidth", "periodinferior", "periodmonospace", 
    "periodsmall", "periodsuperior", "perispomenigreekcmb", 
    "perpendicular", "perthousand", "peseta", "pfsquare", "phabengali", 
    "phadeva", "phagujarati", "phagurmukhi", "phi", "phi1", 
    "phieuphacirclekorean", "phieuphaparenkorean", 
    "phieuphcirclekorean", "phieuphkorean", "phieuphparenkorean", 
    "philatin", "phinthuthai", "phisymbolgreek", "phook", 
    "phophanthai", "phophungthai", "phosamphaothai", "pi", 
    "pieupacirclekorean", "pieupaparenkorean", "pieupcieuckorean", 
    "pieupcirclekorean", "pieupkiyeokkorean", "pieupkorean", 
    "pieupparenkorean", "pieupsioskiyeokkorean", "pieupsioskorean", 
    "pieupsiostikeutkorean", "pieupthieuthkorean", "pieuptikeutkorean", 
    "pihiragana", "pikatakana", "pisymbolgreek", "piwrarmenian", 
    "plus", "plusbelowcmb", "pluscircle", "plusminus", "plusmod", 
    "plusmonospace", "plussmall", "plussuperior", "pmonospace", 
    "pmsquare", "pohiragana", "pointingindexdownwhite", 
    "pointingindexleftwhite", "pointingindexrightwhite", 
    "pointingindexupwhite", "pokatakana", "poplathai", "postalmark", 
    "postalmarkface", "pparen", "precedes", "prescription", "primemod", 
    "primereversed", "product", "projective", "prolongedkana", 
    "propellor", "propersubset", "propersuperset", "proportion", 
    "proportional", "psi", "psicyrillic", "psilipneumatacyrilliccmb", 
    "pssquare", "puhiragana", "pukatakana", "pvsquare", "pwsquare", 
    "q", "qadeva", "qadmahebrew", "qafarabic", "qaffinalarabic", 
    "qafinitialarabic", "qafmedialarabic", "qamats", "qamats10", 
    "qamats1a", "qamats1c", "qamats27", "qamats29", "qamats33", 
    "qamatsde", "qamatshebrew", "qamatsnarrowhebrew", 
    "qamatsqatanhebrew", "qamatsqatannarrowhebrew", 
    "qamatsqatanquarterhebrew", "qamatsqatanwidehebrew", 
    "qamatsquarterhebrew", "qamatswidehebrew", "qarneyparahebrew", 
    "qbopomofo", "qcircle", "qhook", "qmonospace", "qof", "qofdagesh", 
    "qofdageshhebrew", "qofhebrew", "qparen", "quarternote", "qubuts", 
    "qubuts18", "qubuts25", "qubuts31", "qubutshebrew", 
    "qubutsnarrowhebrew", "qubutsquarterhebrew", "qubutswidehebrew", 
    "question", "questionarabic", "questionarmenian", "questiondown", 
    "questiondownsmall", "questiongreek", "questionmonospace", 
    "questionsmall", "quotedbl", "quotedblbase", "quotedblleft", 
    "quotedblmonospace", "quotedblprime", "quotedblprimereversed", 
    "quotedblright", "quoteleft", "quoteleftreversed", "quotereversed", 
    "quoteright", "quoterightn", "quotesinglbase", "quotesingle", 
    "quotesinglemonospace", "r", "raarmenian", "rabengali", "racute", 
    "radeva", "radical", "radicalex", "radoverssquare", 
    "radoverssquaredsquare", "radsquare", "rafe", "rafehebrew", 
    "ragujarati", "ragurmukhi", "rahiragana", "rakatakana", 
    "rakatakanahalfwidth", "ralowerdiagonalbengali", 
    "ramiddlediagonalbengali", "ramshorn", "ratio", "rbopomofo", 
    "rcaron", "rcedilla", "rcircle", "rcommaaccent", "rdblgrave", 
    "rdotaccent", "rdotbelow", "rdotbelowmacron", "referencemark", 
    "reflexsubset", "reflexsuperset", "registered", "registersans", 
    "registerserif", "reharabic", "reharmenian", "rehfinalarabic", 
    "rehiragana", "rekatakana", "rekatakanahalfwidth", "resh", 
    "reshdageshhebrew", "reshhebrew", "reversedtilde", "reviahebrew", 
    "reviamugrashhebrew", "revlogicalnot", "rfishhook", 
    "rfishhookreversed", "rhabengali", "rhadeva", "rho", "rhook", 
    "rhookturned", "rhookturnedsuperior", "rhosymbolgreek", 
    "rhotichookmod", "rieulacirclekorean", "rieulaparenkorean", 
    "rieulcirclekorean", "rieulhieuhkorean", "rieulkiyeokkorean", 
    "rieulkiyeoksioskorean", "rieulkorean", "rieulmieumkorean", 
    "rieulpansioskorean", "rieulparenkorean", "rieulphieuphkorean", 
    "rieulpieupkorean", "rieulpieupsioskorean", "rieulsioskorean", 
    "rieulthieuthkorean", "rieultikeutkorean", 
    "rieulyeorinhieuhkorean", "rightangle", "righttackbelowcmb", 
    "righttriangle", "rihiragana", "rikatakana", "rikatakanahalfwidth", 
    "ring", "ringbelowcmb", "ringcmb", "ringhalfleft", 
    "ringhalfleftarmenian", "ringhalfleftbelowcmb", 
    "ringhalfleftcentered", "ringhalfright", "ringhalfrightbelowcmb", 
    "ringhalfrightcentered", "rinvertedbreve", "rittorusquare", 
    "rlinebelow", "rlongleg", "rlonglegturned", "rmonospace", 
    "rohiragana", "rokatakana", "rokatakanahalfwidth", "roruathai", 
    "rparen", "rrabengali", "rradeva", "rragurmukhi", "rreharabic", 
    "rrehfinalarabic", "rrvocalicbengali", "rrvocalicdeva", 
    "rrvocalicgujarati", "rrvocalicvowelsignbengali", 
    "rrvocalicvowelsigndeva", "rrvocalicvowelsigngujarati", 
    "rsuperior", "rtblock", "rturned", "rturnedsuperior", "ruhiragana", 
    "rukatakana", "rukatakanahalfwidth", "rupeemarkbengali", 
    "rupeesignbengali", "rupiah", "ruthai", "rvocalicbengali", 
    "rvocalicdeva", "rvocalicgujarati", "rvocalicvowelsignbengali", 
    "rvocalicvowelsigndeva", "rvocalicvowelsigngujarati", "s", 
    "sabengali", "sacute", "sacutedotaccent", "sadarabic", "sadeva", 
    "sadfinalarabic", "sadinitialarabic", "sadmedialarabic", 
    "sagujarati", "sagurmukhi", "sahiragana", "sakatakana", 
    "sakatakanahalfwidth", "sallallahoualayhewasallamarabic", "samekh", 
    "samekhdagesh", "samekhdageshhebrew", "samekhhebrew", "saraaathai", 
    "saraaethai", "saraaimaimalaithai", "saraaimaimuanthai", 
    "saraamthai", "saraathai", "saraethai", "saraiileftthai", 
    "saraiithai", "saraileftthai", "saraithai", "saraothai", 
    "saraueeleftthai", "saraueethai", "saraueleftthai", "sarauethai", 
    "sarauthai", "sarauuthai", "sbopomofo", "scaron", 
    "scarondotaccent", "scedilla", "schwa", "schwacyrillic", 
    "schwadieresiscyrillic", "schwahook", "scircle", "scircumflex", 
    "scommaaccent", "sdotaccent", "sdotbelow", "sdotbelowdotaccent", 
    "seagullbelowcmb", "second", "secondtonechinese", "section", 
    "seenarabic", "seenfinalarabic", "seeninitialarabic", 
    "seenmedialarabic", "segol", "segol13", "segol1f", "segol2c", 
    "segolhebrew", "segolnarrowhebrew", "segolquarterhebrew", 
    "segoltahebrew", "segolwidehebrew", "seharmenian", "sehiragana", 
    "sekatakana", "sekatakanahalfwidth", "semicolon", 
    "semicolonarabic", "semicolonmonospace", "semicolonsmall", 
    "semivoicedmarkkana", "semivoicedmarkkanahalfwidth", "sentisquare", 
    "sentosquare", "seven", "sevenarabic", "sevenbengali", 
    "sevencircle", "sevencircleinversesansserif", "sevendeva", 
    "seveneighths", "sevengujarati", "sevengurmukhi", 
    "sevenhackarabic", "sevenhangzhou", "sevenideographicparen", 
    "seveninferior", "sevenmonospace", "sevenoldstyle", "sevenparen", 
    "sevenperiod", "sevenpersian", "sevenroman", "sevensuperior", 
    "seventeencircle", "seventeenparen", "seventeenperiod", 
    "seventhai", "sfthyphen", "shaarmenian", "shabengali", 
    "shacyrillic", "shaddaarabic", "shaddadammaarabic", 
    "shaddadammatanarabic", "shaddafathaarabic", "shaddakasraarabic", 
    "shaddakasratanarabic", "shade", "shadedark", "shadelight", 
    "shademedium", "shadeva", "shagujarati", "shagurmukhi", 
    "shalshelethebrew", "shbopomofo", "shchacyrillic", "sheenarabic", 
    "sheenfinalarabic", "sheeninitialarabic", "sheenmedialarabic", 
    "sheicoptic", "sheqel", "sheqelhebrew", "sheva", "sheva115", 
    "sheva15", "sheva22", "sheva2e", "shevahebrew", 
    "shevanarrowhebrew", "shevaquarterhebrew", "shevawidehebrew", 
    "shhacyrillic", "shimacoptic", "shin", "shindagesh", 
    "shindageshhebrew", "shindageshshindot", "shindageshshindothebrew", 
    "shindageshsindot", "shindageshsindothebrew", "shindothebrew", 
    "shinhebrew", "shinshindot", "shinshindothebrew", "shinsindot", 
    "shinsindothebrew", "shook", "sigma", "sigma1", "sigmafinal", 
    "sigmalunatesymbolgreek", "sihiragana", "sikatakana", 
    "sikatakanahalfwidth", "siluqhebrew", "siluqlefthebrew", "similar", 
    "sindothebrew", "siosacirclekorean", "siosaparenkorean", 
    "sioscieuckorean", "sioscirclekorean", "sioskiyeokkorean", 
    "sioskorean", "siosnieunkorean", "siosparenkorean", 
    "siospieupkorean", "siostikeutkorean", "six", "sixarabic", 
    "sixbengali", "sixcircle", "sixcircleinversesansserif", "sixdeva", 
    "sixgujarati", "sixgurmukhi", "sixhackarabic", "sixhangzhou", 
    "sixideographicparen", "sixinferior", "sixmonospace", 
    "sixoldstyle", "sixparen", "sixperiod", "sixpersian", "sixroman", 
    "sixsuperior", "sixteencircle", 
    "sixteencurrencydenominatorbengali", "sixteenparen", 
    "sixteenperiod", "sixthai", "slash", "slashmonospace", "slong", 
    "slongdotaccent", "smileface", "smonospace", "sofpasuqhebrew", 
    "softhyphen", "softsigncyrillic", "sohiragana", "sokatakana", 
    "sokatakanahalfwidth", "soliduslongoverlaycmb", 
    "solidusshortoverlaycmb", "sorusithai", "sosalathai", "sosothai", 
    "sosuathai", "space", "spacehackarabic", "spade", "spadesuitblack", 
    "spadesuitwhite", "sparen", "squarebelowcmb", "squarecc", 
    "squarecm", "squarediagonalcrosshatchfill", "squarehorizontalfill", 
    "squarekg", "squarekm", "squarekmcapital", "squareln", "squarelog", 
    "squaremg", "squaremil", "squaremm", "squaremsquared", 
    "squareorthogonalcrosshatchfill", 
    "squareupperlefttolowerrightfill", 
    "squareupperrighttolowerleftfill", "squareverticalfill", 
    "squarewhitewithsmallblack", "srsquare", "ssabengali", "ssadeva", 
    "ssagujarati", "ssangcieuckorean", "ssanghieuhkorean", 
    "ssangieungkorean", "ssangkiyeokkorean", "ssangnieunkorean", 
    "ssangpieupkorean", "ssangsioskorean", "ssangtikeutkorean", 
    "ssuperior", "sterling", "sterlingmonospace", 
    "strokelongoverlaycmb", "strokeshortoverlaycmb", "subset", 
    "subsetnotequal", "subsetorequal", "succeeds", "suchthat", 
    "suhiragana", "sukatakana", "sukatakanahalfwidth", "sukunarabic", 
    "summation", "sun", "superset", "supersetnotequal", 
    "supersetorequal", "svsquare", "syouwaerasquare", "t", "tabengali", 
    "tackdown", "tackleft", "tadeva", "tagujarati", "tagurmukhi", 
    "taharabic", "tahfinalarabic", "tahinitialarabic", "tahiragana", 
    "tahmedialarabic", "taisyouerasquare", "takatakana", 
    "takatakanahalfwidth", "tatweelarabic", "tau", "tav", "tavdages", 
    "tavdagesh", "tavdageshhebrew", "tavhebrew", "tbar", "tbopomofo", 
    "tcaron", "tccurl", "tcedilla", "tcheharabic", "tchehfinalarabic", 
    "tchehinitialarabic", "tchehmedialarabic", "tcircle", 
    "tcircumflexbelow", "tcommaaccent", "tdieresis", "tdotaccent", 
    "tdotbelow", "tecyrillic", "tedescendercyrillic", "teharabic", 
    "tehfinalarabic", "tehhahinitialarabic", "tehhahisolatedarabic", 
    "tehinitialarabic", "tehiragana", "tehjeeminitialarabic", 
    "tehjeemisolatedarabic", "tehmarbutaarabic", 
    "tehmarbutafinalarabic", "tehmedialarabic", "tehmeeminitialarabic", 
    "tehmeemisolatedarabic", "tehnoonfinalarabic", "tekatakana", 
    "tekatakanahalfwidth", "telephone", "telephoneblack", 
    "telishagedolahebrew", "telishaqetanahebrew", "tencircle", 
    "tenideographicparen", "tenparen", "tenperiod", "tenroman", "tesh", 
    "tet", "tetdagesh", "tetdageshhebrew", "tethebrew", 
    "tetsecyrillic", "tevirhebrew", "tevirlefthebrew", "thabengali", 
    "thadeva", "thagujarati", "thagurmukhi", "thalarabic", 
    "thalfinalarabic", "thanthakhatlowleftthai", 
    "thanthakhatlowrightthai", "thanthakhatthai", 
    "thanthakhatupperleftthai", "theharabic", "thehfinalarabic", 
    "thehinitialarabic", "thehmedialarabic", "thereexists", 
    "therefore", "theta", "theta1", "thetasymbolgreek", 
    "thieuthacirclekorean", "thieuthaparenkorean", 
    "thieuthcirclekorean", "thieuthkorean", "thieuthparenkorean", 
    "thirteencircle", "thirteenparen", "thirteenperiod", 
    "thonangmonthothai", "thook", "thophuthaothai", "thorn", 
    "thothahanthai", "thothanthai", "thothongthai", "thothungthai", 
    "thousandcyrillic", "thousandsseparatorarabic", 
    "thousandsseparatorpersian", "three", "threearabic", 
    "threebengali", "threecircle", "threecircleinversesansserif", 
    "threedeva", "threeeighths", "threegujarati", "threegurmukhi", 
    "threehackarabic", "threehangzhou", "threeideographicparen", 
    "threeinferior", "threemonospace", "threenumeratorbengali", 
    "threeoldstyle", "threeparen", "threeperiod", "threepersian", 
    "threequarters", "threequartersemdash", "threeroman", 
    "threesuperior", "threethai", "thzsquare", "tihiragana", 
    "tikatakana", "tikatakanahalfwidth", "tikeutacirclekorean", 
    "tikeutaparenkorean", "tikeutcirclekorean", "tikeutkorean", 
    "tikeutparenkorean", "tilde", "tildebelowcmb", "tildecmb", 
    "tildecomb", "tildedoublecmb", "tildeoperator", "tildeoverlaycmb", 
    "tildeverticalcmb", "timescircle", "tipehahebrew", 
    "tipehalefthebrew", "tippigurmukhi", "titlocyrilliccmb", 
    "tiwnarmenian", "tlinebelow", "tmonospace", "toarmenian", 
    "tohiragana", "tokatakana", "tokatakanahalfwidth", 
    "tonebarextrahighmod", "tonebarextralowmod", "tonebarhighmod", 
    "tonebarlowmod", "tonebarmidmod", "tonefive", "tonesix", "tonetwo", 
    "tonos", "tonsquare", "topatakthai", "tortoiseshellbracketleft", 
    "tortoiseshellbracketleftsmall", 
    "tortoiseshellbracketleftvertical", "tortoiseshellbracketright", 
    "tortoiseshellbracketrightsmall", 
    "tortoiseshellbracketrightvertical", "totaothai", "tpalatalhook", 
    "tparen", "trademark", "trademarksans", "trademarkserif", 
    "tretroflexhook", "triagdn", "triaglf", "triagrt", "triagup", "ts", 
    "tsadi", "tsadidagesh", "tsadidageshhebrew", "tsadihebrew", 
    "tsecyrillic", "tsere", "tsere12", "tsere1e", "tsere2b", 
    "tserehebrew", "tserenarrowhebrew", "tserequarterhebrew", 
    "tserewidehebrew", "tshecyrillic", "tsuperior", "ttabengali", 
    "ttadeva", "ttagujarati", "ttagurmukhi", "tteharabic", 
    "ttehfinalarabic", "ttehinitialarabic", "ttehmedialarabic", 
    "tthabengali", "tthadeva", "tthagujarati", "tthagurmukhi", 
    "tturned", "tuhiragana", "tukatakana", "tukatakanahalfwidth", 
    "tusmallhiragana", "tusmallkatakana", "tusmallkatakanahalfwidth", 
    "twelvecircle", "twelveparen", "twelveperiod", "twelveroman", 
    "twentycircle", "twentyhangzhou", "twentyparen", "twentyperiod", 
    "two", "twoarabic", "twobengali", "twocircle", 
    "twocircleinversesansserif", "twodeva", "twodotenleader", 
    "twodotleader", "twodotleadervertical", "twogujarati", 
    "twogurmukhi", "twohackarabic", "twohangzhou", 
    "twoideographicparen", "twoinferior", "twomonospace", 
    "twonumeratorbengali", "twooldstyle", "twoparen", "twoperiod", 
    "twopersian", "tworoman", "twostroke", "twosuperior", "twothai", 
    "twothirds", "u", "uacute", "ubar", "ubengali", "ubopomofo", 
    "ubreve", "ucaron", "ucircle", "ucircumflex", "ucircumflexbelow", 
    "ucyrillic", "udattadeva", "udblacute", "udblgrave", "udeva", 
    "udieresis", "udieresisacute", "udieresisbelow", "udieresiscaron", 
    "udieresiscyrillic", "udieresisgrave", "udieresismacron", 
    "udotbelow", "ugrave", "ugujarati", "ugurmukhi", "uhiragana", 
    "uhookabove", "uhorn", "uhornacute", "uhorndotbelow", "uhorngrave", 
    "uhornhookabove", "uhorntilde", "uhungarumlaut", 
    "uhungarumlautcyrillic", "uinvertedbreve", "ukatakana", 
    "ukatakanahalfwidth", "ukcyrillic", "ukorean", "umacron", 
    "umacroncyrillic", "umacrondieresis", "umatragurmukhi", 
    "umonospace", "underscore", "underscoredbl", "underscoremonospace", 
    "underscorevertical", "underscorewavy", "union", "universal", 
    "uogonek", "uparen", "upblock", "upperdothebrew", "upsilon", 
    "upsilondieresis", "upsilondieresistonos", "upsilonlatin", 
    "upsilontonos", "uptackbelowcmb", "uptackmod", "uragurmukhi", 
    "uring", "ushortcyrillic", "usmallhiragana", "usmallkatakana", 
    "usmallkatakanahalfwidth", "ustraightcyrillic", 
    "ustraightstrokecyrillic", "utilde", "utildeacute", "utildebelow", 
    "uubengali", "uudeva", "uugujarati", "uugurmukhi", 
    "uumatragurmukhi", "uuvowelsignbengali", "uuvowelsigndeva", 
    "uuvowelsigngujarati", "uvowelsignbengali", "uvowelsigndeva", 
    "uvowelsigngujarati", "v", "vadeva", "vagujarati", "vagurmukhi", 
    "vakatakana", "vav", "vavdagesh", "vavdagesh65", "vavdageshhebrew", 
    "vavhebrew", "vavholam", "vavholamhebrew", "vavvavhebrew", 
    "vavyodhebrew", "vcircle", "vdotbelow", "vecyrillic", "veharabic", 
    "vehfinalarabic", "vehinitialarabic", "vehmedialarabic", 
    "vekatakana", "venus", "verticalbar", "verticallineabovecmb", 
    "verticallinebelowcmb", "verticallinelowmod", "verticallinemod", 
    "vewarmenian", "vhook", "vikatakana", "viramabengali", 
    "viramadeva", "viramagujarati", "visargabengali", "visargadeva", 
    "visargagujarati", "vmonospace", "voarmenian", 
    "voicediterationhiragana", "voicediterationkatakana", 
    "voicedmarkkana", "voicedmarkkanahalfwidth", "vokatakana", 
    "vparen", "vtilde", "vturned", "vuhiragana", "vukatakana", "w", 
    "wacute", "waekorean", "wahiragana", "wakatakana", 
    "wakatakanahalfwidth", "wakorean", "wasmallhiragana", 
    "wasmallkatakana", "wattosquare", "wavedash", 
    "wavyunderscorevertical", "wawarabic", "wawfinalarabic", 
    "wawhamzaabovearabic", "wawhamzaabovefinalarabic", "wbsquare", 
    "wcircle", "wcircumflex", "wdieresis", "wdotaccent", "wdotbelow", 
    "wehiragana", "weierstrass", "wekatakana", "wekorean", "weokorean", 
    "wgrave", "whitebullet", "whitecircle", "whitecircleinverse", 
    "whitecornerbracketleft", "whitecornerbracketleftvertical", 
    "whitecornerbracketright", "whitecornerbracketrightvertical", 
    "whitediamond", "whitediamondcontainingblacksmalldiamond", 
    "whitedownpointingsmalltriangle", "whitedownpointingtriangle", 
    "whiteleftpointingsmalltriangle", "whiteleftpointingtriangle", 
    "whitelenticularbracketleft", "whitelenticularbracketright", 
    "whiterightpointingsmalltriangle", "whiterightpointingtriangle", 
    "whitesmallsquare", "whitesmilingface", "whitesquare", "whitestar", 
    "whitetelephone", "whitetortoiseshellbracketleft", 
    "whitetortoiseshellbracketright", "whiteuppointingsmalltriangle", 
    "whiteuppointingtriangle", "wihiragana", "wikatakana", "wikorean", 
    "wmonospace", "wohiragana", "wokatakana", "wokatakanahalfwidth", 
    "won", "wonmonospace", "wowaenthai", "wparen", "wring", 
    "wsuperior", "wturned", "wynn", "x", "xabovecmb", "xbopomofo", 
    "xcircle", "xdieresis", "xdotaccent", "xeharmenian", "xi", 
    "xmonospace", "xparen", "xsuperior", "y", "yaadosquare", 
    "yabengali", "yacute", "yadeva", "yaekorean", "yagujarati", 
    "yagurmukhi", "yahiragana", "yakatakana", "yakatakanahalfwidth", 
    "yakorean", "yamakkanthai", "yasmallhiragana", "yasmallkatakana", 
    "yasmallkatakanahalfwidth", "yatcyrillic", "ycircle", 
    "ycircumflex", "ydieresis", "ydotaccent", "ydotbelow", "yeharabic", 
    "yehbarreearabic", "yehbarreefinalarabic", "yehfinalarabic", 
    "yehhamzaabovearabic", "yehhamzaabovefinalarabic", 
    "yehhamzaaboveinitialarabic", "yehhamzaabovemedialarabic", 
    "yehinitialarabic", "yehmedialarabic", "yehmeeminitialarabic", 
    "yehmeemisolatedarabic", "yehnoonfinalarabic", 
    "yehthreedotsbelowarabic", "yekorean", "yen", "yenmonospace", 
    "yeokorean", "yeorinhieuhkorean", "yerahbenyomohebrew", 
    "yerahbenyomolefthebrew", "yericyrillic", "yerudieresiscyrillic", 
    "yesieungkorean", "yesieungpansioskorean", "yesieungsioskorean", 
    "yetivhebrew", "ygrave", "yhook", "yhookabove", "yiarmenian", 
    "yicyrillic", "yikorean", "yinyang", "yiwnarmenian", "ymonospace", 
    "yod", "yoddagesh", "yoddageshhebrew", "yodhebrew", "yodyodhebrew", 
    "yodyodpatahhebrew", "yohiragana", "yoikorean", "yokatakana", 
    "yokatakanahalfwidth", "yokorean", "yosmallhiragana", 
    "yosmallkatakana", "yosmallkatakanahalfwidth", "yotgreek", 
    "yoyaekorean", "yoyakorean", "yoyakthai", "yoyingthai", "yparen", 
    "ypogegrammeni", "ypogegrammenigreekcmb", "yr", "yring", 
    "ysuperior", "ytilde", "yturned", "yuhiragana", "yuikorean", 
    "yukatakana", "yukatakanahalfwidth", "yukorean", "yusbigcyrillic", 
    "yusbigiotifiedcyrillic", "yuslittlecyrillic", 
    "yuslittleiotifiedcyrillic", "yusmallhiragana", "yusmallkatakana", 
    "yusmallkatakanahalfwidth", "yuyekorean", "yuyeokorean", 
    "yyabengali", "yyadeva", "z", "zaarmenian", "zacute", "zadeva", 
    "zagurmukhi", "zaharabic", "zahfinalarabic", "zahinitialarabic", 
    "zahiragana", "zahmedialarabic", "zainarabic", "zainfinalarabic", 
    "zakatakana", "zaqefgadolhebrew", "zaqefqatanhebrew", 
    "zarqahebrew", "zayin", "zayindagesh", "zayindageshhebrew", 
    "zayinhebrew", "zbopomofo", "zcaron", "zcircle", "zcircumflex", 
    "zcurl", "zdot", "zdotaccent", "zdotbelow", "zecyrillic", 
    "zedescendercyrillic", "zedieresiscyrillic", "zehiragana", 
    "zekatakana", "zero", "zeroarabic", "zerobengali", "zerodeva", 
    "zerogujarati", "zerogurmukhi", "zerohackarabic", "zeroinferior", 
    "zeromonospace", "zerooldstyle", "zeropersian", "zerosuperior", 
    "zerothai", "zerowidthjoiner", "zerowidthnonjoiner", 
    "zerowidthspace", "zeta", "zhbopomofo", "zhearmenian", 
    "zhebrevecyrillic", "zhecyrillic", "zhedescendercyrillic", 
    "zhedieresiscyrillic", "zihiragana", "zikatakana", "zinorhebrew", 
    "zlinebelow", "zmonospace", "zohiragana", "zokatakana", "zparen", 
    "zretroflexhook", "zstroke", "zuhiragana", "zukatakana", 
};

glyph glyph_intern(char const *glyphname)
{
    int i, j, k, c;

    i = -1;
    j = lenof(ps_glyphs_alphabetic);
    while (j-i > 1) {
	k = (i + j) / 2;
	c = strcmp(glyphname, ps_glyphs_alphabetic[k]);

	if (c == 0)
	    return k;
	else if (c < 0)
	    j = k;
	else
	    i = k;
    }

    return NOGLYPH;		       /* illegal value means not found */
}

char const *glyph_extern(glyph glyph)
{

    if (glyph == NOGLYPH) return ".notdef";
    assert(glyph < lenof(ps_glyphs_alphabetic));
    return ps_glyphs_alphabetic[glyph];
}

/* ----------------------------------------------------------------------
 * Mapping between PS character names (/aacute, /zcaron etc) and
 * Unicode code points.
 * 
 * Generated from the Adobe Glyph List at
 * 
 *   http://partners.adobe.com/public/developer/en/opentype/glyphlist.txt
 * 
 * by another Perl/sh fragment.

grep '^[^#;][^;]*;[^ ][^ ][^ ][^ ]$' glyphlist.txt | sort -t\; +0 -1 | \
    cut -f2 -d\; | perl -ne 'chomp; print "0x$_, "' | \
    fold -s -w68 | sed 's/^/    /'; echo

 */
static const unsigned short ps_codes_alphabetic[] = {
    0x0041, 0x00C6, 0x01FC, 0x01E2, 0xF7E6, 0x00C1, 0xF7E1, 0x0102, 
    0x1EAE, 0x04D0, 0x1EB6, 0x1EB0, 0x1EB2, 0x1EB4, 0x01CD, 0x24B6, 
    0x00C2, 0x1EA4, 0x1EAC, 0x1EA6, 0x1EA8, 0xF7E2, 0x1EAA, 0xF6C9, 
    0xF7B4, 0x0410, 0x0200, 0x00C4, 0x04D2, 0x01DE, 0xF7E4, 0x1EA0, 
    0x01E0, 0x00C0, 0xF7E0, 0x1EA2, 0x04D4, 0x0202, 0x0391, 0x0386, 
    0x0100, 0xFF21, 0x0104, 0x00C5, 0x01FA, 0x1E00, 0xF7E5, 0xF761, 
    0x00C3, 0xF7E3, 0x0531, 0x0042, 0x24B7, 0x1E02, 0x1E04, 0x0411, 
    0x0532, 0x0392, 0x0181, 0x1E06, 0xFF22, 0xF6F4, 0xF762, 0x0182, 
    0x0043, 0x053E, 0x0106, 0xF6CA, 0xF6F5, 0x010C, 0x00C7, 0x1E08, 
    0xF7E7, 0x24B8, 0x0108, 0x010A, 0x010A, 0xF7B8, 0x0549, 0x04BC, 
    0x0427, 0x04BE, 0x04B6, 0x04F4, 0x0543, 0x04CB, 0x04B8, 0x03A7, 
    0x0187, 0xF6F6, 0xFF23, 0x0551, 0xF763, 0x0044, 0x01F1, 0x01C4, 
    0x0534, 0x0189, 0x010E, 0x1E10, 0x24B9, 0x1E12, 0x0110, 0x1E0A, 
    0x1E0C, 0x0414, 0x03EE, 0x2206, 0x0394, 0x018A, 0xF6CB, 0xF6CC, 
    0xF6CD, 0xF7A8, 0x03DC, 0x0402, 0x1E0E, 0xFF24, 0xF6F7, 0x0110, 
    0xF764, 0x018B, 0x01F2, 0x01C5, 0x04E0, 0x0405, 0x040F, 0x0045, 
    0x00C9, 0xF7E9, 0x0114, 0x011A, 0x1E1C, 0x0535, 0x24BA, 0x00CA, 
    0x1EBE, 0x1E18, 0x1EC6, 0x1EC0, 0x1EC2, 0xF7EA, 0x1EC4, 0x0404, 
    0x0204, 0x00CB, 0xF7EB, 0x0116, 0x0116, 0x1EB8, 0x0424, 0x00C8, 
    0xF7E8, 0x0537, 0x1EBA, 0x2167, 0x0206, 0x0464, 0x041B, 0x216A, 
    0x0112, 0x1E16, 0x1E14, 0x041C, 0xFF25, 0x041D, 0x04A2, 0x014A, 
    0x04A4, 0x04C7, 0x0118, 0x0190, 0x0395, 0x0388, 0x0420, 0x018E, 
    0x042D, 0x0421, 0x04AA, 0x01A9, 0xF765, 0x0397, 0x0538, 0x0389, 
    0x00D0, 0xF7F0, 0x1EBC, 0x1E1A, 0x20AC, 0x01B7, 0x01EE, 0x01B8, 
    0x0046, 0x24BB, 0x1E1E, 0x0556, 0x03E4, 0x0191, 0x0472, 0x2164, 
    0xFF26, 0x2163, 0xF766, 0x0047, 0x3387, 0x01F4, 0x0393, 0x0194, 
    0x03EA, 0x011E, 0x01E6, 0x0122, 0x24BC, 0x011C, 0x0122, 0x0120, 
    0x0120, 0x0413, 0x0542, 0x0494, 0x0492, 0x0490, 0x0193, 0x0533, 
    0x0403, 0x1E20, 0xFF27, 0xF6CE, 0xF760, 0xF767, 0x029B, 0x01E4, 
    0x0048, 0x25CF, 0x25AA, 0x25AB, 0x25A1, 0x33CB, 0x04A8, 0x04B2, 
    0x042A, 0x0126, 0x1E2A, 0x1E28, 0x24BD, 0x0124, 0x1E26, 0x1E22, 
    0x1E24, 0xFF28, 0x0540, 0x03E8, 0xF768, 0xF6CF, 0xF6F8, 0x3390, 
    0x0049, 0x042F, 0x0132, 0x042E, 0x00CD, 0xF7ED, 0x012C, 0x01CF, 
    0x24BE, 0x00CE, 0xF7EE, 0x0406, 0x0208, 0x00CF, 0x1E2E, 0x04E4, 
    0xF7EF, 0x0130, 0x0130, 0x1ECA, 0x04D6, 0x0415, 0x2111, 0x00CC, 
    0xF7EC, 0x1EC8, 0x0418, 0x020A, 0x0419, 0x012A, 0x04E2, 0xFF29, 
    0x053B, 0x0401, 0x012E, 0x0399, 0x0196, 0x03AA, 0x038A, 0xF769, 
    0x0197, 0x0128, 0x1E2C, 0x0474, 0x0476, 0x004A, 0x0541, 0x24BF, 
    0x0134, 0x0408, 0x054B, 0xFF2A, 0xF76A, 0x004B, 0x3385, 0x33CD, 
    0x04A0, 0x1E30, 0x041A, 0x049A, 0x04C3, 0x039A, 0x049E, 0x049C, 
    0x01E8, 0x0136, 0x24C0, 0x0136, 0x1E32, 0x0554, 0x053F, 0x0425, 
    0x03E6, 0x0198, 0x040C, 0x1E34, 0xFF2B, 0x0480, 0x03DE, 0x046E, 
    0xF76B, 0x004C, 0x01C7, 0xF6BF, 0x0139, 0x039B, 0x013D, 0x013B, 
    0x24C1, 0x1E3C, 0x013B, 0x013F, 0x013F, 0x1E36, 0x1E38, 0x053C, 
    0x01C8, 0x0409, 0x1E3A, 0xFF2C, 0x0141, 0xF6F9, 0xF76C, 0x004D, 
    0x3386, 0xF6D0, 0xF7AF, 0x1E3E, 0x24C2, 0x1E40, 0x1E42, 0x0544, 
    0xFF2D, 0xF76D, 0x019C, 0x039C, 0x004E, 0x01CA, 0x0143, 0x0147, 
    0x0145, 0x24C3, 0x1E4A, 0x0145, 0x1E44, 0x1E46, 0x019D, 0x2168, 
    0x01CB, 0x040A, 0x1E48, 0xFF2E, 0x0546, 0xF76E, 0x00D1, 0xF7F1, 
    0x039D, 0x004F, 0x0152, 0xF6FA, 0x00D3, 0xF7F3, 0x04E8, 0x04EA, 
    0x014E, 0x01D1, 0x019F, 0x24C4, 0x00D4, 0x1ED0, 0x1ED8, 0x1ED2, 
    0x1ED4, 0xF7F4, 0x1ED6, 0x041E, 0x0150, 0x020C, 0x00D6, 0x04E6, 
    0xF7F6, 0x1ECC, 0xF6FB, 0x00D2, 0xF7F2, 0x0555, 0x2126, 0x1ECE, 
    0x01A0, 0x1EDA, 0x1EE2, 0x1EDC, 0x1EDE, 0x1EE0, 0x0150, 0x01A2, 
    0x020E, 0x014C, 0x1E52, 0x1E50, 0x2126, 0x0460, 0x03A9, 0x047A, 
    0x047C, 0x038F, 0x039F, 0x038C, 0xFF2F, 0x2160, 0x01EA, 0x01EC, 
    0x0186, 0x00D8, 0x01FE, 0xF7F8, 0xF76F, 0x01FE, 0x047E, 0x00D5, 
    0x1E4C, 0x1E4E, 0xF7F5, 0x0050, 0x1E54, 0x24C5, 0x1E56, 0x041F, 
    0x054A, 0x04A6, 0x03A6, 0x01A4, 0x03A0, 0x0553, 0xFF30, 0x03A8, 
    0x0470, 0xF770, 0x0051, 0x24C6, 0xFF31, 0xF771, 0x0052, 0x054C, 
    0x0154, 0x0158, 0x0156, 0x24C7, 0x0156, 0x0210, 0x1E58, 0x1E5A, 
    0x1E5C, 0x0550, 0x211C, 0x03A1, 0xF6FC, 0x0212, 0x1E5E, 0xFF32, 
    0xF772, 0x0281, 0x02B6, 0x0053, 0x250C, 0x2514, 0x2510, 0x2518, 
    0x253C, 0x252C, 0x2534, 0x251C, 0x2524, 0x2500, 0x2502, 0x2561, 
    0x2562, 0x2556, 0x2555, 0x2563, 0x2551, 0x2557, 0x255D, 0x255C, 
    0x255B, 0x255E, 0x255F, 0x255A, 0x2554, 0x2569, 0x2566, 0x2560, 
    0x2550, 0x256C, 0x2567, 0x2568, 0x2564, 0x2565, 0x2559, 0x2558, 
    0x2552, 0x2553, 0x256B, 0x256A, 0x015A, 0x1E64, 0x03E0, 0x0160, 
    0x1E66, 0xF6FD, 0x015E, 0x018F, 0x04D8, 0x04DA, 0x24C8, 0x015C, 
    0x0218, 0x1E60, 0x1E62, 0x1E68, 0x054D, 0x2166, 0x0547, 0x0428, 
    0x0429, 0x03E2, 0x04BA, 0x03EC, 0x03A3, 0x2165, 0xFF33, 0x042C, 
    0xF773, 0x03DA, 0x0054, 0x03A4, 0x0166, 0x0164, 0x0162, 0x24C9, 
    0x1E70, 0x0162, 0x1E6A, 0x1E6C, 0x0422, 0x04AC, 0x2169, 0x04B4, 
    0x0398, 0x01AC, 0x00DE, 0xF7FE, 0x2162, 0xF6FE, 0x054F, 0x1E6E, 
    0xFF34, 0x0539, 0x01BC, 0x0184, 0x01A7, 0x01AE, 0x0426, 0x040B, 
    0xF774, 0x216B, 0x2161, 0x0055, 0x00DA, 0xF7FA, 0x016C, 0x01D3, 
    0x24CA, 0x00DB, 0x1E76, 0xF7FB, 0x0423, 0x0170, 0x0214, 0x00DC, 
    0x01D7, 0x1E72, 0x01D9, 0x04F0, 0x01DB, 0x01D5, 0xF7FC, 0x1EE4, 
    0x00D9, 0xF7F9, 0x1EE6, 0x01AF, 0x1EE8, 0x1EF0, 0x1EEA, 0x1EEC, 
    0x1EEE, 0x0170, 0x04F2, 0x0216, 0x0478, 0x016A, 0x04EE, 0x1E7A, 
    0xFF35, 0x0172, 0x03A5, 0x03D2, 0x03D3, 0x01B1, 0x03AB, 0x03D4, 
    0x03D2, 0x038E, 0x016E, 0x040E, 0xF775, 0x04AE, 0x04B0, 0x0168, 
    0x1E78, 0x1E74, 0x0056, 0x24CB, 0x1E7E, 0x0412, 0x054E, 0x01B2, 
    0xFF36, 0x0548, 0xF776, 0x1E7C, 0x0057, 0x1E82, 0x24CC, 0x0174, 
    0x1E84, 0x1E86, 0x1E88, 0x1E80, 0xFF37, 0xF777, 0x0058, 0x24CD, 
    0x1E8C, 0x1E8A, 0x053D, 0x039E, 0xFF38, 0xF778, 0x0059, 0x00DD, 
    0xF7FD, 0x0462, 0x24CE, 0x0176, 0x0178, 0xF7FF, 0x1E8E, 0x1EF4, 
    0x042B, 0x04F8, 0x1EF2, 0x01B3, 0x1EF6, 0x0545, 0x0407, 0x0552, 
    0xFF39, 0xF779, 0x1EF8, 0x046A, 0x046C, 0x0466, 0x0468, 0x005A, 
    0x0536, 0x0179, 0x017D, 0xF6FF, 0x24CF, 0x1E90, 0x017B, 0x017B, 
    0x1E92, 0x0417, 0x0498, 0x04DE, 0x0396, 0x053A, 0x04C1, 0x0416, 
    0x0496, 0x04DC, 0x1E94, 0xFF3A, 0xF77A, 0x01B5, 0x0061, 0x0986, 
    0x00E1, 0x0906, 0x0A86, 0x0A06, 0x0A3E, 0x3303, 0x09BE, 0x093E, 
    0x0ABE, 0x055F, 0x0970, 0x0985, 0x311A, 0x0103, 0x1EAF, 0x04D1, 
    0x1EB7, 0x1EB1, 0x1EB3, 0x1EB5, 0x01CE, 0x24D0, 0x00E2, 0x1EA5, 
    0x1EAD, 0x1EA7, 0x1EA9, 0x1EAB, 0x00B4, 0x0317, 0x0301, 0x0301, 
    0x0954, 0x02CF, 0x0341, 0x0430, 0x0201, 0x0A71, 0x0905, 0x00E4, 
    0x04D3, 0x01DF, 0x1EA1, 0x01E1, 0x00E6, 0x01FD, 0x3150, 0x01E3, 
    0x2015, 0x20A4, 0x0410, 0x0411, 0x0412, 0x0413, 0x0414, 0x0415, 
    0x0401, 0x0416, 0x0417, 0x0418, 0x0419, 0x041A, 0x041B, 0x041C, 
    0x041D, 0x041E, 0x041F, 0x0420, 0x0421, 0x0422, 0x0423, 0x0424, 
    0x0425, 0x0426, 0x0427, 0x0428, 0x0429, 0x042A, 0x042B, 0x042C, 
    0x042D, 0x042E, 0x042F, 0x0490, 0x0402, 0x0403, 0x0404, 0x0405, 
    0x0406, 0x0407, 0x0408, 0x0409, 0x040A, 0x040B, 0x040C, 0x040E, 
    0xF6C4, 0xF6C5, 0x0430, 0x0431, 0x0432, 0x0433, 0x0434, 0x0435, 
    0x0451, 0x0436, 0x0437, 0x0438, 0x0439, 0x043A, 0x043B, 0x043C, 
    0x043D, 0x043E, 0x043F, 0x0440, 0x0441, 0x0442, 0x0443, 0x0444, 
    0x0445, 0x0446, 0x0447, 0x0448, 0x0449, 0x044A, 0x044B, 0x044C, 
    0x044D, 0x044E, 0x044F, 0x0491, 0x0452, 0x0453, 0x0454, 0x0455, 
    0x0456, 0x0457, 0x0458, 0x0459, 0x045A, 0x045B, 0x045C, 0x045E, 
    0x040F, 0x0462, 0x0472, 0x0474, 0xF6C6, 0x045F, 0x0463, 0x0473, 
    0x0475, 0xF6C7, 0xF6C8, 0x04D9, 0x200E, 0x200F, 0x200D, 0x066A, 
    0x060C, 0x0660, 0x0661, 0x0662, 0x0663, 0x0664, 0x0665, 0x0666, 
    0x0667, 0x0668, 0x0669, 0x061B, 0x061F, 0x0621, 0x0622, 0x0623, 
    0x0624, 0x0625, 0x0626, 0x0627, 0x0628, 0x0629, 0x062A, 0x062B, 
    0x062C, 0x062D, 0x062E, 0x062F, 0x0630, 0x0631, 0x0632, 0x0633, 
    0x0634, 0x0635, 0x0636, 0x0637, 0x0638, 0x0639, 0x063A, 0x0640, 
    0x0641, 0x0642, 0x0643, 0x0644, 0x0645, 0x0646, 0x0648, 0x0649, 
    0x064A, 0x064B, 0x064C, 0x064D, 0x064E, 0x064F, 0x0650, 0x0651, 
    0x0652, 0x0647, 0x06A4, 0x067E, 0x0686, 0x0698, 0x06AF, 0x0679, 
    0x0688, 0x0691, 0x06BA, 0x06D2, 0x06D5, 0x20AA, 0x05BE, 0x05C3, 
    0x05D0, 0x05D1, 0x05D2, 0x05D3, 0x05D4, 0x05D5, 0x05D6, 0x05D7, 
    0x05D8, 0x05D9, 0x05DA, 0x05DB, 0x05DC, 0x05DD, 0x05DE, 0x05DF, 
    0x05E0, 0x05E1, 0x05E2, 0x05E3, 0x05E4, 0x05E5, 0x05E6, 0x05E7, 
    0x05E8, 0x05E9, 0x05EA, 0xFB2A, 0xFB2B, 0xFB4B, 0xFB1F, 0x05F0, 
    0x05F1, 0x05F2, 0xFB35, 0x05B4, 0x05B5, 0x05B6, 0x05BB, 0x05B8, 
    0x05B7, 0x05B0, 0x05B2, 0x05B1, 0x05B3, 0x05C2, 0x05C1, 0x05B9, 
    0x05BC, 0x05BD, 0x05BF, 0x05C0, 0x02BC, 0x2105, 0x2113, 0x2116, 
    0x202C, 0x202D, 0x202E, 0x200C, 0x066D, 0x02BD, 0x00E0, 0x0A85, 
    0x0A05, 0x3042, 0x1EA3, 0x0990, 0x311E, 0x0910, 0x04D5, 0x0A90, 
    0x0A10, 0x0A48, 0x0639, 0xFECA, 0xFECB, 0xFECC, 0x0203, 0x09C8, 
    0x0948, 0x0AC8, 0x30A2, 0xFF71, 0x314F, 0x05D0, 0x0627, 0xFB30, 
    0xFE8E, 0x0623, 0xFE84, 0x0625, 0xFE88, 0x05D0, 0xFB4F, 0x0622, 
    0xFE82, 0x0649, 0xFEF0, 0xFEF3, 0xFEF4, 0xFB2E, 0xFB2F, 0x2135, 
    0x224C, 0x03B1, 0x03AC, 0x0101, 0xFF41, 0x0026, 0xFF06, 0xF726, 
    0x33C2, 0x3122, 0x3124, 0x0E5A, 0x2220, 0x3008, 0xFE3F, 0x3009, 
    0xFE40, 0x2329, 0x232A, 0x212B, 0x0387, 0x0952, 0x0982, 0x0902, 
    0x0A82, 0x0105, 0x3300, 0x249C, 0x055A, 0x02BC, 0xF8FF, 0x2250, 
    0x2248, 0x2252, 0x2245, 0x318E, 0x318D, 0x2312, 0x1E9A, 0x00E5, 
    0x01FB, 0x1E01, 0x2194, 0x21E3, 0x21E0, 0x21E2, 0x21E1, 0x21D4, 
    0x21D3, 0x21D0, 0x21D2, 0x21D1, 0x2193, 0x2199, 0x2198, 0x21E9, 
    0x02C5, 0x02C2, 0x02C3, 0x02C4, 0xF8E7, 0x2190, 0x21D0, 0x21CD, 
    0x21C6, 0x21E6, 0x2192, 0x21CF, 0x279E, 0x21C4, 0x21E8, 0x21E4, 
    0x21E5, 0x2191, 0x2195, 0x21A8, 0x21A8, 0x2196, 0x21C5, 0x2197, 
    0x21E7, 0xF8E6, 0x005E, 0xFF3E, 0x007E, 0xFF5E, 0x0251, 0x0252, 
    0x3041, 0x30A1, 0xFF67, 0x002A, 0x066D, 0x066D, 0x2217, 0xFF0A, 
    0xFE61, 0x2042, 0xF6E9, 0x2243, 0x0040, 0x00E3, 0xFF20, 0xFE6B, 
    0x0250, 0x0994, 0x3120, 0x0914, 0x0A94, 0x0A14, 0x09D7, 0x0A4C, 
    0x09CC, 0x094C, 0x0ACC, 0x093D, 0x0561, 0x05E2, 0xFB20, 0x05E2, 
    0x0062, 0x09AC, 0x005C, 0xFF3C, 0x092C, 0x0AAC, 0x0A2C, 0x3070, 
    0x0E3F, 0x30D0, 0x007C, 0xFF5C, 0x3105, 0x24D1, 0x1E03, 0x1E05, 
    0x266C, 0x2235, 0x0431, 0x0628, 0xFE90, 0xFE91, 0x3079, 0xFE92, 
    0xFC9F, 0xFC08, 0xFC6D, 0x30D9, 0x0562, 0x05D1, 0x03B2, 0x03D0, 
    0xFB31, 0xFB31, 0x05D1, 0xFB4C, 0x09AD, 0x092D, 0x0AAD, 0x0A2D, 
    0x0253, 0x3073, 0x30D3, 0x0298, 0x0A02, 0x3331, 0x25CF, 0x25C6, 
    0x25BC, 0x25C4, 0x25C0, 0x3010, 0xFE3B, 0x3011, 0xFE3C, 0x25E3, 
    0x25E2, 0x25AC, 0x25BA, 0x25B6, 0x25AA, 0x263B, 0x25A0, 0x2605, 
    0x25E4, 0x25E5, 0x25B4, 0x25B2, 0x2423, 0x1E07, 0x2588, 0xFF42, 
    0x0E1A, 0x307C, 0x30DC, 0x249D, 0x33C3, 0xF8F4, 0x007B, 0xF8F3, 
    0xF8F2, 0xFF5B, 0xFE5B, 0xF8F1, 0xFE37, 0x007D, 0xF8FE, 0xF8FD, 
    0xFF5D, 0xFE5C, 0xF8FC, 0xFE38, 0x005B, 0xF8F0, 0xF8EF, 0xFF3B, 
    0xF8EE, 0x005D, 0xF8FB, 0xF8FA, 0xFF3D, 0xF8F9, 0x02D8, 0x032E, 
    0x0306, 0x032F, 0x0311, 0x0361, 0x032A, 0x033A, 0x00A6, 0x0180, 
    0xF6EA, 0x0183, 0x3076, 0x30D6, 0x2022, 0x25D8, 0x2219, 0x25CE, 
    0x0063, 0x056E, 0x099A, 0x0107, 0x091A, 0x0A9A, 0x0A1A, 0x3388, 
    0x0981, 0x0310, 0x0901, 0x0A81, 0x21EA, 0x2105, 0x02C7, 0x032C, 
    0x030C, 0x21B5, 0x3118, 0x010D, 0x00E7, 0x1E09, 0x24D2, 0x0109, 
    0x0255, 0x010B, 0x010B, 0x33C5, 0x00B8, 0x0327, 0x00A2, 0x2103, 
    0xF6DF, 0xFFE0, 0xF7A2, 0xF6E0, 0x0579, 0x099B, 0x091B, 0x0A9B, 
    0x0A1B, 0x3114, 0x04BD, 0x2713, 0x0447, 0x04BF, 0x04B7, 0x04F5, 
    0x0573, 0x04CC, 0x04B9, 0x03C7, 0x3277, 0x3217, 0x3269, 0x314A, 
    0x3209, 0x0E0A, 0x0E08, 0x0E09, 0x0E0C, 0x0188, 0x3276, 0x3216, 
    0x3268, 0x3148, 0x3208, 0x321C, 0x25CB, 0x2297, 0x2299, 0x2295, 
    0x3036, 0x25D0, 0x25D1, 0x02C6, 0x032D, 0x0302, 0x2327, 0x01C2, 
    0x01C0, 0x01C1, 0x01C3, 0x2663, 0x2663, 0x2667, 0x33A4, 0xFF43, 
    0x33A0, 0x0581, 0x003A, 0x20A1, 0xFF1A, 0x20A1, 0xFE55, 0x02D1, 
    0x02D0, 0x002C, 0x0313, 0x0315, 0xF6C3, 0x060C, 0x055D, 0xF6E1, 
    0xFF0C, 0x0314, 0x02BD, 0xFE50, 0xF6E2, 0x0312, 0x02BB, 0x263C, 
    0x2245, 0x222E, 0x2303, 0x0006, 0x0007, 0x0008, 0x0018, 0x000D, 
    0x0011, 0x0012, 0x0013, 0x0014, 0x007F, 0x0010, 0x0019, 0x0005, 
    0x0004, 0x001B, 0x0017, 0x0003, 0x000C, 0x001C, 0x001D, 0x0009, 
    0x000A, 0x0015, 0x001E, 0x000F, 0x000E, 0x0002, 0x0001, 0x001A, 
    0x0016, 0x001F, 0x000B, 0x00A9, 0xF8E9, 0xF6D9, 0x300C, 0xFF62, 
    0xFE41, 0x300D, 0xFF63, 0xFE42, 0x337F, 0x33C7, 0x33C6, 0x249E, 
    0x20A2, 0x0297, 0x22CF, 0x22CE, 0x00A4, 0xF6D1, 0xF6D2, 0xF6D4, 
    0xF6D5, 0x0064, 0x0564, 0x09A6, 0x0636, 0x0926, 0xFEBE, 0xFEBF, 
    0xFEC0, 0x05BC, 0x05BC, 0x2020, 0x2021, 0x0AA6, 0x0A26, 0x3060, 
    0x30C0, 0x062F, 0x05D3, 0xFB33, 0xFB33, 0x05D3, 0xFEAA, 0x064F, 
    0x064F, 0x064C, 0x064C, 0x0964, 0x05A7, 0x05A7, 0x0485, 0xF6D3, 
    0x300A, 0xFE3D, 0x300B, 0xFE3E, 0x032B, 0x21D4, 0x21D2, 0x0965, 
    0xF6D6, 0x030F, 0x222C, 0x2017, 0x0333, 0x033F, 0x02BA, 0x2016, 
    0x030E, 0x3109, 0x33C8, 0x010F, 0x1E11, 0x24D3, 0x1E13, 0x0111, 
    0x09A1, 0x0921, 0x0AA1, 0x0A21, 0x0688, 0xFB89, 0x095C, 0x09A2, 
    0x0922, 0x0AA2, 0x0A22, 0x1E0B, 0x1E0D, 0x066B, 0x066B, 0x0434, 
    0x00B0, 0x05AD, 0x3067, 0x03EF, 0x30C7, 0x232B, 0x2326, 0x03B4, 
    0x018D, 0x09F8, 0x02A4, 0x09A7, 0x0927, 0x0AA7, 0x0A27, 0x0257, 
    0x0385, 0x0344, 0x2666, 0x2662, 0x00A8, 0xF6D7, 0x0324, 0x0308, 
    0xF6D8, 0x0385, 0x3062, 0x30C2, 0x3003, 0x00F7, 0x2223, 0x2215, 
    0x0452, 0x2593, 0x1E0F, 0x3397, 0x0111, 0xFF44, 0x2584, 0x0E0E, 
    0x0E14, 0x3069, 0x30C9, 0x0024, 0xF6E3, 0xFF04, 0xF724, 0xFE69, 
    0xF6E4, 0x20AB, 0x3326, 0x02D9, 0x0307, 0x0323, 0x0323, 0x30FB, 
    0x0131, 0xF6BE, 0x0284, 0x22C5, 0x25CC, 0xFB1F, 0xFB1F, 0x031E, 
    0x02D5, 0x249F, 0xF6EB, 0x0256, 0x018C, 0x3065, 0x30C5, 0x01F3, 
    0x02A3, 0x01C6, 0x02A5, 0x04E1, 0x0455, 0x045F, 0x0065, 0x00E9, 
    0x2641, 0x098F, 0x311C, 0x0115, 0x090D, 0x0A8D, 0x0945, 0x0AC5, 
    0x011B, 0x1E1D, 0x0565, 0x0587, 0x24D4, 0x00EA, 0x1EBF, 0x1E19, 
    0x1EC7, 0x1EC1, 0x1EC3, 0x1EC5, 0x0454, 0x0205, 0x090F, 0x00EB, 
    0x0117, 0x0117, 0x1EB9, 0x0A0F, 0x0A47, 0x0444, 0x00E8, 0x0A8F, 
    0x0567, 0x311D, 0x3048, 0x1EBB, 0x311F, 0x0038, 0x0668, 0x09EE, 
    0x2467, 0x2791, 0x096E, 0x2471, 0x2485, 0x2499, 0x0AEE, 0x0A6E, 
    0x0668, 0x3028, 0x266B, 0x3227, 0x2088, 0xFF18, 0xF738, 0x247B, 
    0x248F, 0x06F8, 0x2177, 0x2078, 0x0E58, 0x0207, 0x0465, 0x30A8, 
    0xFF74, 0x0A74, 0x3154, 0x043B, 0x2208, 0x246A, 0x247E, 0x2492, 
    0x217A, 0x2026, 0x22EE, 0x0113, 0x1E17, 0x1E15, 0x043C, 0x2014, 
    0xFE31, 0xFF45, 0x055B, 0x2205, 0x3123, 0x043D, 0x2013, 0xFE32, 
    0x04A3, 0x014B, 0x3125, 0x04A5, 0x04C8, 0x2002, 0x0119, 0x3153, 
    0x025B, 0x029A, 0x025C, 0x025E, 0x025D, 0x24A0, 0x03B5, 0x03AD, 
    0x003D, 0xFF1D, 0xFE66, 0x207C, 0x2261, 0x3126, 0x0440, 0x0258, 
    0x044D, 0x0441, 0x04AB, 0x0283, 0x0286, 0x090E, 0x0946, 0x01AA, 
    0x0285, 0x3047, 0x30A7, 0xFF6A, 0x212E, 0xF6EC, 0x03B7, 0x0568, 
    0x03AE, 0x00F0, 0x1EBD, 0x1E1B, 0x0591, 0x0591, 0x0591, 0x0591, 
    0x01DD, 0x3161, 0x20AC, 0x09C7, 0x0947, 0x0AC7, 0x0021, 0x055C, 
    0x203C, 0x00A1, 0xF7A1, 0xFF01, 0xF721, 0x2203, 0x0292, 0x01EF, 
    0x0293, 0x01B9, 0x01BA, 0x0066, 0x095E, 0x0A5E, 0x2109, 0x064E, 
    0x064E, 0x064B, 0x3108, 0x24D5, 0x1E1F, 0x0641, 0x0586, 0xFED2, 
    0xFED3, 0xFED4, 0x03E5, 0x2640, 0xFB00, 0xFB03, 0xFB04, 0xFB01, 
    0x246E, 0x2482, 0x2496, 0x2012, 0x25A0, 0x25AC, 0x05DA, 0xFB3A, 
    0xFB3A, 0x05DA, 0x05DD, 0x05DD, 0x05DF, 0x05DF, 0x05E3, 0x05E3, 
    0x05E5, 0x05E5, 0x02C9, 0x25C9, 0x0473, 0x0035, 0x0665, 0x09EB, 
    0x2464, 0x278E, 0x096B, 0x215D, 0x0AEB, 0x0A6B, 0x0665, 0x3025, 
    0x3224, 0x2085, 0xFF15, 0xF735, 0x2478, 0x248C, 0x06F5, 0x2174, 
    0x2075, 0x0E55, 0xFB02, 0x0192, 0xFF46, 0x3399, 0x0E1F, 0x0E1D, 
    0x0E4F, 0x2200, 0x0034, 0x0664, 0x09EA, 0x2463, 0x278D, 0x096A, 
    0x0AEA, 0x0A6A, 0x0664, 0x3024, 0x3223, 0x2084, 0xFF14, 0x09F7, 
    0xF734, 0x2477, 0x248B, 0x06F4, 0x2173, 0x2074, 0x246D, 0x2481, 
    0x2495, 0x0E54, 0x02CB, 0x24A1, 0x2044, 0x20A3, 0x0067, 0x0997, 
    0x01F5, 0x0917, 0x06AF, 0xFB93, 0xFB94, 0xFB95, 0x0A97, 0x0A17, 
    0x304C, 0x30AC, 0x03B3, 0x0263, 0x02E0, 0x03EB, 0x310D, 0x011F, 
    0x01E7, 0x0123, 0x24D6, 0x011D, 0x0123, 0x0121, 0x0121, 0x0433, 
    0x3052, 0x30B2, 0x2251, 0x059C, 0x05F3, 0x059D, 0x00DF, 0x059E, 
    0x05F4, 0x3013, 0x0998, 0x0572, 0x0918, 0x0A98, 0x0A18, 0x063A, 
    0xFECE, 0xFECF, 0xFED0, 0x0495, 0x0493, 0x0491, 0x095A, 0x0A5A, 
    0x0260, 0x3393, 0x304E, 0x30AE, 0x0563, 0x05D2, 0xFB32, 0xFB32, 
    0x05D2, 0x0453, 0x01BE, 0x0294, 0x0296, 0x02C0, 0x0295, 0x02C1, 
    0x02E4, 0x02A1, 0x02A2, 0x1E21, 0xFF47, 0x3054, 0x30B4, 0x24A2, 
    0x33AC, 0x2207, 0x0060, 0x0316, 0x0300, 0x0300, 0x0953, 0x02CE, 
    0xFF40, 0x0340, 0x003E, 0x2265, 0x22DB, 0xFF1E, 0x2273, 0x2277, 
    0x2267, 0xFE65, 0x0261, 0x01E5, 0x3050, 0x00AB, 0x00BB, 0x2039, 
    0x203A, 0x30B0, 0x3318, 0x33C9, 0x0068, 0x04A9, 0x06C1, 0x09B9, 
    0x04B3, 0x0939, 0x0AB9, 0x0A39, 0x062D, 0xFEA2, 0xFEA3, 0x306F, 
    0xFEA4, 0x332A, 0x30CF, 0xFF8A, 0x0A4D, 0x0621, 0x0621, 0x3164, 
    0x044A, 0x21BC, 0x21C0, 0x33CA, 0x05B2, 0x05B2, 0x05B2, 0x05B2, 
    0x05B2, 0x05B2, 0x05B2, 0x05B2, 0x05B3, 0x05B3, 0x05B3, 0x05B3, 
    0x05B3, 0x05B3, 0x05B3, 0x05B3, 0x05B1, 0x05B1, 0x05B1, 0x05B1, 
    0x05B1, 0x05B1, 0x05B1, 0x05B1, 0x0127, 0x310F, 0x1E2B, 0x1E29, 
    0x24D7, 0x0125, 0x1E27, 0x1E23, 0x1E25, 0x05D4, 0x2665, 0x2665, 
    0x2661, 0xFB34, 0xFB34, 0x06C1, 0x0647, 0x05D4, 0xFBA7, 0xFEEA, 
    0xFEEA, 0xFBA5, 0xFBA4, 0xFBA8, 0xFEEB, 0x3078, 0xFBA9, 0xFEEC, 
    0x337B, 0x30D8, 0xFF8D, 0x3336, 0x0267, 0x3339, 0x05D7, 0x05D7, 
    0x0266, 0x02B1, 0x327B, 0x321B, 0x326D, 0x314E, 0x320D, 0x3072, 
    0x30D2, 0xFF8B, 0x05B4, 0x05B4, 0x05B4, 0x05B4, 0x05B4, 0x05B4, 
    0x05B4, 0x05B4, 0x1E96, 0xFF48, 0x0570, 0x0E2B, 0x307B, 0x30DB, 
    0xFF8E, 0x05B9, 0x05B9, 0x05B9, 0x05B9, 0x05B9, 0x05B9, 0x05B9, 
    0x05B9, 0x0E2E, 0x0309, 0x0309, 0x0321, 0x0322, 0x3342, 0x03E9, 
    0x2015, 0x031B, 0x2668, 0x2302, 0x24A3, 0x02B0, 0x0265, 0x3075, 
    0x3333, 0x30D5, 0xFF8C, 0x02DD, 0x030B, 0x0195, 0x002D, 0xF6E5, 
    0xFF0D, 0xFE63, 0xF6E6, 0x2010, 0x0069, 0x00ED, 0x044F, 0x0987, 
    0x3127, 0x012D, 0x01D0, 0x24D8, 0x00EE, 0x0456, 0x0209, 0x328F, 
    0x328B, 0x323F, 0x323A, 0x32A5, 0x3006, 0x3001, 0xFF64, 0x3237, 
    0x32A3, 0x322F, 0x323D, 0x329D, 0x3240, 0x3296, 0x3236, 0x322B, 
    0x3232, 0x32A4, 0x3005, 0x3298, 0x3238, 0x32A7, 0x32A6, 0x32A9, 
    0x322E, 0x322A, 0x3234, 0x3002, 0x329E, 0x3243, 0x3239, 0x323E, 
    0x32A8, 0x3299, 0x3242, 0x3233, 0x3000, 0x3235, 0x3231, 0x323B, 
    0x3230, 0x323C, 0x322C, 0x322D, 0x3007, 0x328E, 0x328A, 0x3294, 
    0x3290, 0x328C, 0x328D, 0x0907, 0x00EF, 0x1E2F, 0x04E5, 0x1ECB, 
    0x04D7, 0x0435, 0x3275, 0x3215, 0x3267, 0x3147, 0x3207, 0x00EC, 
    0x0A87, 0x0A07, 0x3044, 0x1EC9, 0x0988, 0x0438, 0x0908, 0x0A88, 
    0x0A08, 0x0A40, 0x020B, 0x0439, 0x09C0, 0x0940, 0x0AC0, 0x0133, 
    0x30A4, 0xFF72, 0x3163, 0x02DC, 0x05AC, 0x012B, 0x04E3, 0x2253, 
    0x0A3F, 0xFF49, 0x2206, 0x221E, 0x056B, 0x222B, 0x2321, 0x2321, 
    0xF8F5, 0x2320, 0x2320, 0x2229, 0x3305, 0x25D8, 0x25D9, 0x263B, 
    0x0451, 0x012F, 0x03B9, 0x03CA, 0x0390, 0x0269, 0x03AF, 0x24A4, 
    0x0A72, 0x3043, 0x30A3, 0xFF68, 0x09FA, 0x0268, 0xF6ED, 0x309D, 
    0x30FD, 0x0129, 0x1E2D, 0x3129, 0x044E, 0x09BF, 0x093F, 0x0ABF, 
    0x0475, 0x0477, 0x006A, 0x0571, 0x099C, 0x091C, 0x0A9C, 0x0A1C, 
    0x3110, 0x01F0, 0x24D9, 0x0135, 0x029D, 0x025F, 0x0458, 0x062C, 
    0xFE9E, 0xFE9F, 0xFEA0, 0x0698, 0xFB8B, 0x099D, 0x091D, 0x0A9D, 
    0x0A1D, 0x057B, 0x3004, 0xFF4A, 0x24A5, 0x02B2, 0x006B, 0x04A1, 
    0x0995, 0x1E31, 0x043A, 0x049B, 0x0915, 0x05DB, 0x0643, 0xFB3B, 
    0xFB3B, 0xFEDA, 0x05DB, 0xFEDB, 0xFEDC, 0xFB4D, 0x0A95, 0x0A15, 
    0x304B, 0x04C4, 0x30AB, 0xFF76, 0x03BA, 0x03F0, 0x3171, 0x3184, 
    0x3178, 0x3179, 0x330D, 0x0640, 0x0640, 0x30F5, 0x3384, 0x0650, 
    0x064D, 0x049F, 0xFF70, 0x049D, 0x310E, 0x3389, 0x01E9, 0x0137, 
    0x24DA, 0x0137, 0x1E33, 0x0584, 0x3051, 0x30B1, 0xFF79, 0x056F, 
    0x30F6, 0x0138, 0x0996, 0x0445, 0x0916, 0x0A96, 0x0A16, 0x062E, 
    0xFEA6, 0xFEA7, 0xFEA8, 0x03E7, 0x0959, 0x0A59, 0x3278, 0x3218, 
    0x326A, 0x314B, 0x320A, 0x0E02, 0x0E05, 0x0E03, 0x0E04, 0x0E5B, 
    0x0199, 0x0E06, 0x3391, 0x304D, 0x30AD, 0xFF77, 0x3315, 0x3316, 
    0x3314, 0x326E, 0x320E, 0x3260, 0x3131, 0x3200, 0x3133, 0x045C, 
    0x1E35, 0x3398, 0x33A6, 0xFF4B, 0x33A2, 0x3053, 0x33C0, 0x0E01, 
    0x30B3, 0xFF7A, 0x331E, 0x0481, 0x327F, 0x0343, 0x24A6, 0x33AA, 
    0x046F, 0x33CF, 0x029E, 0x304F, 0x30AF, 0xFF78, 0x33B8, 0x33BE, 
    0x006C, 0x09B2, 0x013A, 0x0932, 0x0AB2, 0x0A32, 0x0E45, 0xFEFC, 
    0xFEF8, 0xFEF7, 0xFEFA, 0xFEF9, 0xFEFB, 0xFEF6, 0xFEF5, 0x0644, 
    0x03BB, 0x019B, 0x05DC, 0xFB3C, 0xFB3C, 0x05DC, 0xFEDE, 0xFCCA, 
    0xFEDF, 0xFCC9, 0xFCCB, 0xFDF2, 0xFEE0, 0xFD88, 0xFCCC, 0x25EF, 
    0x019A, 0x026C, 0x310C, 0x013E, 0x013C, 0x24DB, 0x1E3D, 0x013C, 
    0x0140, 0x0140, 0x1E37, 0x1E39, 0x031A, 0x0318, 0x003C, 0x2264, 
    0x22DA, 0xFF1C, 0x2272, 0x2276, 0x2266, 0xFE64, 0x026E, 0x258C, 
    0x026D, 0x20A4, 0x056C, 0x01C9, 0x0459, 0xF6C0, 0x0933, 0x0AB3, 
    0x1E3B, 0x0934, 0x09E1, 0x0961, 0x09E3, 0x0963, 0x026B, 0xFF4C, 
    0x33D0, 0x0E2C, 0x2227, 0x00AC, 0x2310, 0x2228, 0x0E25, 0x017F, 
    0xFE4E, 0x0332, 0xFE4D, 0x25CA, 0x24A7, 0x0142, 0x2113, 0xF6EE, 
    0x2591, 0x0E26, 0x098C, 0x090C, 0x09E2, 0x0962, 0x33D3, 0x006D, 
    0x09AE, 0x00AF, 0x0331, 0x0304, 0x02CD, 0xFFE3, 0x1E3F, 0x092E, 
    0x0AAE, 0x0A2E, 0x05A4, 0x05A4, 0x307E, 0xF895, 0xF894, 0x0E4B, 
    0xF893, 0xF88C, 0xF88B, 0x0E48, 0xF88A, 0xF884, 0x0E31, 0xF889, 
    0x0E47, 0xF88F, 0xF88E, 0x0E49, 0xF88D, 0xF892, 0xF891, 0x0E4A, 
    0xF890, 0x0E46, 0x30DE, 0xFF8F, 0x2642, 0x3347, 0x05BE, 0x2642, 
    0x05AF, 0x3383, 0x3107, 0x33D4, 0x24DC, 0x33A5, 0x1E41, 0x1E43, 
    0x0645, 0xFEE2, 0xFEE3, 0xFEE4, 0xFCD1, 0xFC48, 0x334D, 0x3081, 
    0x337E, 0x30E1, 0xFF92, 0x05DE, 0xFB3E, 0xFB3E, 0x05DE, 0x0574, 
    0x05A5, 0x05A6, 0x05A6, 0x05A5, 0x0271, 0x3392, 0xFF65, 0x00B7, 
    0x3272, 0x3212, 0x3264, 0x3141, 0x3170, 0x3204, 0x316E, 0x316F, 
    0x307F, 0x30DF, 0xFF90, 0x2212, 0x0320, 0x2296, 0x02D7, 0x2213, 
    0x2032, 0x334A, 0x3349, 0x0270, 0x3396, 0x33A3, 0xFF4D, 0x339F, 
    0x3082, 0x33C1, 0x30E2, 0xFF93, 0x33D6, 0x0E21, 0x33A7, 0x33A8, 
    0x24A8, 0x33AB, 0x33B3, 0xF6EF, 0x026F, 0x00B5, 0x00B5, 0x3382, 
    0x226B, 0x226A, 0x338C, 0x03BC, 0x338D, 0x3080, 0x30E0, 0xFF91, 
    0x3395, 0x00D7, 0x339B, 0x05A3, 0x05A3, 0x266A, 0x266B, 0x266D, 
    0x266F, 0x33B2, 0x33B6, 0x33BC, 0x33B9, 0x33B7, 0x33BF, 0x33BD, 
    0x006E, 0x09A8, 0x2207, 0x0144, 0x0928, 0x0AA8, 0x0A28, 0x306A, 
    0x30CA, 0xFF85, 0x0149, 0x3381, 0x310B, 0x00A0, 0x0148, 0x0146, 
    0x24DD, 0x1E4B, 0x0146, 0x1E45, 0x1E47, 0x306D, 0x30CD, 0xFF88, 
    0x20AA, 0x338B, 0x0999, 0x0919, 0x0A99, 0x0A19, 0x0E07, 0x3093, 
    0x0272, 0x0273, 0x326F, 0x320F, 0x3135, 0x3261, 0x3136, 0x3134, 
    0x3168, 0x3201, 0x3167, 0x3166, 0x306B, 0x30CB, 0xFF86, 0xF899, 
    0x0E4D, 0x0039, 0x0669, 0x09EF, 0x2468, 0x2792, 0x096F, 0x0AEF, 
    0x0A6F, 0x0669, 0x3029, 0x3228, 0x2089, 0xFF19, 0xF739, 0x247C, 
    0x2490, 0x06F9, 0x2178, 0x2079, 0x2472, 0x2486, 0x249A, 0x0E59, 
    0x01CC, 0x045A, 0x30F3, 0xFF9D, 0x019E, 0x1E49, 0xFF4E, 0x339A, 
    0x09A3, 0x0923, 0x0AA3, 0x0A23, 0x0929, 0x306E, 0x30CE, 0xFF89, 
    0x00A0, 0x0E13, 0x0E19, 0x0646, 0xFEE6, 0x06BA, 0xFB9F, 0xFEE7, 
    0xFCD2, 0xFC4B, 0xFEE8, 0xFCD5, 0xFC4E, 0xFC8D, 0x220C, 0x2209, 
    0x2209, 0x2260, 0x226F, 0x2271, 0x2279, 0x2262, 0x226E, 0x2270, 
    0x2226, 0x2280, 0x2284, 0x2281, 0x2285, 0x0576, 0x24A9, 0x33B1, 
    0x207F, 0x00F1, 0x03BD, 0x306C, 0x30CC, 0xFF87, 0x09BC, 0x093C, 
    0x0ABC, 0x0A3C, 0x0023, 0xFF03, 0xFE5F, 0x0374, 0x0375, 0x2116, 
    0x05E0, 0xFB40, 0xFB40, 0x05E0, 0x33B5, 0x33BB, 0x099E, 0x091E, 
    0x0A9E, 0x0A1E, 0x006F, 0x00F3, 0x0E2D, 0x0275, 0x04E9, 0x04EB, 
    0x0993, 0x311B, 0x014F, 0x0911, 0x0A91, 0x0949, 0x0AC9, 0x01D2, 
    0x24DE, 0x00F4, 0x1ED1, 0x1ED9, 0x1ED3, 0x1ED5, 0x1ED7, 0x043E, 
    0x0151, 0x020D, 0x0913, 0x00F6, 0x04E7, 0x1ECD, 0x0153, 0x315A, 
    0x02DB, 0x0328, 0x00F2, 0x0A93, 0x0585, 0x304A, 0x1ECF, 0x01A1, 
    0x1EDB, 0x1EE3, 0x1EDD, 0x1EDF, 0x1EE1, 0x0151, 0x01A3, 0x020F, 
    0x30AA, 0xFF75, 0x3157, 0x05AB, 0x014D, 0x1E53, 0x1E51, 0x0950, 
    0x03C9, 0x03D6, 0x0461, 0x0277, 0x047B, 0x047D, 0x03CE, 0x0AD0, 
    0x03BF, 0x03CC, 0xFF4F, 0x0031, 0x0661, 0x09E7, 0x2460, 0x278A, 
    0x0967, 0x2024, 0x215B, 0xF6DC, 0x0AE7, 0x0A67, 0x0661, 0x00BD, 
    0x3021, 0x3220, 0x2081, 0xFF11, 0x09F4, 0xF731, 0x2474, 0x2488, 
    0x06F1, 0x00BC, 0x2170, 0x00B9, 0x0E51, 0x2153, 0x01EB, 0x01ED, 
    0x0A13, 0x0A4B, 0x0254, 0x24AA, 0x25E6, 0x2325, 0x00AA, 0x00BA, 
    0x221F, 0x0912, 0x094A, 0x00F8, 0x01FF, 0x3049, 0x30A9, 0xFF6B, 
    0x01FF, 0xF6F0, 0x047F, 0x00F5, 0x1E4D, 0x1E4F, 0x3121, 0x203E, 
    0xFE4A, 0x0305, 0xFE49, 0xFE4C, 0xFE4B, 0x00AF, 0x09CB, 0x094B, 
    0x0ACB, 0x0070, 0x3380, 0x332B, 0x09AA, 0x1E55, 0x092A, 0x21DF, 
    0x21DE, 0x0AAA, 0x0A2A, 0x3071, 0x0E2F, 0x30D1, 0x0484, 0x04C0, 
    0x317F, 0x00B6, 0x2225, 0x0028, 0xFD3E, 0xF8ED, 0xF8EC, 0x208D, 
    0xFF08, 0xFE59, 0x207D, 0xF8EB, 0xFE35, 0x0029, 0xFD3F, 0xF8F8, 
    0xF8F7, 0x208E, 0xFF09, 0xFE5A, 0x207E, 0xF8F6, 0xFE36, 0x2202, 
    0x05C0, 0x0599, 0x33A9, 0x05B7, 0x05B7, 0x05B7, 0x05B7, 0x05B7, 
    0x05B7, 0x05B7, 0x05B7, 0x05A1, 0x3106, 0x24DF, 0x1E57, 0x05E4, 
    0x043F, 0xFB44, 0xFB44, 0x333B, 0xFB43, 0x067E, 0x057A, 0x05E4, 
    0xFB57, 0xFB58, 0x307A, 0xFB59, 0x30DA, 0x04A7, 0xFB4E, 0x0025, 
    0x066A, 0xFF05, 0xFE6A, 0x002E, 0x0589, 0x00B7, 0xFF61, 0xF6E7, 
    0xFF0E, 0xFE52, 0xF6E8, 0x0342, 0x22A5, 0x2030, 0x20A7, 0x338A, 
    0x09AB, 0x092B, 0x0AAB, 0x0A2B, 0x03C6, 0x03D5, 0x327A, 0x321A, 
    0x326C, 0x314D, 0x320C, 0x0278, 0x0E3A, 0x03D5, 0x01A5, 0x0E1E, 
    0x0E1C, 0x0E20, 0x03C0, 0x3273, 0x3213, 0x3176, 0x3265, 0x3172, 
    0x3142, 0x3205, 0x3174, 0x3144, 0x3175, 0x3177, 0x3173, 0x3074, 
    0x30D4, 0x03D6, 0x0583, 0x002B, 0x031F, 0x2295, 0x00B1, 0x02D6, 
    0xFF0B, 0xFE62, 0x207A, 0xFF50, 0x33D8, 0x307D, 0x261F, 0x261C, 
    0x261E, 0x261D, 0x30DD, 0x0E1B, 0x3012, 0x3020, 0x24AB, 0x227A, 
    0x211E, 0x02B9, 0x2035, 0x220F, 0x2305, 0x30FC, 0x2318, 0x2282, 
    0x2283, 0x2237, 0x221D, 0x03C8, 0x0471, 0x0486, 0x33B0, 0x3077, 
    0x30D7, 0x33B4, 0x33BA, 0x0071, 0x0958, 0x05A8, 0x0642, 0xFED6, 
    0xFED7, 0xFED8, 0x05B8, 0x05B8, 0x05B8, 0x05B8, 0x05B8, 0x05B8, 
    0x05B8, 0x05B8, 0x05B8, 0x05B8, 0x05B8, 0x05B8, 0x05B8, 0x05B8, 
    0x05B8, 0x05B8, 0x059F, 0x3111, 0x24E0, 0x02A0, 0xFF51, 0x05E7, 
    0xFB47, 0xFB47, 0x05E7, 0x24AC, 0x2669, 0x05BB, 0x05BB, 0x05BB, 
    0x05BB, 0x05BB, 0x05BB, 0x05BB, 0x05BB, 0x003F, 0x061F, 0x055E, 
    0x00BF, 0xF7BF, 0x037E, 0xFF1F, 0xF73F, 0x0022, 0x201E, 0x201C, 
    0xFF02, 0x301E, 0x301D, 0x201D, 0x2018, 0x201B, 0x201B, 0x2019, 
    0x0149, 0x201A, 0x0027, 0xFF07, 0x0072, 0x057C, 0x09B0, 0x0155, 
    0x0930, 0x221A, 0xF8E5, 0x33AE, 0x33AF, 0x33AD, 0x05BF, 0x05BF, 
    0x0AB0, 0x0A30, 0x3089, 0x30E9, 0xFF97, 0x09F1, 0x09F0, 0x0264, 
    0x2236, 0x3116, 0x0159, 0x0157, 0x24E1, 0x0157, 0x0211, 0x1E59, 
    0x1E5B, 0x1E5D, 0x203B, 0x2286, 0x2287, 0x00AE, 0xF8E8, 0xF6DA, 
    0x0631, 0x0580, 0xFEAE, 0x308C, 0x30EC, 0xFF9A, 0x05E8, 0xFB48, 
    0x05E8, 0x223D, 0x0597, 0x0597, 0x2310, 0x027E, 0x027F, 0x09DD, 
    0x095D, 0x03C1, 0x027D, 0x027B, 0x02B5, 0x03F1, 0x02DE, 0x3271, 
    0x3211, 0x3263, 0x3140, 0x313A, 0x3169, 0x3139, 0x313B, 0x316C, 
    0x3203, 0x313F, 0x313C, 0x316B, 0x313D, 0x313E, 0x316A, 0x316D, 
    0x221F, 0x0319, 0x22BF, 0x308A, 0x30EA, 0xFF98, 0x02DA, 0x0325, 
    0x030A, 0x02BF, 0x0559, 0x031C, 0x02D3, 0x02BE, 0x0339, 0x02D2, 
    0x0213, 0x3351, 0x1E5F, 0x027C, 0x027A, 0xFF52, 0x308D, 0x30ED, 
    0xFF9B, 0x0E23, 0x24AD, 0x09DC, 0x0931, 0x0A5C, 0x0691, 0xFB8D, 
    0x09E0, 0x0960, 0x0AE0, 0x09C4, 0x0944, 0x0AC4, 0xF6F1, 0x2590, 
    0x0279, 0x02B4, 0x308B, 0x30EB, 0xFF99, 0x09F2, 0x09F3, 0xF6DD, 
    0x0E24, 0x098B, 0x090B, 0x0A8B, 0x09C3, 0x0943, 0x0AC3, 0x0073, 
    0x09B8, 0x015B, 0x1E65, 0x0635, 0x0938, 0xFEBA, 0xFEBB, 0xFEBC, 
    0x0AB8, 0x0A38, 0x3055, 0x30B5, 0xFF7B, 0xFDFA, 0x05E1, 0xFB41, 
    0xFB41, 0x05E1, 0x0E32, 0x0E41, 0x0E44, 0x0E43, 0x0E33, 0x0E30, 
    0x0E40, 0xF886, 0x0E35, 0xF885, 0x0E34, 0x0E42, 0xF888, 0x0E37, 
    0xF887, 0x0E36, 0x0E38, 0x0E39, 0x3119, 0x0161, 0x1E67, 0x015F, 
    0x0259, 0x04D9, 0x04DB, 0x025A, 0x24E2, 0x015D, 0x0219, 0x1E61, 
    0x1E63, 0x1E69, 0x033C, 0x2033, 0x02CA, 0x00A7, 0x0633, 0xFEB2, 
    0xFEB3, 0xFEB4, 0x05B6, 0x05B6, 0x05B6, 0x05B6, 0x05B6, 0x05B6, 
    0x05B6, 0x0592, 0x05B6, 0x057D, 0x305B, 0x30BB, 0xFF7E, 0x003B, 
    0x061B, 0xFF1B, 0xFE54, 0x309C, 0xFF9F, 0x3322, 0x3323, 0x0037, 
    0x0667, 0x09ED, 0x2466, 0x2790, 0x096D, 0x215E, 0x0AED, 0x0A6D, 
    0x0667, 0x3027, 0x3226, 0x2087, 0xFF17, 0xF737, 0x247A, 0x248E, 
    0x06F7, 0x2176, 0x2077, 0x2470, 0x2484, 0x2498, 0x0E57, 0x00AD, 
    0x0577, 0x09B6, 0x0448, 0x0651, 0xFC61, 0xFC5E, 0xFC60, 0xFC62, 
    0xFC5F, 0x2592, 0x2593, 0x2591, 0x2592, 0x0936, 0x0AB6, 0x0A36, 
    0x0593, 0x3115, 0x0449, 0x0634, 0xFEB6, 0xFEB7, 0xFEB8, 0x03E3, 
    0x20AA, 0x20AA, 0x05B0, 0x05B0, 0x05B0, 0x05B0, 0x05B0, 0x05B0, 
    0x05B0, 0x05B0, 0x05B0, 0x04BB, 0x03ED, 0x05E9, 0xFB49, 0xFB49, 
    0xFB2C, 0xFB2C, 0xFB2D, 0xFB2D, 0x05C1, 0x05E9, 0xFB2A, 0xFB2A, 
    0xFB2B, 0xFB2B, 0x0282, 0x03C3, 0x03C2, 0x03C2, 0x03F2, 0x3057, 
    0x30B7, 0xFF7C, 0x05BD, 0x05BD, 0x223C, 0x05C2, 0x3274, 0x3214, 
    0x317E, 0x3266, 0x317A, 0x3145, 0x317B, 0x3206, 0x317D, 0x317C, 
    0x0036, 0x0666, 0x09EC, 0x2465, 0x278F, 0x096C, 0x0AEC, 0x0A6C, 
    0x0666, 0x3026, 0x3225, 0x2086, 0xFF16, 0xF736, 0x2479, 0x248D, 
    0x06F6, 0x2175, 0x2076, 0x246F, 0x09F9, 0x2483, 0x2497, 0x0E56, 
    0x002F, 0xFF0F, 0x017F, 0x1E9B, 0x263A, 0xFF53, 0x05C3, 0x00AD, 
    0x044C, 0x305D, 0x30BD, 0xFF7F, 0x0338, 0x0337, 0x0E29, 0x0E28, 
    0x0E0B, 0x0E2A, 0x0020, 0x0020, 0x2660, 0x2660, 0x2664, 0x24AE, 
    0x033B, 0x33C4, 0x339D, 0x25A9, 0x25A4, 0x338F, 0x339E, 0x33CE, 
    0x33D1, 0x33D2, 0x338E, 0x33D5, 0x339C, 0x33A1, 0x25A6, 0x25A7, 
    0x25A8, 0x25A5, 0x25A3, 0x33DB, 0x09B7, 0x0937, 0x0AB7, 0x3149, 
    0x3185, 0x3180, 0x3132, 0x3165, 0x3143, 0x3146, 0x3138, 0xF6F2, 
    0x00A3, 0xFFE1, 0x0336, 0x0335, 0x2282, 0x228A, 0x2286, 0x227B, 
    0x220B, 0x3059, 0x30B9, 0xFF7D, 0x0652, 0x2211, 0x263C, 0x2283, 
    0x228B, 0x2287, 0x33DC, 0x337C, 0x0074, 0x09A4, 0x22A4, 0x22A3, 
    0x0924, 0x0AA4, 0x0A24, 0x0637, 0xFEC2, 0xFEC3, 0x305F, 0xFEC4, 
    0x337D, 0x30BF, 0xFF80, 0x0640, 0x03C4, 0x05EA, 0xFB4A, 0xFB4A, 
    0xFB4A, 0x05EA, 0x0167, 0x310A, 0x0165, 0x02A8, 0x0163, 0x0686, 
    0xFB7B, 0xFB7C, 0xFB7D, 0x24E3, 0x1E71, 0x0163, 0x1E97, 0x1E6B, 
    0x1E6D, 0x0442, 0x04AD, 0x062A, 0xFE96, 0xFCA2, 0xFC0C, 0xFE97, 
    0x3066, 0xFCA1, 0xFC0B, 0x0629, 0xFE94, 0xFE98, 0xFCA4, 0xFC0E, 
    0xFC73, 0x30C6, 0xFF83, 0x2121, 0x260E, 0x05A0, 0x05A9, 0x2469, 
    0x3229, 0x247D, 0x2491, 0x2179, 0x02A7, 0x05D8, 0xFB38, 0xFB38, 
    0x05D8, 0x04B5, 0x059B, 0x059B, 0x09A5, 0x0925, 0x0AA5, 0x0A25, 
    0x0630, 0xFEAC, 0xF898, 0xF897, 0x0E4C, 0xF896, 0x062B, 0xFE9A, 
    0xFE9B, 0xFE9C, 0x2203, 0x2234, 0x03B8, 0x03D1, 0x03D1, 0x3279, 
    0x3219, 0x326B, 0x314C, 0x320B, 0x246C, 0x2480, 0x2494, 0x0E11, 
    0x01AD, 0x0E12, 0x00FE, 0x0E17, 0x0E10, 0x0E18, 0x0E16, 0x0482, 
    0x066C, 0x066C, 0x0033, 0x0663, 0x09E9, 0x2462, 0x278C, 0x0969, 
    0x215C, 0x0AE9, 0x0A69, 0x0663, 0x3023, 0x3222, 0x2083, 0xFF13, 
    0x09F6, 0xF733, 0x2476, 0x248A, 0x06F3, 0x00BE, 0xF6DE, 0x2172, 
    0x00B3, 0x0E53, 0x3394, 0x3061, 0x30C1, 0xFF81, 0x3270, 0x3210, 
    0x3262, 0x3137, 0x3202, 0x02DC, 0x0330, 0x0303, 0x0303, 0x0360, 
    0x223C, 0x0334, 0x033E, 0x2297, 0x0596, 0x0596, 0x0A70, 0x0483, 
    0x057F, 0x1E6F, 0xFF54, 0x0569, 0x3068, 0x30C8, 0xFF84, 0x02E5, 
    0x02E9, 0x02E6, 0x02E8, 0x02E7, 0x01BD, 0x0185, 0x01A8, 0x0384, 
    0x3327, 0x0E0F, 0x3014, 0xFE5D, 0xFE39, 0x3015, 0xFE5E, 0xFE3A, 
    0x0E15, 0x01AB, 0x24AF, 0x2122, 0xF8EA, 0xF6DB, 0x0288, 0x25BC, 
    0x25C4, 0x25BA, 0x25B2, 0x02A6, 0x05E6, 0xFB46, 0xFB46, 0x05E6, 
    0x0446, 0x05B5, 0x05B5, 0x05B5, 0x05B5, 0x05B5, 0x05B5, 0x05B5, 
    0x05B5, 0x045B, 0xF6F3, 0x099F, 0x091F, 0x0A9F, 0x0A1F, 0x0679, 
    0xFB67, 0xFB68, 0xFB69, 0x09A0, 0x0920, 0x0AA0, 0x0A20, 0x0287, 
    0x3064, 0x30C4, 0xFF82, 0x3063, 0x30C3, 0xFF6F, 0x246B, 0x247F, 
    0x2493, 0x217B, 0x2473, 0x5344, 0x2487, 0x249B, 0x0032, 0x0662, 
    0x09E8, 0x2461, 0x278B, 0x0968, 0x2025, 0x2025, 0xFE30, 0x0AE8, 
    0x0A68, 0x0662, 0x3022, 0x3221, 0x2082, 0xFF12, 0x09F5, 0xF732, 
    0x2475, 0x2489, 0x06F2, 0x2171, 0x01BB, 0x00B2, 0x0E52, 0x2154, 
    0x0075, 0x00FA, 0x0289, 0x0989, 0x3128, 0x016D, 0x01D4, 0x24E4, 
    0x00FB, 0x1E77, 0x0443, 0x0951, 0x0171, 0x0215, 0x0909, 0x00FC, 
    0x01D8, 0x1E73, 0x01DA, 0x04F1, 0x01DC, 0x01D6, 0x1EE5, 0x00F9, 
    0x0A89, 0x0A09, 0x3046, 0x1EE7, 0x01B0, 0x1EE9, 0x1EF1, 0x1EEB, 
    0x1EED, 0x1EEF, 0x0171, 0x04F3, 0x0217, 0x30A6, 0xFF73, 0x0479, 
    0x315C, 0x016B, 0x04EF, 0x1E7B, 0x0A41, 0xFF55, 0x005F, 0x2017, 
    0xFF3F, 0xFE33, 0xFE4F, 0x222A, 0x2200, 0x0173, 0x24B0, 0x2580, 
    0x05C4, 0x03C5, 0x03CB, 0x03B0, 0x028A, 0x03CD, 0x031D, 0x02D4, 
    0x0A73, 0x016F, 0x045E, 0x3045, 0x30A5, 0xFF69, 0x04AF, 0x04B1, 
    0x0169, 0x1E79, 0x1E75, 0x098A, 0x090A, 0x0A8A, 0x0A0A, 0x0A42, 
    0x09C2, 0x0942, 0x0AC2, 0x09C1, 0x0941, 0x0AC1, 0x0076, 0x0935, 
    0x0AB5, 0x0A35, 0x30F7, 0x05D5, 0xFB35, 0xFB35, 0xFB35, 0x05D5, 
    0xFB4B, 0xFB4B, 0x05F0, 0x05F1, 0x24E5, 0x1E7F, 0x0432, 0x06A4, 
    0xFB6B, 0xFB6C, 0xFB6D, 0x30F9, 0x2640, 0x007C, 0x030D, 0x0329, 
    0x02CC, 0x02C8, 0x057E, 0x028B, 0x30F8, 0x09CD, 0x094D, 0x0ACD, 
    0x0983, 0x0903, 0x0A83, 0xFF56, 0x0578, 0x309E, 0x30FE, 0x309B, 
    0xFF9E, 0x30FA, 0x24B1, 0x1E7D, 0x028C, 0x3094, 0x30F4, 0x0077, 
    0x1E83, 0x3159, 0x308F, 0x30EF, 0xFF9C, 0x3158, 0x308E, 0x30EE, 
    0x3357, 0x301C, 0xFE34, 0x0648, 0xFEEE, 0x0624, 0xFE86, 0x33DD, 
    0x24E6, 0x0175, 0x1E85, 0x1E87, 0x1E89, 0x3091, 0x2118, 0x30F1, 
    0x315E, 0x315D, 0x1E81, 0x25E6, 0x25CB, 0x25D9, 0x300E, 0xFE43, 
    0x300F, 0xFE44, 0x25C7, 0x25C8, 0x25BF, 0x25BD, 0x25C3, 0x25C1, 
    0x3016, 0x3017, 0x25B9, 0x25B7, 0x25AB, 0x263A, 0x25A1, 0x2606, 
    0x260F, 0x3018, 0x3019, 0x25B5, 0x25B3, 0x3090, 0x30F0, 0x315F, 
    0xFF57, 0x3092, 0x30F2, 0xFF66, 0x20A9, 0xFFE6, 0x0E27, 0x24B2, 
    0x1E98, 0x02B7, 0x028D, 0x01BF, 0x0078, 0x033D, 0x3112, 0x24E7, 
    0x1E8D, 0x1E8B, 0x056D, 0x03BE, 0xFF58, 0x24B3, 0x02E3, 0x0079, 
    0x334E, 0x09AF, 0x00FD, 0x092F, 0x3152, 0x0AAF, 0x0A2F, 0x3084, 
    0x30E4, 0xFF94, 0x3151, 0x0E4E, 0x3083, 0x30E3, 0xFF6C, 0x0463, 
    0x24E8, 0x0177, 0x00FF, 0x1E8F, 0x1EF5, 0x064A, 0x06D2, 0xFBAF, 
    0xFEF2, 0x0626, 0xFE8A, 0xFE8B, 0xFE8C, 0xFEF3, 0xFEF4, 0xFCDD, 
    0xFC58, 0xFC94, 0x06D1, 0x3156, 0x00A5, 0xFFE5, 0x3155, 0x3186, 
    0x05AA, 0x05AA, 0x044B, 0x04F9, 0x3181, 0x3183, 0x3182, 0x059A, 
    0x1EF3, 0x01B4, 0x1EF7, 0x0575, 0x0457, 0x3162, 0x262F, 0x0582, 
    0xFF59, 0x05D9, 0xFB39, 0xFB39, 0x05D9, 0x05F2, 0xFB1F, 0x3088, 
    0x3189, 0x30E8, 0xFF96, 0x315B, 0x3087, 0x30E7, 0xFF6E, 0x03F3, 
    0x3188, 0x3187, 0x0E22, 0x0E0D, 0x24B4, 0x037A, 0x0345, 0x01A6, 
    0x1E99, 0x02B8, 0x1EF9, 0x028E, 0x3086, 0x318C, 0x30E6, 0xFF95, 
    0x3160, 0x046B, 0x046D, 0x0467, 0x0469, 0x3085, 0x30E5, 0xFF6D, 
    0x318B, 0x318A, 0x09DF, 0x095F, 0x007A, 0x0566, 0x017A, 0x095B, 
    0x0A5B, 0x0638, 0xFEC6, 0xFEC7, 0x3056, 0xFEC8, 0x0632, 0xFEB0, 
    0x30B6, 0x0595, 0x0594, 0x0598, 0x05D6, 0xFB36, 0xFB36, 0x05D6, 
    0x3117, 0x017E, 0x24E9, 0x1E91, 0x0291, 0x017C, 0x017C, 0x1E93, 
    0x0437, 0x0499, 0x04DF, 0x305C, 0x30BC, 0x0030, 0x0660, 0x09E6, 
    0x0966, 0x0AE6, 0x0A66, 0x0660, 0x2080, 0xFF10, 0xF730, 0x06F0, 
    0x2070, 0x0E50, 0xFEFF, 0x200C, 0x200B, 0x03B6, 0x3113, 0x056A, 
    0x04C2, 0x0436, 0x0497, 0x04DD, 0x3058, 0x30B8, 0x05AE, 0x1E95, 
    0xFF5A, 0x305E, 0x30BE, 0x24B5, 0x0290, 0x01B6, 0x305A, 0x30BA, 
};

wchar_t ps_glyph_to_unicode(glyph g)
{
    if (g == NOGLYPH) return 0xFFFF;
    return ps_codes_alphabetic[g];
}

/* ----------------------------------------------------------------------
 * Data about the standard fonts: available glyphs and font metrics.
 */

/*
 * Character set list extracted from the back of the PostScript
 * Language Reference Manual.
 * 
 * I'm going to include a bit of shell which creates a file
 * containing the list of characters, and then several further bits
 * of shell that process it in different ways.
 */

/*

tr -s ' \n' '\n' << EOF > stdchars.txt
A B C D E F G H I J K L M N O P Q R S T U V W X Y Z
a b c d e f g h i j k l m n o p q r s t u v w x y z
AE Aacute Acircumflex Adieresis Agrave Aring Atilde
Ccedilla
Eacute Ecircumflex Edieresis Egrave
Eth
Iacute Icircumflex Idieresis Igrave
Lslash
Ntilde
OE Oacute Ocircumflex Odieresis Ograve Oslash Otilde
Scaron
Thorn
Uacute Ucircumflex Udieresis Ugrave
Yacute Ydieresis
Zcaron
ae aacute acircumflex adieresis agrave aring atilde
ccedilla
eacute ecircumflex edieresis egrave
eth
iacute icircumflex idieresis igrave
lslash
ntilde
oe oacute ocircumflex odieresis ograve oslash otilde
scaron
thorn
uacute ucircumflex udieresis ugrave
yacute ydieresis
zcaron
acute ampersand asciicircum asciitilde asterisk at backslash bar
braceleft braceright bracketleft bracketright breve brokenbar bullet
caron cedilla cent circumflex colon comma copyright currency dagger
daggerdbl degree dieresis divide dollar dotaccent dotlessi eight
ellipsis emdash endash equal exclam exclamdown fi five fl florin
four fraction germandbls grave greater guillemotleft guillemotright
guilsinglleft guilsinglright hungarumlaut hyphen less logicalnot
macron minus mu multiply nine numbersign ogonek one onehalf onequarter
onesuperior ordfeminine ordmasculine paragraph parenleft parenright
percent period periodcentered perthousand plus plusminus question
questiondown quotedbl quotedblbase quotedblleft quotedblright quoteleft
quoteright quotesinglbase quotesingle registered ring section semicolon
seven six slash space sterling three threequarters threesuperior tilde
trademark two twosuperior underscore yen zero
EOF

 */

/*
 * A simple string array giving the glyph list.
 */

/*

perl -ne 'chomp; print "\"$_\", "; END { print "NULL" }' stdchars.txt | \
    fold -s -w68 | sed 's/^/    /'

*/

const char *const ps_std_glyphs[] = {
    "A", "B", "C", "D", "E", "F", "G", "H", "I", "J", "K", "L", "M", 
    "N", "O", "P", "Q", "R", "S", "T", "U", "V", "W", "X", "Y", "Z", 
    "a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l", "m", 
    "n", "o", "p", "q", "r", "s", "t", "u", "v", "w", "x", "y", "z", 
    "AE", "Aacute", "Acircumflex", "Adieresis", "Agrave", "Aring", 
    "Atilde", "Ccedilla", "Eacute", "Ecircumflex", "Edieresis", 
    "Egrave", "Eth", "Iacute", "Icircumflex", "Idieresis", "Igrave", 
    "Lslash", "Ntilde", "OE", "Oacute", "Ocircumflex", "Odieresis", 
    "Ograve", "Oslash", "Otilde", "Scaron", "Thorn", "Uacute", 
    "Ucircumflex", "Udieresis", "Ugrave", "Yacute", "Ydieresis", 
    "Zcaron", "ae", "aacute", "acircumflex", "adieresis", "agrave", 
    "aring", "atilde", "ccedilla", "eacute", "ecircumflex", 
    "edieresis", "egrave", "eth", "iacute", "icircumflex", "idieresis", 
    "igrave", "lslash", "ntilde", "oe", "oacute", "ocircumflex", 
    "odieresis", "ograve", "oslash", "otilde", "scaron", "thorn", 
    "uacute", "ucircumflex", "udieresis", "ugrave", "yacute", 
    "ydieresis", "zcaron", "acute", "ampersand", "asciicircum", 
    "asciitilde", "asterisk", "at", "backslash", "bar", "braceleft", 
    "braceright", "bracketleft", "bracketright", "breve", "brokenbar", 
    "bullet", "caron", "cedilla", "cent", "circumflex", "colon", 
    "comma", "copyright", "currency", "dagger", "daggerdbl", "degree", 
    "dieresis", "divide", "dollar", "dotaccent", "dotlessi", "eight", 
    "ellipsis", "emdash", "endash", "equal", "exclam", "exclamdown", 
    "fi", "five", "fl", "florin", "four", "fraction", "germandbls", 
    "grave", "greater", "guillemotleft", "guillemotright", 
    "guilsinglleft", "guilsinglright", "hungarumlaut", "hyphen", 
    "less", "logicalnot", "macron", "minus", "mu", "multiply", "nine", 
    "numbersign", "ogonek", "one", "onehalf", "onequarter", 
    "onesuperior", "ordfeminine", "ordmasculine", "paragraph", 
    "parenleft", "parenright", "percent", "period", "periodcentered", 
    "perthousand", "plus", "plusminus", "question", "questiondown", 
    "quotedbl", "quotedblbase", "quotedblleft", "quotedblright", 
    "quoteleft", "quoteright", "quotesinglbase", "quotesingle", 
    "registered", "ring", "section", "semicolon", "seven", "six", 
    "slash", "space", "sterling", "three", "threequarters", 
    "threesuperior", "tilde", "trademark", "two", "twosuperior", 
    "underscore", "yen", "zero", NULL
};

/*
 * Character width information from the standard PS fonts.
 * Units are UNITS_PER_PT defined in paper.h.
 * Extracted from the AFM files in
 * <http://partners.adobe.com/public/developer/en/pdf/Core14_AFMs.tar>
 * by the following script:

fonts="Times-Roman Times-Italic Times-Bold Times-BoldItalic \
       Helvetica Helvetica-Oblique Helvetica-Bold Helvetica-BoldOblique \
       Courier Courier-Oblique Courier-Bold Courier-BoldOblique"

for i in $fonts; do
  printf 'static const kern_pair %s_kerns[] = {\n' $(echo $i | tr 'A-Z\-' a-z_)
  perl -e '
open S, "stdchars.txt" or die;
chomp(@s = <S>);
open G, "glyphnames.txt" or die;
chomp(@g = <G>); %g = map(($_, $i++), @g); %g = map(($_, $g{$_}), @s); 
open M, "$ARGV[0].afm" or die;
while (<M>) { /KPX (\S+) (\S+) (\S+)/ and exists $g{$1} and exists $g{$2} and
  print "{$g{$1},$g{$2},$3}, "; } print "\n"' $i |\
    fold -sw 68 | sed 's/^/    /'
  printf '    {NOGLYPH,NOGLYPH,0}\n};\n'
  printf 'static const ligature %s_ligs[] = {\n' $(echo $i | tr 'A-Z\-' a-z_)
  perl -e '
open S, "stdchars.txt" or die;
chomp(@s = <S>);
open G, "glyphnames.txt" or die;
chomp(@g = <G>); %g = map(($_, $i++), @g); %g = map(($_, $g{$_}), @s); 
open M, "$ARGV[0].afm" or die;
while (<M>) { / N (\S+) / and $l = $1;
  while (/ L (\S+) (\S+) /g) { exists $g{$l} and exists $g{$1} and
    exists $g{$2} and print "{$g{$l},$g{$1},$g{$2}}, "; } } print "\n"' $i |\
    fold -sw 68 | sed 's/^/    /'
  printf '    {NOGLYPH,NOGLYPH,NOGLYPH}\n};\n'
done

cat <<DECL

static const struct ps_std_font_data {
    char const *name;
    kern_pair const *kerns;
    ligature const *ligs;
    int widths[lenof(ps_std_glyphs)-1];
} ps_std_fonts[] = {
DECL
for i in $fonts; do
  printf '    { "%s",\n\t%s_kerns, %s_ligs, {\n' $i \
    $(echo $i | tr 'A-Z\-' a-z_) $(echo $i | tr 'A-Z\-' a-z_)
  perl -e '
open M, "$ARGV[0].afm" or die;
while (<M>) { /WX (\d+) ; N (\S+)/ and $m{$2} = $1; }
open G, "stdchars.txt" or die;
while (<G>) { chomp; print "$m{$_}, " }; print "\n"' $i |\
    fold -sw 68 | sed 's/^/	/'
  printf '    }},\n'
done
printf '};\n'

 * The AFM files have the following copyright and licence: 
 *
 * Copyright (c) 1985, 1987, 1989, 1990, 1991, 1992, 1993, 1997
 * Adobe Systems Incorporated.  All Rights Reserved.
 *
 * This file and the 14 PostScript(R) AFM files it accompanies may be
 * used, copied, and distributed for any purpose and without charge,
 * with or without modification, provided that all copyright notices
 * are retained; that the AFM files are not distributed without this
 * file; that all modifications to this file or any of the AFM files
 * are prominently noted in the modified file(s); and that this
 * paragraph is not modified. Adobe Systems has no responsibility or
 * obligation to support the use of the AFM files.
 */

static const kern_pair times_roman_kerns[] = {
    {0,64,-40}, {0,70,-40}, {0,203,-40}, {0,393,-55}, {0,396,-55}, 
    {0,404,-55}, {0,414,-55}, {0,419,-55}, {0,449,-55}, {0,455,-55}, 
    {0,474,-55}, {0,570,-111}, {0,603,-55}, {0,604,-55}, {0,609,-55}, 
    {0,615,-55}, {0,624,-55}, {0,658,-135}, {0,668,-90}, {0,686,-105}, 
    {0,687,-105}, {0,692,-105}, {0,3143,-111}, {0,3902,-74}, 
    {0,3951,-92}, {0,4031,-92}, {0,4034,-92}, {0,4050,-92}, {5,64,-40}, 
    {5,70,-40}, {5,203,-40}, {5,393,-55}, {5,396,-55}, {5,404,-55}, 
    {5,414,-55}, {5,419,-55}, {5,449,-55}, {5,455,-55}, {5,474,-55}, 
    {5,570,-111}, {5,603,-55}, {5,604,-55}, {5,609,-55}, {5,615,-55}, 
    {5,624,-55}, {5,658,-135}, {5,668,-90}, {5,686,-105}, {5,687,-105}, 
    {5,692,-105}, {5,3143,-111}, {5,3902,-74}, {5,3951,-92}, 
    {5,4031,-92}, {5,4034,-92}, {5,4050,-92}, {16,64,-40}, {16,70,-40}, 
    {16,203,-40}, {16,393,-55}, {16,396,-55}, {16,404,-55}, 
    {16,414,-55}, {16,419,-55}, {16,449,-55}, {16,455,-55}, 
    {16,474,-55}, {16,570,-111}, {16,603,-55}, {16,604,-55}, 
    {16,609,-55}, {16,615,-55}, {16,624,-55}, {16,658,-135}, 
    {16,668,-90}, {16,686,-105}, {16,687,-105}, {16,692,-105}, 
    {16,3143,-111}, {16,3902,-74}, {16,3951,-92}, {16,4031,-92}, 
    {16,4034,-92}, {16,4050,-92}, {27,64,-40}, {27,70,-40}, 
    {27,203,-40}, {27,393,-55}, {27,396,-55}, {27,404,-55}, 
    {27,414,-55}, {27,419,-55}, {27,449,-55}, {27,455,-55}, 
    {27,474,-55}, {27,570,-111}, {27,603,-55}, {27,604,-55}, 
    {27,609,-55}, {27,615,-55}, {27,624,-55}, {27,658,-135}, 
    {27,668,-90}, {27,686,-105}, {27,687,-105}, {27,692,-105}, 
    {27,3143,-111}, {27,3902,-74}, {27,3951,-92}, {27,4031,-92}, 
    {27,4034,-92}, {27,4050,-92}, {33,64,-40}, {33,70,-40}, 
    {33,203,-40}, {33,393,-55}, {33,396,-55}, {33,404,-55}, 
    {33,414,-55}, {33,419,-55}, {33,449,-55}, {33,455,-55}, 
    {33,474,-55}, {33,570,-111}, {33,603,-55}, {33,604,-55}, 
    {33,609,-55}, {33,615,-55}, {33,624,-55}, {33,658,-135}, 
    {33,668,-90}, {33,686,-105}, {33,687,-105}, {33,692,-105}, 
    {33,3143,-111}, {33,3902,-74}, {33,3951,-92}, {33,4031,-92}, 
    {33,4034,-92}, {33,4050,-92}, {43,64,-40}, {43,70,-40}, 
    {43,203,-40}, {43,393,-55}, {43,396,-55}, {43,404,-55}, 
    {43,414,-55}, {43,419,-55}, {43,449,-55}, {43,455,-55}, 
    {43,474,-55}, {43,570,-111}, {43,603,-55}, {43,604,-55}, 
    {43,609,-55}, {43,615,-55}, {43,624,-55}, {43,658,-135}, 
    {43,668,-90}, {43,686,-105}, {43,687,-105}, {43,692,-105}, 
    {43,3143,-111}, {43,3902,-74}, {43,3951,-92}, {43,4031,-92}, 
    {43,4034,-92}, {43,4050,-92}, {48,64,-40}, {48,70,-40}, 
    {48,203,-40}, {48,393,-55}, {48,396,-55}, {48,404,-55}, 
    {48,414,-55}, {48,419,-55}, {48,449,-55}, {48,455,-55}, 
    {48,474,-55}, {48,570,-111}, {48,603,-55}, {48,604,-55}, 
    {48,609,-55}, {48,615,-55}, {48,624,-55}, {48,658,-135}, 
    {48,668,-90}, {48,686,-105}, {48,687,-105}, {48,692,-105}, 
    {48,3143,-111}, {48,3902,-74}, {48,3951,-92}, {48,4031,-92}, 
    {48,4034,-92}, {48,4050,-92}, {51,0,-35}, {51,5,-35}, {51,16,-35}, 
    {51,27,-35}, {51,33,-35}, {51,43,-35}, {51,48,-35}, {51,603,-10}, 
    {51,604,-10}, {51,609,-10}, {51,615,-10}, {51,624,-10}, {93,0,-40}, 
    {93,5,-40}, {93,16,-40}, {93,27,-40}, {93,33,-40}, {93,43,-40}, 
    {93,48,-40}, {93,658,-40}, {93,668,-30}, {93,686,-55}, 
    {93,687,-55}, {93,692,-55}, {192,0,-74}, {192,5,-74}, {192,16,-74}, 
    {192,27,-74}, {192,33,-74}, {192,43,-74}, {192,48,-74}, 
    {192,734,-15}, {192,736,-15}, {192,758,-15}, {192,775,-15}, 
    {192,1030,-15}, {192,1111,-15}, {192,1173,-15}, {192,1409,-80}, 
    {192,2794,-15}, {192,2795,-15}, {192,2809,-15}, {192,2819,-15}, 
    {192,2826,-15}, {192,2899,-15}, {192,2907,-15}, {192,2995,-80}, 
    {301,0,-60}, {301,5,-60}, {301,16,-60}, {301,27,-60}, {301,33,-60}, 
    {301,43,-60}, {301,48,-60}, {309,393,-30}, {309,396,-30}, 
    {309,404,-30}, {309,414,-30}, {309,419,-30}, {309,449,-30}, 
    {309,455,-30}, {309,1630,-25}, {309,1631,-25}, {309,1645,-25}, 
    {309,1655,-25}, {309,1662,-25}, {309,2794,-35}, {309,2795,-35}, 
    {309,2809,-35}, {309,2819,-35}, {309,2826,-35}, {309,2899,-35}, 
    {309,2907,-35}, {309,3816,-15}, {309,3817,-15}, {309,3824,-15}, 
    {309,3831,-15}, {309,3839,-15}, {309,4031,-25}, {309,4034,-25}, 
    {309,4050,-25}, {337,570,-92}, {337,658,-100}, {337,668,-74}, 
    {337,686,-100}, {337,687,-100}, {337,692,-100}, {337,3143,-92}, 
    {337,4031,-55}, {337,4034,-55}, {337,4050,-55}, {356,570,-92}, 
    {356,658,-100}, {356,668,-74}, {356,686,-100}, {356,687,-100}, 
    {356,692,-100}, {356,3143,-92}, {356,4031,-55}, {356,4034,-55}, 
    {356,4050,-55}, {372,0,-35}, {372,5,-35}, {372,16,-35}, 
    {372,27,-35}, {372,33,-35}, {372,43,-35}, {372,48,-35}, 
    {390,0,-35}, {390,5,-35}, {390,16,-35}, {390,27,-35}, {390,33,-35}, 
    {390,43,-35}, {390,48,-35}, {393,0,-35}, {393,5,-35}, {393,16,-35}, 
    {393,27,-35}, {393,33,-35}, {393,43,-35}, {393,48,-35}, 
    {393,570,-40}, {393,658,-50}, {393,668,-35}, {393,678,-40}, 
    {393,686,-50}, {393,687,-50}, {393,692,-50}, {396,0,-35}, 
    {396,5,-35}, {396,16,-35}, {396,27,-35}, {396,33,-35}, 
    {396,43,-35}, {396,48,-35}, {396,570,-40}, {396,658,-50}, 
    {396,668,-35}, {396,678,-40}, {396,686,-50}, {396,687,-50}, 
    {396,692,-50}, {404,0,-35}, {404,5,-35}, {404,16,-35}, 
    {404,27,-35}, {404,33,-35}, {404,43,-35}, {404,48,-35}, 
    {404,570,-40}, {404,658,-50}, {404,668,-35}, {404,678,-40}, 
    {404,686,-50}, {404,687,-50}, {404,692,-50}, {414,0,-35}, 
    {414,5,-35}, {414,16,-35}, {414,27,-35}, {414,33,-35}, 
    {414,43,-35}, {414,48,-35}, {414,570,-40}, {414,658,-50}, 
    {414,668,-35}, {414,678,-40}, {414,686,-50}, {414,687,-50}, 
    {414,692,-50}, {419,0,-35}, {419,5,-35}, {419,16,-35}, 
    {419,27,-35}, {419,33,-35}, {419,43,-35}, {419,48,-35}, 
    {419,570,-40}, {419,658,-50}, {419,668,-35}, {419,678,-40}, 
    {419,686,-50}, {419,687,-50}, {419,692,-50}, {449,0,-35}, 
    {449,5,-35}, {449,16,-35}, {449,27,-35}, {449,33,-35}, 
    {449,43,-35}, {449,48,-35}, {449,570,-40}, {449,658,-50}, 
    {449,668,-35}, {449,678,-40}, {449,686,-50}, {449,687,-50}, 
    {449,692,-50}, {455,0,-35}, {455,5,-35}, {455,16,-35}, 
    {455,27,-35}, {455,33,-35}, {455,43,-35}, {455,48,-35}, 
    {455,570,-40}, {455,658,-50}, {455,668,-35}, {455,678,-40}, 
    {455,686,-50}, {455,687,-50}, {455,692,-50}, {459,0,-92}, 
    {459,5,-92}, {459,16,-92}, {459,27,-92}, {459,33,-92}, 
    {459,43,-92}, {459,48,-92}, {459,734,-15}, {459,736,-15}, 
    {459,758,-15}, {459,775,-15}, {459,1030,-15}, {459,1111,-15}, 
    {459,1173,-15}, {459,1409,-111}, {459,2995,-111}, {474,603,-10}, 
    {474,604,-10}, {474,609,-10}, {474,615,-10}, {474,624,-10}, 
    {478,393,-40}, {478,396,-40}, {478,404,-40}, {478,414,-40}, 
    {478,419,-40}, {478,449,-40}, {478,455,-40}, {478,570,-60}, 
    {478,603,-40}, {478,604,-40}, {478,609,-40}, {478,615,-40}, 
    {478,624,-40}, {478,658,-80}, {478,668,-55}, {478,686,-65}, 
    {478,687,-65}, {478,692,-65}, {570,0,-93}, {570,5,-93}, 
    {570,16,-93}, {570,27,-93}, {570,33,-93}, {570,43,-93}, 
    {570,48,-93}, {570,393,-18}, {570,396,-18}, {570,404,-18}, 
    {570,414,-18}, {570,419,-18}, {570,449,-18}, {570,455,-18}, 
    {570,734,-80}, {570,736,-80}, {570,758,-80}, {570,775,-40}, 
    {570,1030,-40}, {570,1111,-80}, {570,1173,-40}, {570,1402,-50}, 
    {570,1409,-74}, {570,1630,-70}, {570,1631,-70}, {570,1645,-70}, 
    {570,1655,-30}, {570,1662,-70}, {570,2126,-92}, {570,2132,-35}, 
    {570,2133,-35}, {570,2794,-80}, {570,2795,-80}, {570,2809,-80}, 
    {570,2819,-80}, {570,2826,-80}, {570,2899,-80}, {570,2907,-80}, 
    {570,2995,-74}, {570,3148,-35}, {570,3351,-55}, {570,3816,-45}, 
    {570,3817,-45}, {570,3824,-45}, {570,3831,-45}, {570,3839,-45}, 
    {570,3951,-80}, {570,4031,-80}, {570,4034,-80}, {570,4050,-80}, 
    {603,0,-40}, {603,5,-40}, {603,16,-40}, {603,27,-40}, {603,33,-40}, 
    {603,43,-40}, {603,48,-40}, {604,0,-40}, {604,5,-40}, {604,16,-40}, 
    {604,27,-40}, {604,33,-40}, {604,43,-40}, {604,48,-40}, 
    {609,0,-40}, {609,5,-40}, {609,16,-40}, {609,27,-40}, {609,33,-40}, 
    {609,43,-40}, {609,48,-40}, {615,0,-40}, {615,5,-40}, {615,16,-40}, 
    {615,27,-40}, {615,33,-40}, {615,43,-40}, {615,48,-40}, 
    {624,0,-40}, {624,5,-40}, {624,16,-40}, {624,27,-40}, {624,33,-40}, 
    {624,43,-40}, {624,48,-40}, {658,0,-135}, {658,5,-135}, 
    {658,16,-135}, {658,27,-135}, {658,33,-135}, {658,43,-135}, 
    {658,48,-135}, {658,203,-15}, {658,393,-40}, {658,396,-40}, 
    {658,404,-40}, {658,414,-40}, {658,419,-40}, {658,449,-40}, 
    {658,455,-40}, {658,734,-111}, {658,736,-111}, {658,758,-71}, 
    {658,775,-71}, {658,1030,-71}, {658,1111,-111}, {658,1173,-71}, 
    {658,1402,-74}, {658,1409,-129}, {658,1630,-111}, {658,1631,-111}, 
    {658,1645,-71}, {658,1655,-71}, {658,1662,-71}, {658,2126,-100}, 
    {658,2132,-60}, {658,2133,-60}, {658,2140,-20}, {658,2196,-20}, 
    {658,2207,-20}, {658,2794,-129}, {658,2795,-129}, {658,2809,-129}, 
    {658,2819,-89}, {658,2826,-89}, {658,2899,-129}, {658,2907,-89}, 
    {658,2995,-129}, {658,3351,-74}, {658,3816,-75}, {658,3817,-75}, 
    {658,3824,-75}, {658,3831,-75}, {658,3839,-75}, {668,0,-120}, 
    {668,5,-120}, {668,16,-120}, {668,27,-120}, {668,33,-120}, 
    {668,43,-120}, {668,48,-120}, {668,393,-10}, {668,396,-10}, 
    {668,404,-10}, {668,414,-10}, {668,419,-10}, {668,449,-10}, 
    {668,455,-10}, {668,734,-80}, {668,736,-80}, {668,758,-80}, 
    {668,775,-80}, {668,1030,-80}, {668,1111,-80}, {668,1173,-80}, 
    {668,1402,-37}, {668,1409,-92}, {668,1630,-80}, {668,1631,-80}, 
    {668,1645,-80}, {668,1655,-40}, {668,1662,-40}, {668,2126,-65}, 
    {668,2132,-40}, {668,2133,-40}, {668,2794,-80}, {668,2795,-80}, 
    {668,2809,-80}, {668,2819,-80}, {668,2826,-80}, {668,2899,-80}, 
    {668,2907,-80}, {668,2995,-92}, {668,3351,-37}, {668,3816,-50}, 
    {668,3817,-50}, {668,3824,-50}, {668,3831,-50}, {668,3839,-50}, 
    {668,4031,-73}, {668,4034,-73}, {668,4050,-73}, {686,0,-120}, 
    {686,5,-120}, {686,16,-120}, {686,27,-120}, {686,33,-120}, 
    {686,43,-120}, {686,48,-120}, {686,393,-30}, {686,396,-30}, 
    {686,404,-30}, {686,414,-30}, {686,419,-30}, {686,449,-30}, 
    {686,455,-30}, {686,734,-100}, {686,736,-100}, {686,758,-100}, 
    {686,775,-60}, {686,1030,-60}, {686,1111,-100}, {686,1173,-60}, 
    {686,1402,-92}, {686,1409,-129}, {686,1630,-100}, {686,1631,-100}, 
    {686,1645,-100}, {686,1655,-60}, {686,1662,-60}, {686,2126,-111}, 
    {686,2132,-55}, {686,2133,-55}, {686,2794,-110}, {686,2795,-110}, 
    {686,2809,-110}, {686,2819,-70}, {686,2826,-70}, {686,2899,-110}, 
    {686,2907,-70}, {686,2995,-129}, {686,3351,-92}, {686,3816,-111}, 
    {686,3817,-111}, {686,3824,-111}, {686,3831,-71}, {686,3839,-71}, 
    {687,0,-120}, {687,5,-120}, {687,16,-120}, {687,27,-120}, 
    {687,33,-120}, {687,43,-120}, {687,48,-120}, {687,393,-30}, 
    {687,396,-30}, {687,404,-30}, {687,414,-30}, {687,419,-30}, 
    {687,449,-30}, {687,455,-30}, {687,734,-100}, {687,736,-100}, 
    {687,758,-100}, {687,775,-60}, {687,1030,-60}, {687,1111,-100}, 
    {687,1173,-60}, {687,1402,-92}, {687,1409,-129}, {687,1630,-100}, 
    {687,1631,-100}, {687,1645,-100}, {687,1655,-60}, {687,1662,-60}, 
    {687,2126,-111}, {687,2132,-55}, {687,2133,-55}, {687,2794,-110}, 
    {687,2795,-110}, {687,2809,-110}, {687,2819,-70}, {687,2826,-70}, 
    {687,2899,-110}, {687,2907,-70}, {687,2995,-129}, {687,3351,-92}, 
    {687,3816,-111}, {687,3817,-111}, {687,3824,-111}, {687,3831,-71}, 
    {687,3839,-71}, {692,0,-120}, {692,5,-120}, {692,16,-120}, 
    {692,27,-120}, {692,33,-120}, {692,43,-120}, {692,48,-120}, 
    {692,393,-30}, {692,396,-30}, {692,404,-30}, {692,414,-30}, 
    {692,419,-30}, {692,449,-30}, {692,455,-30}, {692,734,-100}, 
    {692,736,-100}, {692,758,-100}, {692,775,-60}, {692,1030,-60}, 
    {692,1111,-100}, {692,1173,-100}, {692,1402,-92}, {692,1409,-129}, 
    {692,1630,-100}, {692,1631,-100}, {692,1645,-100}, {692,1655,-60}, 
    {692,1662,-60}, {692,2126,-111}, {692,2132,-55}, {692,2133,-55}, 
    {692,2794,-110}, {692,2795,-110}, {692,2809,-110}, {692,2819,-70}, 
    {692,2826,-70}, {692,2899,-110}, {692,2907,-70}, {692,2995,-129}, 
    {692,3351,-92}, {692,3816,-111}, {692,3817,-111}, {692,3824,-111}, 
    {692,3831,-71}, {692,3839,-71}, {734,3902,-20}, {734,3951,-15}, 
    {736,3902,-20}, {736,3951,-15}, {758,3902,-20}, {758,3951,-15}, 
    {775,3902,-20}, {775,3951,-15}, {1030,3902,-20}, {1030,3951,-15}, 
    {1111,3902,-20}, {1111,3951,-15}, {1173,3902,-20}, {1173,3951,-15}, 
    {1192,2995,-40}, {1192,3816,-20}, {1192,3817,-20}, {1192,3824,-20}, 
    {1192,3831,-20}, {1192,3839,-20}, {1192,3902,-15}, {1312,4031,-15}, 
    {1312,4034,-15}, {1312,4050,-15}, {1332,4031,-15}, {1332,4034,-15}, 
    {1332,4050,-15}, {1409,3139,-70}, {1409,3143,-70}, {1630,1886,-15}, 
    {1630,3902,-25}, {1630,3951,-25}, {1630,4020,-15}, {1630,4031,-15}, 
    {1630,4034,-15}, {1630,4050,-15}, {1631,1886,-15}, {1631,3902,-25}, 
    {1631,3951,-25}, {1631,4020,-15}, {1631,4031,-15}, {1631,4034,-15}, 
    {1631,4050,-15}, {1645,1886,-15}, {1645,3902,-25}, {1645,3951,-25}, 
    {1645,4020,-15}, {1645,4031,-15}, {1645,4034,-15}, {1645,4050,-15}, 
    {1655,1886,-15}, {1655,3902,-25}, {1655,3951,-25}, {1655,4020,-15}, 
    {1655,4031,-15}, {1655,4034,-15}, {1655,4050,-15}, {1662,1886,-15}, 
    {1662,3902,-25}, {1662,3951,-25}, {1662,4020,-15}, {1662,4031,-15}, 
    {1662,4034,-15}, {1662,4050,-15}, {1787,734,-10}, {1787,736,-10}, 
    {1787,758,-10}, {1787,775,-10}, {1787,1030,-10}, {1787,1111,-10}, 
    {1787,1173,-10}, {1787,1608,-50}, {1787,1787,-25}, {1787,2132,-20}, 
    {1787,2133,-20}, {1787,3143,55}, {1886,734,-5}, {1886,736,-5}, 
    {1886,758,-5}, {1886,775,-5}, {1886,1030,-5}, {1886,1111,-5}, 
    {1886,1173,-5}, {1988,4031,-5}, {1988,4034,-5}, {1988,4050,-5}, 
    {2132,3902,-25}, {2133,3902,-25}, {2140,3902,-25}, {2196,3902,-25}, 
    {2207,3902,-25}, {2302,1630,-10}, {2302,1631,-10}, {2302,1645,-10}, 
    {2302,1655,-10}, {2302,1662,-10}, {2302,2794,-10}, {2302,2795,-10}, 
    {2302,2809,-10}, {2302,2819,-10}, {2302,2826,-10}, {2302,2899,-10}, 
    {2302,2907,-10}, {2302,4031,-15}, {2302,4034,-15}, {2302,4050,-15}, 
    {2416,3951,-10}, {2501,3951,-10}, {2648,3902,-40}, {2648,4031,-15}, 
    {2648,4034,-15}, {2648,4050,-15}, {2769,3902,-40}, {2769,4031,-15}, 
    {2769,4034,-15}, {2769,4050,-15}, {2794,3902,-15}, {2794,3951,-25}, 
    {2794,4031,-10}, {2794,4034,-10}, {2794,4050,-10}, {2795,3902,-15}, 
    {2795,3951,-25}, {2795,4031,-10}, {2795,4034,-10}, {2795,4050,-10}, 
    {2809,3902,-15}, {2809,3951,-25}, {2809,4031,-10}, {2809,4034,-10}, 
    {2809,4050,-10}, {2819,3902,-15}, {2819,3951,-25}, {2819,4031,-10}, 
    {2819,4034,-10}, {2819,4050,-10}, {2826,3902,-15}, {2826,3951,-25}, 
    {2826,4031,-10}, {2826,4034,-10}, {2826,4050,-10}, {2899,3902,-15}, 
    {2899,3951,-25}, {2899,4031,-10}, {2899,4034,-10}, {2899,4050,-10}, 
    {2907,3902,-15}, {2907,3951,-25}, {2907,4031,-10}, {2907,4034,-10}, 
    {2907,4050,-10}, {2921,4031,-10}, {2921,4034,-10}, {2921,4050,-10}, 
    {2995,3139,-70}, {2995,3143,-70}, {3135,0,-80}, {3135,5,-80}, 
    {3135,16,-80}, {3135,27,-80}, {3135,33,-80}, {3135,43,-80}, 
    {3135,48,-80}, {3140,0,-80}, {3140,5,-80}, {3140,16,-80}, 
    {3140,27,-80}, {3140,33,-80}, {3140,43,-80}, {3140,48,-80}, 
    {3140,3140,-74}, {3143,1481,-50}, {3143,2416,-10}, {3143,2501,-10}, 
    {3143,3143,-74}, {3143,3148,-50}, {3143,3279,-55}, {3143,3317,-55}, 
    {3143,3498,-74}, {3143,3556,-18}, {3143,3902,-50}, {3148,1409,-40}, 
    {3148,1886,-18}, {3148,2126,-20}, {3148,2995,-55}, {3498,0,-55}, 
    {3498,5,-55}, {3498,16,-55}, {3498,27,-55}, {3498,33,-55}, 
    {3498,43,-55}, {3498,48,-55}, {3498,570,-18}, {3498,658,-50}, 
    {3498,668,-30}, {3498,686,-90}, {3498,687,-90}, {3498,692,-90}, 
    {3902,734,-25}, {3902,736,-25}, {3902,758,-25}, {3902,775,-25}, 
    {3902,1030,-25}, {3902,1111,-25}, {3902,1173,-25}, {3902,1409,-65}, 
    {3902,1630,-15}, {3902,1631,-15}, {3902,1645,-15}, {3902,1655,-15}, 
    {3902,1662,-15}, {3902,2794,-20}, {3902,2795,-20}, {3902,2809,-20}, 
    {3902,2819,-20}, {3902,2826,-20}, {3902,2899,-20}, {3902,2907,-20}, 
    {3902,2995,-65}, {3951,734,-10}, {3951,736,-10}, {3951,758,-10}, 
    {3951,775,-10}, {3951,1030,-10}, {3951,1111,-10}, {3951,1173,-10}, 
    {3951,1409,-65}, {3951,2794,-10}, {3951,2795,-10}, {3951,2809,-10}, 
    {3951,2819,-10}, {3951,2826,-10}, {3951,2899,-10}, {3951,2907,-10}, 
    {3951,2995,-65}, {4020,1630,-15}, {4020,1631,-15}, {4020,1645,-15}, 
    {4020,1655,-15}, {4020,1662,-15}, {4031,1409,-65}, {4031,2995,-65}, 
    {4034,1409,-65}, {4034,2995,-65}, {4050,1409,-65}, {4050,2995,-65}, 
    {NOGLYPH,NOGLYPH,0}
};
static const ligature times_roman_ligs[] = {
    {1787,2132,1807}, {1787,2416,1850}, 
    {NOGLYPH,NOGLYPH,NOGLYPH}
};
static const kern_pair times_italic_kerns[] = {
    {0,64,-30}, {0,70,-30}, {0,203,-35}, {0,393,-40}, {0,396,-40}, 
    {0,404,-40}, {0,414,-40}, {0,419,-40}, {0,449,-40}, {0,455,-40}, 
    {0,474,-40}, {0,570,-37}, {0,603,-50}, {0,604,-50}, {0,609,-50}, 
    {0,615,-50}, {0,624,-50}, {0,658,-105}, {0,668,-95}, {0,686,-55}, 
    {0,687,-55}, {0,692,-55}, {0,3143,-37}, {0,3816,-20}, {0,3817,-20}, 
    {0,3824,-20}, {0,3831,-20}, {0,3839,-20}, {0,3902,-55}, 
    {0,3951,-55}, {0,4031,-55}, {0,4034,-55}, {0,4050,-55}, {5,64,-30}, 
    {5,70,-30}, {5,203,-35}, {5,393,-40}, {5,396,-40}, {5,404,-40}, 
    {5,414,-40}, {5,419,-40}, {5,449,-40}, {5,455,-40}, {5,474,-40}, 
    {5,570,-37}, {5,603,-50}, {5,604,-50}, {5,609,-50}, {5,615,-50}, 
    {5,624,-50}, {5,658,-105}, {5,668,-95}, {5,686,-55}, {5,687,-55}, 
    {5,692,-55}, {5,3143,-37}, {5,3816,-20}, {5,3817,-20}, 
    {5,3824,-20}, {5,3831,-20}, {5,3839,-20}, {5,3902,-55}, 
    {5,3951,-55}, {5,4031,-55}, {5,4034,-55}, {5,4050,-55}, 
    {16,64,-30}, {16,70,-30}, {16,203,-35}, {16,393,-40}, {16,396,-40}, 
    {16,404,-40}, {16,414,-40}, {16,419,-40}, {16,449,-40}, 
    {16,455,-40}, {16,474,-40}, {16,570,-37}, {16,603,-50}, 
    {16,604,-50}, {16,609,-50}, {16,615,-50}, {16,624,-50}, 
    {16,658,-105}, {16,668,-95}, {16,686,-55}, {16,687,-55}, 
    {16,692,-55}, {16,3143,-37}, {16,3816,-20}, {16,3817,-20}, 
    {16,3824,-20}, {16,3831,-20}, {16,3839,-20}, {16,3902,-55}, 
    {16,3951,-55}, {16,4031,-55}, {16,4034,-55}, {16,4050,-55}, 
    {27,64,-30}, {27,70,-30}, {27,203,-35}, {27,393,-40}, {27,396,-40}, 
    {27,404,-40}, {27,414,-40}, {27,419,-40}, {27,449,-40}, 
    {27,455,-40}, {27,474,-40}, {27,570,-37}, {27,603,-50}, 
    {27,604,-50}, {27,609,-50}, {27,615,-50}, {27,624,-50}, 
    {27,658,-105}, {27,668,-95}, {27,686,-55}, {27,687,-55}, 
    {27,692,-55}, {27,3143,-37}, {27,3816,-20}, {27,3817,-20}, 
    {27,3824,-20}, {27,3831,-20}, {27,3839,-20}, {27,3902,-55}, 
    {27,3951,-55}, {27,4031,-55}, {27,4034,-55}, {27,4050,-55}, 
    {33,64,-30}, {33,70,-30}, {33,203,-35}, {33,393,-40}, {33,396,-40}, 
    {33,404,-40}, {33,414,-40}, {33,419,-40}, {33,449,-40}, 
    {33,455,-40}, {33,474,-40}, {33,570,-37}, {33,603,-50}, 
    {33,604,-50}, {33,609,-50}, {33,615,-50}, {33,624,-50}, 
    {33,658,-105}, {33,668,-95}, {33,686,-55}, {33,687,-55}, 
    {33,692,-55}, {33,3143,-37}, {33,3816,-20}, {33,3817,-20}, 
    {33,3824,-20}, {33,3831,-20}, {33,3839,-20}, {33,3902,-55}, 
    {33,3951,-55}, {33,4031,-55}, {33,4034,-55}, {33,4050,-55}, 
    {43,64,-30}, {43,70,-30}, {43,203,-35}, {43,393,-40}, {43,396,-40}, 
    {43,404,-40}, {43,414,-40}, {43,419,-40}, {43,449,-40}, 
    {43,455,-40}, {43,474,-40}, {43,570,-37}, {43,603,-50}, 
    {43,604,-50}, {43,609,-50}, {43,615,-50}, {43,624,-50}, 
    {43,658,-105}, {43,668,-95}, {43,686,-55}, {43,687,-55}, 
    {43,692,-55}, {43,3143,-37}, {43,3816,-20}, {43,3817,-20}, 
    {43,3824,-20}, {43,3831,-20}, {43,3839,-20}, {43,3902,-55}, 
    {43,3951,-55}, {43,4031,-55}, {43,4034,-55}, {43,4050,-55}, 
    {48,64,-30}, {48,70,-30}, {48,203,-35}, {48,393,-40}, {48,396,-40}, 
    {48,404,-40}, {48,414,-40}, {48,419,-40}, {48,449,-40}, 
    {48,455,-40}, {48,474,-40}, {48,570,-37}, {48,603,-50}, 
    {48,604,-50}, {48,609,-50}, {48,615,-50}, {48,624,-50}, 
    {48,658,-105}, {48,668,-95}, {48,686,-55}, {48,687,-55}, 
    {48,692,-55}, {48,3143,-37}, {48,3816,-20}, {48,3817,-20}, 
    {48,3824,-20}, {48,3831,-20}, {48,3839,-20}, {48,3902,-55}, 
    {48,3951,-55}, {48,4031,-55}, {48,4034,-55}, {48,4050,-55}, 
    {51,0,-25}, {51,5,-25}, {51,16,-25}, {51,27,-25}, {51,33,-25}, 
    {51,43,-25}, {51,48,-25}, {51,603,-10}, {51,604,-10}, {51,609,-10}, 
    {51,615,-10}, {51,624,-10}, {93,0,-35}, {93,5,-35}, {93,16,-35}, 
    {93,27,-35}, {93,33,-35}, {93,43,-35}, {93,48,-35}, {93,658,-40}, 
    {93,668,-40}, {93,686,-40}, {93,687,-40}, {93,692,-40}, 
    {192,0,-115}, {192,5,-115}, {192,16,-115}, {192,27,-115}, 
    {192,33,-115}, {192,43,-115}, {192,48,-115}, {192,734,-75}, 
    {192,736,-75}, {192,758,-75}, {192,775,-75}, {192,1030,-75}, 
    {192,1111,-75}, {192,1173,-75}, {192,1409,-135}, {192,1630,-75}, 
    {192,1631,-75}, {192,1645,-75}, {192,1655,-75}, {192,1662,-75}, 
    {192,2132,-45}, {192,2133,-45}, {192,2140,-45}, {192,2196,-45}, 
    {192,2207,-45}, {192,2794,-105}, {192,2795,-105}, {192,2809,-105}, 
    {192,2819,-105}, {192,2826,-105}, {192,2899,-105}, {192,2907,-105}, 
    {192,2995,-135}, {192,3148,-55}, {301,0,-40}, {301,5,-40}, 
    {301,16,-40}, {301,27,-40}, {301,33,-40}, {301,43,-40}, 
    {301,48,-40}, {301,734,-35}, {301,736,-35}, {301,758,-35}, 
    {301,775,-35}, {301,1030,-35}, {301,1111,-35}, {301,1173,-35}, 
    {301,1409,-25}, {301,1630,-25}, {301,1631,-25}, {301,1645,-25}, 
    {301,1655,-25}, {301,1662,-25}, {301,2794,-25}, {301,2795,-25}, 
    {301,2809,-25}, {301,2819,-25}, {301,2826,-25}, {301,2899,-25}, 
    {301,2907,-25}, {301,2995,-25}, {301,3816,-35}, {301,3817,-35}, 
    {301,3824,-35}, {301,3831,-35}, {301,3839,-35}, {309,393,-50}, 
    {309,396,-50}, {309,404,-50}, {309,414,-50}, {309,419,-50}, 
    {309,449,-50}, {309,455,-50}, {309,1630,-35}, {309,1631,-35}, 
    {309,1645,-35}, {309,1655,-35}, {309,1662,-35}, {309,2794,-40}, 
    {309,2795,-40}, {309,2809,-40}, {309,2819,-40}, {309,2826,-40}, 
    {309,2899,-40}, {309,2907,-40}, {309,3816,-40}, {309,3817,-40}, 
    {309,3824,-40}, {309,3831,-40}, {309,3839,-40}, {309,4031,-40}, 
    {309,4034,-40}, {309,4050,-40}, {337,570,-20}, {337,658,-55}, 
    {337,668,-55}, {337,686,-20}, {337,687,-20}, {337,692,-20}, 
    {337,3143,-37}, {337,4031,-30}, {337,4034,-30}, {337,4050,-30}, 
    {356,570,-20}, {356,658,-55}, {356,668,-55}, {356,686,-20}, 
    {356,687,-20}, {356,692,-20}, {356,3143,-37}, {356,4031,-30}, 
    {356,4034,-30}, {356,4050,-30}, {372,0,-27}, {372,5,-27}, 
    {372,16,-27}, {372,27,-27}, {372,33,-27}, {372,43,-27}, 
    {372,48,-27}, {390,0,-27}, {390,5,-27}, {390,16,-27}, {390,27,-27}, 
    {390,33,-27}, {390,43,-27}, {390,48,-27}, {393,0,-55}, {393,5,-55}, 
    {393,16,-55}, {393,27,-55}, {393,33,-55}, {393,43,-55}, 
    {393,48,-55}, {393,570,-40}, {393,658,-50}, {393,668,-50}, 
    {393,678,-40}, {393,686,-50}, {393,687,-50}, {393,692,-50}, 
    {396,0,-55}, {396,5,-55}, {396,16,-55}, {396,27,-55}, {396,33,-55}, 
    {396,43,-55}, {396,48,-55}, {396,570,-40}, {396,658,-50}, 
    {396,668,-50}, {396,678,-40}, {396,686,-50}, {396,687,-50}, 
    {396,692,-50}, {404,0,-55}, {404,5,-55}, {404,16,-55}, 
    {404,27,-55}, {404,33,-55}, {404,43,-55}, {404,48,-55}, 
    {404,570,-40}, {404,658,-50}, {404,668,-50}, {404,678,-40}, 
    {404,686,-50}, {404,687,-50}, {404,692,-50}, {414,0,-55}, 
    {414,5,-55}, {414,16,-55}, {414,27,-55}, {414,33,-55}, 
    {414,43,-55}, {414,48,-55}, {414,570,-40}, {414,658,-50}, 
    {414,668,-50}, {414,678,-40}, {414,686,-50}, {414,687,-50}, 
    {414,692,-50}, {419,0,-55}, {419,5,-55}, {419,16,-55}, 
    {419,27,-55}, {419,33,-55}, {419,43,-55}, {419,48,-55}, 
    {419,570,-40}, {419,658,-50}, {419,668,-50}, {419,678,-40}, 
    {419,686,-50}, {419,687,-50}, {419,692,-50}, {449,0,-55}, 
    {449,5,-55}, {449,16,-55}, {449,27,-55}, {449,33,-55}, 
    {449,43,-55}, {449,48,-55}, {449,570,-40}, {449,658,-50}, 
    {449,668,-50}, {449,678,-40}, {449,686,-50}, {449,687,-50}, 
    {449,692,-50}, {455,0,-55}, {455,5,-55}, {455,16,-55}, 
    {455,27,-55}, {455,33,-55}, {455,43,-55}, {455,48,-55}, 
    {455,570,-40}, {455,658,-50}, {455,668,-50}, {455,678,-40}, 
    {455,686,-50}, {455,687,-50}, {455,692,-50}, {459,0,-90}, 
    {459,5,-90}, {459,16,-90}, {459,27,-90}, {459,33,-90}, 
    {459,43,-90}, {459,48,-90}, {459,734,-80}, {459,736,-80}, 
    {459,758,-80}, {459,775,-80}, {459,1030,-80}, {459,1111,-80}, 
    {459,1173,-80}, {459,1409,-135}, {459,1630,-80}, {459,1631,-80}, 
    {459,1645,-80}, {459,1655,-80}, {459,1662,-80}, {459,2794,-80}, 
    {459,2795,-80}, {459,2809,-80}, {459,2819,-80}, {459,2826,-80}, 
    {459,2899,-80}, {459,2907,-80}, {459,2995,-135}, {474,603,-10}, 
    {474,604,-10}, {474,609,-10}, {474,615,-10}, {474,624,-10}, 
    {478,393,-40}, {478,396,-40}, {478,404,-40}, {478,414,-40}, 
    {478,419,-40}, {478,449,-40}, {478,455,-40}, {478,603,-40}, 
    {478,604,-40}, {478,609,-40}, {478,615,-40}, {478,624,-40}, 
    {478,658,-18}, {478,668,-18}, {478,686,-18}, {478,687,-18}, 
    {478,692,-18}, {570,0,-50}, {570,5,-50}, {570,16,-50}, 
    {570,27,-50}, {570,33,-50}, {570,43,-50}, {570,48,-50}, 
    {570,393,-18}, {570,396,-18}, {570,404,-18}, {570,414,-18}, 
    {570,419,-18}, {570,449,-18}, {570,455,-18}, {570,734,-92}, 
    {570,736,-92}, {570,758,-92}, {570,775,-92}, {570,1030,-92}, 
    {570,1111,-92}, {570,1173,-92}, {570,1402,-55}, {570,1409,-74}, 
    {570,1630,-92}, {570,1631,-92}, {570,1645,-52}, {570,1655,-52}, 
    {570,1662,-52}, {570,2126,-74}, {570,2132,-55}, {570,2133,-55}, 
    {570,2794,-92}, {570,2795,-92}, {570,2809,-92}, {570,2819,-92}, 
    {570,2826,-92}, {570,2899,-92}, {570,2907,-92}, {570,2995,-74}, 
    {570,3148,-55}, {570,3351,-65}, {570,3816,-55}, {570,3817,-55}, 
    {570,3824,-55}, {570,3831,-55}, {570,3839,-55}, {570,3951,-74}, 
    {570,4031,-74}, {570,4034,-74}, {570,4050,-34}, {603,0,-40}, 
    {603,5,-40}, {603,16,-40}, {603,27,-40}, {603,33,-40}, 
    {603,43,-40}, {603,48,-40}, {603,1409,-25}, {603,2995,-25}, 
    {604,0,-40}, {604,5,-40}, {604,16,-40}, {604,27,-40}, {604,33,-40}, 
    {604,43,-40}, {604,48,-40}, {604,1409,-25}, {604,2995,-25}, 
    {609,0,-40}, {609,5,-40}, {609,16,-40}, {609,27,-40}, {609,33,-40}, 
    {609,43,-40}, {609,48,-40}, {609,1409,-25}, {609,2995,-25}, 
    {615,0,-40}, {615,5,-40}, {615,16,-40}, {615,27,-40}, {615,33,-40}, 
    {615,43,-40}, {615,48,-40}, {615,1409,-25}, {615,2995,-25}, 
    {624,0,-40}, {624,5,-40}, {624,16,-40}, {624,27,-40}, {624,33,-40}, 
    {624,43,-40}, {624,48,-40}, {624,1409,-25}, {624,2995,-25}, 
    {658,0,-60}, {658,5,-60}, {658,16,-60}, {658,27,-60}, {658,33,-60}, 
    {658,43,-60}, {658,48,-60}, {658,393,-30}, {658,396,-30}, 
    {658,404,-30}, {658,414,-30}, {658,419,-30}, {658,449,-30}, 
    {658,455,-30}, {658,734,-111}, {658,736,-111}, {658,758,-111}, 
    {658,775,-111}, {658,1030,-111}, {658,1111,-111}, {658,1173,-111}, 
    {658,1402,-65}, {658,1409,-129}, {658,1630,-111}, {658,1631,-111}, 
    {658,1645,-111}, {658,1655,-71}, {658,1662,-71}, {658,2126,-55}, 
    {658,2132,-74}, {658,2133,-74}, {658,2140,-34}, {658,2196,-34}, 
    {658,2207,-34}, {658,2794,-111}, {658,2795,-111}, {658,2809,-111}, 
    {658,2819,-111}, {658,2826,-111}, {658,2899,-111}, {658,2907,-111}, 
    {658,2995,-129}, {658,3351,-74}, {658,3816,-74}, {658,3817,-74}, 
    {658,3824,-74}, {658,3831,-74}, {658,3839,-74}, {668,0,-60}, 
    {668,5,-60}, {668,16,-60}, {668,27,-60}, {668,33,-60}, 
    {668,43,-60}, {668,48,-60}, {668,393,-25}, {668,396,-25}, 
    {668,404,-25}, {668,414,-25}, {668,419,-25}, {668,449,-25}, 
    {668,455,-25}, {668,734,-92}, {668,736,-92}, {668,758,-92}, 
    {668,775,-92}, {668,1030,-92}, {668,1111,-92}, {668,1173,-92}, 
    {668,1402,-65}, {668,1409,-92}, {668,1630,-92}, {668,1631,-92}, 
    {668,1645,-92}, {668,1655,-52}, {668,1662,-52}, {668,2126,-37}, 
    {668,2132,-55}, {668,2133,-55}, {668,2794,-92}, {668,2795,-92}, 
    {668,2809,-92}, {668,2819,-92}, {668,2826,-92}, {668,2899,-92}, 
    {668,2907,-92}, {668,2995,-92}, {668,3351,-65}, {668,3816,-55}, 
    {668,3817,-55}, {668,3824,-55}, {668,3831,-55}, {668,3839,-55}, 
    {668,4031,-70}, {668,4034,-70}, {668,4050,-70}, {686,0,-50}, 
    {686,5,-50}, {686,16,-50}, {686,27,-50}, {686,33,-50}, 
    {686,43,-50}, {686,48,-50}, {686,393,-15}, {686,396,-15}, 
    {686,404,-15}, {686,414,-15}, {686,419,-15}, {686,449,-15}, 
    {686,455,-15}, {686,734,-92}, {686,736,-92}, {686,758,-92}, 
    {686,775,-92}, {686,1030,-92}, {686,1111,-92}, {686,1173,-92}, 
    {686,1402,-65}, {686,1409,-92}, {686,1630,-92}, {686,1631,-92}, 
    {686,1645,-92}, {686,1655,-52}, {686,1662,-52}, {686,2126,-74}, 
    {686,2132,-74}, {686,2133,-74}, {686,2140,-34}, {686,2196,-34}, 
    {686,2207,-34}, {686,2794,-92}, {686,2795,-92}, {686,2809,-92}, 
    {686,2819,-92}, {686,2826,-92}, {686,2899,-92}, {686,2907,-92}, 
    {686,2995,-92}, {686,3351,-65}, {686,3816,-92}, {686,3817,-92}, 
    {686,3824,-92}, {686,3831,-92}, {686,3839,-92}, {687,0,-50}, 
    {687,5,-50}, {687,16,-50}, {687,27,-50}, {687,33,-50}, 
    {687,43,-50}, {687,48,-50}, {687,393,-15}, {687,396,-15}, 
    {687,404,-15}, {687,414,-15}, {687,419,-15}, {687,449,-15}, 
    {687,455,-15}, {687,734,-92}, {687,736,-92}, {687,758,-92}, 
    {687,775,-92}, {687,1030,-92}, {687,1111,-92}, {687,1173,-92}, 
    {687,1402,-65}, {687,1409,-92}, {687,1630,-92}, {687,1631,-92}, 
    {687,1645,-92}, {687,1655,-52}, {687,1662,-52}, {687,2126,-74}, 
    {687,2132,-74}, {687,2133,-74}, {687,2140,-34}, {687,2196,-34}, 
    {687,2207,-34}, {687,2794,-92}, {687,2795,-92}, {687,2809,-92}, 
    {687,2819,-92}, {687,2826,-92}, {687,2899,-92}, {687,2907,-92}, 
    {687,2995,-92}, {687,3351,-65}, {687,3816,-92}, {687,3817,-92}, 
    {687,3824,-92}, {687,3831,-92}, {687,3839,-92}, {692,0,-50}, 
    {692,5,-50}, {692,16,-50}, {692,27,-50}, {692,33,-50}, 
    {692,43,-50}, {692,48,-50}, {692,393,-15}, {692,396,-15}, 
    {692,404,-15}, {692,414,-15}, {692,419,-15}, {692,449,-15}, 
    {692,455,-15}, {692,734,-92}, {692,736,-92}, {692,758,-92}, 
    {692,775,-92}, {692,1030,-92}, {692,1111,-92}, {692,1173,-92}, 
    {692,1402,-65}, {692,1409,-92}, {692,1630,-92}, {692,1631,-92}, 
    {692,1645,-92}, {692,1655,-52}, {692,1662,-52}, {692,2126,-74}, 
    {692,2132,-74}, {692,2133,-74}, {692,2140,-34}, {692,2196,-34}, 
    {692,2207,-34}, {692,2794,-92}, {692,2795,-92}, {692,2809,-92}, 
    {692,2819,-92}, {692,2826,-92}, {692,2899,-92}, {692,2907,-92}, 
    {692,2995,-92}, {692,3351,-65}, {692,3816,-92}, {692,3817,-92}, 
    {692,3824,-92}, {692,3831,-92}, {692,3839,-92}, {734,1886,-10}, 
    {736,1886,-10}, {758,1886,-10}, {775,1886,-10}, {1030,1886,-10}, 
    {1111,1886,-10}, {1173,1886,-10}, {1192,2995,-40}, {1192,3816,-20}, 
    {1192,3817,-20}, {1192,3824,-20}, {1192,3831,-20}, {1192,3839,-20}, 
    {1312,1988,-15}, {1312,2302,-20}, {1332,1988,-15}, {1332,2302,-20}, 
    {1409,3139,-140}, {1409,3143,-140}, {1630,1409,-10}, 
    {1630,1886,-40}, {1630,2995,-15}, {1630,3902,-15}, {1630,3951,-15}, 
    {1630,4020,-20}, {1630,4031,-30}, {1630,4034,-30}, {1630,4050,-30}, 
    {1631,1409,-10}, {1631,1886,-40}, {1631,2995,-15}, {1631,3902,-15}, 
    {1631,3951,-15}, {1631,4020,-20}, {1631,4031,-30}, {1631,4034,-30}, 
    {1631,4050,-30}, {1645,1409,-10}, {1645,1886,-40}, {1645,2995,-15}, 
    {1645,3902,-15}, {1645,3951,-15}, {1645,4020,-20}, {1645,4031,-30}, 
    {1645,4034,-30}, {1645,4050,-30}, {1655,1409,-10}, {1655,1886,-40}, 
    {1655,2995,-15}, {1655,3902,-15}, {1655,3951,-15}, {1655,4020,-20}, 
    {1655,4031,-30}, {1655,4034,-30}, {1655,4050,-30}, {1662,1409,-10}, 
    {1662,1886,-40}, {1662,2995,-15}, {1662,3902,-15}, {1662,3951,-15}, 
    {1662,4020,-20}, {1662,4031,-30}, {1662,4034,-30}, {1662,4050,-30}, 
    {1787,1409,-10}, {1787,1608,-60}, {1787,1787,-18}, {1787,2132,-20}, 
    {1787,2995,-15}, {1787,3143,92}, {1886,1409,-10}, {1886,1630,-10}, 
    {1886,1631,-10}, {1886,1645,-10}, {1886,1655,-10}, {1886,1662,-10}, 
    {1886,1886,-10}, {1886,2995,-15}, {2302,1630,-10}, {2302,1631,-10}, 
    {2302,1645,-10}, {2302,1655,-10}, {2302,1662,-10}, {2302,2794,-10}, 
    {2302,2795,-10}, {2302,2809,-10}, {2302,2819,-10}, {2302,2826,-10}, 
    {2302,2899,-10}, {2302,2907,-10}, {2302,4031,-10}, {2302,4034,-10}, 
    {2302,4050,-10}, {2648,3902,-40}, {2769,3902,-40}, {2794,1886,-10}, 
    {2794,3902,-10}, {2795,1886,-10}, {2795,3902,-10}, {2809,1886,-10}, 
    {2809,3902,-10}, {2819,1886,-10}, {2819,3902,-10}, {2826,1886,-10}, 
    {2826,3902,-10}, {2899,1886,-10}, {2899,3902,-10}, {2907,1886,-10}, 
    {2907,3902,-10}, {2995,3139,-140}, {2995,3143,-140}, 
    {3140,3140,-111}, {3143,1481,-25}, {3143,3143,-111}, 
    {3143,3148,-25}, {3143,3279,-40}, {3143,3317,-40}, 
    {3143,3498,-111}, {3143,3556,-30}, {3143,3902,-10}, {3148,734,-15}, 
    {3148,736,-15}, {3148,758,-15}, {3148,775,-15}, {3148,1030,-15}, 
    {3148,1111,-15}, {3148,1173,-15}, {3148,1312,-37}, {3148,1332,-37}, 
    {3148,1409,-111}, {3148,1481,-37}, {3148,1630,-37}, 
    {3148,1631,-37}, {3148,1645,-37}, {3148,1655,-37}, {3148,1662,-37}, 
    {3148,1886,-37}, {3148,2126,-20}, {3148,2794,-45}, {3148,2795,-45}, 
    {3148,2809,-45}, {3148,2819,-45}, {3148,2826,-45}, {3148,2899,-45}, 
    {3148,2907,-45}, {3148,2995,-111}, {3148,3083,-37}, 
    {3148,3279,-10}, {3148,3317,-10}, {3498,0,-18}, {3498,5,-18}, 
    {3498,16,-18}, {3498,27,-18}, {3498,33,-18}, {3498,43,-18}, 
    {3498,48,-18}, {3498,570,-18}, {3498,658,-35}, {3498,668,-40}, 
    {3498,686,-75}, {3498,687,-75}, {3498,692,-75}, {3902,1409,-74}, 
    {3902,2995,-74}, {3951,1409,-74}, {3951,2995,-74}, {4031,1409,-55}, 
    {4031,2995,-55}, {4034,1409,-55}, {4034,2995,-55}, {4050,1409,-55}, 
    {4050,2995,-55}, 
    {NOGLYPH,NOGLYPH,0}
};
static const ligature times_italic_ligs[] = {
    {1787,2132,1807}, {1787,2416,1850}, 
    {NOGLYPH,NOGLYPH,NOGLYPH}
};
static const kern_pair times_bold_kerns[] = {
    {0,64,-55}, {0,70,-55}, {0,203,-55}, {0,393,-45}, {0,396,-45}, 
    {0,404,-45}, {0,414,-45}, {0,419,-45}, {0,449,-45}, {0,455,-45}, 
    {0,474,-45}, {0,570,-95}, {0,603,-50}, {0,604,-50}, {0,609,-50}, 
    {0,615,-50}, {0,624,-50}, {0,658,-145}, {0,668,-130}, {0,686,-100}, 
    {0,687,-100}, {0,692,-100}, {0,2921,-25}, {0,3143,-74}, 
    {0,3816,-50}, {0,3817,-50}, {0,3824,-50}, {0,3831,-50}, 
    {0,3839,-50}, {0,3902,-100}, {0,3951,-90}, {0,4031,-74}, 
    {0,4034,-74}, {0,4050,-74}, {5,64,-55}, {5,70,-55}, {5,203,-55}, 
    {5,393,-45}, {5,396,-45}, {5,404,-45}, {5,414,-45}, {5,419,-45}, 
    {5,449,-45}, {5,455,-45}, {5,474,-45}, {5,570,-95}, {5,603,-50}, 
    {5,604,-50}, {5,609,-50}, {5,615,-50}, {5,624,-50}, {5,658,-145}, 
    {5,668,-130}, {5,686,-100}, {5,687,-100}, {5,692,-100}, 
    {5,2921,-25}, {5,3143,-74}, {5,3816,-50}, {5,3817,-50}, 
    {5,3824,-50}, {5,3831,-50}, {5,3839,-50}, {5,3902,-100}, 
    {5,3951,-90}, {5,4031,-74}, {5,4034,-74}, {5,4050,-74}, 
    {16,64,-55}, {16,70,-55}, {16,203,-55}, {16,393,-45}, {16,396,-45}, 
    {16,404,-45}, {16,414,-45}, {16,419,-45}, {16,449,-45}, 
    {16,455,-45}, {16,474,-45}, {16,570,-95}, {16,603,-50}, 
    {16,604,-50}, {16,609,-50}, {16,615,-50}, {16,624,-50}, 
    {16,658,-145}, {16,668,-130}, {16,686,-100}, {16,687,-100}, 
    {16,692,-100}, {16,2921,-25}, {16,3143,-74}, {16,3816,-50}, 
    {16,3817,-50}, {16,3824,-50}, {16,3831,-50}, {16,3839,-50}, 
    {16,3902,-100}, {16,3951,-90}, {16,4031,-74}, {16,4034,-74}, 
    {16,4050,-74}, {27,64,-55}, {27,70,-55}, {27,203,-55}, 
    {27,393,-45}, {27,396,-45}, {27,404,-45}, {27,414,-45}, 
    {27,419,-45}, {27,449,-45}, {27,455,-45}, {27,474,-45}, 
    {27,570,-95}, {27,603,-50}, {27,604,-50}, {27,609,-50}, 
    {27,615,-50}, {27,624,-50}, {27,658,-145}, {27,668,-130}, 
    {27,686,-100}, {27,687,-100}, {27,692,-100}, {27,2921,-25}, 
    {27,3143,-74}, {27,3816,-50}, {27,3817,-50}, {27,3824,-50}, 
    {27,3831,-50}, {27,3839,-50}, {27,3902,-100}, {27,3951,-90}, 
    {27,4031,-74}, {27,4034,-74}, {27,4050,-74}, {33,64,-55}, 
    {33,70,-55}, {33,203,-55}, {33,393,-45}, {33,396,-45}, 
    {33,404,-45}, {33,414,-45}, {33,419,-45}, {33,449,-45}, 
    {33,455,-45}, {33,474,-45}, {33,570,-95}, {33,603,-50}, 
    {33,604,-50}, {33,609,-50}, {33,615,-50}, {33,624,-50}, 
    {33,658,-145}, {33,668,-130}, {33,686,-100}, {33,687,-100}, 
    {33,692,-100}, {33,2921,-25}, {33,3143,-74}, {33,3816,-50}, 
    {33,3817,-50}, {33,3824,-50}, {33,3831,-50}, {33,3839,-50}, 
    {33,3902,-100}, {33,3951,-90}, {33,4031,-74}, {33,4034,-74}, 
    {33,4050,-74}, {43,64,-55}, {43,70,-55}, {43,203,-55}, 
    {43,393,-45}, {43,396,-45}, {43,404,-45}, {43,414,-45}, 
    {43,419,-45}, {43,449,-45}, {43,455,-45}, {43,474,-45}, 
    {43,570,-95}, {43,603,-50}, {43,604,-50}, {43,609,-50}, 
    {43,615,-50}, {43,624,-50}, {43,658,-145}, {43,668,-130}, 
    {43,686,-100}, {43,687,-100}, {43,692,-100}, {43,2921,-25}, 
    {43,3143,-74}, {43,3816,-50}, {43,3817,-50}, {43,3824,-50}, 
    {43,3831,-50}, {43,3839,-50}, {43,3902,-100}, {43,3951,-90}, 
    {43,4031,-74}, {43,4034,-74}, {43,4050,-74}, {48,64,-55}, 
    {48,70,-55}, {48,203,-55}, {48,393,-45}, {48,396,-45}, 
    {48,404,-45}, {48,414,-45}, {48,419,-45}, {48,449,-45}, 
    {48,455,-45}, {48,474,-45}, {48,570,-95}, {48,603,-50}, 
    {48,604,-50}, {48,609,-50}, {48,615,-50}, {48,624,-50}, 
    {48,658,-145}, {48,668,-130}, {48,686,-100}, {48,687,-100}, 
    {48,692,-100}, {48,2921,-25}, {48,3143,-74}, {48,3816,-50}, 
    {48,3817,-50}, {48,3824,-50}, {48,3831,-50}, {48,3839,-50}, 
    {48,3902,-100}, {48,3951,-90}, {48,4031,-74}, {48,4034,-74}, 
    {48,4050,-74}, {51,0,-30}, {51,5,-30}, {51,16,-30}, {51,27,-30}, 
    {51,33,-30}, {51,43,-30}, {51,48,-30}, {51,603,-10}, {51,604,-10}, 
    {51,609,-10}, {51,615,-10}, {51,624,-10}, {93,0,-35}, {93,5,-35}, 
    {93,16,-35}, {93,27,-35}, {93,33,-35}, {93,43,-35}, {93,48,-35}, 
    {93,658,-40}, {93,668,-40}, {93,686,-40}, {93,687,-40}, 
    {93,692,-40}, {93,2995,-20}, {192,0,-90}, {192,5,-90}, 
    {192,16,-90}, {192,27,-90}, {192,33,-90}, {192,43,-90}, 
    {192,48,-90}, {192,734,-25}, {192,736,-25}, {192,758,-25}, 
    {192,775,-25}, {192,1030,-25}, {192,1111,-25}, {192,1173,-25}, 
    {192,1409,-92}, {192,1630,-25}, {192,1631,-25}, {192,1645,-25}, 
    {192,1655,-25}, {192,1662,-25}, {192,2794,-25}, {192,2795,-25}, 
    {192,2809,-25}, {192,2819,-25}, {192,2826,-25}, {192,2899,-25}, 
    {192,2907,-25}, {192,2995,-110}, {301,0,-30}, {301,5,-30}, 
    {301,16,-30}, {301,27,-30}, {301,33,-30}, {301,43,-30}, 
    {301,48,-30}, {301,734,-15}, {301,736,-15}, {301,758,-15}, 
    {301,775,-15}, {301,1030,-15}, {301,1111,-15}, {301,1173,-15}, 
    {301,1630,-15}, {301,1631,-15}, {301,1645,-15}, {301,1655,-15}, 
    {301,1662,-15}, {301,2794,-15}, {301,2795,-15}, {301,2809,-15}, 
    {301,2819,-15}, {301,2826,-15}, {301,2899,-15}, {301,2907,-15}, 
    {301,2995,-20}, {301,3816,-15}, {301,3817,-15}, {301,3824,-15}, 
    {301,3831,-15}, {301,3839,-15}, {309,393,-30}, {309,396,-30}, 
    {309,404,-30}, {309,414,-30}, {309,419,-30}, {309,449,-30}, 
    {309,455,-30}, {309,1630,-25}, {309,1631,-25}, {309,1645,-25}, 
    {309,1655,-25}, {309,1662,-25}, {309,2794,-25}, {309,2795,-25}, 
    {309,2809,-25}, {309,2819,-25}, {309,2826,-25}, {309,2899,-25}, 
    {309,2907,-25}, {309,3816,-15}, {309,3817,-15}, {309,3824,-15}, 
    {309,3831,-15}, {309,3839,-15}, {309,4031,-45}, {309,4034,-45}, 
    {309,4050,-45}, {337,570,-92}, {337,658,-92}, {337,668,-92}, 
    {337,686,-92}, {337,687,-92}, {337,692,-92}, {337,3139,-20}, 
    {337,3143,-110}, {337,4031,-55}, {337,4034,-55}, {337,4050,-55}, 
    {356,570,-92}, {356,658,-92}, {356,668,-92}, {356,686,-92}, 
    {356,687,-92}, {356,692,-92}, {356,3139,-20}, {356,3143,-110}, 
    {356,4031,-55}, {356,4034,-55}, {356,4050,-55}, {372,0,-20}, 
    {372,5,-20}, {372,16,-20}, {372,27,-20}, {372,33,-20}, 
    {372,43,-20}, {372,48,-20}, {390,0,-20}, {390,5,-20}, {390,16,-20}, 
    {390,27,-20}, {390,33,-20}, {390,43,-20}, {390,48,-20}, 
    {393,0,-40}, {393,5,-40}, {393,16,-40}, {393,27,-40}, {393,33,-40}, 
    {393,43,-40}, {393,48,-40}, {393,570,-40}, {393,658,-50}, 
    {393,668,-50}, {393,678,-40}, {393,686,-50}, {393,687,-50}, 
    {393,692,-50}, {396,0,-40}, {396,5,-40}, {396,16,-40}, 
    {396,27,-40}, {396,33,-40}, {396,43,-40}, {396,48,-40}, 
    {396,570,-40}, {396,658,-50}, {396,668,-50}, {396,678,-40}, 
    {396,686,-50}, {396,687,-50}, {396,692,-50}, {404,0,-40}, 
    {404,5,-40}, {404,16,-40}, {404,27,-40}, {404,33,-40}, 
    {404,43,-40}, {404,48,-40}, {404,570,-40}, {404,658,-50}, 
    {404,668,-50}, {404,678,-40}, {404,686,-50}, {404,687,-50}, 
    {404,692,-50}, {414,0,-40}, {414,5,-40}, {414,16,-40}, 
    {414,27,-40}, {414,33,-40}, {414,43,-40}, {414,48,-40}, 
    {414,570,-40}, {414,658,-50}, {414,668,-50}, {414,678,-40}, 
    {414,686,-50}, {414,687,-50}, {414,692,-50}, {419,0,-40}, 
    {419,5,-40}, {419,16,-40}, {419,27,-40}, {419,33,-40}, 
    {419,43,-40}, {419,48,-40}, {419,570,-40}, {419,658,-50}, 
    {419,668,-50}, {419,678,-40}, {419,686,-50}, {419,687,-50}, 
    {419,692,-50}, {449,0,-40}, {449,5,-40}, {449,16,-40}, 
    {449,27,-40}, {449,33,-40}, {449,43,-40}, {449,48,-40}, 
    {449,570,-40}, {449,658,-50}, {449,668,-50}, {449,678,-40}, 
    {449,686,-50}, {449,687,-50}, {449,692,-50}, {455,0,-40}, 
    {455,5,-40}, {455,16,-40}, {455,27,-40}, {455,33,-40}, 
    {455,43,-40}, {455,48,-40}, {455,570,-40}, {455,658,-50}, 
    {455,668,-50}, {455,678,-40}, {455,686,-50}, {455,687,-50}, 
    {455,692,-50}, {459,0,-74}, {459,5,-74}, {459,16,-74}, 
    {459,27,-74}, {459,33,-74}, {459,43,-74}, {459,48,-74}, 
    {459,734,-10}, {459,736,-10}, {459,758,-10}, {459,775,-10}, 
    {459,1030,-10}, {459,1111,-10}, {459,1173,-10}, {459,1409,-92}, 
    {459,1630,-20}, {459,1631,-20}, {459,1645,-20}, {459,1655,-20}, 
    {459,1662,-20}, {459,2794,-20}, {459,2795,-20}, {459,2809,-20}, 
    {459,2819,-20}, {459,2826,-20}, {459,2899,-20}, {459,2907,-20}, 
    {459,2995,-110}, {474,603,-10}, {474,604,-10}, {474,609,-10}, 
    {474,615,-10}, {474,624,-10}, {474,2995,-20}, {478,393,-30}, 
    {478,396,-30}, {478,404,-30}, {478,414,-30}, {478,419,-30}, 
    {478,449,-30}, {478,455,-30}, {478,570,-40}, {478,603,-30}, 
    {478,604,-30}, {478,609,-30}, {478,615,-30}, {478,624,-30}, 
    {478,658,-55}, {478,668,-35}, {478,686,-35}, {478,687,-35}, 
    {478,692,-35}, {570,0,-90}, {570,5,-90}, {570,16,-90}, 
    {570,27,-90}, {570,33,-90}, {570,43,-90}, {570,48,-90}, 
    {570,393,-18}, {570,396,-18}, {570,404,-18}, {570,414,-18}, 
    {570,419,-18}, {570,449,-18}, {570,455,-18}, {570,734,-92}, 
    {570,736,-92}, {570,758,-52}, {570,775,-52}, {570,1030,-52}, 
    {570,1111,-92}, {570,1173,-52}, {570,1402,-74}, {570,1409,-74}, 
    {570,1630,-92}, {570,1631,-92}, {570,1645,-92}, {570,1655,-52}, 
    {570,1662,-52}, {570,2126,-92}, {570,2132,-18}, {570,2133,-18}, 
    {570,2794,-92}, {570,2795,-92}, {570,2809,-92}, {570,2819,-92}, 
    {570,2826,-92}, {570,2899,-92}, {570,2907,-92}, {570,2995,-90}, 
    {570,3148,-74}, {570,3351,-74}, {570,3816,-92}, {570,3817,-92}, 
    {570,3824,-92}, {570,3831,-92}, {570,3839,-92}, {570,3951,-74}, 
    {570,4031,-34}, {570,4034,-34}, {570,4050,-34}, {603,0,-60}, 
    {603,5,-60}, {603,16,-60}, {603,27,-60}, {603,33,-60}, 
    {603,43,-60}, {603,48,-60}, {603,1409,-50}, {603,2995,-50}, 
    {604,0,-60}, {604,5,-60}, {604,16,-60}, {604,27,-60}, {604,33,-60}, 
    {604,43,-60}, {604,48,-60}, {604,1409,-50}, {604,2995,-50}, 
    {609,0,-60}, {609,5,-60}, {609,16,-60}, {609,27,-60}, {609,33,-60}, 
    {609,43,-60}, {609,48,-60}, {609,1409,-50}, {609,2995,-50}, 
    {615,0,-60}, {615,5,-60}, {615,16,-60}, {615,27,-60}, {615,33,-60}, 
    {615,43,-60}, {615,48,-60}, {615,1409,-50}, {615,2995,-50}, 
    {624,0,-60}, {624,5,-60}, {624,16,-60}, {624,27,-60}, {624,33,-60}, 
    {624,43,-60}, {624,48,-60}, {624,1409,-50}, {624,2995,-50}, 
    {658,0,-135}, {658,5,-135}, {658,16,-135}, {658,27,-135}, 
    {658,33,-135}, {658,43,-135}, {658,48,-135}, {658,203,-30}, 
    {658,393,-45}, {658,396,-45}, {658,404,-45}, {658,414,-45}, 
    {658,419,-45}, {658,449,-45}, {658,455,-45}, {658,734,-92}, 
    {658,736,-92}, {658,758,-92}, {658,775,-92}, {658,1030,-92}, 
    {658,1111,-92}, {658,1173,-92}, {658,1402,-92}, {658,1409,-129}, 
    {658,1630,-100}, {658,1631,-100}, {658,1645,-100}, {658,1655,-100}, 
    {658,1662,-100}, {658,2126,-74}, {658,2132,-37}, {658,2133,-37}, 
    {658,2140,-37}, {658,2196,-37}, {658,2207,-37}, {658,2794,-100}, 
    {658,2795,-100}, {658,2809,-100}, {658,2819,-100}, {658,2826,-100}, 
    {658,2899,-100}, {658,2907,-100}, {658,2995,-145}, {658,3351,-92}, 
    {658,3816,-92}, {658,3817,-92}, {658,3824,-92}, {658,3831,-92}, 
    {658,3839,-92}, {668,0,-120}, {668,5,-120}, {668,16,-120}, 
    {668,27,-120}, {668,33,-120}, {668,43,-120}, {668,48,-120}, 
    {668,393,-10}, {668,396,-10}, {668,404,-10}, {668,414,-10}, 
    {668,419,-10}, {668,449,-10}, {668,455,-10}, {668,734,-65}, 
    {668,736,-65}, {668,758,-65}, {668,775,-65}, {668,1030,-65}, 
    {668,1111,-65}, {668,1173,-65}, {668,1402,-55}, {668,1409,-92}, 
    {668,1630,-65}, {668,1631,-65}, {668,1645,-65}, {668,1655,-65}, 
    {668,1662,-65}, {668,2126,-37}, {668,2132,-18}, {668,2133,-18}, 
    {668,2794,-75}, {668,2795,-75}, {668,2809,-75}, {668,2819,-75}, 
    {668,2826,-75}, {668,2899,-75}, {668,2907,-75}, {668,2995,-92}, 
    {668,3351,-55}, {668,3816,-50}, {668,3817,-50}, {668,3824,-50}, 
    {668,3831,-50}, {668,3839,-50}, {668,4031,-60}, {668,4034,-60}, 
    {668,4050,-60}, {686,0,-110}, {686,5,-110}, {686,16,-110}, 
    {686,27,-110}, {686,33,-110}, {686,43,-110}, {686,48,-110}, 
    {686,393,-35}, {686,396,-35}, {686,404,-35}, {686,414,-35}, 
    {686,419,-35}, {686,449,-35}, {686,455,-35}, {686,734,-85}, 
    {686,736,-85}, {686,758,-85}, {686,775,-85}, {686,1030,-85}, 
    {686,1111,-85}, {686,1173,-85}, {686,1402,-92}, {686,1409,-92}, 
    {686,1630,-111}, {686,1631,-111}, {686,1645,-111}, {686,1655,-71}, 
    {686,1662,-71}, {686,2126,-92}, {686,2132,-37}, {686,2133,-37}, 
    {686,2794,-111}, {686,2795,-111}, {686,2809,-111}, {686,2819,-111}, 
    {686,2826,-111}, {686,2899,-111}, {686,2907,-111}, {686,2995,-92}, 
    {686,3351,-92}, {686,3816,-92}, {686,3817,-92}, {686,3824,-92}, 
    {686,3831,-92}, {686,3839,-92}, {687,0,-110}, {687,5,-110}, 
    {687,16,-110}, {687,27,-110}, {687,33,-110}, {687,43,-110}, 
    {687,48,-110}, {687,393,-35}, {687,396,-35}, {687,404,-35}, 
    {687,414,-35}, {687,419,-35}, {687,449,-35}, {687,455,-35}, 
    {687,734,-85}, {687,736,-85}, {687,758,-85}, {687,775,-85}, 
    {687,1030,-85}, {687,1111,-85}, {687,1173,-85}, {687,1402,-92}, 
    {687,1409,-92}, {687,1630,-111}, {687,1631,-111}, {687,1645,-111}, 
    {687,1655,-71}, {687,1662,-71}, {687,2126,-92}, {687,2132,-37}, 
    {687,2133,-37}, {687,2794,-111}, {687,2795,-111}, {687,2809,-111}, 
    {687,2819,-111}, {687,2826,-111}, {687,2899,-111}, {687,2907,-111}, 
    {687,2995,-92}, {687,3351,-92}, {687,3816,-92}, {687,3817,-92}, 
    {687,3824,-92}, {687,3831,-92}, {687,3839,-92}, {692,0,-110}, 
    {692,5,-110}, {692,16,-110}, {692,27,-110}, {692,33,-110}, 
    {692,43,-110}, {692,48,-110}, {692,393,-35}, {692,396,-35}, 
    {692,404,-35}, {692,414,-35}, {692,419,-35}, {692,449,-35}, 
    {692,455,-35}, {692,734,-85}, {692,736,-85}, {692,758,-85}, 
    {692,775,-85}, {692,1030,-85}, {692,1111,-85}, {692,1173,-85}, 
    {692,1402,-92}, {692,1409,-92}, {692,1630,-111}, {692,1631,-111}, 
    {692,1645,-111}, {692,1655,-71}, {692,1662,-71}, {692,2126,-92}, 
    {692,2132,-37}, {692,2133,-37}, {692,2794,-111}, {692,2795,-111}, 
    {692,2809,-111}, {692,2819,-111}, {692,2826,-111}, {692,2899,-111}, 
    {692,2907,-111}, {692,2995,-92}, {692,3351,-92}, {692,3816,-92}, 
    {692,3817,-92}, {692,3824,-92}, {692,3831,-92}, {692,3839,-92}, 
    {734,3902,-25}, {736,3902,-25}, {758,3902,-25}, {775,3902,-25}, 
    {1030,3902,-25}, {1111,3902,-25}, {1173,3902,-25}, {1192,1192,-10}, 
    {1192,2995,-40}, {1192,3816,-20}, {1192,3817,-20}, {1192,3824,-20}, 
    {1192,3831,-20}, {1192,3839,-20}, {1192,3902,-15}, {1409,3139,-45}, 
    {1409,3143,-55}, {1481,3951,-15}, {1630,3902,-15}, {1631,3902,-15}, 
    {1645,3902,-15}, {1655,3902,-15}, {1662,3902,-15}, {1787,1409,-15}, 
    {1787,1608,-35}, {1787,2132,-25}, {1787,2794,-25}, {1787,2795,-25}, 
    {1787,2809,-25}, {1787,2819,-25}, {1787,2826,-25}, {1787,2899,-25}, 
    {1787,2907,-25}, {1787,2995,-15}, {1787,3139,50}, {1787,3143,55}, 
    {1886,2995,-15}, {1988,4031,-15}, {1988,4034,-15}, {1988,4050,-15}, 
    {2132,3902,-10}, {2133,3902,-10}, {2140,3902,-10}, {2196,3902,-10}, 
    {2207,3902,-10}, {2302,1630,-10}, {2302,1631,-10}, {2302,1645,-10}, 
    {2302,1655,-10}, {2302,1662,-10}, {2302,2794,-15}, {2302,2795,-15}, 
    {2302,2809,-15}, {2302,2819,-15}, {2302,2826,-15}, {2302,2899,-15}, 
    {2302,2907,-15}, {2302,4031,-15}, {2302,4034,-15}, {2302,4050,-15}, 
    {2648,3902,-40}, {2769,3902,-40}, {2794,3902,-10}, {2794,3951,-10}, 
    {2795,3902,-10}, {2795,3951,-10}, {2809,3902,-10}, {2809,3951,-10}, 
    {2819,3902,-10}, {2819,3951,-10}, {2826,3902,-10}, {2826,3951,-10}, 
    {2899,3902,-10}, {2899,3951,-10}, {2907,3902,-10}, {2907,3951,-10}, 
    {2995,3139,-55}, {2995,3143,-55}, {3135,0,-10}, {3135,5,-10}, 
    {3135,16,-10}, {3135,27,-10}, {3135,33,-10}, {3135,43,-10}, 
    {3135,48,-10}, {3140,0,-10}, {3140,5,-10}, {3140,16,-10}, 
    {3140,27,-10}, {3140,33,-10}, {3140,43,-10}, {3140,48,-10}, 
    {3140,3140,-63}, {3143,1481,-20}, {3143,3143,-63}, {3143,3148,-20}, 
    {3143,3279,-37}, {3143,3317,-37}, {3143,3498,-74}, {3143,3902,-20}, 
    {3148,1312,-18}, {3148,1332,-18}, {3148,1409,-92}, {3148,1630,-18}, 
    {3148,1631,-18}, {3148,1645,-18}, {3148,1655,-18}, {3148,1662,-18}, 
    {3148,1886,-10}, {3148,2126,-37}, {3148,2648,-15}, {3148,2769,-15}, 
    {3148,2794,-18}, {3148,2795,-18}, {3148,2809,-18}, {3148,2819,-18}, 
    {3148,2826,-18}, {3148,2899,-18}, {3148,2907,-18}, {3148,2921,-10}, 
    {3148,2995,-100}, {3148,3083,-18}, {3148,3902,-10}, {3498,0,-55}, 
    {3498,5,-55}, {3498,16,-55}, {3498,27,-55}, {3498,33,-55}, 
    {3498,43,-55}, {3498,48,-55}, {3498,570,-30}, {3498,658,-45}, 
    {3498,668,-30}, {3498,686,-55}, {3498,687,-55}, {3498,692,-55}, 
    {3902,734,-10}, {3902,736,-10}, {3902,758,-10}, {3902,775,-10}, 
    {3902,1030,-10}, {3902,1111,-10}, {3902,1173,-10}, {3902,1409,-55}, 
    {3902,1630,-10}, {3902,1631,-10}, {3902,1645,-10}, {3902,1655,-10}, 
    {3902,1662,-10}, {3902,2794,-10}, {3902,2795,-10}, {3902,2809,-10}, 
    {3902,2819,-10}, {3902,2826,-10}, {3902,2899,-10}, {3902,2907,-10}, 
    {3902,2995,-70}, {3951,1409,-55}, {3951,2794,-10}, {3951,2795,-10}, 
    {3951,2809,-10}, {3951,2819,-10}, {3951,2826,-10}, {3951,2899,-10}, 
    {3951,2907,-10}, {3951,2995,-70}, {4031,1409,-55}, {4031,1630,-10}, 
    {4031,1631,-10}, {4031,1645,-10}, {4031,1655,-10}, {4031,1662,-10}, 
    {4031,2794,-25}, {4031,2795,-25}, {4031,2809,-25}, {4031,2819,-25}, 
    {4031,2826,-25}, {4031,2899,-25}, {4031,2907,-25}, {4031,2995,-70}, 
    {4034,1409,-55}, {4034,1630,-10}, {4034,1631,-10}, {4034,1645,-10}, 
    {4034,1655,-10}, {4034,1662,-10}, {4034,2794,-25}, {4034,2795,-25}, 
    {4034,2809,-25}, {4034,2819,-25}, {4034,2826,-25}, {4034,2899,-25}, 
    {4034,2907,-25}, {4034,2995,-70}, {4050,1409,-55}, {4050,1630,-10}, 
    {4050,1631,-10}, {4050,1645,-10}, {4050,1655,-10}, {4050,1662,-10}, 
    {4050,2794,-25}, {4050,2795,-25}, {4050,2809,-25}, {4050,2819,-25}, 
    {4050,2826,-25}, {4050,2899,-25}, {4050,2907,-25}, {4050,2995,-70}, 
    {NOGLYPH,NOGLYPH,0}
};
static const ligature times_bold_ligs[] = {
    {1787,2132,1807}, {1787,2416,1850}, 
    {NOGLYPH,NOGLYPH,NOGLYPH}
};
static const kern_pair times_bolditalic_kerns[] = {
    {0,64,-65}, {0,70,-65}, {0,203,-60}, {0,393,-50}, {0,396,-50}, 
    {0,404,-50}, {0,414,-50}, {0,419,-50}, {0,449,-50}, {0,455,-50}, 
    {0,474,-55}, {0,570,-55}, {0,603,-50}, {0,604,-50}, {0,609,-50}, 
    {0,615,-50}, {0,624,-50}, {0,658,-95}, {0,668,-100}, {0,686,-70}, 
    {0,687,-70}, {0,692,-70}, {0,3143,-74}, {0,3816,-30}, {0,3817,-30}, 
    {0,3824,-30}, {0,3831,-30}, {0,3839,-30}, {0,3902,-74}, 
    {0,3951,-74}, {0,4031,-74}, {0,4034,-74}, {0,4050,-74}, {5,64,-65}, 
    {5,70,-65}, {5,203,-60}, {5,393,-50}, {5,396,-50}, {5,404,-50}, 
    {5,414,-50}, {5,419,-50}, {5,449,-50}, {5,455,-50}, {5,474,-55}, 
    {5,570,-55}, {5,603,-50}, {5,604,-50}, {5,609,-50}, {5,615,-50}, 
    {5,624,-50}, {5,658,-95}, {5,668,-100}, {5,686,-70}, {5,687,-70}, 
    {5,692,-70}, {5,3143,-74}, {5,3816,-30}, {5,3817,-30}, 
    {5,3824,-30}, {5,3831,-30}, {5,3839,-30}, {5,3902,-74}, 
    {5,3951,-74}, {5,4031,-74}, {5,4034,-74}, {5,4050,-74}, 
    {16,64,-65}, {16,70,-65}, {16,203,-60}, {16,393,-50}, {16,396,-50}, 
    {16,404,-50}, {16,414,-50}, {16,419,-50}, {16,449,-50}, 
    {16,455,-50}, {16,474,-55}, {16,570,-55}, {16,603,-50}, 
    {16,604,-50}, {16,609,-50}, {16,615,-50}, {16,624,-50}, 
    {16,658,-95}, {16,668,-100}, {16,686,-70}, {16,687,-70}, 
    {16,692,-70}, {16,3143,-74}, {16,3816,-30}, {16,3817,-30}, 
    {16,3824,-30}, {16,3831,-30}, {16,3839,-30}, {16,3902,-74}, 
    {16,3951,-74}, {16,4031,-74}, {16,4034,-74}, {16,4050,-74}, 
    {27,64,-65}, {27,70,-65}, {27,203,-60}, {27,393,-50}, {27,396,-50}, 
    {27,404,-50}, {27,414,-50}, {27,419,-50}, {27,449,-50}, 
    {27,455,-50}, {27,474,-55}, {27,570,-55}, {27,603,-50}, 
    {27,604,-50}, {27,609,-50}, {27,615,-50}, {27,624,-50}, 
    {27,658,-95}, {27,668,-100}, {27,686,-70}, {27,687,-70}, 
    {27,692,-70}, {27,3143,-74}, {27,3816,-30}, {27,3817,-30}, 
    {27,3824,-30}, {27,3831,-30}, {27,3839,-30}, {27,3902,-74}, 
    {27,3951,-74}, {27,4031,-74}, {27,4034,-74}, {27,4050,-74}, 
    {33,64,-65}, {33,70,-65}, {33,203,-60}, {33,393,-50}, {33,396,-50}, 
    {33,404,-50}, {33,414,-50}, {33,419,-50}, {33,449,-50}, 
    {33,455,-50}, {33,474,-55}, {33,570,-55}, {33,603,-50}, 
    {33,604,-50}, {33,609,-50}, {33,615,-50}, {33,624,-50}, 
    {33,658,-95}, {33,668,-100}, {33,686,-70}, {33,687,-70}, 
    {33,692,-70}, {33,3143,-74}, {33,3816,-30}, {33,3817,-30}, 
    {33,3824,-30}, {33,3831,-30}, {33,3839,-30}, {33,3902,-74}, 
    {33,3951,-74}, {33,4031,-74}, {33,4034,-74}, {33,4050,-74}, 
    {43,64,-65}, {43,70,-65}, {43,203,-60}, {43,393,-50}, {43,396,-50}, 
    {43,404,-50}, {43,414,-50}, {43,419,-50}, {43,449,-50}, 
    {43,455,-50}, {43,474,-55}, {43,570,-55}, {43,603,-50}, 
    {43,604,-50}, {43,609,-50}, {43,615,-50}, {43,624,-50}, 
    {43,658,-95}, {43,668,-100}, {43,686,-70}, {43,687,-70}, 
    {43,692,-70}, {43,3143,-74}, {43,3816,-30}, {43,3817,-30}, 
    {43,3824,-30}, {43,3831,-30}, {43,3839,-30}, {43,3902,-74}, 
    {43,3951,-74}, {43,4031,-74}, {43,4034,-74}, {43,4050,-74}, 
    {48,64,-65}, {48,70,-65}, {48,203,-60}, {48,393,-50}, {48,396,-50}, 
    {48,404,-50}, {48,414,-50}, {48,419,-50}, {48,449,-50}, 
    {48,455,-50}, {48,474,-55}, {48,570,-55}, {48,603,-50}, 
    {48,604,-50}, {48,609,-50}, {48,615,-50}, {48,624,-50}, 
    {48,658,-95}, {48,668,-100}, {48,686,-70}, {48,687,-70}, 
    {48,692,-70}, {48,3143,-74}, {48,3816,-30}, {48,3817,-30}, 
    {48,3824,-30}, {48,3831,-30}, {48,3839,-30}, {48,3902,-74}, 
    {48,3951,-74}, {48,4031,-74}, {48,4034,-74}, {48,4050,-74}, 
    {51,0,-25}, {51,5,-25}, {51,16,-25}, {51,27,-25}, {51,33,-25}, 
    {51,43,-25}, {51,48,-25}, {51,603,-10}, {51,604,-10}, {51,609,-10}, 
    {51,615,-10}, {51,624,-10}, {93,0,-25}, {93,5,-25}, {93,16,-25}, 
    {93,27,-25}, {93,33,-25}, {93,43,-25}, {93,48,-25}, {93,658,-50}, 
    {93,668,-40}, {93,686,-50}, {93,687,-50}, {93,692,-50}, 
    {192,0,-100}, {192,5,-100}, {192,16,-100}, {192,27,-100}, 
    {192,33,-100}, {192,43,-100}, {192,48,-100}, {192,734,-95}, 
    {192,736,-95}, {192,758,-95}, {192,775,-95}, {192,1030,-95}, 
    {192,1111,-95}, {192,1173,-95}, {192,1409,-129}, {192,1630,-100}, 
    {192,1631,-100}, {192,1645,-100}, {192,1655,-100}, {192,1662,-100}, 
    {192,2132,-40}, {192,2133,-40}, {192,2140,-40}, {192,2196,-40}, 
    {192,2207,-40}, {192,2794,-70}, {192,2795,-70}, {192,2809,-70}, 
    {192,2819,-70}, {192,2826,-70}, {192,2899,-70}, {192,2907,-70}, 
    {192,2995,-129}, {192,3148,-50}, {301,0,-25}, {301,5,-25}, 
    {301,16,-25}, {301,27,-25}, {301,33,-25}, {301,43,-25}, 
    {301,48,-25}, {301,734,-40}, {301,736,-40}, {301,758,-40}, 
    {301,775,-40}, {301,1030,-40}, {301,1111,-40}, {301,1173,-40}, 
    {301,1409,-10}, {301,1630,-40}, {301,1631,-40}, {301,1645,-40}, 
    {301,1655,-40}, {301,1662,-40}, {301,2794,-40}, {301,2795,-40}, 
    {301,2809,-40}, {301,2819,-40}, {301,2826,-40}, {301,2899,-40}, 
    {301,2907,-40}, {301,2995,-10}, {301,3816,-40}, {301,3817,-40}, 
    {301,3824,-40}, {301,3831,-40}, {301,3839,-40}, {309,393,-30}, 
    {309,396,-30}, {309,404,-30}, {309,414,-30}, {309,419,-30}, 
    {309,449,-30}, {309,455,-30}, {309,1630,-25}, {309,1631,-25}, 
    {309,1645,-25}, {309,1655,-25}, {309,1662,-25}, {309,2794,-25}, 
    {309,2795,-25}, {309,2809,-25}, {309,2819,-25}, {309,2826,-25}, 
    {309,2899,-25}, {309,2907,-25}, {309,3816,-20}, {309,3817,-20}, 
    {309,3824,-20}, {309,3831,-20}, {309,3839,-20}, {309,4031,-20}, 
    {309,4034,-20}, {309,4050,-20}, {337,570,-18}, {337,658,-37}, 
    {337,668,-37}, {337,686,-37}, {337,687,-37}, {337,692,-37}, 
    {337,3143,-55}, {337,4031,-37}, {337,4034,-37}, {337,4050,-37}, 
    {356,570,-18}, {356,658,-37}, {356,668,-37}, {356,686,-37}, 
    {356,687,-37}, {356,692,-37}, {356,3143,-55}, {356,4031,-37}, 
    {356,4034,-37}, {356,4050,-37}, {372,0,-30}, {372,5,-30}, 
    {372,16,-30}, {372,27,-30}, {372,33,-30}, {372,43,-30}, 
    {372,48,-30}, {390,0,-30}, {390,5,-30}, {390,16,-30}, {390,27,-30}, 
    {390,33,-30}, {390,43,-30}, {390,48,-30}, {393,0,-40}, {393,5,-40}, 
    {393,16,-40}, {393,27,-40}, {393,33,-40}, {393,43,-40}, 
    {393,48,-40}, {393,570,-40}, {393,658,-50}, {393,668,-50}, 
    {393,678,-40}, {393,686,-50}, {393,687,-50}, {393,692,-50}, 
    {396,0,-40}, {396,5,-40}, {396,16,-40}, {396,27,-40}, {396,33,-40}, 
    {396,43,-40}, {396,48,-40}, {396,570,-40}, {396,658,-50}, 
    {396,668,-50}, {396,678,-40}, {396,686,-50}, {396,687,-50}, 
    {396,692,-50}, {404,0,-40}, {404,5,-40}, {404,16,-40}, 
    {404,27,-40}, {404,33,-40}, {404,43,-40}, {404,48,-40}, 
    {404,570,-40}, {404,658,-50}, {404,668,-50}, {404,678,-40}, 
    {404,686,-50}, {404,687,-50}, {404,692,-50}, {414,0,-40}, 
    {414,5,-40}, {414,16,-40}, {414,27,-40}, {414,33,-40}, 
    {414,43,-40}, {414,48,-40}, {414,570,-40}, {414,658,-50}, 
    {414,668,-50}, {414,678,-40}, {414,686,-50}, {414,687,-50}, 
    {414,692,-50}, {419,0,-40}, {419,5,-40}, {419,16,-40}, 
    {419,27,-40}, {419,33,-40}, {419,43,-40}, {419,48,-40}, 
    {419,570,-40}, {419,658,-50}, {419,668,-50}, {419,678,-40}, 
    {419,686,-50}, {419,687,-50}, {419,692,-50}, {449,0,-40}, 
    {449,5,-40}, {449,16,-40}, {449,27,-40}, {449,33,-40}, 
    {449,43,-40}, {449,48,-40}, {449,570,-40}, {449,658,-50}, 
    {449,668,-50}, {449,678,-40}, {449,686,-50}, {449,687,-50}, 
    {449,692,-50}, {455,0,-40}, {455,5,-40}, {455,16,-40}, 
    {455,27,-40}, {455,33,-40}, {455,43,-40}, {455,48,-40}, 
    {455,570,-40}, {455,658,-50}, {455,668,-50}, {455,678,-40}, 
    {455,686,-50}, {455,687,-50}, {455,692,-50}, {459,0,-85}, 
    {459,5,-85}, {459,16,-85}, {459,27,-85}, {459,33,-85}, 
    {459,43,-85}, {459,48,-85}, {459,734,-40}, {459,736,-40}, 
    {459,758,-40}, {459,775,-40}, {459,1030,-40}, {459,1111,-40}, 
    {459,1173,-40}, {459,1409,-129}, {459,1630,-50}, {459,1631,-50}, 
    {459,1645,-50}, {459,1655,-50}, {459,1662,-50}, {459,2794,-55}, 
    {459,2795,-55}, {459,2809,-55}, {459,2819,-55}, {459,2826,-55}, 
    {459,2899,-55}, {459,2907,-55}, {459,2995,-129}, {474,603,-10}, 
    {474,604,-10}, {474,609,-10}, {474,615,-10}, {474,624,-10}, 
    {478,393,-40}, {478,396,-40}, {478,404,-40}, {478,414,-40}, 
    {478,419,-40}, {478,449,-40}, {478,455,-40}, {478,570,-30}, 
    {478,603,-40}, {478,604,-40}, {478,609,-40}, {478,615,-40}, 
    {478,624,-40}, {478,658,-18}, {478,668,-18}, {478,686,-18}, 
    {478,687,-18}, {478,692,-18}, {570,0,-55}, {570,5,-55}, 
    {570,16,-55}, {570,27,-55}, {570,33,-55}, {570,43,-55}, 
    {570,48,-55}, {570,393,-18}, {570,396,-18}, {570,404,-18}, 
    {570,414,-18}, {570,419,-18}, {570,449,-18}, {570,455,-18}, 
    {570,734,-92}, {570,736,-92}, {570,758,-92}, {570,775,-92}, 
    {570,1030,-92}, {570,1111,-92}, {570,1173,-92}, {570,1402,-74}, 
    {570,1409,-92}, {570,1630,-92}, {570,1631,-92}, {570,1645,-92}, 
    {570,1655,-52}, {570,1662,-52}, {570,2126,-92}, {570,2132,-37}, 
    {570,2133,-37}, {570,2794,-95}, {570,2795,-95}, {570,2809,-95}, 
    {570,2819,-95}, {570,2826,-95}, {570,2899,-95}, {570,2907,-95}, 
    {570,2995,-92}, {570,3148,-37}, {570,3351,-74}, {570,3816,-37}, 
    {570,3817,-37}, {570,3824,-37}, {570,3831,-37}, {570,3839,-37}, 
    {570,3951,-37}, {570,4031,-37}, {570,4034,-37}, {570,4050,-37}, 
    {603,0,-45}, {603,5,-45}, {603,16,-45}, {603,27,-45}, {603,33,-45}, 
    {603,43,-45}, {603,48,-45}, {604,0,-45}, {604,5,-45}, {604,16,-45}, 
    {604,27,-45}, {604,33,-45}, {604,43,-45}, {604,48,-45}, 
    {609,0,-45}, {609,5,-45}, {609,16,-45}, {609,27,-45}, {609,33,-45}, 
    {609,43,-45}, {609,48,-45}, {615,0,-45}, {615,5,-45}, {615,16,-45}, 
    {615,27,-45}, {615,33,-45}, {615,43,-45}, {615,48,-45}, 
    {624,0,-45}, {624,5,-45}, {624,16,-45}, {624,27,-45}, {624,33,-45}, 
    {624,43,-45}, {624,48,-45}, {658,0,-85}, {658,5,-85}, {658,16,-85}, 
    {658,27,-85}, {658,33,-85}, {658,43,-85}, {658,48,-85}, 
    {658,203,-10}, {658,393,-30}, {658,396,-30}, {658,404,-30}, 
    {658,414,-30}, {658,419,-30}, {658,449,-30}, {658,455,-30}, 
    {658,734,-111}, {658,736,-111}, {658,758,-111}, {658,775,-111}, 
    {658,1030,-111}, {658,1111,-111}, {658,1173,-111}, {658,1402,-74}, 
    {658,1409,-129}, {658,1630,-111}, {658,1631,-111}, {658,1645,-111}, 
    {658,1655,-71}, {658,1662,-71}, {658,2126,-70}, {658,2132,-55}, 
    {658,2133,-55}, {658,2794,-111}, {658,2795,-111}, {658,2809,-111}, 
    {658,2819,-111}, {658,2826,-111}, {658,2899,-111}, {658,2907,-111}, 
    {658,2995,-129}, {658,3351,-74}, {658,3816,-55}, {658,3817,-55}, 
    {658,3824,-55}, {658,3831,-55}, {658,3839,-55}, {668,0,-74}, 
    {668,5,-74}, {668,16,-74}, {668,27,-74}, {668,33,-74}, 
    {668,43,-74}, {668,48,-74}, {668,393,-15}, {668,396,-15}, 
    {668,404,-15}, {668,414,-15}, {668,419,-15}, {668,449,-15}, 
    {668,455,-15}, {668,734,-85}, {668,736,-85}, {668,758,-85}, 
    {668,775,-85}, {668,1030,-85}, {668,1111,-85}, {668,1173,-85}, 
    {668,1402,-55}, {668,1409,-74}, {668,1630,-90}, {668,1631,-90}, 
    {668,1645,-90}, {668,1655,-50}, {668,1662,-50}, {668,2126,-50}, 
    {668,2132,-37}, {668,2133,-37}, {668,2794,-80}, {668,2795,-80}, 
    {668,2809,-80}, {668,2819,-80}, {668,2826,-80}, {668,2899,-80}, 
    {668,2907,-80}, {668,2995,-74}, {668,3351,-55}, {668,3816,-55}, 
    {668,3817,-55}, {668,3824,-55}, {668,3831,-55}, {668,3839,-55}, 
    {668,4031,-55}, {668,4034,-55}, {668,4050,-55}, {686,0,-74}, 
    {686,5,-74}, {686,16,-74}, {686,27,-74}, {686,33,-74}, 
    {686,43,-74}, {686,48,-74}, {686,393,-25}, {686,396,-25}, 
    {686,404,-25}, {686,414,-25}, {686,419,-25}, {686,449,-25}, 
    {686,455,-25}, {686,734,-92}, {686,736,-92}, {686,758,-92}, 
    {686,775,-92}, {686,1030,-92}, {686,1111,-92}, {686,1173,-92}, 
    {686,1402,-92}, {686,1409,-92}, {686,1630,-111}, {686,1631,-111}, 
    {686,1645,-71}, {686,1655,-71}, {686,1662,-71}, {686,2126,-92}, 
    {686,2132,-55}, {686,2133,-55}, {686,2794,-111}, {686,2795,-111}, 
    {686,2809,-111}, {686,2819,-111}, {686,2826,-111}, {686,2899,-111}, 
    {686,2907,-111}, {686,2995,-74}, {686,3351,-92}, {686,3816,-92}, 
    {686,3817,-92}, {686,3824,-92}, {686,3831,-92}, {686,3839,-92}, 
    {687,0,-74}, {687,5,-74}, {687,16,-74}, {687,27,-74}, {687,33,-74}, 
    {687,43,-74}, {687,48,-74}, {687,393,-25}, {687,396,-25}, 
    {687,404,-25}, {687,414,-25}, {687,419,-25}, {687,449,-25}, 
    {687,455,-25}, {687,734,-92}, {687,736,-92}, {687,758,-92}, 
    {687,775,-92}, {687,1030,-92}, {687,1111,-92}, {687,1173,-92}, 
    {687,1402,-92}, {687,1409,-92}, {687,1630,-111}, {687,1631,-111}, 
    {687,1645,-71}, {687,1655,-71}, {687,1662,-71}, {687,2126,-92}, 
    {687,2132,-55}, {687,2133,-55}, {687,2794,-111}, {687,2795,-111}, 
    {687,2809,-111}, {687,2819,-111}, {687,2826,-111}, {687,2899,-111}, 
    {687,2907,-111}, {687,2995,-74}, {687,3351,-92}, {687,3816,-92}, 
    {687,3817,-92}, {687,3824,-92}, {687,3831,-92}, {687,3839,-92}, 
    {692,0,-74}, {692,5,-74}, {692,16,-74}, {692,27,-74}, {692,33,-74}, 
    {692,43,-74}, {692,48,-74}, {692,393,-25}, {692,396,-25}, 
    {692,404,-25}, {692,414,-25}, {692,419,-25}, {692,449,-25}, 
    {692,455,-25}, {692,734,-92}, {692,736,-92}, {692,758,-92}, 
    {692,775,-92}, {692,1030,-92}, {692,1111,-92}, {692,1173,-92}, 
    {692,1402,-92}, {692,1409,-92}, {692,1630,-111}, {692,1631,-111}, 
    {692,1645,-71}, {692,1655,-71}, {692,1662,-71}, {692,2126,-92}, 
    {692,2132,-55}, {692,2133,-55}, {692,2794,-111}, {692,2795,-111}, 
    {692,2809,-111}, {692,2819,-111}, {692,2826,-111}, {692,2899,-111}, 
    {692,2907,-111}, {692,2995,-74}, {692,3351,-92}, {692,3816,-92}, 
    {692,3817,-92}, {692,3824,-92}, {692,3831,-92}, {692,3839,-92}, 
    {1192,1192,-10}, {1192,2995,-40}, {1192,3816,-20}, {1192,3817,-20}, 
    {1192,3824,-20}, {1192,3831,-20}, {1192,3839,-20}, {1312,1988,-10}, 
    {1312,2302,-10}, {1332,1988,-10}, {1332,2302,-10}, {1409,3139,-95}, 
    {1409,3143,-95}, {1630,1192,-10}, {1631,1192,-10}, {1645,1192,-10}, 
    {1655,1192,-10}, {1662,1192,-10}, {1787,1409,-10}, {1787,1608,-30}, 
    {1787,1630,-10}, {1787,1631,-10}, {1787,1787,-18}, {1787,2794,-10}, 
    {1787,2795,-10}, {1787,2809,-10}, {1787,2826,-10}, {1787,2899,-10}, 
    {1787,2907,-10}, {1787,2995,-10}, {1787,3143,55}, {2302,1630,-30}, 
    {2302,1631,-30}, {2302,1645,-30}, {2302,1655,-30}, {2302,1662,-30}, 
    {2302,2794,-10}, {2302,2795,-10}, {2302,2809,-10}, {2302,2819,-10}, 
    {2302,2826,-10}, {2302,2899,-10}, {2302,2907,-10}, {2648,3902,-40}, 
    {2769,3902,-40}, {2794,3902,-15}, {2794,3951,-25}, {2794,4020,-10}, 
    {2794,4031,-10}, {2794,4034,-10}, {2794,4050,-10}, {2795,3902,-15}, 
    {2795,3951,-25}, {2795,4020,-10}, {2795,4031,-10}, {2795,4034,-10}, 
    {2795,4050,-10}, {2809,3902,-15}, {2809,3951,-25}, {2809,4020,-10}, 
    {2809,4031,-10}, {2809,4034,-10}, {2809,4050,-10}, {2819,3902,-15}, 
    {2819,3951,-25}, {2819,4020,-10}, {2819,4031,-10}, {2819,4034,-10}, 
    {2819,4050,-10}, {2826,3902,-15}, {2826,3951,-25}, {2826,4020,-10}, 
    {2826,4031,-10}, {2826,4034,-10}, {2826,4050,-10}, {2899,3902,-15}, 
    {2899,3951,-25}, {2899,4020,-10}, {2899,4031,-10}, {2899,4034,-10}, 
    {2899,4050,-10}, {2907,3902,-15}, {2907,3951,-25}, {2907,4020,-10}, 
    {2907,4031,-10}, {2907,4034,-10}, {2907,4050,-10}, {2995,3139,-95}, 
    {2995,3143,-95}, {3140,3140,-74}, {3143,1481,-15}, {3143,3143,-74}, 
    {3143,3148,-15}, {3143,3279,-74}, {3143,3317,-74}, {3143,3498,-74}, 
    {3143,3556,-37}, {3143,3902,-15}, {3148,1409,-65}, {3148,2995,-65}, 
    {3498,0,-37}, {3498,5,-37}, {3498,16,-37}, {3498,27,-37}, 
    {3498,33,-37}, {3498,43,-37}, {3498,48,-37}, {3498,658,-70}, 
    {3498,668,-70}, {3498,686,-70}, {3498,687,-70}, {3498,692,-70}, 
    {3902,1409,-37}, {3902,1630,-15}, {3902,1631,-15}, {3902,1645,-15}, 
    {3902,1655,-15}, {3902,1662,-15}, {3902,2794,-15}, {3902,2795,-15}, 
    {3902,2809,-15}, {3902,2819,-15}, {3902,2826,-15}, {3902,2899,-15}, 
    {3902,2907,-15}, {3902,2995,-37}, {3951,734,-10}, {3951,736,-10}, 
    {3951,758,-10}, {3951,775,-10}, {3951,1030,-10}, {3951,1111,-10}, 
    {3951,1173,-10}, {3951,1409,-37}, {3951,1630,-10}, {3951,1631,-10}, 
    {3951,1645,-10}, {3951,1655,-10}, {3951,1662,-10}, {3951,2794,-15}, 
    {3951,2795,-15}, {3951,2809,-15}, {3951,2819,-15}, {3951,2826,-15}, 
    {3951,2899,-15}, {3951,2907,-15}, {3951,2995,-37}, {4020,1630,-10}, 
    {4020,1631,-10}, {4020,1645,-10}, {4020,1655,-10}, {4020,1662,-10}, 
    {4031,1409,-37}, {4031,2995,-37}, {4034,1409,-37}, {4034,2995,-37}, 
    {4050,1409,-37}, {4050,2995,-37}, 
    {NOGLYPH,NOGLYPH,0}
};
static const ligature times_bolditalic_ligs[] = {
    {1787,2132,1807}, {1787,2416,1850}, 
    {NOGLYPH,NOGLYPH,NOGLYPH}
};
static const kern_pair helvetica_kerns[] = {
    {0,64,-30}, {0,70,-30}, {0,203,-30}, {0,393,-30}, {0,396,-30}, 
    {0,404,-30}, {0,414,-30}, {0,419,-30}, {0,449,-30}, {0,455,-30}, 
    {0,474,-30}, {0,570,-120}, {0,603,-50}, {0,604,-50}, {0,609,-50}, 
    {0,615,-50}, {0,624,-50}, {0,658,-70}, {0,668,-50}, {0,686,-100}, 
    {0,687,-100}, {0,692,-100}, {0,3816,-30}, {0,3817,-30}, 
    {0,3824,-30}, {0,3831,-30}, {0,3839,-30}, {0,3902,-40}, 
    {0,3951,-40}, {0,4031,-40}, {0,4034,-40}, {0,4050,-40}, {5,64,-30}, 
    {5,70,-30}, {5,203,-30}, {5,393,-30}, {5,396,-30}, {5,404,-30}, 
    {5,414,-30}, {5,419,-30}, {5,449,-30}, {5,455,-30}, {5,474,-30}, 
    {5,570,-120}, {5,603,-50}, {5,604,-50}, {5,609,-50}, {5,615,-50}, 
    {5,624,-50}, {5,658,-70}, {5,668,-50}, {5,686,-100}, {5,687,-100}, 
    {5,692,-100}, {5,3816,-30}, {5,3817,-30}, {5,3824,-30}, 
    {5,3831,-30}, {5,3839,-30}, {5,3902,-40}, {5,3951,-40}, 
    {5,4031,-40}, {5,4034,-40}, {5,4050,-40}, {16,64,-30}, {16,70,-30}, 
    {16,203,-30}, {16,393,-30}, {16,396,-30}, {16,404,-30}, 
    {16,414,-30}, {16,419,-30}, {16,449,-30}, {16,455,-30}, 
    {16,474,-30}, {16,570,-120}, {16,603,-50}, {16,604,-50}, 
    {16,609,-50}, {16,615,-50}, {16,624,-50}, {16,658,-70}, 
    {16,668,-50}, {16,686,-100}, {16,687,-100}, {16,692,-100}, 
    {16,3816,-30}, {16,3817,-30}, {16,3824,-30}, {16,3831,-30}, 
    {16,3839,-30}, {16,3902,-40}, {16,3951,-40}, {16,4031,-40}, 
    {16,4034,-40}, {16,4050,-40}, {27,64,-30}, {27,70,-30}, 
    {27,203,-30}, {27,393,-30}, {27,396,-30}, {27,404,-30}, 
    {27,414,-30}, {27,419,-30}, {27,449,-30}, {27,455,-30}, 
    {27,474,-30}, {27,570,-120}, {27,603,-50}, {27,604,-50}, 
    {27,609,-50}, {27,615,-50}, {27,624,-50}, {27,658,-70}, 
    {27,668,-50}, {27,686,-100}, {27,687,-100}, {27,692,-100}, 
    {27,3816,-30}, {27,3817,-30}, {27,3824,-30}, {27,3831,-30}, 
    {27,3839,-30}, {27,3902,-40}, {27,3951,-40}, {27,4031,-40}, 
    {27,4034,-40}, {27,4050,-40}, {33,64,-30}, {33,70,-30}, 
    {33,203,-30}, {33,393,-30}, {33,396,-30}, {33,404,-30}, 
    {33,414,-30}, {33,419,-30}, {33,449,-30}, {33,455,-30}, 
    {33,474,-30}, {33,570,-120}, {33,603,-50}, {33,604,-50}, 
    {33,609,-50}, {33,615,-50}, {33,624,-50}, {33,658,-70}, 
    {33,668,-50}, {33,686,-100}, {33,687,-100}, {33,692,-100}, 
    {33,3816,-30}, {33,3817,-30}, {33,3824,-30}, {33,3831,-30}, 
    {33,3839,-30}, {33,3902,-40}, {33,3951,-40}, {33,4031,-40}, 
    {33,4034,-40}, {33,4050,-40}, {43,64,-30}, {43,70,-30}, 
    {43,203,-30}, {43,393,-30}, {43,396,-30}, {43,404,-30}, 
    {43,414,-30}, {43,419,-30}, {43,449,-30}, {43,455,-30}, 
    {43,474,-30}, {43,570,-120}, {43,603,-50}, {43,604,-50}, 
    {43,609,-50}, {43,615,-50}, {43,624,-50}, {43,658,-70}, 
    {43,668,-50}, {43,686,-100}, {43,687,-100}, {43,692,-100}, 
    {43,3816,-30}, {43,3817,-30}, {43,3824,-30}, {43,3831,-30}, 
    {43,3839,-30}, {43,3902,-40}, {43,3951,-40}, {43,4031,-40}, 
    {43,4034,-40}, {43,4050,-40}, {48,64,-30}, {48,70,-30}, 
    {48,203,-30}, {48,393,-30}, {48,396,-30}, {48,404,-30}, 
    {48,414,-30}, {48,419,-30}, {48,449,-30}, {48,455,-30}, 
    {48,474,-30}, {48,570,-120}, {48,603,-50}, {48,604,-50}, 
    {48,609,-50}, {48,615,-50}, {48,624,-50}, {48,658,-70}, 
    {48,668,-50}, {48,686,-100}, {48,687,-100}, {48,692,-100}, 
    {48,3816,-30}, {48,3817,-30}, {48,3824,-30}, {48,3831,-30}, 
    {48,3839,-30}, {48,3902,-40}, {48,3951,-40}, {48,4031,-40}, 
    {48,4034,-40}, {48,4050,-40}, {51,603,-10}, {51,604,-10}, 
    {51,609,-10}, {51,615,-10}, {51,624,-10}, {51,1409,-20}, 
    {51,2995,-20}, {64,1409,-30}, {64,2995,-30}, {70,1409,-30}, 
    {70,2995,-30}, {93,0,-40}, {93,5,-40}, {93,16,-40}, {93,27,-40}, 
    {93,33,-40}, {93,43,-40}, {93,48,-40}, {93,658,-70}, {93,668,-40}, 
    {93,686,-90}, {93,687,-90}, {93,692,-90}, {93,1409,-70}, 
    {93,2995,-70}, {192,0,-80}, {192,5,-80}, {192,16,-80}, 
    {192,27,-80}, {192,33,-80}, {192,43,-80}, {192,48,-80}, 
    {192,734,-50}, {192,736,-50}, {192,758,-50}, {192,775,-50}, 
    {192,1030,-50}, {192,1111,-50}, {192,1173,-50}, {192,1409,-150}, 
    {192,1630,-30}, {192,1631,-30}, {192,1645,-30}, {192,1655,-30}, 
    {192,1662,-30}, {192,2794,-30}, {192,2795,-30}, {192,2809,-30}, 
    {192,2819,-30}, {192,2826,-30}, {192,2899,-30}, {192,2907,-30}, 
    {192,2995,-150}, {192,3148,-45}, {301,0,-20}, {301,5,-20}, 
    {301,16,-20}, {301,27,-20}, {301,33,-20}, {301,43,-20}, 
    {301,48,-20}, {301,734,-20}, {301,736,-20}, {301,758,-20}, 
    {301,775,-20}, {301,1030,-20}, {301,1111,-20}, {301,1173,-20}, 
    {301,1409,-30}, {301,2995,-30}, {301,3816,-20}, {301,3817,-20}, 
    {301,3824,-20}, {301,3831,-20}, {301,3839,-20}, {309,393,-50}, 
    {309,396,-50}, {309,404,-50}, {309,414,-50}, {309,419,-50}, 
    {309,449,-50}, {309,455,-50}, {309,1630,-40}, {309,1631,-40}, 
    {309,1645,-40}, {309,1655,-40}, {309,1662,-40}, {309,2794,-40}, 
    {309,2795,-40}, {309,2809,-40}, {309,2819,-40}, {309,2826,-40}, 
    {309,2899,-40}, {309,2907,-40}, {309,3816,-30}, {309,3817,-30}, 
    {309,3824,-30}, {309,3831,-30}, {309,3839,-30}, {309,4031,-50}, 
    {309,4034,-50}, {309,4050,-50}, {337,570,-110}, {337,658,-110}, 
    {337,668,-70}, {337,686,-140}, {337,687,-140}, {337,692,-140}, 
    {337,3139,-140}, {337,3143,-160}, {337,4031,-30}, {337,4034,-30}, 
    {337,4050,-30}, {356,570,-110}, {356,658,-110}, {356,668,-70}, 
    {356,686,-140}, {356,687,-140}, {356,692,-140}, {356,3139,-140}, 
    {356,3143,-160}, {356,4031,-30}, {356,4034,-30}, {356,4050,-30}, 
    {393,0,-20}, {393,5,-20}, {393,16,-20}, {393,27,-20}, {393,33,-20}, 
    {393,43,-20}, {393,48,-20}, {393,570,-40}, {393,658,-50}, 
    {393,668,-30}, {393,678,-60}, {393,686,-70}, {393,687,-70}, 
    {393,692,-70}, {393,1409,-40}, {393,2995,-40}, {396,0,-20}, 
    {396,5,-20}, {396,16,-20}, {396,27,-20}, {396,33,-20}, 
    {396,43,-20}, {396,48,-20}, {396,570,-40}, {396,658,-50}, 
    {396,668,-30}, {396,678,-60}, {396,686,-70}, {396,687,-70}, 
    {396,692,-70}, {396,1409,-40}, {396,2995,-40}, {404,0,-20}, 
    {404,5,-20}, {404,16,-20}, {404,27,-20}, {404,33,-20}, 
    {404,43,-20}, {404,48,-20}, {404,570,-40}, {404,658,-50}, 
    {404,668,-30}, {404,678,-60}, {404,686,-70}, {404,687,-70}, 
    {404,692,-70}, {404,1409,-40}, {404,2995,-40}, {414,0,-20}, 
    {414,5,-20}, {414,16,-20}, {414,27,-20}, {414,33,-20}, 
    {414,43,-20}, {414,48,-20}, {414,570,-40}, {414,658,-50}, 
    {414,668,-30}, {414,678,-60}, {414,686,-70}, {414,687,-70}, 
    {414,692,-70}, {414,1409,-40}, {414,2995,-40}, {419,0,-20}, 
    {419,5,-20}, {419,16,-20}, {419,27,-20}, {419,33,-20}, 
    {419,43,-20}, {419,48,-20}, {419,570,-40}, {419,658,-50}, 
    {419,668,-30}, {419,678,-60}, {419,686,-70}, {419,687,-70}, 
    {419,692,-70}, {419,1409,-40}, {419,2995,-40}, {449,0,-20}, 
    {449,5,-20}, {449,16,-20}, {449,27,-20}, {449,33,-20}, 
    {449,43,-20}, {449,48,-20}, {449,570,-40}, {449,658,-50}, 
    {449,668,-30}, {449,678,-60}, {449,686,-70}, {449,687,-70}, 
    {449,692,-70}, {449,1409,-40}, {449,2995,-40}, {455,0,-20}, 
    {455,5,-20}, {455,16,-20}, {455,27,-20}, {455,33,-20}, 
    {455,43,-20}, {455,48,-20}, {455,570,-40}, {455,658,-50}, 
    {455,668,-30}, {455,678,-60}, {455,686,-70}, {455,687,-70}, 
    {455,692,-70}, {455,1409,-40}, {455,2995,-40}, {459,0,-120}, 
    {459,5,-120}, {459,16,-120}, {459,27,-120}, {459,33,-120}, 
    {459,43,-120}, {459,48,-120}, {459,734,-40}, {459,736,-40}, 
    {459,758,-40}, {459,775,-40}, {459,1030,-40}, {459,1111,-40}, 
    {459,1173,-40}, {459,1409,-180}, {459,1630,-50}, {459,1631,-50}, 
    {459,1645,-50}, {459,1655,-50}, {459,1662,-50}, {459,2794,-50}, 
    {459,2795,-50}, {459,2809,-50}, {459,2819,-50}, {459,2826,-50}, 
    {459,2899,-50}, {459,2907,-50}, {459,2995,-180}, {474,603,-10}, 
    {474,604,-10}, {474,609,-10}, {474,615,-10}, {474,624,-10}, 
    {478,393,-20}, {478,396,-20}, {478,404,-20}, {478,414,-20}, 
    {478,419,-20}, {478,449,-20}, {478,455,-20}, {478,570,-30}, 
    {478,603,-40}, {478,604,-40}, {478,609,-40}, {478,615,-40}, 
    {478,624,-40}, {478,658,-50}, {478,668,-30}, {478,686,-50}, 
    {478,687,-50}, {478,692,-50}, {499,1409,-20}, {499,2995,-20}, 
    {543,1409,-20}, {543,2995,-20}, {570,0,-120}, {570,5,-120}, 
    {570,16,-120}, {570,27,-120}, {570,33,-120}, {570,43,-120}, 
    {570,48,-120}, {570,393,-40}, {570,396,-40}, {570,404,-40}, 
    {570,414,-40}, {570,419,-40}, {570,449,-40}, {570,455,-40}, 
    {570,734,-120}, {570,736,-120}, {570,758,-120}, {570,775,-120}, 
    {570,1030,-120}, {570,1111,-120}, {570,1173,-60}, {570,1402,-20}, 
    {570,1409,-120}, {570,1630,-120}, {570,1631,-120}, {570,1645,-120}, 
    {570,1655,-120}, {570,1662,-60}, {570,2126,-140}, {570,2794,-120}, 
    {570,2795,-120}, {570,2809,-120}, {570,2819,-120}, {570,2826,-120}, 
    {570,2899,-120}, {570,2907,-60}, {570,2995,-120}, {570,3148,-120}, 
    {570,3351,-20}, {570,3816,-120}, {570,3817,-120}, {570,3824,-120}, 
    {570,3831,-120}, {570,3839,-120}, {570,3951,-120}, {570,4031,-120}, 
    {570,4034,-120}, {570,4050,-60}, {603,0,-40}, {603,5,-40}, 
    {603,16,-40}, {603,27,-40}, {603,33,-40}, {603,43,-40}, 
    {603,48,-40}, {603,1409,-40}, {603,2995,-40}, {604,0,-40}, 
    {604,5,-40}, {604,16,-40}, {604,27,-40}, {604,33,-40}, 
    {604,43,-40}, {604,48,-40}, {604,1409,-40}, {604,2995,-40}, 
    {609,0,-40}, {609,5,-40}, {609,16,-40}, {609,27,-40}, {609,33,-40}, 
    {609,43,-40}, {609,48,-40}, {609,1409,-40}, {609,2995,-40}, 
    {615,0,-40}, {615,5,-40}, {615,16,-40}, {615,27,-40}, {615,33,-40}, 
    {615,43,-40}, {615,48,-40}, {615,1409,-40}, {615,2995,-40}, 
    {624,0,-40}, {624,5,-40}, {624,16,-40}, {624,27,-40}, {624,33,-40}, 
    {624,43,-40}, {624,48,-40}, {624,1409,-40}, {624,2995,-40}, 
    {658,0,-80}, {658,5,-80}, {658,16,-80}, {658,27,-80}, {658,33,-80}, 
    {658,43,-80}, {658,48,-80}, {658,203,-40}, {658,393,-40}, 
    {658,396,-40}, {658,404,-40}, {658,414,-40}, {658,419,-40}, 
    {658,449,-40}, {658,455,-40}, {658,734,-70}, {658,736,-70}, 
    {658,758,-70}, {658,775,-70}, {658,1030,-70}, {658,1111,-70}, 
    {658,1173,-70}, {658,1402,-40}, {658,1409,-125}, {658,1630,-80}, 
    {658,1631,-80}, {658,1645,-80}, {658,1655,-80}, {658,1662,-80}, 
    {658,2126,-80}, {658,2794,-80}, {658,2795,-80}, {658,2809,-80}, 
    {658,2819,-80}, {658,2826,-80}, {658,2899,-80}, {658,2907,-80}, 
    {658,2995,-125}, {658,3351,-40}, {658,3816,-70}, {658,3817,-70}, 
    {658,3824,-70}, {658,3831,-70}, {658,3839,-70}, {668,0,-50}, 
    {668,5,-50}, {668,16,-50}, {668,27,-50}, {668,33,-50}, 
    {668,43,-50}, {668,48,-50}, {668,393,-20}, {668,396,-20}, 
    {668,404,-20}, {668,414,-20}, {668,419,-20}, {668,449,-20}, 
    {668,455,-20}, {668,734,-40}, {668,736,-40}, {668,758,-40}, 
    {668,775,-40}, {668,1030,-40}, {668,1111,-40}, {668,1173,-40}, 
    {668,1409,-80}, {668,1630,-30}, {668,1631,-30}, {668,1645,-30}, 
    {668,1655,-30}, {668,1662,-30}, {668,2126,-40}, {668,2794,-30}, 
    {668,2795,-30}, {668,2809,-30}, {668,2819,-30}, {668,2826,-30}, 
    {668,2899,-30}, {668,2907,-30}, {668,2995,-80}, {668,3816,-30}, 
    {668,3817,-30}, {668,3824,-30}, {668,3831,-30}, {668,3839,-30}, 
    {668,4031,-20}, {668,4034,-20}, {668,4050,-20}, {686,0,-110}, 
    {686,5,-110}, {686,16,-110}, {686,27,-110}, {686,33,-110}, 
    {686,43,-110}, {686,48,-110}, {686,393,-85}, {686,396,-85}, 
    {686,404,-85}, {686,414,-85}, {686,419,-85}, {686,449,-85}, 
    {686,455,-85}, {686,734,-140}, {686,736,-140}, {686,758,-140}, 
    {686,775,-140}, {686,1030,-140}, {686,1111,-140}, {686,1173,-140}, 
    {686,1402,-60}, {686,1409,-140}, {686,1630,-140}, {686,1631,-140}, 
    {686,1645,-140}, {686,1655,-140}, {686,1662,-140}, {686,2126,-140}, 
    {686,2132,-20}, {686,2133,-20}, {686,2794,-140}, {686,2795,-140}, 
    {686,2809,-140}, {686,2819,-140}, {686,2826,-140}, {686,2899,-140}, 
    {686,2907,-140}, {686,2995,-140}, {686,3351,-60}, {686,3816,-110}, 
    {686,3817,-110}, {686,3824,-110}, {686,3831,-110}, {686,3839,-110}, 
    {687,0,-110}, {687,5,-110}, {687,16,-110}, {687,27,-110}, 
    {687,33,-110}, {687,43,-110}, {687,48,-110}, {687,393,-85}, 
    {687,396,-85}, {687,404,-85}, {687,414,-85}, {687,419,-85}, 
    {687,449,-85}, {687,455,-85}, {687,734,-140}, {687,736,-140}, 
    {687,758,-140}, {687,775,-140}, {687,1030,-140}, {687,1111,-140}, 
    {687,1173,-70}, {687,1402,-60}, {687,1409,-140}, {687,1630,-140}, 
    {687,1631,-140}, {687,1645,-140}, {687,1655,-140}, {687,1662,-140}, 
    {687,2126,-140}, {687,2132,-20}, {687,2133,-20}, {687,2794,-140}, 
    {687,2795,-140}, {687,2809,-140}, {687,2819,-140}, {687,2826,-140}, 
    {687,2899,-140}, {687,2907,-140}, {687,2995,-140}, {687,3351,-60}, 
    {687,3816,-110}, {687,3817,-110}, {687,3824,-110}, {687,3831,-110}, 
    {687,3839,-110}, {692,0,-110}, {692,5,-110}, {692,16,-110}, 
    {692,27,-110}, {692,33,-110}, {692,43,-110}, {692,48,-110}, 
    {692,393,-85}, {692,396,-85}, {692,404,-85}, {692,414,-85}, 
    {692,419,-85}, {692,449,-85}, {692,455,-85}, {692,734,-140}, 
    {692,736,-140}, {692,758,-140}, {692,775,-140}, {692,1030,-140}, 
    {692,1111,-140}, {692,1173,-70}, {692,1402,-60}, {692,1409,-140}, 
    {692,1630,-140}, {692,1631,-140}, {692,1645,-140}, {692,1655,-140}, 
    {692,1662,-140}, {692,2126,-140}, {692,2132,-20}, {692,2133,-20}, 
    {692,2794,-140}, {692,2795,-140}, {692,2809,-140}, {692,2819,-140}, 
    {692,2826,-140}, {692,2899,-140}, {692,2907,-140}, {692,2995,-140}, 
    {692,3351,-60}, {692,3816,-110}, {692,3817,-110}, {692,3824,-110}, 
    {692,3831,-110}, {692,3839,-110}, {734,3902,-20}, {734,3951,-20}, 
    {734,4031,-30}, {734,4034,-30}, {734,4050,-30}, {736,3902,-20}, 
    {736,3951,-20}, {736,4031,-30}, {736,4034,-30}, {736,4050,-30}, 
    {758,3902,-20}, {758,3951,-20}, {758,4031,-30}, {758,4034,-30}, 
    {758,4050,-30}, {775,3902,-20}, {775,3951,-20}, {775,4031,-30}, 
    {775,4034,-30}, {775,4050,-30}, {1030,3902,-20}, {1030,3951,-20}, 
    {1030,4031,-30}, {1030,4034,-30}, {1030,4050,-30}, {1111,3902,-20}, 
    {1111,3951,-20}, {1111,4031,-30}, {1111,4034,-30}, {1111,4050,-30}, 
    {1173,3902,-20}, {1173,3951,-20}, {1173,4031,-30}, {1173,4034,-30}, 
    {1173,4050,-30}, {1192,1192,-10}, {1192,1409,-40}, {1192,2416,-20}, 
    {1192,2501,-20}, {1192,2995,-40}, {1192,3816,-20}, {1192,3817,-20}, 
    {1192,3824,-20}, {1192,3831,-20}, {1192,3839,-20}, {1192,3902,-20}, 
    {1192,4031,-20}, {1192,4034,-20}, {1192,4050,-20}, {1312,1409,-15}, 
    {1312,2302,-20}, {1332,1409,-15}, {1332,2302,-20}, {1402,3498,-50}, 
    {1409,3139,-100}, {1409,3143,-100}, {1630,1409,-15}, 
    {1630,2995,-15}, {1630,3902,-30}, {1630,3951,-20}, {1630,4020,-30}, 
    {1630,4031,-20}, {1630,4034,-20}, {1630,4050,-20}, {1631,1409,-15}, 
    {1631,2995,-15}, {1631,3902,-30}, {1631,3951,-20}, {1631,4020,-30}, 
    {1631,4031,-20}, {1631,4034,-20}, {1631,4050,-20}, {1645,1409,-15}, 
    {1645,2995,-15}, {1645,3902,-30}, {1645,3951,-20}, {1645,4020,-30}, 
    {1645,4031,-20}, {1645,4034,-20}, {1645,4050,-20}, {1655,1409,-15}, 
    {1655,2995,-15}, {1655,3902,-30}, {1655,3951,-20}, {1655,4020,-30}, 
    {1655,4031,-20}, {1655,4034,-20}, {1655,4050,-20}, {1662,1409,-15}, 
    {1662,2995,-15}, {1662,3902,-30}, {1662,3951,-20}, {1662,4020,-30}, 
    {1662,4031,-20}, {1662,4034,-20}, {1662,4050,-20}, {1787,734,-30}, 
    {1787,736,-30}, {1787,758,-30}, {1787,775,-30}, {1787,1030,-30}, 
    {1787,1111,-30}, {1787,1173,-30}, {1787,1409,-30}, {1787,1608,-28}, 
    {1787,1630,-30}, {1787,1631,-30}, {1787,1645,-30}, {1787,1655,-30}, 
    {1787,1662,-30}, {1787,2794,-30}, {1787,2795,-30}, {1787,2809,-30}, 
    {1787,2819,-30}, {1787,2826,-30}, {1787,2899,-30}, {1787,2907,-30}, 
    {1787,2995,-30}, {1787,3139,60}, {1787,3143,50}, {1886,3148,-10}, 
    {1988,4031,-30}, {1988,4034,-30}, {1988,4050,-30}, {2302,1630,-20}, 
    {2302,1631,-20}, {2302,1645,-20}, {2302,1655,-20}, {2302,1662,-20}, 
    {2302,2794,-20}, {2302,2795,-20}, {2302,2809,-20}, {2302,2819,-20}, 
    {2302,2826,-20}, {2302,2899,-20}, {2302,2907,-20}, {2511,3816,-10}, 
    {2511,3817,-10}, {2511,3824,-10}, {2511,3831,-10}, {2511,3839,-10}, 
    {2511,4031,-15}, {2511,4034,-15}, {2511,4050,-15}, {2648,3816,-10}, 
    {2648,3817,-10}, {2648,3824,-10}, {2648,3831,-10}, {2648,3839,-10}, 
    {2648,3902,-20}, {2648,4031,-15}, {2648,4034,-15}, {2648,4050,-15}, 
    {2769,3816,-10}, {2769,3817,-10}, {2769,3824,-10}, {2769,3831,-10}, 
    {2769,3839,-10}, {2769,3902,-20}, {2769,4031,-15}, {2769,4034,-15}, 
    {2769,4050,-15}, {2794,1409,-40}, {2794,2995,-40}, {2794,3902,-15}, 
    {2794,3951,-15}, {2794,4020,-30}, {2794,4031,-30}, {2794,4034,-30}, 
    {2794,4050,-30}, {2795,1409,-40}, {2795,2995,-40}, {2795,3902,-15}, 
    {2795,3951,-15}, {2795,4020,-30}, {2795,4031,-30}, {2795,4034,-30}, 
    {2795,4050,-30}, {2809,1409,-40}, {2809,2995,-40}, {2809,3902,-15}, 
    {2809,3951,-15}, {2809,4020,-30}, {2809,4031,-30}, {2809,4034,-30}, 
    {2809,4050,-30}, {2819,1409,-40}, {2819,2995,-40}, {2819,3902,-15}, 
    {2819,3951,-15}, {2819,4020,-30}, {2819,4031,-30}, {2819,4034,-30}, 
    {2819,4050,-30}, {2826,1409,-40}, {2826,2995,-40}, {2826,3902,-15}, 
    {2826,3951,-15}, {2826,4020,-30}, {2826,4031,-30}, {2826,4034,-30}, 
    {2826,4050,-30}, {2899,734,-55}, {2899,736,-55}, {2899,758,-55}, 
    {2899,775,-55}, {2899,1030,-55}, {2899,1111,-55}, {2899,1173,-55}, 
    {2899,1192,-55}, {2899,1312,-55}, {2899,1332,-55}, {2899,1409,-95}, 
    {2899,1481,-55}, {2899,1630,-55}, {2899,1631,-55}, {2899,1645,-55}, 
    {2899,1655,-55}, {2899,1662,-55}, {2899,1787,-55}, {2899,1886,-55}, 
    {2899,1988,-55}, {2899,2132,-55}, {2899,2133,-55}, {2899,2140,-55}, 
    {2899,2196,-55}, {2899,2207,-55}, {2899,2274,-55}, {2899,2302,-55}, 
    {2899,2416,-55}, {2899,2501,-55}, {2899,2511,-55}, {2899,2648,-55}, 
    {2899,2769,-55}, {2899,2794,-55}, {2899,2795,-55}, {2899,2809,-55}, 
    {2899,2819,-55}, {2899,2826,-55}, {2899,2899,-55}, {2899,2907,-55}, 
    {2899,2921,-55}, {2899,2995,-95}, {2899,3083,-55}, {2899,3148,-55}, 
    {2899,3279,-55}, {2899,3317,-55}, {2899,3556,-55}, {2899,3816,-55}, 
    {2899,3817,-55}, {2899,3824,-55}, {2899,3831,-55}, {2899,3839,-55}, 
    {2899,3902,-70}, {2899,3951,-70}, {2899,4020,-85}, {2899,4031,-70}, 
    {2899,4034,-70}, {2899,4050,-70}, {2899,4132,-55}, {2899,4153,-55}, 
    {2907,1409,-40}, {2907,2995,-40}, {2907,3902,-15}, {2907,3951,-15}, 
    {2907,4020,-30}, {2907,4031,-30}, {2907,4034,-30}, {2907,4050,-30}, 
    {2921,1409,-35}, {2921,2995,-35}, {2921,4031,-30}, {2921,4034,-30}, 
    {2921,4050,-30}, {2995,3139,-100}, {2995,3143,-100}, 
    {2995,3498,-60}, {3139,3498,-40}, {3140,3140,-57}, {3143,1481,-50}, 
    {3143,3143,-57}, {3143,3148,-50}, {3143,3279,-50}, {3143,3317,-50}, 
    {3143,3498,-70}, {3148,734,-10}, {3148,736,-10}, {3148,758,-10}, 
    {3148,775,-10}, {3148,1030,-10}, {3148,1111,-10}, {3148,1173,-10}, 
    {3148,1402,30}, {3148,1409,-50}, {3148,2132,15}, {3148,2133,15}, 
    {3148,2140,15}, {3148,2196,15}, {3148,2207,15}, {3148,2302,15}, 
    {3148,2416,15}, {3148,2501,15}, {3148,2511,25}, {3148,2648,25}, 
    {3148,2769,25}, {3148,2921,30}, {3148,2995,-50}, {3148,3351,30}, 
    {3148,3556,40}, {3148,3816,15}, {3148,3817,15}, {3148,3824,15}, 
    {3148,3831,15}, {3148,3839,15}, {3148,3902,30}, {3148,4031,30}, 
    {3148,4034,30}, {3148,4050,30}, {3279,1409,-15}, {3279,2995,-15}, 
    {3279,3951,-30}, {3317,1409,-15}, {3317,2995,-15}, {3317,3951,-30}, 
    {3351,3498,-50}, {3498,570,-50}, {3498,658,-50}, {3498,668,-40}, 
    {3498,686,-90}, {3498,687,-90}, {3498,692,-90}, {3498,3135,-30}, 
    {3498,3140,-60}, {3902,734,-25}, {3902,736,-25}, {3902,758,-25}, 
    {3902,775,-25}, {3902,1030,-25}, {3902,1111,-25}, {3902,1173,-25}, 
    {3902,1409,-80}, {3902,1630,-25}, {3902,1631,-25}, {3902,1645,-25}, 
    {3902,1655,-25}, {3902,1662,-25}, {3902,2794,-25}, {3902,2795,-25}, 
    {3902,2809,-25}, {3902,2819,-25}, {3902,2826,-25}, {3902,2899,-25}, 
    {3902,2907,-25}, {3902,2995,-80}, {3951,734,-15}, {3951,736,-15}, 
    {3951,758,-15}, {3951,775,-15}, {3951,1030,-15}, {3951,1111,-15}, 
    {3951,1173,-15}, {3951,1409,-60}, {3951,1630,-10}, {3951,1631,-10}, 
    {3951,1645,-10}, {3951,1655,-10}, {3951,1662,-10}, {3951,2794,-10}, 
    {3951,2795,-10}, {3951,2809,-10}, {3951,2819,-10}, {3951,2826,-10}, 
    {3951,2899,-10}, {3951,2907,-10}, {3951,2995,-60}, {4020,1630,-30}, 
    {4020,1631,-30}, {4020,1645,-30}, {4020,1655,-30}, {4020,1662,-30}, 
    {4031,734,-20}, {4031,736,-20}, {4031,758,-20}, {4031,775,-20}, 
    {4031,1030,-20}, {4031,1111,-20}, {4031,1173,-20}, 
    {4031,1409,-100}, {4031,1630,-20}, {4031,1631,-20}, 
    {4031,1645,-20}, {4031,1655,-20}, {4031,1662,-20}, {4031,2794,-20}, 
    {4031,2795,-20}, {4031,2809,-20}, {4031,2819,-20}, {4031,2826,-20}, 
    {4031,2899,-20}, {4031,2907,-20}, {4031,2995,-100}, {4034,734,-20}, 
    {4034,736,-20}, {4034,758,-20}, {4034,775,-20}, {4034,1030,-20}, 
    {4034,1111,-20}, {4034,1173,-20}, {4034,1409,-100}, 
    {4034,1630,-20}, {4034,1631,-20}, {4034,1645,-20}, {4034,1655,-20}, 
    {4034,1662,-20}, {4034,2794,-20}, {4034,2795,-20}, {4034,2809,-20}, 
    {4034,2819,-20}, {4034,2826,-20}, {4034,2899,-20}, {4034,2907,-20}, 
    {4034,2995,-100}, {4050,734,-20}, {4050,736,-20}, {4050,758,-20}, 
    {4050,775,-20}, {4050,1030,-20}, {4050,1111,-20}, {4050,1173,-20}, 
    {4050,1409,-100}, {4050,1630,-20}, {4050,1631,-20}, 
    {4050,1645,-20}, {4050,1655,-20}, {4050,1662,-20}, {4050,2794,-20}, 
    {4050,2795,-20}, {4050,2809,-20}, {4050,2819,-20}, {4050,2826,-20}, 
    {4050,2899,-20}, {4050,2907,-20}, {4050,2995,-100}, 
    {4132,1630,-15}, {4132,1631,-15}, {4132,1645,-15}, {4132,1655,-15}, 
    {4132,1662,-15}, {4132,2794,-15}, {4132,2795,-15}, {4132,2809,-15}, 
    {4132,2819,-15}, {4132,2826,-15}, {4132,2899,-15}, {4132,2907,-15}, 
    {4153,1630,-15}, {4153,1631,-15}, {4153,1645,-15}, {4153,1655,-15}, 
    {4153,1662,-15}, {4153,2794,-15}, {4153,2795,-15}, {4153,2809,-15}, 
    {4153,2819,-15}, {4153,2826,-15}, {4153,2899,-15}, {4153,2907,-15}, 
    {NOGLYPH,NOGLYPH,0}
};
static const ligature helvetica_ligs[] = {
    {1787,2132,1807}, {1787,2416,1850}, 
    {NOGLYPH,NOGLYPH,NOGLYPH}
};
static const kern_pair helvetica_oblique_kerns[] = {
    {0,64,-30}, {0,70,-30}, {0,203,-30}, {0,393,-30}, {0,396,-30}, 
    {0,404,-30}, {0,414,-30}, {0,419,-30}, {0,449,-30}, {0,455,-30}, 
    {0,474,-30}, {0,570,-120}, {0,603,-50}, {0,604,-50}, {0,609,-50}, 
    {0,615,-50}, {0,624,-50}, {0,658,-70}, {0,668,-50}, {0,686,-100}, 
    {0,687,-100}, {0,692,-100}, {0,3816,-30}, {0,3817,-30}, 
    {0,3824,-30}, {0,3831,-30}, {0,3839,-30}, {0,3902,-40}, 
    {0,3951,-40}, {0,4031,-40}, {0,4034,-40}, {0,4050,-40}, {5,64,-30}, 
    {5,70,-30}, {5,203,-30}, {5,393,-30}, {5,396,-30}, {5,404,-30}, 
    {5,414,-30}, {5,419,-30}, {5,449,-30}, {5,455,-30}, {5,474,-30}, 
    {5,570,-120}, {5,603,-50}, {5,604,-50}, {5,609,-50}, {5,615,-50}, 
    {5,624,-50}, {5,658,-70}, {5,668,-50}, {5,686,-100}, {5,687,-100}, 
    {5,692,-100}, {5,3816,-30}, {5,3817,-30}, {5,3824,-30}, 
    {5,3831,-30}, {5,3839,-30}, {5,3902,-40}, {5,3951,-40}, 
    {5,4031,-40}, {5,4034,-40}, {5,4050,-40}, {16,64,-30}, {16,70,-30}, 
    {16,203,-30}, {16,393,-30}, {16,396,-30}, {16,404,-30}, 
    {16,414,-30}, {16,419,-30}, {16,449,-30}, {16,455,-30}, 
    {16,474,-30}, {16,570,-120}, {16,603,-50}, {16,604,-50}, 
    {16,609,-50}, {16,615,-50}, {16,624,-50}, {16,658,-70}, 
    {16,668,-50}, {16,686,-100}, {16,687,-100}, {16,692,-100}, 
    {16,3816,-30}, {16,3817,-30}, {16,3824,-30}, {16,3831,-30}, 
    {16,3839,-30}, {16,3902,-40}, {16,3951,-40}, {16,4031,-40}, 
    {16,4034,-40}, {16,4050,-40}, {27,64,-30}, {27,70,-30}, 
    {27,203,-30}, {27,393,-30}, {27,396,-30}, {27,404,-30}, 
    {27,414,-30}, {27,419,-30}, {27,449,-30}, {27,455,-30}, 
    {27,474,-30}, {27,570,-120}, {27,603,-50}, {27,604,-50}, 
    {27,609,-50}, {27,615,-50}, {27,624,-50}, {27,658,-70}, 
    {27,668,-50}, {27,686,-100}, {27,687,-100}, {27,692,-100}, 
    {27,3816,-30}, {27,3817,-30}, {27,3824,-30}, {27,3831,-30}, 
    {27,3839,-30}, {27,3902,-40}, {27,3951,-40}, {27,4031,-40}, 
    {27,4034,-40}, {27,4050,-40}, {33,64,-30}, {33,70,-30}, 
    {33,203,-30}, {33,393,-30}, {33,396,-30}, {33,404,-30}, 
    {33,414,-30}, {33,419,-30}, {33,449,-30}, {33,455,-30}, 
    {33,474,-30}, {33,570,-120}, {33,603,-50}, {33,604,-50}, 
    {33,609,-50}, {33,615,-50}, {33,624,-50}, {33,658,-70}, 
    {33,668,-50}, {33,686,-100}, {33,687,-100}, {33,692,-100}, 
    {33,3816,-30}, {33,3817,-30}, {33,3824,-30}, {33,3831,-30}, 
    {33,3839,-30}, {33,3902,-40}, {33,3951,-40}, {33,4031,-40}, 
    {33,4034,-40}, {33,4050,-40}, {43,64,-30}, {43,70,-30}, 
    {43,203,-30}, {43,393,-30}, {43,396,-30}, {43,404,-30}, 
    {43,414,-30}, {43,419,-30}, {43,449,-30}, {43,455,-30}, 
    {43,474,-30}, {43,570,-120}, {43,603,-50}, {43,604,-50}, 
    {43,609,-50}, {43,615,-50}, {43,624,-50}, {43,658,-70}, 
    {43,668,-50}, {43,686,-100}, {43,687,-100}, {43,692,-100}, 
    {43,3816,-30}, {43,3817,-30}, {43,3824,-30}, {43,3831,-30}, 
    {43,3839,-30}, {43,3902,-40}, {43,3951,-40}, {43,4031,-40}, 
    {43,4034,-40}, {43,4050,-40}, {48,64,-30}, {48,70,-30}, 
    {48,203,-30}, {48,393,-30}, {48,396,-30}, {48,404,-30}, 
    {48,414,-30}, {48,419,-30}, {48,449,-30}, {48,455,-30}, 
    {48,474,-30}, {48,570,-120}, {48,603,-50}, {48,604,-50}, 
    {48,609,-50}, {48,615,-50}, {48,624,-50}, {48,658,-70}, 
    {48,668,-50}, {48,686,-100}, {48,687,-100}, {48,692,-100}, 
    {48,3816,-30}, {48,3817,-30}, {48,3824,-30}, {48,3831,-30}, 
    {48,3839,-30}, {48,3902,-40}, {48,3951,-40}, {48,4031,-40}, 
    {48,4034,-40}, {48,4050,-40}, {51,603,-10}, {51,604,-10}, 
    {51,609,-10}, {51,615,-10}, {51,624,-10}, {51,1409,-20}, 
    {51,2995,-20}, {64,1409,-30}, {64,2995,-30}, {70,1409,-30}, 
    {70,2995,-30}, {93,0,-40}, {93,5,-40}, {93,16,-40}, {93,27,-40}, 
    {93,33,-40}, {93,43,-40}, {93,48,-40}, {93,658,-70}, {93,668,-40}, 
    {93,686,-90}, {93,687,-90}, {93,692,-90}, {93,1409,-70}, 
    {93,2995,-70}, {192,0,-80}, {192,5,-80}, {192,16,-80}, 
    {192,27,-80}, {192,33,-80}, {192,43,-80}, {192,48,-80}, 
    {192,734,-50}, {192,736,-50}, {192,758,-50}, {192,775,-50}, 
    {192,1030,-50}, {192,1111,-50}, {192,1173,-50}, {192,1409,-150}, 
    {192,1630,-30}, {192,1631,-30}, {192,1645,-30}, {192,1655,-30}, 
    {192,1662,-30}, {192,2794,-30}, {192,2795,-30}, {192,2809,-30}, 
    {192,2819,-30}, {192,2826,-30}, {192,2899,-30}, {192,2907,-30}, 
    {192,2995,-150}, {192,3148,-45}, {301,0,-20}, {301,5,-20}, 
    {301,16,-20}, {301,27,-20}, {301,33,-20}, {301,43,-20}, 
    {301,48,-20}, {301,734,-20}, {301,736,-20}, {301,758,-20}, 
    {301,775,-20}, {301,1030,-20}, {301,1111,-20}, {301,1173,-20}, 
    {301,1409,-30}, {301,2995,-30}, {301,3816,-20}, {301,3817,-20}, 
    {301,3824,-20}, {301,3831,-20}, {301,3839,-20}, {309,393,-50}, 
    {309,396,-50}, {309,404,-50}, {309,414,-50}, {309,419,-50}, 
    {309,449,-50}, {309,455,-50}, {309,1630,-40}, {309,1631,-40}, 
    {309,1645,-40}, {309,1655,-40}, {309,1662,-40}, {309,2794,-40}, 
    {309,2795,-40}, {309,2809,-40}, {309,2819,-40}, {309,2826,-40}, 
    {309,2899,-40}, {309,2907,-40}, {309,3816,-30}, {309,3817,-30}, 
    {309,3824,-30}, {309,3831,-30}, {309,3839,-30}, {309,4031,-50}, 
    {309,4034,-50}, {309,4050,-50}, {337,570,-110}, {337,658,-110}, 
    {337,668,-70}, {337,686,-140}, {337,687,-140}, {337,692,-140}, 
    {337,3139,-140}, {337,3143,-160}, {337,4031,-30}, {337,4034,-30}, 
    {337,4050,-30}, {356,570,-110}, {356,658,-110}, {356,668,-70}, 
    {356,686,-140}, {356,687,-140}, {356,692,-140}, {356,3139,-140}, 
    {356,3143,-160}, {356,4031,-30}, {356,4034,-30}, {356,4050,-30}, 
    {393,0,-20}, {393,5,-20}, {393,16,-20}, {393,27,-20}, {393,33,-20}, 
    {393,43,-20}, {393,48,-20}, {393,570,-40}, {393,658,-50}, 
    {393,668,-30}, {393,678,-60}, {393,686,-70}, {393,687,-70}, 
    {393,692,-70}, {393,1409,-40}, {393,2995,-40}, {396,0,-20}, 
    {396,5,-20}, {396,16,-20}, {396,27,-20}, {396,33,-20}, 
    {396,43,-20}, {396,48,-20}, {396,570,-40}, {396,658,-50}, 
    {396,668,-30}, {396,678,-60}, {396,686,-70}, {396,687,-70}, 
    {396,692,-70}, {396,1409,-40}, {396,2995,-40}, {404,0,-20}, 
    {404,5,-20}, {404,16,-20}, {404,27,-20}, {404,33,-20}, 
    {404,43,-20}, {404,48,-20}, {404,570,-40}, {404,658,-50}, 
    {404,668,-30}, {404,678,-60}, {404,686,-70}, {404,687,-70}, 
    {404,692,-70}, {404,1409,-40}, {404,2995,-40}, {414,0,-20}, 
    {414,5,-20}, {414,16,-20}, {414,27,-20}, {414,33,-20}, 
    {414,43,-20}, {414,48,-20}, {414,570,-40}, {414,658,-50}, 
    {414,668,-30}, {414,678,-60}, {414,686,-70}, {414,687,-70}, 
    {414,692,-70}, {414,1409,-40}, {414,2995,-40}, {419,0,-20}, 
    {419,5,-20}, {419,16,-20}, {419,27,-20}, {419,33,-20}, 
    {419,43,-20}, {419,48,-20}, {419,570,-40}, {419,658,-50}, 
    {419,668,-30}, {419,678,-60}, {419,686,-70}, {419,687,-70}, 
    {419,692,-70}, {419,1409,-40}, {419,2995,-40}, {449,0,-20}, 
    {449,5,-20}, {449,16,-20}, {449,27,-20}, {449,33,-20}, 
    {449,43,-20}, {449,48,-20}, {449,570,-40}, {449,658,-50}, 
    {449,668,-30}, {449,678,-60}, {449,686,-70}, {449,687,-70}, 
    {449,692,-70}, {449,1409,-40}, {449,2995,-40}, {455,0,-20}, 
    {455,5,-20}, {455,16,-20}, {455,27,-20}, {455,33,-20}, 
    {455,43,-20}, {455,48,-20}, {455,570,-40}, {455,658,-50}, 
    {455,668,-30}, {455,678,-60}, {455,686,-70}, {455,687,-70}, 
    {455,692,-70}, {455,1409,-40}, {455,2995,-40}, {459,0,-120}, 
    {459,5,-120}, {459,16,-120}, {459,27,-120}, {459,33,-120}, 
    {459,43,-120}, {459,48,-120}, {459,734,-40}, {459,736,-40}, 
    {459,758,-40}, {459,775,-40}, {459,1030,-40}, {459,1111,-40}, 
    {459,1173,-40}, {459,1409,-180}, {459,1630,-50}, {459,1631,-50}, 
    {459,1645,-50}, {459,1655,-50}, {459,1662,-50}, {459,2794,-50}, 
    {459,2795,-50}, {459,2809,-50}, {459,2819,-50}, {459,2826,-50}, 
    {459,2899,-50}, {459,2907,-50}, {459,2995,-180}, {474,603,-10}, 
    {474,604,-10}, {474,609,-10}, {474,615,-10}, {474,624,-10}, 
    {478,393,-20}, {478,396,-20}, {478,404,-20}, {478,414,-20}, 
    {478,419,-20}, {478,449,-20}, {478,455,-20}, {478,570,-30}, 
    {478,603,-40}, {478,604,-40}, {478,609,-40}, {478,615,-40}, 
    {478,624,-40}, {478,658,-50}, {478,668,-30}, {478,686,-50}, 
    {478,687,-50}, {478,692,-50}, {499,1409,-20}, {499,2995,-20}, 
    {543,1409,-20}, {543,2995,-20}, {570,0,-120}, {570,5,-120}, 
    {570,16,-120}, {570,27,-120}, {570,33,-120}, {570,43,-120}, 
    {570,48,-120}, {570,393,-40}, {570,396,-40}, {570,404,-40}, 
    {570,414,-40}, {570,419,-40}, {570,449,-40}, {570,455,-40}, 
    {570,734,-120}, {570,736,-120}, {570,758,-120}, {570,775,-120}, 
    {570,1030,-120}, {570,1111,-120}, {570,1173,-60}, {570,1402,-20}, 
    {570,1409,-120}, {570,1630,-120}, {570,1631,-120}, {570,1645,-120}, 
    {570,1655,-120}, {570,1662,-60}, {570,2126,-140}, {570,2794,-120}, 
    {570,2795,-120}, {570,2809,-120}, {570,2819,-120}, {570,2826,-120}, 
    {570,2899,-120}, {570,2907,-60}, {570,2995,-120}, {570,3148,-120}, 
    {570,3351,-20}, {570,3816,-120}, {570,3817,-120}, {570,3824,-120}, 
    {570,3831,-120}, {570,3839,-120}, {570,3951,-120}, {570,4031,-120}, 
    {570,4034,-120}, {570,4050,-60}, {603,0,-40}, {603,5,-40}, 
    {603,16,-40}, {603,27,-40}, {603,33,-40}, {603,43,-40}, 
    {603,48,-40}, {603,1409,-40}, {603,2995,-40}, {604,0,-40}, 
    {604,5,-40}, {604,16,-40}, {604,27,-40}, {604,33,-40}, 
    {604,43,-40}, {604,48,-40}, {604,1409,-40}, {604,2995,-40}, 
    {609,0,-40}, {609,5,-40}, {609,16,-40}, {609,27,-40}, {609,33,-40}, 
    {609,43,-40}, {609,48,-40}, {609,1409,-40}, {609,2995,-40}, 
    {615,0,-40}, {615,5,-40}, {615,16,-40}, {615,27,-40}, {615,33,-40}, 
    {615,43,-40}, {615,48,-40}, {615,1409,-40}, {615,2995,-40}, 
    {624,0,-40}, {624,5,-40}, {624,16,-40}, {624,27,-40}, {624,33,-40}, 
    {624,43,-40}, {624,48,-40}, {624,1409,-40}, {624,2995,-40}, 
    {658,0,-80}, {658,5,-80}, {658,16,-80}, {658,27,-80}, {658,33,-80}, 
    {658,43,-80}, {658,48,-80}, {658,203,-40}, {658,393,-40}, 
    {658,396,-40}, {658,404,-40}, {658,414,-40}, {658,419,-40}, 
    {658,449,-40}, {658,455,-40}, {658,734,-70}, {658,736,-70}, 
    {658,758,-70}, {658,775,-70}, {658,1030,-70}, {658,1111,-70}, 
    {658,1173,-70}, {658,1402,-40}, {658,1409,-125}, {658,1630,-80}, 
    {658,1631,-80}, {658,1645,-80}, {658,1655,-80}, {658,1662,-80}, 
    {658,2126,-80}, {658,2794,-80}, {658,2795,-80}, {658,2809,-80}, 
    {658,2819,-80}, {658,2826,-80}, {658,2899,-80}, {658,2907,-80}, 
    {658,2995,-125}, {658,3351,-40}, {658,3816,-70}, {658,3817,-70}, 
    {658,3824,-70}, {658,3831,-70}, {658,3839,-70}, {668,0,-50}, 
    {668,5,-50}, {668,16,-50}, {668,27,-50}, {668,33,-50}, 
    {668,43,-50}, {668,48,-50}, {668,393,-20}, {668,396,-20}, 
    {668,404,-20}, {668,414,-20}, {668,419,-20}, {668,449,-20}, 
    {668,455,-20}, {668,734,-40}, {668,736,-40}, {668,758,-40}, 
    {668,775,-40}, {668,1030,-40}, {668,1111,-40}, {668,1173,-40}, 
    {668,1409,-80}, {668,1630,-30}, {668,1631,-30}, {668,1645,-30}, 
    {668,1655,-30}, {668,1662,-30}, {668,2126,-40}, {668,2794,-30}, 
    {668,2795,-30}, {668,2809,-30}, {668,2819,-30}, {668,2826,-30}, 
    {668,2899,-30}, {668,2907,-30}, {668,2995,-80}, {668,3816,-30}, 
    {668,3817,-30}, {668,3824,-30}, {668,3831,-30}, {668,3839,-30}, 
    {668,4031,-20}, {668,4034,-20}, {668,4050,-20}, {686,0,-110}, 
    {686,5,-110}, {686,16,-110}, {686,27,-110}, {686,33,-110}, 
    {686,43,-110}, {686,48,-110}, {686,393,-85}, {686,396,-85}, 
    {686,404,-85}, {686,414,-85}, {686,419,-85}, {686,449,-85}, 
    {686,455,-85}, {686,734,-140}, {686,736,-140}, {686,758,-140}, 
    {686,775,-140}, {686,1030,-140}, {686,1111,-140}, {686,1173,-140}, 
    {686,1402,-60}, {686,1409,-140}, {686,1630,-140}, {686,1631,-140}, 
    {686,1645,-140}, {686,1655,-140}, {686,1662,-140}, {686,2126,-140}, 
    {686,2132,-20}, {686,2133,-20}, {686,2794,-140}, {686,2795,-140}, 
    {686,2809,-140}, {686,2819,-140}, {686,2826,-140}, {686,2899,-140}, 
    {686,2907,-140}, {686,2995,-140}, {686,3351,-60}, {686,3816,-110}, 
    {686,3817,-110}, {686,3824,-110}, {686,3831,-110}, {686,3839,-110}, 
    {687,0,-110}, {687,5,-110}, {687,16,-110}, {687,27,-110}, 
    {687,33,-110}, {687,43,-110}, {687,48,-110}, {687,393,-85}, 
    {687,396,-85}, {687,404,-85}, {687,414,-85}, {687,419,-85}, 
    {687,449,-85}, {687,455,-85}, {687,734,-140}, {687,736,-140}, 
    {687,758,-140}, {687,775,-140}, {687,1030,-140}, {687,1111,-140}, 
    {687,1173,-70}, {687,1402,-60}, {687,1409,-140}, {687,1630,-140}, 
    {687,1631,-140}, {687,1645,-140}, {687,1655,-140}, {687,1662,-140}, 
    {687,2126,-140}, {687,2132,-20}, {687,2133,-20}, {687,2794,-140}, 
    {687,2795,-140}, {687,2809,-140}, {687,2819,-140}, {687,2826,-140}, 
    {687,2899,-140}, {687,2907,-140}, {687,2995,-140}, {687,3351,-60}, 
    {687,3816,-110}, {687,3817,-110}, {687,3824,-110}, {687,3831,-110}, 
    {687,3839,-110}, {692,0,-110}, {692,5,-110}, {692,16,-110}, 
    {692,27,-110}, {692,33,-110}, {692,43,-110}, {692,48,-110}, 
    {692,393,-85}, {692,396,-85}, {692,404,-85}, {692,414,-85}, 
    {692,419,-85}, {692,449,-85}, {692,455,-85}, {692,734,-140}, 
    {692,736,-140}, {692,758,-140}, {692,775,-140}, {692,1030,-140}, 
    {692,1111,-140}, {692,1173,-70}, {692,1402,-60}, {692,1409,-140}, 
    {692,1630,-140}, {692,1631,-140}, {692,1645,-140}, {692,1655,-140}, 
    {692,1662,-140}, {692,2126,-140}, {692,2132,-20}, {692,2133,-20}, 
    {692,2794,-140}, {692,2795,-140}, {692,2809,-140}, {692,2819,-140}, 
    {692,2826,-140}, {692,2899,-140}, {692,2907,-140}, {692,2995,-140}, 
    {692,3351,-60}, {692,3816,-110}, {692,3817,-110}, {692,3824,-110}, 
    {692,3831,-110}, {692,3839,-110}, {734,3902,-20}, {734,3951,-20}, 
    {734,4031,-30}, {734,4034,-30}, {734,4050,-30}, {736,3902,-20}, 
    {736,3951,-20}, {736,4031,-30}, {736,4034,-30}, {736,4050,-30}, 
    {758,3902,-20}, {758,3951,-20}, {758,4031,-30}, {758,4034,-30}, 
    {758,4050,-30}, {775,3902,-20}, {775,3951,-20}, {775,4031,-30}, 
    {775,4034,-30}, {775,4050,-30}, {1030,3902,-20}, {1030,3951,-20}, 
    {1030,4031,-30}, {1030,4034,-30}, {1030,4050,-30}, {1111,3902,-20}, 
    {1111,3951,-20}, {1111,4031,-30}, {1111,4034,-30}, {1111,4050,-30}, 
    {1173,3902,-20}, {1173,3951,-20}, {1173,4031,-30}, {1173,4034,-30}, 
    {1173,4050,-30}, {1192,1192,-10}, {1192,1409,-40}, {1192,2416,-20}, 
    {1192,2501,-20}, {1192,2995,-40}, {1192,3816,-20}, {1192,3817,-20}, 
    {1192,3824,-20}, {1192,3831,-20}, {1192,3839,-20}, {1192,3902,-20}, 
    {1192,4031,-20}, {1192,4034,-20}, {1192,4050,-20}, {1312,1409,-15}, 
    {1312,2302,-20}, {1332,1409,-15}, {1332,2302,-20}, {1402,3498,-50}, 
    {1409,3139,-100}, {1409,3143,-100}, {1630,1409,-15}, 
    {1630,2995,-15}, {1630,3902,-30}, {1630,3951,-20}, {1630,4020,-30}, 
    {1630,4031,-20}, {1630,4034,-20}, {1630,4050,-20}, {1631,1409,-15}, 
    {1631,2995,-15}, {1631,3902,-30}, {1631,3951,-20}, {1631,4020,-30}, 
    {1631,4031,-20}, {1631,4034,-20}, {1631,4050,-20}, {1645,1409,-15}, 
    {1645,2995,-15}, {1645,3902,-30}, {1645,3951,-20}, {1645,4020,-30}, 
    {1645,4031,-20}, {1645,4034,-20}, {1645,4050,-20}, {1655,1409,-15}, 
    {1655,2995,-15}, {1655,3902,-30}, {1655,3951,-20}, {1655,4020,-30}, 
    {1655,4031,-20}, {1655,4034,-20}, {1655,4050,-20}, {1662,1409,-15}, 
    {1662,2995,-15}, {1662,3902,-30}, {1662,3951,-20}, {1662,4020,-30}, 
    {1662,4031,-20}, {1662,4034,-20}, {1662,4050,-20}, {1787,734,-30}, 
    {1787,736,-30}, {1787,758,-30}, {1787,775,-30}, {1787,1030,-30}, 
    {1787,1111,-30}, {1787,1173,-30}, {1787,1409,-30}, {1787,1608,-28}, 
    {1787,1630,-30}, {1787,1631,-30}, {1787,1645,-30}, {1787,1655,-30}, 
    {1787,1662,-30}, {1787,2794,-30}, {1787,2795,-30}, {1787,2809,-30}, 
    {1787,2819,-30}, {1787,2826,-30}, {1787,2899,-30}, {1787,2907,-30}, 
    {1787,2995,-30}, {1787,3139,60}, {1787,3143,50}, {1886,3148,-10}, 
    {1988,4031,-30}, {1988,4034,-30}, {1988,4050,-30}, {2302,1630,-20}, 
    {2302,1631,-20}, {2302,1645,-20}, {2302,1655,-20}, {2302,1662,-20}, 
    {2302,2794,-20}, {2302,2795,-20}, {2302,2809,-20}, {2302,2819,-20}, 
    {2302,2826,-20}, {2302,2899,-20}, {2302,2907,-20}, {2511,3816,-10}, 
    {2511,3817,-10}, {2511,3824,-10}, {2511,3831,-10}, {2511,3839,-10}, 
    {2511,4031,-15}, {2511,4034,-15}, {2511,4050,-15}, {2648,3816,-10}, 
    {2648,3817,-10}, {2648,3824,-10}, {2648,3831,-10}, {2648,3839,-10}, 
    {2648,3902,-20}, {2648,4031,-15}, {2648,4034,-15}, {2648,4050,-15}, 
    {2769,3816,-10}, {2769,3817,-10}, {2769,3824,-10}, {2769,3831,-10}, 
    {2769,3839,-10}, {2769,3902,-20}, {2769,4031,-15}, {2769,4034,-15}, 
    {2769,4050,-15}, {2794,1409,-40}, {2794,2995,-40}, {2794,3902,-15}, 
    {2794,3951,-15}, {2794,4020,-30}, {2794,4031,-30}, {2794,4034,-30}, 
    {2794,4050,-30}, {2795,1409,-40}, {2795,2995,-40}, {2795,3902,-15}, 
    {2795,3951,-15}, {2795,4020,-30}, {2795,4031,-30}, {2795,4034,-30}, 
    {2795,4050,-30}, {2809,1409,-40}, {2809,2995,-40}, {2809,3902,-15}, 
    {2809,3951,-15}, {2809,4020,-30}, {2809,4031,-30}, {2809,4034,-30}, 
    {2809,4050,-30}, {2819,1409,-40}, {2819,2995,-40}, {2819,3902,-15}, 
    {2819,3951,-15}, {2819,4020,-30}, {2819,4031,-30}, {2819,4034,-30}, 
    {2819,4050,-30}, {2826,1409,-40}, {2826,2995,-40}, {2826,3902,-15}, 
    {2826,3951,-15}, {2826,4020,-30}, {2826,4031,-30}, {2826,4034,-30}, 
    {2826,4050,-30}, {2899,734,-55}, {2899,736,-55}, {2899,758,-55}, 
    {2899,775,-55}, {2899,1030,-55}, {2899,1111,-55}, {2899,1173,-55}, 
    {2899,1192,-55}, {2899,1312,-55}, {2899,1332,-55}, {2899,1409,-95}, 
    {2899,1481,-55}, {2899,1630,-55}, {2899,1631,-55}, {2899,1645,-55}, 
    {2899,1655,-55}, {2899,1662,-55}, {2899,1787,-55}, {2899,1886,-55}, 
    {2899,1988,-55}, {2899,2132,-55}, {2899,2133,-55}, {2899,2140,-55}, 
    {2899,2196,-55}, {2899,2207,-55}, {2899,2274,-55}, {2899,2302,-55}, 
    {2899,2416,-55}, {2899,2501,-55}, {2899,2511,-55}, {2899,2648,-55}, 
    {2899,2769,-55}, {2899,2794,-55}, {2899,2795,-55}, {2899,2809,-55}, 
    {2899,2819,-55}, {2899,2826,-55}, {2899,2899,-55}, {2899,2907,-55}, 
    {2899,2921,-55}, {2899,2995,-95}, {2899,3083,-55}, {2899,3148,-55}, 
    {2899,3279,-55}, {2899,3317,-55}, {2899,3556,-55}, {2899,3816,-55}, 
    {2899,3817,-55}, {2899,3824,-55}, {2899,3831,-55}, {2899,3839,-55}, 
    {2899,3902,-70}, {2899,3951,-70}, {2899,4020,-85}, {2899,4031,-70}, 
    {2899,4034,-70}, {2899,4050,-70}, {2899,4132,-55}, {2899,4153,-55}, 
    {2907,1409,-40}, {2907,2995,-40}, {2907,3902,-15}, {2907,3951,-15}, 
    {2907,4020,-30}, {2907,4031,-30}, {2907,4034,-30}, {2907,4050,-30}, 
    {2921,1409,-35}, {2921,2995,-35}, {2921,4031,-30}, {2921,4034,-30}, 
    {2921,4050,-30}, {2995,3139,-100}, {2995,3143,-100}, 
    {2995,3498,-60}, {3139,3498,-40}, {3140,3140,-57}, {3143,1481,-50}, 
    {3143,3143,-57}, {3143,3148,-50}, {3143,3279,-50}, {3143,3317,-50}, 
    {3143,3498,-70}, {3148,734,-10}, {3148,736,-10}, {3148,758,-10}, 
    {3148,775,-10}, {3148,1030,-10}, {3148,1111,-10}, {3148,1173,-10}, 
    {3148,1402,30}, {3148,1409,-50}, {3148,2132,15}, {3148,2133,15}, 
    {3148,2140,15}, {3148,2196,15}, {3148,2207,15}, {3148,2302,15}, 
    {3148,2416,15}, {3148,2501,15}, {3148,2511,25}, {3148,2648,25}, 
    {3148,2769,25}, {3148,2921,30}, {3148,2995,-50}, {3148,3351,30}, 
    {3148,3556,40}, {3148,3816,15}, {3148,3817,15}, {3148,3824,15}, 
    {3148,3831,15}, {3148,3839,15}, {3148,3902,30}, {3148,4031,30}, 
    {3148,4034,30}, {3148,4050,30}, {3279,1409,-15}, {3279,2995,-15}, 
    {3279,3951,-30}, {3317,1409,-15}, {3317,2995,-15}, {3317,3951,-30}, 
    {3351,3498,-50}, {3498,570,-50}, {3498,658,-50}, {3498,668,-40}, 
    {3498,686,-90}, {3498,687,-90}, {3498,692,-90}, {3498,3135,-30}, 
    {3498,3140,-60}, {3902,734,-25}, {3902,736,-25}, {3902,758,-25}, 
    {3902,775,-25}, {3902,1030,-25}, {3902,1111,-25}, {3902,1173,-25}, 
    {3902,1409,-80}, {3902,1630,-25}, {3902,1631,-25}, {3902,1645,-25}, 
    {3902,1655,-25}, {3902,1662,-25}, {3902,2794,-25}, {3902,2795,-25}, 
    {3902,2809,-25}, {3902,2819,-25}, {3902,2826,-25}, {3902,2899,-25}, 
    {3902,2907,-25}, {3902,2995,-80}, {3951,734,-15}, {3951,736,-15}, 
    {3951,758,-15}, {3951,775,-15}, {3951,1030,-15}, {3951,1111,-15}, 
    {3951,1173,-15}, {3951,1409,-60}, {3951,1630,-10}, {3951,1631,-10}, 
    {3951,1645,-10}, {3951,1655,-10}, {3951,1662,-10}, {3951,2794,-10}, 
    {3951,2795,-10}, {3951,2809,-10}, {3951,2819,-10}, {3951,2826,-10}, 
    {3951,2899,-10}, {3951,2907,-10}, {3951,2995,-60}, {4020,1630,-30}, 
    {4020,1631,-30}, {4020,1645,-30}, {4020,1655,-30}, {4020,1662,-30}, 
    {4031,734,-20}, {4031,736,-20}, {4031,758,-20}, {4031,775,-20}, 
    {4031,1030,-20}, {4031,1111,-20}, {4031,1173,-20}, 
    {4031,1409,-100}, {4031,1630,-20}, {4031,1631,-20}, 
    {4031,1645,-20}, {4031,1655,-20}, {4031,1662,-20}, {4031,2794,-20}, 
    {4031,2795,-20}, {4031,2809,-20}, {4031,2819,-20}, {4031,2826,-20}, 
    {4031,2899,-20}, {4031,2907,-20}, {4031,2995,-100}, {4034,734,-20}, 
    {4034,736,-20}, {4034,758,-20}, {4034,775,-20}, {4034,1030,-20}, 
    {4034,1111,-20}, {4034,1173,-20}, {4034,1409,-100}, 
    {4034,1630,-20}, {4034,1631,-20}, {4034,1645,-20}, {4034,1655,-20}, 
    {4034,1662,-20}, {4034,2794,-20}, {4034,2795,-20}, {4034,2809,-20}, 
    {4034,2819,-20}, {4034,2826,-20}, {4034,2899,-20}, {4034,2907,-20}, 
    {4034,2995,-100}, {4050,734,-20}, {4050,736,-20}, {4050,758,-20}, 
    {4050,775,-20}, {4050,1030,-20}, {4050,1111,-20}, {4050,1173,-20}, 
    {4050,1409,-100}, {4050,1630,-20}, {4050,1631,-20}, 
    {4050,1645,-20}, {4050,1655,-20}, {4050,1662,-20}, {4050,2794,-20}, 
    {4050,2795,-20}, {4050,2809,-20}, {4050,2819,-20}, {4050,2826,-20}, 
    {4050,2899,-20}, {4050,2907,-20}, {4050,2995,-100}, 
    {4132,1630,-15}, {4132,1631,-15}, {4132,1645,-15}, {4132,1655,-15}, 
    {4132,1662,-15}, {4132,2794,-15}, {4132,2795,-15}, {4132,2809,-15}, 
    {4132,2819,-15}, {4132,2826,-15}, {4132,2899,-15}, {4132,2907,-15}, 
    {4153,1630,-15}, {4153,1631,-15}, {4153,1645,-15}, {4153,1655,-15}, 
    {4153,1662,-15}, {4153,2794,-15}, {4153,2795,-15}, {4153,2809,-15}, 
    {4153,2819,-15}, {4153,2826,-15}, {4153,2899,-15}, {4153,2907,-15}, 
    {NOGLYPH,NOGLYPH,0}
};
static const ligature helvetica_oblique_ligs[] = {
    {1787,2132,1807}, {1787,2416,1850}, 
    {NOGLYPH,NOGLYPH,NOGLYPH}
};
static const kern_pair helvetica_bold_kerns[] = {
    {0,64,-40}, {0,70,-40}, {0,203,-50}, {0,393,-40}, {0,396,-40}, 
    {0,404,-40}, {0,414,-40}, {0,419,-40}, {0,449,-40}, {0,455,-40}, 
    {0,474,-40}, {0,570,-90}, {0,603,-50}, {0,604,-50}, {0,609,-50}, 
    {0,615,-50}, {0,624,-50}, {0,658,-80}, {0,668,-60}, {0,686,-110}, 
    {0,687,-110}, {0,692,-110}, {0,3816,-30}, {0,3817,-30}, 
    {0,3824,-30}, {0,3831,-30}, {0,3839,-30}, {0,3902,-40}, 
    {0,3951,-30}, {0,4031,-30}, {0,4034,-30}, {0,4050,-30}, {5,64,-40}, 
    {5,70,-40}, {5,203,-50}, {5,393,-40}, {5,396,-40}, {5,404,-40}, 
    {5,414,-40}, {5,419,-40}, {5,449,-40}, {5,455,-40}, {5,474,-40}, 
    {5,570,-90}, {5,603,-50}, {5,604,-50}, {5,609,-50}, {5,615,-50}, 
    {5,624,-50}, {5,658,-80}, {5,668,-60}, {5,686,-110}, {5,687,-110}, 
    {5,692,-110}, {5,3816,-30}, {5,3817,-30}, {5,3824,-30}, 
    {5,3831,-30}, {5,3839,-30}, {5,3902,-40}, {5,3951,-30}, 
    {5,4031,-30}, {5,4034,-30}, {5,4050,-30}, {16,64,-40}, {16,70,-40}, 
    {16,203,-50}, {16,393,-40}, {16,396,-40}, {16,404,-40}, 
    {16,414,-40}, {16,419,-40}, {16,449,-40}, {16,455,-40}, 
    {16,474,-40}, {16,570,-90}, {16,603,-50}, {16,604,-50}, 
    {16,609,-50}, {16,615,-50}, {16,624,-50}, {16,658,-80}, 
    {16,668,-60}, {16,686,-110}, {16,687,-110}, {16,692,-110}, 
    {16,3816,-30}, {16,3817,-30}, {16,3824,-30}, {16,3831,-30}, 
    {16,3839,-30}, {16,3902,-40}, {16,3951,-30}, {16,4031,-30}, 
    {16,4034,-30}, {16,4050,-30}, {27,64,-40}, {27,70,-40}, 
    {27,203,-50}, {27,393,-40}, {27,396,-40}, {27,404,-40}, 
    {27,414,-40}, {27,419,-40}, {27,449,-40}, {27,455,-40}, 
    {27,474,-40}, {27,570,-90}, {27,603,-50}, {27,604,-50}, 
    {27,609,-50}, {27,615,-50}, {27,624,-50}, {27,658,-80}, 
    {27,668,-60}, {27,686,-110}, {27,687,-110}, {27,692,-110}, 
    {27,3816,-30}, {27,3817,-30}, {27,3824,-30}, {27,3831,-30}, 
    {27,3839,-30}, {27,3902,-40}, {27,3951,-30}, {27,4031,-30}, 
    {27,4034,-30}, {27,4050,-30}, {33,64,-40}, {33,70,-40}, 
    {33,203,-50}, {33,393,-40}, {33,396,-40}, {33,404,-40}, 
    {33,414,-40}, {33,419,-40}, {33,449,-40}, {33,455,-40}, 
    {33,474,-40}, {33,570,-90}, {33,603,-50}, {33,604,-50}, 
    {33,609,-50}, {33,615,-50}, {33,624,-50}, {33,658,-80}, 
    {33,668,-60}, {33,686,-110}, {33,687,-110}, {33,692,-110}, 
    {33,3816,-30}, {33,3817,-30}, {33,3824,-30}, {33,3831,-30}, 
    {33,3839,-30}, {33,3902,-40}, {33,3951,-30}, {33,4031,-30}, 
    {33,4034,-30}, {33,4050,-30}, {43,64,-40}, {43,70,-40}, 
    {43,203,-50}, {43,393,-40}, {43,396,-40}, {43,404,-40}, 
    {43,414,-40}, {43,419,-40}, {43,449,-40}, {43,455,-40}, 
    {43,474,-40}, {43,570,-90}, {43,603,-50}, {43,604,-50}, 
    {43,609,-50}, {43,615,-50}, {43,624,-50}, {43,658,-80}, 
    {43,668,-60}, {43,686,-110}, {43,687,-110}, {43,692,-110}, 
    {43,3816,-30}, {43,3817,-30}, {43,3824,-30}, {43,3831,-30}, 
    {43,3839,-30}, {43,3902,-40}, {43,3951,-30}, {43,4031,-30}, 
    {43,4034,-30}, {43,4050,-30}, {48,64,-40}, {48,70,-40}, 
    {48,203,-50}, {48,393,-40}, {48,396,-40}, {48,404,-40}, 
    {48,414,-40}, {48,419,-40}, {48,449,-40}, {48,455,-40}, 
    {48,474,-40}, {48,570,-90}, {48,603,-50}, {48,604,-50}, 
    {48,609,-50}, {48,615,-50}, {48,624,-50}, {48,658,-80}, 
    {48,668,-60}, {48,686,-110}, {48,687,-110}, {48,692,-110}, 
    {48,3816,-30}, {48,3817,-30}, {48,3824,-30}, {48,3831,-30}, 
    {48,3839,-30}, {48,3902,-40}, {48,3951,-30}, {48,4031,-30}, 
    {48,4034,-30}, {48,4050,-30}, {51,0,-30}, {51,5,-30}, {51,16,-30}, 
    {51,27,-30}, {51,33,-30}, {51,43,-30}, {51,48,-30}, {51,603,-10}, 
    {51,604,-10}, {51,609,-10}, {51,615,-10}, {51,624,-10}, {93,0,-40}, 
    {93,5,-40}, {93,16,-40}, {93,27,-40}, {93,33,-40}, {93,43,-40}, 
    {93,48,-40}, {93,658,-40}, {93,668,-40}, {93,686,-70}, 
    {93,687,-70}, {93,692,-70}, {93,1409,-30}, {93,2995,-30}, 
    {192,0,-80}, {192,5,-80}, {192,16,-80}, {192,27,-80}, {192,33,-80}, 
    {192,43,-80}, {192,48,-80}, {192,734,-20}, {192,736,-20}, 
    {192,758,-20}, {192,775,-20}, {192,1030,-20}, {192,1111,-20}, 
    {192,1173,-20}, {192,1409,-100}, {192,2995,-100}, {301,0,-20}, 
    {301,5,-20}, {301,16,-20}, {301,27,-20}, {301,33,-20}, 
    {301,43,-20}, {301,48,-20}, {301,1409,-20}, {301,2995,-20}, 
    {301,3816,-20}, {301,3817,-20}, {301,3824,-20}, {301,3831,-20}, 
    {301,3839,-20}, {309,393,-30}, {309,396,-30}, {309,404,-30}, 
    {309,414,-30}, {309,419,-30}, {309,449,-30}, {309,455,-30}, 
    {309,1630,-15}, {309,1631,-15}, {309,1645,-15}, {309,1655,-15}, 
    {309,1662,-15}, {309,2794,-35}, {309,2795,-35}, {309,2809,-35}, 
    {309,2819,-35}, {309,2826,-35}, {309,2899,-35}, {309,2907,-35}, 
    {309,3816,-30}, {309,3817,-30}, {309,3824,-30}, {309,3831,-30}, 
    {309,3839,-30}, {309,4031,-40}, {309,4034,-40}, {309,4050,-40}, 
    {337,570,-90}, {337,658,-110}, {337,668,-80}, {337,686,-120}, 
    {337,687,-120}, {337,692,-120}, {337,3139,-140}, {337,3143,-140}, 
    {337,4031,-30}, {337,4034,-30}, {337,4050,-30}, {356,570,-90}, 
    {356,658,-110}, {356,668,-80}, {356,686,-120}, {356,687,-120}, 
    {356,692,-120}, {356,3139,-140}, {356,3143,-140}, {356,4031,-30}, 
    {356,4034,-30}, {356,4050,-30}, {393,0,-50}, {393,5,-50}, 
    {393,16,-50}, {393,27,-50}, {393,33,-50}, {393,43,-50}, 
    {393,48,-50}, {393,570,-40}, {393,658,-50}, {393,668,-50}, 
    {393,678,-50}, {393,686,-70}, {393,687,-70}, {393,692,-70}, 
    {393,1409,-40}, {393,2995,-40}, {396,0,-50}, {396,5,-50}, 
    {396,16,-50}, {396,27,-50}, {396,33,-50}, {396,43,-50}, 
    {396,48,-50}, {396,570,-40}, {396,658,-50}, {396,668,-50}, 
    {396,678,-50}, {396,686,-70}, {396,687,-70}, {396,692,-70}, 
    {396,1409,-40}, {396,2995,-40}, {404,0,-50}, {404,5,-50}, 
    {404,16,-50}, {404,27,-50}, {404,33,-50}, {404,43,-50}, 
    {404,48,-50}, {404,570,-40}, {404,658,-50}, {404,668,-50}, 
    {404,678,-50}, {404,686,-70}, {404,687,-70}, {404,692,-70}, 
    {404,1409,-40}, {404,2995,-40}, {414,0,-50}, {414,5,-50}, 
    {414,16,-50}, {414,27,-50}, {414,33,-50}, {414,43,-50}, 
    {414,48,-50}, {414,570,-40}, {414,658,-50}, {414,668,-50}, 
    {414,678,-50}, {414,686,-70}, {414,687,-70}, {414,692,-70}, 
    {414,1409,-40}, {414,2995,-40}, {419,0,-50}, {419,5,-50}, 
    {419,16,-50}, {419,27,-50}, {419,33,-50}, {419,43,-50}, 
    {419,48,-50}, {419,570,-40}, {419,658,-50}, {419,668,-50}, 
    {419,678,-50}, {419,686,-70}, {419,687,-70}, {419,692,-70}, 
    {419,1409,-40}, {419,2995,-40}, {449,0,-50}, {449,5,-50}, 
    {449,16,-50}, {449,27,-50}, {449,33,-50}, {449,43,-50}, 
    {449,48,-50}, {449,570,-40}, {449,658,-50}, {449,668,-50}, 
    {449,678,-50}, {449,686,-70}, {449,687,-70}, {449,692,-70}, 
    {449,1409,-40}, {449,2995,-40}, {455,0,-50}, {455,5,-50}, 
    {455,16,-50}, {455,27,-50}, {455,33,-50}, {455,43,-50}, 
    {455,48,-50}, {455,570,-40}, {455,658,-50}, {455,668,-50}, 
    {455,678,-50}, {455,686,-70}, {455,687,-70}, {455,692,-70}, 
    {455,1409,-40}, {455,2995,-40}, {459,0,-100}, {459,5,-100}, 
    {459,16,-100}, {459,27,-100}, {459,33,-100}, {459,43,-100}, 
    {459,48,-100}, {459,734,-30}, {459,736,-30}, {459,758,-30}, 
    {459,775,-30}, {459,1030,-30}, {459,1111,-30}, {459,1173,-30}, 
    {459,1409,-120}, {459,1630,-30}, {459,1631,-30}, {459,1645,-30}, 
    {459,1655,-30}, {459,1662,-30}, {459,2794,-40}, {459,2795,-40}, 
    {459,2809,-40}, {459,2819,-40}, {459,2826,-40}, {459,2899,-40}, 
    {459,2907,-40}, {459,2995,-120}, {474,603,-10}, {474,604,-10}, 
    {474,609,-10}, {474,615,-10}, {474,624,-10}, {474,1409,20}, 
    {474,2995,20}, {478,393,-20}, {478,396,-20}, {478,404,-20}, 
    {478,414,-20}, {478,419,-20}, {478,449,-20}, {478,455,-20}, 
    {478,570,-20}, {478,603,-20}, {478,604,-20}, {478,609,-20}, 
    {478,615,-20}, {478,624,-20}, {478,658,-50}, {478,668,-40}, 
    {478,686,-50}, {478,687,-50}, {478,692,-50}, {570,0,-90}, 
    {570,5,-90}, {570,16,-90}, {570,27,-90}, {570,33,-90}, 
    {570,43,-90}, {570,48,-90}, {570,393,-40}, {570,396,-40}, 
    {570,404,-40}, {570,414,-40}, {570,419,-40}, {570,449,-40}, 
    {570,455,-40}, {570,734,-80}, {570,736,-80}, {570,758,-80}, 
    {570,775,-80}, {570,1030,-80}, {570,1111,-80}, {570,1173,-80}, 
    {570,1402,-40}, {570,1409,-80}, {570,1630,-60}, {570,1631,-60}, 
    {570,1645,-60}, {570,1655,-60}, {570,1662,-60}, {570,2126,-120}, 
    {570,2794,-80}, {570,2795,-80}, {570,2809,-80}, {570,2819,-80}, 
    {570,2826,-80}, {570,2899,-80}, {570,2907,-80}, {570,2995,-80}, 
    {570,3148,-80}, {570,3351,-40}, {570,3816,-90}, {570,3817,-90}, 
    {570,3824,-90}, {570,3831,-90}, {570,3839,-90}, {570,3951,-60}, 
    {570,4031,-60}, {570,4034,-60}, {570,4050,-60}, {603,0,-50}, 
    {603,5,-50}, {603,16,-50}, {603,27,-50}, {603,33,-50}, 
    {603,43,-50}, {603,48,-50}, {603,1409,-30}, {603,2995,-30}, 
    {604,0,-50}, {604,5,-50}, {604,16,-50}, {604,27,-50}, {604,33,-50}, 
    {604,43,-50}, {604,48,-50}, {604,1409,-30}, {604,2995,-30}, 
    {609,0,-50}, {609,5,-50}, {609,16,-50}, {609,27,-50}, {609,33,-50}, 
    {609,43,-50}, {609,48,-50}, {609,1409,-30}, {609,2995,-30}, 
    {615,0,-50}, {615,5,-50}, {615,16,-50}, {615,27,-50}, {615,33,-50}, 
    {615,43,-50}, {615,48,-50}, {615,1409,-30}, {615,2995,-30}, 
    {624,0,-50}, {624,5,-50}, {624,16,-50}, {624,27,-50}, {624,33,-50}, 
    {624,43,-50}, {624,48,-50}, {624,1409,-30}, {624,2995,-30}, 
    {658,0,-80}, {658,5,-80}, {658,16,-80}, {658,27,-80}, {658,33,-80}, 
    {658,43,-80}, {658,48,-80}, {658,203,-50}, {658,393,-50}, 
    {658,396,-50}, {658,404,-50}, {658,414,-50}, {658,419,-50}, 
    {658,449,-50}, {658,455,-50}, {658,734,-60}, {658,736,-60}, 
    {658,758,-60}, {658,775,-60}, {658,1030,-60}, {658,1111,-60}, 
    {658,1173,-60}, {658,1402,-40}, {658,1409,-120}, {658,1630,-50}, 
    {658,1631,-50}, {658,1645,-50}, {658,1655,-50}, {658,1662,-50}, 
    {658,2126,-80}, {658,2794,-90}, {658,2795,-90}, {658,2809,-90}, 
    {658,2819,-90}, {658,2826,-90}, {658,2899,-90}, {658,2907,-90}, 
    {658,2995,-120}, {658,3351,-40}, {658,3816,-60}, {658,3817,-60}, 
    {658,3824,-60}, {658,3831,-60}, {658,3839,-60}, {668,0,-60}, 
    {668,5,-60}, {668,16,-60}, {668,27,-60}, {668,33,-60}, 
    {668,43,-60}, {668,48,-60}, {668,393,-20}, {668,396,-20}, 
    {668,404,-20}, {668,414,-20}, {668,419,-20}, {668,449,-20}, 
    {668,455,-20}, {668,734,-40}, {668,736,-40}, {668,758,-40}, 
    {668,775,-40}, {668,1030,-40}, {668,1111,-40}, {668,1173,-40}, 
    {668,1402,-10}, {668,1409,-80}, {668,1630,-35}, {668,1631,-35}, 
    {668,1645,-35}, {668,1655,-35}, {668,1662,-35}, {668,2126,-40}, 
    {668,2794,-60}, {668,2795,-60}, {668,2809,-60}, {668,2819,-60}, 
    {668,2826,-60}, {668,2899,-60}, {668,2907,-60}, {668,2995,-80}, 
    {668,3351,-10}, {668,3816,-45}, {668,3817,-45}, {668,3824,-45}, 
    {668,3831,-45}, {668,3839,-45}, {668,4031,-20}, {668,4034,-20}, 
    {668,4050,-20}, {686,0,-110}, {686,5,-110}, {686,16,-110}, 
    {686,27,-110}, {686,33,-110}, {686,43,-110}, {686,48,-110}, 
    {686,393,-70}, {686,396,-70}, {686,404,-70}, {686,414,-70}, 
    {686,419,-70}, {686,449,-70}, {686,455,-70}, {686,734,-90}, 
    {686,736,-90}, {686,758,-90}, {686,775,-90}, {686,1030,-90}, 
    {686,1111,-90}, {686,1173,-90}, {686,1402,-50}, {686,1409,-100}, 
    {686,1630,-80}, {686,1631,-80}, {686,1645,-80}, {686,1655,-80}, 
    {686,1662,-80}, {686,2794,-100}, {686,2795,-100}, {686,2809,-100}, 
    {686,2819,-100}, {686,2826,-100}, {686,2899,-100}, {686,2907,-100}, 
    {686,2995,-100}, {686,3351,-50}, {686,3816,-100}, {686,3817,-100}, 
    {686,3824,-100}, {686,3831,-100}, {686,3839,-100}, {687,0,-110}, 
    {687,5,-110}, {687,16,-110}, {687,27,-110}, {687,33,-110}, 
    {687,43,-110}, {687,48,-110}, {687,393,-70}, {687,396,-70}, 
    {687,404,-70}, {687,414,-70}, {687,419,-70}, {687,449,-70}, 
    {687,455,-70}, {687,734,-90}, {687,736,-90}, {687,758,-90}, 
    {687,775,-90}, {687,1030,-90}, {687,1111,-90}, {687,1173,-90}, 
    {687,1402,-50}, {687,1409,-100}, {687,1630,-80}, {687,1631,-80}, 
    {687,1645,-80}, {687,1655,-80}, {687,1662,-80}, {687,2794,-100}, 
    {687,2795,-100}, {687,2809,-100}, {687,2819,-100}, {687,2826,-100}, 
    {687,2899,-100}, {687,2907,-100}, {687,2995,-100}, {687,3351,-50}, 
    {687,3816,-100}, {687,3817,-100}, {687,3824,-100}, {687,3831,-100}, 
    {687,3839,-100}, {692,0,-110}, {692,5,-110}, {692,16,-110}, 
    {692,27,-110}, {692,33,-110}, {692,43,-110}, {692,48,-110}, 
    {692,393,-70}, {692,396,-70}, {692,404,-70}, {692,414,-70}, 
    {692,419,-70}, {692,449,-70}, {692,455,-70}, {692,734,-90}, 
    {692,736,-90}, {692,758,-90}, {692,775,-90}, {692,1030,-90}, 
    {692,1111,-90}, {692,1173,-90}, {692,1402,-50}, {692,1409,-100}, 
    {692,1630,-80}, {692,1631,-80}, {692,1645,-80}, {692,1655,-80}, 
    {692,1662,-80}, {692,2794,-100}, {692,2795,-100}, {692,2809,-100}, 
    {692,2819,-100}, {692,2826,-100}, {692,2899,-100}, {692,2907,-100}, 
    {692,2995,-100}, {692,3351,-50}, {692,3816,-100}, {692,3817,-100}, 
    {692,3824,-100}, {692,3831,-100}, {692,3839,-100}, {734,1886,-10}, 
    {734,3902,-15}, {734,3951,-15}, {734,4031,-20}, {734,4034,-20}, 
    {734,4050,-20}, {736,1886,-10}, {736,3902,-15}, {736,3951,-15}, 
    {736,4031,-20}, {736,4034,-20}, {736,4050,-20}, {758,1886,-10}, 
    {758,3902,-15}, {758,3951,-15}, {758,4031,-20}, {758,4034,-20}, 
    {758,4050,-20}, {775,1886,-10}, {775,3902,-15}, {775,3951,-15}, 
    {775,4031,-20}, {775,4034,-20}, {775,4050,-20}, {1030,1886,-10}, 
    {1030,3902,-15}, {1030,3951,-15}, {1030,4031,-20}, {1030,4034,-20}, 
    {1030,4050,-20}, {1111,1886,-10}, {1111,3902,-15}, {1111,3951,-15}, 
    {1111,4031,-20}, {1111,4034,-20}, {1111,4050,-20}, {1173,1886,-10}, 
    {1173,3902,-15}, {1173,3951,-15}, {1173,4031,-20}, {1173,4034,-20}, 
    {1173,4050,-20}, {1192,2416,-10}, {1192,2501,-10}, {1192,3816,-20}, 
    {1192,3817,-20}, {1192,3824,-20}, {1192,3831,-20}, {1192,3839,-20}, 
    {1192,3902,-20}, {1192,4031,-20}, {1192,4034,-20}, {1192,4050,-20}, 
    {1312,1988,-10}, {1312,2302,-20}, {1312,2416,-20}, {1312,2501,-20}, 
    {1312,4031,-10}, {1312,4034,-10}, {1312,4050,-10}, {1332,1988,-10}, 
    {1332,2302,-20}, {1332,2416,-20}, {1332,2501,-20}, {1332,4031,-10}, 
    {1332,4034,-10}, {1332,4050,-10}, {1402,3498,-40}, 
    {1409,3139,-120}, {1409,3143,-120}, {1409,3498,-40}, 
    {1481,1481,-10}, {1481,3902,-15}, {1481,3951,-15}, {1481,4031,-15}, 
    {1481,4034,-15}, {1481,4050,-15}, {1630,1409,10}, {1630,2995,20}, 
    {1630,3902,-15}, {1630,3951,-15}, {1630,4020,-15}, {1630,4031,-15}, 
    {1630,4034,-15}, {1630,4050,-15}, {1631,1409,10}, {1631,2995,20}, 
    {1631,3902,-15}, {1631,3951,-15}, {1631,4020,-15}, {1631,4031,-15}, 
    {1631,4034,-15}, {1631,4050,-15}, {1645,1409,10}, {1645,2995,20}, 
    {1645,3902,-15}, {1645,3951,-15}, {1645,4020,-15}, {1645,4031,-15}, 
    {1645,4034,-15}, {1645,4050,-15}, {1655,1409,10}, {1655,2995,20}, 
    {1655,3902,-15}, {1655,3951,-15}, {1655,4020,-15}, {1655,4031,-15}, 
    {1655,4034,-15}, {1655,4050,-15}, {1662,1409,10}, {1662,2995,20}, 
    {1662,3902,-15}, {1662,3951,-15}, {1662,4020,-15}, {1662,4031,-15}, 
    {1662,4034,-15}, {1662,4050,-15}, {1787,1409,-10}, {1787,1630,-10}, 
    {1787,1631,-10}, {1787,1645,-10}, {1787,1655,-10}, {1787,1662,-10}, 
    {1787,2794,-20}, {1787,2795,-20}, {1787,2809,-20}, {1787,2819,-20}, 
    {1787,2826,-20}, {1787,2899,-20}, {1787,2907,-20}, {1787,2995,-10}, 
    {1787,3139,30}, {1787,3143,30}, {1886,1630,10}, {1886,1631,10}, 
    {1886,1645,10}, {1886,1655,10}, {1886,1662,10}, {1886,1886,-10}, 
    {1988,4031,-20}, {1988,4034,-20}, {1988,4050,-20}, {2302,2794,-15}, 
    {2302,2795,-15}, {2302,2809,-15}, {2302,2819,-15}, {2302,2826,-15}, 
    {2302,2899,-15}, {2302,2907,-15}, {2416,3951,-15}, {2416,4031,-15}, 
    {2416,4034,-15}, {2416,4050,-15}, {2501,3951,-15}, {2501,4031,-15}, 
    {2501,4034,-15}, {2501,4050,-15}, {2511,3816,-20}, {2511,3817,-20}, 
    {2511,3824,-20}, {2511,3831,-20}, {2511,3839,-20}, {2511,4031,-30}, 
    {2511,4034,-30}, {2511,4050,-30}, {2648,3816,-10}, {2648,3817,-10}, 
    {2648,3824,-10}, {2648,3831,-10}, {2648,3839,-10}, {2648,3902,-40}, 
    {2648,4031,-20}, {2648,4034,-20}, {2648,4050,-20}, {2769,3816,-10}, 
    {2769,3817,-10}, {2769,3824,-10}, {2769,3831,-10}, {2769,3839,-10}, 
    {2769,3902,-40}, {2769,4031,-20}, {2769,4034,-20}, {2769,4050,-20}, 
    {2794,3902,-20}, {2794,3951,-15}, {2794,4020,-30}, {2794,4031,-20}, 
    {2794,4034,-20}, {2794,4050,-20}, {2795,3902,-20}, {2795,3951,-15}, 
    {2795,4020,-30}, {2795,4031,-20}, {2795,4034,-20}, {2795,4050,-20}, 
    {2809,3902,-20}, {2809,3951,-15}, {2809,4020,-30}, {2809,4031,-20}, 
    {2809,4034,-20}, {2809,4050,-20}, {2819,3902,-20}, {2819,3951,-15}, 
    {2819,4020,-30}, {2819,4031,-20}, {2819,4034,-20}, {2819,4050,-20}, 
    {2826,3902,-20}, {2826,3951,-15}, {2826,4020,-30}, {2826,4031,-20}, 
    {2826,4034,-20}, {2826,4050,-20}, {2899,3902,-20}, {2899,3951,-15}, 
    {2899,4020,-30}, {2899,4031,-20}, {2899,4034,-20}, {2899,4050,-20}, 
    {2907,3902,-20}, {2907,3951,-15}, {2907,4020,-30}, {2907,4031,-20}, 
    {2907,4034,-20}, {2907,4050,-20}, {2921,4031,-15}, {2921,4034,-15}, 
    {2921,4050,-15}, {2995,3139,-120}, {2995,3143,-120}, 
    {2995,3498,-40}, {3139,3498,-80}, {3140,3140,-46}, {3143,1481,-80}, 
    {3143,2416,-20}, {3143,2501,-20}, {3143,3143,-46}, {3143,3148,-40}, 
    {3143,3279,-60}, {3143,3317,-60}, {3143,3498,-80}, {3143,3902,-20}, 
    {3148,1312,-20}, {3148,1332,-20}, {3148,1409,-60}, {3148,1481,-20}, 
    {3148,1886,-15}, {3148,2126,-20}, {3148,2794,-20}, {3148,2795,-20}, 
    {3148,2809,-20}, {3148,2819,-20}, {3148,2826,-20}, {3148,2899,-20}, 
    {3148,2907,-20}, {3148,2995,-60}, {3148,3083,-20}, {3148,3279,-15}, 
    {3148,3317,-15}, {3148,3556,20}, {3148,3902,10}, {3148,4031,10}, 
    {3148,4034,10}, {3148,4050,10}, {3279,3951,-15}, {3317,3951,-15}, 
    {3351,3498,-40}, {3498,570,-100}, {3498,658,-80}, {3498,668,-80}, 
    {3498,686,-120}, {3498,687,-120}, {3498,692,-120}, {3498,3135,-80}, 
    {3498,3140,-60}, {3902,734,-20}, {3902,736,-20}, {3902,758,-20}, 
    {3902,775,-20}, {3902,1030,-20}, {3902,1111,-20}, {3902,1173,-20}, 
    {3902,1409,-80}, {3902,2794,-30}, {3902,2795,-30}, {3902,2809,-30}, 
    {3902,2819,-30}, {3902,2826,-30}, {3902,2899,-30}, {3902,2907,-30}, 
    {3902,2995,-80}, {3951,1409,-40}, {3951,2794,-20}, {3951,2795,-20}, 
    {3951,2809,-20}, {3951,2819,-20}, {3951,2826,-20}, {3951,2899,-20}, 
    {3951,2907,-20}, {3951,2995,-40}, {4020,1630,-10}, {4020,1631,-10}, 
    {4020,1645,-10}, {4020,1655,-10}, {4020,1662,-10}, {4031,734,-30}, 
    {4031,736,-30}, {4031,758,-30}, {4031,775,-30}, {4031,1030,-30}, 
    {4031,1111,-30}, {4031,1173,-30}, {4031,1409,-80}, {4031,1630,-10}, 
    {4031,1631,-10}, {4031,1645,-10}, {4031,1655,-10}, {4031,1662,-10}, 
    {4031,2794,-25}, {4031,2795,-25}, {4031,2809,-25}, {4031,2819,-25}, 
    {4031,2826,-25}, {4031,2899,-25}, {4031,2907,-25}, {4031,2995,-80}, 
    {4034,734,-30}, {4034,736,-30}, {4034,758,-30}, {4034,775,-30}, 
    {4034,1030,-30}, {4034,1111,-30}, {4034,1173,-30}, {4034,1409,-80}, 
    {4034,1630,-10}, {4034,1631,-10}, {4034,1645,-10}, {4034,1655,-10}, 
    {4034,1662,-10}, {4034,2794,-25}, {4034,2795,-25}, {4034,2809,-25}, 
    {4034,2819,-25}, {4034,2826,-25}, {4034,2899,-25}, {4034,2907,-25}, 
    {4034,2995,-80}, {4050,734,-30}, {4050,736,-30}, {4050,758,-30}, 
    {4050,775,-30}, {4050,1030,-30}, {4050,1111,-30}, {4050,1173,-30}, 
    {4050,1409,-80}, {4050,1630,-10}, {4050,1631,-10}, {4050,1645,-10}, 
    {4050,1655,-10}, {4050,1662,-10}, {4050,2794,-25}, {4050,2795,-25}, 
    {4050,2809,-25}, {4050,2819,-25}, {4050,2826,-25}, {4050,2899,-25}, 
    {4050,2907,-25}, {4050,2995,-80}, {4132,1630,10}, {4132,1631,10}, 
    {4132,1645,10}, {4132,1655,10}, {4132,1662,10}, {4153,1630,10}, 
    {4153,1631,10}, {4153,1645,10}, {4153,1655,10}, {4153,1662,10}, 
    {NOGLYPH,NOGLYPH,0}
};
static const ligature helvetica_bold_ligs[] = {
    {1787,2132,1807}, {1787,2416,1850}, 
    {NOGLYPH,NOGLYPH,NOGLYPH}
};
static const kern_pair helvetica_boldoblique_kerns[] = {
    {0,64,-40}, {0,70,-40}, {0,203,-50}, {0,393,-40}, {0,396,-40}, 
    {0,404,-40}, {0,414,-40}, {0,419,-40}, {0,449,-40}, {0,455,-40}, 
    {0,474,-40}, {0,570,-90}, {0,603,-50}, {0,604,-50}, {0,609,-50}, 
    {0,615,-50}, {0,624,-50}, {0,658,-80}, {0,668,-60}, {0,686,-110}, 
    {0,687,-110}, {0,692,-110}, {0,3816,-30}, {0,3817,-30}, 
    {0,3824,-30}, {0,3831,-30}, {0,3839,-30}, {0,3902,-40}, 
    {0,3951,-30}, {0,4031,-30}, {0,4034,-30}, {0,4050,-30}, {5,64,-40}, 
    {5,70,-40}, {5,203,-50}, {5,393,-40}, {5,396,-40}, {5,404,-40}, 
    {5,414,-40}, {5,419,-40}, {5,449,-40}, {5,455,-40}, {5,474,-40}, 
    {5,570,-90}, {5,603,-50}, {5,604,-50}, {5,609,-50}, {5,615,-50}, 
    {5,624,-50}, {5,658,-80}, {5,668,-60}, {5,686,-110}, {5,687,-110}, 
    {5,692,-110}, {5,3816,-30}, {5,3817,-30}, {5,3824,-30}, 
    {5,3831,-30}, {5,3839,-30}, {5,3902,-40}, {5,3951,-30}, 
    {5,4031,-30}, {5,4034,-30}, {5,4050,-30}, {16,64,-40}, {16,70,-40}, 
    {16,203,-50}, {16,393,-40}, {16,396,-40}, {16,404,-40}, 
    {16,414,-40}, {16,419,-40}, {16,449,-40}, {16,455,-40}, 
    {16,474,-40}, {16,570,-90}, {16,603,-50}, {16,604,-50}, 
    {16,609,-50}, {16,615,-50}, {16,624,-50}, {16,658,-80}, 
    {16,668,-60}, {16,686,-110}, {16,687,-110}, {16,692,-110}, 
    {16,3816,-30}, {16,3817,-30}, {16,3824,-30}, {16,3831,-30}, 
    {16,3839,-30}, {16,3902,-40}, {16,3951,-30}, {16,4031,-30}, 
    {16,4034,-30}, {16,4050,-30}, {27,64,-40}, {27,70,-40}, 
    {27,203,-50}, {27,393,-40}, {27,396,-40}, {27,404,-40}, 
    {27,414,-40}, {27,419,-40}, {27,449,-40}, {27,455,-40}, 
    {27,474,-40}, {27,570,-90}, {27,603,-50}, {27,604,-50}, 
    {27,609,-50}, {27,615,-50}, {27,624,-50}, {27,658,-80}, 
    {27,668,-60}, {27,686,-110}, {27,687,-110}, {27,692,-110}, 
    {27,3816,-30}, {27,3817,-30}, {27,3824,-30}, {27,3831,-30}, 
    {27,3839,-30}, {27,3902,-40}, {27,3951,-30}, {27,4031,-30}, 
    {27,4034,-30}, {27,4050,-30}, {33,64,-40}, {33,70,-40}, 
    {33,203,-50}, {33,393,-40}, {33,396,-40}, {33,404,-40}, 
    {33,414,-40}, {33,419,-40}, {33,449,-40}, {33,455,-40}, 
    {33,474,-40}, {33,570,-90}, {33,603,-50}, {33,604,-50}, 
    {33,609,-50}, {33,615,-50}, {33,624,-50}, {33,658,-80}, 
    {33,668,-60}, {33,686,-110}, {33,687,-110}, {33,692,-110}, 
    {33,3816,-30}, {33,3817,-30}, {33,3824,-30}, {33,3831,-30}, 
    {33,3839,-30}, {33,3902,-40}, {33,3951,-30}, {33,4031,-30}, 
    {33,4034,-30}, {33,4050,-30}, {43,64,-40}, {43,70,-40}, 
    {43,203,-50}, {43,393,-40}, {43,396,-40}, {43,404,-40}, 
    {43,414,-40}, {43,419,-40}, {43,449,-40}, {43,455,-40}, 
    {43,474,-40}, {43,570,-90}, {43,603,-50}, {43,604,-50}, 
    {43,609,-50}, {43,615,-50}, {43,624,-50}, {43,658,-80}, 
    {43,668,-60}, {43,686,-110}, {43,687,-110}, {43,692,-110}, 
    {43,3816,-30}, {43,3817,-30}, {43,3824,-30}, {43,3831,-30}, 
    {43,3839,-30}, {43,3902,-40}, {43,3951,-30}, {43,4031,-30}, 
    {43,4034,-30}, {43,4050,-30}, {48,64,-40}, {48,70,-40}, 
    {48,203,-50}, {48,393,-40}, {48,396,-40}, {48,404,-40}, 
    {48,414,-40}, {48,419,-40}, {48,449,-40}, {48,455,-40}, 
    {48,474,-40}, {48,570,-90}, {48,603,-50}, {48,604,-50}, 
    {48,609,-50}, {48,615,-50}, {48,624,-50}, {48,658,-80}, 
    {48,668,-60}, {48,686,-110}, {48,687,-110}, {48,692,-110}, 
    {48,3816,-30}, {48,3817,-30}, {48,3824,-30}, {48,3831,-30}, 
    {48,3839,-30}, {48,3902,-40}, {48,3951,-30}, {48,4031,-30}, 
    {48,4034,-30}, {48,4050,-30}, {51,0,-30}, {51,5,-30}, {51,16,-30}, 
    {51,27,-30}, {51,33,-30}, {51,43,-30}, {51,48,-30}, {51,603,-10}, 
    {51,604,-10}, {51,609,-10}, {51,615,-10}, {51,624,-10}, {93,0,-40}, 
    {93,5,-40}, {93,16,-40}, {93,27,-40}, {93,33,-40}, {93,43,-40}, 
    {93,48,-40}, {93,658,-40}, {93,668,-40}, {93,686,-70}, 
    {93,687,-70}, {93,692,-70}, {93,1409,-30}, {93,2995,-30}, 
    {192,0,-80}, {192,5,-80}, {192,16,-80}, {192,27,-80}, {192,33,-80}, 
    {192,43,-80}, {192,48,-80}, {192,734,-20}, {192,736,-20}, 
    {192,758,-20}, {192,775,-20}, {192,1030,-20}, {192,1111,-20}, 
    {192,1173,-20}, {192,1409,-100}, {192,2995,-100}, {301,0,-20}, 
    {301,5,-20}, {301,16,-20}, {301,27,-20}, {301,33,-20}, 
    {301,43,-20}, {301,48,-20}, {301,1409,-20}, {301,2995,-20}, 
    {301,3816,-20}, {301,3817,-20}, {301,3824,-20}, {301,3831,-20}, 
    {301,3839,-20}, {309,393,-30}, {309,396,-30}, {309,404,-30}, 
    {309,414,-30}, {309,419,-30}, {309,449,-30}, {309,455,-30}, 
    {309,1630,-15}, {309,1631,-15}, {309,1645,-15}, {309,1655,-15}, 
    {309,1662,-15}, {309,2794,-35}, {309,2795,-35}, {309,2809,-35}, 
    {309,2819,-35}, {309,2826,-35}, {309,2899,-35}, {309,2907,-35}, 
    {309,3816,-30}, {309,3817,-30}, {309,3824,-30}, {309,3831,-30}, 
    {309,3839,-30}, {309,4031,-40}, {309,4034,-40}, {309,4050,-40}, 
    {337,570,-90}, {337,658,-110}, {337,668,-80}, {337,686,-120}, 
    {337,687,-120}, {337,692,-120}, {337,3139,-140}, {337,3143,-140}, 
    {337,4031,-30}, {337,4034,-30}, {337,4050,-30}, {356,570,-90}, 
    {356,658,-110}, {356,668,-80}, {356,686,-120}, {356,687,-120}, 
    {356,692,-120}, {356,3139,-140}, {356,3143,-140}, {356,4031,-30}, 
    {356,4034,-30}, {356,4050,-30}, {393,0,-50}, {393,5,-50}, 
    {393,16,-50}, {393,27,-50}, {393,33,-50}, {393,43,-50}, 
    {393,48,-50}, {393,570,-40}, {393,658,-50}, {393,668,-50}, 
    {393,678,-50}, {393,686,-70}, {393,687,-70}, {393,692,-70}, 
    {393,1409,-40}, {393,2995,-40}, {396,0,-50}, {396,5,-50}, 
    {396,16,-50}, {396,27,-50}, {396,33,-50}, {396,43,-50}, 
    {396,48,-50}, {396,570,-40}, {396,658,-50}, {396,668,-50}, 
    {396,678,-50}, {396,686,-70}, {396,687,-70}, {396,692,-70}, 
    {396,1409,-40}, {396,2995,-40}, {404,0,-50}, {404,5,-50}, 
    {404,16,-50}, {404,27,-50}, {404,33,-50}, {404,43,-50}, 
    {404,48,-50}, {404,570,-40}, {404,658,-50}, {404,668,-50}, 
    {404,678,-50}, {404,686,-70}, {404,687,-70}, {404,692,-70}, 
    {404,1409,-40}, {404,2995,-40}, {414,0,-50}, {414,5,-50}, 
    {414,16,-50}, {414,27,-50}, {414,33,-50}, {414,43,-50}, 
    {414,48,-50}, {414,570,-40}, {414,658,-50}, {414,668,-50}, 
    {414,678,-50}, {414,686,-70}, {414,687,-70}, {414,692,-70}, 
    {414,1409,-40}, {414,2995,-40}, {419,0,-50}, {419,5,-50}, 
    {419,16,-50}, {419,27,-50}, {419,33,-50}, {419,43,-50}, 
    {419,48,-50}, {419,570,-40}, {419,658,-50}, {419,668,-50}, 
    {419,678,-50}, {419,686,-70}, {419,687,-70}, {419,692,-70}, 
    {419,1409,-40}, {419,2995,-40}, {449,0,-50}, {449,5,-50}, 
    {449,16,-50}, {449,27,-50}, {449,33,-50}, {449,43,-50}, 
    {449,48,-50}, {449,570,-40}, {449,658,-50}, {449,668,-50}, 
    {449,678,-50}, {449,686,-70}, {449,687,-70}, {449,692,-70}, 
    {449,1409,-40}, {449,2995,-40}, {455,0,-50}, {455,5,-50}, 
    {455,16,-50}, {455,27,-50}, {455,33,-50}, {455,43,-50}, 
    {455,48,-50}, {455,570,-40}, {455,658,-50}, {455,668,-50}, 
    {455,678,-50}, {455,686,-70}, {455,687,-70}, {455,692,-70}, 
    {455,1409,-40}, {455,2995,-40}, {459,0,-100}, {459,5,-100}, 
    {459,16,-100}, {459,27,-100}, {459,33,-100}, {459,43,-100}, 
    {459,48,-100}, {459,734,-30}, {459,736,-30}, {459,758,-30}, 
    {459,775,-30}, {459,1030,-30}, {459,1111,-30}, {459,1173,-30}, 
    {459,1409,-120}, {459,1630,-30}, {459,1631,-30}, {459,1645,-30}, 
    {459,1655,-30}, {459,1662,-30}, {459,2794,-40}, {459,2795,-40}, 
    {459,2809,-40}, {459,2819,-40}, {459,2826,-40}, {459,2899,-40}, 
    {459,2907,-40}, {459,2995,-120}, {474,603,-10}, {474,604,-10}, 
    {474,609,-10}, {474,615,-10}, {474,624,-10}, {474,1409,20}, 
    {474,2995,20}, {478,393,-20}, {478,396,-20}, {478,404,-20}, 
    {478,414,-20}, {478,419,-20}, {478,449,-20}, {478,455,-20}, 
    {478,570,-20}, {478,603,-20}, {478,604,-20}, {478,609,-20}, 
    {478,615,-20}, {478,624,-20}, {478,658,-50}, {478,668,-40}, 
    {478,686,-50}, {478,687,-50}, {478,692,-50}, {570,0,-90}, 
    {570,5,-90}, {570,16,-90}, {570,27,-90}, {570,33,-90}, 
    {570,43,-90}, {570,48,-90}, {570,393,-40}, {570,396,-40}, 
    {570,404,-40}, {570,414,-40}, {570,419,-40}, {570,449,-40}, 
    {570,455,-40}, {570,734,-80}, {570,736,-80}, {570,758,-80}, 
    {570,775,-80}, {570,1030,-80}, {570,1111,-80}, {570,1173,-80}, 
    {570,1402,-40}, {570,1409,-80}, {570,1630,-60}, {570,1631,-60}, 
    {570,1645,-60}, {570,1655,-60}, {570,1662,-60}, {570,2126,-120}, 
    {570,2794,-80}, {570,2795,-80}, {570,2809,-80}, {570,2819,-80}, 
    {570,2826,-80}, {570,2899,-80}, {570,2907,-80}, {570,2995,-80}, 
    {570,3148,-80}, {570,3351,-40}, {570,3816,-90}, {570,3817,-90}, 
    {570,3824,-90}, {570,3831,-90}, {570,3839,-90}, {570,3951,-60}, 
    {570,4031,-60}, {570,4034,-60}, {570,4050,-60}, {603,0,-50}, 
    {603,5,-50}, {603,16,-50}, {603,27,-50}, {603,33,-50}, 
    {603,43,-50}, {603,48,-50}, {603,1409,-30}, {603,2995,-30}, 
    {604,0,-50}, {604,5,-50}, {604,16,-50}, {604,27,-50}, {604,33,-50}, 
    {604,43,-50}, {604,48,-50}, {604,1409,-30}, {604,2995,-30}, 
    {609,0,-50}, {609,5,-50}, {609,16,-50}, {609,27,-50}, {609,33,-50}, 
    {609,43,-50}, {609,48,-50}, {609,1409,-30}, {609,2995,-30}, 
    {615,0,-50}, {615,5,-50}, {615,16,-50}, {615,27,-50}, {615,33,-50}, 
    {615,43,-50}, {615,48,-50}, {615,1409,-30}, {615,2995,-30}, 
    {624,0,-50}, {624,5,-50}, {624,16,-50}, {624,27,-50}, {624,33,-50}, 
    {624,43,-50}, {624,48,-50}, {624,1409,-30}, {624,2995,-30}, 
    {658,0,-80}, {658,5,-80}, {658,16,-80}, {658,27,-80}, {658,33,-80}, 
    {658,43,-80}, {658,48,-80}, {658,203,-50}, {658,393,-50}, 
    {658,396,-50}, {658,404,-50}, {658,414,-50}, {658,419,-50}, 
    {658,449,-50}, {658,455,-50}, {658,734,-60}, {658,736,-60}, 
    {658,758,-60}, {658,775,-60}, {658,1030,-60}, {658,1111,-60}, 
    {658,1173,-60}, {658,1402,-40}, {658,1409,-120}, {658,1630,-50}, 
    {658,1631,-50}, {658,1645,-50}, {658,1655,-50}, {658,1662,-50}, 
    {658,2126,-80}, {658,2794,-90}, {658,2795,-90}, {658,2809,-90}, 
    {658,2819,-90}, {658,2826,-90}, {658,2899,-90}, {658,2907,-90}, 
    {658,2995,-120}, {658,3351,-40}, {658,3816,-60}, {658,3817,-60}, 
    {658,3824,-60}, {658,3831,-60}, {658,3839,-60}, {668,0,-60}, 
    {668,5,-60}, {668,16,-60}, {668,27,-60}, {668,33,-60}, 
    {668,43,-60}, {668,48,-60}, {668,393,-20}, {668,396,-20}, 
    {668,404,-20}, {668,414,-20}, {668,419,-20}, {668,449,-20}, 
    {668,455,-20}, {668,734,-40}, {668,736,-40}, {668,758,-40}, 
    {668,775,-40}, {668,1030,-40}, {668,1111,-40}, {668,1173,-40}, 
    {668,1402,-10}, {668,1409,-80}, {668,1630,-35}, {668,1631,-35}, 
    {668,1645,-35}, {668,1655,-35}, {668,1662,-35}, {668,2126,-40}, 
    {668,2794,-60}, {668,2795,-60}, {668,2809,-60}, {668,2819,-60}, 
    {668,2826,-60}, {668,2899,-60}, {668,2907,-60}, {668,2995,-80}, 
    {668,3351,-10}, {668,3816,-45}, {668,3817,-45}, {668,3824,-45}, 
    {668,3831,-45}, {668,3839,-45}, {668,4031,-20}, {668,4034,-20}, 
    {668,4050,-20}, {686,0,-110}, {686,5,-110}, {686,16,-110}, 
    {686,27,-110}, {686,33,-110}, {686,43,-110}, {686,48,-110}, 
    {686,393,-70}, {686,396,-70}, {686,404,-70}, {686,414,-70}, 
    {686,419,-70}, {686,449,-70}, {686,455,-70}, {686,734,-90}, 
    {686,736,-90}, {686,758,-90}, {686,775,-90}, {686,1030,-90}, 
    {686,1111,-90}, {686,1173,-90}, {686,1402,-50}, {686,1409,-100}, 
    {686,1630,-80}, {686,1631,-80}, {686,1645,-80}, {686,1655,-80}, 
    {686,1662,-80}, {686,2794,-100}, {686,2795,-100}, {686,2809,-100}, 
    {686,2819,-100}, {686,2826,-100}, {686,2899,-100}, {686,2907,-100}, 
    {686,2995,-100}, {686,3351,-50}, {686,3816,-100}, {686,3817,-100}, 
    {686,3824,-100}, {686,3831,-100}, {686,3839,-100}, {687,0,-110}, 
    {687,5,-110}, {687,16,-110}, {687,27,-110}, {687,33,-110}, 
    {687,43,-110}, {687,48,-110}, {687,393,-70}, {687,396,-70}, 
    {687,404,-70}, {687,414,-70}, {687,419,-70}, {687,449,-70}, 
    {687,455,-70}, {687,734,-90}, {687,736,-90}, {687,758,-90}, 
    {687,775,-90}, {687,1030,-90}, {687,1111,-90}, {687,1173,-90}, 
    {687,1402,-50}, {687,1409,-100}, {687,1630,-80}, {687,1631,-80}, 
    {687,1645,-80}, {687,1655,-80}, {687,1662,-80}, {687,2794,-100}, 
    {687,2795,-100}, {687,2809,-100}, {687,2819,-100}, {687,2826,-100}, 
    {687,2899,-100}, {687,2907,-100}, {687,2995,-100}, {687,3351,-50}, 
    {687,3816,-100}, {687,3817,-100}, {687,3824,-100}, {687,3831,-100}, 
    {687,3839,-100}, {692,0,-110}, {692,5,-110}, {692,16,-110}, 
    {692,27,-110}, {692,33,-110}, {692,43,-110}, {692,48,-110}, 
    {692,393,-70}, {692,396,-70}, {692,404,-70}, {692,414,-70}, 
    {692,419,-70}, {692,449,-70}, {692,455,-70}, {692,734,-90}, 
    {692,736,-90}, {692,758,-90}, {692,775,-90}, {692,1030,-90}, 
    {692,1111,-90}, {692,1173,-90}, {692,1402,-50}, {692,1409,-100}, 
    {692,1630,-80}, {692,1631,-80}, {692,1645,-80}, {692,1655,-80}, 
    {692,1662,-80}, {692,2794,-100}, {692,2795,-100}, {692,2809,-100}, 
    {692,2819,-100}, {692,2826,-100}, {692,2899,-100}, {692,2907,-100}, 
    {692,2995,-100}, {692,3351,-50}, {692,3816,-100}, {692,3817,-100}, 
    {692,3824,-100}, {692,3831,-100}, {692,3839,-100}, {734,1886,-10}, 
    {734,3902,-15}, {734,3951,-15}, {734,4031,-20}, {734,4034,-20}, 
    {734,4050,-20}, {736,1886,-10}, {736,3902,-15}, {736,3951,-15}, 
    {736,4031,-20}, {736,4034,-20}, {736,4050,-20}, {758,1886,-10}, 
    {758,3902,-15}, {758,3951,-15}, {758,4031,-20}, {758,4034,-20}, 
    {758,4050,-20}, {775,1886,-10}, {775,3902,-15}, {775,3951,-15}, 
    {775,4031,-20}, {775,4034,-20}, {775,4050,-20}, {1030,1886,-10}, 
    {1030,3902,-15}, {1030,3951,-15}, {1030,4031,-20}, {1030,4034,-20}, 
    {1030,4050,-20}, {1111,1886,-10}, {1111,3902,-15}, {1111,3951,-15}, 
    {1111,4031,-20}, {1111,4034,-20}, {1111,4050,-20}, {1173,1886,-10}, 
    {1173,3902,-15}, {1173,3951,-15}, {1173,4031,-20}, {1173,4034,-20}, 
    {1173,4050,-20}, {1192,2416,-10}, {1192,2501,-10}, {1192,3816,-20}, 
    {1192,3817,-20}, {1192,3824,-20}, {1192,3831,-20}, {1192,3839,-20}, 
    {1192,3902,-20}, {1192,4031,-20}, {1192,4034,-20}, {1192,4050,-20}, 
    {1312,1988,-10}, {1312,2302,-20}, {1312,2416,-20}, {1312,2501,-20}, 
    {1312,4031,-10}, {1312,4034,-10}, {1312,4050,-10}, {1332,1988,-10}, 
    {1332,2302,-20}, {1332,2416,-20}, {1332,2501,-20}, {1332,4031,-10}, 
    {1332,4034,-10}, {1332,4050,-10}, {1402,3498,-40}, 
    {1409,3139,-120}, {1409,3143,-120}, {1409,3498,-40}, 
    {1481,1481,-10}, {1481,3902,-15}, {1481,3951,-15}, {1481,4031,-15}, 
    {1481,4034,-15}, {1481,4050,-15}, {1630,1409,10}, {1630,2995,20}, 
    {1630,3902,-15}, {1630,3951,-15}, {1630,4020,-15}, {1630,4031,-15}, 
    {1630,4034,-15}, {1630,4050,-15}, {1631,1409,10}, {1631,2995,20}, 
    {1631,3902,-15}, {1631,3951,-15}, {1631,4020,-15}, {1631,4031,-15}, 
    {1631,4034,-15}, {1631,4050,-15}, {1645,1409,10}, {1645,2995,20}, 
    {1645,3902,-15}, {1645,3951,-15}, {1645,4020,-15}, {1645,4031,-15}, 
    {1645,4034,-15}, {1645,4050,-15}, {1655,1409,10}, {1655,2995,20}, 
    {1655,3902,-15}, {1655,3951,-15}, {1655,4020,-15}, {1655,4031,-15}, 
    {1655,4034,-15}, {1655,4050,-15}, {1662,1409,10}, {1662,2995,20}, 
    {1662,3902,-15}, {1662,3951,-15}, {1662,4020,-15}, {1662,4031,-15}, 
    {1662,4034,-15}, {1662,4050,-15}, {1787,1409,-10}, {1787,1630,-10}, 
    {1787,1631,-10}, {1787,1645,-10}, {1787,1655,-10}, {1787,1662,-10}, 
    {1787,2794,-20}, {1787,2795,-20}, {1787,2809,-20}, {1787,2819,-20}, 
    {1787,2826,-20}, {1787,2899,-20}, {1787,2907,-20}, {1787,2995,-10}, 
    {1787,3139,30}, {1787,3143,30}, {1886,1630,10}, {1886,1631,10}, 
    {1886,1645,10}, {1886,1655,10}, {1886,1662,10}, {1886,1886,-10}, 
    {1988,4031,-20}, {1988,4034,-20}, {1988,4050,-20}, {2302,2794,-15}, 
    {2302,2795,-15}, {2302,2809,-15}, {2302,2819,-15}, {2302,2826,-15}, 
    {2302,2899,-15}, {2302,2907,-15}, {2416,3951,-15}, {2416,4031,-15}, 
    {2416,4034,-15}, {2416,4050,-15}, {2501,3951,-15}, {2501,4031,-15}, 
    {2501,4034,-15}, {2501,4050,-15}, {2511,3816,-20}, {2511,3817,-20}, 
    {2511,3824,-20}, {2511,3831,-20}, {2511,3839,-20}, {2511,4031,-30}, 
    {2511,4034,-30}, {2511,4050,-30}, {2648,3816,-10}, {2648,3817,-10}, 
    {2648,3824,-10}, {2648,3831,-10}, {2648,3839,-10}, {2648,3902,-40}, 
    {2648,4031,-20}, {2648,4034,-20}, {2648,4050,-20}, {2769,3816,-10}, 
    {2769,3817,-10}, {2769,3824,-10}, {2769,3831,-10}, {2769,3839,-10}, 
    {2769,3902,-40}, {2769,4031,-20}, {2769,4034,-20}, {2769,4050,-20}, 
    {2794,3902,-20}, {2794,3951,-15}, {2794,4020,-30}, {2794,4031,-20}, 
    {2794,4034,-20}, {2794,4050,-20}, {2795,3902,-20}, {2795,3951,-15}, 
    {2795,4020,-30}, {2795,4031,-20}, {2795,4034,-20}, {2795,4050,-20}, 
    {2809,3902,-20}, {2809,3951,-15}, {2809,4020,-30}, {2809,4031,-20}, 
    {2809,4034,-20}, {2809,4050,-20}, {2819,3902,-20}, {2819,3951,-15}, 
    {2819,4020,-30}, {2819,4031,-20}, {2819,4034,-20}, {2819,4050,-20}, 
    {2826,3902,-20}, {2826,3951,-15}, {2826,4020,-30}, {2826,4031,-20}, 
    {2826,4034,-20}, {2826,4050,-20}, {2899,3902,-20}, {2899,3951,-15}, 
    {2899,4020,-30}, {2899,4031,-20}, {2899,4034,-20}, {2899,4050,-20}, 
    {2907,3902,-20}, {2907,3951,-15}, {2907,4020,-30}, {2907,4031,-20}, 
    {2907,4034,-20}, {2907,4050,-20}, {2921,4031,-15}, {2921,4034,-15}, 
    {2921,4050,-15}, {2995,3139,-120}, {2995,3143,-120}, 
    {2995,3498,-40}, {3139,3498,-80}, {3140,3140,-46}, {3143,1481,-80}, 
    {3143,2416,-20}, {3143,2501,-20}, {3143,3143,-46}, {3143,3148,-40}, 
    {3143,3279,-60}, {3143,3317,-60}, {3143,3498,-80}, {3143,3902,-20}, 
    {3148,1312,-20}, {3148,1332,-20}, {3148,1409,-60}, {3148,1481,-20}, 
    {3148,1886,-15}, {3148,2126,-20}, {3148,2794,-20}, {3148,2795,-20}, 
    {3148,2809,-20}, {3148,2819,-20}, {3148,2826,-20}, {3148,2899,-20}, 
    {3148,2907,-20}, {3148,2995,-60}, {3148,3083,-20}, {3148,3279,-15}, 
    {3148,3317,-15}, {3148,3556,20}, {3148,3902,10}, {3148,4031,10}, 
    {3148,4034,10}, {3148,4050,10}, {3279,3951,-15}, {3317,3951,-15}, 
    {3351,3498,-40}, {3498,570,-100}, {3498,658,-80}, {3498,668,-80}, 
    {3498,686,-120}, {3498,687,-120}, {3498,692,-120}, {3498,3135,-80}, 
    {3498,3140,-60}, {3902,734,-20}, {3902,736,-20}, {3902,758,-20}, 
    {3902,775,-20}, {3902,1030,-20}, {3902,1111,-20}, {3902,1173,-20}, 
    {3902,1409,-80}, {3902,2794,-30}, {3902,2795,-30}, {3902,2809,-30}, 
    {3902,2819,-30}, {3902,2826,-30}, {3902,2899,-30}, {3902,2907,-30}, 
    {3902,2995,-80}, {3951,1409,-40}, {3951,2794,-20}, {3951,2795,-20}, 
    {3951,2809,-20}, {3951,2819,-20}, {3951,2826,-20}, {3951,2899,-20}, 
    {3951,2907,-20}, {3951,2995,-40}, {4020,1630,-10}, {4020,1631,-10}, 
    {4020,1645,-10}, {4020,1655,-10}, {4020,1662,-10}, {4031,734,-30}, 
    {4031,736,-30}, {4031,758,-30}, {4031,775,-30}, {4031,1030,-30}, 
    {4031,1111,-30}, {4031,1173,-30}, {4031,1409,-80}, {4031,1630,-10}, 
    {4031,1631,-10}, {4031,1645,-10}, {4031,1655,-10}, {4031,1662,-10}, 
    {4031,2794,-25}, {4031,2795,-25}, {4031,2809,-25}, {4031,2819,-25}, 
    {4031,2826,-25}, {4031,2899,-25}, {4031,2907,-25}, {4031,2995,-80}, 
    {4034,734,-30}, {4034,736,-30}, {4034,758,-30}, {4034,775,-30}, 
    {4034,1030,-30}, {4034,1111,-30}, {4034,1173,-30}, {4034,1409,-80}, 
    {4034,1630,-10}, {4034,1631,-10}, {4034,1645,-10}, {4034,1655,-10}, 
    {4034,1662,-10}, {4034,2794,-25}, {4034,2795,-25}, {4034,2809,-25}, 
    {4034,2819,-25}, {4034,2826,-25}, {4034,2899,-25}, {4034,2907,-25}, 
    {4034,2995,-80}, {4050,734,-30}, {4050,736,-30}, {4050,758,-30}, 
    {4050,775,-30}, {4050,1030,-30}, {4050,1111,-30}, {4050,1173,-30}, 
    {4050,1409,-80}, {4050,1630,-10}, {4050,1631,-10}, {4050,1645,-10}, 
    {4050,1655,-10}, {4050,1662,-10}, {4050,2794,-25}, {4050,2795,-25}, 
    {4050,2809,-25}, {4050,2819,-25}, {4050,2826,-25}, {4050,2899,-25}, 
    {4050,2907,-25}, {4050,2995,-80}, {4132,1630,10}, {4132,1631,10}, 
    {4132,1645,10}, {4132,1655,10}, {4132,1662,10}, {4153,1630,10}, 
    {4153,1631,10}, {4153,1645,10}, {4153,1655,10}, {4153,1662,10}, 
    {NOGLYPH,NOGLYPH,0}
};
static const ligature helvetica_boldoblique_ligs[] = {
    {1787,2132,1807}, {1787,2416,1850}, 
    {NOGLYPH,NOGLYPH,NOGLYPH}
};
static const kern_pair courier_kerns[] = {
    
    {NOGLYPH,NOGLYPH,0}
};
static const ligature courier_ligs[] = {
    {1787,2132,1807}, {1787,2416,1850}, 
    {NOGLYPH,NOGLYPH,NOGLYPH}
};
static const kern_pair courier_oblique_kerns[] = {
    
    {NOGLYPH,NOGLYPH,0}
};
static const ligature courier_oblique_ligs[] = {
    {1787,2132,1807}, {1787,2416,1850}, 
    {NOGLYPH,NOGLYPH,NOGLYPH}
};
static const kern_pair courier_bold_kerns[] = {
    
    {NOGLYPH,NOGLYPH,0}
};
static const ligature courier_bold_ligs[] = {
    {1787,2132,1807}, {1787,2416,1850}, 
    {NOGLYPH,NOGLYPH,NOGLYPH}
};
static const kern_pair courier_boldoblique_kerns[] = {
    
    {NOGLYPH,NOGLYPH,0}
};
static const ligature courier_boldoblique_ligs[] = {
    {1787,2132,1807}, {1787,2416,1850}, 
    {NOGLYPH,NOGLYPH,NOGLYPH}
};

static const struct ps_std_font_data {
    char const *name;
    kern_pair const *kerns;
    ligature const *ligs;
    int widths[lenof(ps_std_glyphs)-1];
} ps_std_fonts[] = {
    { "Times-Roman",
	times_roman_kerns, times_roman_ligs, {
	722, 667, 667, 722, 611, 556, 722, 722, 333, 389, 722, 611, 889, 
	722, 722, 556, 722, 667, 556, 611, 722, 722, 944, 722, 722, 611, 
	444, 500, 444, 500, 444, 333, 500, 500, 278, 278, 500, 278, 778, 
	500, 500, 500, 500, 333, 389, 278, 500, 500, 722, 500, 500, 444, 
	889, 722, 722, 722, 722, 722, 722, 667, 611, 611, 611, 611, 722, 
	333, 333, 333, 333, 611, 722, 889, 722, 722, 722, 722, 722, 722, 
	556, 556, 722, 722, 722, 722, 722, 722, 611, 667, 444, 444, 444, 
	444, 444, 444, 444, 444, 444, 444, 444, 500, 278, 278, 278, 278, 
	278, 500, 722, 500, 500, 500, 500, 500, 500, 389, 500, 500, 500, 
	500, 500, 500, 500, 444, 333, 778, 469, 541, 500, 921, 278, 200, 
	480, 480, 333, 333, 333, 200, 350, 333, 333, 500, 333, 278, 250, 
	760, 500, 500, 500, 400, 333, 564, 500, 333, 278, 500, 1000, 1000, 
	500, 564, 333, 333, 556, 500, 556, 500, 500, 167, 500, 333, 564, 
	500, 500, 333, 333, 333, 333, 564, 564, 333, 564, 500, 564, 500, 
	500, 333, 500, 750, 750, 300, 276, 310, 453, 333, 333, 833, 250, 
	250, 1000, 564, 564, 444, 444, 408, 444, 444, 444, 333, 333, 333, 
	180, 760, 333, 500, 278, 500, 500, 278, 250, 500, 500, 750, 300, 
	333, 980, 500, 300, 500, 500, 500, 
    }},
    { "Times-Italic",
	times_italic_kerns, times_italic_ligs, {
	611, 611, 667, 722, 611, 611, 722, 722, 333, 444, 667, 556, 833, 
	667, 722, 611, 722, 611, 500, 556, 722, 611, 833, 611, 556, 556, 
	500, 500, 444, 500, 444, 278, 500, 500, 278, 278, 444, 278, 722, 
	500, 500, 500, 500, 389, 389, 278, 500, 444, 667, 444, 444, 389, 
	889, 611, 611, 611, 611, 611, 611, 667, 611, 611, 611, 611, 722, 
	333, 333, 333, 333, 556, 667, 944, 722, 722, 722, 722, 722, 722, 
	500, 611, 722, 722, 722, 722, 556, 556, 556, 667, 500, 500, 500, 
	500, 500, 500, 444, 444, 444, 444, 444, 500, 278, 278, 278, 278, 
	278, 500, 667, 500, 500, 500, 500, 500, 500, 389, 500, 500, 500, 
	500, 500, 444, 444, 389, 333, 778, 422, 541, 500, 920, 278, 275, 
	400, 400, 389, 389, 333, 275, 350, 333, 333, 500, 333, 333, 250, 
	760, 500, 500, 500, 400, 333, 675, 500, 333, 278, 500, 889, 889, 
	500, 675, 333, 389, 500, 500, 500, 500, 500, 167, 500, 333, 675, 
	500, 500, 333, 333, 333, 333, 675, 675, 333, 675, 500, 675, 500, 
	500, 333, 500, 750, 750, 300, 276, 310, 523, 333, 333, 833, 250, 
	250, 1000, 675, 675, 500, 500, 420, 556, 556, 556, 333, 333, 333, 
	214, 760, 333, 500, 333, 500, 500, 278, 250, 500, 500, 750, 300, 
	333, 980, 500, 300, 500, 500, 500, 
    }},
    { "Times-Bold",
	times_bold_kerns, times_bold_ligs, {
	722, 667, 722, 722, 667, 611, 778, 778, 389, 500, 778, 667, 944, 
	722, 778, 611, 778, 722, 556, 667, 722, 722, 1000, 722, 722, 667, 
	500, 556, 444, 556, 444, 333, 500, 556, 278, 333, 556, 278, 833, 
	556, 500, 556, 556, 444, 389, 333, 556, 500, 722, 500, 500, 444, 
	1000, 722, 722, 722, 722, 722, 722, 722, 667, 667, 667, 667, 722, 
	389, 389, 389, 389, 667, 722, 1000, 778, 778, 778, 778, 778, 778, 
	556, 611, 722, 722, 722, 722, 722, 722, 667, 722, 500, 500, 500, 
	500, 500, 500, 444, 444, 444, 444, 444, 500, 278, 278, 278, 278, 
	278, 556, 722, 500, 500, 500, 500, 500, 500, 389, 556, 556, 556, 
	556, 556, 500, 500, 444, 333, 833, 581, 520, 500, 930, 278, 220, 
	394, 394, 333, 333, 333, 220, 350, 333, 333, 500, 333, 333, 250, 
	747, 500, 500, 500, 400, 333, 570, 500, 333, 278, 500, 1000, 1000, 
	500, 570, 333, 333, 556, 500, 556, 500, 500, 167, 556, 333, 570, 
	500, 500, 333, 333, 333, 333, 570, 570, 333, 570, 556, 570, 500, 
	500, 333, 500, 750, 750, 300, 300, 330, 540, 333, 333, 1000, 250, 
	250, 1000, 570, 570, 500, 500, 555, 500, 500, 500, 333, 333, 333, 
	278, 747, 333, 500, 333, 500, 500, 278, 250, 500, 500, 750, 300, 
	333, 1000, 500, 300, 500, 500, 500, 
    }},
    { "Times-BoldItalic",
	times_bolditalic_kerns, times_bolditalic_ligs, {
	667, 667, 667, 722, 667, 667, 722, 778, 389, 500, 667, 611, 889, 
	722, 722, 611, 722, 667, 556, 611, 722, 667, 889, 667, 611, 611, 
	500, 500, 444, 500, 444, 333, 500, 556, 278, 278, 500, 278, 778, 
	556, 500, 500, 500, 389, 389, 278, 556, 444, 667, 500, 444, 389, 
	944, 667, 667, 667, 667, 667, 667, 667, 667, 667, 667, 667, 722, 
	389, 389, 389, 389, 611, 722, 944, 722, 722, 722, 722, 722, 722, 
	556, 611, 722, 722, 722, 722, 611, 611, 611, 722, 500, 500, 500, 
	500, 500, 500, 444, 444, 444, 444, 444, 500, 278, 278, 278, 278, 
	278, 556, 722, 500, 500, 500, 500, 500, 500, 389, 500, 556, 556, 
	556, 556, 444, 444, 389, 333, 778, 570, 570, 500, 832, 278, 220, 
	348, 348, 333, 333, 333, 220, 350, 333, 333, 500, 333, 333, 250, 
	747, 500, 500, 500, 400, 333, 570, 500, 333, 278, 500, 1000, 1000, 
	500, 570, 389, 389, 556, 500, 556, 500, 500, 167, 500, 333, 570, 
	500, 500, 333, 333, 333, 333, 570, 606, 333, 606, 576, 570, 500, 
	500, 333, 500, 750, 750, 300, 266, 300, 500, 333, 333, 833, 250, 
	250, 1000, 570, 570, 500, 500, 555, 500, 500, 500, 333, 333, 333, 
	278, 747, 333, 500, 333, 500, 500, 278, 250, 500, 500, 750, 300, 
	333, 1000, 500, 300, 500, 500, 500, 
    }},
    { "Helvetica",
	helvetica_kerns, helvetica_ligs, {
	667, 667, 722, 722, 667, 611, 778, 722, 278, 500, 667, 556, 833, 
	722, 778, 667, 778, 722, 667, 611, 722, 667, 944, 667, 667, 611, 
	556, 556, 500, 556, 556, 278, 556, 556, 222, 222, 500, 222, 833, 
	556, 556, 556, 556, 333, 500, 278, 556, 500, 722, 500, 500, 500, 
	1000, 667, 667, 667, 667, 667, 667, 722, 667, 667, 667, 667, 722, 
	278, 278, 278, 278, 556, 722, 1000, 778, 778, 778, 778, 778, 778, 
	667, 667, 722, 722, 722, 722, 667, 667, 611, 889, 556, 556, 556, 
	556, 556, 556, 500, 556, 556, 556, 556, 556, 278, 278, 278, 278, 
	222, 556, 944, 556, 556, 556, 556, 611, 556, 500, 556, 556, 556, 
	556, 556, 500, 500, 500, 333, 667, 469, 584, 389, 1015, 278, 260, 
	334, 334, 278, 278, 333, 260, 350, 333, 333, 556, 333, 278, 278, 
	737, 556, 556, 556, 400, 333, 584, 556, 333, 278, 556, 1000, 1000, 
	556, 584, 278, 333, 500, 556, 500, 556, 556, 167, 611, 333, 584, 
	556, 556, 333, 333, 333, 333, 584, 584, 333, 584, 556, 584, 556, 
	556, 333, 556, 834, 834, 333, 370, 365, 537, 333, 333, 889, 278, 
	278, 1000, 584, 584, 556, 611, 355, 333, 333, 333, 222, 222, 222, 
	191, 737, 333, 556, 278, 556, 556, 278, 278, 556, 556, 834, 333, 
	333, 1000, 556, 333, 556, 556, 556, 
    }},
    { "Helvetica-Oblique",
	helvetica_oblique_kerns, helvetica_oblique_ligs, {
	667, 667, 722, 722, 667, 611, 778, 722, 278, 500, 667, 556, 833, 
	722, 778, 667, 778, 722, 667, 611, 722, 667, 944, 667, 667, 611, 
	556, 556, 500, 556, 556, 278, 556, 556, 222, 222, 500, 222, 833, 
	556, 556, 556, 556, 333, 500, 278, 556, 500, 722, 500, 500, 500, 
	1000, 667, 667, 667, 667, 667, 667, 722, 667, 667, 667, 667, 722, 
	278, 278, 278, 278, 556, 722, 1000, 778, 778, 778, 778, 778, 778, 
	667, 667, 722, 722, 722, 722, 667, 667, 611, 889, 556, 556, 556, 
	556, 556, 556, 500, 556, 556, 556, 556, 556, 278, 278, 278, 278, 
	222, 556, 944, 556, 556, 556, 556, 611, 556, 500, 556, 556, 556, 
	556, 556, 500, 500, 500, 333, 667, 469, 584, 389, 1015, 278, 260, 
	334, 334, 278, 278, 333, 260, 350, 333, 333, 556, 333, 278, 278, 
	737, 556, 556, 556, 400, 333, 584, 556, 333, 278, 556, 1000, 1000, 
	556, 584, 278, 333, 500, 556, 500, 556, 556, 167, 611, 333, 584, 
	556, 556, 333, 333, 333, 333, 584, 584, 333, 584, 556, 584, 556, 
	556, 333, 556, 834, 834, 333, 370, 365, 537, 333, 333, 889, 278, 
	278, 1000, 584, 584, 556, 611, 355, 333, 333, 333, 222, 222, 222, 
	191, 737, 333, 556, 278, 556, 556, 278, 278, 556, 556, 834, 333, 
	333, 1000, 556, 333, 556, 556, 556, 
    }},
    { "Helvetica-Bold",
	helvetica_bold_kerns, helvetica_bold_ligs, {
	722, 722, 722, 722, 667, 611, 778, 722, 278, 556, 722, 611, 833, 
	722, 778, 667, 778, 722, 667, 611, 722, 667, 944, 667, 667, 611, 
	556, 611, 556, 611, 556, 333, 611, 611, 278, 278, 556, 278, 889, 
	611, 611, 611, 611, 389, 556, 333, 611, 556, 778, 556, 556, 500, 
	1000, 722, 722, 722, 722, 722, 722, 722, 667, 667, 667, 667, 722, 
	278, 278, 278, 278, 611, 722, 1000, 778, 778, 778, 778, 778, 778, 
	667, 667, 722, 722, 722, 722, 667, 667, 611, 889, 556, 556, 556, 
	556, 556, 556, 556, 556, 556, 556, 556, 611, 278, 278, 278, 278, 
	278, 611, 944, 611, 611, 611, 611, 611, 611, 556, 611, 611, 611, 
	611, 611, 556, 556, 500, 333, 722, 584, 584, 389, 975, 278, 280, 
	389, 389, 333, 333, 333, 280, 350, 333, 333, 556, 333, 333, 278, 
	737, 556, 556, 556, 400, 333, 584, 556, 333, 278, 556, 1000, 1000, 
	556, 584, 333, 333, 611, 556, 611, 556, 556, 167, 611, 333, 584, 
	556, 556, 333, 333, 333, 333, 584, 584, 333, 584, 611, 584, 556, 
	556, 333, 556, 834, 834, 333, 370, 365, 556, 333, 333, 889, 278, 
	278, 1000, 584, 584, 611, 611, 474, 500, 500, 500, 278, 278, 278, 
	238, 737, 333, 556, 333, 556, 556, 278, 278, 556, 556, 834, 333, 
	333, 1000, 556, 333, 556, 556, 556, 
    }},
    { "Helvetica-BoldOblique",
	helvetica_boldoblique_kerns, helvetica_boldoblique_ligs, {
	722, 722, 722, 722, 667, 611, 778, 722, 278, 556, 722, 611, 833, 
	722, 778, 667, 778, 722, 667, 611, 722, 667, 944, 667, 667, 611, 
	556, 611, 556, 611, 556, 333, 611, 611, 278, 278, 556, 278, 889, 
	611, 611, 611, 611, 389, 556, 333, 611, 556, 778, 556, 556, 500, 
	1000, 722, 722, 722, 722, 722, 722, 722, 667, 667, 667, 667, 722, 
	278, 278, 278, 278, 611, 722, 1000, 778, 778, 778, 778, 778, 778, 
	667, 667, 722, 722, 722, 722, 667, 667, 611, 889, 556, 556, 556, 
	556, 556, 556, 556, 556, 556, 556, 556, 611, 278, 278, 278, 278, 
	278, 611, 944, 611, 611, 611, 611, 611, 611, 556, 611, 611, 611, 
	611, 611, 556, 556, 500, 333, 722, 584, 584, 389, 975, 278, 280, 
	389, 389, 333, 333, 333, 280, 350, 333, 333, 556, 333, 333, 278, 
	737, 556, 556, 556, 400, 333, 584, 556, 333, 278, 556, 1000, 1000, 
	556, 584, 333, 333, 611, 556, 611, 556, 556, 167, 611, 333, 584, 
	556, 556, 333, 333, 333, 333, 584, 584, 333, 584, 611, 584, 556, 
	556, 333, 556, 834, 834, 333, 370, 365, 556, 333, 333, 889, 278, 
	278, 1000, 584, 584, 611, 611, 474, 500, 500, 500, 278, 278, 278, 
	238, 737, 333, 556, 333, 556, 556, 278, 278, 556, 556, 834, 333, 
	333, 1000, 556, 333, 556, 556, 556, 
    }},
    { "Courier",
	courier_kerns, courier_ligs, {
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 
    }},
    { "Courier-Oblique",
	courier_oblique_kerns, courier_oblique_ligs, {
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 
    }},
    { "Courier-Bold",
	courier_bold_kerns, courier_bold_ligs, {
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 
    }},
    { "Courier-BoldOblique",
	courier_boldoblique_kerns, courier_boldoblique_ligs, {
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 
    }},
};

void init_std_fonts(void) {
    int i, j;
    ligature const *lig;
    kern_pair const *kern;
    static int done = FALSE;

    if (done) return;
    for (i = 0; i < (int)lenof(ps_std_fonts); i++) {
	font_info *fi = snew(font_info);
	fi->fontfile = NULL;
	fi->name = ps_std_fonts[i].name;
	fi->widths = newtree234(width_cmp);
	for (j = 0; j < (int)lenof(fi->bmp); j++)
	    fi->bmp[j] = NOGLYPH;
	for (j = 0; j < (int)lenof(ps_std_glyphs) - 1; j++) {
	    glyph_width *w = snew(glyph_width);
	    wchar_t ucs;
	    w->glyph = glyph_intern(ps_std_glyphs[j]);
	    w->width = ps_std_fonts[i].widths[j];
	    add234(fi->widths, w);
	    ucs = ps_glyph_to_unicode(w->glyph);
	    assert(ucs != 0xFFFF);
	    fi->bmp[ucs] = w->glyph;
	}
	fi->kerns = newtree234(kern_cmp);
	for (kern = ps_std_fonts[i].kerns; kern->left != NOGLYPH; kern++)
	    add234(fi->kerns, (void *)kern);
	fi->ligs = newtree234(lig_cmp);
	for (lig = ps_std_fonts[i].ligs; lig->left != NOGLYPH; lig++)
	    add234(fi->ligs, (void *)lig);
	fi->next = all_fonts;
	all_fonts = fi;
    }
    done = TRUE;
}

const int *ps_std_font_widths(char const *fontname)
{
    int i;

    for (i = 0; i < (int)lenof(ps_std_fonts); i++)
	if (!strcmp(ps_std_fonts[i].name, fontname))
	    return ps_std_fonts[i].widths;

    return NULL;
}

const kern_pair *ps_std_font_kerns(char const *fontname)
{
    int i;

    for (i = 0; i < (int)lenof(ps_std_fonts); i++)
	if (!strcmp(ps_std_fonts[i].name, fontname))
	    return ps_std_fonts[i].kerns;

    return NULL;
}
