/*******************************************************************************
 * Part of "Intel(R) Active Management Technology (Intel(R) AMT)
 *                   User Notification Service (UNS)"
 *
 * Copyright (c) 2007 Intel Corp.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification.
 * 2. Redistributions in binary form must reproduce at minimum a disclaimer
 *    substantially similar to the "NO WARRANTY" disclaimer below
 *    ("Disclaimer") and any redistribution must be conditioned upon
 *    including a substantially similar Disclaimer requirement for further
 *    binary redistribution.
 * 3. Neither the names of the above-listed copyright holders nor the names
 *    of any contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * Alternatively, this software may be distributed under the terms of the
 * GNU General Public License ("GPL") version 2 as published by the Free
 * Software Foundation.
 *
 * NO WARRANTY
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * HOLDERS OR CONTRIBUTORS BE LIABLE FOR SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGES.
 *******************************************************************************/

//*****************************************************************************
//
// Class:	UNSWSManClient
// Date created:25 Dec 2006
// Description:	This class is used to subscribe the User Notification Service
//		for events at the local AMT machine.
//
//*****************************************************************************

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include "UNSWSManClient.h"
#include "UNSEventLogger.h"
#include "UNSCommonDefs.h"
#include "UNSStatus.h"
#include "openssl/rand.h"

using namespace CimClassNamespace;
const int UNSWSManClient::GENERATED_USERNAME_LEN = 8;
const int UNSWSManClient::GENERATED_PASSWORD_LEN = 8;
const std::string UNSWSManClient::DEFAULT_USER = "$$uns";
const std::string UNSWSManClient::DEFAULT_PASS = "$$uns";

// Constants for the common use
static const unsigned char UNS_POLICY_ID= 0x86;	// Hard-coded policyID for UNS
static unsigned short UNS_PORT_PROTOCOL = 2;
static unsigned short UNS_INFO_FORMAT = 200;

//************************************************************************
// Default Constructor.
// Params:	certName - Certificate for SSL.
//************************************************************************
UNSWSManClient::UNSWSManClient() : BaseWSManClient(CRED_REG_INFO::UNS,DEFAULT_USER,DEFAULT_PASS)
{
  m_subscribed = false; // Not subscribed.
}

//************************************************************************
// Name			: UNSWSManClient
// Description	: Destructor.
//************************************************************************
UNSWSManClient::~UNSWSManClient(void)
{
}


//************************************************************************
// Name		: DeletePreviousSubscriptions.
// Description	: Enumerate the UNS PolicyID subscriptions and cencel them
//                all.
//************************************************************************
unsigned long UNSWSManClient::DeletePreviousSubscriptions()
{
  printf("\nDeleting Previous Subscriptions...\n");
  for(int i = 0; i < 2; i++)
    {
      if (!m_endpoint)
	SetEndpoint(i == 0 ? false : true);

      printf("Sending wsman enumeration command (%s)\n",endpoint.c_str());
      vector<AMT_SOAPEventSubscriber > subscriptions;
      try {
	AMT_SOAPEventSubscriber::Enumerate(subscriptions, NULL, m_client);
      }
      catch (GeneralWsmanException& e)
	{
	  printf("\nError: failed while calling AMT_SOAPEventSubscriber::Enumerate routine\n");
	  printf("GeneralWsmanException:\n");
	  printf("%s\n", e.what());
	  m_endpoint = false;
	  continue;
	}
      catch (exception& e)
	{
	  printf("\nError: failed while calling AMT_SOAPEventSubscriber::Enumerate routine\n");
	  printf("%s\n", e.what());
	  m_endpoint = false;
	  continue;
	}

      printf("Wsman send successfully.\n");
      m_endpoint = true;

      for (int i = 0; i < (int)subscriptions.size(); i++)
	{
	  if (subscriptions[i].PolicyID == UNS_POLICY_ID) {
	    printf ("Deleting old subscription ID=%d\n",subscriptions[i].PolicyID);
	    try {
	      subscriptions[i].Delete(m_client);
	    }
	    catch (GeneralWsmanException& e)
	      {
		printf("\nError: failed while calling AMT_SOAPEventSubscriber::Delete routine\n");
		printf("GeneralWsmanException:\n");
		printf("%s\n", e.what());
	      }
	    catch (exception& e)
	      {
		printf("\nError: failed while calling AMT_SOAPEventSubscriber::Delete routine\n");
	      }
	  }
	}
      return UNS_STATUS_SUCCESS;
    }

  return UNS_STATUS_SUBSCRIPTION_ERROR;
}

//************************************************************************
// Name			: SubscribeForGeneralAlert.
// Description	: Subscribe for alert at the local AMT machine.
//************************************************************************
unsigned long UNSWSManClient::SubscribeForGeneralAlert(int port)
{
  // Do nothing if already subscribed.
  if (m_subscribed) {
    printf("UNS already subscribed\n");
    return UNS_STATUS_SUCCESS;
  }

  // First cencel all previous subscriptions
  if (DeletePreviousSubscriptions() != UNS_STATUS_SUCCESS)
    return UNS_STATUS_SUBSCRIPTION_ERROR;

  // Set address to localhost.
  string accessInfo = "http://localhost:";
  char strPort[16];
  sprintf(strPort,"%d",port);
  accessInfo.append(strPort);

  // Fill some dummy values required by the FW.
  string dummy = "n/a";
  m_Subscriber.CreationClassName			= dummy;
  m_Subscriber.Description				= dummy;
  m_Subscriber.DescriptionSpecified			= true;
  m_Subscriber.ElementName				= dummy;
  m_Subscriber.ElementNameSpecified			= true;
  m_Subscriber.Name						= dummy;
  m_Subscriber.SystemCreationClassName	= dummy;
  m_Subscriber.SystemName					= dummy;
  m_Subscriber.PortInfo					= dummy;

  m_Subscriber.AccessInfo				= accessInfo;
  m_Subscriber.InfoFormat				= UNS_INFO_FORMAT;
  m_Subscriber.PortProtocol			= UNS_PORT_PROTOCOL;
  m_Subscriber.PortProtocolSpecified	= true;
  m_Subscriber.PolicyID				= UNS_POLICY_ID;

#ifdef NO_AUTH
#else
  m_Subscriber.UserName = m_SubscribeCred.username;
  m_Subscriber.UserNameSpecified = true;
  m_Subscriber.Password = m_SubscribeCred.password;
  m_Subscriber.PasswordSpecified = true;
#endif

  // Set the authentication options.
#ifdef BASIC_AUTH
  m_Subscriber.AlertAuthenticationOptions.push_back(2);	// Allow Basic authentication.
#endif
#ifdef DIGEST_AUTH
  m_Subscriber.AlertAuthenticationOptions.push_back(1);	// Allow Diget authentication.
#endif


  // Send request, try both HTTP and HTTPS
  for(int i = 0; i < 2; i++)
    {
      if (!m_endpoint)
	SetEndpoint(i == 0 ? false : true);

      printf ("\nCreating a new AMT_SOAPEventSubscriber (%s).\n", endpoint.c_str());
      try {
	m_Subscriber.Create(m_client);
      }
      catch (GeneralWsmanException& e)
	{
	  printf("\nError: failed while calling AMT_SOAPEventSubscriber::Create routine\n");
	  printf("GeneralWsmanException:\n");
	  printf("%s\n", e.what());
	  m_endpoint = false;
	  continue;
	}
      catch (exception& e)
	{
	  printf("\nError: failed while calling AMT_SOAPEventSubscriber::Create routine\n");
	  printf("%s\n", e.what());
	  m_endpoint = false;
	  continue;
	}
      m_endpoint    = true;
      m_subscribed  = true;
      printf ("AMT_SoapAlertTarget instance created successfuly\n");
      return UNS_STATUS_SUCCESS;
    }

  m_subscribed = false;
  return UNS_STATUS_SUBSCRIPTION_ERROR;
}

//************************************************************************
// Name			: CancelAlertSubscription.
// Description	: Cencel the subscription at the local AMT machine.
//************************************************************************
unsigned long UNSWSManClient::CancelAlertSubscription(unsigned int *handle)
{
  // If not subscribed return success
  if (m_subscribed == false)
    return UNS_STATUS_SUCCESS;

  for(int i = 0; i < 2; i++)
    {
      if (!m_endpoint)
	SetEndpoint(i == 0 ? false : true);

      printf ("\nDeleting AMT_SoapAlertTarget instance (%s).\n", endpoint.c_str());
      try {
	m_Subscriber.Delete(m_client);
      }
      catch (GeneralWsmanException& e)
	{
	  printf("\nError: failed while calling AMT_SOAPEventSubscriber::Delete routine\n");
	  printf("GeneralWsmanException:\n");
	  printf("%s\n", e.what());
	  m_endpoint = false;
	  continue;
	}
      catch (exception& e)
	{
	  printf("\nError: failed while calling AMT_SOAPEventSubscriber::Delete routine\n");
	  printf("%s\n", e.what());
	  m_endpoint = false;
	  continue;
	}
      m_endpoint = true;
      printf ("\nInstance deleted successfuly.\n");
      return UNS_STATUS_SUCCESS;
    }
  return UNS_STATUS_CENCEL_ERROR;;
}

bool UNSWSManClient::GenRandAsciiBuf(unsigned char *pBuf, int len)
{
  //Generate random data
  RAND_METHOD *rmeth;

  rmeth = RAND_SSLeay();

  if (rmeth == NULL)
    return false;

  if (!RAND_bytes(pBuf, len-1))
    return false;

  RAND_cleanup();

  const unsigned char *end = pBuf + len - 1;
  //Turn it into all alphabetic characters
  unsigned char *p;
  for (p = pBuf; p != end; p++)
    *p = (*p % 26) + 'A';

  //add trailing zero
  *p = '\0';
  return true;
}


//************************************************************************
// Name			: GenerateCredentials.
// Description	: Generate random userid and passwd.
//************************************************************************
bool UNSWSManClient::GenerateCredentials()
{
  unsigned char user[GENERATED_USERNAME_LEN+1];
  unsigned char pass[GENERATED_PASSWORD_LEN+1];

  if (!GenRandAsciiBuf(user, GENERATED_USERNAME_LEN+1))
    return false;

  if (!GenRandAsciiBuf(pass, GENERATED_PASSWORD_LEN+1))
    return false;

  m_SubscribeCred.username = (char*)user;
  m_SubscribeCred.password = (char*)pass;

  return true;
}
