/* SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use, copy,
 * modify, merge, publish, distribute, sublicense, and/or sell copies
 * of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 * Copyright:
 *   2020      Evan Nemerson <evan@nemerson.com>
 *   2020      Himanshi Mathur <himanshi18037@iiitd.ac.in>
 *   2020      Hidayat Khan <huk2209@gmail.com>
 */

#define SIMDE_TEST_X86_AVX512_INSN cvt

#include <test/x86/avx512/test-avx512.h>
#include <simde/x86/avx512/set.h>
#include <simde/x86/avx512/cvt.h>

static int
test_simde_mm512_cvtepi16_epi8(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512i a;
    simde__m256i r;
  } test_vec[8] = {
    { simde_mm512_set_epi16(INT16_C( 14423), INT16_C(  3775), INT16_C( 16156), INT16_C( 17811),
                            INT16_C(-14881), INT16_C(-30283), INT16_C( 27295), INT16_C(-12290),
                            INT16_C( 12394), INT16_C( 32764), INT16_C(  8681), INT16_C( 21255),
                            INT16_C(-21785), INT16_C(-24065), INT16_C(-28005), INT16_C( 15206),
                            INT16_C(  6131), INT16_C(-29323), INT16_C( -9530), INT16_C( -6655),
                            INT16_C( 14785), INT16_C( -9158), INT16_C(  7009), INT16_C(  4834),
                            INT16_C(-15579), INT16_C(  5296), INT16_C( 20054), INT16_C( 12832),
                            INT16_C( 15724), INT16_C(  5918), INT16_C( 25398), INT16_C( 14084)),
      simde_mm256_set_epi8(INT8_C(  87), INT8_C( -65), INT8_C(  28), INT8_C(-109),
                           INT8_C( -33), INT8_C( -75), INT8_C( -97), INT8_C(  -2),
                           INT8_C( 106), INT8_C(  -4), INT8_C( -23), INT8_C(   7),
                           INT8_C( -25), INT8_C(  -1), INT8_C(-101), INT8_C( 102),
                           INT8_C( -13), INT8_C( 117), INT8_C( -58), INT8_C(   1),
                           INT8_C( -63), INT8_C(  58), INT8_C(  97), INT8_C( -30),
                           INT8_C(  37), INT8_C( -80), INT8_C(  86), INT8_C(  32),
                           INT8_C( 108), INT8_C(  30), INT8_C(  54), INT8_C(   4)) },
    { simde_mm512_set_epi16(INT16_C(  8455), INT16_C(  1140), INT16_C(-23383), INT16_C( 22825),
                            INT16_C(-21438), INT16_C(  8713), INT16_C(-25940), INT16_C(-31180),
                            INT16_C(-13214), INT16_C( 10200), INT16_C(-21253), INT16_C(  2612),
                            INT16_C(-27891), INT16_C( 14031), INT16_C( -9014), INT16_C( 10287),
                            INT16_C(-11660), INT16_C( 26858), INT16_C(-19518), INT16_C(  2472),
                            INT16_C( 27637), INT16_C( 14857), INT16_C( 30034), INT16_C(-24153),
                            INT16_C( 31935), INT16_C( -6397), INT16_C( -2502), INT16_C( 31062),
                            INT16_C( 30236), INT16_C(  5156), INT16_C( 18439), INT16_C(-13074)),
      simde_mm256_set_epi8(INT8_C(   7), INT8_C( 116), INT8_C( -87), INT8_C(  41),
                           INT8_C(  66), INT8_C(   9), INT8_C( -84), INT8_C(  52),
                           INT8_C(  98), INT8_C( -40), INT8_C(  -5), INT8_C(  52),
                           INT8_C(  13), INT8_C( -49), INT8_C( -54), INT8_C(  47),
                           INT8_C( 116), INT8_C( -22), INT8_C( -62), INT8_C( -88),
                           INT8_C( -11), INT8_C(   9), INT8_C(  82), INT8_C( -89),
                           INT8_C( -65), INT8_C(   3), INT8_C(  58), INT8_C(  86),
                           INT8_C(  28), INT8_C(  36), INT8_C(   7), INT8_C( -18)) },
    { simde_mm512_set_epi16(INT16_C( 18175), INT16_C( -3760), INT16_C( 10318), INT16_C(-31849),
                            INT16_C(-32429), INT16_C(-26500), INT16_C( 24084), INT16_C(-23946),
                            INT16_C(  2525), INT16_C(  2478), INT16_C(-15141), INT16_C(-27410),
                            INT16_C( 30961), INT16_C(-31554), INT16_C( -9533), INT16_C(-20012),
                            INT16_C(-21820), INT16_C( 11767), INT16_C(-17849), INT16_C( 24518),
                            INT16_C(-22206), INT16_C(-24996), INT16_C(-19566), INT16_C( 17826),
                            INT16_C( 25765), INT16_C( 29123), INT16_C( 28065), INT16_C(  1432),
                            INT16_C(-24949), INT16_C( 30580), INT16_C( 20499), INT16_C(-29164)),
      simde_mm256_set_epi8(INT8_C(  -1), INT8_C(  80), INT8_C(  78), INT8_C(-105),
                           INT8_C(  83), INT8_C( 124), INT8_C(  20), INT8_C( 118),
                           INT8_C( -35), INT8_C( -82), INT8_C( -37), INT8_C( -18),
                           INT8_C( -15), INT8_C( -66), INT8_C( -61), INT8_C( -44),
                           INT8_C( -60), INT8_C(  -9), INT8_C(  71), INT8_C( -58),
                           INT8_C(  66), INT8_C(  92), INT8_C(-110), INT8_C( -94),
                           INT8_C( -91), INT8_C( -61), INT8_C( -95), INT8_C(-104),
                           INT8_C(-117), INT8_C( 116), INT8_C(  19), INT8_C(  20)) },
    { simde_mm512_set_epi16(INT16_C( 10816), INT16_C( 16713), INT16_C( 29707), INT16_C( 15186),
                            INT16_C( 31860), INT16_C(-28520), INT16_C( 18947), INT16_C(-27460),
                            INT16_C( 10883), INT16_C(   310), INT16_C(  8277), INT16_C(-28768),
                            INT16_C( -4553), INT16_C( 23273), INT16_C(-27696), INT16_C(-20678),
                            INT16_C( 13089), INT16_C( -6620), INT16_C( 31575), INT16_C(-20169),
                            INT16_C( 14440), INT16_C( -9264), INT16_C(-26919), INT16_C(-25720),
                            INT16_C(-18371), INT16_C( 25765), INT16_C(-13162), INT16_C(-16808),
                            INT16_C(  5695), INT16_C(-25080), INT16_C( 19142), INT16_C(  3825)),
      simde_mm256_set_epi8(INT8_C(  64), INT8_C(  73), INT8_C(  11), INT8_C(  82),
                           INT8_C( 116), INT8_C(-104), INT8_C(   3), INT8_C( -68),
                           INT8_C(-125), INT8_C(  54), INT8_C(  85), INT8_C( -96),
                           INT8_C(  55), INT8_C( -23), INT8_C( -48), INT8_C(  58),
                           INT8_C(  33), INT8_C(  36), INT8_C(  87), INT8_C(  55),
                           INT8_C( 104), INT8_C( -48), INT8_C( -39), INT8_C(-120),
                           INT8_C(  61), INT8_C( -91), INT8_C(-106), INT8_C(  88),
                           INT8_C(  63), INT8_C(   8), INT8_C( -58), INT8_C( -15)) },
    { simde_mm512_set_epi16(INT16_C(  5079), INT16_C(-24746), INT16_C( 23487), INT16_C(-22087),
                            INT16_C( -8346), INT16_C( 29848), INT16_C( 14241), INT16_C( 18254),
                            INT16_C( -3124), INT16_C(-16186), INT16_C(-13364), INT16_C( 10652),
                            INT16_C( 31028), INT16_C( 21346), INT16_C(  1443), INT16_C(-20222),
                            INT16_C(-17028), INT16_C(-21899), INT16_C( 18933), INT16_C(  6935),
                            INT16_C( 24619), INT16_C(  1737), INT16_C( 12596), INT16_C( 31606),
                            INT16_C(-32691), INT16_C( 11392), INT16_C( 32126), INT16_C(-32712),
                            INT16_C( 20927), INT16_C(-27859), INT16_C( 22640), INT16_C(  8969)),
      simde_mm256_set_epi8(INT8_C( -41), INT8_C(  86), INT8_C( -65), INT8_C( -71),
                           INT8_C( 102), INT8_C(-104), INT8_C( -95), INT8_C(  78),
                           INT8_C( -52), INT8_C( -58), INT8_C( -52), INT8_C(-100),
                           INT8_C(  52), INT8_C(  98), INT8_C( -93), INT8_C(   2),
                           INT8_C( 124), INT8_C( 117), INT8_C( -11), INT8_C(  23),
                           INT8_C(  43), INT8_C( -55), INT8_C(  52), INT8_C( 118),
                           INT8_C(  77), INT8_C(-128), INT8_C( 126), INT8_C(  56),
                           INT8_C( -65), INT8_C(  45), INT8_C( 112), INT8_C(   9)) },
    { simde_mm512_set_epi16(INT16_C(  6901), INT16_C(-23435), INT16_C(-26040), INT16_C(-11295),
                            INT16_C(   623), INT16_C(-23058), INT16_C( 17549), INT16_C(-23291),
                            INT16_C( 17215), INT16_C( -4892), INT16_C(  -849), INT16_C( 21086),
                            INT16_C(-13056), INT16_C( 19549), INT16_C( 16492), INT16_C(-22767),
                            INT16_C(-24079), INT16_C(  6429), INT16_C( 15302), INT16_C( -9175),
                            INT16_C( 17671), INT16_C(-29856), INT16_C(-12718), INT16_C(-22914),
                            INT16_C(-19613), INT16_C( 14088), INT16_C(-10443), INT16_C( 31757),
                            INT16_C( 24994), INT16_C( 24174), INT16_C( -9596), INT16_C(-22481)),
      simde_mm256_set_epi8(INT8_C( -11), INT8_C( 117), INT8_C(  72), INT8_C( -31),
                           INT8_C( 111), INT8_C( -18), INT8_C(-115), INT8_C(   5),
                           INT8_C(  63), INT8_C( -28), INT8_C( -81), INT8_C(  94),
                           INT8_C(   0), INT8_C(  93), INT8_C( 108), INT8_C(  17),
                           INT8_C( -15), INT8_C(  29), INT8_C( -58), INT8_C(  41),
                           INT8_C(   7), INT8_C(  96), INT8_C(  82), INT8_C( 126),
                           INT8_C(  99), INT8_C(   8), INT8_C(  53), INT8_C(  13),
                           INT8_C( -94), INT8_C( 110), INT8_C(-124), INT8_C(  47)) },
    { simde_mm512_set_epi16(INT16_C( 15520), INT16_C( 15679), INT16_C(  8541), INT16_C(-20376),
                            INT16_C(  8861), INT16_C( 12926), INT16_C( 25712), INT16_C( -8433),
                            INT16_C( -7066), INT16_C(-23691), INT16_C(-20251), INT16_C( 18056),
                            INT16_C(  5498), INT16_C(-18751), INT16_C(-26321), INT16_C(  7918),
                            INT16_C(  1647), INT16_C( 21774), INT16_C(  5430), INT16_C(-19512),
                            INT16_C(-14894), INT16_C( 12466), INT16_C( -9612), INT16_C(-23130),
                            INT16_C( 18357), INT16_C( 32349), INT16_C(-25760), INT16_C( -6559),
                            INT16_C(-24198), INT16_C( 13614), INT16_C( 13473), INT16_C(-25578)),
      simde_mm256_set_epi8(INT8_C( -96), INT8_C(  63), INT8_C(  93), INT8_C( 104),
                           INT8_C( -99), INT8_C( 126), INT8_C( 112), INT8_C(  15),
                           INT8_C( 102), INT8_C( 117), INT8_C( -27), INT8_C(-120),
                           INT8_C( 122), INT8_C( -63), INT8_C(  47), INT8_C( -18),
                           INT8_C( 111), INT8_C(  14), INT8_C(  54), INT8_C( -56),
                           INT8_C( -46), INT8_C( -78), INT8_C( 116), INT8_C( -90),
                           INT8_C( -75), INT8_C(  93), INT8_C(  96), INT8_C(  97),
                           INT8_C( 122), INT8_C(  46), INT8_C( -95), INT8_C(  22)) },
    { simde_mm512_set_epi16(INT16_C(-13944), INT16_C( 30422), INT16_C( 10523), INT16_C( 28986),
                            INT16_C(-23789), INT16_C(-20754), INT16_C( 29282), INT16_C(-10845),
                            INT16_C( 10721), INT16_C(  2777), INT16_C(-18838), INT16_C(  8324),
                            INT16_C( 19192), INT16_C(   114), INT16_C( -9073), INT16_C(  2615),
                            INT16_C( 21008), INT16_C( 12652), INT16_C(-14859), INT16_C(  5734),
                            INT16_C( -5598), INT16_C(-10707), INT16_C(  2170), INT16_C( 23903),
                            INT16_C( 29988), INT16_C( 24405), INT16_C(  5383), INT16_C(-29994),
                            INT16_C(  7143), INT16_C( 22270), INT16_C( -1480), INT16_C( 15491)),
      simde_mm256_set_epi8(INT8_C(-120), INT8_C( -42), INT8_C(  27), INT8_C(  58),
                           INT8_C(  19), INT8_C( -18), INT8_C(  98), INT8_C( -93),
                           INT8_C( -31), INT8_C( -39), INT8_C( 106), INT8_C(-124),
                           INT8_C(  -8), INT8_C( 114), INT8_C(-113), INT8_C(  55),
                           INT8_C(  16), INT8_C( 108), INT8_C( -11), INT8_C( 102),
                           INT8_C(  34), INT8_C(  45), INT8_C( 122), INT8_C(  95),
                           INT8_C(  36), INT8_C(  85), INT8_C(   7), INT8_C( -42),
                           INT8_C( -25), INT8_C(  -2), INT8_C(  56), INT8_C(-125)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm512_cvtepi16_epi8(test_vec[i].a);
    simde_assert_m256i_i8(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_mask_cvtepi16_epi8(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m256i src;
    simde__mmask32 k;
    simde__m512i a;
    simde__m256i r;
  } test_vec[8] = {
    { simde_mm256_set_epi8(INT8_C(-112), INT8_C(  50), INT8_C( -90), INT8_C( -47),
                           INT8_C(  24), INT8_C( -14), INT8_C( -76), INT8_C(  -4),
                           INT8_C(-104), INT8_C( 115), INT8_C( -75), INT8_C(  98),
                           INT8_C( -58), INT8_C( -14), INT8_C(  98), INT8_C(  29),
                           INT8_C( -34), INT8_C(  91), INT8_C(  -9), INT8_C( -32),
                           INT8_C( 105), INT8_C( -54), INT8_C(  11), INT8_C(  76),
                           INT8_C(  83), INT8_C(   3), INT8_C(  48), INT8_C(   2),
                           INT8_C(  92), INT8_C( -54), INT8_C(  99), INT8_C(  95)),
      UINT32_C(     36055),
      simde_mm512_set_epi16(INT16_C( 29253), INT16_C(-14914), INT16_C(  8284), INT16_C( 18521),
                            INT16_C( 32034), INT16_C( 27278), INT16_C( -3730), INT16_C( -7695),
                            INT16_C(  8989), INT16_C(-29300), INT16_C(-14890), INT16_C( 11419),
                            INT16_C( -1355), INT16_C( 25284), INT16_C(-28026), INT16_C(  1548),
                            INT16_C( 26140), INT16_C( -8634), INT16_C( 26242), INT16_C(  1035),
                            INT16_C(-29578), INT16_C( -2997), INT16_C( 22546), INT16_C(-28782),
                            INT16_C(-11973), INT16_C( 12912), INT16_C(-22923), INT16_C(-12898),
                            INT16_C(  4984), INT16_C(   989), INT16_C(  2511), INT16_C( 26483)),
      simde_mm256_set_epi8(INT8_C(-112), INT8_C(  50), INT8_C( -90), INT8_C( -47),
                           INT8_C(  24), INT8_C( -14), INT8_C( -76), INT8_C(  -4),
                           INT8_C(-104), INT8_C( 115), INT8_C( -75), INT8_C(  98),
                           INT8_C( -58), INT8_C( -14), INT8_C(  98), INT8_C(  29),
                           INT8_C(  28), INT8_C(  91), INT8_C(  -9), INT8_C( -32),
                           INT8_C( 118), INT8_C(  75), INT8_C(  11), INT8_C(  76),
                           INT8_C(  59), INT8_C( 112), INT8_C(  48), INT8_C( -98),
                           INT8_C(  92), INT8_C( -35), INT8_C( -49), INT8_C( 115)) },
    { simde_mm256_set_epi8(INT8_C( -93), INT8_C( -75), INT8_C( 109), INT8_C(  43),
                           INT8_C( -79), INT8_C( -91), INT8_C( -13), INT8_C( 103),
                           INT8_C(  -6), INT8_C( -39), INT8_C(   3), INT8_C(-115),
                           INT8_C(  30), INT8_C( -62), INT8_C(  30), INT8_C( 115),
                           INT8_C( -28), INT8_C( -92), INT8_C( 110), INT8_C( -10),
                           INT8_C(  20), INT8_C( -82), INT8_C(  59), INT8_C(  62),
                           INT8_C(  57), INT8_C(  97), INT8_C(  29), INT8_C(  -4),
                           INT8_C( -48), INT8_C(   1), INT8_C(  47), INT8_C(  43)),
      UINT32_C(     13848),
      simde_mm512_set_epi16(INT16_C( 19920), INT16_C( 28417), INT16_C(-26944), INT16_C( -1327),
                            INT16_C(-18966), INT16_C(-19374), INT16_C(  9639), INT16_C(-25572),
                            INT16_C(-16315), INT16_C( 16363), INT16_C( -4686), INT16_C(-14474),
                            INT16_C( 26743), INT16_C( 20737), INT16_C(-16355), INT16_C( 24251),
                            INT16_C(-20830), INT16_C( 19809), INT16_C(-32085), INT16_C(-29115),
                            INT16_C(-21999), INT16_C( 14843), INT16_C( 13075), INT16_C(-28846),
                            INT16_C(-12894), INT16_C( 31357), INT16_C( 16553), INT16_C(-16546),
                            INT16_C(-16544), INT16_C( 30528), INT16_C( -9494), INT16_C(  8241)),
      simde_mm256_set_epi8(INT8_C( -93), INT8_C( -75), INT8_C( 109), INT8_C(  43),
                           INT8_C( -79), INT8_C( -91), INT8_C( -13), INT8_C( 103),
                           INT8_C(  -6), INT8_C( -39), INT8_C(   3), INT8_C(-115),
                           INT8_C(  30), INT8_C( -62), INT8_C(  30), INT8_C( 115),
                           INT8_C( -28), INT8_C( -92), INT8_C( -85), INT8_C(  69),
                           INT8_C(  20), INT8_C(  -5), INT8_C(  19), INT8_C(  62),
                           INT8_C(  57), INT8_C(  97), INT8_C(  29), INT8_C(  94),
                           INT8_C(  96), INT8_C(   1), INT8_C(  47), INT8_C(  43)) },
    { simde_mm256_set_epi8(INT8_C(  57), INT8_C( 119), INT8_C(   6), INT8_C( -62),
                           INT8_C( -27), INT8_C( -22), INT8_C( -69), INT8_C( -61),
                           INT8_C(   8), INT8_C(-101), INT8_C( -24), INT8_C(  69),
                           INT8_C(-111), INT8_C(  66), INT8_C( -48), INT8_C(-122),
                           INT8_C( -19), INT8_C( -25), INT8_C( -88), INT8_C(  96),
                           INT8_C( -81), INT8_C(  28), INT8_C( -73), INT8_C(-105),
                           INT8_C( 109), INT8_C( -84), INT8_C(  26), INT8_C( 108),
                           INT8_C(  16), INT8_C(  69), INT8_C( -67), INT8_C(-122)),
      UINT32_C(     52950),
      simde_mm512_set_epi16(INT16_C(-28100), INT16_C(  2824), INT16_C(-32113), INT16_C(-30059),
                            INT16_C(-19864), INT16_C(-29923), INT16_C( 19573), INT16_C(-11183),
                            INT16_C(-18980), INT16_C( 26281), INT16_C( -7946), INT16_C( 14491),
                            INT16_C( 28715), INT16_C( 26138), INT16_C( 16023), INT16_C( 24398),
                            INT16_C( 20578), INT16_C( -1642), INT16_C( 24774), INT16_C( 26937),
                            INT16_C(-19881), INT16_C(-20408), INT16_C( 26365), INT16_C( -2980),
                            INT16_C( -4479), INT16_C(-10298), INT16_C( 13784), INT16_C(-25535),
                            INT16_C(-26583), INT16_C(-31618), INT16_C(  -202), INT16_C( 28295)),
      simde_mm256_set_epi8(INT8_C(  57), INT8_C( 119), INT8_C(   6), INT8_C( -62),
                           INT8_C( -27), INT8_C( -22), INT8_C( -69), INT8_C( -61),
                           INT8_C(   8), INT8_C(-101), INT8_C( -24), INT8_C(  69),
                           INT8_C(-111), INT8_C(  66), INT8_C( -48), INT8_C(-122),
                           INT8_C(  98), INT8_C(-106), INT8_C( -88), INT8_C(  96),
                           INT8_C(  87), INT8_C(  72), INT8_C(  -3), INT8_C(-105),
                           INT8_C(-127), INT8_C( -58), INT8_C(  26), INT8_C(  65),
                           INT8_C(  16), INT8_C( 126), INT8_C(  54), INT8_C(-122)) },
    { simde_mm256_set_epi8(INT8_C(  89), INT8_C(  16), INT8_C(  86), INT8_C( 124),
                           INT8_C(-106), INT8_C(  54), INT8_C(  30), INT8_C( -60),
                           INT8_C(  41), INT8_C(  45), INT8_C(-103), INT8_C( -75),
                           INT8_C( -46), INT8_C(  -2), INT8_C( 119), INT8_C(  28),
                           INT8_C(  69), INT8_C( -84), INT8_C(  78), INT8_C( -36),
                           INT8_C(  42), INT8_C( -59), INT8_C(  42), INT8_C(   5),
                           INT8_C( -74), INT8_C( -70), INT8_C( 107), INT8_C(  22),
                           INT8_C(  91), INT8_C(  10), INT8_C( -44), INT8_C(  28)),
      UINT32_C(      4183),
      simde_mm512_set_epi16(INT16_C(  8531), INT16_C(  2537), INT16_C(  7090), INT16_C( 32184),
                            INT16_C(   918), INT16_C( -4406), INT16_C( -1230), INT16_C(-20248),
                            INT16_C( 28454), INT16_C( -8033), INT16_C( 29491), INT16_C(  9038),
                            INT16_C( 31537), INT16_C(-32476), INT16_C( 15213), INT16_C(  2771),
                            INT16_C(  9158), INT16_C( 15700), INT16_C( 24392), INT16_C(-14500),
                            INT16_C( 20701), INT16_C( -9424), INT16_C( -5862), INT16_C(  8150),
                            INT16_C(-14293), INT16_C( 29409), INT16_C(-21051), INT16_C(-16951),
                            INT16_C(-32102), INT16_C(-16442), INT16_C(  4517), INT16_C(-32738)),
      simde_mm256_set_epi8(INT8_C(  89), INT8_C(  16), INT8_C(  86), INT8_C( 124),
                           INT8_C(-106), INT8_C(  54), INT8_C(  30), INT8_C( -60),
                           INT8_C(  41), INT8_C(  45), INT8_C(-103), INT8_C( -75),
                           INT8_C( -46), INT8_C(  -2), INT8_C( 119), INT8_C(  28),
                           INT8_C(  69), INT8_C( -84), INT8_C(  78), INT8_C(  92),
                           INT8_C(  42), INT8_C( -59), INT8_C(  42), INT8_C(   5),
                           INT8_C( -74), INT8_C( -31), INT8_C( 107), INT8_C( -55),
                           INT8_C(  91), INT8_C( -58), INT8_C( -91), INT8_C(  30)) },
    { simde_mm256_set_epi8(INT8_C(  66), INT8_C( -53), INT8_C( -22), INT8_C(-109),
                           INT8_C(-122), INT8_C( -34), INT8_C(  49), INT8_C( -51),
                           INT8_C(  45), INT8_C(  96), INT8_C(  21), INT8_C(   9),
                           INT8_C(-107), INT8_C(  88), INT8_C(  41), INT8_C(  63),
                           INT8_C( -15), INT8_C(  66), INT8_C( -60), INT8_C(  80),
                           INT8_C( -27), INT8_C(   9), INT8_C(  30), INT8_C( -73),
                           INT8_C( -55), INT8_C( -22), INT8_C(-122), INT8_C(  86),
                           INT8_C( -35), INT8_C( -54), INT8_C(  95), INT8_C( -17)),
      UINT32_C(     34749),
      simde_mm512_set_epi16(INT16_C(  6349), INT16_C( -1940), INT16_C( 12009), INT16_C( 26974),
                            INT16_C( 15374), INT16_C(  6913), INT16_C(-19915), INT16_C(-14530),
                            INT16_C(-31337), INT16_C( 22983), INT16_C(  6281), INT16_C(  -506),
                            INT16_C(-24168), INT16_C(-22228), INT16_C(-32449), INT16_C(-30658),
                            INT16_C(-16400), INT16_C( -7823), INT16_C( -6600), INT16_C( -5428),
                            INT16_C( 10840), INT16_C(-16201), INT16_C(-15359), INT16_C(-30650),
                            INT16_C(  6966), INT16_C(-30042), INT16_C( 32539), INT16_C(-32588),
                            INT16_C(-23367), INT16_C(-13235), INT16_C(-19835), INT16_C( 15017)),
      simde_mm256_set_epi8(INT8_C(  66), INT8_C( -53), INT8_C( -22), INT8_C(-109),
                           INT8_C(-122), INT8_C( -34), INT8_C(  49), INT8_C( -51),
                           INT8_C(  45), INT8_C(  96), INT8_C(  21), INT8_C(   9),
                           INT8_C(-107), INT8_C(  88), INT8_C(  41), INT8_C(  63),
                           INT8_C( -16), INT8_C(  66), INT8_C( -60), INT8_C(  80),
                           INT8_C( -27), INT8_C( -73), INT8_C(   1), INT8_C(  70),
                           INT8_C(  54), INT8_C( -22), INT8_C(  27), INT8_C( -76),
                           INT8_C( -71), INT8_C(  77), INT8_C(  95), INT8_C( -87)) },
    { simde_mm256_set_epi8(INT8_C(-124), INT8_C(  59), INT8_C( -81), INT8_C(  66),
                           INT8_C( -65), INT8_C( -38), INT8_C( -36), INT8_C(   5),
                           INT8_C(  15), INT8_C(  28), INT8_C( -18), INT8_C( -54),
                           INT8_C(  82), INT8_C(  30), INT8_C(-110), INT8_C(-114),
                           INT8_C(   3), INT8_C(  71), INT8_C(  64), INT8_C(  21),
                           INT8_C( 115), INT8_C( 123), INT8_C( -22), INT8_C(-111),
                           INT8_C( -10), INT8_C(  18), INT8_C(   3), INT8_C(  -8),
                           INT8_C( -97), INT8_C(  26), INT8_C(  72), INT8_C( -94)),
      UINT32_C(     31044),
      simde_mm512_set_epi16(INT16_C(-26750), INT16_C(-23902), INT16_C( 29963), INT16_C(  2819),
                            INT16_C(  9258), INT16_C( 16800), INT16_C(-21230), INT16_C( -2332),
                            INT16_C(-12889), INT16_C( 23107), INT16_C( 17922), INT16_C(  3552),
                            INT16_C( 16956), INT16_C(-21244), INT16_C( -9865), INT16_C( 24672),
                            INT16_C(-32513), INT16_C( -3970), INT16_C( 14993), INT16_C(-21626),
                            INT16_C(-29335), INT16_C( -2219), INT16_C(  4209), INT16_C( 11969),
                            INT16_C( -6560), INT16_C(-26729), INT16_C(  7233), INT16_C( 27170),
                            INT16_C(  5881), INT16_C( -9473), INT16_C(-30967), INT16_C(  3275)),
      simde_mm256_set_epi8(INT8_C(-124), INT8_C(  59), INT8_C( -81), INT8_C(  66),
                           INT8_C( -65), INT8_C( -38), INT8_C( -36), INT8_C(   5),
                           INT8_C(  15), INT8_C(  28), INT8_C( -18), INT8_C( -54),
                           INT8_C(  82), INT8_C(  30), INT8_C(-110), INT8_C(-114),
                           INT8_C(   3), INT8_C( 126), INT8_C(-111), INT8_C(-122),
                           INT8_C( 105), INT8_C( 123), INT8_C( -22), INT8_C( -63),
                           INT8_C( -10), INT8_C(-105), INT8_C(   3), INT8_C(  -8),
                           INT8_C( -97), INT8_C(  -1), INT8_C(  72), INT8_C( -94)) },
    { simde_mm256_set_epi8(INT8_C(  76), INT8_C( -68), INT8_C(   3), INT8_C( 100),
                           INT8_C(  64), INT8_C( -71), INT8_C( -39), INT8_C(  30),
                           INT8_C( 110), INT8_C(  44), INT8_C(  96), INT8_C(  10),
                           INT8_C(  66), INT8_C(  40), INT8_C(  31), INT8_C( -85),
                           INT8_C( 120), INT8_C(  70), INT8_C( -37), INT8_C( -25),
                           INT8_C(  51), INT8_C( -19), INT8_C( 124), INT8_C( -52),
                           INT8_C(  69), INT8_C( 107), INT8_C(  96), INT8_C( 106),
                           INT8_C(-126), INT8_C(  61), INT8_C( -71), INT8_C(   9)),
      UINT32_C(     63997),
      simde_mm512_set_epi16(INT16_C( 25271), INT16_C( 20153), INT16_C(-23804), INT16_C(-24091),
                            INT16_C(  6064), INT16_C(  3189), INT16_C( -2682), INT16_C(  5283),
                            INT16_C( 14900), INT16_C(   731), INT16_C(-14623), INT16_C( 14729),
                            INT16_C( -3836), INT16_C( 26379), INT16_C( 13131), INT16_C( 14975),
                            INT16_C( 19045), INT16_C( 14845), INT16_C(-21672), INT16_C(  4155),
                            INT16_C(  9032), INT16_C(-30375), INT16_C( 14167), INT16_C( 25860),
                            INT16_C( -6683), INT16_C(-21473), INT16_C( -6588), INT16_C( 22432),
                            INT16_C( -4408), INT16_C( -2180), INT16_C( 26333), INT16_C( 18369)),
      simde_mm256_set_epi8(INT8_C(  76), INT8_C( -68), INT8_C(   3), INT8_C( 100),
                           INT8_C(  64), INT8_C( -71), INT8_C( -39), INT8_C(  30),
                           INT8_C( 110), INT8_C(  44), INT8_C(  96), INT8_C(  10),
                           INT8_C(  66), INT8_C(  40), INT8_C(  31), INT8_C( -85),
                           INT8_C( 101), INT8_C(  -3), INT8_C(  88), INT8_C(  59),
                           INT8_C(  72), INT8_C( -19), INT8_C( 124), INT8_C(   4),
                           INT8_C( -27), INT8_C(  31), INT8_C(  68), INT8_C( -96),
                           INT8_C( -56), INT8_C( 124), INT8_C( -71), INT8_C( -63)) },
    { simde_mm256_set_epi8(INT8_C(  40), INT8_C( -41), INT8_C(-126), INT8_C(   8),
                           INT8_C(-115), INT8_C( 108), INT8_C(  31), INT8_C(  41),
                           INT8_C( -21), INT8_C( -60), INT8_C(  76), INT8_C(  74),
                           INT8_C(  86), INT8_C(  39), INT8_C(  41), INT8_C( -61),
                           INT8_C( 120), INT8_C(  -6), INT8_C(-117), INT8_C(  43),
                           INT8_C(  64), INT8_C( -40), INT8_C( -63), INT8_C(  39),
                           INT8_C(  82), INT8_C(  -3), INT8_C(  -8), INT8_C(-102),
                           INT8_C(  21), INT8_C(-109), INT8_C(  -6), INT8_C( 102)),
      UINT32_C(     16734),
      simde_mm512_set_epi16(INT16_C(-25905), INT16_C( 19727), INT16_C( 28735), INT16_C(  3852),
                            INT16_C(-23084), INT16_C( -6530), INT16_C( -1505), INT16_C(  9601),
                            INT16_C( -7362), INT16_C(  8505), INT16_C(-26382), INT16_C( 25139),
                            INT16_C(  4198), INT16_C( -1011), INT16_C( -5955), INT16_C( 29084),
                            INT16_C( 25996), INT16_C( 30463), INT16_C( -4775), INT16_C( 11032),
                            INT16_C(-28689), INT16_C(-14740), INT16_C( -1416), INT16_C(  8406),
                            INT16_C(-23209), INT16_C( 25079), INT16_C( 23521), INT16_C( 23507),
                            INT16_C( 15383), INT16_C(-27993), INT16_C(  2371), INT16_C(-19992)),
      simde_mm256_set_epi8(INT8_C(  40), INT8_C( -41), INT8_C(-126), INT8_C(   8),
                           INT8_C(-115), INT8_C( 108), INT8_C(  31), INT8_C(  41),
                           INT8_C( -21), INT8_C( -60), INT8_C(  76), INT8_C(  74),
                           INT8_C(  86), INT8_C(  39), INT8_C(  41), INT8_C( -61),
                           INT8_C( 120), INT8_C(  -1), INT8_C(-117), INT8_C(  43),
                           INT8_C(  64), INT8_C( -40), INT8_C( -63), INT8_C( -42),
                           INT8_C(  82), INT8_C(  -9), INT8_C(  -8), INT8_C( -45),
                           INT8_C(  23), INT8_C( -89), INT8_C(  67), INT8_C( 102)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm512_mask_cvtepi16_epi8(test_vec[i].src, test_vec[i].k, test_vec[i].a);
    simde_assert_m256i_i8(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_maskz_cvtepi16_epi8(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__mmask32 k;
    simde__m512i a;
    simde__m256i r;
  } test_vec[8] = {
    { UINT32_C(     25439),
      simde_mm512_set_epi16(INT16_C( 26140), INT16_C( -8634), INT16_C( 26242), INT16_C(  1035),
                            INT16_C(-29578), INT16_C( -2997), INT16_C( 22546), INT16_C(-28782),
                            INT16_C(-11973), INT16_C( 12912), INT16_C(-22923), INT16_C(-12898),
                            INT16_C(  4984), INT16_C(   989), INT16_C(  2511), INT16_C( 26483),
                            INT16_C(-18247), INT16_C( 15612), INT16_C( -5009), INT16_C(-29481),
                            INT16_C(-28622), INT16_C(-22831), INT16_C(  6386), INT16_C(-19204),
                            INT16_C(-26509), INT16_C(-19102), INT16_C(-14606), INT16_C( 25117),
                            INT16_C( -8613), INT16_C( -2080), INT16_C( 27082), INT16_C(  2892)),
      simde_mm256_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(  -4), INT8_C( 111), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C( -14), INT8_C(  -4),
                           INT8_C(   0), INT8_C(  98), INT8_C(   0), INT8_C(  29),
                           INT8_C(  91), INT8_C( -32), INT8_C( -54), INT8_C(  76)) },
    { UINT32_C(      1548),
      simde_mm512_set_epi16(INT16_C( -5255), INT16_C( 10791), INT16_C(-28009), INT16_C( 13848),
                            INT16_C(-23627), INT16_C( 27947), INT16_C(-20059), INT16_C( -3225),
                            INT16_C( -1319), INT16_C(   909), INT16_C(  7874), INT16_C(  7795),
                            INT16_C( -7004), INT16_C( 28406), INT16_C(  5294), INT16_C( 15166),
                            INT16_C( 14689), INT16_C(  7676), INT16_C(-12287), INT16_C( 12075),
                            INT16_C( 29253), INT16_C(-14914), INT16_C(  8284), INT16_C( 18521),
                            INT16_C( 32034), INT16_C( 27278), INT16_C( -3730), INT16_C( -7695),
                            INT16_C(  8989), INT16_C(-29300), INT16_C(-14890), INT16_C( 11419)),
      simde_mm256_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C( -66), INT8_C(  92), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(  29), INT8_C(-116), INT8_C(   0), INT8_C(   0)) },
    { UINT32_C(      8241),
      simde_mm512_set_epi16(INT16_C( 28076), INT16_C(  6764), INT16_C(  4165), INT16_C(-17018),
                            INT16_C( 19920), INT16_C( 28417), INT16_C(-26944), INT16_C( -1327),
                            INT16_C(-18966), INT16_C(-19374), INT16_C(  9639), INT16_C(-25572),
                            INT16_C(-16315), INT16_C( 16363), INT16_C( -4686), INT16_C(-14474),
                            INT16_C( 26743), INT16_C( 20737), INT16_C(-16355), INT16_C( 24251),
                            INT16_C(-20830), INT16_C( 19809), INT16_C(-32085), INT16_C(-29115),
                            INT16_C(-21999), INT16_C( 14843), INT16_C( 13075), INT16_C(-28846),
                            INT16_C(-12894), INT16_C( 31357), INT16_C( 16553), INT16_C(-16546)),
      simde_mm256_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(  29), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(  19), INT8_C(  82),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(  94)) },
    { UINT32_C(     46999),
      simde_mm512_set_epi16(INT16_C( 28715), INT16_C( 26138), INT16_C( 16023), INT16_C( 24398),
                            INT16_C( 20578), INT16_C( -1642), INT16_C( 24774), INT16_C( 26937),
                            INT16_C(-19881), INT16_C(-20408), INT16_C( 26365), INT16_C( -2980),
                            INT16_C( -4479), INT16_C(-10298), INT16_C( 13784), INT16_C(-25535),
                            INT16_C(-26583), INT16_C(-31618), INT16_C(  -202), INT16_C( 28295),
                            INT16_C(-12554), INT16_C( -5929), INT16_C(-27764), INT16_C(-12586),
                            INT16_C( 14711), INT16_C(  1730), INT16_C( -6678), INT16_C(-17469),
                            INT16_C(  2203), INT16_C( -6075), INT16_C(-28350), INT16_C(-12154)),
      simde_mm256_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(  41), INT8_C(   0), INT8_C(  54), INT8_C(-121),
                           INT8_C(   0), INT8_C( -41), INT8_C(-116), INT8_C( -42),
                           INT8_C( 119), INT8_C(   0), INT8_C(   0), INT8_C( -61),
                           INT8_C(   0), INT8_C(  69), INT8_C(  66), INT8_C(-122)) },
    { UINT32_C(     14491),
      simde_mm512_set_epi16(INT16_C(-32102), INT16_C(-16442), INT16_C(  4517), INT16_C(-32738),
                            INT16_C(  -320), INT16_C(  2839), INT16_C( 18963), INT16_C(  4183),
                            INT16_C( 22800), INT16_C( 22140), INT16_C(-27082), INT16_C(  7876),
                            INT16_C( 10541), INT16_C(-26187), INT16_C(-11522), INT16_C( 30492),
                            INT16_C( 17836), INT16_C( 20188), INT16_C( 10949), INT16_C( 10757),
                            INT16_C(-18758), INT16_C( 27414), INT16_C( 23306), INT16_C(-11236),
                            INT16_C(-28100), INT16_C(  2824), INT16_C(-32113), INT16_C(-30059),
                            INT16_C(-19864), INT16_C(-29923), INT16_C( 19573), INT16_C(-11183)),
      simde_mm256_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C( -59), INT8_C(   5),
                           INT8_C( -70), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(  60), INT8_C(   0), INT8_C(   0), INT8_C(-107),
                           INT8_C( 104), INT8_C(   0), INT8_C( 117), INT8_C(  81)) },
    { UINT32_C(     48585),
      simde_mm512_set_epi16(INT16_C( -3774), INT16_C(-15280), INT16_C( -6903), INT16_C(  7863),
                            INT16_C(-13846), INT16_C(-31146), INT16_C( -8758), INT16_C( 24559),
                            INT16_C(  8531), INT16_C(  2537), INT16_C(  7090), INT16_C( 32184),
                            INT16_C(   918), INT16_C( -4406), INT16_C( -1230), INT16_C(-20248),
                            INT16_C( 28454), INT16_C( -8033), INT16_C( 29491), INT16_C(  9038),
                            INT16_C( 31537), INT16_C(-32476), INT16_C( 15213), INT16_C(  2771),
                            INT16_C(  9158), INT16_C( 15700), INT16_C( 24392), INT16_C(-14500),
                            INT16_C( 20701), INT16_C( -9424), INT16_C( -5862), INT16_C(  8150)),
      simde_mm256_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(  38), INT8_C(   0), INT8_C(  51), INT8_C(  78),
                           INT8_C(  49), INT8_C(  36), INT8_C(   0), INT8_C( -45),
                           INT8_C( -58), INT8_C(  84), INT8_C(   0), INT8_C(   0),
                           INT8_C( -35), INT8_C(   0), INT8_C(   0), INT8_C( -42)) },
    { UINT32_C(     10559),
      simde_mm512_set_epi16(INT16_C(-31337), INT16_C( 22983), INT16_C(  6281), INT16_C(  -506),
                            INT16_C(-24168), INT16_C(-22228), INT16_C(-32449), INT16_C(-30658),
                            INT16_C(-16400), INT16_C( -7823), INT16_C( -6600), INT16_C( -5428),
                            INT16_C( 10840), INT16_C(-16201), INT16_C(-15359), INT16_C(-30650),
                            INT16_C(  6966), INT16_C(-30042), INT16_C( 32539), INT16_C(-32588),
                            INT16_C(-23367), INT16_C(-13235), INT16_C(-19835), INT16_C( 15017),
                            INT16_C( -4677), INT16_C(-14834), INT16_C(  9957), INT16_C(-30787),
                            INT16_C( 17099), INT16_C( -5485), INT16_C(-31010), INT16_C( 12749)),
      simde_mm256_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(  27), INT8_C(   0),
                           INT8_C( -71), INT8_C(   0), INT8_C(   0), INT8_C( -87),
                           INT8_C(   0), INT8_C(   0), INT8_C( -27), INT8_C( -67),
                           INT8_C( -53), INT8_C(-109), INT8_C( -34), INT8_C( -51)) },
    { UINT32_C(     51006),
      simde_mm512_set_epi16(INT16_C( -6560), INT16_C(-26729), INT16_C(  7233), INT16_C( 27170),
                            INT16_C(  5881), INT16_C( -9473), INT16_C(-30967), INT16_C(  3275),
                            INT16_C( -2646), INT16_C( 14621), INT16_C( 19871), INT16_C( 31044),
                            INT16_C(-31685), INT16_C(-20670), INT16_C(-16422), INT16_C( -9211),
                            INT16_C(  3868), INT16_C( -4406), INT16_C( 21022), INT16_C(-28018),
                            INT16_C(   839), INT16_C( 16405), INT16_C( 29563), INT16_C( -5487),
                            INT16_C( -2542), INT16_C(  1016), INT16_C(-24806), INT16_C( 18594),
                            INT16_C(  6349), INT16_C( -1940), INT16_C( 12009), INT16_C( 26974)),
      simde_mm256_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(  28), INT8_C( -54), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(  21), INT8_C( 123), INT8_C(-111),
                           INT8_C(   0), INT8_C(   0), INT8_C(  26), INT8_C( -94),
                           INT8_C( -51), INT8_C( 108), INT8_C( -23), INT8_C(   0)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm512_maskz_cvtepi16_epi8(test_vec[i].k, test_vec[i].a);
    simde_assert_m256i_i8(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_cvtepi8_epi16(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m256i a;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm256_set_epi8(INT8_C(   7), INT8_C(  68), INT8_C( -86), INT8_C( -36),
                           INT8_C( -19), INT8_C(  73), INT8_C(  92), INT8_C( -27),
                           INT8_C(  55), INT8_C( -65), INT8_C( -50), INT8_C(  19),
                           INT8_C(-111), INT8_C( -79), INT8_C( -16), INT8_C(  70),
                           INT8_C(  27), INT8_C( -28), INT8_C( 116), INT8_C(  42),
                           INT8_C(  -4), INT8_C(  78), INT8_C(  31), INT8_C(  51),
                           INT8_C(  92), INT8_C(  39), INT8_C(-125), INT8_C(  94),
                           INT8_C( -78), INT8_C(  67), INT8_C( -43), INT8_C( -71)),
      simde_mm512_set_epi16(INT16_C(     7), INT16_C(    68), INT16_C(   -86), INT16_C(   -36),
                            INT16_C(   -19), INT16_C(    73), INT16_C(    92), INT16_C(   -27),
                            INT16_C(    55), INT16_C(   -65), INT16_C(   -50), INT16_C(    19),
                            INT16_C(  -111), INT16_C(   -79), INT16_C(   -16), INT16_C(    70),
                            INT16_C(    27), INT16_C(   -28), INT16_C(   116), INT16_C(    42),
                            INT16_C(    -4), INT16_C(    78), INT16_C(    31), INT16_C(    51),
                            INT16_C(    92), INT16_C(    39), INT16_C(  -125), INT16_C(    94),
                            INT16_C(   -78), INT16_C(    67), INT16_C(   -43), INT16_C(   -71)) },
    { simde_mm256_set_epi8(INT8_C(  29), INT8_C( -37), INT8_C(  27), INT8_C(  10),
                           INT8_C( -22), INT8_C(  -9), INT8_C(-125), INT8_C(  -3),
                           INT8_C( -53), INT8_C(  92), INT8_C( 103), INT8_C(  92),
                           INT8_C( 123), INT8_C(  74), INT8_C(  36), INT8_C(  59),
                           INT8_C(  46), INT8_C( -29), INT8_C(-103), INT8_C(  -4),
                           INT8_C( 109), INT8_C( -54), INT8_C(  41), INT8_C(  79),
                           INT8_C(  15), INT8_C( -92), INT8_C( 102), INT8_C( 116),
                           INT8_C( -42), INT8_C(  52), INT8_C( -61), INT8_C( -99)),
      simde_mm512_set_epi16(INT16_C(    29), INT16_C(   -37), INT16_C(    27), INT16_C(    10),
                            INT16_C(   -22), INT16_C(    -9), INT16_C(  -125), INT16_C(    -3),
                            INT16_C(   -53), INT16_C(    92), INT16_C(   103), INT16_C(    92),
                            INT16_C(   123), INT16_C(    74), INT16_C(    36), INT16_C(    59),
                            INT16_C(    46), INT16_C(   -29), INT16_C(  -103), INT16_C(    -4),
                            INT16_C(   109), INT16_C(   -54), INT16_C(    41), INT16_C(    79),
                            INT16_C(    15), INT16_C(   -92), INT16_C(   102), INT16_C(   116),
                            INT16_C(   -42), INT16_C(    52), INT16_C(   -61), INT16_C(   -99)) },
    { simde_mm256_set_epi8(INT8_C(  -9), INT8_C( -47), INT8_C( 107), INT8_C( -74),
                           INT8_C(-126), INT8_C(  34), INT8_C(  64), INT8_C( 115),
                           INT8_C( -65), INT8_C(-124), INT8_C(  54), INT8_C(  27),
                           INT8_C(  41), INT8_C( 112), INT8_C(  61), INT8_C(   6),
                           INT8_C(   7), INT8_C(  39), INT8_C(-109), INT8_C( -99),
                           INT8_C(  63), INT8_C( -35), INT8_C(-111), INT8_C( -72),
                           INT8_C( 109), INT8_C( -39), INT8_C( -99), INT8_C(  26),
                           INT8_C(  66), INT8_C( -78), INT8_C(  30), INT8_C(  38)),
      simde_mm512_set_epi16(INT16_C(    -9), INT16_C(   -47), INT16_C(   107), INT16_C(   -74),
                            INT16_C(  -126), INT16_C(    34), INT16_C(    64), INT16_C(   115),
                            INT16_C(   -65), INT16_C(  -124), INT16_C(    54), INT16_C(    27),
                            INT16_C(    41), INT16_C(   112), INT16_C(    61), INT16_C(     6),
                            INT16_C(     7), INT16_C(    39), INT16_C(  -109), INT16_C(   -99),
                            INT16_C(    63), INT16_C(   -35), INT16_C(  -111), INT16_C(   -72),
                            INT16_C(   109), INT16_C(   -39), INT16_C(   -99), INT16_C(    26),
                            INT16_C(    66), INT16_C(   -78), INT16_C(    30), INT16_C(    38)) },
    { simde_mm256_set_epi8(INT8_C( -72), INT8_C( -80), INT8_C( 101), INT8_C(  81),
                           INT8_C(  23), INT8_C( -68), INT8_C( -57), INT8_C(-111),
                           INT8_C(  -3), INT8_C(  21), INT8_C( 121), INT8_C( -22),
                           INT8_C(-104), INT8_C( -10), INT8_C( -37), INT8_C(  66),
                           INT8_C( -93), INT8_C( -80), INT8_C(  34), INT8_C( 104),
                           INT8_C( -39), INT8_C( -99), INT8_C(  18), INT8_C( 110),
                           INT8_C(-118), INT8_C(  38), INT8_C( 112), INT8_C( -67),
                           INT8_C(  60), INT8_C(  47), INT8_C(  32), INT8_C(  33)),
      simde_mm512_set_epi16(INT16_C(   -72), INT16_C(   -80), INT16_C(   101), INT16_C(    81),
                            INT16_C(    23), INT16_C(   -68), INT16_C(   -57), INT16_C(  -111),
                            INT16_C(    -3), INT16_C(    21), INT16_C(   121), INT16_C(   -22),
                            INT16_C(  -104), INT16_C(   -10), INT16_C(   -37), INT16_C(    66),
                            INT16_C(   -93), INT16_C(   -80), INT16_C(    34), INT16_C(   104),
                            INT16_C(   -39), INT16_C(   -99), INT16_C(    18), INT16_C(   110),
                            INT16_C(  -118), INT16_C(    38), INT16_C(   112), INT16_C(   -67),
                            INT16_C(    60), INT16_C(    47), INT16_C(    32), INT16_C(    33)) },
    { simde_mm256_set_epi8(INT8_C( 120), INT8_C( -90), INT8_C(-101), INT8_C(-106),
                           INT8_C(  70), INT8_C( -49), INT8_C(  29), INT8_C( -43),
                           INT8_C( -42), INT8_C(  38), INT8_C(  16), INT8_C( -43),
                           INT8_C( -40), INT8_C( -76), INT8_C( -67), INT8_C(  53),
                           INT8_C( -73), INT8_C( -17), INT8_C(  66), INT8_C(  57),
                           INT8_C( -65), INT8_C( -63), INT8_C(  17), INT8_C(  -9),
                           INT8_C(  95), INT8_C( -50), INT8_C(-118), INT8_C( 114),
                           INT8_C(  58), INT8_C( -28), INT8_C( -81), INT8_C( -37)),
      simde_mm512_set_epi16(INT16_C(   120), INT16_C(   -90), INT16_C(  -101), INT16_C(  -106),
                            INT16_C(    70), INT16_C(   -49), INT16_C(    29), INT16_C(   -43),
                            INT16_C(   -42), INT16_C(    38), INT16_C(    16), INT16_C(   -43),
                            INT16_C(   -40), INT16_C(   -76), INT16_C(   -67), INT16_C(    53),
                            INT16_C(   -73), INT16_C(   -17), INT16_C(    66), INT16_C(    57),
                            INT16_C(   -65), INT16_C(   -63), INT16_C(    17), INT16_C(    -9),
                            INT16_C(    95), INT16_C(   -50), INT16_C(  -118), INT16_C(   114),
                            INT16_C(    58), INT16_C(   -28), INT16_C(   -81), INT16_C(   -37)) },
    { simde_mm256_set_epi8(INT8_C( -97), INT8_C(  10), INT8_C( -75), INT8_C(-120),
                           INT8_C( -32), INT8_C(-105), INT8_C( -75), INT8_C(-101),
                           INT8_C(  71), INT8_C(-122), INT8_C(-112), INT8_C(  -2),
                           INT8_C(  60), INT8_C( -71), INT8_C( 101), INT8_C(  -1),
                           INT8_C(  95), INT8_C( -58), INT8_C( -70), INT8_C( 102),
                           INT8_C( 115), INT8_C( -68), INT8_C(-110), INT8_C( -36),
                           INT8_C(   6), INT8_C(  58), INT8_C(  73), INT8_C(  97),
                           INT8_C( -51), INT8_C(  -4), INT8_C(  58), INT8_C(  31)),
      simde_mm512_set_epi16(INT16_C(   -97), INT16_C(    10), INT16_C(   -75), INT16_C(  -120),
                            INT16_C(   -32), INT16_C(  -105), INT16_C(   -75), INT16_C(  -101),
                            INT16_C(    71), INT16_C(  -122), INT16_C(  -112), INT16_C(    -2),
                            INT16_C(    60), INT16_C(   -71), INT16_C(   101), INT16_C(    -1),
                            INT16_C(    95), INT16_C(   -58), INT16_C(   -70), INT16_C(   102),
                            INT16_C(   115), INT16_C(   -68), INT16_C(  -110), INT16_C(   -36),
                            INT16_C(     6), INT16_C(    58), INT16_C(    73), INT16_C(    97),
                            INT16_C(   -51), INT16_C(    -4), INT16_C(    58), INT16_C(    31)) },
    { simde_mm256_set_epi8(INT8_C( -73), INT8_C(-123), INT8_C( -11), INT8_C(  62),
                           INT8_C( -96), INT8_C(-103), INT8_C(  85), INT8_C(  88),
                           INT8_C( -19), INT8_C(  28), INT8_C(-107), INT8_C( -81),
                           INT8_C(-125), INT8_C(  88), INT8_C(  84), INT8_C( 115),
                           INT8_C( 105), INT8_C( -47), INT8_C(  68), INT8_C(-124),
                           INT8_C(  32), INT8_C(-100), INT8_C(  10), INT8_C( -69),
                           INT8_C( 124), INT8_C( -51), INT8_C( -89), INT8_C( -72),
                           INT8_C( -92), INT8_C(  -5), INT8_C( -46), INT8_C( 115)),
      simde_mm512_set_epi16(INT16_C(   -73), INT16_C(  -123), INT16_C(   -11), INT16_C(    62),
                            INT16_C(   -96), INT16_C(  -103), INT16_C(    85), INT16_C(    88),
                            INT16_C(   -19), INT16_C(    28), INT16_C(  -107), INT16_C(   -81),
                            INT16_C(  -125), INT16_C(    88), INT16_C(    84), INT16_C(   115),
                            INT16_C(   105), INT16_C(   -47), INT16_C(    68), INT16_C(  -124),
                            INT16_C(    32), INT16_C(  -100), INT16_C(    10), INT16_C(   -69),
                            INT16_C(   124), INT16_C(   -51), INT16_C(   -89), INT16_C(   -72),
                            INT16_C(   -92), INT16_C(    -5), INT16_C(   -46), INT16_C(   115)) },
    { simde_mm256_set_epi8(INT8_C( 104), INT8_C(  66), INT8_C(  51), INT8_C(  81),
                           INT8_C( -69), INT8_C( 104), INT8_C( 126), INT8_C( -43),
                           INT8_C( -40), INT8_C(  23), INT8_C(-124), INT8_C(  98),
                           INT8_C(-125), INT8_C(  95), INT8_C( -36), INT8_C(  46),
                           INT8_C(-115), INT8_C( -93), INT8_C(   2), INT8_C( -77),
                           INT8_C(  80), INT8_C(-116), INT8_C(  61), INT8_C( -89),
                           INT8_C( -37), INT8_C(   9), INT8_C(  84), INT8_C( -64),
                           INT8_C(  94), INT8_C(  67), INT8_C( -53), INT8_C( 111)),
      simde_mm512_set_epi16(INT16_C(   104), INT16_C(    66), INT16_C(    51), INT16_C(    81),
                            INT16_C(   -69), INT16_C(   104), INT16_C(   126), INT16_C(   -43),
                            INT16_C(   -40), INT16_C(    23), INT16_C(  -124), INT16_C(    98),
                            INT16_C(  -125), INT16_C(    95), INT16_C(   -36), INT16_C(    46),
                            INT16_C(  -115), INT16_C(   -93), INT16_C(     2), INT16_C(   -77),
                            INT16_C(    80), INT16_C(  -116), INT16_C(    61), INT16_C(   -89),
                            INT16_C(   -37), INT16_C(     9), INT16_C(    84), INT16_C(   -64),
                            INT16_C(    94), INT16_C(    67), INT16_C(   -53), INT16_C(   111)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_cvtepi8_epi16(test_vec[i].a);
    simde_assert_m512i_i16(r, ==, test_vec[i].r);
  }

  return 0;
}

SIMDE_TEST_FUNC_LIST_BEGIN
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_cvtepi8_epi16)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_cvtepi16_epi8)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_mask_cvtepi16_epi8)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_maskz_cvtepi16_epi8)
SIMDE_TEST_FUNC_LIST_END

#include <test/x86/avx512/test-avx512-footer.h>
