/** 
 * @file gfxroutines.c 
 * @brief 
 * @created 2003-04-15  
 * @date 2007-09-02
 * @author Sam Hocevar <sam@zoy.org> 
 */
/* 
 * copyright (c) 2003-2007 Sam Hocevar <sam@zoy.org> 
 * $Id: gfxroutines.c,v 1.10 2007/09/02 19:57:10 gurumeditation Exp $
 *
 * Powermanga is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Powermanga is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 */
#include <stdbool.h>
#include "powermanga.h"
#include "electrical_shock.h"


void
_type_routine_gfx (Sint32 * adresse)
{
  *adresse = 20;
}

typedef struct _compress
{
  Uint32 offset;
  Uint16 r1;
  Uint16 r2;
}
_compress;

/* To test these functions: the intro animation */

void
conv8_16 (char *src, char *dest, Uint16 * _pPalette16, Uint32 size)
{
  Uint16 *pDestination = (Uint16 *) dest;
  unsigned char *pSource = (unsigned char *) src;
  while (size--)
    {
      *pDestination++ = _pPalette16[*pSource++];
    }
}

void
conv8_24 (char *src, char *dest, Uint32 * _pPalette24, Uint32 size)
{
  fprintf (stderr, "conv8_24 not implemented\n");
}

void
conv8_32 (char *src, char *dest, Uint32 * _pPalette32, Uint32 size)
{
  Uint32 *pDestination = (Uint32 *) dest;
  unsigned char *pSource = (unsigned char *) src;
  while (size--)
    {
      *pDestination++ = _pPalette32[*pSource++];
    }
}

/* ??? */

void
copie4octets (char *src, char *dest,
              Uint32 width, Uint32 height, Uint32 _iOffset, Uint32 _iOffset2)
{
  fprintf (stderr, "copie4octets not implemented\n");
}

/**
 * Clear a offscreen
 * @param oscreen Pointer to a offscreen destination
 * @param with Width of region to clear in 32-bit long words
 * @param height Height of region to clear in lines
 * @param offset Offet of the next line
 */
void
clear_offscreen (char *oscreen, Uint32 width, Uint32 height, Uint32 offset)
{
  width *= 4;
  while (height--)
    {
      memset (oscreen, 0, width);
      oscreen += width + offset;
    }
}

/**
 * Convert a 24-bit RGB palette to a 16-bit RGB palette 
 * @param src 24-bit RGB palette source
 * @param dest 16-bit RGB palette destination 
 */
void
convert_palette_24_to_16 (unsigned char *src, Uint16 * dest)
{
  Sint32 i = 256;
  while (i--)
    {
      Uint16 d = 0;
      d |= (Uint16) ((*src++) & 0x000000f8) << 8;
      d |= (Uint16) ((*src++) & 0xfffffffc) << 3;
      d |= (Uint16) ((*src++) & 0x000000f8) >> 3;
      *dest++ = d;
    }
}

/**
 * Convert a 24-bit RGB palette to a 15-bit RGB palette 
 * @param src 24-bit RGB palette source
 * @param dest 15-bit RGB palette destination 
 */
void
convert_palette_24_to_15 (unsigned char *src, Uint16 * dest)
{
  Sint32 i = 256;
  while (i--)
    {
      Uint16 d = 0;
      d |= ((Uint16) (*src++) >> 3) << 10;
      d |= ((Uint16) (*src++) >> 3) << 5;
      d |= ((Uint16) (*src++) >> 3) << 0;
      *dest++ = d;
    }
}

/* To test these functions: the intro animation with the --640 flag */

#define COPY2X(TYPE) \
    TYPE *src = (TYPE *)_src; \
    TYPE *dest = (TYPE *)_dest; \
    \
    while(height--) \
    { \
        Sint32 i; \
        for(i = width / 8; i--;) \
        { \
            dest[0] = dest[1] = *src++; dest += 2; \
            dest[0] = dest[1] = *src++; dest += 2; \
            dest[0] = dest[1] = *src++; dest += 2; \
            dest[0] = dest[1] = *src++; dest += 2; \
            dest[0] = dest[1] = *src++; dest += 2; \
            dest[0] = dest[1] = *src++; dest += 2; \
            dest[0] = dest[1] = *src++; dest += 2; \
            dest[0] = dest[1] = *src++; dest += 2; \
        } \
        for(i = width % 8; i--;) \
        { \
            dest[0] = dest[1] = *src++; dest += 2; \
        } \
        src += _iOffset / sizeof(TYPE); \
        dest += _iOffset2 / sizeof(TYPE); \
    }

void
_COPY2X8BITS (char *_src, char *_dest,
              Uint32 width, Uint32 height, Uint32 _iOffset, Uint32 _iOffset2)
{
  COPY2X (unsigned char);
}

void
_COPY2X16BITS (char *_src, char *_dest,
               Uint32 width, Uint32 height, Uint32 _iOffset, Uint32 _iOffset2)
{
  COPY2X (Uint16);
}

void
_COPY2X24BITS (char *_src, char *_dest,
               Uint32 width, Uint32 height, Uint32 _iOffset, Uint32 _iOffset2)
{
  fprintf (stderr, "_COPY2X24BITS not implemented\n");
}

void
_COPY2X32BITS (char *_src, char *_dest,
               Uint32 width, Uint32 height, Uint32 _iOffset, Uint32 _iOffset2)
{
  COPY2X (Uint32);
}

/**
 * Draw a 8x8 font (8 / 16 / 24 or 32-bit depth)
 * @param src
 * @param dest
 * @param with
 * @param height
 * @param src_offset
 * @param dest_offset
 */
void
draw_bitmap_char_8 (unsigned char *src, unsigned char *dest,
                    Uint32 width, Uint32 height,
                    Uint32 src_offset, Uint32 dest_offset)
{
  Uint32 h, l;
  unsigned char p;
  unsigned char *s = src;
  unsigned char *d = dest;
  for (h = 0; h < height; h++)
    {
      for (l = 0; l < width; l++)
        {
          p = *(s++);
          if (p != 0)
            {
              *d = p;
            }
          d++;
        }
      s = s + src_offset;
      d = d + dest_offset;
    }
}
void
draw_bitmap_char_16 (unsigned char *src, unsigned char *dest,
                     Uint32 width, Uint32 height,
                     Uint32 src_offset, Uint32 dest_offset)
{
  Uint16 *s = (Uint16 *) src;
  Uint16 *d = (Uint16 *) dest;
  src_offset = src_offset / 2;
  dest_offset = dest_offset / 2;
  for (Uint32 h = 0; h < height; h++)
    {
      for (Uint32 l = 0; l < width; l++)
        {
          Uint16 p = *(s++);
          if (p != 0)
            *d = p;
          d++;
        }
      s = s + src_offset;
      d = d + dest_offset;
    }
}
void
draw_bitmap_char_24 (unsigned char *src, unsigned char *dest,
                     Uint32 width, Uint32 height,
                     Uint32 src_offset, Uint32 dest_offset)
{
  fprintf (stderr, "draw_bitmap_char_24 not implemented\n");
}

void
draw_bitmap_char_32 (unsigned char *src, unsigned char *dest,
                     Uint32 width, Uint32 height,
                     Uint32 src_offset, Uint32 dest_offset)
{
  Uint32 h, l, p;
  Uint32 *s = (Uint32 *) src;
  Uint32 *d = (Uint32 *) dest;
  src_offset = src_offset / 4;
  dest_offset = dest_offset / 4;
  for (h = 0; h < height; h++)
    {
      for (l = 0; l < width; l++)
        {
          p = *(s++);
          if (p != 0)
            {
              *d = p;
            }
          d++;
        }
      s = s + src_offset;
      d = d + dest_offset;
    }
}

/* To test these functions: the main menu, the whole game */

#define PUTSPRITE(TYPE) \
    Uint32 *pCompression = (Uint32 *)_pCompression; \
    \
    while(size--) \
    { \
        dest += *pCompression++; \
        \
        Uint32 data = *pCompression++; \
        Uint32 i = ((data & 0x0000ffff) << 2) /* 32 bits words */ \
                       | ((data & 0xffff0000) \
                           / (0x10000 / sizeof(TYPE))); /* 8 bits words */ \
        \
        memcpy( dest, src, i ); \
        dest += i; \
        src += i; \
    }

void
put_sprite_8 (char *src, char *dest, char *_pCompression, Uint32 size)
{

  unsigned char *s = (unsigned char *) src;
  unsigned char *p = (unsigned char *) dest;
  _compress *t = (_compress *) _pCompression;
  do
    {
      register Uint32 d = t->offset;
      p = p + d;
      Uint16 z = t->r1;
      Uint32 *s2 = (Uint32 *) s;
      Uint32 *p2 = (Uint32 *) p;
      memcpy (p2, s2, z * 4);
      p2 = p2 + z;
      s2 = s2 + z;
      s = (unsigned char *) s2;
      p = (unsigned char *) p2;
      z = t->r2;
      memcpy (p2, s2, z);
      p = p + z;
      s = s + z;
      t++;
      size = size - 1;
    }
  while (size > 0);
}

void
put_sprite_16 (char *src, char *dest, char *_pCompression, Uint32 size)
{
  Uint16 *s = (Uint16 *) src;
  Uint16 *p = (Uint16 *) dest;
  _compress *t = (_compress *) _pCompression;
  do
    {
      register Uint32 d = t->offset;
      d = d >> 1;
      p = p + d;
      Uint16 z = t->r1;
      Uint32 *s2 = (Uint32 *) s;
      Uint32 *p2 = (Uint32 *) p;
      memcpy (p2, s2, z * 4);
      p2 = p2 + z;
      s2 = s2 + z;
      s = (Uint16 *) s2;
      p = (Uint16 *) p2;
      z = t->r2;
      memcpy (p2, s2, z * 2);
      p = p + z;
      s = s + z;
      t++;
      size = size - 1;
    }
  while (size > 0);
}

/**
 *
 */
void
put_sprite_24 (char *src, char *dest, char *_pCompression, Uint32 size)
{
  fprintf (stderr, "put_sprite_24 not implemented\n");
}

/**
 *
 */
void
put_sprite_32 (char *src, char *dest, char *_pCompression, Uint32 size)
{
  Uint32 *s = (Uint32 *) src;
  Uint32 *p = (Uint32 *) dest;
  _compress *t = (_compress *) _pCompression;
  do
    {
      register Uint32 d = t->offset;
      d = d >> 2;
      p = p + d;
      register Uint16 z = t->r1;
      memcpy (p, s, z * 4);
      p = p + z;
      s = s + z;
      z = t->r2;
      memcpy (p, s, z * 4);
      p = p + z;
      s = s + z;
      t++;
      size = size - 1;
    }
  while (size > 0);
}


/* To test these functions: when an enemy gets killed */

#define PUTCOLOR(TYPE) \
    TYPE *pDestination = (TYPE *)dest; \
    Uint32 *pCompression = (Uint32 *)_pCompression; \
    \
    while(size--) \
    { \
        pDestination += *pCompression++ / sizeof(TYPE); \
        \
        Uint32 data = *pCompression++; \
        Uint32 i = ((data & 0x0000ffff) \
                           * (4 / sizeof(TYPE))) /* 32 bits words */ \
                       | ((data & 0xffff0000) >> 16); /* remaining */ \
        \
        if( sizeof(TYPE) == 1 ) \
        { \
            memset(pDestination, _iColor, i); \
            pDestination += i; \
        } \
        else \
        { \
            while(i--) *pDestination++ = _iColor; \
        } \
    }

void
put_sprite_mask_8bits (Uint32 _iColor, char *dest,
                   char *_pCompression, Uint32 size)
{
  unsigned char *p = (unsigned char *) dest;
  _compress *t = (_compress *) _pCompression;
  do
    {
      Uint32 d = t->offset;
      p = p + d;
      Uint16 z = t->r1;
      Uint32 *p2 = (Uint32 *) p;
      memset (p2, _iColor, z * 4);
      p2 = p2 + z;
      p = (unsigned char *) p2;
      z = t->r2;
      memset (p2, _iColor, z);
      p = p + z;
      t++;
      size = size - 1;
    }
  while (size > 0);
}

void
put_sprite_mask_16bits (Uint32 _iColor, char *dest,
                    char *_pCompression, Uint32 size)
{
  Uint32 d = _iColor << 16;
  _iColor = _iColor | d;
  Uint16 *p = (Uint16 *) dest;
  _compress *t = (_compress *) _pCompression;
  do
    {
      d = t->offset;
      d = d >> 1;
      p = p + d;
      Uint16 z = t->r1;
      Uint32 *p2 = (Uint32 *) p;
      memset (p2, _iColor, z * 4);
      p2 = p2 + z;
      p = (Uint16 *) p2;
      z = t->r2;
      memset (p2, _iColor, z * 2);
      p = p + z;
      t++;
      size = size - 1;
    }
  while (size > 0);

}

void
put_sprite_mask_24bits (Uint32 _iColor, char *dest,
                    char *_pCompression, Uint32 size)
{
  fprintf (stderr, "put_sprite_mask_24bits not implemented\n");
}

void
put_sprite_mask_32bits (Uint32 _iColor, char *dest,
                    char *_pCompression, Uint32 size)
{
  PUTCOLOR (Uint32);
}

/* To test these functions: the nuke (purple gem) */

#define POLY(TYPE) \
    TYPE *pDestination = (TYPE *)dest; \
    \
    if(sizeof(TYPE) == 1) \
        memset(pDestination, _iCouleur, _iNombrePixel); \
    else \
        while(_iNombrePixel--) \
            *pDestination++ = _iCouleur

void
poly8bits (char *dest, Sint32 _iNombrePixel, Sint32 _iCouleur)
{
  POLY (unsigned char);
}

void
poly16bits (char *dest, Sint32 _iNombrePixel, Sint32 _iCouleur)
{
  POLY (Uint16);
}

void
poly24bits (char *dest, Sint32 _iNombrePixel, Sint32 _iCouleur)
{
  fprintf (stderr, "poly24bits not implemented\n");
}

void
poly32bits (char *dest, Sint32 _iNombrePixel, Sint32 _iCouleur)
{
  POLY (Uint32);
}

/* To test these functions: the lightning (6 yellow gems) */

#define BRESENHAM(cur1,cur2,inc1,inc2,d1,d2) do { \
    Sint32 dp = d2 << 1; \
    Sint32 dpu = dp - (d1 << 1); \
    Sint32 p = dp - d1; \
    \
    while(d1--) \
    { \
        pDestination[cur1 + cur2 - inc2] = _pEclair->col2; \
        pDestination[cur1 + cur2] = _pEclair->col1; \
        pDestination[cur1 + cur2 + inc2] = _pEclair->col2; \
        cur1 += inc1; \
        if (p > 0) \
        { \
            cur2 += inc2; \
            p += dpu; \
        } \
        else \
            p += dp; \
    } \
} while(0)

#define DRAW_ECLAIR(TYPE,CALLBACK) \
    TYPE *pDestination = (TYPE *)dest; \
    \
    if(_iIter--) \
    { \
        Sint32 dx, dy, midx, midy, r, oldx, oldy; \
        \
        dx = (_pEclair->dx - _pEclair->sx) / 2; \
        dy = (_pEclair->dy - _pEclair->sy) / 2; \
        \
        /* Add a little random normal deviation */ \
        r = randomize_eclair(_pEclair) / (1 << 24); \
        midx = _pEclair->sx + dx + ((dy * r) / (1 << 8)); \
        midy = _pEclair->sy + dy - ((dx * r) / (1 << 8)); \
        \
        /* Recurse on both halves */ \
        oldx = _pEclair->sx; _pEclair->sx = midx; \
        oldy = _pEclair->sy; _pEclair->sy = midy; \
        CALLBACK(dest, _pEclair, _iIter); \
        _pEclair->sx = oldx; \
        _pEclair->sy = oldy; \
        \
        oldx = _pEclair->dx; _pEclair->dx = midx; \
        oldy = _pEclair->dy; _pEclair->dy = midy; \
        CALLBACK(dest, _pEclair, _iIter); \
        _pEclair->dx = oldx; \
        _pEclair->dy = oldy; \
    } \
    else \
    { \
        /* Draw a line using Bresenham */ \
        Sint32 dx = abs(_pEclair->dx - _pEclair->sx); \
        Sint32 dy = abs(_pEclair->dy - _pEclair->sy); \
        \
        Sint32 xcur = _pEclair->sx; \
        Sint32 ycur = _pEclair->sy * 512; \
        \
        Sint32 xinc = _pEclair->sx > _pEclair->dx ? -1 : 1; \
        Sint32 yinc = _pEclair->sy > _pEclair->dy ? -512 : 512; \
        \
        if(dx >= dy) \
            BRESENHAM(xcur,ycur,xinc,yinc,dx,dy); \
        else \
            BRESENHAM(ycur,xcur,yinc,xinc,dy,dx); \
    }

static Sint32
randomize_eclair (Eclair * _pEclair)
{
  Sint32 a = _pEclair->r1, b = _pEclair->r2, c = _pEclair->r3;

  a = (a << 13) | (a >> 3);
  a = a ^ c;
  b = b ^ c;
  b = (b >> 7) | (b << 9);
  c += 27;
  a = a ^ c;
  b = b ^ c;

  _pEclair->r1 = c;
  _pEclair->r2 = a;
  _pEclair->r3 = b;

  return c;
}

static void
draw_electrical_shock_8_in (char *dest, Eclair * _pEclair, Sint32 _iIter)
{
  DRAW_ECLAIR (unsigned char, draw_electrical_shock_8_in);
}

static void
draw_electrical_shock_16_in (char *dest, Eclair * _pEclair, Sint32 _iIter)
{
  DRAW_ECLAIR (Uint16, draw_electrical_shock_16_in);
}

static void
draw_electrical_shock_32_in (char *dest, Eclair * _pEclair, Sint32 _iIter)
{
  DRAW_ECLAIR (Uint32, draw_electrical_shock_32_in);
}

void
draw_electrical_shock_8 (char *dest, Eclair * _pEclair, Sint32 _iIter)
{
  Sint32 a = _pEclair->r1, b = _pEclair->r2, c = _pEclair->r3;
  draw_electrical_shock_8_in (dest, _pEclair, _iIter);
  _pEclair->r1 = a;
  _pEclair->r2 = b;
  _pEclair->r3 = c;
}

void
draw_electrical_shock_16 (char *dest, Eclair * _pEclair, Sint32 _iIter)
{
  Sint32 a = _pEclair->r1, b = _pEclair->r2, c = _pEclair->r3;
  draw_electrical_shock_16_in (dest, _pEclair, _iIter);
  _pEclair->r1 = a;
  _pEclair->r2 = b;
  _pEclair->r3 = c;
}

void
draw_electrical_shock_24 (char *dest, Eclair * _pEclair, Sint32 _iIter)
{
  fprintf (stderr, "draw_electrical_shock_24 not implemented\n");
}

void
draw_electrical_shock_32 (char *dest, Eclair * _pEclair, Sint32 _iIter)
{
  Sint32 a = _pEclair->r1, b = _pEclair->r2, c = _pEclair->r3;
  draw_electrical_shock_32_in (dest, _pEclair, _iIter);
  _pEclair->r1 = a;
  _pEclair->r2 = b;
  _pEclair->r3 = c;
}

/* To test these functions: the main menu, the game */

#define COPY2X_512x440(TYPE) \
    TYPE *src = (TYPE *)_src; \
    TYPE *dest = (TYPE *)_dest; \
    \
    for(int i = height; i--;) \
    { \
        for(int j = 256 / 8; j--;) \
        { \
            dest[0] = dest[1] = *src++; dest += 2; \
            dest[0] = dest[1] = *src++; dest += 2; \
            dest[0] = dest[1] = *src++; dest += 2; \
            dest[0] = dest[1] = *src++; dest += 2; \
            dest[0] = dest[1] = *src++; dest += 2; \
            dest[0] = dest[1] = *src++; dest += 2; \
            dest[0] = dest[1] = *src++; dest += 2; \
            dest[0] = dest[1] = *src++; dest += 2; \
        } \
        src += 256; \
        dest += 640 + (640 - 512); \
    }

void
_COPY2X8BITS_512x440 (char *_src, char *_dest, Uint32 height)
{
  COPY2X_512x440 (unsigned char);
}

void
_COPY2X16BITS_512x440 (char *_src, char *_dest, Uint32 height)
{
  COPY2X_512x440 (Uint16);
}

void
_COPY2X24BITS_512x440 (char *_src, char *_dest, Uint32 height)
{
  fprintf (stderr, "_COPY2X24BITS_512x440 not implemented\n");
}

void
_COPY2X32BITS_512x440 (char *_src, char *_dest, Uint32 height)
{
  COPY2X_512x440 (Uint32);
}

/* To test these functions: the cursor in the "ORDER" menu and name input */

#define PUTRECT(TYPE) \
    TYPE *dest = (TYPE *)adresse + (512 * y + x); \
    Sint32 i; \
    \
    /* Top line */ \
    i = width; while(i--) *dest++ = coul; \
    \
    dest += 512 - width; \
    height -= 2; \
    \
    /* Side lines */ \
    while(height-- > 0) \
    { \
        dest[0] = dest[width-1] = coul; \
        dest += 512; \
    } \
    \
    /* Bottom line */ \
    i = width; while(i--) *dest++ = coul

void
draw_rectangle_8 (char *adresse, Sint32 x, Sint32 y,
                        Sint32 coul, Sint32 width, Sint32 height)
{
  PUTRECT (unsigned char);
}

void
draw_rectangle_16 (char *adresse, Sint32 x, Sint32 y,
                         Sint32 coul, Sint32 width, Sint32 height)
{
  PUTRECT (Uint16);
}

void
draw_rectangle_24 (char *adresse, Sint32 x, Sint32 y,
                         Sint32 coul, Sint32 width, Sint32 height)
{
  fprintf (stderr, "draw_rectangle_16 not implemented\n");
}

void
draw_rectangle_32 (char *adresse, Sint32 x, Sint32 y,
                         Sint32 coul, Sint32 width, Sint32 height)
{
  PUTRECT (Uint32);
}
