/*	Checker

PIRL CVS ID: Checker.java,v 1.6 2012/04/16 06:18:24 castalia Exp

Copyright (C) 2004-2012  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/
package	PIRL.Utilities;

/**	A <i>Checker</i> is used to compare expected against obtained results
	for unit testing.
<p>
	A set of Check methods compares an expected argument value with an
	obtained argument value for equality. When the arguments match the
	word "PASS" is printed, otherwise the word "FAIL" is printed. This
	is followed by ": " and the description String that very briefly
	identifies what was checked (i.e. the unit test that occured). The
	obtained value will then be printed following " - " if the
	Show_Obtained flag is enabled. If the comparison does not match, or
	the Check is Verbose, the expected and obtained arguments are
	printed on the next two lines following an "====> expected: " and
	"====> obtained: " prefix.
<p>
	Check methods are provided for integer, floating point, boolean and char values as
	well as String and generic Object values.
	String values are printed enclosed
	in double quotes (") and char values are listed enclosed in single
	quotes (').
<p>
	The total number of checks performed (Checks_Total), and the number
	in which the comparison matched (Checks_Passed), is accumulated by
	each Check method.
<p>
	@author		Bradford Castalia
	@version	1.6
*/
public class Checker
{
public static final String
	ID = "PIRL.Utilities.Checker (1.6 2012/04/16 06:18:24)";

/**	Total number of Checks performed.
*/
public int
	Checks_Total	= 0;

/**	Total number of Checks where expected matched obtained.
*/
public int
	Checks_Passed	= 0;


/**	Set to true to always show both the expected and obtained values.
<p>
	Default is false;
*/
public boolean
	Verbose			= false;

/**	Set to true to always show the obtained value.
<p>
	Default is false;
*/
public boolean
	Show_Obtained	= false;

/*==============================================================================
	Checks
*/
/**	Checks integer values.

	@param	description	A brief description of what was checked.
	@param	expected	The expected value.
	@param	obtained	The obtained value.
	@return	true if expected equals obtained; false otherwise.
*/
public boolean Check
	(
	String	description,
	long	expected,
	long	obtained
	)
{
boolean
	passed = obtained == expected;
System.out.print ((passed ? "PASS" : "FAIL") + ": " + description);
if (Verbose || ! passed)
	System.out.println ('\n'
	+"====> expected: " + expected + '\n'
	+"====> obtained: " + obtained);
else if (Show_Obtained)
	System.out.println (" - " + obtained);
else
	System.out.println ();
return Check (passed);
}

/**	Checks floating point values.

	@param	description	A brief description of what was checked.
	@param	expected	The expected value.
	@param	obtained	The obtained value.
	@return	true if expected equals obtained; false otherwise.
*/
public boolean Check
	(
	String	description,
	double	expected,
	double	obtained
	)
{
boolean
	passed = obtained == expected;
System.out.print ((passed ? "PASS" : "FAIL") + ": " + description);
if (Verbose || ! passed)
	System.out.println ('\n'
	+"====> expected: " + expected + '\n'
	+"====> obtained: " + obtained);
else if (Show_Obtained)
	System.out.println (" - " + obtained);
else
	System.out.println ();
return Check (passed);
}

/**	Checks boolean values.

	@param	description	A brief description of what was checked.
	@param	expected	The expected value.
	@param	obtained	The obtained value.
	@return	true if expected equals obtained; false otherwise.
*/
public boolean Check
	(
	String	description,
	boolean	expected,
	boolean	obtained
	)
{
boolean
	passed = obtained == expected;
System.out.print ((passed ? "PASS" : "FAIL") + ": " + description);
if (Verbose || ! passed)
	System.out.println ('\n'
	+"====> expected: " + expected + '\n'
	+"====> obtained: " + obtained);
else if (Show_Obtained)
	System.out.println (" - " + obtained);
else
	System.out.println ();
return Check (passed);
}

/**	Checks char values.

	@param	description	A brief description of what was checked.
	@param	expected	The expected value.
	@param	obtained	The obtained value.
	@return	true if expected equals obtained; false otherwise.
*/
public boolean Check
	(
	String	description,
	char	expected,
	char	obtained
	)
{
boolean
	passed = obtained == expected;
System.out.print ((passed ? "PASS" : "FAIL") + ": " + description);
if (Verbose || ! passed)
	System.out.println ('\n'
	+"====> expected: '" + expected + "'\n"
	+"====> obtained: '" + obtained + "'");
else if (Show_Obtained)
	System.out.println (" - '" + obtained + "'");
else
	System.out.println ();
return Check (passed);
}

/**	Checks String values.

	@param	description	A brief description of what was checked.
	@param	expected	The expected value.
	@param	obtained	The obtained value.
	@return	true if expected equals obtained; false otherwise.
*/
public boolean Check
	(
	String	description,
	String	expected,
	String	obtained
	)
{
boolean
	passed =
		(expected == null && obtained == null) ||
		(expected != null && expected.equals (obtained));
System.out.print ((passed ? "PASS" : "FAIL") + ": " + description);
if (Verbose || ! passed)
	System.out.println ('\n'
	+"====> expected: \"" + expected + "\"\n"
	+"====> obtained: \"" + obtained + '"');
else if (Show_Obtained)
	System.out.println (" - \"" + obtained + '"');
else
	System.out.println ();
return Check (passed);
}

/**	Checks Exceptions.
<p>
	If both the excpected and obtained exception are  null they are
	equal. If both are non-null the {@link Object#getClass() class} of
	each are checked for equality. If only one exception is null they are
	not equal.
<p>
	@param	description	A brief description of what was checked.
	@param	expected	The expected Exception.
	@param	obtained	The obtained Exception.
	@return	true if expected equals obtained; false otherwise.
*/
public boolean Check
	(
	String	description,
	Exception	expected,
	Exception	obtained
	)
{
boolean
	passed = 
		(expected == null && obtained == null) ||
		(expected != null && obtained != null &&
			expected.getClass ().equals (obtained.getClass ()));
System.out.print ((passed ? "PASS" : "FAIL") + ": " + description);
if (Verbose || ! passed)
	System.out.println ('\n'
	+"====> expected: " + expected + '\n'
	+"====> obtained: " + obtained);
else if (Show_Obtained)
	System.out.println (" - " + obtained);
else
	System.out.println ();
return Check (passed);
}

/**	Checks Objects.
<p>
	If both objects are null they are equal. <b>N.B.</b>: The {@link
	Object#equals(Object)} method is used to compare a non-null expected
	Object with the obtained object, which may not be what is expected
	since this comparison is only true if both arguments are the same
	Object reference.
<p>
	@param	description	A brief description of what was checked.
	@param	expected	The expected Object.
	@param	obtained	The obtained Object.
	@return	true if expected equals obtained; false otherwise.
*/
public boolean Check
	(
	String	description,
	Object	expected,
	Object	obtained
	)
{
boolean
	passed = 
		(expected == null && obtained == null) ||
		(expected != null && expected.equals (obtained));
System.out.print ((passed ? "PASS" : "FAIL") + ": " + description);
if (Verbose || ! passed)
	System.out.println ('\n'
	+"====> expected: " + expected + '\n'
	+"====> obtained: " + obtained);
else if (Show_Obtained)
	System.out.println (" - " + obtained);
else
	System.out.println ();
return Check (passed);
}

/**	Updates the check accounting.
<p>
	The Checks_Total accumulator is incremented. The Checks_Passed
	accumulator is incremented if passed is true.
<p>
	@param	passed	true if the external test passed; false otherwise.
	@return	The passed value.
*/
public boolean Check
	(
	boolean	passed
	)
{
Checks_Total++;
if (passed)
	Checks_Passed++;
return passed;
}

}
