/*	Database

PIRL CVS  ID: Database.java,v 2.35 2012/04/16 06:08:57 castalia Exp

Copyright (C) 2001-2008  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/

package PIRL.Database;

import PIRL.Configuration.Configuration;
import PIRL.Configuration.Configuration_Exception;

import java.sql.Connection;
import java.util.Vector;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.Collection;


/**	A <i>Database</i> provides an implementation of a <i>Data_Port</i> to
	manage access to any database for which a driver is available.
<p>
	Every Database will have a <i>Configuration</i>. If one is not
	specified when a Database object is created a default Configuration
	is created. The application may provide a Configuration directly to
	the new Database object. The Configuration is a group of PVL
	Parameters that provide needed characterization for accessing
	specific databases.
<p>
	A Database is like a File in that instantitating an object of this
	class does not establish a data access connection to a database's
	contents. This is done with a <code>{@link #Connect(String, String)
	Connect}</code> method. To Connect to a database server its type
	(e.g. "MySQL") must be specified or a default <code>{@link
	#TYPE TYPE}</code> parameter must be found in the
	Configuration. The Database will extract from its Configuration only
	those Parameters applicable to the specified type and use them to
	<code>{@link #Open(Configuration) Open}</code> the Data_Port. The
	Open method will load a Data_Port class, having a name beginning
	with the database type name and ending with "_Data_Port", that
	implements the Data_Port interface and knows how to access that type
	of database (it may, for example, load a JDBC driver class, perhaps
	named by a "Driver" parameter). The Database class implements the
	Data_Port interface itself by passing its Data_Port method calls on
	to its current dynamically loaded Data_Port object.
<p>
	If a specific database catalog to access from the server is
	specified with the <code>Connect</code> method, then a <code>{@link
	#CATALOG CATALOG}</code> parameter will be set in the
	Data_Port Configuration. The Data_Port is likely to supplement the
	Configuration it is given with its own default Parameters.
<p>
	There are a set of parameter names that are known to the Database
	class:
<p>
<dl>
<dt><b>{@link #SERVER Server}</b>
	<dd>The list of server configuration groups in a Configuration. Each
	group must have the name of one of the names in the Server list. The
	first name in the list is the default server.
<dt><b>{@link #TYPE Type}</b>
	<dd>The type of database server to connect to. This name, with the
		addition of "_Data_Port", is the name of the Data_Port class
		providing access to the server. There is no default; it must
		be specified.
<dt><b>{@link #DRIVER Driver}</b>
	<dd>The formal name, as a Java pathname, of a database driver class
		that might be needed by a Data_Port. Usually each Data_Port
		class implementation provides its own default.
<dt><b>{@link #CATALOG Catalog}</b>
	<dd>The name of the data catalog to use on the server.
<dt><b>{@link #TABLE Table}</b>
	<dd>The name of one or more tables within a catalog to be accessed
		for data. The first table listed is the default table.
<dt><b>{@link #FIELDS Fields}</b>
	<dd>The name of one or more fields within a table.
</dl>
<p>
@see		Configuration
@see		Data_Port

@author		Bradford Castalia - UA/PIRL
@version	2.35 
*/
public class Database
	implements Data_Port
{
/**	Class name and version identification.
*/
public static final String
	ID = "PIRL.Database.Database (2.35 2012/04/16 06:08:57)";


/**	The parameter name for the default database server: "Server".
*/
public static final String
	SERVER				= "Server";

/**	The parameter name for the type of database server which
	corresponds to the Data_Port class to use: "Type".
*/
public static final String
	TYPE				= "Type";

/**	The parameter name for the driver class to provide access to the
	database server: "Driver".
*/
public static final String
	DRIVER				= "Driver";

/**	The parameter name for the name of the database on the server to be
	used: "Database".
<p>
	<b>N.B.</b>: Some database servers require the value of this
	parameter to make a client connection (e.g. PostgreSQL), while others
	do not use it (e.g. MySQL).
<p>
	@see	#CATALOG
*/
public static final String
	DATABASE			= "Database";

/**	The parameter name for a container of data tables: "Catalog".
<p>
	<b>N.B.</b>: The meaning of the term "catalog" to the Database class
	is the database server structure that contains a collection of data
	tables. For some database servers the corresponding term is
	"database" (e.g. MySQL), while for others it is "schema" (e.g.
	PostgreSQL).
*/
public static final String
	CATALOG				= "Catalog";

/**	The parameter name for the name of one or more tables within a
	catalog: "Table".
*/
public static final String
	TABLE				= "Table";

/**	The parameter name for the name of one or more fields within a
	table: "Field".
*/
public static final String
	FIELDS				= "Fields";

/**	The "value" to specify when setting a field to NULL.
*/
public static final String
	NULL_VALUE			= "NULL";

/**	The default for determining if database entity identifiers are case
	sensitive.
<p>
	A database server that does not use case sensitive entity identifiers
	will coerce the identifiers to lowercase. When these identifiers are
	returned from a query they will not match user specified identifiers
	that are in mixed case.
<p>
	Because some database servers on some operating systems are not case
	sensitive in handling identifiers (the names of catalogs, tables, and
	field names) it may be necessary to enforce case insensitivity when
	matching user specified names against identifiers returned from the
	database server.
<p>
	When a Data_Port is {@link #Open(Configuration) open}ed this flag is
	initialized from the Data_Port {@link Data_Port#Case_Sensitive() case
	sensitive} indicator.
<p>
	<b>N.B.</b>: The setting of this flag does not itself determine the
	case sensitivity of the identifiers, rather it is expected to inform
	the Database software if case sensitive matches are allowed. Setting
	this flag to true when the database server uses case insensitive
	identifiers (mixed case identifiers provided by the user to the
	database server being always read back as all lowercase) will
	ultimately result in unexpected mismatches. Only set this flag to
	true when it is known that all database servers that the Database
	class will be used with are case sensitive.
<p>
	Unless case sensitive matches are required case insensitive
	identifier comparisions should be used.
<p>
	@see	#Matches(String, String)
*/
private static boolean
	Case_Sensitive_Default			= false;

/**	The default configuration filename: "Database.conf".
*/
public static final String
	DEFAULT_CONFIGURATION_FILENAME	= "Database.conf";
	
private Configuration
	The_Configuration;

private Data_Port
	Data_Port						= null;

private static final String
	DATA_PORT_CLASS_NAME_SUFFIX		= "_Data_Port";

/**	The new-line sequence for the local host system.
*/
public static final String
	NL								= System.getProperty ("line.separator");


//	DEBUG control.
private static final int
	DEBUG_OFF			= 0,
	DEBUG_CONFIGURE		= 1 << 0,
	DEBUG_DESCRIPTION	= 1 << 1,
	DEBUG_QUERY			= 1 << 2,
	DEBUG_UTILITY		= 1 << 3,
	DEBUG_LOAD			= 1 << 4,
	DEBUG_CONNECT		= 1 << 5,
	DEBUG_VALUE			= 1 << 6,
	DEBUG_ALL			= -1,

	DEBUG				= DEBUG_OFF;


/*==============================================================================
	Constructors
*/
/**	Creates a Database with a default Configuration.
*/
public Database ()
	throws Database_Exception
{
try {The_Configuration = new Configuration (DEFAULT_CONFIGURATION_FILENAME);}
catch (Configuration_Exception exception)
	{throw new Database_Exception (exception);}
}

/**	Creates a Database with a specific Configuration.
<p>
	@param	configuration	The Configuration to use when {@link
		#Connect(String, String) connecting} to the database server. If
		the configuration is null, a default Configuration is created.
*/
public Database
	(
	Configuration	configuration
	)
	throws Database_Exception
{
try
	{
	The_Configuration = (configuration == null) ?
		new Configuration (DEFAULT_CONFIGURATION_FILENAME) : configuration;
	}
catch (Configuration_Exception exception)
	{
	throw new Database_Exception (exception);
	}
}

/*==============================================================================
	Accessors
*/
/**	Gets the Configuration for the Database.
<p>
	@return	The Configuration of the Database.
*/
public Configuration Configuration ()
{return The_Configuration;}

/**	Gets the current Data_Port backing the Database.
<p>
	@return	The Data_Port currently being used by the Database. This
		will be null if the Database does not have a connection to a
		database server.
*/
public Data_Port Data_Port ()
{return Data_Port;}

/*==============================================================================
	Methods
*/
/*--------------------------------------------------------------------------
	Connect:
*/
/**	Connects the Database to a named database server and a specific
	catalog of the database as a data source.
<p>
	To connect to a database server the Configuration parameters used
	to <code>{@link Database#Open(Configuration) Open}</code> the
	connection must be found in the Configuration of this Database in a
	<code>{@link Configuration#Group(String) Group}</code> with the
	database server name. The case sensitivity of the search for the
	server name is controlled by the Configuration <code>{@link
	Configuration#Case_Sensitive() Case_Sensitive}</code> setting. If
	the name of this Database Configuration matches the server name,
	then the entire Configuration is used. If the server name is null,
	then the value of the top_level <code>{@link #SERVER SERVER}</code>
	parameter will be used, if present; otherwise the value of the top
	level <code>{@link #TYPE TYPE}</code> parameter will be used.
<p>
	<b>Note</b>: The use of the <code>TYPE</code> parameter is
	deprecated; it is provided for backwards compatibility with
	previous versions of the Database configuration rules in which the
	database server was known by its <code>TYPE</code> name.
<p>
	If a non-null catalog name is provided, a <code>{@link #CATALOG
	CATALOG}</code> parameter is set in the server Configuration group
	with this name.
<p>
	Once the server Configuration group has been setup it is used to
	<code>Open</code> the <code>Data_Port</code> of this Database.
<p>
	@param	server	The name of the database server to use. If null the
		{@link #Default_Server_Name() default server name} is used.
	@param	catalog	The name of database catalog to use.
		This may be null if no specific catalog is selected.
	@return	This Database.
	@throws	Configuration_Exception	If the database server name can not
		be found in the Configuration, or its server group does not
		contain a <code>{@link #TYPE TYPE}</code> parameter.
	@throws	Database_Exception	If the Data_Port could not be
		<code>Open</code>ed.
	@see	Configuration#Group(String)
	@see	#Open(Configuration)
*/
public Database Connect
	(
	String	server,
	String	catalog
	)
	throws Database_Exception, Configuration_Exception
{
if ((DEBUG & (DEBUG_CONNECT | DEBUG_CONFIGURE)) != 0)
	System.out.println
		(">>> Database.Connect:" + NL
		+"     server: " + server + NL
		+"    catalog: " + catalog + NL
		+"    configuration -" + NL
		+ The_Configuration);
//	Get the Configuration for the database server.
Configuration
	configuration = Server_Configuration (server);
if (configuration == null)
	{
	if (server == null &&
		(server = Default_Server_Name ()) == null)
		{
		throw new Configuration_Exception
			(
			ID + NL
			+ "To Connect to a database a server name must be specified." + NL
			+ "The \"" + SERVER +
				"\" parameter in a Configuration may be used to do this."
			);
		}
	throw new Configuration_Exception
		(
		ID + NL
		+ "Unable to Connect to the database server." + NL
		+ "Missing \"" + server + "\" Configuration Group."
		);
	}

if ((catalog != null &&
	catalog.length () != 0))
	{
	//	Set the CATALOG parameter.
	if ((DEBUG & DEBUG_CONNECT) != 0)
		System.out.println
			("    Database.Connect: Set " + CATALOG + " = " + catalog);
	configuration.Set (CATALOG, catalog);
	}

//	Open the data port.
Open (configuration);

if ((DEBUG & DEBUG_CONNECT) != 0)
	System.out.println
		("<<< Database.Connect");
return this;
}

/**	Connects the Database to a named database server.
<p>
	@param	server	The name of the database server to use.
	@return	This Database.
	@throws	Database_Exception	If establishing the Data_Port
		Configuration failed, or there was a problem opening the
		Data_Port.
	@see	#Connect(String, String)
*/
public Database Connect
	(
	String	server
	)
	throws Database_Exception, Configuration_Exception
{return Connect (server, null);}

/**	Connects the Database to the default database server.
<p>
	@return	This Database.
	@throws	Database_Exception	If establishing the Data_Port
		Configuration failed, or there was a problem opening the
		Data_Port.
	@see	#Connect(String, String)
*/

public Database Connect ()
	throws Database_Exception, Configuration_Exception
{return Connect (null, null);}

/**	Disconnect the Database (its backing Data_Port) from its
	database server.
<p>
	If the Data_Port is already closed, nothing is done.
*/
public void Disconnect ()
	throws Database_Exception
{
if (Data_Port != null &&
	Data_Port.is_Open ())
	Data_Port.Close ();
Data_Port = null;
}

//	Load the Data_Port class.
private Data_Port Load_Data_Port
	(
	String	type
	)
	throws Database_Exception
{
if ((DEBUG & DEBUG_LOAD) != 0)
	System.out.println
		(">>> Database.Load_Data_Port: type = " + type );
Data_Port
	data_port = null;
String
	data_port_class = type + DATA_PORT_CLASS_NAME_SUFFIX;
try
	{
	data_port =
		(Data_Port)Class.forName (data_port_class).newInstance ();
	}
catch (ClassNotFoundException exception)
	{
	//	Try again in this package.
	data_port_class = "PIRL.Database." + data_port_class;
	try
		{
		data_port =
			(Data_Port)Class.forName (data_port_class).newInstance ();
		}
	catch (ClassNotFoundException sigh)
		{
		throw new Database_Exception
			(
			ID + NL
			+ "Unable to find a \""
				+ type + DATA_PORT_CLASS_NAME_SUFFIX
				+ "\" class."
			);
		}
	catch (Exception bummer) {}
	}
catch (Exception bummer) {}
if (data_port == null)
	{
	throw new Database_Exception
		(
		ID + NL
		+ "Unable to load the \"" + data_port_class + "\" class."
		);
	}
if ((DEBUG & DEBUG_LOAD) != 0)
	System.out.println ("<<< Database.Load_Data_Port: " + data_port);
return data_port;
}

/**	Gets the required and optional parameters used by a Data_Port type.
<p>
	@param	type	The name of the Data_Port type. If null, the
		currently connected Data_Port will be queried.
	@return	A Configuration of Required and Optional Data_Port
		parameters.
	@throws	Database_Exception If the Data_Port class can not be loaded.
	@throws	IllegalArgumentException	If the Data_Port type is null
		and there is no currently connected Data_Port.
	@throws Configuration_Exception		If the Data_Port could not
		be loaded.
	@see	Data_Port#Parameters()
*/
public Configuration Parameters
	(
	String	type
	)
	throws Database_Exception, Configuration_Exception
{
Data_Port
	data_port;
if (type == null)
	{
	if (Data_Port == null)
		throw new Database_Exception
			(
			ID + NL
			+ "Unable to get Parameters for an unknown type of Data Port."
			);
	else
		data_port = Data_Port;
	}
else
	data_port = Load_Data_Port (type);

return data_port.Parameters ();
}

/**	Gets the required and optional parameters used by the default
	Data_Port.
<p>
	@see	#Parameters(String)
*/
public Configuration Parameters ()
	throws Database_Exception, Configuration_Exception
{return Parameters (null);}

/**	Gets the default server name from the Database Configuration.
<p>
	If a <code>{@link #SERVER Server}</code> parameter is not found in
	the {@link #Configuration() Configuration} of the Database the
	<code>{@link #TYPE Type}</code> parameter is obtained. These
	parameters must be at the top level of the Configuration.
<p>
	@return	The default server name for use in making a {@link
		#Connect(String, String) connection} to a database server. This
		will be null if the Database Configuration does not contain the
		required information.
*/
public String Default_Server_Name ()
{
String
	server = The_Configuration.Get_One
		(Configuration.Absolute_Pathname (null, SERVER));
if (server == null)
	server = The_Configuration.Get_One
		(Configuration.Absolute_Pathname (null, TYPE));
if ((DEBUG & DEBUG_CONFIGURE) != 0)
	System.out.println
		(">-< Database.Default_Server_Name: " + server);
return server;
}

/**	Gets a copy of the configuration for a database server.
<p>
	The server name is expected to be the name of a Configuration Group
	in the Database Configuration.
<p>
	If the server name matches the name of the Database Configuration a
	copy of the entire Database Configuration is returned. Otherwise a
	copy of the first Group in the Database Configuration with the
	server name is returned.
<p>
	@param	server	The server name. If null the {@link
		#Default_Server_Name() default server name} is used.
	@return	A Configuration for the server. This will be null if the
		server name is null and no default server could be identified,
		or if no Group with the server name could be found.
	@throws	Configuration_Exception	If there was a problem while
		manipulating a Configuration.
	@see	#Configuration()
*/
public Configuration Server_Configuration
	(
	String	server
	)
	throws Configuration_Exception
{
if ((DEBUG & DEBUG_CONFIGURE) != 0)
	System.out.println
		(">>> Database.Server_Configuration: " + server);
if (server == null &&
	(server = Default_Server_Name ()) == null)
	{
	//	No default server name.
	if ((DEBUG & DEBUG_CONFIGURE) != 0)
		System.out.println
			("<<< Database.Server_Configuration");
	return null;
	}

Configuration
	configuration = null;
if (The_Configuration.Case_Sensitive () ?
	The_Configuration.Name ().equals (server) :
	The_Configuration.Name ().equalsIgnoreCase (server))
	{
	//	Use the entire Configuration.
	if ((DEBUG & DEBUG_CONFIGURE) != 0)
		System.out.println
			("    Database.Server_Configuration: Using entire Configuration - "
			+ The_Configuration.Name ());
	configuration = new Configuration (The_Configuration);
	}
else
	{
	configuration = The_Configuration.Group (server);
	if ((DEBUG & DEBUG_CONFIGURE) != 0)
		System.out.println
			("    Database.Server_Configuration: "
			+ ((configuration == null) ? "couldn't find" : " found")
			+" Configuration Group " + server);
	}

if (configuration != null)
	//	Ensure there is a TYPE parameter; use the server name by default.
	configuration.Set_Conditionally (TYPE, server);

if ((DEBUG & DEBUG_CONFIGURE) != 0)
	System.out.println
		("<<< Database.Server_Configuration");
return configuration;
}

/**	Gets the default server configuration.
<p>
	@return	A Configuration for the default server. This will be
		null if no default server can be found.
	@throws	Configuration_Exception	If there was a problem while
		manipulating a Configuration.
	@see	#Server_Configuration(String)
*/
public Configuration Server_Configuration ()
	throws Configuration_Exception
{return Server_Configuration (null);}

/*==============================================================================
	Data_Port implementation
*/
/*------------------------------------------------------------------------------
	Access:
*/
/**	Opens a Data_Port using the Configuration.
<p>
	If the Data_Port for this Database is already open, it is closed
	and then re-opened.
<p>
	Before a Data_Port can be opened its implementing class must be
	loaded. The configuration must contain a <code>{@link
	#TYPE Type}</code> parameter which is used to select
	the Data_Port class that will be loaded to manage access to the
	database server. The name of this class must be "type"_Data_Port,
	where "type" is the value of the Configuration TYPE parameter. If
	the class is not found in the current classpath, it is presumed to
	be in PIRL.Database package. The <code>Open</code> method of the
	new Data_Port instance will be given the configuration to use in
	establishing its connection to the database server.
<p>
	<b>Note</b>: The configuration of the Data_Port may be completely
	separate from the configuration associated with this Database; the
	preferred method to open access to a database is to use one of the
	<code>{@link #Connect(String, String) Connect}</code> methods.
<p>
	@param	configuration	The Configuration for the Data_Port. If the
		configuration is null the default <code>{@link #Connect()
		Connect}</code> method is used to select a
		database type and its configuration.
	@throws	Database_Exception	If the TYPE parameter could not be
		found in the configuration; the implementing Data_Port class
		could not be loaded; or the Data_Port Open failed.
*/
public void Open
	(
	Configuration	configuration
	)
	throws Database_Exception, Configuration_Exception
{
if ((DEBUG & DEBUG_CONNECT) != 0)
	System.out.println
		(">>> Database.Open: Configuration - " + configuration.Name ());
if (configuration == null)
	{
	//	Use the default configuration, which will recursively call Open.
	Connect ();
	if ((DEBUG & DEBUG_CONNECT) != 0)
		System.out.println ("<<< Database.Open");
	return;
	}
if (is_Open ())
	{
	//	Close the existing connection.
	if ((DEBUG & DEBUG_CONNECT) != 0)
		System.out.println ("    Database.Open: Disconnect");
	Disconnect ();
	}

//	Load the Data_Port class.
String
	type = configuration.Get_One (TYPE);
if ((DEBUG & DEBUG_CONNECT) != 0)
	System.out.println ("    Database.Open: " + TYPE + " = " + type);
if (type == null)
	throw new Configuration_Exception
		(
		ID + NL
		+ "To Open a database a \"" + TYPE
			+ "\" parameter must be in the Configuration."
		);
Data_Port = Load_Data_Port (type);

//	Open the Data_Port.
Data_Port.Open (configuration);

//	Initialize identifier case sensitivity.
Case_Sensitive_Default = Data_Port.Case_Sensitive_Identifiers ();
if ((DEBUG & DEBUG_CONNECT) != 0)
	System.out.println ("<<< Database.Open");
}

/**	Closes the Database connection to the database server.
<p>
	@see	#Disconnect()
*/
public void		Close ()
	throws Database_Exception
{Disconnect ();}

/**	Tests if a connection to the database server is open.
<p>
	@return	true if the Data_Port is open; false otherwise.
*/
public boolean	is_Open ()
{
return Data_Port == null ?
	false : Data_Port.is_Open ();
}

/**	Gets the JDBC Connection object.

	<b>Note</b>: A Data_Port is not required to use a JDBC Connection.
<p>
	@return	A JDBC Connection object associated with the Data_Port
		backing the Database. This will be null if the Data_Port
		is closed or it does not use a JDBC Connection.
*/
public Connection Connection ()
{
if (Data_Port != null)
	return Data_Port.Connection ();
return null;
}

/**	Adds a SQL_Listener to the list of listeners to be notified of
	SQL statements before they are sent to the database server.
<p>
	@param	listener	A SQL_Listener to be added. If null, or the
		listener is already listed, nothing is done.
	@return	This Data_Port, or null if the Data_Port is closed.
	@see	SQL_Listener
*/
public Data_Port Add_SQL_Listener
	(
	SQL_Listener	listener
	)
{
if (Data_Port != null)
	return Data_Port.Add_SQL_Listener (listener);
return null;
}

/**	Removes a SQL_Listener from the list of listeners to be notified of
	SQL statements before they are sent to the database server.
<p>
	@param	listener	A SQL_Listener to be removed.
	@return	true if the listener was removed; false if the listener was
		not listed or the the Data_Port is closed.
	@see	SQL_Listener
*/
public boolean Remove_SQL_Listener
	(
	SQL_Listener	listener
	)
{
if (Data_Port != null)
	return Data_Port.Remove_SQL_Listener (listener);
return false;
}

/*------------------------------------------------------------------------------
	Description:
*/
/**	@see	Data_Port#toString()
*/
public String toString ()
{
return Data_Port == null ?
	ID : ID + NL + Data_Port.toString ();
}

/**	@see	Data_Port#Description()
*/
public String Description ()
{
return Data_Port == null ?
	ID : ID + NL + Data_Port.Description ();
}

/**	@see	Data_Port#Contents(String, String)
*/
public String Contents
	(
	String	catalog,
	String	table
	)
{
return Data_Port == null ?
	ID : ID + NL + Data_Port.Contents (catalog, table);
}

/**	@see	Data_Port#Catalogs()
*/
public Vector Catalogs ()
	throws Database_Exception
{
connect_check ();
return Data_Port.Catalogs ();
}

/**	@see	Data_Port#Tables(String)
*/
public Vector Tables
	(
	String	catalog
	)
	throws Database_Exception
{
connect_check ();
return Data_Port.Tables (catalog);
}

/**	@see	Data_Port#Field_Names(String)
*/
public Vector Field_Names
	(
	String	table
	)
	throws Database_Exception
{
connect_check ();
return Data_Port.Field_Names (table);
}

/**	@see	Data_Port#Field_Types(String)
*/
public Vector Field_Types
	(
	String	table
	)
	throws Database_Exception
{
connect_check ();
return Data_Port.Field_Types (table);
}

/**	@see	Data_Port#Fields(String, String)
*/
public Vector Fields
	(
	String	table,
	String	field_info
	)
	throws Database_Exception
{
connect_check ();
return Data_Port.Fields (table, field_info);
}

/**	@see	Data_Port#Keys(String)
*/
public Vector Keys
	(
	String	table
	)
	throws Database_Exception
{
connect_check ();
return Data_Port.Keys (table);
}

/*------------------------------------------------------------------------------
	Query:
*/
/**	@see	Data_Port#Query(String, int)
*/
public Vector Query
	(
	String	SQL_Query,
	int		limit
	)
	throws Database_Exception
{
connect_check ();
return Data_Port.Query (SQL_Query, limit);
}

/**	@see	Data_Port#Select(Vector, Vector, String, int)
*/
public Vector Select
	(
	Vector	tables,
	Vector	fields,
	String	conditions,
	int		limit
	)
	throws Database_Exception
{
connect_check ();
return Data_Port.Select (tables, fields, conditions, limit);
}

/*------------------------------------------------------------------------------
	Update:
*/
/**	@see	Data_Port#Update(String)
*/
public int Update
	(
	String	SQL_Update
	)
	throws Database_Exception
{
connect_check ();
return Data_Port.Update (SQL_Update);
}

/*..............................................................................
	Catalogs:
*/
/**	@see	Data_Port#Create(String)
*/
public void Create
	(
	String	catalog
	)
	throws Database_Exception
{
connect_check ();
Data_Port.Create (catalog);
}

/**	@see	Data_Port#Delete(String)
*/
public void Delete
	(
	String	catalog
	)
	throws Database_Exception
{
connect_check ();
Data_Port.Delete (catalog);
}

/*..............................................................................
	Tables:
*/
/**	@see	Data_Port#Create(String, Vector, Vector)
*/
public void Create
	(
	String	table,
	Vector	fields,
	Vector	types
	)
	throws Database_Exception
{
connect_check ();
Data_Port.Create (table, fields, types);
}

/**	@see	Data_Port#Delete(String, Vector)
*/
public void Delete
	(
	String	table,
	Vector	fields
	)
	throws Database_Exception
{
connect_check ();
Data_Port.Delete (table, fields);
}

/**	@see	Data_Port#Rename(String, String)
*/
public void	Rename
	(
	String	table,
	String	name
	)
	throws Database_Exception
{
connect_check ();
Data_Port.Rename (table, name);
}

/**	@see	Data_Port#Rename(String, Vector, Vector)
*/
public void Rename
	(
	String	table,
	Vector	fields,
	Vector	names
	)
	throws Database_Exception
{
connect_check ();
Data_Port.Rename (table, fields, names);
}

/*..............................................................................
	Fields:
*/
/**	@see	Data_Port#Insert(String, Vector, Vector)
*/
public int Insert
	(
	String	table,
	Vector	fields,
	Vector	values
	)
	throws Database_Exception
{
connect_check ();
return Data_Port.Insert (table, fields, values);
}

/**	@see	Data_Port#Update(String, Vector, Vector, String)
*/
public int Update
	(
	String	table,
	Vector	fields,
	Vector	values,
	String	conditions
	)
	throws Database_Exception
{
connect_check ();
return Data_Port.Update (table, fields, values, conditions);
}

/**	@see	Data_Port#Delete(String, String)
*/
public int Delete
	(
	String	table,
	String	conditions
	)
	throws Database_Exception
{
connect_check ();
return Data_Port.Delete (table, conditions);
}

/*..............................................................................
	Utility:
*/
/**	@see	Data_Port#Table_Reference(String, String)
*/
public String Table_Reference
	(
	String	catalog,
	String	table
	)
	throws Database_Exception
{
connect_check ();
return Data_Port.Table_Reference (catalog, table);
}

/**	@see	Data_Port#Catalog_Name(String)
*/
public String Catalog_Name
	(
	String	table_reference
	)
	throws Database_Exception
{
connect_check ();
return Data_Port.Catalog_Name (table_reference);
}

/**	@see	Data_Port#Database_Catalog_Name(String)
*/
public String Database_Catalog_Name
	(
	String	catalog
	)
	throws Database_Exception
{
connect_check ();
return Data_Port.Database_Catalog_Name (catalog);
}

/**	@see	Data_Port#Table_Name(String)
*/
public String Table_Name
	(
	String	table_reference
	)
	throws Database_Exception
{
connect_check ();
return Data_Port.Table_Name (table_reference);
}

/**	@see	Data_Port#Database_Table_Name(String)
*/
public String Database_Table_Name
	(
	String	table
	)
	throws Database_Exception
{
connect_check ();
return Data_Port.Database_Table_Name (table);
}

/**	@see	Data_Port#Table_Reference_Component_Delimiter()
*/
public String Table_Reference_Component_Delimiter ()
	throws Database_Exception
{
connect_check ();
return Data_Port.Table_Reference_Component_Delimiter ();
}

/**	@see	Data_Port#Case_Sensitive_Identifiers()
*/
public boolean Case_Sensitive_Identifiers ()
	throws Database_Exception
{
connect_check ();
return Data_Port.Case_Sensitive_Identifiers ();
}

/**	@see	Data_Port#Case_Sensitive_Identifiers(boolean)
*/
public void Case_Sensitive_Identifiers
	(
	boolean		case_sensitive
	)
	throws Database_Exception
{
connect_check ();
Data_Port.Case_Sensitive_Identifiers (case_sensitive);
}

/*==============================================================================
	Convenience methods
*/
/*..............................................................................
	Select:
*/
/**	Selects fields conditionally from a list of tables with no
	limit on the number of records returned.
<p>
	@see	#Select(Vector, Vector, String, int)
*/
public Vector	Select (Vector tables, Vector fields, String conditional)
	throws Database_Exception
{return Select (tables, fields, conditional, -1);}

/**	Selects fields conditionally from a single table.
<p>
	@see	#Select(Vector, Vector, String, int)
*/
public Vector	Select (String table, Vector fields, String conditional)
	throws Database_Exception
{
Vector tables = new Vector (1);
tables.add (table);
return Select (tables, fields, conditional, -1);
}

/**	Selects the entire default table.
<p>
	@see	#Select(Vector, Vector, String, int)
*/
public Vector	Select ()
	throws Database_Exception
{
return Select ((Vector)null, null, null, -1);
}

/**	Selects an entire single table.
<p>
	@see	#Select(Vector, Vector, String, int)
*/
public Vector	Select (String table)
	throws Database_Exception
{
return Select (table, null, null);
}

/**	Selects unconditionally from the fields of a single table.
<p>
	@see	#Select(Vector, Vector, String, int)
*/
public Vector	Select (String table, Vector fields)
	throws Database_Exception
{
return Select (table, fields, null);
}

/**	Selects all fields conditionally from a single table.
<p>
	@see	#Select(Vector, Vector, String, int)
*/
public Vector	Select (String table, String conditional)
	throws Database_Exception
{
return Select (table, null, conditional);
}

/**	Selects fields unconditionally from the default table.
<p>
	@see	#Select(Vector, Vector, String, int)
*/
public Vector	Select (Vector fields)
	throws Database_Exception
{
return Select ((Vector)null, fields, null, -1);
}

/**	Selects fields conditionally from the default table.
<p>
	@see	#Select(Vector, Vector, String, int)
*/
public Vector	Select (Vector fields, String conditional)
	throws Database_Exception
{
return Select ((Vector)null, fields, conditional, -1);
}

/**	Unlimited Query.
<p>
	@see	#Query(String, int)
*/
public Vector Query
	(
	String	SQL_Query
	)
	throws Database_Exception
{
connect_check ();
return Data_Port.Query (SQL_Query, -1);
}

/*..............................................................................
	Insert:
*/
/**	Inserts the set of records into a table.
<p>
	The set of records is like that returned from the <code>{@link
	#Select(Vector, Vector, String, int) Select}</code> or <code>{@link
	#Query(String, int) Query}</code> methods: a Vector of record
	Vectors, where the first Vector contains the String names of the
	fields corresponding to the data values in the subsequent Vectors.
	The data records do not necessarily have to contain Strings, as
	long as each object's toString method produces a valid
	representation of the field value. Though all the records must be
	the same size, not all fields in the table need to be included; the
	database is expected to provide the default for missing fields.
<p>
	An entire table can be copied from an old_table into a new_table
	using:

<PRE>database.Insert (new_table, database.Select (old_table))
</PRE>
<p>
	@param	table	The name of the table in which to insert the
		records. This may be null to use the default table.
	@param	records	The field names record followed by a set of
		data records.
	@return	The number of records inserted.
	@throws	Database_Exception	If an insertion failed.
	@see	#Insert(String, Vector, Vector)
*/
public int Insert
	(
	String	table,
	Vector	records
	)
	throws Database_Exception
{
if (records == null)
	return 0;
int
	count;
for (count = 1;
	 count < records.size ();
	 count++)
	Insert (table, (Vector)records.get (0), (Vector)records.get (count));
return --count;
}

/*==============================================================================
	Utility methods
*/
/**	Constructs a Hashtable from a Vector of keys and a corresponding
	Vector of values.
<p>
	@param	keys	The Vector of objects to be used as the Hashtable
		keys.
	@param	values	The Vector of objects to be used as the Hashtable
		values.
	@return	A Hashtable of key/value pairs. This will be null if either
		the keys or values Vectors are null.
	@throws	Database_Exception	If the keys and values Vectors are not
		the same size, there is a duplicate (equals) key object, or a
		key or value is null.
	@see	Hashtable#put(Object, Object)
*/
public static Hashtable Hash_Map
	(
	Vector	keys,
	Vector	values
	)
	throws Database_Exception
{
if ((DEBUG & DEBUG_UTILITY) != 0)
	System.out.println (">>> Database.Hash_Map:" + NL
		+ "    keys - " + keys + NL
		+ "    values - " + values);
if (keys == null || values == null)
	return null;
if (keys.size () != values.size ())
	throw new Database_Exception
		(
		ID + NL
		+ "Hash_Map was given " + keys.size () + " key"
			+ ((keys.size () == 1) ? " " : "s ") + "but "
			+ values.size () + " value"
			+ ((values.size () == 1) ? "." : "s.")
		);
Hashtable
	keys_map = new Hashtable ();
Iterator
	key_list = keys.iterator (),
	value_list = values.iterator ();
while (key_list.hasNext ())
	{
	String key = (String)key_list.next ();
	if (keys_map.put (key, value_list.next ()) != null)
		throw new Database_Exception
			(
			ID + NL
			+ "Invalid hash map entry for key \"" + key + "\"."
			);
	}
if ((DEBUG & DEBUG_UTILITY) != 0)
	System.out.println ("<<< Database.Hash_Map: " + keys_map);
return keys_map;
}

/**	Tests if two strings match.
<P>
	This is a helper function used by the Matches methods.
<P>
	@param	string_1	A String to be compared.
	@param	string_2	The other String for the comparison.
	@param	case_sensitive	If true, the string comparison will be case
		sensitive; otherwise the comparison will ignore case.
	@return	true if the strings match, false otherwise. If both strings
		are null they match. If only one string is null, they do not
		match.
	@see	#Matches(String, String)
*/
public static boolean Matches
	(
	String	string_1,
	String	string_2,
	boolean	case_sensitive
	)
{
if (string_1 == null &&
	string_2 == null)
	return true;
if (string_1 == null ||
	string_2 == null)
	return false;
return case_sensitive ?
	string_1.equals (string_2) :
	string_1.equalsIgnoreCase (string_2);
}

/**	Tests if two strings match.
<P>
	The {@link #Matches(String, String, boolean) Matches} function will
	be used with case sensitivity determined by the {@link
	#Case_Sensitive_Identifiers() Data_Port identifier case sensitivity}
	or the {@link #Case_Sensitive_Default} value if the Database is not
	bound to a Data_Port.
<P>
	@param	string_1	A String to be compared.
	@param	string_2	The other String for the comparison.
	@return	true if the strings match, false otherwise. If both strings
		are null they match.
	@see	#Matches(String, String, boolean)
*/
public boolean Matches
	(
	String	string_1,
	String	string_2
	)
{
boolean
	case_sensitive = Case_Sensitive_Default;
if (Data_Port != null)
	{
	try {case_sensitive = Data_Port.Case_Sensitive_Identifiers ();}
	catch (Database_Exception exception) {/* Can't happen */}
	}
return Matches (string_1, string_2, case_sensitive);
}

/**	Tests if a string matches any element of a Vector.
<P>
	This is a helper function used by the corresponding Matches method.
<P>
	@param	vector	The Vector to search for a matching String. Each
		element of the Vector is cast to a String. If null, false is
		returned.
	@param	string	The String to be compared. May be null, in which
		case the vector must contain a null element for true to be
		returned.
	@param	case_sensitive	If true, the string comparison will be case
		sensitive; otherwise the comparison will ignore case.
	@return true if the string matches any element of the vector,
		false otherwise.
	@see	#Matches(String, String, boolean)
*/
public static boolean Matches
	(
	Vector	vector,
	String	string,
	boolean	case_sensitive
	)
{
if (vector == null)
	return false;

for (int
		index = 0,
		size = vector.size ();
		index < size;
		index++)
	if (Matches (string, (String)vector.get (index), case_sensitive))
		return true;
return false;
}

/**	Tests if a string matches any element of a Vector.
<P>
	The {@link #Matches(Vector, String, boolean) Matches} function will
	be used with case sensitivity determined by the {@link
	#Case_Sensitive_Identifiers() Data_Port identifier case sensitivity}
	or the {@link #Case_Sensitive_Default} value if the Database is not
	bound to a Data_Port.
<P>
	@param	vector	The Vector to search for a matching String. Each
		element of the Vector is cast to a String. If null, false is
		returned.
	@param	string	The String to be compared. May be null, in which
		case the vector must contain a null element for true to be
		returned.
	@return true if the string matches any element of the vector,
		false otherwise.
*/
public boolean Matches
	(
	Vector	vector,
	String	string
	)
{
boolean
	case_sensitive = Case_Sensitive_Default;
if (Data_Port != null)
	{
	try {case_sensitive = Data_Port.Case_Sensitive_Identifiers ();}
	catch (Database_Exception exception) {/* Can't happen */}
	}
return Matches (vector, string, case_sensitive);
}

/**	Gets the index of the string that matches an element of a Vector.
<P>
	This is a helper function used by the corresponding Index method.
<P>
	@param	vector	The Vector to search for a matching String. Each
		element of the Vector is cast to a String. If null, -1 is
		returned.
	@param	string	The String to be compared. May be null, in which
		case the index of the first null element of the vector, if any,
		will be returned.
	@return The index of the first element of the vector that matches
		the string, or -1 if no match is found.
	@see	#Matches(String, String, boolean)
*/
public static int Index
	(
	Vector	vector,
	String	string,
	boolean	case_sensitive
	)
{
if (vector == null)
	return -1;

for (int
		index = 0,
		size = vector.size ();
		index < size;
		index++)
	if (Matches (string, (String)vector.get (index), case_sensitive))
		return index;
return -1;
}

/**	Gets the index of the string that matches an element of a Vector.
<P>
	The {@link #Index(Vector, String, boolean) Index} function will
	be used with case sensitivity determined by the {@link
	#Case_Sensitive_Identifiers() Data_Port identifier case sensitivity}
	or the {@link #Case_Sensitive_Default} value if the Database is not
	bound to a Data_Port.
<P>
	@param	vector	The Vector to search for a matching String. Each
		element of the Vector is cast to a String. If null, -1 is
		returned.
	@param	string	The String to be compared. May be null, in which
		case the index of the first null element of the vector, if any,
		will be returned.
	@return The index of the first element of the vector that matches
		the string, or -1 if no match is found.
*/
public int Index
	(
	Vector	vector,
	String	string
	)
{
boolean
	case_sensitive = Case_Sensitive_Default;
if (Data_Port != null)
	{
	try {case_sensitive = Data_Port.Case_Sensitive_Identifiers ();}
	catch (Database_Exception exception) {/* Can't happen */}
	}
return Index (vector, string, case_sensitive);
}

/**	Gets the index of the string that matches an element of an array.
<P>
	This is a helper function used by the corresponding Index method.
<P>
	@param	array	The array of Strings to search for a matching String.
		If null, -1 is returned.
	@param	string	The String to be compared. May be null, in which
		case the index of the first null element of the array, if any,
		will be returned.
	@return The index of the first element of the array that matches
		the string, or -1 if no match is found.
	@see	#Matches(String, String, boolean)
*/
public static int Index
	(
	String[]	array,
	String		string,
	boolean		case_sensitive
	)
{
if (array == null)
	return -1;

for (int
		index = 0;
		index < array.length;
		index++)
	if (Matches (string, array[index], case_sensitive))
		return index;
return -1;
}

/**	Gets the index of the string that matches an element of an array.
<P>
	The {@link #Index(String[], String, boolean) Index} function will
	be used with case sensitivity determined by the {@link
	#Case_Sensitive_Identifiers() Data_Port identifier case sensitivity}
	or the {@link #Case_Sensitive_Default} value if the Database is not
	bound to a Data_Port.
<P>
	@param	array	The array of Strings to search for a matching String.
		If null, -1 is returned.
	@param	string	The String to be compared. May be null, in which
		case the index of the first null element of the array, if any,
		will be returned.
	@return The index of the first element of the array that matches
		the string, or -1 if no match is found.
*/
public int Index
	(
	String[]	array,
	String		string
	)
{
boolean
	case_sensitive = Case_Sensitive_Default;
if (Data_Port != null)
	{
	try {case_sensitive = Data_Port.Case_Sensitive_Identifiers ();}
	catch (Database_Exception exception) {/* Can't happen */}
	}
return Index (array, string, case_sensitive);
}

/**	Substitutes special characters for escape sequences.
<p>
	The following escape sequences, and their corresponding special
	characters, are recognized:
<blockquote>
	\b - Backspace (BS)<br>
	\t - Horizontal tab (HT)<br>
	\n - Newline (NL)<br>
	\f - Form feed (FF)<br>
	\r - Carriage return (CR)<br>
	\X - The character X<br>
	\0nnn - The character having the octal value nnn (0 <= nnn <= 377)<br>
</blockquote>
	The escape sequences will be substituted for their corresponding
	special characters. All backslash characters, except those that are
	themselves escaped, will be removed.
<p>
	@param	string	The String to be un-escaped.
	@return	The un-escaped String.
*/
public static String Escape_to_Special
	(
	String	string
	)
{
if (string == null)
	return null;
StringBuffer
	mung = new StringBuffer (string);
int
	 index;
for (index = 0;
	 index < mung.length ();
	 index++)
	{
	if (mung.charAt (index) == '\\')
		{
		//	Delete the backslash.
		mung.deleteCharAt (index);
		//	Check the escaped character.
		switch (mung.charAt (index))
			{
			case 'b':
				//	BS
				mung.setCharAt (index, '\b');
				break;
			case 't':
				//	HT
				mung.setCharAt (index, '\t');
				break;
			case 'n':
				//	NL
				mung.setCharAt (index, '\n');
				break;
			case 'f':
				//	FF
				mung.setCharAt (index, '\f');
				break;
			case 'r':
				//	CR
				mung.setCharAt (index, '\r');
				break;
			case '0':
				//	Octal value.
				char
					character,
					new_character = 0;
				int
					 end_index;
				for (end_index = index + 1;
					 end_index < mung.length () &&
					(end_index - index) < 4;	//	No more that four digits.
					 end_index++)
					{
					//	Only octal digits are acceptable.
					if ((character = mung.charAt (end_index)) > '7' ||
						 character < '0')
						break;
					new_character *= 8;
					new_character += character - 48;
					}
				if (new_character < 256)
					{
					mung.setCharAt (index, new_character);
					mung.delete (index + 1, end_index);
					}
			}
		}
	}
return mung.toString ();
}

/**	Substitutes escape sequences for special characters in a String.
<p>
	All control characters - less than ' ' (32), plus single quote
	(39), double quote (34), backslash (92) and DEL (127) - will be
	substituted with escape sequences:
<blockquote>
	\0 - for an ASCII 0 (NUL, 0) character<br>
	\b - for backspace (BS, 8)<br>
	\t - for horizontal tab (HT, 9)<br>
	\n - for newline (NL, 10)<br>
	\f - for form feed (FF, 12)<br>
	\r - for carriage return (CR, 13)<br>
	\' - for a single quote (', 39)<br>
	\" - for a double quote (", 34)<br>
	\\ - for backslash (\, 92)<br>
	\0nnn - for all other special characters where nnn is the octal
		value of the character.<br>
</blockquote>
<p>
	Escape sequences in the string will be recognized and left
	unchanged.
<p>
	@param	string	The String to be escaped.
	@return	The escaped String.
*/
public static String Special_to_Escape
	(
	String	string
	)
{
if (string == null)
	return null;
if ((DEBUG & DEBUG_UTILITY) != 0)
	System.out.println (">>> Database.Special_to_Escape: " + string);
StringBuffer
	mung = new StringBuffer (string);
char
	character;
String
	escape;
int
	index = 0;
while (index < mung.length ())
	{
	character = mung.charAt (index);
	if (character < 32 ||
		character == '"' ||
		character == '\'' ||
		character >= 127)
		{
		//	Check the special character.
		switch (character)
			{
			case '\b':
				//	BS
				escape = "b";
				break;
			case '\t':
				//	HT
				escape = "t";
				break;
			case '\n':
				//	NL
				escape = "n";
				break;
			case '\f':
				//	FF
				escape = "f";
				break;
			case '\r':
				//	CR
				escape = "r";
				break;
			case '\'':
				//	CR
				escape = "'";
				break;
			case '"':
				//	CR
				escape = "\"";
				break;
			default:
				//	Octal value.
				escape = "0" + Integer.toString ((int)character, 8);
			}
		if ((DEBUG & DEBUG_UTILITY) != 0)
			System.out.print ("|\\" + escape + "|");
		mung.setCharAt (index++, '\\');
		mung.insert (index, escape);
		index += escape.length ();
		}
	else if (character == '\\')
		{
		//	Backslash
		if (++index < mung.length () &&
		   (character = mung.charAt (index)) == '\\' ||
			character == 'b' ||
			character == 't' ||
			character == 'n' ||
			character == 'f' ||
			character == 'r' ||
			character == '\'' ||
			character == '"' ||
			character == '0')
			{
			//	Escape sequence
			if ((DEBUG & DEBUG_UTILITY) != 0)
				System.out.print ("\\" + character);
			index++;
			continue;
			}
		//	Escape the backslash
		if ((DEBUG & DEBUG_UTILITY) != 0)
			System.out.print ("|\\\\|");
		mung.insert (index++, '\\');
		}
	else
		{
		if ((DEBUG & DEBUG_UTILITY) != 0)
			System.out.print (character);
		index++;
		}
	}
if ((DEBUG & DEBUG_UTILITY) != 0)
	System.out.println (NL
		+"<<< Database.Special_to_Escape: " + mung.toString ());
return mung.toString ();
}

/**	Filters a data value to protect special characters.
<p>
	If the data type is non-numeric it is <code>{@link
	#Special_to_Escape(String) Special_to_Escape}</code> filtered and
	enclosed in single quote (') characters. A numeric data type is
	identified by containing in its name one of these (case
	insensitive) substrings:
<p>
<ul>
	<li>INT
	<li>REAL
	<li>FLOAT
	<li>DOUBLE
</ul>
	The data value of a numeric data type is not altered.
<p>
	@param	value	A String representing a data value.
	@param	type	The JDBC name of the value's data type.
	@return	The possibly altered value string.
*/
public static String Value_Syntax
	(
	String	value,
	String	type
	)
{
if (type == null)
	type = "";
if (value == null)
	value = NULL_VALUE;
type = type.toUpperCase ();
if (type.indexOf (INT_TYPE) >= 0 ||
	type.indexOf (REAL_TYPE) >= 0 ||
	type.indexOf (FLOAT_TYPE) >= 0 ||
	type.indexOf (DOUBLE_TYPE) >= 0 ||
	value.equals (NULL_VALUE))
	{
	if ((DEBUG & DEBUG_VALUE) != 0)
		System.out.println (">-< Database.Value_Syntax: " + value);
	return value;
	}
if ((DEBUG & DEBUG_VALUE) != 0)
	System.out.println (">-< Database.Value_Syntax: '" + value + "'");
return "'" + Special_to_Escape (value) + "'";
}

private static final String
	INT_TYPE	= "INT",
	REAL_TYPE	= "REAL",
	FLOAT_TYPE	= "FLOAT",
	DOUBLE_TYPE	= "DOUBLE";

/*==============================================================================
	Private helper methods
*/
private void connect_check ()
	throws Database_Exception
{
if (Data_Port == null)
	throw new Database_Exception
		(
		ID + NL
		+ "The database is not connected."
		);
}


}	//	End of class

