<?xml version="1.0"?>
<xsl:stylesheet xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
                version="1.0">

<!--
     This style sheet generates a brief summary of a PhotoML file. (A
     more detailed representation is provided by the 'detailed.xsl'
     style sheet developed by Oskar Ojala, who also helped clean up
     the output of this file.)

     Copyright © 2005-2011 Brendt Wohlberg <photoml@wohlberg.net>

     This is free software; you can redistribute it and/or modify it 
     under the terms of version 2 of the GNU General Public License 
     at http://www.gnu.org/licenses/gpl-2.0.txt.

     This software is distributed in the hope that it will be useful,
     but WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the 
     GNU General Public License for more details.
-->


<xsl:output method="xml" indent="yes" encoding="utf-8"
  doctype-public="-//W3C//DTD XHTML 1.0 Strict//EN"
  doctype-system="http://www.w3.org/TR/xhtml1/DTD/xhtml1-strict.dtd" />

<!-- Include the XSL encapsulating the CSS definition -->
<xsl:include href="cssinc.xsl"/>

<!-- Stylesheet parameter determining the primary font size -->
<xsl:param name="font-size">11</xsl:param>
<!-- Stylesheet parameter determining display of exposure details -->
<xsl:param name="disp-exposure">0</xsl:param>
<!-- Stylesheet parameter determining display of location map -->
<xsl:param name="disp-map">0</xsl:param>


<!-- Variable determining whether location map is to be displayed -->
<xsl:variable name="map-active" select="boolean(number($disp-map)) and 
                                        count(//location/coordinates)>0"/>


<!-- Key matching all digimage elements with specific group-id -->
<xsl:key name="gid" match="digimage" use="@group-id"/>
<!-- Key matching all digimage elements with specific group-id and frame-id -->
<xsl:key name="gfid" match="digimage" use="concat(@group-id,'-',@frame-id)"/>



<!-- Template matching the main document element -->
<xsl:template match="/photo">
  <html>
    <head>
      <title>Photo Metadata</title>

      <xsl:call-template name="include-stylesheet">
        <xsl:with-param name="font-size" select="$font-size"/>
      </xsl:call-template>

      <xsl:if test="$map-active">
        <xsl:call-template name="gmap-init"/>
      </xsl:if>

    </head>

    <body>
      <xsl:if test="$map-active">
        <xsl:attribute name="onload">initialize()</xsl:attribute>
      </xsl:if>

      <xsl:for-each select="sheet">
        <xsl:sort select="@id"/>
        <xsl:apply-templates select="."/>
      </xsl:for-each>

      <xsl:for-each select="roll">
        <xsl:sort select="@id"/>
        <xsl:apply-templates select="."/>
      </xsl:for-each>

       <xsl:for-each select="digital">
        <xsl:sort select="@id"/>
        <xsl:apply-templates select="."/>
      </xsl:for-each>

      <xsl:if test="$map-active">
        <div>
          <div id="map_canvas" style="height:400px;border:2px solid black;
                                      margin-bottom: 3em;"/>
        </div>
      </xsl:if>

      <xsl:call-template name="digimage-table"/>
    </body>
  </html>

</xsl:template>



<!-- Template matching roll elements -->
<xsl:template match="roll">

      <!-- Main heading includes the roll identity information -->
      <div class="gtitle">
        Film Roll Description: <xsl:call-template name="group-identity"/>
      </div>

      <xsl:call-template name="collection-details"/>

      <xsl:call-template name="group-properties-table"/>
      <p/>
      <xsl:call-template name="group-frames-table"/>
      <div class="gend"/>

</xsl:template>



<!-- Template matching digital elements -->
<xsl:template match="digital">

      <!-- Main heading includes the digital group identity information -->
      <div class="gtitle">
        Digital Image Description: <xsl:call-template name="group-identity"/>
      </div>

      <xsl:call-template name="collection-details"/>

      <xsl:call-template name="group-properties-table"/>
      <p/>
      <xsl:call-template name="group-frames-table"/>
      <div class="gend"/>

</xsl:template>



<!-- Template matching sheet elements -->
<xsl:template match="sheet">

      <!-- Main heading includes the sheet identity information -->
      <div class="gtitle">
        Sheet Film Description: <xsl:call-template name="group-identity"/>
      </div>

      <xsl:call-template name="collection-details"/>

      <xsl:call-template name="group-properties-table"/>
      <p/>
      <xsl:call-template name="group-frames-table">
        <xsl:with-param name="frame-col-flag" select="0"/>
      </xsl:call-template>
      <div class="gend"/>

</xsl:template>



<!-- Named template for constructing the group identity string -->
<xsl:template name="group-identity">
  <span class="gid">
    <xsl:if test="count((//roll|//sheet|//digital)[@id=current()/@id])=1">
      <xsl:attribute name="id">
        <xsl:value-of select="concat(name(.),'-',@id)"/>
      </xsl:attribute>
    </xsl:if>
    <xsl:value-of select="@id"/>
  </span>
</xsl:template>


<!-- Named template for displaying the details of any collections -->
<xsl:template name="collection-details">
  <xsl:if test="collection">
    <div class="cblock">
      <xsl:for-each select="collection">
        <span class="citem">
          <xsl:value-of select="@id"/>-<xsl:value-of select="@cgid"/>
          <xsl:if test="@fstfid|@lstfid">
            <xsl:text>[</xsl:text>
            <xsl:value-of select="@fstfid"/>-<xsl:value-of select="@lstfid"/>
            <xsl:text>]</xsl:text>
          </xsl:if>
          <xsl:text> </xsl:text>
        </span>
      </xsl:for-each>
    </div>
    <p/>
  </xsl:if>
</xsl:template>


<!-- Named template for constructing a table of information pertaining to 
     the whole group -->
<xsl:template name="group-properties-table">
  <table>
    <tr>
      <!-- Columns are only defined if the corresponding table entry 
           is non-null. -->
      <xsl:if test="film"><th>Film</th></xsl:if>
      <xsl:if test="processing/lab"><th>Laboratory</th></xsl:if>
      <xsl:if test="processing/date">
        <th>Process Date</th>
      </xsl:if>
      <xsl:if test="processing/tag"><th>Tag Number</th></xsl:if>
      <xsl:if test="load/date">
        <th>Load Date</th>
      </xsl:if>
      <xsl:if test="unload/date">
        <th>Unload Date</th>
      </xsl:if>
      <!-- List the photographer name if there is only a single
           photographer name, or if they are all equal -->
      <xsl:if test="count(descendant::photographer/name[.= following::photographer/name or count(following::photographer/name) = 0]) = count(descendant::photographer/name) and count(descendant::photographer/name) > 0">
        <th>Photographer</th>
      </xsl:if>
      <!-- List the camera body if there is only a single
           definition, or if they are all equal -->
      <xsl:if test="count(descendant::equipment/body[.= following::equipment/body or count(following::equipment/body) = 0]) = count(descendant::equipment/body) and count(descendant::equipment/body) > 0">
        <th>Camera Body</th>
      </xsl:if>
    </tr>
    <tr>
      <xsl:call-template name="format-film"/>
      
      <xsl:if test="processing/lab">
        <td align="center">
          <xsl:if test="processing/lab/name">
            <xsl:choose>
              <xsl:when test="processing/lab[@href]">
                <xsl:variable name="url" select="processing/lab/@href"/> 
                <a href="{$url}">
                  <xsl:value-of select="processing/lab/name"/>
                </a>
              </xsl:when>
              <xsl:otherwise>
                <xsl:value-of select="processing/lab/name"/>
              </xsl:otherwise>
            </xsl:choose>
            <xsl:if test="processing/lab/address">
              <xsl:text>, </xsl:text>
              <xsl:value-of select="processing/lab/address"/>
            </xsl:if>
          </xsl:if>
        </td>
      </xsl:if>
      
      <xsl:call-template name="non-empty-cell">
        <xsl:with-param name="expr" select="processing/date"/>
      </xsl:call-template>
      
      <xsl:call-template name="non-empty-cell">
        <xsl:with-param name="expr" select="processing/tag"/>
      </xsl:call-template>
      
      <xsl:call-template name="non-empty-cell">
        <xsl:with-param name="expr" select="load[position()=1]/date"/>
      </xsl:call-template>
      
      <xsl:call-template name="non-empty-cell">
        <xsl:with-param name="expr" select="unload[position()=last()]/date"/>
      </xsl:call-template>
      
      <!-- List the photographer name if only there is only a single
           photographer name, or if they are all equal -->
      <xsl:if test="count(descendant::photographer/name[.=following::photographer/name or count(following::photographer/name) = 0]) = count(descendant::photographer/name) and count(descendant::photographer/name) > 0">
        <td><xsl:value-of select="descendant::photographer/name"/></td>
      </xsl:if>
      <!-- List the camera body if only there is only a single
           definition, or if they are all equal -->
      <xsl:if test="count(descendant::equipment/body[.=following::equipment/body or count(following::equipment/body) = 0]) = count(descendant::equipment/body) and count(descendant::equipment/body) > 0">
        <td>
          <xsl:apply-templates select="descendant::equipment[1]/body"/>
        </td>
      </xsl:if>
    </tr>
  </table>
</xsl:template>



<!-- Named template for constructing a table with a row for each frame -->
<xsl:template name="group-frames-table">
  <xsl:param name="frame-col-flag" select="1"/>
  <table>
    <tr>
      <xsl:if test="$frame-col-flag=1">
        <th>Frame</th>
      </xsl:if>
      <th>Date</th>
      <!-- Include a location column only if a location is specified
           for at least one frame -->
      <xsl:if test="descendant::frame/scene/location">
        <th>Location</th>
      </xsl:if>
      <th>Description</th>
      <!-- Include an exposure column only if requested and exposure
           is specified for at least one frame -->
      <xsl:if test="boolean(number($disp-exposure)) and frame/exposure">
        <th>Exposure</th>
      </xsl:if>
    </tr>
    <xsl:for-each select="descendant::frame[not(ancestor::defaults)]">
      <tr>
        <xsl:if test="$frame-col-flag=1">
          <td>
            <xsl:value-of select="@id"/>
            <xsl:if test="@fn">
              <span class="fn">
                <xsl:text> (</xsl:text>
                <xsl:value-of select="@fn"/>
                <xsl:text>)</xsl:text>
              </span>
            </xsl:if>
          </td>
        </xsl:if>
        <td>
          <xsl:value-of select="scene/occasion/date"/>
          <xsl:if test="scene/occasion/time">
            <xsl:text> </xsl:text>
            <xsl:value-of select="scene/occasion/time"/>
          </xsl:if>
        </td>
        <xsl:call-template name="format-location"/>
        <td><xsl:value-of select="scene/description"/></td>
        <xsl:call-template name="format-exposure"/>
      </tr>
    </xsl:for-each>
  </table>
</xsl:template>


<!-- Template for formatting camera body descriptions -->
<xsl:template match="equipment/body">
  <xsl:value-of select="concat(make,' ',model)"/>
</xsl:template>


<!-- Named template for formatting the film description table cell -->
<xsl:template name="format-film">
  <xsl:if test="film">
    <td>
      <xsl:value-of select="film/make"/>
      <xsl:text> </xsl:text>
      <xsl:value-of select="film/name"/>
      <xsl:if test="film/speed">
        (ISO <xsl:value-of select="film/speed"/>)
      </xsl:if>
    </td>
  </xsl:if>
</xsl:template>



<!-- Named template for formatting the exposure description table cell -->
<xsl:template name="format-exposure">
  <xsl:if test="boolean(number($disp-exposure)) and 
                ancestor::*/frame/exposure">
    <td>
      <xsl:if test="exposure/aperture">
        f/<xsl:value-of select="exposure/aperture"/>
    </xsl:if>
    <xsl:text> </xsl:text>
    <xsl:if test="exposure/shutter">
      <xsl:value-of select="exposure/shutter"/>s
    </xsl:if>
    <xsl:text> </xsl:text>
    <xsl:if test="exposure/exp-comp">
      (<xsl:value-of select="exposure/exp-comp"/>)
    </xsl:if>
   </td>
  </xsl:if>
</xsl:template>



<!-- Named template for formatting the location description table cell -->
<xsl:template name="format-location">
  <xsl:if test="ancestor::*/frame/scene/location">
    <td>
      <xsl:call-template name="comma-seperated-list">
        <xsl:with-param name="expr" select="scene/location/*[name() != 'coordinates' and name()!='note']"/>
        <xsl:with-param name="reverse">1</xsl:with-param>
      </xsl:call-template>
      <xsl:if test="scene/location/coordinates">
        <xsl:variable name="dscr">
          <xsl:call-template name="comma-seperated-list">
            <xsl:with-param name="expr" select="scene/location/*[name() != 'coordinates' and name()!='note']"/>
            <xsl:with-param name="reverse">1</xsl:with-param>
          </xsl:call-template>
        </xsl:variable>
        <xsl:variable name="gmdscr">
          <xsl:if test="$dscr != ''">
            <xsl:value-of select="concat('+(',
                                  translate($dscr,' ()','+[]'),')')"/>
          </xsl:if>
        </xsl:variable>
        <xsl:variable name="gmap" select="concat('http://maps.google.com/maps?q=',scene/location/coordinates/latitude,'+',scene/location/coordinates/longitude,$gmdscr,'&amp;ll=',scene/location/coordinates/latitude,',',scene/location/coordinates/longitude,'&amp;z=14&amp;hl=en&amp;t=h&amp;iwloc=A')"/>
        <xsl:variable name="gctt" select="concat('Latitude: ',scene/location/coordinates/latitude, '&#176;  Longitude: ', scene/location/coordinates/longitude, '&#176;')"/>
        <span class="gc" title="{$gctt}">
          <xsl:text> </xsl:text>
          <a href="{$gmap}">
            <xsl:text>&#x2318;</xsl:text>
          </a>
        </span>
      </xsl:if>
    </td>
  </xsl:if>
</xsl:template>



<!-- Named template for constructing a table cell only if the content
     is non-null -->
<xsl:template name="non-empty-cell">
  <xsl:param name="expr"/>
  <xsl:if test="$expr">
    <td align="center">
      <xsl:value-of select="$expr"/>
    </td>
  </xsl:if>
</xsl:template>



<!-- Named template for formatting the children of an element as a
     comma separated list -->
<xsl:template name="comma-seperated-list">
  <xsl:param name="expr"/>
  <xsl:param name="reverse">0</xsl:param>
  <xsl:choose>
    <xsl:when test="$reverse = 0">
      <xsl:for-each select="$expr">
        <xsl:value-of select="."/>
        <xsl:if test="position() != last()">, </xsl:if>
      </xsl:for-each>
    </xsl:when>
    <xsl:otherwise>
      <xsl:for-each select="$expr">
        <xsl:sort data-type="number" order="descending" select="position()"/>
        <xsl:value-of select="."/>
        <xsl:if test="position() != last()">, </xsl:if>
      </xsl:for-each>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>



<!-- Named template for constructing the table of digimage elements -->
<xsl:template name="digimage-table">
  <!-- Construct the table if there are any digimage elements -->
  <xsl:if test="//photo//digimage[not(ancestor::frame)]">
    <!-- Table headings -->
    <div class="gtitle">Digital Image Details</div>
    <table>
      <tr>
        <th>Grp</th><th>Frm</th><th>Ver.</th>
        <th>Origin</th><th>Date</th><th>Format</th>
        <th>Width</th><th>Height</th><th>Bits</th>
      </tr>
      <!-- Iterate over each unique digimage group-id -->
      <xsl:for-each select="//digimage[generate-id(.)=generate-id(key('gid',@group-id)[1])]">
        <xsl:sort select="@group-id"/>
        <xsl:variable name="gid" select="@group-id"/>
        <tr>
          <td rowspan="{count(key('gid',@group-id)) + 1 + count(//digimage[generate-id(.)=generate-id(key('gfid',concat($gid,'-',@frame-id))[1])])}">
            <span id="{concat('digimage-',$gid)}">
              <xsl:value-of select="@group-id"/>
            </span>
          </td>
          <td colspan="8" style="display: none;"/>
        </tr>
        <!-- Iterate over each unique digimage frame-idx with a
             specific corresponding group-id -->
        <xsl:for-each select="//digimage[generate-id(.)=generate-id(key('gfid',concat($gid,'-',@frame-id))[1])]">
          <xsl:sort select="@frame-id"/>
          <xsl:variable name="fid" select="@frame-id"/>
          <tr>
            <td rowspan="{count(key('gfid',concat($gid,'-',@frame-id)))+1}">
              <xsl:value-of select="@frame-id"/>
            </td>
            <td colspan="7" style="display: none;"/>
          </tr>
          <!-- Iterate over each digimage with specified group-id and
               frame-id -->
          <xsl:for-each select="key('gfid',concat($gid,'-',$fid))">
            <xsl:sort select="@image-id"/>
            <xsl:variable name="iid" select="@image-id"/>
            <tr>
              <td>
                <span id="{generate-id(.)}">
                  <xsl:value-of select="@image-id"/>
                </span>
              </td>
              <xsl:apply-templates select="."/>
            </tr>
          </xsl:for-each>
        </xsl:for-each>
      </xsl:for-each>
    </table>
  </xsl:if>
</xsl:template>



<!-- Template for formatting digimage elements -->
<xsl:template match="digimage">
  <xsl:variable name="gid" select="@group-id"/>
  <xsl:variable name="fid" select="@frame-id"/>
  <td>
    <xsl:if test="history/origin">
      <xsl:choose>
        <xsl:when test="history/origin/source-image">
          <xsl:text>Source: Version </xsl:text>
          <xsl:variable name="pvid"
                        select="history/origin/source-image/@image-id"/>
          <xsl:variable name="parentid" 
                        select="generate-id(//digimage[@group-id=$gid and 
                                @frame-id = $fid and @image-id = $pvid])"/>
          <a>
            <xsl:if test="$parentid">
            <xsl:attribute name="href">#<xsl:value-of select="$parentid"/></xsl:attribute>
             </xsl:if>
             <xsl:value-of select="history/origin/source-image/@image-id"/>
          </a>
        </xsl:when>
        <xsl:when test="history/origin/scanner">
          <xsl:text>Scanner: </xsl:text>
          <xsl:value-of select="history/origin/scanner/hardware/make"/>
          <xsl:text> </xsl:text>
          <xsl:value-of select="history/origin/scanner/hardware/model"/>
        </xsl:when>
        <xsl:when test="history/origin/camera">
          <xsl:text>Camera: </xsl:text>
          <xsl:value-of select="history/origin/camera/hardware/make"/>
          <xsl:text> </xsl:text>
          <xsl:value-of select="history/origin/camera/hardware/model"/>
        </xsl:when>
      </xsl:choose>
    </xsl:if>
  </td>
  <td class="left">
    <xsl:value-of select="history/origin//date"/>
    <xsl:if test="history/origin//time">
      <xsl:text> </xsl:text>
      <xsl:value-of select="history/origin//time"/>
    </xsl:if>
  </td>
  <td class="center">
    <xsl:apply-templates select="properties/file-format"/>
  </td>
  <td class="right">
    <xsl:value-of select="properties/width"/>
  </td>
  <td class="right">
    <xsl:value-of select="properties/height"/>
  </td>
  <td class="right">
    <xsl:value-of select="properties/bit-depth"/>
  </td>
</xsl:template>



<!-- Template for formatting fileformat elements -->
<xsl:template match="file-format">
  <xsl:choose>
    <xsl:when test="@std">
      <xsl:value-of select="translate(@std,'bdefgijkmnpst','BDEFGIJKMNPST')"/>
    </xsl:when>
    <xsl:otherwise>
      <xsl:value-of select="."/>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>



<!-- Template for setting up Google Maps javascript -->
<xsl:template name="gmap-init">
  <script type="text/javascript" src="http://maps.google.com/maps/api/js?sensor=false"></script>
  <script type="text/javascript" src="http://google-maps-utility-library-v3.googlecode.com/svn/trunk/markerclusterer/src/markerclusterer.js"></script>
  <script type="text/javascript" src="http://google-maps-utility-library-v3.googlecode.com/svn/trunk/styledmarker/src/StyledMarker.js"></script>
  <script type="text/javascript">
    <xsl:text disable-output-escaping="yes">&#10;//&lt;![CDATA[
<![CDATA[
function initgmap(mrkdat) {
  //Initialise arrays for LatLng and Marker objects
  var latlng = new Array();
  var marker = new Array();
  //Initialise map
  var mapopt = {mapTypeId: google.maps.MapTypeId.HYBRID};
  var map = new google.maps.Map(document.getElementById("map_canvas"),mapopt);
  //Construct map bounds object
  var bounds = new google.maps.LatLngBounds();

  //Iterate over marker data entries, constructing corresponding
  //LatLng and Marker objects
  for (k=0; k < mrkdat.length; k++) {
    latlng[k] = new google.maps.LatLng(mrkdat[k]['lat']+2e-5*Math.random(),
                                       mrkdat[k]['lon']+2e-5*Math.random());
    //Construct marker title
    var ttl = mrkdat[k]['gtp'] + ': ' + mrkdat[k]['grp'] + ' frame: ' +
        mrkdat[k]['frm'];
    //Set marker colour and text according to rlv field
    var mcl = 'fe7569';
    var mtx = '';
    switch (mrkdat[k]['rlv']) {
    case 1: 
       mcl = 'ff3010';
       mtx = 'pt';
       break;
    case 2:
       mcl = 'c03010';
       mtx = 'ar';
       break;
    }
    //Construct marker
    marker[k] = new StyledMarker(
                    {styleIcon:new StyledIcon(StyledIconTypes.MARKER,
                                              {color: mcl, text: mtx}),
        position: latlng[k],
        map: map,
        title: ttl,
        draggable: true
    });
    //If position uncertainty specified, construct circle to indicate it
    if (mrkdat[k]['unc'] > 0) {
       var circle = new google.maps.Circle({
         map: map,
         radius: mrkdat[k]['unc'],
         fillColor: '#AA0000',
         fillOpacity: 0.2,
         strokeOpacity: 0.5,
         strokeWeight: 1
       });
       circle.bindTo('center', marker[k], 'position');
    }
    //Extend map bounds for current marker
    bounds.extend(latlng[k]);
  }

  //Fit map to markers
  map.fitBounds(bounds);

  //Function providing effectively static maxzoom variable
  function maxzoom(mxz) {
    if (typeof mxz == 'number') {
      maxzoom.value = mxz;
    }
    return maxzoom.value;
  }

  //Use MaxZoomService to ensure that zoom level is not too
  //high for available map data within map bounds
  var maxZoomService = new google.maps.MaxZoomService();
  var mpc = bounds.getCenter();
  maxZoomService.getMaxZoomAtLatLng(mpc, function(response) {
    if (response.status != google.maps.MaxZoomStatus.OK) {
      maxzoom(17);
    } else {
      var mxz = maxzoom(response.zoom);
      if (map.getZoom() > mxz) {
        map.setZoom(mxz);
      }
    }
  });
 
  //Construct InfoWindow object
  var infowindow = new google.maps.InfoWindow({content: 'Null'});
  //Define function setting info window content depending on user
  //selected marker
  function openinfowin(marker, m) {
  var pos = marker.getPosition();
  var iws = '<p><b>' + mrkdat[m]['gtp'] + ': </b>' + 
            mrkdat[m]['grp'] + '<b> frame: </b>' +
            mrkdat[m]['frm'] + '</p>';
  if (mrkdat[m]['dat'] != '') {
    iws = iws + '<p><b>date: </b>' + mrkdat[m]['dat'] + '</p>';
  }
  infowindow.setContent(iws+'<p><b>coords:</b> '+
                        pos.lat().toFixed(6)+', '+
                        pos.lng().toFixed(6)+'</p>');
    infowindow.open(map, marker);
  }
  //Info window is to be closed if the map is clicked 
  google.maps.event.addListener(map, 'click', function () {
    infowindow.close()});

  //Iterate over markers, setting click response function for each
  for (k=0; k < mrkdat.length;k++) {
    (function(m) {
      google.maps.event.addListener(marker[m], 'click', function() {
        openinfowin(marker[m], m);
        });
     }(k));
  }
   
  //Add marker array to cluster manager
  var markerCluster = new MarkerClusterer(map, marker, {maxZoom: 17});

  //If map clicked, center at click position 
  google.maps.event.addListener(map,'click',
    function(event){
      map.setCenter(event.latLng);
    }
  );

  //Workaround for bug: 
  //http://code.google.com/p/google-maps-utility-library-v3/issues/detail?id=11
  google.maps.event.addListener(map,'zoom_changed',
    function(){
      if (map.getZoom() > maxzoom()) {
         map.setZoom(maxzoom()-1);
         map.setZoom(maxzoom());
      }
    }
  );
  google.maps.event.addListener(map,'center_changed',
    function(){
      var mpc = map.getCenter();
      maxZoomService.getMaxZoomAtLatLng(mpc, function(response) {
      if (response.status == google.maps.MaxZoomStatus.OK) {
        maxzoom(response.zoom);
      }
    });
  });

}

function initialize() {
    //Define marker data
    initgmap([
]]></xsl:text>
  <xsl:for-each select="//coordinates">
    <xsl:variable name="relevance">
      <xsl:choose>
        <xsl:when test="@relevance='point'">1</xsl:when>
        <xsl:when test="@relevance='area'">2</xsl:when>
        <xsl:otherwise>0</xsl:otherwise>
      </xsl:choose>
    </xsl:variable>
    <xsl:variable name="uncertainty">
      <xsl:choose>
        <xsl:when test="@uncertainty">
          <xsl:value-of select="@uncertainty"/>
        </xsl:when>
        <xsl:otherwise>0</xsl:otherwise>
      </xsl:choose>
    </xsl:variable>
    <xsl:variable name="date">
      <xsl:if test="ancestor::frame/scene/occasion/date">
        <xsl:value-of select="ancestor::frame/scene/occasion/date"/>
        <xsl:if test="ancestor::frame/scene/occasion/time">
          <xsl:text>T</xsl:text>
          <xsl:value-of select="ancestor::frame/scene/occasion/time"/>
        </xsl:if>
      </xsl:if>
    </xsl:variable>
    <xsl:text>{lat: </xsl:text>
    <xsl:value-of select="latitude"/>
    <xsl:text>, lon: </xsl:text>
    <xsl:value-of select="longitude"/>
    <xsl:text>, rlv: </xsl:text>
    <xsl:value-of select="$relevance"/>
    <xsl:text>, unc: </xsl:text>
    <xsl:value-of select="$uncertainty"/>
    <xsl:text>, gtp: '</xsl:text>
    <xsl:value-of select="name(ancestor::roll|ancestor::sheet|
                          ancestor::digital)"/>
    <xsl:text>'</xsl:text>
    <xsl:text>, grp: '</xsl:text>
    <xsl:value-of select="(ancestor::roll|ancestor::sheet|
                          ancestor::digital)/@id"/>
    <xsl:text>'</xsl:text>
    <xsl:text>, frm: '</xsl:text>
    <xsl:value-of select="ancestor::frame/@id"/>
    <xsl:text>'</xsl:text>
    <xsl:text>, dat: '</xsl:text>
    <xsl:value-of select="$date"/>
    <xsl:text>'</xsl:text>
    <xsl:text>}</xsl:text>
    <xsl:if test="position() != last()">,</xsl:if>
    <xsl:text>&#10;</xsl:text>
  </xsl:for-each>
<![CDATA[
      ]);
}]]>
    <xsl:text disable-output-escaping="yes">//]]&gt;&#10;</xsl:text>
  </script>
</xsl:template>



</xsl:stylesheet>
