///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <CGAL/Exact_predicates_inexact_constructions_kernel.h>
#include <CGAL/Weighted_alpha_shape_euclidean_traits_3.h>
#include <CGAL/Delaunay_triangulation_3.h>
#include <CGAL/Triangulation_hierarchy_3.h>
#include <CGAL/Triangulation_utils_3.h>
#include <CGAL/Alpha_shape_3.h>
#include <CGAL/Unique_hash_map.h>

#include <CGAL/Surface_mesh_simplification/HalfedgeGraph_Polyhedron_3.h>
#include <CGAL/Surface_mesh_simplification/Policies/Edge_collapse/Count_ratio_stop_predicate.h>
#include <CGAL/Surface_mesh_simplification/edge_collapse.h>
#include <CGAL/Polyhedron_incremental_builder_3.h>

#include <CGAL/Subdivision_method_3.h>

#include <core/Core.h>
#include <core/gui/RolloutContainer.h>
#include <core/scene/animation/AnimManager.h>
#include <core/data/DataSetManager.h>
#include <core/scene/ObjectNode.h>
#include <core/scene/objects/geometry/MeshObject.h>
#include <core/utilities/ProgressIndicator.h>
#include "AlphaShapeUtility.h"
#include <atomviz/atoms/AtomsObject.h>
#include <atomviz/modifier/deleteatoms/DeleteAtomsModifier.h>
#include <atomviz/modifier/selection/SelectExpressionModifier.h>

namespace CrystalAnalysis {

IMPLEMENT_PLUGIN_CLASS(AlphaShapeUtility, UtilityPlugin)

/******************************************************************************
* Initializes the utility applet.
******************************************************************************/
AlphaShapeUtility::AlphaShapeUtility() : UtilityPlugin(), rollout(NULL)
{
}

/******************************************************************************
* Shows the UI of the utility in the given RolloutContainer.
******************************************************************************/
void AlphaShapeUtility::openUtility(RolloutContainer* container, const RolloutInsertionParameters& rolloutParams)
{
	// Create a rollout.
	rollout = new QWidget();
	container->addRollout(rollout, tr("Alpha Shape Utility"), rolloutParams);

    // Create the rollout contents.
	QVBoxLayout* layout = new QVBoxLayout(rollout);
	layout->setContentsMargins(4,4,4,4);
	layout->setSpacing(4);

	QGridLayout* contentLayout = new QGridLayout();
	contentLayout->setContentsMargins(0,0,0,0);
	contentLayout->setSpacing(0);
	contentLayout->setColumnStretch(1, 1);
	layout->addLayout(contentLayout);

	contentLayout->addWidget(new QLabel(tr("Alpha value:"), rollout), 0, 0);
	QLineEdit* alphaValueBox = new QLineEdit(rollout);
	contentLayout->addWidget(alphaValueBox, 0, 1);
	alphaValueSpinner = new SpinnerWidget(rollout);
	alphaValueSpinner->setTextBox(alphaValueBox);
	alphaValueSpinner->setUnit(UNITS_MANAGER.getUnit(PLUGINCLASSINFO(WorldParameterUnit)));
	alphaValueSpinner->setMinValue(0.0);
	alphaValueSpinner->setFloatValue(30.0);
	contentLayout->addWidget(alphaValueSpinner, 0, 2);

	smoothCheckBox = new QCheckBox(tr("Smooth mesh"), rollout);
	contentLayout->addWidget(smoothCheckBox, 1, 0, 1, 3);

	processAllClustersCheckBox = new QCheckBox(tr("Process all clusters"), rollout);
	contentLayout->addWidget(processAllClustersCheckBox, 2, 0, 1, 3);

	deleteProcessedAtomsCheckBox = new QCheckBox(tr("Delete processed atoms"), rollout);
	deleteProcessedAtomsCheckBox->setChecked(true);
	contentLayout->addWidget(deleteProcessedAtomsCheckBox, 3, 0, 1, 3);

	QPushButton* createShapeButton = new QPushButton(tr("Create Alpha Shape"), rollout);
	layout->addWidget(createShapeButton);
	connect(createShapeButton, SIGNAL(clicked(bool)), this, SLOT(onCreateShape()));
}

/******************************************************************************
* Removes the UI of the utility from the rollout container.
******************************************************************************/
void AlphaShapeUtility::closeUtility(RolloutContainer* container)
{
	delete rollout;
}

/******************************************************************************
* Is called when the user presses the Create Shape button.
******************************************************************************/
void AlphaShapeUtility::onCreateShape()
{
	UNDO_MANAGER.beginCompoundOperation(tr("Create alpha shape"));
	try {
		FloatType alpha = alphaValueSpinner->floatValue();
		bool smooth = smoothCheckBox->isChecked();
		bool processAllClusters = processAllClustersCheckBox->isChecked();
		bool deleteProcessedAtoms = deleteProcessedAtomsCheckBox->isChecked();

		// Retrieve the selected atoms object.
		AtomsObject::SmartPtr atoms = NULL;
		ObjectNode::SmartPtr atomsNode = NULL;

		Q_FOREACH(SceneNode* node, DATASET_MANAGER.currentSet()->selection()->nodes()) {
			ObjectNode* objNode = dynamic_object_cast<ObjectNode>(node);
			if(!objNode) continue;

			const PipelineFlowState& flowState = objNode->evalPipeline(ANIM_MANAGER.time());
			atoms = dynamic_object_cast<AtomsObject>(flowState.result());
			if(atoms) {
				atomsNode = objNode;
				break;
			}
		}
		if(!atoms)
			throw Exception(tr("No atoms object selected. Please select an atoms object first before pressing the button."));

		// Extract the selected atoms from the AtomsObject.
		DataChannel* posChannel = atoms->getStandardDataChannel(DataChannel::PositionChannel);
		if(!posChannel) throw Exception(tr("Input object does not contain a position data channel."));

		TimeInterval iv;
		QVector<FloatType> atomRadii = atoms->getAtomRadii(ANIM_MANAGER.time(), iv);

		QMap<int, QVector<Point3> > clusters;
		if(processAllClusters) {
			DataChannel* clusterChannel = atoms->getStandardDataChannel(DataChannel::ClusterChannel);
			if(!clusterChannel) throw Exception(tr("Input object does not contain a cluster data channel."));
			for(size_t i = 0; i < clusterChannel->size(); i++) {
				if(clusterChannel->getInt(i) < 0) continue;
				QVector<Point3>& cluster = clusters[clusterChannel->getInt(i)];
				const Point3& center = posChannel->getPoint3(i);
				if(!smooth) {
					cluster.push_back(center);
				}
				else {
					FloatType radius = atomRadii[i];
					cluster.push_back(Point3(center.X + radius, center.Y, center.Z));
					cluster.push_back(Point3(center.X - radius, center.Y, center.Z));
					cluster.push_back(Point3(center.X, center.Y + radius, center.Z));
					cluster.push_back(Point3(center.X, center.Y - radius, center.Z));
					cluster.push_back(Point3(center.X, center.Y, center.Z + radius));
					cluster.push_back(Point3(center.X, center.Y, center.Z - radius));
				}
			}
		}
		else {
			DataChannel* selectionChannel = atoms->getStandardDataChannel(DataChannel::SelectionChannel);
			if(!selectionChannel) throw Exception(tr("Input object does not contain a selection data channel. Please select some atoms first for which the alpha shape should be created."));
			QVector<Point3>& cluster = clusters[0];
			for(size_t i = 0; i < posChannel->size(); i++) {
				if(selectionChannel->getInt(i)) {
					const Point3& center = posChannel->getPoint3(i);
					if(!smooth) {
						cluster.push_back(center);
					}
					else {
						FloatType radius = atomRadii[i];
						cluster.push_back(Point3(center.X + radius, center.Y, center.Z));
						cluster.push_back(Point3(center.X - radius, center.Y, center.Z));
						cluster.push_back(Point3(center.X, center.Y + radius, center.Z));
						cluster.push_back(Point3(center.X, center.Y - radius, center.Z));
						cluster.push_back(Point3(center.X, center.Y, center.Z + radius));
						cluster.push_back(Point3(center.X, center.Y, center.Z - radius));
					}
				}
			}
			if(cluster.empty())
				throw Exception(tr("No atoms are selected. Please select some atoms first for which the alpha shape should be created."));
		}

		for(QMap<int, QVector<Point3> >::const_iterator cluster = clusters.constBegin(); cluster != clusters.constEnd(); ++cluster) {
			// Create alpha shape mesh
			MeshObject::SmartPtr meshObj = new MeshObject();
			if(!createAlphaShape(alpha, smooth, cluster.value(), meshObj->mesh()))
				break;	// Aborted by user.

			// Create a new scene node that contains the generated mesh.
			ObjectNode::SmartPtr objNode = new ObjectNode();
			objNode->setSceneObject(meshObj);

			SceneRoot* scene = DATASET_MANAGER.currentSet()->sceneRoot();
			objNode->setName(scene->makeNameUnique(tr("AlphaShape%1").arg(cluster.key())));

			// Position the new mesh node at the original position of the atoms.
			TimeInterval iv;
			const AffineTransformation& tm = atomsNode->getWorldTransform(ANIM_MANAGER.time(), iv);
			objNode->transformationController()->setValue(ANIM_MANAGER.time(), tm);
			objNode->setObjectTransform(atomsNode->objectTransform());

			// Insert node into scene.
			scene->addChild(objNode);

			DATASET_MANAGER.currentSet()->selection()->clear();
			DATASET_MANAGER.currentSet()->selection()->add(objNode);
		}

		if(deleteProcessedAtoms) {
			if(processAllClusters) {
				// Select cluster atoms.
				SelectExpressionModifier::SmartPtr selectMod = new SelectExpressionModifier();
				selectMod->setExpression("Cluster>=0");
				atomsNode->applyModifier(selectMod);
			}
			// Appply a delete atoms modifier to remove source atoms.
			DeleteAtomsModifier::SmartPtr delAtomsMod = new DeleteAtomsModifier();
			atomsNode->applyModifier(delAtomsMod);
		}
	}
	catch(const Exception& ex) {
		ex.showError();
	}
	UNDO_MANAGER.endCompoundOperation();
}

template<class Alpha_shape_3, class HDS>
class Alpha_shape_to_polyhedron : public CGAL::Modifier_base<HDS> {
public:
	Alpha_shape_3& as;
	bool isValid;
	Alpha_shape_to_polyhedron(Alpha_shape_3& shape) : as(shape) {}
    void operator()(HDS& hds) {

    	// Count vertices.
		int number_of_vertices = 0;
		CGAL::Unique_hash_map<typename Alpha_shape_3::Vertex_handle, int> V;
		typename Alpha_shape_3::Alpha_shape_vertices_iterator vit,
			vit_begin = as.Alpha_shape_vertices_begin(),
			vit_end = as.Alpha_shape_vertices_end();
		for(vit = vit_begin; vit != vit_end; ++vit) {
			V[*vit] = number_of_vertices++;
		}

		// Count facets.
		typename Alpha_shape_3::Alpha_shape_facets_iterator fit,
			fit_begin = as.Alpha_shape_facets_begin(),
			fit_end = as.Alpha_shape_facets_end();
		int number_of_facets = 0;
		for(fit = fit_begin; fit != fit_end; ++fit)
			number_of_facets++;

    	CGAL::Polyhedron_incremental_builder_3<HDS> B(hds, true);
        B.begin_surface(number_of_vertices, number_of_facets);
        typedef typename HDS::Vertex   Vertex;
        typedef typename Vertex::Point Point;
		// Transfer vertices.
		for(vit = vit_begin; vit != vit_end; ++vit)
        	B.add_vertex((*vit)->point());
		// Transfer facets.
		for(fit = fit_begin; fit != fit_end; ++fit) {
			typename Alpha_shape_3::Cell_handle c = fit->first;
			int i = fit->second;
			OVITO_ASSERT(as.classify(*fit) == Alpha_shape_3::REGULAR);
			int i0 = CGAL::Triangulation_utils_3::vertex_triple_index(i,0);
			int i1 = CGAL::Triangulation_utils_3::vertex_triple_index(i,1);
			int i2 = CGAL::Triangulation_utils_3::vertex_triple_index(i,2);
			// the following ensures that regular facets are output
			// in ccw order
			if(as.classify(c) == Alpha_shape_3::INTERIOR)
				swap(i0, i1);
			OVITO_ASSERT(V.is_defined(c->vertex(i0)));
			OVITO_ASSERT(V.is_defined(c->vertex(i1)));
			OVITO_ASSERT(V.is_defined(c->vertex(i2)));
			B.begin_facet();
        	B.add_vertex_to_facet(V[c->vertex(i0)]);
        	B.add_vertex_to_facet(V[c->vertex(i1)]);
        	B.add_vertex_to_facet(V[c->vertex(i2)]);
        	B.end_facet();
        }
        B.end_surface();
        isValid = !B.error();
    }
};


/******************************************************************************
* Calls the CGAL library to build the slpha shape.
******************************************************************************/
bool AlphaShapeUtility::createAlphaShape(FloatType alpha, bool smooth, const QVector<Point3>& points, TriMesh& mesh)
{
	typedef CGAL::Exact_predicates_inexact_constructions_kernel K;

	typedef CGAL::Alpha_shape_vertex_base_3<K>               Vb;
	typedef CGAL::Triangulation_hierarchy_vertex_base_3<Vb>  Vbh;
	typedef CGAL::Alpha_shape_cell_base_3<K>                 Fb;
	typedef CGAL::Triangulation_data_structure_3<Vbh,Fb>     Tds;
	typedef CGAL::Delaunay_triangulation_3<K,Tds>            Delaunay;
	typedef CGAL::Triangulation_hierarchy_3<Delaunay>        Delaunay_hierarchy;
	typedef CGAL::Alpha_shape_3<Delaunay_hierarchy>          Alpha_shape_3;

	typedef K::Point_3                                  Point;
	typedef Alpha_shape_3::Alpha_iterator               Alpha_iterator;
	typedef Alpha_shape_3::NT                           NT;
	typedef Alpha_shape_3::Facet 						Facet;
	typedef Alpha_shape_3::Cell  						Cell;
	typedef Alpha_shape_3::Vertex 						Vertex;
	typedef Alpha_shape_3::Cell_handle  				Cell_handle;
	typedef Alpha_shape_3::Vertex_handle 				Vertex_handle;

	typedef CGAL::Polyhedron_3<K>                               Polyhedron;

	// Insert all points into delaunay triangulation structure.
	ProgressIndicator progress(tr("Computing delaunay triangulation for %1 points.").arg(points.size()), points.size());
	Delaunay_hierarchy dt;
	for(int i=0; i<points.size(); i++) {
		const Point3& p = points[i];
		dt.insert(Point(p.X, p.Y, p.Z));

		if((i % 1000) == 0) {
			progress.setValue(i);
			if(progress.isCanceled())
				return false;
		}
	}
	progress.setMaximum(0);

	// compute alpha shape.
	progress.setLabelText(tr("Computing alpha shape."));
	if(progress.isCanceled()) return false;
	Alpha_shape_3 as(dt);
	as.set_alpha(alpha);
	if(progress.isCanceled()) return false;

	if(!smooth) {
		// Convert alpha shape to triangle mesh.
		progress.setLabelText(tr("Building triangle mesh."));
		if(progress.isCanceled()) return false;

		// Finite vertices coordinates.
		Alpha_shape_3::Alpha_shape_vertices_iterator vit,
			vit_begin = as.Alpha_shape_vertices_begin(),
			vit_end = as.Alpha_shape_vertices_end();

		CGAL::Unique_hash_map<Vertex_handle, int> V;
		int number_of_vertices = 0;
		for(vit = vit_begin; vit != vit_end; ++vit) {
			V[*vit] = number_of_vertices++;
		}

		// Count facets.
		Alpha_shape_3::Alpha_shape_facets_iterator fit,
			fit_begin = as.Alpha_shape_facets_begin(),
			fit_end = as.Alpha_shape_facets_end();

		int number_of_facets = 0;
		for(fit = fit_begin; fit != fit_end; ++fit)
			number_of_facets++;

		mesh.clearMesh();
		mesh.setVertexCount(number_of_vertices);
		mesh.setFaceCount(number_of_facets);

		// Transfer mesh vertices.
		QVector<Point3>::iterator v = mesh.vertices().begin();
		for(vit = vit_begin; vit != vit_end; ++vit, ++v) {
			v->X = (*vit)->point().x();
			v->Y = (*vit)->point().y();
			v->Z = (*vit)->point().z();
		}

		QVector<TriMeshFace>::iterator f = mesh.faces().begin();
		for(fit = fit_begin; fit != fit_end; ++fit, ++f) {

			Cell_handle c = fit->first;
			int i = fit->second;
			OVITO_ASSERT(as.classify(*fit) == Alpha_shape_3::REGULAR);
			int i0 = CGAL::Triangulation_utils_3::vertex_triple_index(i,0);
			int i1 = CGAL::Triangulation_utils_3::vertex_triple_index(i,1);
			int i2 = CGAL::Triangulation_utils_3::vertex_triple_index(i,2);
			// the following ensures that regular facets are output
			// in ccw order
			if(as.classify(c) == Alpha_shape_3::INTERIOR)
				swap(i0, i1);
			OVITO_ASSERT(V.is_defined(c->vertex(i0)));
			OVITO_ASSERT(V.is_defined(c->vertex(i1)));
			OVITO_ASSERT(V.is_defined(c->vertex(i2)));
			f->setVertex(0, V[c->vertex(i0)]);
			f->setVertex(1, V[c->vertex(i1)]);
			f->setVertex(2, V[c->vertex(i2)]);
		}
		mesh.invalidateFaces();
		mesh.invalidateVertices();
	}
	else {

		progress.setLabelText(tr("Simplifying alpha shape mesh."));
		if(progress.isCanceled()) return false;

		Polyhedron p;
		Alpha_shape_to_polyhedron<Alpha_shape_3, Polyhedron::HalfedgeDS> builder(as);
		p.delegate(builder);
		if(!builder.isValid)
			throw Exception(tr("Cannot create a smoothed mesh for the input atoms. The resulting alpha shape has an invalid topology preventing it from being smoothed."));

		CGAL::Surface_mesh_simplification::Count_ratio_stop_predicate<Polyhedron> stop(0.1);
		int r = CGAL::Surface_mesh_simplification::edge_collapse(p, stop,
					CGAL::vertex_index_map(boost::get(CGAL::vertex_external_index,p))
						  .edge_index_map(boost::get(CGAL::edge_external_index,p))
					);

		progress.setLabelText(tr("Smoothing alpha shape mesh."));
		if(progress.isCanceled()) return false;
		int d = 1;
		CGAL::Subdivision_method_3::CatmullClark_subdivision(p,d);

		progress.setLabelText(tr("Converting mesh representation."));
		if(progress.isCanceled()) return false;

		typedef Polyhedron::Vertex_const_iterator                  VCI;
		typedef Polyhedron::Facet_const_iterator                   FCI;
		typedef Polyhedron::Halfedge_around_facet_const_circulator HFCC;

		mesh.clearMesh();
		mesh.setVertexCount(p.size_of_vertices());

		// Transfer vertices.
		QVector<Point3>::iterator v = mesh.vertices().begin();
		for(VCI vi = p.vertices_begin(); vi != p.vertices_end(); ++vi, ++v) {
			v->X = vi->point().x();
			v->Y = vi->point().y();
			v->Z = vi->point().z();
		}
		typedef CGAL::Inverse_index<VCI> Index;
		Index index(p.vertices_begin(), p.vertices_end());

		// Transfer faces.
		for(FCI fi = p.facets_begin(); fi != p.facets_end(); ++fi) {
			HFCC hc = fi->facet_begin();
			HFCC hc_end = hc;
			size_t n = circulator_size(hc);
			OVITO_ASSERT(n >= 3);
			if(n < 3) continue;
			int v0 = index[VCI(hc->vertex())]; ++hc;
			int v1 = index[VCI(hc->vertex())]; ++hc;
			int v2 = index[VCI(hc->vertex())]; ++hc;
			for(;;) {
				TriMeshFace& f = mesh.addFace();
				f.setVertices(v0, v1, v2);
				f.setSmoothingGroup(1);
				if(hc == hc_end) break;
				v1 = v2;
				v2 = index[VCI(hc->vertex())]; ++hc;
			}
		}
		mesh.invalidateFaces();
		mesh.invalidateVertices();
	}

	return true;
}


};
