///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <base/Base.h>
#include <base/linalg/Rotation.h>
#include <base/linalg/AffineTransformation.h>
#include <base/linalg/Vector3.h>
#include <base/linalg/Quaternion.h>

namespace Base {

/// This dummy instance should be passed to the Rotation class constructor to initialize it to zero.
BASE_DLLEXPORT NullRotation NULL_ROTATION;

/******************************************************************************
* Initializes the object from rotational part of the matrix.
******************************************************************************/
Rotation::Rotation(const AffineTransformation& tm)
{
	axis.X = (tm(2,1) - tm(1,2));
	axis.Y = (tm(0,2) - tm(2,0));
	axis.Z = (tm(1,0) - tm(0,1));
	if(axis == NULL_VECTOR) {
		axis.Z = 1.0;
		angle = 0.0;
	}
	else {
		FloatType trace = (tm(0,0) + tm(1,1) + tm(2,2) - 1.0);
		FloatType s = Length(axis);
		axis /= s;
		angle = atan2(s, trace);
	}
}

/******************************************************************************
* Constructs a rotation that rotates one vector into a second vector.
******************************************************************************/
Rotation::Rotation(const Vector3& a, const Vector3& b)
{
	Vector3 an = Normalize(a);
	Vector3 bn = Normalize(b);
	FloatType cos = DotProduct(an, bn);
	if(cos > 1.0 - FLOATTYPE_EPSILON) {
		angle = 0.0;
		axis = Vector3(0,0,1);
	}
	else if(cos < -1.0 + FLOATTYPE_EPSILON) {
		angle = FLOATTYPE_PI;
		axis = Vector3(0,0,1);
	}
	else {
		angle = acos(cos);
		axis = Normalize(CrossProduct(a, b));
	}
	OVITO_ASSERT(bn.equals(Matrix3::rotation(*this) * an, FLOATTYPE_EPSILON));
}

inline Vector3 interpolateAxis(FloatType time, const Vector3& axis0, const Vector3& axis1)
{
	// assert:  axis0 and axis1 are unit length
	// assert:  DotProduct(axis0, axis1) >= 0
	// assert:  0 <= time <= 1

	FloatType cos = DotProduct(axis0, axis1);  // >= 0 by assertion
	OVITO_ASSERT(cos >= 0.0);
	if(cos > 1.0) cos = 1.0; // round-off error might create problems in acos call

	FloatType angle = acos(cos);
	FloatType invSin = 1.0 / sin(angle);
	FloatType timeAngle = time * angle;
	FloatType coeff0 = sin(angle - timeAngle) * invSin;
	FloatType coeff1 = sin(timeAngle) * invSin;

	return (coeff0 * axis0 + coeff1 * axis1);
}

inline Quaternion slerpExtraSpins(FloatType t, const Quaternion& p, const Quaternion& q, int iExtraSpins)
{
	FloatType fCos = DotProduct(p,q);
	OVITO_ASSERT(fCos >= 0.0);

	// Numerical round-off error could create problems in call to acos.
	if(fCos < -1.0) fCos = -1.0;
	else if(fCos > 1.0) fCos = 1.0;

	FloatType fAngle = acos(fCos);
	FloatType fSin = sin(fAngle);  // fSin >= 0 since fCos >= 0

	if(fSin < 0.001) {
		return p;
	}
	else {
		FloatType fPhase = FLOATTYPE_PI * (FloatType)iExtraSpins * t;
		FloatType fInvSin = 1.0 / fSin;
		FloatType fCoeff0 = sin((1.0f - t) * fAngle - fPhase) * fInvSin;
		FloatType fCoeff1 = sin(t * fAngle + fPhase) * fInvSin;
		return Quaternion(fCoeff0*p.X + fCoeff1*q.X, fCoeff0*p.Y + fCoeff1*q.Y,
		                        fCoeff0*p.Z + fCoeff1*q.Z, fCoeff0*p.W + fCoeff1*q.W);
	}
}


/******************************************************************************
* Interpolates between the two rotation structures and handles multiple revolutions.
******************************************************************************/
Quaternion Rotation::interpolate(const Rotation& rot1, const Rotation& _rot2, FloatType t)
{
	OVITO_ASSERT(t >= 0.0 && t <= 1.0);

	Rotation rot2;
	if(DotProduct(rot1.axis, _rot2.axis) < 0.0) {
		rot2.angle = -_rot2.angle;
		rot2.axis = -_rot2.axis;
	}
	else rot2 = _rot2;

	Quaternion q1 = (Quaternion)rot1;
	Quaternion q2 = (Quaternion)rot2;

	// Eliminate any non-acute angles between successive quaternions. This
	// is done to prevent potential discontinuities that are the result of
	// invalid intermediate value quaternions.
	if(DotProduct(q1,q2) < 0.0) {
		q2 = -q2;
	}

	// Clamp identity quaternions so that |w| <= 1 (avoids problems with
	// call to acos in SlerpExtraSpins).
	if(q1.W < -1.0) q1.W = -1.0; else if(q1.W > 1.0) q1.W = 1.0;
	if(q2.W < -1.0) q2.W = -1.0; else if(q2.W > 1.0) q2.W = 1.0;

	// Determine interpolation type, compute extra spins, and adjust angles accordingly.
	FloatType fDiff = rot1.angle - rot2.angle;
	int iExtraSpins;
	if(abs(fDiff) < FLOATTYPE_PI*2.0) {
		return Quaternion::interpolate(q1, q2, t);
	}
	else {
		iExtraSpins = (int)(fDiff/(FLOATTYPE_PI*2.0));

		if(rot1.axis.equals(rot2.axis, FLOATTYPE_EPSILON)) {
			Rotation result(rot1.axis, (1.0f-t) * rot1.angle + t * rot2.angle);
			return (Quaternion)result;
		}
		else if(rot1.angle != 0.0)
			return slerpExtraSpins(t, q1, q2, iExtraSpins);
		else {
			Rotation result(interpolateAxis(t, rot1.axis, rot2.axis), (1.0 - t) * rot1.angle + t * rot2.angle);
			return (Quaternion)result;
		}
	}
}

/******************************************************************************
* Interpolates between the two rotations using spherical quadratic interpolation.
******************************************************************************/
Quaternion Rotation::interpolateQuad(const Rotation& rot1, const Rotation& rot2, const Rotation& out, const Rotation& in, FloatType t)
{
	Quaternion slerpP = interpolate(rot1, rot2, t);
	Quaternion slerpQ = interpolate(out, in, t);
	FloatType Ti = 2.0*t*(1.0-t);
	return Quaternion::interpolate(slerpP, slerpQ, Ti);
}

};	// End of namespace Base
