//*******************************************************************
//
// License:  See top level LICENSE.txt file.
//
// Author:  David Burken
//
// Description:  Command line application for copying images "image copy".
// Can be used to cut images, convert formats.  Works in image space
// (no resampler).
//
//*******************************************************************
//  $Id: icp.cpp 13641 2008-10-01 13:02:40Z gpotts $
#include <iostream>
#include <algorithm>
#include <iterator>
#include <exception>

#include <ossim/base/ossimConstants.h>
#include <ossim/base/ossimKeywordNames.h>
#include <ossim/base/ossimTrace.h>
#include <ossim/base/ossimStdOutProgress.h>
#include <ossim/base/ossimFilename.h>
#include <ossim/base/ossimKeywordlist.h>
#include <ossim/base/ossimDrect.h>
#include <ossim/base/ossimImageTypeLut.h>
#include <ossim/imaging/ossimJpegWriter.h>
#include <ossim/imaging/ossimImageHandler.h>
#include <ossim/imaging/ossimRLevelFilter.h>
#include <ossim/imaging/ossimImageSource.h>
#include <ossim/imaging/ossimImageHandlerRegistry.h>
#include <ossim/imaging/ossimImageWriterFactoryRegistry.h>
#include <ossim/imaging/ossimImageWriterFactory.h>
#include <ossim/imaging/ossimImageFileWriter.h>
#include <ossim/imaging/ossimCacheTileSource.h>
#include <ossim/imaging/ossimBandSelector.h>
#include <ossim/imaging/ossimCibCadrgTileSource.h>
#include <ossim/init/ossimInit.h>

#include <ossim/base/ossimArgumentParser.h>
#include <ossim/base/ossimApplicationUsage.h>

static ossimTrace traceDebug("icp:main");

static void usage();
static void outputWriterTypes();

ossimString massageQuotedValue(const ossimString& value)
{
   char quote = '\0';
   if(*value.begin() == '"')
   {
      quote = '"';
   }
   else if(*value.begin() == '\'')
   {
      quote = '\'';
   }
   
   if(quote == '\0')
   {
      return value;
   }
   std::vector<ossimString> splitString;
   value.split(splitString, quote);
   if(splitString.size() == 3)
   {
      return splitString[1];
   }
   return value;
}

int main(int argc, char* argv[])
{
   static const char MODULE[] = "icp:main";
   std::string tempString;
   ossimArgumentParser::ossimParameter stringParam(tempString);
   ossimArgumentParser argumentParser(&argc, argv);
   ossimInit::instance()->addOptions(argumentParser);
   ossimInit::instance()->initialize(argumentParser);
   
   argumentParser.getApplicationUsage()->setApplicationName(argumentParser.getApplicationName());
   argumentParser.getApplicationUsage()->setDescription(argumentParser.getApplicationName()+" copys any supported input image format to any supported output image format format");
   argumentParser.getApplicationUsage()->setCommandLineUsage(argumentParser.getApplicationName()+" [options] <output_type> <input_file> <output_file>");
   argumentParser.getApplicationUsage()->addCommandLineOption("-h or --help","Display this information");
   argumentParser.getApplicationUsage()->addCommandLineOption("-a or --use-scalar-remapper", "Uses scalar remapper, transforms to 8-bit");
   argumentParser.getApplicationUsage()->addCommandLineOption("-o or --create-overview", "Creates and overview for the output image");
   argumentParser.getApplicationUsage()->addCommandLineOption("-b or --bands", "uses the specified bands: ex. \"1, 2, 4\" will select bands 1 2 and 4 of the input image.  Note: it is 1 based");
   argumentParser.getApplicationUsage()->addCommandLineOption("-c or --compression-type", "Uses compression.  Currently valid for only tiff output -c jpeg will use jpeg compression");
   argumentParser.getApplicationUsage()->addCommandLineOption("-e or --entry", "For multi image handlers which entry do you wish to extract");
   
   argumentParser.getApplicationUsage()->addCommandLineOption("-q or --compression-quality", "Uses compression.  Valid for jpeg type. default is 75 where 100 is the best and 1 is the worst");

   argumentParser.getApplicationUsage()->addCommandLineOption("--pixel-type", "Valid values: area or point, this will determine if the tie point is upper left corner of the upper left pixel (area) or the center of the upper left corner (point), default=point.  NOTE: This option will only affect the tiff writer.");

   argumentParser.getApplicationUsage()->addCommandLineOption("-r or --res-level", "Which res level to extract from the input: ex -r 1 will get res level 1");
   argumentParser.getApplicationUsage()->addCommandLineOption("-l or --start-line", "Which start line do you wish to copy from the input. If none is given then 0 is used");
   argumentParser.getApplicationUsage()->addCommandLineOption("-L or --end-line", "Which end line do you wish to copy from the input.  If none is given then max line is used");
   argumentParser.getApplicationUsage()->addCommandLineOption("-s or --start-sample", "Which start sample do you wish to copy from the input.  If none is given then 0 is used");
   argumentParser.getApplicationUsage()->addCommandLineOption("-p or --end-sample", "Which end sample do you wish to copy from the input.  If none is given then max sample is used");
   argumentParser.getApplicationUsage()->addCommandLineOption("-t or --create-thumbnail", "Takes an argument which is the maximum pixel dimension desired.\nCreate thumbnail flag is enabled");
   argumentParser.getApplicationUsage()->addCommandLineOption("-w or --tile-width", "Defines the tile width for the handlers that support tiled output");
   argumentParser.getApplicationUsage()->addCommandLineOption("-writer-prop", "adds a property to send to the writer. format is name=value");
   
   
   if (traceDebug()) CLOG << " Entered..." << std::endl;
   
   // Keyword list to initialize image writers with.
   ossimKeywordlist kwl;
   const char* PREFIX = "imagewriter.";

   bool        lineEndIsSpecified       = false;
   bool        sampEndIsSpecified       = false;
   bool        lineStartIsSpecified     = false;
   bool        sampStartIsSpecified     = false;
   bool        create_overview          = false;
   bool        create_thumbnail         = false;
   bool        use_band_selector        = false;
   bool        use_scalar_remapper      = false;
   ossim_int32 tile_width               = 0;
   ossim_int32 max_thumbnail_dimension  = 0;
   ossim_int32 rr_level                 = 0;
   ossim_int32 line_start               = 0;
   ossim_int32 line_stop                = 0;
   ossim_int32 sample_start             = 0;
   ossim_int32 sample_stop              = 0;
   ossim_int32 cibcadrg_entry           = 0;
   vector<ossimString> band_list(0);
   std::map<ossimString, ossimString> theWriterPropertyMap;

   if (argumentParser.read("-h") ||
       argumentParser.read("--help")||(argumentParser.argc() < 2))
   {
      argumentParser.getApplicationUsage()->write(ossimNotify(ossimNotifyLevel_NOTICE));
      usage(); // for writer output types
      exit(0);
   }
   
   while(argumentParser.read("--writer-prop", stringParam))
   {
      std::vector<ossimString> nameValue;
      ossimString(tempString).split(nameValue, "=");
      if(nameValue.size() == 2)
      {
         theWriterPropertyMap.insert(std::make_pair(nameValue[0], massageQuotedValue(nameValue[1])));
      }
   }
   while(argumentParser.read("-a") ||
         argumentParser.read("--use-scalar-remapper"))
   {
      use_scalar_remapper = true;        
   }
   
   while(argumentParser.read("-o") ||
         argumentParser.read("--create-overview"))
   {
      create_overview = true;
   }
   
   while(argumentParser.read("-b", stringParam) ||
         argumentParser.read("--bands", stringParam))
   {
      use_band_selector = true;
      ossimString s = tempString;
      band_list = s.split(",");
   }
   
   while(argumentParser.read("-c", stringParam) ||
         argumentParser.read("--compression-type", stringParam))
   {
      ossimString s = tempString;
      s.downcase();
      kwl.add(PREFIX, ossimKeywordNames::COMPRESSION_TYPE_KW, s.c_str(), true);
   }
   
   while(argumentParser.read("-e", stringParam) ||
         argumentParser.read("--entry", stringParam))
   {
      cibcadrg_entry = ossimString(tempString).toInt();
   }
   
   while(argumentParser.read("-q", stringParam) ||
         argumentParser.read("--compression-quality", stringParam))
   {
      // Set the jpeg compression quality level.
      kwl.add(PREFIX,
              ossimKeywordNames::COMPRESSION_QUALITY_KW,
              tempString.c_str(),
              true);
   }
   while(argumentParser.read("-r", stringParam) ||
         argumentParser.read("--res-level", stringParam))
   {
      rr_level = ossimString(tempString).toInt();
   }
   
   while(argumentParser.read("-l", stringParam) ||
         argumentParser.read("--start-line", stringParam))
   {
      lineStartIsSpecified = true;
      line_start = ossimString(tempString).toInt();
   }
   
   while(argumentParser.read("-L", stringParam) ||
         argumentParser.read("--end-line", stringParam))
   {
      lineEndIsSpecified = true;
      line_stop = ossimString(tempString).toInt();
   }
   while(argumentParser.read("-s", stringParam) ||
         argumentParser.read("--start-sample", stringParam))
   {
      sampStartIsSpecified = true;
      sample_start = ossimString(tempString).toInt();
   }
   
   while(argumentParser.read("-p", stringParam) ||
         argumentParser.read("--end-sample", stringParam))
   {
      sampEndIsSpecified = true;
      sample_stop = ossimString(tempString).toInt();
   }
   
   while(argumentParser.read("-t", stringParam) ||
         argumentParser.read("--create-thumbnail", stringParam))
   {
      create_thumbnail = true;
      max_thumbnail_dimension=ossimString(tempString).toInt();
   }
   
   while(argumentParser.read("-w", stringParam) ||
         argumentParser.read("-tile-width", stringParam))
   {
      tile_width = ossimString(tempString).toInt();
      if ((tile_width % 16) != 0)
      {
         ossimNotify(ossimNotifyLevel_NOTICE)
            << MODULE << " NOTICE:"
            << "\nTile width must be a multiple of 16!"
            << "\nDefaulting to 128"
            << std::endl;
         tile_width = 0;
      }
   }
   if (argumentParser.read("--pixel-type", stringParam))
   {
      ossimString os = tempString;
      os.downcase();
      if (os.contains("area"))
      {
         kwl.add(PREFIX, ossimKeywordNames::PIXEL_TYPE_KW, "area", true);
      }
      else
      {
         kwl.add(PREFIX, ossimKeywordNames::PIXEL_TYPE_KW, "point", true);
 
      }
   }
   
   argumentParser.reportRemainingOptionsAsUnrecognized();
   
   // Three required args:  output_type, input file, and output file.
   if (argumentParser.errors())
   {
      argumentParser.writeErrorMessages(ossimNotify(ossimNotifyLevel_NOTICE));
      exit(0);
   }
   if (argumentParser.argc() < 4)
   {
      argumentParser.getApplicationUsage()->write(ossimNotify(ossimNotifyLevel_NOTICE));
      usage(); // for writer output types
      exit(0);
   }
   
   //---
   // Set the writer type and the image type.
   //---
   ossimString output_type = argumentParser.argv()[argumentParser.argc()-3];
//   output_type.downcase();
   
   
   kwl.add(PREFIX, ossimKeywordNames::TYPE_KW, output_type.c_str(), true);
   // Get the input file.
   const char* input_file = argv[argumentParser.argc()-2];
   
   // Get the output file.
   ossimFilename    output_file = argv[argumentParser.argc()-1];
   
   if (traceDebug())
   {
      CLOG << "DEBUG:"
           << "\noutput type:  "
           << argumentParser.argv()[argumentParser.argc()-3]
           << "\ninput file:   "
           << argumentParser.argv()[argumentParser.argc()-2]
           << "\noutput file:  "
           << argumentParser.argv()[argumentParser.argc()-1]
           << std::endl;
      
      if (tile_width) ossimNotify(ossimNotifyLevel_NOTICE) << "tile_width:  " << tile_width << std::endl;
   }
   
   // Get an image handler for the input file.
   ossimRefPtr<ossimImageHandler> ih =
      ossimImageHandlerRegistry::instance()->open(ossimFilename(input_file));
   
   ossimCibCadrgTileSource* its = PTR_CAST(ossimCibCadrgTileSource, ih.get());
   
   if (its)
   {
      if (cibcadrg_entry > 0)
      {
         its->setCurrentEntry(cibcadrg_entry);
      }
   }
   
   if (!ih)
   {
      ossimNotify(ossimNotifyLevel_NOTICE) << "Unsupported image file:  " << input_file
           << "\nExiting application..." << std::endl;
      exit(0);
   }
   
   // Initialize the 
   if (ih->getErrorStatus() == ossimErrorCodes::OSSIM_ERROR)
   {
      ossimNotify(ossimNotifyLevel_FATAL)
         << "Error reading image:  " << input_file
         << "Exiting application..." << std::endl; 
      exit(1);
   }
   
   ih->initialize();
   
   if (traceDebug())
   {
      CLOG << "DEBUG:"
           << "\nImage Handler:  " << ih->getLongName()
           << std::endl;
   }
   
   ossimRefPtr<ossimImageSource> source = ih.get();
   ossimRefPtr<ossimBandSelector> bs = 0;
   
   if (use_band_selector && (ih->getNumberOfOutputBands() > 1))
   {
      // Build the band list.
      ossim_uint32 bands = ih->getNumberOfOutputBands();
      vector<ossim_uint32> bl;
      ossim_uint32 i;
      for (i=0; i<band_list.size(); ++i)
      {
         bl.push_back(band_list[i].toULong()-1);
      }
      
      // Check the list.  Make sure all the bands are within range.
      for (i=0; i<bl.size(); ++i)
      {
         if (bl[i] >= bands)
         {
            ossimNotify(ossimNotifyLevel_FATAL)
               << MODULE << " ERROR:"
               << "\nBand list range error!"
               << "\nHighest available band:  " << bands
               << std::endl;
            exit(1);
         }
      }
      
      bs = new ossimBandSelector();
      bs->connectMyInputTo(ih.get());
      bs->setOutputBandList(bl);
      bs->enableSource();
      bs->initialize();
      source = bs.get();
      
      if (traceDebug())
      {
         CLOG << "DEBUG:"
              << "\nZero based output band list:" << std::endl;
         for (i=0; i<bl.size(); ++i)
         {
            ossimNotify(ossimNotifyLevel_NOTICE)
               << "   band[" << i << "]:  " << bl[i] << std::endl;
         }
         ossimNotify(ossimNotifyLevel_NOTICE) << std::endl;
      }
   }
   
   //---
   // Get the image rectangle for the rrLevel selected.
   //---
   ossimIrect output_rect;
   ossimRefPtr<ossimRLevelFilter> rlevelFilter = 0;
   
   if (create_thumbnail == true)
   {
      rlevelFilter = new ossimRLevelFilter;
      
      rlevelFilter->connectMyInputTo(source.get());
      
      source = rlevelFilter.get();
      // Get the rlevel that <= max thumbnail dimension.
      int max   = 0;
      int level = 0;
      
      while (level < ((ossim_int32)ih->getNumberOfDecimationLevels()-1))
      {
         int lines   = ih->getNumberOfLines(level);
         int samples = ih->getNumberOfSamples(level);
         max = lines > samples ? lines : samples;
         if (max <= max_thumbnail_dimension)
         {
            break;
         }
         level++;
      }
      
      if (max > max_thumbnail_dimension)
      {
         ossimNotify(ossimNotifyLevel_NOTICE) << " NOTICE:"
              << "\nLowest rlevel not small enough to fullfill "
              << "max_thumbnail_dimension requirement!" << std::endl;
      }
      
      // Set the reduced res level.  This will override the -r option.
      rr_level = level;
      rlevelFilter->setCurrentRLevel(rr_level);
      if (rr_level)
      {
         rlevelFilter->setOverrideGeometryFlag(true);
      }


      // Set the rectangle to that of the res level.
      output_rect = rlevelFilter->getBoundingRect();

   } // end of "if  (create_thumbnail == true)
   else
   {
      // modified to only do an rlevel if one is requested
      if(rr_level != 0)
      {
         //***
         // Check for a valid reduced resolution level.
         // If the operator entered an invalid rr_level with the -r option,
         // spit out a warning and set to default "0".
         //***
         if (rr_level >= (ossim_int32)(ih->getNumberOfDecimationLevels()))
         {
            ossimNotify(ossimNotifyLevel_NOTICE) << " WARNING:"
            << "\n\t Selected res level greater than available res levels."
            << "\n\t Defaulting to res level zero. " << std::endl;
            rr_level = 0;
         }
         
         rlevelFilter = new ossimRLevelFilter;
         
         rlevelFilter->connectMyInputTo(source.get());
         
         source = rlevelFilter.get();
         
         rlevelFilter->setCurrentRLevel(rr_level);
         if (rr_level)
         {
            rlevelFilter->setOverrideGeometryFlag(true);
         }
      }
      output_rect = source->getBoundingRect(rr_level);
      
      //***
      // If any of these are true the user wants to cut the rectangle.
      //***
      if ( lineStartIsSpecified || lineEndIsSpecified ||
           sampStartIsSpecified || sampEndIsSpecified)
      {
         if (!lineStartIsSpecified) line_start   = output_rect.ul().y;
         if (!lineEndIsSpecified)   line_stop    = output_rect.lr().y;
         if (!sampStartIsSpecified) sample_start = output_rect.ul().x;
         if (!sampEndIsSpecified)   sample_stop  = output_rect.lr().x;
                                                 
         //***
         // Check the start and stop points and make sure they are in
         // the right order; if not, swap them.
         //***
         if (line_stop < line_start)
         {
            ossimNotify(ossimNotifyLevel_NOTICE) << " WARNING:\n"
                 << "\t Line end is less than line start, swapping."
                 << std::endl;
            int tmp    = line_start;
            line_start = line_stop;
            line_stop  = tmp;
         }
         if (sample_stop < sample_start)
         {
            ossimNotify(ossimNotifyLevel_WARN)
               << " WARNING:\n"
               << "\t Sample end is less than sample start, swapping."
               << std::endl;
            int tmp      = sample_start;
            sample_start = sample_stop;
            sample_stop  = tmp;
         }

//          if (sample_start > output_rect.ul().x &&
//              sample_start < output_rect.lr().x)
//          {
             output_rect.set_ulx(sample_start);
//          }
//          if (sample_stop > output_rect.ul().x &&
//              sample_start < output_rect.lr().x)
//          {
             output_rect.set_lrx(sample_stop);
//          }
//          if (line_start > output_rect.ul().y &&
//              line_start < output_rect.lr().y)
//          {
             output_rect.set_uly(line_start);
//          }
//          if (line_stop > output_rect.ul().y &&
//              line_start < output_rect.lr().y)
//          {
             output_rect.set_lry(line_stop);
//          }
         
      } // End of "if ((lineEndIsSpecified) ||..."
   }

   if (traceDebug())
   {
      CLOG << "icp:main debug"
           << "\nrr_level:  " << rr_level
           << "\noutput_rect:   " << output_rect
           << "\nkeyword list:  " << kwl << std::endl;
   }
   
   
   ossimRefPtr<ossimImageFileWriter> writer =
      ossimImageWriterFactoryRegistry::instance()->createWriter(kwl, PREFIX);

   if( writer == 0 )
   {
      ossimNotify(ossimNotifyLevel_NOTICE)
         << "\nCould not create writer of type:  "
         << output_type
         << std::endl;
      usage();
      exit(1);
   }

   writer->connectMyInputTo(0, source.get());

   if (tile_width)
   {
      // Set the tile size...
      writer->setTileSize(ossimIpt(tile_width, tile_width));
   }
   
   writer->open(output_file);
   
   // Add a listener to get percent complete.
   ossimStdOutProgress prog(0, true);
   writer->addListener(&prog);

   if (writer->getErrorStatus() == ossimErrorCodes::OSSIM_OK)
   {
      if( (ih->getOutputScalarType() != OSSIM_UCHAR) &&
          ((PTR_CAST(ossimJpegWriter, writer.get())) ||
           use_scalar_remapper))
      {
         writer->setScaleToEightBitFlag(true);
      }

      ossimRefPtr<ossimCacheTileSource> cache = new ossimCacheTileSource;
      ossimIpt tileWidthHeight(ih->getImageTileWidth(),
                               ih->getImageTileHeight());
      // only use the cache if its stripped
      if(static_cast<ossim_uint32>(tileWidthHeight.x) ==
         ih->getBoundingRect().width())
      {
         cache->connectMyInputTo(0, source.get());
         cache->setTileSize(tileWidthHeight);
         writer->connectMyInputTo(0, cache.get());
      }
      else
      {
         writer->connectMyInputTo(0, source.get());
      }
      writer->initialize();
      writer->setAreaOfInterest(output_rect); // Set the output rectangle.

      try
      {
         ossimPropertyInterface* propInterface = (ossimPropertyInterface*)writer.get();
         std::map<ossimString, ossimString>::iterator iter = theWriterPropertyMap.begin();
         while(iter!=theWriterPropertyMap.end())
         {
            propInterface->setProperty(iter->first, iter->second);
            ++iter;
         }
         writer->execute();
      }
      catch(std::exception& e)
      {
         ossimNotify(ossimNotifyLevel_FATAL)
            << "icp:main ERROR:\n"
            << "Caught exception!\n"
            << e.what()
            << std::endl;
      }
      catch(...)
      {
         ossimNotify(ossimNotifyLevel_FATAL)
            << "icp:main ERROR:\n"
            << "Unknown exception caught!\n"
            << std::endl;
      }

      cache = 0;
   }
   else
   {
      ossimNotify(ossimNotifyLevel_FATAL)
         << "Error detected in the image writer..."
         << "\nExiting application..." << std::endl;

      exit(1);
   }
   
   if (create_overview == true)
   {
      writer->writeOverviewFile();
   }
   
   exit(0);
}

void usage()
{
   ossimNotify(ossimNotifyLevel_NOTICE)
      << "\nNOTES:"
      << "\nValid output writer types:"
      << "\n";
   outputWriterTypes();
}

void outputWriterTypes()
{
   std::vector<ossimString> outputType;
   
   ossimImageWriterFactoryRegistry::instance()->getImageTypeList(outputType);
   std::copy(outputType.begin(),
             outputType.end(),
             std::ostream_iterator<ossimString>(ossimNotify(ossimNotifyLevel_NOTICE), "\t\n"));
}
