%% SPDX-License-Identifier: GPL-3.0-or-later
%% Copyright (C) 2016-2022, 2024 Colin B. Macdonald
%%
%% This file is part of OctSymPy.
%%
%% OctSymPy is free software; you can redistribute it and/or modify
%% it under the terms of the GNU General Public License as published
%% by the Free Software Foundation; either version 3 of the License,
%% or (at your option) any later version.
%%
%% This software is distributed in the hope that it will be useful,
%% but WITHOUT ANY WARRANTY; without even the implied warranty
%% of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
%% the GNU General Public License for more details.
%%
%% You should have received a copy of the GNU General Public
%% License along with this software; see the file COPYING.
%% If not, see <http://www.gnu.org/licenses/>.

%% -*- texinfo -*-
%% @documentencoding UTF-8
%% @defun coshint (@var{x})
%% Numerical coshint function.
%%
%% Example:
%% @example
%% @group
%% coshint (1.1)
%%   @result{} ans = 0.9907
%% @end group
%% @end example
%%
%% @strong{Note} this function may be slow for large numbers of inputs.
%% This is because it is not a native double-precision implementation
%% but rather the numerical evaluation of the Python @code{mpmath} function
%% @code{chi}.
%%
%% Note: this file is autogenerated: if you want to edit it, you might
%% want to make changes to 'generate_functions.py' instead.
%%
%% @seealso{@@sym/coshint}
%% @end defun


function y = coshint (x)
  if (nargin ~= 1)
    print_usage ();
  end
  cmd = { 'L = _ins[0]'
          'A = [complex(mpmath.chi(x)) for x in L]'
          'return A,' };
  c = pycall_sympy__ (cmd, num2cell (x(:)));
  y = reshape (cell2mat (c), size (x));
end


%!error coshint (1, 2)

%!test
%! x = 1.1;
%! y = sym(11)/10;
%! A = coshint (x);
%! B = double (coshint (y));
%! assert (A, B, -4*eps);

%!test
%! y = [2 3 sym(pi); exp(sym(1)) 5 6];
%! x = double (y);
%! A = coshint (x);
%! B = double (coshint (y));
%! assert (A, B, -4*eps);

%!test
%! % maple:
%! % > A := [1+2*I, -2 + 5*I, 100, 10*I, -1e-4 + 1e-6*I, -20 + I];
%! % > for a in A do evalf(Chi(a)) end do;
%! x = [1+2i; -2+5i; 100; 10i; -1e-4 + 1e-6*1i; -20-1i];
%! A = [  0.58447599687824767874 + 1.8682915044330306402*1i
%!       -0.63131069034703116988 + 1.8986171211850702957*1i
%!        0.13577763724269399110e42
%!       -0.045456433004455372635 + 1.5707963267948966192*1i
%!       -8.6330747070747332203 + 3.1315929868531280002*1i
%!        0.74701205140887966531e7 + 0.10381444259644068585e8*1i ];
%! B = coshint (x);
%! assert (A, B, -eps)

%!assert (coshint (inf), inf)
%!assert (isinf (coshint (-inf)))
%!assert (imag (coshint (-inf)), pi)
