# -*-ruby; coding: utf-8 -*- vim:set ft=ruby:
#
# Copyright (c) 2004-2006 SASADA Koichi <ko1 at atdot.net>
#
# This program is free software with ABSOLUTELY NO WARRANTY.
# You can re-distribute and/or modify this program under
# the same terms of the Ruby's license.
#

=begin

== Abstract

Answer weather information using "Livedoor Weather Web Service / LWWS".

LWWS: http://weather.livedoor.com/weather_hacks/webservice


== Usage

  tenki> [CITY] 

  [CITY] should be city name in Kanji listed on following table.
  http://weather.livedoor.com/forecast/rss/primary_area.xml


== Configuration

BotConfig = [
{
  :name => :TenkiBot,
  :ch   => /nadoka/,  # default: //
}
]


=end

require 'open-uri'
require 'pp'
require 'kconv'
require 'date'
begin
  require 'json'
rescue LoadError
  require 'rubygems'
  require 'json'
end

module Tenki
  CityIDs = {}

  def init_tenki
    open('http://weather.livedoor.com/forecast/rss/primary_area.xml') do |f|
      f.each_line do |line|
        if /city title="(.+?)" id="(\d+)"/ =~ line
          CityIDs[$1.toutf8] = $2
        end
      end
    end
  end

  def tenki(city)
    unless city_id = CityIDs[city]
      return "Unknown city.  Check city title on http://weather.livedoor.com/forecast/rss/primary_area.xml"
    end
    json = open("http://weather.livedoor.com/forecast/webservice/json/v1?city=#{city_id}") do |f|
      JSON.parse f.read
    end

    tenki = "#{json['title']}: "
    tenki << json['forecasts'].map do |forecast|
      max = forecast['temperature']['max']
      min = forecast['temperature']['min']
      celsius = []
      celsius << "min:#{min['celsius']}" if min
      celsius << "max:#{max['celsius']}" if max
      unless celsius.empty?
        temperature = "(#{celsius.join(',')})"
      end
      "#{forecast['dateLabel']}:#{forecast['telop']}#{temperature}"
    end.join(', ')
    desc = json['description']
    text, = desc['text'].split(/\n\n/, 2)
    text.gsub!(/\n/, '')
    tenki << " - #{text}(#{desc['publicTime']})"
    tenki << " - #{json['link']}"

    tenki
  end
end

if __FILE__ == $0
  include Tenki
  if ARGV.empty?
    puts "#$0 city"
  else
    init_tenki
    ARGV.each do |city|
      puts tenki(city)
    end
  end
  exit
end

class TenkiBot < Nadoka::NDK_Bot
  include Tenki

  def bot_initialize
    bot_init_utils
    init_tenki
    @nkf = @bot_config[:nkf] || "-Wj"
  end

  def on_privmsg prefix, ch, msg
    return unless @available_channel === ch
    return if same_bot?(ch)
    msg = NKF.nkf('-w', msg)
    if /\Atenki>/ =~ msg
      city = $'.strip.toutf8
      begin
        result = tenki(city)
      rescue => e
        result = "#{e}"
      end
      send_notice ch, NKF.nkf(@nkf, "tenki bot: #{result}".gsub(/\s+/, ' '))
    end
  end
end
