package tests::functional::I18NTest;

use strict;

use base qw/Lire::Test::FunctionalTestCase tests::functional::TestSchemaFixture/;

use Lire::Config;
use Lire::Config::Build qw/ac_info ac_path/;
use File::Basename qw/dirname/;
use File::Path qw/mkpath rmtree/;
use Lire::I18N;
use Lire::DlfStore;
use Lire::Config::SpecParser;
use Lire::Utils qw/file_content/;

use Time::Local;
use Cwd 'realpath';
use Carp;

#our @TESTS = qw//;

sub new {
    my $self = shift->SUPER::new(@_);

    $self->tests::functional::TestSchemaFixture::init();

    $self->{'testsdir'} = realpath(dirname( __FILE__ ) . "/..");
    $self->{'report_cfg'} = "$self->{'testsdir'}/data/test-utf8_cfg.xml";
    $self->{'fr_LocaleData'} =
      ac_info( 'LR_PERL5LIB' ) . "/LocaleData/fr/LC_MESSAGES";

    return $self;
}

sub set_up {
    my $self = $_[0];
    $self->SUPER::set_up();

    $self->set_up_test_schemas();
    $self->set_up_test_specs();
    $self->set_up_TestDlfConverter();
    $self->set_up_test_analysers();
    $self->set_up_test_store();

    system( "cp $self->{'testsdir'}/po/fr.mo $self->{'fr_LocaleData'}/lire-test.mo" );
}

sub tear_down {
    my $self = $_[0];
    $self->SUPER::tear_down();

    $self->tests::functional::TestSchemaFixture::tear_down();
    unlink( $self->{'fr_LocaleData'} );
    $self->locale( 'C' );

    return;
}

sub set_up_i18n_import_job {
    my ( $self, $cfg ) = @_;

    my $job = $cfg->get( 'import_jobs' )->spec()->get( 'import_job' )->instance();
    $cfg->get( 'import_jobs' )->append( $job );
    $job->get( 'name' )->set( 'UTF8 Test' );
    $job->get( 'period' )->set( 'daily' );
    $job->get( 'service' )->set_plugin( 'test_newapi' );
    $job->get( 'log_file' )->set( "$self->{'testsdir'}/data/test-utf8.dlf" );
    $job->get( 'log_encoding' )->set( 'utf-8' );

    return;
}

sub set_up_i18n_report_job {
    my ( $self, $cfg ) = @_;

    my $job = $cfg->get( 'report_jobs' )->spec()->get( 'report_job' )->instance();
    $cfg->get( 'report_jobs' )->append( $job );
    $job->get( 'name' )->set( 'i18n_test' );
    my $sched = $job->get( 'schedules' )->spec()->get( 'schedule' )->instance();
    $job->get( 'schedules' )->append( $sched );

    $sched->get( 'period' )->set( 'daily' );
    $sched->get( 'report_config' )->set( 'test_utf8' );

    my $parser = new Lire::Config::SpecParser();
    my $report_cfg  = $parser->parsefile( $self->{'report_cfg'} )->get( 'test_utf8' )->instance();
    $report_cfg->{'spec'} = $cfg->get( 'reports' )->spec()->get( 'report_cfg' );
    $cfg->get( 'reports' )->append( $report_cfg );

    return
}

# This validate the generation of XML reports in non-ASCII encoding.
# Log file is UTF-8
# Report configuration file is Latin1
# Report specification contains Latin1
sub test_report_with_encoding {
    my $self = $_[0];

    return unless $Lire::I18N::USE_ENCODING;

    Lire::Config->init();

    $self->locale( 'fr_CA.iso8859-1' );

    # lr_cron is run the next day
    my $time = timelocal( 0, 0, 3, 26, 0, 2003 );
    my $store =  Lire::DlfStore->open( $self->homedir() . "/a_test", 1 );
    my $store_path = $store->path();
    my $cfg = $store->config();
    $self->set_up_i18n_import_job( $cfg );
    $self->set_up_i18n_report_job( $cfg );
    $cfg->save();
    $store->close();

    my $result = $self->lire_run( "lr_cron daily $store_path $time" );
    $self->annotate( $result->stderr() );
    $self->assert_num_equals( 0, $result->status() );
    $self->assert( !$result->stdout(), "stdout should be empty" );
    $self->assert_does_not_match( qr/ (crit|err|warning) /, $result->stderr(),
                                  "There were warnings or error messages." );

    my $report = "$store_path/daily_reports/i18n_test/200301/25.xml";
    $self->assert( -f $report, "Report wasn't generated" );
    $self->check_xml_report( "$self->{'testsdir'}/data/test-utf8.xml",
                             $report );
}

sub test_report_with_encoding_no_support {
    my $self = $_[0];

    return if $Lire::I18N::USE_ENCODING;

    my $report_cfg = $self->rundir() . "/test.cfg";
    $self->create_test_file( $report_cfg, <<NOENCODING_CFG );
=section Telechargements
user-downloads-utf8 users_to_show=5
NOENCODING_CFG

    my $result = $self->lire_run( "lr_log2report -c $report_cfg -o xml test_newapi < $self->{'testsdir'}/data/test-utf8.dlf > report.xml" );
    $self->annotate( $result->stderr() );
    $self->assert_num_equals( 0, $result->status() );
    $self->assert( !$result->stdout(), "stdout should be empty" );
    $self->assert_does_not_match( qr/(ERROR|WARNING)/, $result->stderr(),
                                  "There were warnings or error messages." );

    $self->assert( -f $self->rundir() . "/report.xml",
                   "XML report wasn't created" );
    $self->assert( -s $self->rundir() . "/report.xml",
                   "XML report is empty" );

    $self->check_xml_report( "$self->{'testsdir'}/data/test-utf8-nosupport.xml",
                             $self->rundir() . "/report.xml" );
}

sub test_report_i18n {
    my $self = $_[0];

    $self->locale( 'fr_CA.iso8859-1' );

    my $default_file =
      ac_path( 'datadir', 'PACKAGE' ) . '/templates/test-l10n.xml';
    $self->create_test_file( $default_file,
                             file_content( $self->tests_datadir() . "/test-l10n_cfg.xml" ) );

    my $result = $self->lire_run( "lr_log2report --template test-l10n --output-format xml test_newapi $self->{'testsdir'}/data/test.dlf report.xml" );
    $self->annotate( $result->stderr() );
    $self->assert_num_equals( 0, $result->status() );
    $self->assert( !$result->stdout(), "stdout should be empty" );
    $self->assert_does_not_match( qr/(ERROR|WARNING)/, $result->stderr(),
                                  "There were warnings or error messages." );

    $self->assert( -f $self->rundir() . "/report.xml",
                   "XML report wasn't created" );
    $self->assert( -s $self->rundir() . "/report.xml",
                   "XML report is empty" );

    $self->check_xml_report( "$self->{'testsdir'}/data/test-l10n.xml",
                             $self->rundir() . "/report.xml" );
}

1;
