/*
 * $Id: Cglobals.c,v 1.1 2005/03/29 09:27:18 baud Exp $
 */

/*
 * Copyright (C) 1990-1999 by CERN/IT/PDP/DM
 * All rights reserved
 */

#ifndef lint
static char sccsid[] = "@(#)$RCSfile: Cglobals.c,v $ $Revision: 1.1 $ $Date: 2005/03/29 09:27:18 $ CERN/IT/PDP/DM Olof Barring";
#endif /* lint */
/*
 * Castor_globals.c - central entry to maintain all Castor globals
 */

/*
 * All static and global variables in the Castor client part, i.e.
 * routines which can be called from within independent parts of
 * Castor itself as well as routines which are externalized to the
 * users, must be declared through this routine in order to
 * guarantee thread-safeness. The schema works if the calling
 * application is entirely based on Cthread, in which case the
 * Cglobals_init() routine is called from Cthread_init(). If
 * other thread packages are used, the application developer must
 * provide two routines similar to Cthread_getspecific() and
 * Cthread_setspecific() to be passed in a call to Cglobals_init().
 */

#include <stdlib.h>
#include <Cglobals.h>

/*
 * Define a structure to maintain global data in a single-threaded
 * environment.
 */
typedef struct Cglobals {
    void *addr;
    int *key;
} Cglobals_t;

/*
 * Internal static storage. Thread-safeness is guaranteed if
 * Cglobals_init() is called from Cthread_init() or at process
 * startup before any threads (except main) have been created.
 */
#ifdef __STDC__
extern int (*local_getspec)(int *, void **);
extern int (*local_setspec)(int *, void *);
int (*local_getspec)(int *, void **) = NULL;
int (*local_setspec)(int *, void *) = NULL;
static int (*local_getTid)(void) = NULL;
#else
extern int (*local_getspec)();
extern int (*local_setspec)();
int (*local_getspec)() = NULL;
int (*local_setspec)() = NULL;
static int (*local_getTid)() = NULL;
#endif

static Cglobals_t **single_thread_globals = NULL;
static int nb_globals = 0;
/*
 * Single thread globals table allocation size.
 */
#ifdef __STDC__
const int alloc_size = 1000;
#else
int alloc_size = 1000;
#endif

/*
 * Castor errno externals (used as keys in multi-thread env.)
 * Note that we cannot include serrno.h to get the definitions
 * since it re-defines serrno if _REENTRANT.
 */
EXTERN_C int DLL_DECL serrno;
int DLL_DECL serrno;
EXTERN_C int DLL_DECL rfio_errno;
int DLL_DECL rfio_errno;
EXTERN_C int DLL_DECL Copterr;
int DLL_DECL Copterr;
EXTERN_C int DLL_DECL Coptind;
int DLL_DECL Coptind;
EXTERN_C int DLL_DECL Coptopt;
int DLL_DECL Coptopt;
EXTERN_C int DLL_DECL Coptreset;
int DLL_DECL Coptreset;
EXTERN_C char DLL_DECL *Coptarg;
char DLL_DECL *Coptarg;
EXTERN_C int DLL_DECL Coptarg_key;
int DLL_DECL Coptarg_key;
/*
 * Function prototypes for multi-thread env. version of errno externals
 */
int DLL_DECL *C__serrno _PROTO((void));
int DLL_DECL *C__rfio_errno _PROTO((void));
int DLL_DECL *C__Copterr _PROTO((void));
int DLL_DECL *C__Coptind _PROTO((void));
int DLL_DECL *C__Coptopt _PROTO((void));
int DLL_DECL *C__Coptreset _PROTO((void));
char DLL_DECL **C__Coptarg _PROTO((void));

#if defined(hpux) || defined(HPUX10) || defined(sgi) || defined(SOLARIS)
/* 
 * We need to define a thread safe h_errno for these systems...
 */
EXTERN_C int DLL_DECL h_errno;
/*
 * Function prototypes for muti-thread env. version of h_errno external
 */
int DLL_DECL *C__h_errno _PROTO((void));

#endif /* hpux ||HPUX10 || sgi || SOLARIS */

/*
 * Cglobals_init() - assing routines to provide thread local storage. 
 * Globals that existed prior to this call are all re-assigned as
 * thread-specific to calling thread (normally the main thread).
 */
void DLL_DECL Cglobals_init(getspec,setspec,getTid)
    int (*getspec) _PROTO((int *, void **));
    int (*setspec) _PROTO((int *, void *));
    int (*getTid) _PROTO((void));
{
    int i,rc;
    int *key;
    void *addr;

    if ( getspec != NULL && local_getspec == NULL ) local_getspec = getspec;
    if ( setspec != NULL && local_setspec == NULL ) local_setspec = setspec;
    if ( getTid  != NULL && local_getTid  == NULL ) local_getTid = getTid;
    if ( local_getspec != NULL  && local_setspec != NULL ) {
        if ( single_thread_globals != NULL ) {
            /*
             * Re-assign all old single-thread globals to thread
             * specific data for the calling thread.
             */
            for ( i=0; i<nb_globals; i++) {
                key = single_thread_globals[i]->key;
                rc = local_getspec(key,&addr);
                if ( rc == -1 || addr == NULL ) {
                    addr = single_thread_globals[i]->addr;
                    rc = local_setspec(key,addr);
                }
                free(single_thread_globals[i]);
                single_thread_globals[i] = NULL;
            }
            free(single_thread_globals);
        }
        /*
         * Inherit the current values of the errno globals.
         */
        *C__serrno() = serrno;
        *C__rfio_errno() = rfio_errno;
        *C__Copterr() = Copterr;
        *C__Coptind() = Coptind;
        *C__Coptopt() = Coptopt;
        *C__Coptreset() = Coptreset;
        *C__Coptarg() = Coptarg;
#if defined(hpux) || defined(HPUX10) ||  defined(sgi) || defined(SOLARIS)
        *C__h_errno() = h_errno;
#endif /* hpux || HPUX10 || sgi || SOLARIS */

        single_thread_globals = NULL;
    }
    return;    
}    

/*
 * Cglobals_get() - simulate keyed accessed global storage in
 * single-thread environment and transparently call TLS routines
 * in multi-thread environment. This routine should be called for
 * every internal static storage in the Castor library.
 */
/*
 * Returns:       -1            on error
 *                 0            on success and not first call
 *                 1            on success and first call (e.g. alloc)
 */
int DLL_DECL Cglobals_get(key, addr, size)
     int *key;
     void **addr;
     size_t size;
{
       return Cglobals_get_dynamic(key, addr, size, 0);
}

int DLL_DECL Cglobals_get_dynamic(key, addr, size, resize)
     int *key;
     void **addr;
     size_t size;
     int    resize;
{
    if ( key == NULL || addr == NULL || size<= (size_t)0 ) {
        return(-1);
    }
    if ( local_setspec == NULL ) {
        if ( *key <= 0 ) {
            Cglobals_t *tmp;

            if ((*addr = calloc(1,size)) == NULL) {
              return(-1);
            }
            if ( single_thread_globals == NULL ) {
                if ((single_thread_globals = (Cglobals_t **) malloc(sizeof(Cglobals_t *)*alloc_size)) == NULL) {
                  free(*addr);
                  *addr = NULL;
                  return(-1);
                }
            } else if ( nb_globals == alloc_size ) {
                Cglobals_t **dummy;

                if ((dummy =  (Cglobals_t **) realloc(single_thread_globals, sizeof(Cglobals_t *)*(nb_globals+alloc_size))) == NULL) {
                  free(*addr);
                  *addr = NULL;
                  return(-1);
                }
                single_thread_globals = dummy;
            }
            if ((tmp = (Cglobals_t *) malloc(sizeof(Cglobals_t))) == NULL) {
              free(*addr);
              *addr = NULL;
              return(-1);
            }
            tmp->addr = *addr;
            tmp->key = key;
            single_thread_globals[nb_globals] = tmp;
            nb_globals++;
            *key = nb_globals;
            return(1);
        } else if (resize) {
            *addr = (void *) single_thread_globals[(*key) - 1]->addr;
            *addr = realloc(*addr,size);
            single_thread_globals[(*key) - 1]->addr = *addr;
            return(0);
        } else {
            *addr = (void *) single_thread_globals[(*key) - 1]->addr;
            return(0);
        }
    } else {
        int rc;

        rc = local_getspec(key,addr);
        if ( (rc == -1) || (*addr == NULL) ) {
            if ((*addr = calloc(1,size)) == NULL) {
              return(-1);
            }
            if ((rc = local_setspec(key,*addr)) != 0) {
              free(*addr);
              *addr = NULL;
              return(-1);
            } else {
              return(1);
            }
        } else if (resize) {
          if ((*addr = realloc(*addr,size)) == NULL ||
              ( rc = local_setspec(key,*addr) != 0)) {
            free(*addr);
            *addr = NULL;
            return(-1);
          }
          return(1);
        } else {
          return(0);
        }
    }
}

/*
 * Cglobals_getTid() - get current thread Id. -1 is returned
 * if Cglobals_init() has not been called. Note that in a Cthread
 * environment Tid = -1 will always refer to main thread since
 * Cthread_self() returns an error for the main thread (the main
 * thread has *NOT* been created with a Cthread_create()).
 * 
 */ 
void DLL_DECL Cglobals_getTid(Tid)
     int *Tid;
{
    if ( Tid == NULL ) return;
    if ( local_getTid == NULL ) *Tid = -1;
    else *Tid = local_getTid();
    return;
} 

int DLL_DECL *C__serrno() {
    int rc;
    int *addr;

    if ( local_setspec == NULL ) {
        return(&serrno);
    } else {
        addr = NULL;
        /*
         * We re-use the old single thread serrno as key
         */
        rc = local_getspec(&serrno,(void **)&addr);
        if ( rc == -1 || addr == NULL ) {
            addr = (int *)calloc(1,sizeof(int));
            rc = local_setspec(&serrno,(void *)addr);
        }
        /*
         * If memory allocation failed, we can still
         * return the external. This is obviously not
         * thread-safe but at least the application
         * will not die with a strange coredump in a
         * hidden de-reference of C__serrno().
         */
        if ( addr == NULL ) return(&serrno);
        else return(addr);
    }
}

int DLL_DECL *C__rfio_errno() {
    int rc;
    int *addr;

    if ( local_setspec == NULL ) {
        return(&rfio_errno);
    } else {
        addr = NULL;
        /*
         * We re-use the old single thread rfio_errno as key
         */
        rc = local_getspec(&rfio_errno,(void **)&addr);
        if ( rc == -1 || addr == NULL ) {
            addr = (int *)calloc(1,sizeof(int));
            rc = local_setspec(&rfio_errno,(void *)addr);
        }
        /*
         * If memory allocation failed, we can still
         * return the external. This is obviously not
         * thread-safe but at least the application
         * will not die with a strange coredump in a
         * hidden de-reference of C__rfio_errno().
         */
        if ( addr == NULL ) return(&rfio_errno);
        else return(addr);
    }
}

int DLL_DECL *C__Copterr() {
    int rc;
    int *addr;

    if ( local_setspec == NULL ) {
        return(&Copterr);
    } else {
        addr = NULL;
        /*
         * We re-use the old single thread Copterr as key
         */
        rc = local_getspec(&Copterr,(void **)&addr);
        if ( rc == -1 || addr == NULL ) {
            addr = (int *)calloc(1,sizeof(int));
            rc = local_setspec(&Copterr,(void *)addr);
        }
        /*
         * If memory allocation failed, we can still
         * return the external. This is obviously not
         * thread-safe but at least the application
         * will not die with a strange coredump in a
         * hidden de-reference of C_Copterr.
         */
        if ( addr == NULL ) return(&Copterr);
        else return(addr);
    }
}

int DLL_DECL *C__Coptind() {
    int rc;
    int *addr;

    if ( local_setspec == NULL ) {
        return(&Coptind);
    } else {
        addr = NULL;
        /*
         * We re-use the old single thread rfio_errno as key
         */
        rc = local_getspec(&Coptind,(void **)&addr);
        if ( rc == -1 || addr == NULL ) {
            addr = (int *)calloc(1,sizeof(int));
            rc = local_setspec(&Coptind,(void *)addr);
        }
        /*
         * If memory allocation failed, we can still
         * return the external. This is obviously not
         * thread-safe but at least the application
         * will not die with a strange coredump in a
         * hidden de-reference of C__Coptind.
         */
        if ( addr == NULL ) return(&Coptind);
        else return(addr);
    }
}

int DLL_DECL *C__Coptopt() {
    int rc;
    int *addr;

    if ( local_setspec == NULL ) {
        return(&Coptopt);
    } else {
        addr = NULL;
        /*
         * We re-use the old single thread rfio_errno as key
         */
        rc = local_getspec(&Coptopt,(void **)&addr);
        if ( rc == -1 || addr == NULL ) {
            addr = (int *)calloc(1,sizeof(int));
            rc = local_setspec(&Coptopt,(void *)addr);
        }
        /*
         * If memory allocation failed, we can still
         * return the external. This is obviously not
         * thread-safe but at least the application
         * will not die with a strange coredump in a
         * hidden de-reference of C__Coptopt.
         */
        if ( addr == NULL ) return(&Coptopt);
        else return(addr);
    }
}

int DLL_DECL *C__Coptreset() {
    int rc;
    int *addr;

    if ( local_setspec == NULL ) {
        return(&Coptreset);
    } else {
        addr = NULL;
        /*
         * We re-use the old single thread rfio_errno as key
         */
        rc = local_getspec(&Coptreset,(void **)&addr);
        if ( rc == -1 || addr == NULL ) {
            addr = (int *)calloc(1,sizeof(int));
            rc = local_setspec(&Coptreset,(void *)addr);
        }
        /*
         * If memory allocation failed, we can still
         * return the external. This is obviously not
         * thread-safe but at least the application
         * will not die with a strange coredump in a
         * hidden de-reference of C__Coptreset.
         */
        if ( addr == NULL ) return(&Coptreset);
        else return(addr);
    }
}

char DLL_DECL **C__Coptarg() {
    int rc;
    char **addr;

    if ( local_setspec == NULL ) {
        return(&Coptarg);
    } else {
        addr = NULL;
        /*
         * We re-use the old single thread rfio_errno as key
         */
        rc = local_getspec(&Coptarg_key,(void **)&addr);
        if ( rc == -1 || addr == NULL ) {
            addr = (char **)calloc(1,sizeof(char *));
            rc = local_setspec(&Coptarg_key,(void *)addr);
        }
        /*
         * If memory allocation failed, we can still
         * return the external. This is obviously not
         * thread-safe but at least the application
         * will not die with a strange coredump in a
         * hidden de-reference of C__Coptarg.
         */
        if ( addr == NULL ) return(&Coptarg);
        else return(addr);
    }
}

#if defined(hpux) || defined(HPUX10) ||  defined(sgi) || defined(SOLARIS)
int DLL_DECL *C__h_errno() {
    int rc;
    int *addr;

    if ( local_setspec == NULL ) {
        return(&h_errno);
    } else {
        addr = NULL;
        /*
         * We re-use the old single thread h_errno as key
         */
        rc = local_getspec(&h_errno,(void **)&addr);
        if ( rc == -1 || addr == NULL ) {
            addr = (int *)calloc(1,sizeof(int));
            rc = local_setspec(&h_errno,(void *)addr);
        }
        /*
         * If memory allocation failed, we can still
         * return the external. This is obviously not
         * thread-safe but at least the application
         * will not die with a strange coredump in a
         * hidden de-reference of __h_errno().
         */
        if ( addr == NULL ) return(&h_errno);
        else return(addr);
    }
}
#endif /* hpux || HPUX10 ||  sgi || SOLARIS */

