"""Tests for the common state interface over the the Openstack provider

The testcases here largely duplicate those in openstack.tests.test_files as
state handling is a pretty thin layer over the file storage.
"""

from juju.lib import serializer
from juju.lib.testing import TestCase
from juju.providers.openstack.tests import OpenStackTestMixin


class OpenStackStateTest(OpenStackTestMixin, TestCase):

    def test_save(self):
        """Saving a dict puts yaml serialized bytes in provider-state"""
        state = {"zookeeper-instances": [
             [1000, "x1.example.com"],
             ]}
        self.expect_swift_put("testing/provider-state", serializer.dump(state))
        self.mocker.replay()
        return self.get_provider().save_state(state)

    def test_save_missing_container(self):
        """Saving will create the container when it does not exist already"""
        state = {"zookeeper-instances": [
             [1000, "x1.example.com"],
             ]}
        state_bytes = serializer.dump(state)
        self.expect_swift_put("testing/provider-state", state_bytes, code=404)
        self.expect_swift_put_container("testing")
        self.expect_swift_put("testing/provider-state", state_bytes)
        self.mocker.replay()
        return self.get_provider().save_state(state)

    def test_load(self):
        """Loading deserializes yaml from provider-state to a python dict"""
        state = {"zookeeper-instances": []}
        self.expect_swift_get("testing/provider-state",
            response=serializer.dump(state))
        self.mocker.replay()
        deferred = self.get_provider().load_state()
        return deferred.addCallback(self.assertEqual, state)

    def test_load_missing(self):
        """Loading returns False if provider-state does not exist"""
        self.expect_swift_get("testing/provider-state", code=404,
            response={})
        self.mocker.replay()
        deferred = self.get_provider().load_state()
        return deferred.addCallback(self.assertIs, False)

    def test_load_no_content(self):
        """Loading returns False if provider-state is empty"""
        self.expect_swift_get("testing/provider-state", response="")
        self.mocker.replay()
        deferred = self.get_provider().load_state()
        return deferred.addCallback(self.assertIs, False)
