/*
 *  $Id: hdf5file.c 25952 2023-10-25 14:26:59Z yeti-dn $
 *  Copyright (C) 2020-2023 David Necas (Yeti), Petr Klapetek.
 *  E-mail: yeti@gwyddion.net, klapetek@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either version 2 of the License, or (at your option) any
 *  later version.
 *
 *  This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 *  details.
 *
 *  You should have received a copy of the GNU General Public License along with this program; if not, write to the
 *  Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

/**
 * [FILE-MAGIC-USERGUIDE]
 * Asylum Research Ergo HDF5
 * .h5
 * Read
 **/

/**
 * [FILE-MAGIC-USERGUIDE]
 * Shilps Sciences Lucent HDF5
 * .h5
 * Read SPS Volume Curvemap
 **/

/**
 * [FILE-MAGIC-USERGUIDE]
 * Generic HDF5 files
 * .h5
 * Read
 **/

/**
 * [FILE-MAGIC-USERGUIDE]
 * Matlab MAT 7.x files
 * .mat
 * Read
 **/

/**
 * [FILE-MAGIC-MISSING]
 * Avoding clash with a standard file format.
 **/

/*
 * HDF5 changes its APIs between versions incompatibly.  Forward compatibility is mostly preserved, but not
 * guaranteed.  To prevent breakage we need to know which specific version of the API we use and tell the library to
 * provide this one through compatibility macros.
 *
 * Therefore, this file must be compiled with -DH5_USE_18_API
 */
#define H5_USE_18_API

#include "config.h"
#include <stdlib.h>
#include <string.h>
#include <hdf5.h>
#include <hdf5_hl.h>
#include <libgwyddion/gwymacros.h>
#include <libgwyddion/gwymath.h>
#include <libgwyddion/gwyutils.h>
#include <libprocess/datafield.h>
#include <libprocess/correct.h>
#include <libgwymodule/gwymodule-file.h>
#include <app/gwyapp.h>
#include <app/gwymoduleutils-file.h>

#include "err.h"

#define MAGIC "\x89HDF\r\n\x1a\n"
#define MAGIC_SIZE (sizeof(MAGIC)-1)

#define EXTENSION ".h5"

#define MAGIC_MAT70 "MATLAB 7.0 MAT-file"
#define MAGIC_MAT70_SIZE (sizeof(MAGIC_MAT70)-1)
#define MAGIC_MAT73 "MATLAB 7.3 MAT-file"
#define MAGIC_MAT73_SIZE (sizeof(MAGIC_MAT73)-1)

enum {
    MAT7x_HEADER_SIZE = 512,
};

typedef struct _GenericHDF5File GenericHDF5File;

typedef void (*AttrHandlerFunc)(GenericHDF5File *ghfile,
                                hid_t loc_id,
                                const char *attr_name);

typedef struct {
    GArray *idlist;
    const gchar *idprefix;
    H5O_type_t idwhat;
} GatheredIds;

struct _GenericHDF5File {
    GArray *addr;
    GString *path;
    GString *buf;
    GwyContainer *meta;

    /* Generic gathering of some numeric values. */
    guint nlists;
    GatheredIds *lists;

    /* Generic gathering of datasets in the file. */
    GArray *datasets;

    /* File type implementation specifics. */
    AttrHandlerFunc attr_handler;
    gpointer impl;
};

static gboolean module_register(void);

/* Ergo */
typedef struct {
    gchar *name;
    GwySIUnit *xyunit;
    GwySIUnit *zunit;
    gint xypower10;
    gint zpower10;
    gdouble realcoords[4];
} ErgoChannel;

typedef struct {
    GArray *channels;
    gint nframes;
} ErgoFile;

static gint          ergo_detect       (const GwyFileDetectInfo *fileinfo,
                                        gboolean only_name);
static GwyContainer* ergo_load         (const gchar *filename,
                                        GwyRunType mode,
                                        GError **error);
static void          ergo_attr_handler (GenericHDF5File *ghfile,
                                        hid_t loc_id,
                                        const char *attr_name);
static GwyContainer* ergo_read_channels(hid_t file_id,
                                        GenericHDF5File *ghfile,
                                        GError **error);
static GwyDataField* ergo_read_field   (hid_t file_id,
                                        guint r,
                                        ErgoChannel *c,
                                        gint frameid,
                                        const gint *yxres,
                                        GString *str,
                                        GError **error);

/* Lucent */
enum {
    SHILPS_IMAGES  = 0,
    SHILPS_SPECTRA = 1,
    SHILPS_CURVES  = 2,
    SHILPS_VOLUME  = 3,
    SHILPS_XYZ     = 4,
    SHILPS_NDATA
};

typedef struct {
    gint start;
    gint end;
    gchar *name;
} ShilpsSegment;

static gint          shilps_detect                 (const GwyFileDetectInfo *fileinfo,
                                                    gboolean only_name);
static GwyContainer* shilps_load                   (const gchar *filename,
                                                    GwyRunType mode,
                                                    GError **error);
static void          shilps_filter_meta            (GenericHDF5File *ghfile);
static gint          shilps_read_images            (GwyContainer *data,
                                                    hid_t file_id,
                                                    GenericHDF5File *ghfile,
                                                    GError **error);
static GwyDataField* shilps_read_field             (hid_t file_id,
                                                    gint id,
                                                    gint xres,
                                                    gint yres,
                                                    gdouble xreal,
                                                    gdouble yreal,
                                                    const gchar *xyunitstr,
                                                    GString *str,
                                                    GError **error);
static gint          shilps_read_graphs            (GwyContainer *container,
                                                    hid_t file_id,
                                                    GenericHDF5File *ghfile,
                                                    GError **error);
static gboolean      shilps_read_graph_spec        (hid_t file_id,
                                                    gint id,
                                                    const gint *plot_map,
                                                    gint ngraphs,
                                                    GArray *spectra,
                                                    GArray *segments,
                                                    gboolean first_time,
                                                    GString *str,
                                                    GError **error);
static gint          shilps_read_spectra           (GwyContainer *container,
                                                    hid_t file_id,
                                                    GenericHDF5File *ghfile,
                                                    GError **error);
static GwySpectra*   shilps_read_curves_spec       (hid_t file_id,
                                                    gint id,
                                                    GString *str,
                                                    GError **error);
static gint          shilps_read_volumes           (GwyContainer *container,
                                                    hid_t file_id,
                                                    GenericHDF5File *ghfile,
                                                    GError **error);
static GwyBrick*     shilps_read_brick             (hid_t file_id,
                                                    gint id,
                                                    gint xres,
                                                    gint yres,
                                                    gint zres,
                                                    gdouble xreal,
                                                    gdouble yreal,
                                                    gdouble zreal,
                                                    const gchar *xyunitstr,
                                                    GString *str,
                                                    GError **error);
static gint          shilps_read_xyzs              (GwyContainer *container,
                                                    hid_t file_id,
                                                    GenericHDF5File *ghfile,
                                                    GError **error);
static GwySurface*   shilps_read_surface           (hid_t file_id,
                                                    gint id,
                                                    const gchar *xyunitstr,
                                                    GString *str,
                                                    GError **error);
static void          shilps_set_spectra_coordinates(GwyContainer *container,
                                                    gint nspec,
                                                    gint nxyz);

/* Generic */
static gint          ghdf5_detect             (const GwyFileDetectInfo *fileinfo,
                                               gboolean only_name);
static gint          mat7x_detect             (const GwyFileDetectInfo *fileinfo,
                                               gboolean only_name);
static GwyContainer* ghdf5_load               (const gchar *filename,
                                               GwyRunType mode,
                                               GError **error);
static hid_t         quick_check_hdf5         (const GwyFileDetectInfo *fileinfo,
                                               gboolean only_name);
static void          generic_hdf5_init        (GenericHDF5File *ghfile);
static void          generic_hdf5_alloc_lists (GenericHDF5File *ghfile,
                                               guint n);
static void          generic_hdf5_free        (GenericHDF5File *ghfile);
static hid_t         make_string_type_for_attr(hid_t attr_type);
static herr_t        scan_file                (hid_t loc_id,
                                               const char *name,
                                               const H5L_info_t *info,
                                               void *user_data);
static herr_t        process_attribute        (hid_t loc_id,
                                               const char *attr_name,
                                               const H5A_info_t *ainfo,
                                               void *user_data);
static gboolean      get_ints_attr            (hid_t file_id,
                                               const gchar *obj_path,
                                               const gchar *attr_name,
                                               gint expected_rank,
                                               gint *expected_dims,
                                               gint *v,
                                               GError **error);
static gboolean      get_int_attr             (hid_t file_id,
                                               const gchar *obj_path,
                                               const gchar *attr_name,
                                               gint *v,
                                               GError **error);
static gboolean      get_floats_attr          (hid_t file_id,
                                               const gchar *obj_path,
                                               const gchar *attr_name,
                                               gint expected_rank,
                                               gint *expected_dims,
                                               gdouble *v,
                                               GError **error);
static gboolean      get_float_attr           (hid_t file_id,
                                               const gchar *obj_path,
                                               const gchar *attr_name,
                                               gdouble *v,
                                               GError **error);
static gboolean      get_strs_attr            (hid_t file_id,
                                               const gchar *obj_path,
                                               const gchar *attr_name,
                                               gint expected_rank,
                                               gint *expected_dims,
                                               gchar **v,
                                               GError **error);
static gboolean      get_str_attr             (hid_t file_id,
                                               const gchar *obj_path,
                                               const gchar *attr_name,
                                               gchar **v,
                                               GError **error);
static gboolean      get_str_attr_g           (hid_t file_id,
                                               const gchar *obj_path,
                                               const gchar *attr_name,
                                               gchar **v,
                                               GError **error);
static hid_t         open_and_check_attr      (hid_t file_id,
                                               const gchar *obj_path,
                                               const gchar *attr_name,
                                               H5T_class_t expected_class,
                                               gint expected_rank,
                                               gint *expected_dims,
                                               GError **error);
static hid_t         open_and_check_dataset   (hid_t file_id,
                                               const gchar *name,
                                               gint expected_ndims,
                                               gint *dims,
                                               GError **error);
static gboolean      enumerate_indexed        (GString *path,
                                               const gchar *prefix,
                                               GArray *array);
static void          err_HDF5                 (GError **error,
                                               const gchar *where,
                                               glong code);

static GwyModuleInfo module_info = {
    GWY_MODULE_ABI_VERSION,
    &module_register,
    N_("Imports files based on Hierarchical Data Format (HDF), version 5."),
    "Yeti <yeti@gwyddion.net>",
    "1.3",
    "David Nečas (Yeti) & Petr Klapetek",
    "2020",
};

GWY_MODULE_QUERY2(module_info, hdf5file)

static gboolean
module_register(void)
{
    if (H5open() < 0) {
        g_warning("H5open() failed.");
        return FALSE;
    }
#ifndef DEBUG
    H5Eset_auto2(H5E_DEFAULT, NULL, NULL);
#endif

    gwy_file_func_register("hdf5generic",
                           N_("Generic HDF5 files (.h5)"),
                           (GwyFileDetectFunc)&ghdf5_detect,
                           (GwyFileLoadFunc)&ghdf5_load,
                           NULL,
                           NULL);
    /* Read MAT 7.x files using the generic HDF5 reader. None of the Matlab specific stuff is meaningful for us
     * anyway. */
    gwy_file_func_register("mat7xfile",
                           N_("Matlab 7.x HDF5 MAT files (.mat)"),
                           (GwyFileDetectFunc)&mat7x_detect,
                           (GwyFileLoadFunc)&ghdf5_load,
                           NULL,
                           NULL);
    gwy_file_func_register("ergofile",
                           N_("Asylum Research Ergo HDF5 files (.h5)"),
                           (GwyFileDetectFunc)&ergo_detect,
                           (GwyFileLoadFunc)&ergo_load,
                           NULL,
                           NULL);
    gwy_file_func_register("shilpsfile",
                           N_("Shilps Sciences Lucent HDF5 files (.h5)"),
                           (GwyFileDetectFunc)&shilps_detect,
                           (GwyFileLoadFunc)&shilps_load,
                           NULL,
                           NULL);

    return TRUE;
}

/*******************************************************************************************************************
 *
 * Plain/generic HDF5
 *
 *******************************************************************************************************************/

static gint
ghdf5_detect(const GwyFileDetectInfo *fileinfo,
            gboolean only_name)
{
    hid_t file_id;

    if ((file_id = quick_check_hdf5(fileinfo, only_name)) < 0)
        return 0;

    H5Fclose(file_id);

    /* Return a moderate score. We have no idea if we can actually read anything from the file. */
    return 50;
}

static gint
mat7x_detect(const GwyFileDetectInfo *fileinfo,
             gboolean only_name)
{
    GwyFileDetectInfo subinfo;
    hid_t file_id;

    if (fileinfo->buffer_len <= MAT7x_HEADER_SIZE)
        return 0;

    if (memcmp(fileinfo->head, MAGIC_MAT70, MAGIC_MAT70_SIZE) && memcmp(fileinfo->head, MAGIC_MAT73, MAGIC_MAT73_SIZE))
        return 0;

    subinfo = *fileinfo;
    subinfo.buffer_len = fileinfo->buffer_len - MAT7x_HEADER_SIZE;
    subinfo.head = fileinfo->head + MAT7x_HEADER_SIZE;
    subinfo.tail = fileinfo->tail + MAT7x_HEADER_SIZE;

    if ((file_id = quick_check_hdf5(&subinfo, only_name)) < 0)
        return 0;

    H5Fclose(file_id);

    return 75;
}

static GwyContainer*
ghdf5_load(const gchar *filename,
           G_GNUC_UNUSED GwyRunType mode,
           GError **error)
{
    GwyContainer *container = NULL;
    GenericHDF5File ghfile;
    hid_t file_id, dataset;
    G_GNUC_UNUSED herr_t status;
    H5O_info_t infobuf;
    guint i;
    gint id;

    file_id = H5Fopen(filename, H5F_ACC_RDONLY, H5P_DEFAULT);
    gwy_debug("file_id %d", (gint)file_id);
    status = H5Oget_info(file_id, &infobuf);
    gwy_debug("status %d", status);
    if (status < 0) {
        err_HDF5(error, "H5Oget_info", status);
        H5Fclose(file_id);
        return NULL;
    }

    generic_hdf5_init(&ghfile);
    g_array_append_val(ghfile.addr, infobuf.addr);
    status = H5Literate(file_id, H5_INDEX_NAME, H5_ITER_NATIVE, NULL, scan_file, &ghfile);
    if (status < 0) {
        err_HDF5(error, "H5Literate", status);
        H5Fclose(file_id);
        generic_hdf5_free(&ghfile);
        return NULL;
    }

    /* Read generic simple 2D data, i.e. images.
     * TODO: May want to read also other data types. */
    id = 0;
    for (i = 0; i < ghfile.datasets->len; i++) {
        gchar *name = g_array_index(ghfile.datasets, gchar*, i);
        gint dims[2] = { -1, -1 };
        GwyDataField *field;

        if ((dataset = open_and_check_dataset(file_id, name, 2, dims, NULL)) < 0)
            continue;
        if (err_DIMENSION(NULL, dims[0]) || err_DIMENSION(NULL, dims[1]))
            continue;
        field = gwy_data_field_new(dims[1], dims[0], dims[1], dims[0], FALSE);
        status = H5Dread(dataset, H5T_NATIVE_DOUBLE, H5S_ALL, H5S_ALL, H5P_DEFAULT, gwy_data_field_get_data(field));
        gwy_debug("status %d", status);
        H5Dclose(dataset);
        if (status < 0) {
            g_object_unref(field);
            continue;
        }

        if (!container)
            container = gwy_container_new();

        gwy_container_pass_object(container, gwy_app_get_data_key_for_id(id), field);
        id++;
    }

    status = H5Fclose(file_id);
    gwy_debug("status %d", status);

    generic_hdf5_free(&ghfile);

    if (!container)
        err_NO_DATA(error);

    return container;
}

/*******************************************************************************************************************
 *
 * Asylum Research Ergo
 *
 *******************************************************************************************************************/

static gint
ergo_detect(const GwyFileDetectInfo *fileinfo,
            gboolean only_name)
{
    hid_t file_id;
    gchar *format = NULL;
    gint version[3], dim = 3;
    gint score = 0;

    if ((file_id = quick_check_hdf5(fileinfo, only_name)) < 0)
        return 0;

    if (get_str_attr(file_id, ".", "ARFormat", &format, NULL)) {
        if (get_ints_attr(file_id, ".", "ARVersion", 1, &dim, version, NULL))
            score = 100;
        H5free_memory(format);
    }

    H5Fclose(file_id);

    return score;
}

static GwyContainer*
ergo_load(const gchar *filename,
          G_GNUC_UNUSED GwyRunType mode,
          GError **error)
{
    GwyContainer *container = NULL;
    GenericHDF5File ghfile;
    ErgoFile efile;
    hid_t file_id;
    G_GNUC_UNUSED herr_t status;
    H5O_info_t infobuf;
    guint i;

    file_id = H5Fopen(filename, H5F_ACC_RDONLY, H5P_DEFAULT);
    gwy_debug("file_id %d", (gint)file_id);
    status = H5Oget_info(file_id, &infobuf);
    gwy_debug("status %d", status);
    if (status < 0) {
        err_HDF5(error, "H5Oget_info", status);
        H5Fclose(file_id);
        return NULL;
    }

    generic_hdf5_init(&ghfile);
    ghfile.impl = &efile;
    ghfile.attr_handler = ergo_attr_handler;
    generic_hdf5_alloc_lists(&ghfile, 1);
    ghfile.lists[0].idprefix = "/DataSet/Resolution ";
    ghfile.lists[0].idwhat = H5O_TYPE_GROUP;
    g_array_append_val(ghfile.addr, infobuf.addr);

    gwy_clear(&efile, 1);
    efile.channels = g_array_new(FALSE, FALSE, sizeof(ErgoChannel));

    status = H5Literate(file_id, H5_INDEX_NAME, H5_ITER_NATIVE, NULL, scan_file, &ghfile);
    if (status < 0) {
        err_HDF5(error, "H5Literate", status);
        H5Fclose(file_id);
        g_array_free(efile.channels, TRUE);
        generic_hdf5_free(&ghfile);
        return NULL;
    }
    H5Aiterate2(file_id, H5_INDEX_NAME, H5_ITER_NATIVE, NULL, process_attribute, &ghfile);

    if (get_int_attr(file_id, "DataSetInfo", "NumFrames", &efile.nframes, error)) {
        gwy_debug("nframes %d", efile.nframes);
        container = ergo_read_channels(file_id, &ghfile, error);
    }

    status = H5Fclose(file_id);
    gwy_debug("status %d", status);

    for (i = 0; i < efile.channels->len; i++) {
        ErgoChannel *c = &g_array_index(efile.channels, ErgoChannel, i);
        g_free(c->name);
        GWY_OBJECT_UNREF(c->xyunit);
        GWY_OBJECT_UNREF(c->zunit);
    }
    g_array_free(efile.channels, TRUE);
    generic_hdf5_free(&ghfile);

    return container;
}

static GwyContainer*
ergo_read_channels(hid_t file_id, GenericHDF5File *ghfile, GError **error)
{
    ErgoFile *efile = (ErgoFile*)ghfile->impl;
    GwyContainer *meta, *container = NULL;
    GArray *channels = efile->channels;
    GArray *resolutions = ghfile->lists[0].idlist;
    GString *str = ghfile->buf;
    GwyDataField *dfield;
    gint expected2[2] = { 2, 2 }, yxres[2];
    gchar *s, *s2[2];
    gint frameid, id = 0;
    guint i, ri, r;

    for (ri = 0; ri < resolutions->len; ri++) {
        r = g_array_index(resolutions, guint, ri);
        for (i = 0; i < channels->len; i++) {
            ErgoChannel *c = &g_array_index(channels, ErgoChannel, i);

            g_string_printf(str, "DataSetInfo/Global/Channels/%s/ImageDims", c->name);

            if (!get_str_attr(file_id, str->str, "DataUnits", &s, error))
                goto fail;
            gwy_debug("zunit of %s is %s", c->name, s);
            c->zunit = gwy_si_unit_new_parse(s, &c->zpower10);
            H5free_memory(s);

            if (!get_strs_attr(file_id, str->str, "DimUnits", 1, expected2, s2, error))
                goto fail;
            gwy_debug("xyunits of %s are %s and %s", c->name, s2[0], s2[1]);
            if (!gwy_strequal(s2[0], s2[1]))
                g_warning("X and Y units differ, using X");
            c->xyunit = gwy_si_unit_new_parse(s2[1], &c->xypower10);
            H5free_memory(s2[0]);
            H5free_memory(s2[1]);

            /* NB: In all dimensions y is first, then x. */
            if (!get_floats_attr(file_id, str->str, "DimScaling", 2, expected2, c->realcoords, error))
                goto fail;
            gwy_debug("dims of %s are [%g, %g], [%g, %g]",
                      c->name, c->realcoords[2], c->realcoords[3], c->realcoords[0], c->realcoords[1]);

            g_string_append_printf(str, "/Resolution %d", r);
            if (!get_ints_attr(file_id, str->str, "DimExtents", 1, expected2, yxres, error))
                goto fail;
            gwy_debug("resid %u res %dx%d", r, yxres[1], yxres[0]);

            for (frameid = 0; frameid < efile->nframes; frameid++) {
                if (!(dfield = ergo_read_field(file_id, r, c, frameid, yxres, str, error)))
                    goto fail;

                if (!container)
                    container = gwy_container_new();

                gwy_container_pass_object(container, gwy_app_get_data_key_for_id(id), dfield);

                gwy_container_set_const_string(container, gwy_app_get_data_title_key_for_id(id), c->name);

                meta = gwy_container_duplicate(ghfile->meta);
                gwy_container_pass_object(container, gwy_app_get_data_meta_key_for_id(id), meta);

                id++;
            }
        }
    }

    if (container)
        return container;

    err_NO_DATA(error);

fail:
    GWY_OBJECT_UNREF(container);
    return NULL;
}

static GwyDataField*
ergo_read_field(hid_t file_id,
                guint r, ErgoChannel *c, gint frameid, const gint *yxres,
                GString *str, GError **error)
{
    GwyDataField *dfield;
    hid_t dataset;
    gdouble q, xreal, yreal, xoff, yoff;
    herr_t status;

    g_string_printf(str, "DataSet/Resolution %u/Frame %d/%s/Image", r, frameid, c->name);
    if ((dataset = open_and_check_dataset(file_id, str->str, 2, (gint*)yxres, error)) < 0)
        return NULL;

    q = pow10(c->xypower10);

    /* NB: In all dimensions y is first, then x. */
    xreal = c->realcoords[3] - c->realcoords[2];
    sanitise_real_size(&xreal, "x size");
    xoff = MIN(c->realcoords[2], c->realcoords[3]);

    yreal = c->realcoords[1] - c->realcoords[0];
    sanitise_real_size(&yreal, "y size");
    yoff = MIN(c->realcoords[0], c->realcoords[1]);

    dfield = gwy_data_field_new(yxres[1], yxres[0], q*xreal, q*yreal, FALSE);
    gwy_data_field_set_xoffset(dfield, q*xoff);
    gwy_data_field_set_yoffset(dfield, q*yoff);
    gwy_si_unit_assign(gwy_data_field_get_si_unit_xy(dfield), c->xyunit);
    gwy_si_unit_assign(gwy_data_field_get_si_unit_z(dfield), c->zunit);

    status = H5Dread(dataset, H5T_NATIVE_DOUBLE, H5S_ALL, H5S_ALL, H5P_DEFAULT, gwy_data_field_get_data(dfield));
    H5Dclose(dataset);
    gwy_data_field_invert(dfield, TRUE, FALSE, FALSE);
    if (c->zpower10)
        gwy_data_field_multiply(dfield, pow10(c->zpower10));

    if (status < 0) {
        err_HDF5(error, "H5Dread", status);
        GWY_OBJECT_UNREF(dfield);
    }
    return dfield;
}

static void
append_channel_name(GArray *channels, const gchar *name)
{
    ErgoChannel c;

    gwy_debug("found channel %s", name);
    gwy_clear(&c, 1);
    c.name = g_strdup(name);
    g_strstrip(c.name);
    g_array_append_val(channels, c);
}

/* XXX: Handle /DataSetInfo/ChannelNames which do not have unique paths and we just build them during the scan */
static void
ergo_attr_handler(GenericHDF5File *ghfile, hid_t loc_id, const char *attr_name)
{
    ErgoFile *efile = (ErgoFile*)ghfile->impl;
    G_GNUC_UNUSED H5T_cset_t cset = H5T_CSET_ERROR;
    H5T_class_t type_class;
    hid_t attr, attr_type, str_type, space;
    gboolean is_vlenstr = FALSE;
    gint nitems, i;
    herr_t status;

    if (!gwy_strequal(ghfile->path->str, "/DataSetInfo/ChannelNames"))
        return;

    gwy_debug("handling /DataSetInfo/ChannelNames");
    attr = H5Aopen(loc_id, attr_name, H5P_DEFAULT);
    attr_type = H5Aget_type(attr);
    space = H5Aget_space(attr);
    nitems = H5Sget_simple_extent_npoints(space);
    type_class = H5Tget_class(attr_type);

    if (gwy_strequal(ghfile->path->str, "/DataSetInfo/ChannelNames")) {
        if (type_class == H5T_STRING) {
            is_vlenstr = H5Tis_variable_str(attr_type);
            cset = H5Tget_cset(attr_type);
        }

        if (type_class == H5T_STRING && is_vlenstr) {
            if (nitems == 1) {
                gchar *s;

                str_type = make_string_type_for_attr(attr_type);
                if ((status = H5Aread(attr, str_type, &s)) >= 0) {
                    append_channel_name(efile->channels, s);
                    H5free_memory(s);
                }
                H5Tclose(str_type);
            }
            else if (nitems > 0) {
                gchar **s = g_new(gchar*, nitems);

                str_type = make_string_type_for_attr(attr_type);
                if ((status = H5Aread(attr, str_type, s)) >= 0) {
                    for (i = 0; i < nitems; i++) {
                        append_channel_name(efile->channels, s[i]);
                        H5free_memory(s[i]);
                    }
                }
                H5Tclose(str_type);
                g_free(s);
            }
        }
    }

    H5Sclose(space);
    H5Tclose(attr_type);
    H5Aclose(attr);
}

/*******************************************************************************************************************
 *
 * Shilps Sciences Lucent
 *
 *******************************************************************************************************************/

static gint
shilps_detect(const GwyFileDetectInfo *fileinfo,
              gboolean only_name)
{
    hid_t file_id;
    gchar *company;
    gint score = 0;

    if ((file_id = quick_check_hdf5(fileinfo, only_name)) < 0)
        return 0;

    if (get_str_attr(file_id, ".", "Company", &company, NULL)) {
        if (gwy_strequal(company, "Shilps Sciences"))
            score = 100;
        H5free_memory(company);
    }

    H5Fclose(file_id);

    return score;
}

static GwyContainer*
shilps_load(const gchar *filename,
            G_GNUC_UNUSED GwyRunType mode,
            GError **error)
{
    GwyContainer *container;
    GenericHDF5File ghfile;
    hid_t file_id;
    G_GNUC_UNUSED herr_t status;
    H5O_info_t infobuf;
    guint i;
    gint nspec, nxyz;

    file_id = H5Fopen(filename, H5F_ACC_RDONLY, H5P_DEFAULT);
    gwy_debug("file_id %d", (gint)file_id);
    status = H5Oget_info(file_id, &infobuf);
    gwy_debug("status %d", status);
    if (status < 0) {
        err_HDF5(error, "H5Oget_info", status);
        H5Fclose(file_id);
        return NULL;
    }

    generic_hdf5_init(&ghfile);
    generic_hdf5_alloc_lists(&ghfile, SHILPS_NDATA);
    /* Having two things called spectra makes a bit of a mess in the corresponding function names. */
    ghfile.lists[SHILPS_IMAGES].idprefix = "/Session/Scan/Image";
    ghfile.lists[SHILPS_SPECTRA].idprefix = "/Session/Spectra/Graph";
    ghfile.lists[SHILPS_CURVES].idprefix = "/Session/Curves/Spectrum";
    ghfile.lists[SHILPS_VOLUME].idprefix = "/Session/Volume/Image3d";
    ghfile.lists[SHILPS_XYZ].idprefix = "/Session/XYZ/Scatter";
    for (i = 0; i < SHILPS_NDATA; i++)
        ghfile.lists[i].idwhat = H5O_TYPE_DATASET;
    g_array_append_val(ghfile.addr, infobuf.addr);

    status = H5Literate(file_id, H5_INDEX_NAME, H5_ITER_NATIVE, NULL, scan_file, &ghfile);
    if (status < 0) {
        err_HDF5(error, "H5Literate", status);
        H5Fclose(file_id);
        generic_hdf5_free(&ghfile);
        return NULL;
    }
    H5Aiterate2(file_id, H5_INDEX_NAME, H5_ITER_NATIVE, NULL, process_attribute, &ghfile);
    shilps_filter_meta(&ghfile);

    container = gwy_container_new();
    if (shilps_read_images(container, file_id, &ghfile, error) < 0
        || shilps_read_volumes(container, file_id, &ghfile, error) < 0
        || shilps_read_graphs(container, file_id, &ghfile, error) < 0
        || (nxyz = shilps_read_xyzs(container, file_id, &ghfile, error)) < 0
        || (nspec = shilps_read_spectra(container, file_id, &ghfile, error)) < 0)
        GWY_OBJECT_UNREF(container);

    status = H5Fclose(file_id);
    gwy_debug("status %d", status);
    generic_hdf5_free(&ghfile);

    shilps_set_spectra_coordinates(container, nspec, nxyz);

    /* All data types are optional so we might not have failed but also have not read anything. */
    if (container && !gwy_container_get_n_items(container)) {
        GWY_OBJECT_UNREF(container);
        err_NO_DATA(error);
    }

    return container;
}

static void
gather_to_remove(gpointer key, G_GNUC_UNUSED gpointer value, gpointer user_data)
{
    static const gchar *prefixes[] = {
        "Session::Scan::Image", "Session::Spectra::Graph", "Session::Curves::Spectrum", "Session::Volume::Image3d",
        "Session::XYZ:Scatter",
    };
    static guint lenghts[G_N_ELEMENTS(prefixes)] = { 0, };

    GArray *to_remove = (GArray*)user_data;
    GQuark quark = GPOINTER_TO_UINT(key);
    const gchar *s = g_quark_to_string(quark);
    guint i, len;

    if (!lenghts[0]) {
        for (i = 0; i < G_N_ELEMENTS(prefixes); i++)
            lenghts[i] = strlen(prefixes[i]);
    }
    for (i = 0; i < G_N_ELEMENTS(prefixes); i++) {
        len = lenghts[i];
        if (strncmp(s, prefixes[i], len) == 0 && g_ascii_isdigit(s[len])) {
            gwy_debug("filtering out metadata %s", s);
            g_array_append_val(to_remove, quark);
            return;
        }
    }
}

/* Filter out data-specific items. We could do a more refined filtering, keeping information pertaining to the
 * individual data pieces (creating different metadata for each data piece). But there is not much useful there to
 * keep anyway. */
static void
shilps_filter_meta(GenericHDF5File *ghfile)
{
    GwyContainer *meta = ghfile->meta;
    GArray *quarks = g_array_new(FALSE, FALSE, sizeof(GQuark));
    guint i;

    gwy_container_foreach(meta, NULL, gather_to_remove, quarks);
    for (i = 0; i < quarks->len; i++)
        gwy_container_remove(meta, g_array_index(quarks, GQuark, i));
    g_array_free(quarks, TRUE);
}

static gint
shilps_read_images(GwyContainer *container,
                   hid_t file_id, GenericHDF5File *ghfile, GError **error)
{
    const gchar grouppfx[] = "Session/Scan";
    GArray *images = ghfile->lists[SHILPS_IMAGES].idlist;
    GString *str = ghfile->buf;
    GwyDataField *dfield;
    gint xres, yres, power10;
    gdouble xreal, yreal;
    gchar *channel, *scancyc, *title, *xyunitstr;
    GwyContainer *meta;
    guint i, id;

    gwy_debug("nimages %u", images->len);
    if (!images->len)
        return 0;

    if (!get_int_attr(file_id, grouppfx, "X No", &xres, error)
        || !get_int_attr(file_id, grouppfx, "Y No", &yres, error)
        || !get_float_attr(file_id, grouppfx, "X Range", &xreal, error)
        || !get_float_attr(file_id, grouppfx, "Y Range", &yreal, error))
        return -1;

    gwy_debug("xres %d, yres %d", xres, yres);
    if (err_DIMENSION(error, xres) || err_DIMENSION(error, yres))
        return -1;

    gwy_debug("xreal %g, yreal %g", xreal, yreal);
    sanitise_real_size(&xreal, "x size");
    sanitise_real_size(&yreal, "y size");

    if (!get_str_attr_g(file_id, grouppfx, "Range Units", &xyunitstr, NULL))
        xyunitstr = g_strdup("µm");
    g_object_unref(gwy_si_unit_new_parse(xyunitstr, &power10));
    xreal *= pow10(power10);
    yreal *= pow10(power10);

    for (i = 0; i < images->len; i++) {
        id = g_array_index(images, gint, i);
        if (!(dfield = shilps_read_field(file_id, id, xres, yres, xreal, yreal, xyunitstr, str, error))) {
            g_free(xyunitstr);
            return -1;
        }

        gwy_container_pass_object(container, gwy_app_get_data_key_for_id(i), dfield);

        /* shilps_read_field() fills str->str with the correct prefix. */
        if (get_str_attr(file_id, str->str, "Channel", &channel, NULL)) {
            g_strstrip(channel);
            if (get_str_attr(file_id, str->str, "Scan cycle", &scancyc, NULL)) {
                g_strstrip(scancyc);
                title = g_strdup_printf("%s (%s)", channel, scancyc);
                gwy_container_set_const_string(container, gwy_app_get_data_title_key_for_id(i), title);
                g_free(title);
                H5free_memory(scancyc);
            }
            else
                gwy_container_set_const_string(container, gwy_app_get_data_title_key_for_id(i), channel);
            H5free_memory(channel);
        }
        else
            gwy_app_channel_title_fall_back(container, i);

        meta = gwy_container_duplicate(ghfile->meta);
        gwy_container_pass_object(container, gwy_app_get_data_meta_key_for_id(i), meta);
    }
    g_free(xyunitstr);

    return images->len;
}

static GwyDataField*
shilps_read_field(hid_t file_id, gint id,
                  gint xres, gint yres, gdouble xreal, gdouble yreal,
                  const gchar *xyunitstr,
                  GString *str, GError **error)
{
    GwyDataField *dfield = NULL;
    hid_t dataset;
    gint power10;
    gint yxres[2] = { yres, xres };
    gchar *zunitstr = NULL;
    herr_t status = -1;

    g_string_printf(str, "Session/Scan/Image%d", id);
    if ((dataset = open_and_check_dataset(file_id, str->str, 2, yxres, error)) < 0)
        return NULL;

    get_str_attr_g(file_id, str->str, "Units", &zunitstr, NULL);
    gwy_debug("Range units %s, data Units %s", xyunitstr, zunitstr);

    dfield = gwy_data_field_new(xres, yres, xreal, yreal, TRUE);
    gwy_si_unit_set_from_string(gwy_data_field_get_si_unit_xy(dfield), xyunitstr);
    gwy_si_unit_set_from_string_parse(gwy_data_field_get_si_unit_z(dfield), zunitstr, &power10);
    g_free(zunitstr);

    status = H5Dread(dataset, H5T_NATIVE_DOUBLE, H5S_ALL, H5S_ALL, H5P_DEFAULT, gwy_data_field_get_data(dfield));
    H5Dclose(dataset);

    if (status < 0) {
        err_HDF5(error, "H5Dread", status);
        GWY_OBJECT_UNREF(dfield);
    }
    else
        gwy_data_field_multiply(dfield, pow10(power10));

    return dfield;
}

/* This also reads spectra. In fact, this is probably the right SPS data type and the other should be some kind of
 * curve map. */
static gint
shilps_read_graphs(GwyContainer *container,
                   hid_t file_id, GenericHDF5File *ghfile, GError **error)
{
    const gchar grouppfx[] = "Session/Spectra";
    GArray *graphs = ghfile->lists[SHILPS_SPECTRA].idlist;
    GArray *segments;
    GString *str = ghfile->buf;
    GArray *spectra;
    GwySpectra *spec;
    guint i, id;
    hid_t attr;
    gint plots_dims[2] = { -1, 2 };
    gint *plot_map = NULL;
    gboolean ok = FALSE;
    gdouble *d;

    gwy_debug("ngraphs %u", graphs->len);
    if (!graphs->len)
        return 0;

    /* The plot map is a N × 2 array giving (abscissa, ordinate) pairs to plot. If we do not have it we fall back to
     * plotting everything as a function of the data point number. */
    if ((attr = open_and_check_attr(file_id, grouppfx, "Plots", H5T_INTEGER, 2, plots_dims, NULL)) >= 0) {
        H5Aclose(attr);
        gwy_debug("found Plots array %d x %d", plots_dims[0], plots_dims[1]);
        plot_map = g_new(gint, plots_dims[0]*plots_dims[1]);
        if (!get_ints_attr(file_id, grouppfx, "Plots", 2, plots_dims, plot_map, NULL))
            GWY_FREE(plot_map);
        for (i = 0; (gint)i < plots_dims[0]*plots_dims[1]; i++) {
            /* NB: The numbers are indexed from 1, like channel numbers. */
            if (plot_map[i] <= 0) {
                g_warning("Invalid Plots map index #%u: %d < 0", i, plot_map[i]);
                GWY_FREE(plot_map);
                break;
            }
        }
    }

    spectra = g_array_new(FALSE, TRUE, sizeof(GwySpectra*));
    segments = g_array_new(FALSE, FALSE, sizeof(ShilpsSegment));
    if (plot_map)
        g_array_set_size(spectra, plots_dims[0]);

    for (i = 0; i < graphs->len; i++) {
        id = g_array_index(graphs, gint, i);
        if (!shilps_read_graph_spec(file_id, id, plot_map, plot_map ? plots_dims[0] : -1, spectra, segments, !i,
                                    str, error))
            goto fail;
    }

    for (i = 0; i < spectra->len; i++) {
        spec = g_array_index(spectra, GwySpectra*, i);
        if (spec)
            gwy_container_set_object(container, gwy_app_get_spectra_key_for_id(i), spec);
    }
    ok = TRUE;

fail:
    for (i = 0; i < spectra->len; i++) {
        spec = g_array_index(spectra, GwySpectra*, i);
        if (spec) {
            if ((d = g_object_get_data(G_OBJECT(spec), "q"))) {
                g_object_set_data(G_OBJECT(spec), "q", NULL);
                g_free(d);
            }
            g_object_unref(spec);
        }
    }
    g_array_free(spectra, TRUE);
    for (i = 0; i < segments->len; i++)
        g_free(g_array_index(segments, ShilpsSegment, i).name);
    g_array_free(segments, TRUE);
    g_free(plot_map);

    return ok ? graphs->len : -1;
}

/* XXX: This is what we do if we don't know which column should be the abscissa. Probably can happen only for
 * early stage (non-public) files or when something goes awry. Normal files should have the plot map.
 *
 * We give up and show point index on abscissa. The good part is that we do not have any consistency requirement on
 * the channels in individual points and can basically merge anything. */
static void
shilps_read_spec_fallback(hid_t file_id, GArray *spectra,
                          const gdouble *values, gint nrow, gint ncol,
                          gdouble x, gdouble y,
                          GString *str)
{
    GString *attrname;
    gint i, j;

    attrname = g_string_new(NULL);
    for (i = 0; i < ncol; i++) {
        gchar *name = NULL, *unit = NULL;
        GwySpectra *spec;
        GwyDataLine *line;
        gint power10;
        gdouble *d;
        gdouble q;

        /* Skip unnamed spectra as we have to way to identify them. */
        g_string_printf(attrname, "Column%u_Channel", i+1);
        if (!get_str_attr_g(file_id, str->str, attrname->str, &name, NULL))
            continue;
        g_string_printf(attrname, "Column%u_Unit", i+1);
        get_str_attr_g(file_id, str->str, attrname->str, &unit, NULL);

        gwy_debug("col[%u] name=%s, unit=%s", i, name, unit);
        for (j = 0; j < spectra->len; j++) {
            spec = g_array_index(spectra, GwySpectra*, j);
            if (gwy_strequal(gwy_spectra_get_title(spec), name))
                break;
        }
        if (j == spectra->len) {
            spec = gwy_spectra_new();
            if (name) {
                gwy_spectra_set_title(spec, name);
                gwy_spectra_set_spectrum_y_label(spec, name);
            }
            gwy_spectra_set_spectrum_x_label(spec, "index");
            gwy_si_unit_set_from_string(gwy_spectra_get_si_unit_xy(spec), "m");
            g_array_append_val(spectra, spec);
        }

        line = gwy_data_line_new(nrow, nrow, FALSE);
        gwy_si_unit_set_from_string_parse(gwy_data_line_get_si_unit_y(line), unit, &power10);
        q = pow10(power10);

        d = gwy_data_line_get_data(line);
        for (j = 0; j < nrow; j++)
            d[j] = q*values[j*ncol + i];

        gwy_spectra_add_spectrum(spec, line, x, y);
        g_object_unref(line);

        g_free(name);
        g_free(unit);
    }
    g_string_free(attrname, TRUE);
}

static void
regularise_curve_sampling(const gdouble *xdata, const gdouble *ydata, gint n, gint stride,
                          gdouble qx, gdouble qy,
                          GwyDataLine *line)
{
    GwyDataLine *weight;
    gdouble xmin = G_MAXDOUBLE, xmax = -G_MAXDOUBLE;
    gdouble x, dx, s = 0.0;
    gdouble *d, *w;
    gint i, ix, res;

    for (i = 0; i < n; i++) {
        xmin = fmin(xmin, xdata[i*stride]);
        xmax = fmax(xmax, xdata[i*stride]);
    }
    xmin *= qx;
    xmax *= qx;

    if (!(xmax > xmin)) {
        res = 2;
        gwy_data_line_resample(line, res, GWY_INTERPOLATION_NONE);
        gwy_data_line_set_offset(line, xmin);
        gwy_data_line_set_real(line, xmin > 0.0 ? 0.1*xmin : 1.0);
        for (i = 0; i < n; i++)
            s += ydata[i*stride];
        d = gwy_data_line_get_data(line);
        d[0] = d[1] = qy*s/n;
        return;
    }

    res = n;
    dx = (xmax - xmin)/res;
    gwy_data_line_resample(line, res, GWY_INTERPOLATION_NONE);
    gwy_data_line_clear(line);
    gwy_data_line_set_real(line, xmax - xmin);
    gwy_data_line_set_offset(line, xmin);
    d = gwy_data_line_get_data(line);
    weight = gwy_data_line_new_alike(line, TRUE);
    w = gwy_data_line_get_data(weight);

    for (i = 0; i < n; i++) {
        x = (qx*xdata[i*stride] + 0.5*dx - xmin)/dx;
        ix = (gint)floor(x);
        ix = CLAMP(ix, 0, res-1);
        d[ix] += ydata[i*stride];
        w[ix]++;
    }

    for (i = 0; i < res; i++) {
        if (w[i] > 0.0) {
            d[i] *= qy/w[i];
            w[i] = 0.0;
        }
        else
            w[i] = 1.0;
    }

    gwy_data_line_correct_laplace(line, weight);

    g_object_unref(weight);
}

/* This is the usual code path. Plot the specified (absiccsa,ordinate) pairs and ignore anything else. */
static gboolean
shilps_read_spec_plotmap(hid_t file_id, GArray *spectra, GArray *segments,
                         const gdouble *values, gint nrow, gint ncol,
                         gdouble x, gdouble y,
                         const gint *plot_map, gint ngraphs,
                         GString *str,
                         G_GNUC_UNUSED GError **error)
{
    GString *attrname;
    gdouble qx, qy;
    gint start, end, i, j, k, nseg = segments->len ? segments->len : 1;

    attrname = g_string_new(NULL);
    for (i = 0; i < ngraphs; i++) {
        gint ia = plot_map[2*i], io = plot_map[2*i + 1];
        gchar *namea = NULL, *unita = NULL, *nameo = NULL, *unito = NULL;
        GwySpectra *spec;
        GwyDataLine *line;
        gint power10a, power10o;
        gdouble *d;
        gchar *title;

        if (ia > ncol || io > ncol) {
            g_warning("Too few spectrum columns (%d) for plot index %d", ncol, MAX(ia, io));
            continue;
        }

        for (j = 0; j < nseg; j++) {
            k = i*nseg + j;
            if (!(spec = g_array_index(spectra, GwySpectra*, k))) {
                g_string_printf(attrname, "Column%u_Channel", ia);
                get_str_attr_g(file_id, str->str, attrname->str, &namea, NULL);
                g_string_printf(attrname, "Column%u_Unit", ia);
                get_str_attr_g(file_id, str->str, attrname->str, &unita, NULL);
                g_string_printf(attrname, "Column%u_Channel", io);
                get_str_attr_g(file_id, str->str, attrname->str, &nameo, NULL);
                g_string_printf(attrname, "Column%u_Unit", io);
                get_str_attr_g(file_id, str->str, attrname->str, &unito, NULL);

                gwy_debug("creating spectrum %d (%s vs. %s) from (%d, %d)", i, namea, nameo, ia, io);
                spec = gwy_spectra_new();
                title = g_strdup_printf("%s-%s%s%s",
                                        nameo ? nameo : _("Unknown"),
                                        namea ? namea : _("Unknown"),
                                        segments->len ? " " : "",
                                        segments->len ? g_array_index(segments, ShilpsSegment, j).name : "");
                gwy_spectra_set_title(spec, title);
                g_free(title);
                if (namea)
                    gwy_spectra_set_spectrum_x_label(spec, namea);
                if (nameo)
                    gwy_spectra_set_spectrum_y_label(spec, nameo);

                gwy_si_unit_set_from_string(gwy_spectra_get_si_unit_xy(spec), "m");
                g_array_index(spectra, GwySpectra*, k) = spec;

                line = gwy_data_line_new(1, 1, FALSE);
                gwy_si_unit_set_from_string_parse(gwy_data_line_get_si_unit_x(line), unita, &power10a);
                gwy_si_unit_set_from_string_parse(gwy_data_line_get_si_unit_y(line), unito, &power10o);

                d = g_new(gdouble, 2);
                d[0] = qx = pow10(power10a);
                d[1] = qy = pow10(power10o);
                g_object_set_data(G_OBJECT(spec), "q", d);

                g_free(namea);
                g_free(nameo);
                g_free(unita);
                g_free(unito);
            }
            else {
                line = gwy_data_line_new_alike(gwy_spectra_get_spectrum(spec, 0), FALSE);
                d = (gdouble*)g_object_get_data(G_OBJECT(spec), "q");
                qx = d[0];
                qy = d[1];
            }

            if (segments->len) {
                start = g_array_index(segments, ShilpsSegment, j).start;
                end = g_array_index(segments, ShilpsSegment, j).end;
                start = CLAMP(start, 0, nrow-1);
                end = CLAMP(end, 1, nrow);
                regularise_curve_sampling(values + ia-1 + start*ncol, values + io-1 + start*ncol,
                                          end - start, ncol, qx, qy, line);
            }
            else
                regularise_curve_sampling(values + ia-1, values + io-1, nrow, ncol, qx, qy, line);
            gwy_spectra_add_spectrum(spec, line, x, y);
            g_object_unref(line);
        }
    }
    g_string_free(attrname, TRUE);

    return TRUE;
}

/* Read and parse curve segmentation given as an array of Name:start strings.
 * NB: We do not fill the last item of segmentation[] because we do not know the number of data points. The caller
 * has to do it afterwards. */
static gboolean
parse_curve_segmentation(hid_t file_id, const gchar *obj_path, const gchar *attr_name,
                         GArray *segments, GArray *tmpl,
                         GError **error)
{
    ShilpsSegment seg;
    hid_t attr;
    gchar **segnames = NULL;
    gchar *colon;
    gint i, nseg = -1;
    gboolean ok = FALSE;

    if ((attr = open_and_check_attr(file_id, obj_path, attr_name, H5T_STRING, 1, &nseg, error)) < 0)
        return FALSE;
    H5Aclose(attr);

    gwy_debug("nseg %d, tmpl nseg %d", nseg, tmpl ? (gint)tmpl->len : -1);
    if (tmpl && (guint)nseg != tmpl->len) {
        err_INCONSISTENT_SPECTRA(error);
        return FALSE;
    }

    segnames = g_new0(gchar*, nseg);
    if (!get_strs_attr(file_id, obj_path, attr_name, 1, &nseg, segnames, error)) {
        g_free(segnames);
        return FALSE;
    }

    for (i = 0; i < nseg; i++) {
        gwy_debug("segment spec[%d] %s", i, segnames[i]);
        if (!(colon = strchr(segnames[i], ':'))) {
            err_INVALID(error, attr_name);
            goto end;
        }
    }

    for (i = 0; i < nseg; i++) {
        colon = strchr(segnames[i], ':');
        *colon = '\0';
        if (tmpl && !gwy_strequal(segnames[i], g_array_index(tmpl, ShilpsSegment, i).name)) {
            err_INCONSISTENT_SPECTRA(error);
            goto end;
        }
        seg.name = segnames[i];
        seg.start = atoi(colon+1);
        if (i)
            g_array_index(segments, ShilpsSegment, i-1).end = seg.start;
        g_array_append_val(segments, seg);
        gwy_debug("segment spec[%d] %s, starts at %d", i, seg.name, seg.start);
    }

    /* Reallocate string using GLib to unify freeing. */
    for (i = 0; i < nseg; i++)
        g_array_index(segments, ShilpsSegment, i).name = g_strdup(g_array_index(segments, ShilpsSegment, i).name);
    ok = TRUE;

end:
    /* No need to free anything inside because if !ok we have not allocated it anew. */
    if (!ok)
        g_array_set_size(segments, 0);
    for (i = 0; i < nseg; i++)
        H5free_memory(segnames[i]);
    g_free(segnames);

    return ok;
}

static gboolean
shilps_read_graph_spec(hid_t file_id, gint id, const gint *plot_map, gint ngraphs,
                       GArray *spectra, GArray *segments, gboolean first_time,
                       GString *str, GError **error)
{
    gint res[2] = { -1, -1 };
    guint ncol, nrow;
    gdouble x, y;
    gdouble *values = NULL;
    GArray *this_segments = NULL;
    hid_t dataset;
    gint status;
    guint i;
    gboolean ok = FALSE;

    g_string_printf(str, "Session/Spectra/Graph%d", id);
    if ((dataset = open_and_check_dataset(file_id, str->str, 2, res, error)) < 0)
        return FALSE;

    nrow = res[0];
    ncol = res[1];
    gwy_debug("[%u]res rows=%u, cols=%u", id, nrow, ncol);
    if (err_DIMENSION(error, nrow) || err_DIMENSION(error, ncol))
        goto fail;

    if (!get_float_attr(file_id, str->str, "Point_X", &x, error)
        || !get_float_attr(file_id, str->str, "Point_Y", &y, error))
        goto fail;
    /* Micrometers. */
    x *= 1e-6;
    y *= 1e-6;
    gwy_debug("spectrum %d at (%g,%g)", id, x, y);

    if (first_time) {
        parse_curve_segmentation(file_id, str->str, "CurveSegments", segments, NULL, NULL);
        if (segments->len && plot_map)
            g_array_set_size(spectra, segments->len*spectra->len);
    }
    else {
        this_segments = g_array_new(FALSE, FALSE, sizeof(ShilpsSegment));
        parse_curve_segmentation(file_id, str->str, "CurveSegments", this_segments, segments, NULL);
        segments = this_segments;
    }
    if (segments->len)
        g_array_index(segments, ShilpsSegment, segments->len-1).end = nrow;

    values = g_new(gdouble, nrow*ncol);
    status = H5Dread(dataset, H5T_NATIVE_DOUBLE, H5S_ALL, H5S_ALL, H5P_DEFAULT, values);
    if (status < 0) {
        err_HDF5(error, "H5Dread", status);
        goto fail;
    }

    if (plot_map) {
        ok = shilps_read_spec_plotmap(file_id, spectra, segments, values, nrow, ncol, x, y, plot_map, ngraphs,
                                      str, error);
    }
    else {
        shilps_read_spec_fallback(file_id, spectra, values, nrow, ncol, x, y, str);
        ok = TRUE;
    }

fail:
    H5Dclose(dataset);
    g_free(values);
    if (this_segments) {
        for (i = 0; i < this_segments->len; i++)
            g_free(g_array_index(this_segments, ShilpsSegment, i).name);
        g_array_free(this_segments, TRUE);
    }

    return ok;
}

static gint
shilps_read_spectra(GwyContainer *container,
                    hid_t file_id, GenericHDF5File *ghfile, GError **error)
{
    //const gchar grouppfx[] = "Session/Curves";
    GArray *spectra = ghfile->lists[SHILPS_CURVES].idlist;
    GString *str = ghfile->buf;
    GwySpectra *spec;
    gchar *channel;
    guint i, id;

    gwy_debug("nspectra %u", spectra->len);
    if (!spectra->len)
        return 0;

    for (i = 0; i < spectra->len; i++) {
        id = g_array_index(spectra, gint, i);
        if (!(spec = shilps_read_curves_spec(file_id, id, str, error))) {
            return -1;
        }

        /* shilps_read_curves_spec() fills str->str with the correct prefix. */
        if (get_str_attr(file_id, str->str, "Channel", &channel, NULL)) {
            g_strstrip(channel);
            gwy_spectra_set_title(spec, channel);
            H5free_memory(channel);
        }

        gwy_container_pass_object(container, gwy_app_get_spectra_key_for_id(i), spec);
    }

    return spectra->len;
}

static GwySpectra*
shilps_read_curves_spec(hid_t file_id, gint id,
                        GString *str, GError **error)
{
    GwySpectra *spec = NULL;
    GwyDataLine *dline;
    hid_t dataset;
    gdouble *ydata, *cdata;
    gint i, npts, ncurves, power10;
    GwySIUnit *yunit;
    gint res[2] = { -1, -1 };
    gchar *yunitstr = NULL;
    gdouble q;
    herr_t status = -1;

    g_string_printf(str, "Session/Curves/Spectrum%d", id);
    if ((dataset = open_and_check_dataset(file_id, str->str, 2, res, error)) < 0)
        return NULL;
    gwy_debug("curve data dims ncurves=%d npts=%d", res[0], res[1]);
    ncurves = res[0];
    npts = res[1];
    if (err_DIMENSION(error, npts)) {
        H5Dclose(dataset);
        return NULL;
    }

    get_str_attr_g(file_id, str->str, "Units", &yunitstr, NULL);
    gwy_debug("data Units %s", yunitstr);

    /* We set units and point coordinates later using XYZ data. */
    yunit = gwy_si_unit_new_parse(yunitstr, &power10);
    g_free(yunitstr);
    q = pow10(power10);
    spec = gwy_spectra_new();

    status = 0;
    ydata = g_new(gdouble, npts*ncurves);
    status = H5Dread(dataset, H5T_NATIVE_DOUBLE, H5S_ALL, H5S_ALL, H5P_DEFAULT, ydata);
    H5Dclose(dataset);

    if (status < 0) {
        err_HDF5(error, "H5Dread", status);
        GWY_OBJECT_UNREF(spec);
    }
    else {
        for (i = 0; i < ncurves; i++) {
            dline = gwy_data_line_new(npts, npts, FALSE);
            cdata = gwy_data_line_get_data(dline);
            gwy_assign(cdata, ydata + i*npts, npts);
            gwy_data_line_multiply(dline, q);
            gwy_si_unit_assign(gwy_data_line_get_si_unit_y(dline), yunit);
            gwy_spectra_add_spectrum(spec, dline, i, 0.0);
            g_object_unref(dline);
        }
    }
    g_free(ydata);
    GWY_OBJECT_UNREF(yunit);

    return spec;
}

static gint
shilps_read_volumes(GwyContainer *container,
                    hid_t file_id, GenericHDF5File *ghfile, GError **error)
{
    const gchar grouppfx[] = "Session/Volume";
    GArray *volumes = ghfile->lists[SHILPS_VOLUME].idlist;
    GString *str = ghfile->buf;
    GwyBrick *brick;
    gint xres, yres, zres, power10;
    gdouble xreal, yreal, zreal;
    gchar *channel, *xyunitstr;
    GwyContainer *meta;
    guint i, id;

    gwy_debug("nvolumes %u", volumes->len);
    if (!volumes->len)
        return 0;

    if (!get_int_attr(file_id, grouppfx, "X No", &xres, error)
        || !get_int_attr(file_id, grouppfx, "Y No", &yres, error)
        || !get_int_attr(file_id, grouppfx, "Z No", &zres, error)
        || !get_float_attr(file_id, grouppfx, "X Range", &xreal, error)
        || !get_float_attr(file_id, grouppfx, "Y Range", &yreal, error)
        || !get_float_attr(file_id, grouppfx, "Z Range", &zreal, error))
        return -1;

    gwy_debug("xres %d, yres %d, zres %d", xres, yres, zres);
    if (err_DIMENSION(error, xres) || err_DIMENSION(error, yres) || err_DIMENSION(error, zres))
        return -1;

    gwy_debug("xreal %g, yreal %g, zreal %g", xreal, yreal, zreal);
    sanitise_real_size(&xreal, "x size");
    sanitise_real_size(&yreal, "y size");
    sanitise_real_size(&zreal, "z size");

    if (!get_str_attr_g(file_id, grouppfx, "Range Units", &xyunitstr, NULL))
        xyunitstr = g_strdup("µm");
    g_object_unref(gwy_si_unit_new_parse(xyunitstr, &power10));
    xreal *= pow10(power10);
    yreal *= pow10(power10);
    zreal *= pow10(power10);

    for (i = 0; i < volumes->len; i++) {
        id = g_array_index(volumes, gint, i);
        if (!(brick = shilps_read_brick(file_id, id, xres, yres, zres, xreal, yreal, zreal, xyunitstr, str, error))) {
            g_free(xyunitstr);
            return -1;
        }

        gwy_container_pass_object(container, gwy_app_get_brick_key_for_id(i), brick);

        /* shilps_read_brick() fills str->str with the correct prefix. */
        if (get_str_attr(file_id, str->str, "Channel", &channel, NULL)) {
            g_strstrip(channel);
            gwy_container_set_const_string(container, gwy_app_get_brick_title_key_for_id(i), channel);
            H5free_memory(channel);
        }

        meta = gwy_container_duplicate(ghfile->meta);
        gwy_container_pass_object(container, gwy_app_get_brick_meta_key_for_id(i), meta);
    }
    g_free(xyunitstr);

    return volumes->len;
}

static GwyBrick*
shilps_read_brick(hid_t file_id, gint id,
                  gint xres, gint yres, gint zres, gdouble xreal, gdouble yreal, gdouble zreal,
                  const gchar *xyunitstr,
                  GString *str, GError **error)
{
    GwyBrick *brick = NULL, *transposed;
    hid_t dataset;
    gint power10;
    /* The order seems a bit odd, but y is really the slowest varying index and z is the fast varying index within one
     * pixel. Unfortnately, we store volume data by plane so we have to transpose. */
    gint yxzres[3] = { yres, xres, zres };
    gchar *wunitstr = NULL;
    herr_t status = -1;

    g_string_printf(str, "Session/Volume/Image3d%d", id);
    if ((dataset = open_and_check_dataset(file_id, str->str, 3, yxzres, error)) < 0)
        return NULL;

    get_str_attr_g(file_id, str->str, "Units", &wunitstr, NULL);
    gwy_debug("Range units %s, data Units %s", xyunitstr, wunitstr);

    transposed = gwy_brick_new(zres, xres, yres, zreal, xreal, yreal, TRUE);
    status = H5Dread(dataset, H5T_NATIVE_DOUBLE, H5S_ALL, H5S_ALL, H5P_DEFAULT, gwy_brick_get_data(transposed));
    H5Dclose(dataset);

    if (status < 0) {
        err_HDF5(error, "H5Dread", status);
    }
    else {
        brick = gwy_brick_new(xres, yres, zres, xreal, yreal, zreal, FALSE);
        gwy_brick_transpose(transposed, brick, GWY_BRICK_TRANSPOSE_ZXY, FALSE, FALSE, FALSE);
        gwy_si_unit_set_from_string(gwy_brick_get_si_unit_x(brick), xyunitstr);
        gwy_si_unit_set_from_string(gwy_brick_get_si_unit_y(brick), xyunitstr);
        gwy_si_unit_set_from_string(gwy_brick_get_si_unit_z(brick), xyunitstr);
        gwy_si_unit_set_from_string_parse(gwy_brick_get_si_unit_w(brick), wunitstr, &power10);
        gwy_brick_multiply(brick, pow10(power10));
    }
    g_object_unref(transposed);
    g_free(wunitstr);

    return brick;
}

static gint
shilps_read_xyzs(GwyContainer *container,
                 hid_t file_id, GenericHDF5File *ghfile, GError **error)
{
    const gchar grouppfx[] = "Session/XYZ";
    GArray *xyzs = ghfile->lists[SHILPS_XYZ].idlist;
    GString *str = ghfile->buf;
    GwySurface *surface;
    gchar *channel, *xyunitstr;
    GwyContainer *meta;
    guint i, id;

    gwy_debug("nxyzs %u", xyzs->len);
    if (!xyzs->len)
        return 0;

    /* XXX: In other groups it is called Range units. */
    if (!get_str_attr_g(file_id, grouppfx, "Units", &xyunitstr, NULL))
        xyunitstr = g_strdup("µm");

    for (i = 0; i < xyzs->len; i++) {
        id = g_array_index(xyzs, gint, i);
        if (!(surface = shilps_read_surface(file_id, id, xyunitstr, str, error))) {
            g_free(xyunitstr);
            return -1;
        }

        gwy_container_pass_object(container, gwy_app_get_surface_key_for_id(i), surface);

        /* shilps_read_surface() fills str->str with the correct prefix. */
        if (get_str_attr(file_id, str->str, "Channel", &channel, NULL)) {
            g_strstrip(channel);
            gwy_container_set_const_string(container, gwy_app_get_surface_title_key_for_id(i), channel);
            H5free_memory(channel);
        }
        else
            gwy_app_xyz_title_fall_back(container, i);

        meta = gwy_container_duplicate(ghfile->meta);
        gwy_container_pass_object(container, gwy_app_get_surface_meta_key_for_id(i), meta);
    }
    g_free(xyunitstr);

    return xyzs->len;
}

static GwySurface*
shilps_read_surface(hid_t file_id, gint id,
                    const gchar *xyunitstr,
                    GString *str, GError **error)
{
    GwySurface *surface = NULL;
    hid_t dataset;
    GwyXYZ *data;
    gint i, npts, power10xy, power10z;
    gint n3[2] = { -1, 3 };
    gchar *zunitstr = NULL;
    gdouble qxy, qz;
    herr_t status = -1;

    g_string_printf(str, "Session/XYZ/Scatter%d", id);
    if ((dataset = open_and_check_dataset(file_id, str->str, 2, n3, error)) < 0)
        return NULL;
    gwy_debug("xyz data dims %d %d", n3[0], n3[1]);
    npts = n3[0];
    /* In principle, we could also create empty surfaces even though they are weird. Let's just hope valid files do
     * not contain empty XYZ data sets. */
    g_return_val_if_fail(npts > 0, NULL);

    get_str_attr_g(file_id, str->str, "Units", &zunitstr, NULL);
    gwy_debug("Range units %s, data Units %s", xyunitstr, zunitstr);

    surface = gwy_surface_new_sized(npts);
    gwy_si_unit_set_from_string_parse(gwy_surface_get_si_unit_xy(surface), xyunitstr, &power10xy);
    gwy_si_unit_set_from_string_parse(gwy_surface_get_si_unit_z(surface), zunitstr, &power10z);
    g_free(zunitstr);

    status = 0;
    data = gwy_surface_get_data(surface);
    /* NB: Here we are silently typecasting array of GwyXYZ to doubles. */
    status = H5Dread(dataset, H5T_NATIVE_DOUBLE, H5S_ALL, H5S_ALL, H5P_DEFAULT, data);
    H5Dclose(dataset);

    if (status < 0) {
        err_HDF5(error, "H5Dread", status);
        GWY_OBJECT_UNREF(surface);
    }
    else {
        qxy = pow10(power10xy);
        qz = pow10(power10z);
        for (i = 0; i < npts; i++) {
            data[i].x *= qxy;
            data[i].y *= qxy;
            data[i].z *= qz;
        }
    }

    return surface;
}

static void
shilps_set_spectra_coordinates(GwyContainer *container, gint nspec, gint nxyz)
{
    GwySpectra *spec;
    GwySurface *surface;
    const GwyXYZ *xyz;
    gint i, j, n, k;

    for (i = 0; i < nspec; i++) {
        spec = gwy_container_get_object(container, gwy_app_get_spectra_key_for_id(i));
        n = gwy_spectra_get_n_spectra(spec);
        for (j = 0; j < nxyz; j++) {
            surface = gwy_container_get_object(container, gwy_app_get_surface_key_for_id(j));
            if (gwy_surface_get_npoints(surface) == n) {
                gwy_debug("XYZ data #%d has the right number of points for spectra #%d", j, i);
                gwy_si_unit_assign(gwy_spectra_get_si_unit_xy(spec), gwy_surface_get_si_unit_xy(surface));
                xyz = gwy_surface_get_data_const(surface);
                for (k = 0; k < n; k++)
                    gwy_spectra_setpos(spec, k, xyz[k].x, xyz[k].y);
                break;
            }
        }
        if (j == nxyz) {
            gwy_debug("cannot find good XYZ data for spectra #%d; removing it", i);
            gwy_container_remove(container, gwy_app_get_spectra_key_for_id(i));
        }
    }
}

/*******************************************************************************************************************
 *
 * More or less general HDF5 utility functions
 *
 *******************************************************************************************************************/

static hid_t
quick_check_hdf5(const GwyFileDetectInfo *fileinfo,
                 gboolean only_name)
{
    if (only_name)
        return -1;

    if (fileinfo->buffer_len <= MAGIC_SIZE || memcmp(fileinfo->head, MAGIC, MAGIC_SIZE) != 0)
        return -1;

    return H5Fopen(fileinfo->name, H5F_ACC_RDONLY, H5P_DEFAULT);
}

static void
generic_hdf5_init(GenericHDF5File *ghfile)
{
    gwy_clear(ghfile, 1);
    ghfile->meta = gwy_container_new();
    ghfile->path = g_string_new(NULL);
    ghfile->buf = g_string_new(NULL);
    ghfile->addr = g_array_new(FALSE, FALSE, sizeof(haddr_t));
    ghfile->datasets = g_array_new(FALSE, FALSE, sizeof(gchar*));
}

static void
generic_hdf5_alloc_lists(GenericHDF5File *ghfile, guint n)
{
    guint i;

    if (!n)
        return;

    ghfile->nlists = n;
    ghfile->lists = g_new(GatheredIds, n);
    for (i = 0; i < n; i++)
        ghfile->lists[i].idlist = g_array_new(FALSE, FALSE, sizeof(gint));
}

static void
generic_hdf5_free(GenericHDF5File *ghfile)
{
    guint i;

    for (i = 0; i < ghfile->datasets->len; i++)
        g_free(g_array_index(ghfile->datasets, gchar*, i));
    g_array_free(ghfile->datasets, TRUE);
    g_array_free(ghfile->addr, TRUE);
    if (ghfile->nlists) {
        for (i = 0; i < ghfile->nlists; i++)
            g_array_free(ghfile->lists[i].idlist, TRUE);
        g_free(ghfile->lists);
    }
    g_string_free(ghfile->path, TRUE);
    g_string_free(ghfile->buf, TRUE);
    GWY_OBJECT_UNREF(ghfile->meta);
}

/* NB: loc_id is ‘parent’ location and name is particlar item within it. */
static herr_t
scan_file(hid_t loc_id,
          const char *name,
          G_GNUC_UNUSED const H5L_info_t *info,
          void *user_data)
{
    GenericHDF5File *ghfile = (GenericHDF5File*)user_data;
    herr_t status, return_val = 0;
    H5O_info_t infobuf;
    GArray *addr = ghfile->addr;
    GString *path = ghfile->path;
    guint i, len = path->len;
    gchar *s;

    status = H5Oget_info_by_name(loc_id, name, &infobuf, H5P_DEFAULT);
    if (status < 0)
        return status;

    /* Detect loops. */
    for (i = 0; i < addr->len; i++) {
        if (g_array_index(addr, haddr_t, i) == infobuf.addr)
            return -1;
    }

    g_array_append_val(addr, infobuf.addr);
    g_string_append_c(path, '/');
    g_string_append(path, name);
    gwy_debug("path %s", path->str);
    if (infobuf.type == H5O_TYPE_GROUP) {
        return_val = H5Literate_by_name(loc_id, name, H5_INDEX_NAME, H5_ITER_NATIVE,
                                        NULL, scan_file, user_data, H5P_DEFAULT);
    }
    else if (infobuf.type == H5O_TYPE_DATASET) {
        s = g_strdup(path->str);
        g_array_append_val(ghfile->datasets, s);
    }
    /* Nothing to do for other object types. */
    else if (infobuf.type == H5O_TYPE_NAMED_DATATYPE) {
    }
    else {
        gwy_debug("unknown type %d", infobuf.type);
    }

    for (i = 0; i < ghfile->nlists; i++) {
        GatheredIds *gathered = ghfile->lists + i;
        if (infobuf.type == gathered->idwhat)
            enumerate_indexed(path, gathered->idprefix, gathered->idlist);
    }

    if (infobuf.num_attrs > 0) {
        hid_t this_id = H5Oopen(loc_id, name, H5P_DEFAULT);

        H5Aiterate2(this_id, H5_INDEX_NAME, H5_ITER_NATIVE, NULL, process_attribute, user_data);
        H5Oclose(this_id);
    }

    g_string_truncate(path, len);
    g_array_set_size(addr, addr->len-1);

    return return_val;
}

static gint
process_float_attribute(hid_t attr, hid_t attr_type, gint nitems,
                        GString *outbuf)
{
    herr_t status;

    if (nitems <= 0)
        return -1;

    if (nitems == 1) {
        gdouble v;

        gwy_debug("float");
        if ((status = H5Aread(attr, attr_type, &v)) >= 0)
            g_string_printf(outbuf, "%.8g", v);
    }
    else {
        gdouble *v = g_new(gdouble, nitems);
        gint i;

        gwy_debug("float array");
        if ((status = H5Aread(attr, attr_type, v)) >= 0) {
            g_string_printf(outbuf, "%.8g", v[0]);
            for (i = 1; i < nitems; i++)
                g_string_append_printf(outbuf, "; %.8g", v[i]);
        }
        g_free(v);
    }

    return status;
}

static gint
process_integer_attribute(hid_t attr, hid_t attr_type, gint nitems,
                          GString *outbuf)
{
    herr_t status;

    if (nitems <= 0)
        return -1;

    if (nitems == 1) {
        gint v;

        gwy_debug("integer");
        if ((status = H5Aread(attr, attr_type, &v)) >= 0)
            g_string_printf(outbuf, "%d", v);
    }
    else {
        gint *v = g_new(gint, nitems);
        gint i;

        gwy_debug("integer array");
        if ((status = H5Aread(attr, attr_type, v)) >= 0) {
            g_string_printf(outbuf, "%d", v[0]);
            for (i = 1; i < nitems; i++)
                g_string_append_printf(outbuf, "; %d", v[i]);
        }
        g_free(v);
    }

    return status;
}

/* Fixed strings are read differently, into preallocated buffers, but fortunately no one uses them. */
static gint
process_var_string_attribute(hid_t attr, hid_t attr_type, hid_t read_as_type, gint nitems,
                             GString *outbuf)
{
    herr_t status;
    gboolean close_str_type = FALSE;

    if (nitems <= 0)
        return -1;

    if (read_as_type < 0) {
        read_as_type = make_string_type_for_attr(attr_type);
        close_str_type = TRUE;
    }
    if (nitems == 1) {
        gchar *s = NULL;

        gwy_debug("string");
        if ((status = H5Aread(attr, read_as_type, &s)) >= 0) {
            g_strstrip(s);
            g_string_printf(outbuf, "%s", s);
        }
        H5free_memory(s);
    }
    else if (nitems > 0) {
        gchar **s = g_new0(gchar*, nitems);
        gint i;

        gwy_debug("string array");
        if ((status = H5Aread(attr, read_as_type, s)) >= 0) {
            g_strstrip(s[0]);
            g_string_assign(outbuf, s[0]);
            H5free_memory(s[0]);
            for (i = 1; i < nitems; i++) {
                g_strstrip(s[i]);
                g_string_append(outbuf, "; ");
                g_string_append(outbuf, s[i]);
                H5free_memory(s[i]);
            }
        }
        g_free(s);
    }
    if (close_str_type)
        H5Tclose(read_as_type);

    return status;
}

static gint
process_fixed_string_attribute(hid_t attr, hid_t attr_type, gint nitems,
                               GString *outbuf)
{
    hid_t str_type;
    herr_t status;
    gchar *s;

    if (nitems < 0)
        return -1;

    gwy_debug("fixed-size string");
    s = g_new(gchar, nitems+1);
    s[nitems] = '\0';
    str_type = make_string_type_for_attr(attr_type);
    if ((status = H5Aread(attr, str_type, s)) >= 0) {
        g_strstrip(s);
        g_string_printf(outbuf, "%s", s);
    }
    g_free(s);
    H5Tclose(str_type);

    return status;
}

static herr_t
process_attribute(hid_t loc_id,
                  const char *attr_name,
                  G_GNUC_UNUSED const H5A_info_t *ainfo,
                  void *user_data)
{
    GenericHDF5File *ghfile = (GenericHDF5File*)user_data;
    GString *path = ghfile->path, *buf = ghfile->buf;
    G_GNUC_UNUSED H5T_cset_t cset = H5T_CSET_ERROR;
    guint len = path->len, attr_path_len;
    hid_t attr, attr_type, member_type, space;
    gboolean is_vlenstr = FALSE, compound_was_handled = FALSE;
    H5T_class_t type_class, member_class;
    gchar *key, *member_name;
    gint nitems, j, nmembers;
    herr_t status;

    attr = H5Aopen(loc_id, attr_name, H5P_DEFAULT);
    attr_type = H5Aget_type(attr);
    space = H5Aget_space(attr);
    nitems = H5Sget_simple_extent_npoints(space);
    type_class = H5Tget_class(attr_type);
    if (type_class == H5T_STRING) {
        is_vlenstr = H5Tis_variable_str(attr_type);
        cset = H5Tget_cset(attr_type);
    }

    gwy_debug("attr %s, type class %d (is_vlenstr: %d, cset: %d)", attr_name, type_class, is_vlenstr, cset);
    g_string_append_c(path, '/');
    g_string_append(path, attr_name);
    status = -1;
    /* Try to read all attribute types used by Ergo; there are just a few. */
    if (type_class == H5T_INTEGER) {
        status = process_integer_attribute(attr, H5T_NATIVE_INT, nitems, buf);
    }
    else if (type_class == H5T_FLOAT) {
        status = process_float_attribute(attr, H5T_NATIVE_DOUBLE, nitems, buf);
    }
    else if (type_class == H5T_STRING && is_vlenstr) {
        status = process_var_string_attribute(attr, attr_type, -1, nitems, buf);
    }
    else if (type_class == H5T_STRING && nitems == 1) {
        status = process_fixed_string_attribute(attr, attr_type, nitems, buf);
    }
    else if (type_class == H5T_COMPOUND) {
        /* Here we basically need to do it all over again with individual fields because we do not read compounds
         * as structs. */
        attr_path_len = path->len;
        g_string_append_c(path, '/');
        nmembers = H5Tget_nmembers(attr_type);
        gwy_debug("compound, nmembers %d, nitems %d", nmembers, nitems);
        for (j = 0; j < nmembers; j++) {
            member_name = H5Tget_member_name(attr_type, j);
            member_class = H5Tget_member_class(attr_type, j);
            gwy_debug("compound[%d] \"%s\" class %d", j, member_name, member_class);
            g_string_append(path, member_name);

            compound_was_handled = TRUE;
            if (member_class == H5T_INTEGER) {
                member_type = H5Tcreate(H5T_COMPOUND, sizeof(gint));
                H5Tinsert(member_type, member_name, 0, H5T_NATIVE_INT);
                status = process_integer_attribute(attr, member_type, nitems, buf);
            }
            else if (member_class == H5T_FLOAT) {
                member_type = H5Tcreate(H5T_COMPOUND, sizeof(gdouble));
                H5Tinsert(member_type, member_name, 0, H5T_NATIVE_DOUBLE);
                status = process_float_attribute(attr, member_type, nitems, buf);
            }
            else if (member_class == H5T_STRING) {
                hid_t str_type, file_member_type = H5Tget_member_type(attr_type, j);

                if (H5Tis_variable_str(file_member_type)) {
                    member_type = H5Tcreate(H5T_COMPOUND, sizeof(gdouble));
                    str_type = make_string_type_for_attr(file_member_type);
                    H5Tinsert(member_type, member_name, 0, str_type);
                    status = process_var_string_attribute(attr, file_member_type, member_type, nitems, buf);
                    H5Tclose(str_type);
                }
                else {
                    /* I do not see how this variant can be logically represented. Where would we read the fixed
                     * string length? */
                    compound_was_handled = FALSE;
                }
            }
            else {
                compound_was_handled = FALSE;
            }

            if (compound_was_handled) {
                gwy_debug("compound reading status %d (value %s)", status, buf->str);
                H5Tclose(member_type);
                if (status < 0)
                    compound_was_handled = FALSE;
            }

            if (compound_was_handled) {
                gwy_debug("[%s] = <%s>", path->str, buf->str);
                key = gwy_strreplace(path->str+1, "/", "::", (gsize)-1);
                gwy_container_set_const_string_by_name(ghfile->meta, key, buf->str);
                g_free(key);
            }

            H5free_memory(member_name);
            g_string_truncate(path, attr_path_len+1);
        }
    }
    H5Sclose(space);
    H5Tclose(attr_type);
    H5Aclose(attr);

    if (compound_was_handled) {
        /* Do nothing because everything is done. */
    }
    else if (status >= 0) {
        gwy_debug("[%s] = <%s>", path->str, buf->str);
        key = gwy_strreplace(path->str+1, "/", "::", (gsize)-1);
        gwy_container_set_const_string_by_name(ghfile->meta, key, buf->str);
        g_free(key);
    }
    else {
        g_warning("Cannot handle attribute %d(%d)[%d]", type_class, is_vlenstr, nitems);
    }

    if (ghfile->attr_handler) {
        ghfile->attr_handler(ghfile, loc_id, attr_name);
    }

    g_string_truncate(path, len);

    return 0;
}

static hid_t
make_string_type_for_attr(hid_t attr_type)
{
    hid_t str_type;

    str_type = H5Tcopy(H5T_C_S1);
    if (H5Tis_variable_str(attr_type)) {
        H5Tset_size(str_type, H5T_VARIABLE);
        H5Tset_strpad(str_type, H5T_STR_NULLTERM);
    }
    /* This is kind of stupid because the character set is ‘whatever’. Unforutnately, when the attribute character set
     * is ASCII and we ask for UTF-8 it goes poorly. Even though the conversion is no-op HDF5 thinks it cannot do it
     * and fails. So we just count on ASCII and UTF-8 being the only character sets available – and both being fine
     * interpreted as UTF-8. */
    H5Tset_cset(str_type, H5Tget_cset(attr_type));

    return str_type;
}

/* expected_dims[] is non-const because we can either check or query the dimensions. Pass -1 as an expected_dims[]
 * item to query it. Pass NULL expected_dims if you do not care. */
static hid_t
open_and_check_attr(hid_t file_id,
                    const gchar *obj_path,
                    const gchar *attr_name,
                    H5T_class_t expected_class,
                    gint expected_rank,
                    gint *expected_dims,
                    GError **error)
{
    hid_t attr, attr_type, space;
    H5T_class_t type_class;
    gint i, rank, status;
    hsize_t dims[3];

    gwy_debug("looking for %s in %s, class %d, rank %d",
              attr_name, obj_path, expected_class, expected_rank);
    if ((attr = H5Aopen_by_name(file_id, obj_path, attr_name, H5P_DEFAULT, H5P_DEFAULT)) < 0) {
        err_MISSING_FIELD(error, attr_name);
        return -1;
    }

    attr_type = H5Aget_type(attr);
    type_class = H5Tget_class(attr_type);
    gwy_debug("found attr %d of type %d and class %d", (gint)attr, (gint)attr_type, type_class);
    if (type_class != expected_class) {
        H5Tclose(attr_type);
        H5Aclose(attr);
        err_UNSUPPORTED(error, attr_name);
        return -1;
    }

    if ((space = H5Aget_space(attr)) < 0) {
        err_HDF5(error, "H5Aget_space", space);
        H5Tclose(attr_type);
        H5Aclose(attr);
    }
    rank = H5Sget_simple_extent_ndims(space);
    gwy_debug("attr space is %d with rank %d", (gint)space, rank);
    if (rank > 3 || rank != expected_rank) {
        err_UNSUPPORTED(error, attr_name);
        goto fail;
    }

    if ((status = H5Sget_simple_extent_dims(space, dims, NULL)) < 0) {
        gwy_debug("cannot get space %d extent dims", (gint)space);
        err_HDF5(error, "H5Sget_simple_extent_dims", status);
        goto fail;
    }
    if (expected_dims) {
        for (i = 0; i < rank; i++) {
            gwy_debug("dims[%d]=%lu, expecting %d", i, (gulong)dims[i], expected_dims[i]);
            if (expected_dims[i] < 0)
                expected_dims[i] = dims[i];
            else if (dims[i] != (hsize_t)expected_dims[i]) {
                err_UNSUPPORTED(error, attr_name);
                goto fail;
            }
        }
    }

    H5Sclose(space);
    H5Tclose(attr_type);
    gwy_debug("attr %d seems OK", (gint)attr);
    return attr;

fail:
    H5Sclose(space);
    H5Tclose(attr_type);
    H5Aclose(attr);
    return -1;
}

static gboolean
get_ints_attr(hid_t file_id,
              const gchar *obj_path,
              const gchar *attr_name,
              gint expected_rank,
              gint *expected_dims,
              gint *v,
              GError **error)
{
    hid_t attr;
    gint status;

    if ((attr = open_and_check_attr(file_id, obj_path, attr_name, H5T_INTEGER,
                                    expected_rank, expected_dims, error)) < 0)
        return FALSE;

    status = H5Aread(attr, H5T_NATIVE_INT, v);
    H5Aclose(attr);
    if (status < 0) {
        err_HDF5(error, "H5Aread", status);
        return FALSE;
    }
    return TRUE;
}

static gboolean
get_int_attr(hid_t file_id,
             const gchar *obj_path, const gchar *attr_name,
             gint *v, GError **error)
{
    return get_ints_attr(file_id, obj_path, attr_name, 0, NULL, v, error);
}

static gboolean
get_floats_attr(hid_t file_id,
                const gchar *obj_path,
                const gchar *attr_name,
                gint expected_rank,
                gint *expected_dims,
                gdouble *v,
                GError **error)
{
    hid_t attr;
    gint status;

    if ((attr = open_and_check_attr(file_id, obj_path, attr_name, H5T_FLOAT, expected_rank, expected_dims, error)) < 0)
        return FALSE;

    status = H5Aread(attr, H5T_NATIVE_DOUBLE, v);
    H5Aclose(attr);
    if (status < 0) {
        err_HDF5(error, "H5Aread", status);
        return FALSE;
    }
    return TRUE;
}

G_GNUC_UNUSED
static gboolean
get_float_attr(hid_t file_id,
               const gchar *obj_path, const gchar *attr_name,
               gdouble *v, GError **error)
{
    return get_floats_attr(file_id, obj_path, attr_name, 0, NULL, v, error);
}

static gboolean
get_strs_attr(hid_t file_id,
              const gchar *obj_path,
              const gchar *attr_name,
              gint expected_rank,
              gint *expected_dims,
              gchar **v,
              GError **error)
{
    hid_t attr, attr_type, str_type;
    gboolean is_vlenstr;
    gint status;

    if ((attr = open_and_check_attr(file_id, obj_path, attr_name, H5T_STRING, expected_rank, expected_dims, error)) < 0)
        return FALSE;

    attr_type = H5Aget_type(attr);
    if (attr_type < 0) {
        H5Aclose(attr);
        err_HDF5(error, "H5Aget_type", attr_type);
        return FALSE;
    }
    is_vlenstr = H5Tis_variable_str(attr_type);
    gwy_debug("attr %d is%s vlen string", (gint)attr, is_vlenstr ? "" : " not");
    if (!is_vlenstr) {
        H5Tclose(attr_type);
        H5Aclose(attr);
        /* XXX: Be more specific. */
        err_UNSUPPORTED(error, attr_name);
        return FALSE;
    }

    str_type = make_string_type_for_attr(attr_type);
    status = H5Aread(attr, str_type, v);
    H5Tclose(attr_type);
    H5Tclose(str_type);
    H5Aclose(attr);
    if (status < 0) {
        err_HDF5(error, "H5Aread", status);
        return FALSE;
    }
    return TRUE;
}

/* Get string attribute, but as a glib-allocated string so we do not have to track its origin and free it using
 * g_free() as any other string. */
static gboolean
get_str_attr_g(hid_t file_id,
               const gchar *obj_path, const gchar *attr_name,
               gchar **v, GError **error)
{
    gchar *t;

    if (!get_strs_attr(file_id, obj_path, attr_name, 0, NULL, &t, error))
        return FALSE;

    *v = g_strdup(t);
    H5free_memory(t);
    return TRUE;
}

static gboolean
get_str_attr(hid_t file_id,
             const gchar *obj_path, const gchar *attr_name,
             gchar **v, GError **error)
{
    return get_strs_attr(file_id, obj_path, attr_name, 0, NULL, v, error);
}

static gboolean
get_simple_space_dims(hid_t space, gint expected_ndims, gint *dims,
                      const gchar *name, GError **error)
{
    gint i, ndims;
    hsize_t hdims[4];

    g_return_val_if_fail(expected_ndims <= 4, FALSE);

    ndims = H5Sget_simple_extent_ndims(space);
    if (ndims != expected_ndims) {
        g_set_error(error, GWY_MODULE_FILE_ERROR, GWY_MODULE_FILE_ERROR_DATA,
                    _("Dataset %s has %d dimensions instead of expected %d."), name, ndims, expected_ndims);
        return FALSE;
    }

    H5Sget_simple_extent_dims(space, hdims, NULL);
    for (i = 0; i < ndims; i++) {
        if (dims[i] >= 0) {
            if (dims[i] != hdims[i]) {
                g_set_error(error, GWY_MODULE_FILE_ERROR, GWY_MODULE_FILE_ERROR_DATA,
                            _("Dataset %s dimension #%d is %ld, which does not match image resolution %d."),
                            name, i, (glong)hdims[i], dims[i]);
                return FALSE;
            }
        }
        else {
            gwy_debug("dims[%d] %ld (no expectation)", i, (glong)hdims[i]);
            dims[i] = hdims[i];
        }
    }
    return TRUE;
}

static hid_t
open_and_check_dataset(hid_t file_id, const gchar *name,
                       gint expected_ndims, gint *dims,
                       GError **error)
{
    hid_t dataset, space;

    if ((dataset = H5Dopen(file_id, name, H5P_DEFAULT)) < 0) {
        err_HDF5(error, "H5Dopen", dataset);
        return -1;
    }
    gwy_debug("dataset %s is %ld", name, (glong)dataset);

    if ((space = H5Dget_space(dataset)) < 0) {
        err_HDF5(error, "H5Dget_space", space);
        H5Dclose(dataset);
        return -1;
    }
    if (!get_simple_space_dims(space, expected_ndims, dims, name, error)) {
        H5Sclose(space);
        H5Dclose(dataset);
        return -1;
    }
    H5Sclose(space);

    return dataset;
}

/* Gather all the numbers in paths of the form prefix#NUM. */
static gboolean
enumerate_indexed(GString *path, const gchar *prefix, GArray *array)
{
    guint i, len = strlen(prefix);
    const gchar *p;

    if (strncmp(path->str, prefix, len))
        return FALSE;

    p = path->str + len;
    for (i = 0; g_ascii_isdigit(p[i]); i++)
        ;
    if (!i || p[i])
        return FALSE;

    i = atol(p);
    if (i > G_MAXINT)
        return FALSE;

    g_array_append_val(array, i);
    gwy_debug("found indexed %s[%u]", prefix, i);
    return TRUE;
}

static void
err_HDF5(GError **error, const gchar *where, glong code)
{
    g_set_error(error, GWY_MODULE_FILE_ERROR, GWY_MODULE_FILE_ERROR_SPECIFIC,
                _("HDF5 library error %ld in function %s."), code, where);
}

/* vim: set cin columns=120 tw=118 et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
