package events

import (
	"testing"

	"launchpad.net/email-reminder/internal/util"
)

func TestRecipientToEmailRecipient(t *testing.T) {
	tests := []struct {
		name      string
		recipient Recipient
		want      util.EmailRecipient
	}{
		{
			name: "recipient with first and last name",
			recipient: Recipient{
				Name:  "John Doe",
				Email: "john@example.com",
			},
			want: util.EmailRecipient{
				Email:     "john@example.com",
				FirstName: "John",
				LastName:  "Doe",
			},
		},
		{
			name: "recipient with three-part name",
			recipient: Recipient{
				Name:  "John Doe Smith",
				Email: "john@example.com",
			},
			want: util.EmailRecipient{
				Email:     "john@example.com",
				FirstName: "John",
				LastName:  "Doe Smith",
			},
		},
		{
			name: "recipient with single name",
			recipient: Recipient{
				Name:  "John",
				Email: "john@example.com",
			},
			want: util.EmailRecipient{
				Email:     "john@example.com",
				FirstName: "John",
				LastName:  "",
			},
		},
		{
			name: "recipient with email only",
			recipient: Recipient{
				Name:  "",
				Email: "john@example.com",
			},
			want: util.EmailRecipient{
				Email:     "john@example.com",
				FirstName: "",
				LastName:  "",
			},
		},
		{
			name: "recipient with invalid email",
			recipient: Recipient{
				Name:  "John Doe",
				Email: "john@",
			},
			want: util.EmailRecipient{
				Email:     "",
				FirstName: "John",
				LastName:  "Doe",
			},
		},
		{
			name: "recipient with whitespace in name",
			recipient: Recipient{
				Name:  "  John   Doe  ",
				Email: "  john@example.com  ",
			},
			want: util.EmailRecipient{
				Email:     "john@example.com",
				FirstName: "John",
				LastName:  "Doe",
			},
		},
		{
			name: "recipient with whitespace-only name",
			recipient: Recipient{
				Name:  "   ",
				Email: "john@example.com",
			},
			want: util.EmailRecipient{
				Email:     "john@example.com",
				FirstName: "",
				LastName:  "",
			},
		},
		{
			name: "recipient with multiple spaces between names",
			recipient: Recipient{
				Name:  "John    Doe    Smith",
				Email: "john@example.com",
			},
			want: util.EmailRecipient{
				Email:     "john@example.com",
				FirstName: "John",
				LastName:  "Doe Smith",
			},
		},
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			got := tt.recipient.ToEmailRecipient()
			if got.Email != tt.want.Email {
				t.Errorf("ToEmailRecipient().Email = %q, want %q", got.Email, tt.want.Email)
			}
			if got.FirstName != tt.want.FirstName {
				t.Errorf("ToEmailRecipient().FirstName = %q, want %q", got.FirstName, tt.want.FirstName)
			}
			if got.LastName != tt.want.LastName {
				t.Errorf("ToEmailRecipient().LastName = %q, want %q", got.LastName, tt.want.LastName)
			}
		})
	}
}
