/*
  This file provides an interface to the /dev/dsp and /dev/Mixer devices of 
  the Open Sound System

  Author: Bob Dean
  Copyright (c) 1999

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public Licensse as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
 
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

*/

#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <glib.h>

#include "dbaudio.h"
#include <dbdebug.h>
#include <dbsoundcard.h>

#ifndef SUCCESS /* this is a hack for now...*/
#define SUCCESS  0
#define FAILURE -1
#endif

oss_control_struct * oss_ctrl = NULL, *cue_ctrl = NULL;

oss_control_struct * init_audio(char * dsp_path, int flags, int output_stdout, 
								int numbuffs)
{
	oss_control_struct * local_oss_ctrl;
	char * dsp_device;
	int    frag;

	local_oss_ctrl = (oss_control_struct *) malloc(sizeof(oss_control_struct)); 

	local_oss_ctrl->sample_format = AFMT_S16_NE;
	local_oss_ctrl->channels = DBSTEREO;
	local_oss_ctrl->sample_rate = 44100; 

	if (numbuffs <= 0) 
	{
		numbuffs = DB_NUM_FRAGMENTS;
		Debug("init_audio: invalid value for numbuffs, resetting to be DB_NUM_FRAGMENTS");
	}

	if (output_stdout)
	{
		local_oss_ctrl->fd_dsp = 1;
		return local_oss_ctrl;
	}
	
	/* allow user to specify which audio device output is sent to */
	if (dsp_path == NULL)
	{
		Debug("init_audio: dsp_path is NULL.");
		dsp_device = "/dev/dsp";
	}
	else
	{
		dsp_device = dsp_path;
	}

	/*try to open sound device*/
	if ((local_oss_ctrl->fd_dsp = open(dsp_device,flags,0)) == -1)
	{
		char str[256];
		sprintf(str,"init_audio: cannot open device \"%s\"",dsp_device);
		perror(str);
		return NULL;
	}
	
	/* set fragment number and size */
	if (numbuffs > 0)
	{
		frag = (numbuffs << 16) | DB_FRAGSIZE;
		
		if (ioctl(local_oss_ctrl->fd_dsp,SNDCTL_DSP_SETFRAGMENT,&frag) == -1)
		{
			perror("init_audio: could not set fragment size.");
			close_audio(local_oss_ctrl);
			return NULL;
		}
		else
		{
			Debug("init_audio: num fragments %d  fragement size: %d",DB_NUM_FRAGMENTS,2^DB_FRAGSIZE);
		}
	}

	/*set audio format*/
	if (ioctl(local_oss_ctrl->fd_dsp,SNDCTL_DSP_SETFMT,&(local_oss_ctrl->sample_format)) == -1)
	{
		perror("init_audio: failure setting format.");
		/*perror("SNDCTL_DSP_SETFMT");*/
		close_audio(local_oss_ctrl);
		return NULL;
	}

	/*set full duplex if allowed, remember if not*/	

	/*  if (ioctl(local_oss_ctrl->fd_dsp, SNDCTL_DSP_SETDUPLEX, 0) == -1)
		{
		perror("init_audio: failure setting full duplex mode:");
		local_oss_ctrl->fullduplex = FALSE;
		}
		else
		{ local_oss_ctrl->fullduplex = TRUE; }
	*/


	/*set channels to be stero*/
	local_oss_ctrl->channels = DBSTEREO;


	/* 	local_oss_ctrl->channels = DBMONO; */


	if (ioctl(local_oss_ctrl->fd_dsp,SNDCTL_DSP_STEREO,&(local_oss_ctrl->channels)) == -1)
	{
		perror("init_audio: Error setting stereo");
		/*perror("SNDCTL_DSP_STEREO");*/
		close_audio(local_oss_ctrl);
		return NULL;
	}


	/*set sample rate*/
	if (ioctl(local_oss_ctrl->fd_dsp,SNDCTL_DSP_SPEED,&(local_oss_ctrl->sample_rate)) == -1)
	{
		perror("init_audio: Could not set sample rate.");
		/*perror("SNDCTL_DSP_SPEED");*/
		close_audio(local_oss_ctrl);
		return NULL;
	}
  
	if (local_oss_ctrl->sample_rate < 44000)
	{
		printf("ERROR: soundcard does not support minimum sampling rate of 44.1 kHz\n");
		printf("       sampling rate returned by driver is: %d\n",local_oss_ctrl->sample_rate);
		exit(-1);
	}

	{
		int frag_size;
		audio_buf_info info;

		Debug("****************");
		Debug("device: %s",dsp_device);
		Debug("sample rate: %d",local_oss_ctrl->sample_rate);
		Debug("getting fragment data...");
		if (ioctl(local_oss_ctrl->fd_dsp, SNDCTL_DSP_GETBLKSIZE, &frag_size) == -1) perror("getting frag size");
		
		Debug("Frag size:  %d",frag_size);

		ioctl(local_oss_ctrl->fd_dsp, SNDCTL_DSP_GETOSPACE, &info);

		Debug("fragments:  %d",info.fragments);
		Debug("fragstotal: %d",info.fragstotal);
		Debug("fragsize:   %d",info.fragsize);
		Debug("bytes:      %d",info.bytes);
		Debug("****************");
	}


	return local_oss_ctrl;
}


oss_control_struct * get_audio_info()
{
	return oss_ctrl;
}


void close_audio(oss_control_struct * in_ctrl)
{
	if (in_ctrl == NULL) return;

	close(in_ctrl->fd_dsp);
	free(in_ctrl);
	return;
}


void pause_audio(oss_control_struct * in_ctrl)
{
	if (in_ctrl == NULL) return;

	if (in_ctrl->fd_dsp == 1) return;

    ioctl(in_ctrl->fd_dsp, SNDCTL_DSP_POST, 0);
    return;
}


int write_audio(oss_control_struct * in_ctrl, void* buf, int size)
{
	if (in_ctrl == NULL) return 0;

	return write(in_ctrl->fd_dsp,buf,size);
}


int read_audio(oss_control_struct * in_ctrl, void* buf, int size)
{
	if (in_ctrl == NULL) return 0;

	return read(in_ctrl->fd_dsp,buf,size);
}
