//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      Sample/HardParticle/IFormFactorPolyhedron.cpp
//! @brief     Implements interface IFormFactorPolyhedron.
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

//! The mathematics implemented here is described in full detail in a paper
//! by Joachim Wuttke, entitled
//! "Form factor (Fourier shape transform) of polygon and polyhedron."

#include "Sample/HardParticle/IFormFactorPolyhedron.h"
#include "Base/Types/Span.h"
#include "Base/Util/Assert.h"
#include "Sample/Particle/PolyhedralUtil.h"
#include <ff/Polyhedron.h>

IFormFactorPolyhedron::IFormFactorPolyhedron(const std::vector<double>& PValues)
    : IFormFactor(PValues)
{
}

IFormFactorPolyhedron::~IFormFactorPolyhedron() = default;

//! Called by child classes to set faces and other internal variables.

void IFormFactorPolyhedron::setPolyhedron(const ff::Topology& topology, double z_bottom,
                                          const std::vector<R3>& vertices) const
{
    pimpl = std::make_unique<ff::Polyhedron>(topology, vertices, R3(0, 0, -z_bottom));
}

double IFormFactorPolyhedron::volume() const
{
    ASSERT(m_validated);
    return pimpl->volume();
}

double IFormFactorPolyhedron::radialExtension() const
{
    ASSERT(m_validated);
    return pimpl->radius();
}

Span IFormFactorPolyhedron::spanZ(const IRotation* rotation) const
{
    ASSERT(m_validated);
    return PolyhedralUtil::spanZ(pimpl->vertices(), rotation);
}

complex_t IFormFactorPolyhedron::formfactor(C3 q) const
{
    ASSERT(m_validated);
    return pimpl->formfactor(q);
}
