//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/View/Project/ProjectManager.h
//! @brief     Defines class ProjectManager
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#ifndef BORNAGAIN_GUI_VIEW_PROJECT_PROJECTMANAGER_H
#define BORNAGAIN_GUI_VIEW_PROJECT_PROJECTMANAGER_H

#include "GUI/Model/Project/ProjectDocument.h"

class AutosaveController;

//! Handles activity related to opening/save projects.

class ProjectManager : public QObject {
    Q_OBJECT
public:
    ProjectManager(QObject* parent);
    ~ProjectManager() override;

    static ProjectManager* instance();

    void readSettings();
    void writeSettings();

    QStringList recentProjects();
    QString projectDir() const;
    QString userImportDir() const;
    QString recentlyUsedImportFilter1D() const;
    QString recentlyUsedImportFilter2D() const;
    void setImportDir(const QString& dirname);
    void setImportDirFromFilePath(const QString& filePath);
    void setRecentlyUsedImportFilter1D(const QString& filter);
    void setRecentlyUsedImportFilter2D(const QString& filter);

    bool isAutosaveEnabled() const;
    AutosaveController* autosaveController() const;

signals:
    void aboutToCloseDocument();
    void documentOpenedOrClosed(bool opened);
    void documentModified();
    void recentListModified();

public slots:
    void setAutosaveEnabled(bool value);
    void clearRecentProjects();
    ProjectDocument* newProject();
    bool closeCurrentProject();
    bool saveProject(QString projectPullPath = "");
    bool saveProjectAs();
    void openProject(QString projectPullPath = "");

private:
    void createNewProject();
    void deleteCurrentProject();
    ProjectDocument::ReadResult loadProject(const QString& fullPathAndName,
                                            MessageService& messageService);
    QString acquireProjectPullPath();
    void addToRecentProjects();

    QString workingDirectory();
    QString untitledProjectName();

    void riseProjectLoadFailedDialog(const MessageService& messageService);
    void riseProjectLoadProblemDialog(const MessageService& messageService);
    bool restoreProjectDialog(const QString& projectFileName, QString autosaveName);

    //!< Name of directory where project directory was created.
    QString m_workingDirectory;

    //!< Name of directory from there user prefer to import files
    QString m_importDirectory;

    //! Recently used import filter for 1D files
    QString m_importFilter1D;

    //! Recently used import filter for 2D files
    QString m_importFilter2D;

    QStringList m_recentProjects;
    std::unique_ptr<AutosaveController> m_autosave;

    static ProjectManager* s_instance;
};

#endif // BORNAGAIN_GUI_VIEW_PROJECT_PROJECTMANAGER_H
