//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/View/PlotUtil/ColorMap.h
//! @brief     Defines class ColorMap
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#ifndef BORNAGAIN_GUI_VIEW_PLOTUTIL_COLORMAP_H
#define BORNAGAIN_GUI_VIEW_PLOTUTIL_COLORMAP_H

#include "GUI/View/PlotUtil/ScientificPlot.h"

class IntensityDataItem;
class QCPRange;
class QCPColorScale;
class QCPColorMap;
class QCPLayoutGrid;
class UpdateTimer;

//! Presents 2D intensity data from IntensityDataItem as color map.

//! Provides a minimal functionality for data plotting and axes interaction.
//! Is used as a component of more complicated plotting widgets.

class ColorMap : public ScientificPlot {
    Q_OBJECT

public:
    explicit ColorMap(QWidget* parent = nullptr);

    void setIntensityItem(IntensityDataItem* intensityItem) override;

    QSize sizeHint() const override { return QSize(500, 400); }
    QSize minimumSizeHint() const override { return QSize(128, 128); }

    QCustomPlot* customPlot() override { return m_customPlot; }
    const QCustomPlot* customPlot() const override { return m_customPlot; }
    QCPColorScale* colorScale() { return m_colorScale; }

    //! Returns rectangle representing current axes zoom state in widget coordinates
    QRectF viewportRectangleInWidgetCoordinates();

    //! Returns PlotEventInfo corresponding to given axes coordinates.
    PlotEventInfo eventInfo(double xpos, double ypos) const override;

signals:
    void marginsChanged(double left, double right);

private:
    void onIntensityModified();
    void onUnitsChanged();
    void setGradient();
    void setInterpolation();
    void setLogz();

    void onDataRangeChanged(QCPRange newRange);
    void onXaxisRangeChanged(QCPRange newRange);
    void onYaxisRangeChanged(QCPRange newRange);
    void replot();
    void onTimeToReplot();
    void marginsChangedNotify();

    void initColorMap();
    void connectItem();

    void setConnected(bool isConnected);
    void setAxesRangeConnected(bool isConnected);
    void setDataRangeConnected(bool isConnected);
    void setUpdateTimerConnected(bool isConnected);

    void setFixedColorMapMargins();

    void setColorMapFromItem();
    void setAxesRangeFromItem();
    void setAxesZoomFromItem();
    void setAxesLabelsFromItem();
    void setDataFromItem();
    void setColorScaleAppearanceFromItem();
    void setDataRangeFromItem();
    void setColorScaleVisible();

    IntensityDataItem* intensityItem() const;

    QCustomPlot* m_customPlot;
    QCPColorMap* m_colorMap;
    QCPColorScale* m_colorScale;
    UpdateTimer* m_updateTimer;
    QCPLayoutGrid* m_colorBarLayout;
};

#endif // BORNAGAIN_GUI_VIEW_PLOTUTIL_COLORMAP_H
