//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/View/Job/JobSelectorWidget.cpp
//! @brief     Implements class JobSelectorWidget
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "GUI/View/Job/JobSelectorWidget.h"
#include "Base/Util/Assert.h"
#include "GUI/View/Job/JobListView.h"
#include "GUI/View/Job/JobPropertiesWidget.h"
#include "GUI/View/Tool/mainwindow_constants.h"
#include <QSplitter>
#include <QVBoxLayout>

namespace {

QVariant listToQVariant(const QList<int>& list)
{
    QList<QVariant> var_list;
    for (int val : list)
        var_list.push_back(QVariant(val));
    return QVariant(var_list);
}

QList<int> qVariantToList(const QVariant& var)
{
    QList<QVariant> var_list = var.toList();
    QList<int> list;
    for (QVariant var_val : var_list)
        list.push_back(var_val.toInt());
    return list;
}

} // namespace
JobSelectorWidget::JobSelectorWidget(JobModel* jobModel, QWidget* parent)
    : QWidget(parent)
    , m_jobModel(jobModel)
{
    setWindowTitle(GUI::Constants::JobSelectorWidgetName);
    setObjectName("JobSelectorWidget");

    auto* layout = new QVBoxLayout(this);
    layout->setContentsMargins(0, 0, 0, 0);
    layout->setSpacing(0);
    layout->setContentsMargins(0, 0, 0, 0);

    m_splitter = new QSplitter(Qt::Vertical, this);
    m_splitter->setChildrenCollapsible(true);
    layout->addWidget(m_splitter);

    m_jobListView = new JobListView(m_jobModel, m_splitter);
    m_splitter->addWidget(m_jobListView);
    m_splitter->setCollapsible(0, false);
    m_splitter->setStretchFactor(0, 1);

    m_jobProperties = new JobPropertiesWidget(this);
    m_splitter->addWidget(m_jobProperties);
    m_splitter->setCollapsible(1, false);
    m_splitter->setStretchFactor(1, 0);

    connect(m_jobListView, &JobListView::selectedJobsChanged, this,
            &JobSelectorWidget::onSelectedJobsChanged);
    connect(m_jobListView, &JobListView::selectedJobsChanged, this,
            &JobSelectorWidget::selectedJobsChanged);

    onSelectedJobsChanged(selectedJobItems());

    setFixedWidth(GUI::Constants::JOB_SELECTOR_PANEL_WIDTH);
    applySettings();
}

JobSelectorWidget::~JobSelectorWidget()
{
    saveSettings();
}

void JobSelectorWidget::resizeEvent(QResizeEvent* event)
{
    QWidget::resizeEvent(event);
    setMinimumWidth(0);
    setMaximumWidth(QWIDGETSIZE_MAX);
}

QVector<JobItem*> JobSelectorWidget::selectedJobItems() const
{
    return m_jobListView->selectedJobItems();
}

void JobSelectorWidget::makeJobItemSelected(JobItem* item)
{
    ASSERT(item);
    m_jobListView->selectJob(item);
}

void JobSelectorWidget::onSelectedJobsChanged(const QVector<JobItem*>& jobs)
{
    if (jobs.size() == 1)
        m_jobProperties->setJobItem(jobs.front());
    else
        m_jobProperties->setJobItem(nullptr);
}


void JobSelectorWidget::applySettings()
{
    QSettings settings;
    if (settings.childGroups().contains(GUI::Constants::S_JOB_SELECTOR_PANEL)) {
        settings.beginGroup(GUI::Constants::S_JOB_SELECTOR_PANEL);
        setFixedWidth(settings.value(GUI::Constants::S_JOB_SELECTOR_PANEL_WIDTH).toInt());
        m_splitter->setSizes(
            qVariantToList(settings.value(GUI::Constants::S_JOB_SELECTOR_SPLITTER_SIZES)));
        settings.endGroup();
    }
}

void JobSelectorWidget::saveSettings()
{
    QSettings settings;
    settings.beginGroup(GUI::Constants::S_JOB_SELECTOR_PANEL);
    settings.setValue(GUI::Constants::S_JOB_SELECTOR_PANEL_WIDTH, width());
    settings.setValue(GUI::Constants::S_JOB_SELECTOR_SPLITTER_SIZES,
                      listToQVariant(m_splitter->sizes()));
    settings.endGroup();
    settings.sync();
}
