/*
 * This file is used to generate the image files to visually describe the color
 * scheme types
 */

#include <list>
#include <math.h>
#include <cairomm/cairomm.h>
#include <glib.h>

const gdouble width = 150.0;
const gdouble size = width * 0.9;
const gdouble ring_width = (width / 2.0) / 2.0;
const gdouble marker_offset = 0.03;

#define INTENSITY(r, g, b) ((r) * 0.30 + (g) * 0.59 + (b) * 0.11)

static void
hsv_to_rgb (gdouble *h,
            gdouble *s,
            gdouble *v)
{
    gdouble hue, saturation, value;
    gdouble f, p, q, t;

    if (*s == 0.0)
    {
        *h = *v;
        *s = *v;
        *v = *v; /* heh */
    }
    else
    {
        hue = *h * 6.0;
        saturation = *s;
        value = *v;

        if (hue == 6.0)
            hue = 0.0;

        f = hue - (int) hue;
        p = value * (1.0 - saturation);
        q = value * (1.0 - saturation * f);
        t = value * (1.0 - saturation * (1.0 - f));

        switch ((int) hue)
        {
            case 0:
                *h = value;
                *s = t;
                *v = p;
                break;

            case 1:
                *h = q;
                *s = value;
                *v = p;
                break;

            case 2:
                *h = p;
                *s = value;
                *v = t;
                break;

            case 3:
                *h = p;
                *s = q;
                *v = value;
                break;

            case 4:
                *h = t;
                *s = p;
                *v = value;
                break;

            case 5:
                *h = value;
                *s = p;
                *v = q;
                break;

            default:
                g_assert_not_reached ();
        }
    }
}

static void
paint_ring (Cairo::RefPtr<Cairo::Context> cr,
            gint     width,
            gint     height,
            std::list<gdouble> stops = std::list<gdouble>())
{
    int xx, yy;
    gdouble dx, dy, dist;
    gdouble center;
    gdouble inner, outer;
    guint32 *buf, *p;
    gdouble angle;
    gdouble hue;
    gdouble r, g, b;
    Cairo::RefPtr<Cairo::Surface> source;

    center = width / 2.0;

    outer = size / 2.0;
    inner = outer - ring_width;

    /* Create an image initialized with the ring colors */

    buf = g_new (guint32, width * height);

    // for each row in the image data buffer
    for (yy = 0; yy < height; yy++)
    {
        // pointer to the start of the row
        p = buf + yy * width;

        // y-distance from the center of the circle
        dy = -(yy - center);

        // for each column in the row
        for (xx = 0; xx < width; xx++)
        {
            // x-distance from the center of the circle
            dx = xx - center;

            // calculate the radial distance to the center of the circle squared
            dist = dx * dx + dy * dy;
            // check whether the calculated point falls within the are that we
            // want to draw
            if (dist < ((inner-1) * (inner-1)) || dist > ((outer+1) * (outer+1)))
            {
                // if not, give up and try the next pixel
                *p++ = 0;
                continue;
            }

            // find the angle from the center of the circle to the current pixel
            angle = atan2 (dy, dx);
            // normalize from 0 to 2*PI
            if (angle < 0.0)
                angle += 2.0 * G_PI;

            // the angle is equal to the hue, normalized to a scale of 0 to 1.0
            hue = angle / (2.0 * G_PI);

            // calculate the rgb values for the specific hue assuming full
            // saturation and value
            r = hue;
            g = 1.0;
            b = 1.0;
            hsv_to_rgb (&r, &g, &b);

            // store the pixel value in the data buffer
            *p++ = (((int)floor (r * 255 + 0.5) << 16) |
                    ((int)floor (g * 255 + 0.5) << 8) |
                    (int)floor (b * 255 + 0.5));
        }
    }

    // create an image surface using the data buffer that we just filled
    source = Cairo::ImageSurface::create((unsigned char *)buf,
            Cairo::FORMAT_RGB24,
            width, height, 4 * width);

    /* Draw the ring using the source image */
    cr->save ();

    cr->set_source (source, 0, 0);

    cr->set_line_width (ring_width);
    cr->begin_new_path ();
    cr->arc (center, center,
            size / 2. - ring_width / 2.,
            0, 2 * G_PI);
    cr->stroke ();

    cr->restore ();

  /* Now draw the stop markers onto the source image
   */
    for (std::list<gdouble>::iterator stop_iter = stops.begin ();
            stop_iter != stops.end (); ++stop_iter)
    {
        cr->save ();

        r = *stop_iter < 0.0 ? *stop_iter + 1.0 : *stop_iter;
        g = 1.0;
        b = 1.0;
        hsv_to_rgb (&r, &g, &b);

        // the angle for calculating the hue is the inverse of the angle for
        // drawing in cairo
        gdouble ratio = -(*stop_iter);

        gdouble lower_angle = (ratio - marker_offset) * 2.0 * G_PI;
        gdouble upper_angle = (ratio + marker_offset) * 2.0 * G_PI;
        gdouble marker_extend = outer * 0.04;
        gdouble marker_inner = inner - marker_extend;
        gdouble marker_outer = outer + marker_extend;

        cr->begin_new_sub_path ();
        cr->arc (center, center, marker_outer, lower_angle, upper_angle);
        cr->line_to (center + cos (upper_angle) * marker_inner,
                     center + sin (upper_angle) * marker_inner);
        cr->arc_negative (center, center, marker_inner, upper_angle,
                          lower_angle);
        cr->close_path ();
        cr->restore ();

        cr->save ();
        cr->set_source_rgb (r, g, b);
        cr->fill_preserve ();

        cr->set_source_rgb (0., 0., 0.);
        cr->set_line_width (size / 100.0);
        cr->stroke ();
        cr->restore ();
    }

    g_free (buf);
}


int main(int argc, char** argv)
{
    // complements
    Cairo::RefPtr<Cairo::ImageSurface> comp_surface = Cairo::ImageSurface::create (Cairo::FORMAT_ARGB32, (int)width, (int)width);
    Cairo::RefPtr<Cairo::Context> complements_cr = Cairo::Context::create (comp_surface);
    std::list<gdouble> comp_stops;
    comp_stops.push_back(0.0);
    comp_stops.push_back(0.5);
    paint_ring (complements_cr, (int)width, (int)width, comp_stops);
    comp_surface->write_to_png ("complements.png");

    // split complements
    Cairo::RefPtr<Cairo::ImageSurface> split_surface = Cairo::ImageSurface::create (Cairo::FORMAT_ARGB32, (int)width, (int)width);
    Cairo::RefPtr<Cairo::Context> split_cr = Cairo::Context::create (split_surface);
    std::list<gdouble> split_stops;
    split_stops.push_back(0.0);
    split_stops.push_back(0.4333);
    split_stops.push_back(0.5666);
    paint_ring (split_cr, (int)width, (int)width, split_stops);
    split_surface->write_to_png ("split-complements.png");

    // triads
    Cairo::RefPtr<Cairo::ImageSurface> triad_surface = Cairo::ImageSurface::create (Cairo::FORMAT_ARGB32, (int)width, (int)width);
    Cairo::RefPtr<Cairo::Context> triad_cr = Cairo::Context::create (triad_surface);
    std::list<gdouble> triad_stops;
    triad_stops.push_back(0.0);
    triad_stops.push_back(0.33);
    triad_stops.push_back(0.666);
    paint_ring (triad_cr, (int)width, (int)width, triad_stops);
    triad_surface->write_to_png ("triads.png");

    // tetrads
    Cairo::RefPtr<Cairo::ImageSurface> tetrad_surface = Cairo::ImageSurface::create (Cairo::FORMAT_ARGB32, (int)width, (int)width);
    Cairo::RefPtr<Cairo::Context> tetrad_cr = Cairo::Context::create (tetrad_surface);
    std::list<gdouble> tetrad_stops;
    tetrad_stops.push_back(0.0);
    tetrad_stops.push_back(0.25);
    tetrad_stops.push_back(0.5);
    tetrad_stops.push_back(0.75);
    paint_ring (tetrad_cr, (int)width, (int)width, tetrad_stops);
    tetrad_surface->write_to_png ("tetrads.png");

    // analogous
    Cairo::RefPtr<Cairo::ImageSurface> analogous_surface = Cairo::ImageSurface::create (Cairo::FORMAT_ARGB32, (int)width, (int)width);
    Cairo::RefPtr<Cairo::Context> analogous_cr = Cairo::Context::create (analogous_surface);
    std::list<gdouble> analogous_stops;
    analogous_stops.push_back(-0.0833);
    analogous_stops.push_back(0.0);
    analogous_stops.push_back(0.0833);
    paint_ring (analogous_cr, (int)width, (int)width, analogous_stops);
    analogous_surface->write_to_png ("analogous.png");
}

