/* elf386tweakrelocs.c:
 *	for elf386 object tweaks given by address relocation to the given symbol
 *
 * usage:
 *	elf386tweakrelocs objfile (entry=symbol)+
 *
 * Author: A. Chentsov
 * Copying: LGPL
 *
 * ToDo: RELA sections, section names to search rel entry as an argument
 */ 

#include <sys/types.h>
#include <asm/types.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>

#include <unistd.h>
#include <fcntl.h>

#include <limits.h>		// ULONG_MAX

#include <sys/mman.h>		// mmap/munmap

#include <elf.h>

struct reloc_pair {
	__u32 entry;
	union {
		char *symbol;
		int sym_idx;
	};
};

int verbose;

void err_usage(char *program) {
	fprintf (stderr, "Usage: %s [--verbose] objfile relocentry=newsymbol\n", program);
}

int acceptable_objectfile (void *file) {
	return 1;
}

void *mapfile (char *filename);
int fixrelocs (char *file, struct reloc_pair *rpairs, int relocsnum); 
   
int main(int argc, char *argv[])
{
	// process args
	if (argc < 3) {
		err_usage (argv[0]); 
		return 1;
	}

#ifdef RELOCFIX_DEBUG
	verbose = 1;
#else
	verbose = 0;
#endif

	char *filename = NULL;
	struct reloc_pair *relocs;
	int rnum = 0;

	relocs = calloc (argc - 2, sizeof (struct reloc_pair));

	char **next = argv + 1;
	do {
		char *eqptr = strchr (next[0], '=');
		
		if (! eqptr) {
			if (strcmp (*next, "--verbose") == 0) 
				verbose = 1;

			else if (filename) {
				fprintf (stderr, "%s: pair expected\n", argv[0]); 
				err_usage (argv[0]);
				return 2;
			}
			else filename = *next;
		}
		// or pair
		else {
			char *ptr; 
			unsigned long relocentry;
			
			*eqptr = '\0';
			relocentry = strtoul (next[0], &ptr, 0);
			if ( ptr != eqptr || relocentry == ULONG_MAX || strchr (next[0], '-') ) {
				*eqptr = '='; 
				printf ("skipping bad relocentry %s\n", next[0]);
				continue;
			}
			
			*eqptr = '='; 
			relocs[rnum].entry = relocentry;
			relocs[rnum].symbol = eqptr + 1;
			rnum++;
		}	
				
	} while (*++next);

#ifdef RELOCFIX_DEBUG
	printf ("%d relocations\n", rnum);
	for (argc = 0; argc < rnum; argc++) 
		printf ("\t%d = %s\n", relocs[argc].entry, relocs[argc].symbol);
#endif
	
	// let's proceed with substitution

	// open & map the object file
	void *file = MAP_FAILED;
	file = mapfile (filename);
	if (file == MAP_FAILED) 
		return 3;

#ifdef RELOCFIX_DEBUG
	printf ("%s mapped at 0x%x\n", filename, file);
#endif

	if (! acceptable_objectfile (file)) {
		fprintf (stderr, "%s:  file %s\n", argv[0], filename);
		return 4;
	}
	
	return fixrelocs (file, relocs, rnum);
}

// do the thing

int fixrelocs (char *file, struct reloc_pair *rpairs, int relocsnum) 
{
	Elf32_Ehdr *elf_header =  (Elf32_Ehdr *) file;
	int result = 0; 	// Ok

	// drop unknown symbols?
	// fill in indeces and filter unneeded


	// dup relocs
	struct reloc_pair *reloc_symunknown[relocsnum];
	int unknown_num = relocsnum;
	// when pair comes to the next list the sym_idx field becames effective not symbol name
	struct reloc_pair *reloc_symresolved[relocsnum]; 
	int resolved_num = 0;
	
	{ 
		int count; 
		for (count = 0; count < relocsnum; count++) 
			reloc_symunknown[count] = &rpairs[count];
	}

	Elf32_Shdr *section_headers = (Elf32_Shdr *) (file + elf_header->e_shoff);
	Elf32_Shdr *symbols_hdr;
	Elf32_Sym *syms;
	char *strtab;
	char *sec_strings = file + section_headers[elf_header->e_shstrndx].sh_offset;

	int section;
	// in the manner of linux/kernel/module.h
	for (section = 1; section < elf_header->e_shnum; section++) 
		if (section_headers[section].sh_type == SHT_SYMTAB) {
			symbols_hdr = section_headers + section;
			
			syms = (Elf32_Sym *) (file + symbols_hdr->sh_offset);
			strtab = file + section_headers[symbols_hdr->sh_link].sh_offset;
			break;
		}

#ifdef RELOCFIX_DEBUG
	printf ("found symsection %s\n", sec_strings + symbols_hdr->sh_name);
	fflush (stdout);
#endif

	// resolve symbol indeces
	int idx, number = symbols_hdr->sh_size / sizeof (Elf32_Sym);

	for (idx = 0; idx < number; idx++) {
		// looking for symbol index
		char *symbol = strtab + syms[idx].st_name;
		int u_idx;

#ifdef RELOCFIX_DEBUG
//		printf ("processing symbol %s\n", symbol);
//		fflush (stdout);
#endif
		u_idx = 0; 
		while (u_idx < unknown_num) {
			if (! strcmp (symbol, reloc_symunknown[u_idx]->symbol)) {
				// resolve
				reloc_symunknown[u_idx]->sym_idx = idx;
				// moving to resolved
				reloc_symresolved[resolved_num++] =  reloc_symunknown[u_idx];

				// delete from unknown
				if (u_idx != --unknown_num) 
					reloc_symunknown[u_idx] = reloc_symunknown[unknown_num];
#ifdef RELOCFIX_DEBUG
				printf ("resolved %s, ndx %d\n", symbol, idx);
#endif
				// symbol can occur more than once	
				// break; // from unknown reloc enum

				// redo
				continue;
			}

			u_idx++;
		}
	}
			

#ifdef RELOCFIX_DEBUG
		printf ("syms done\n");
		fflush (stdout);
#endif
	// linux/arch/i386/kernel/module.c,  apply_relocate function
	for (section = 1; section < elf_header->e_shnum; section++) {
		unsigned int info = section_headers[section].sh_info;

		if (info >= elf_header->e_shnum) 
			continue;

		if ( (section_headers[section].sh_type == SHT_REL) && 
		     (section_headers[info].sh_flags & SHF_ALLOC) ) {

			Elf32_Rel *entry = (Elf32_Rel *) (file + section_headers[section].sh_offset);
			int entries_num = section_headers[section].sh_size / sizeof (Elf32_Rel);

			int entry_idx;

			if (verbose)
				printf ("processing relocation section %s\n", sec_strings + section_headers[section].sh_name);
			
			for (entry_idx = 0; entry_idx < entries_num; entry_idx++, entry++) {
				int fix_idx;
				for (fix_idx = 0; fix_idx < resolved_num; fix_idx++) {
					switch (ELF32_R_TYPE(entry->r_info)) {
					case R_386_32:
					case R_386_PC32:
						if (entry->r_offset == reloc_symresolved[fix_idx]->entry) {
							// fixing
							entry->r_info = ELF32_R_INFO (reloc_symresolved[fix_idx]->sym_idx, ELF32_R_TYPE (entry->r_info));
							if (verbose) {
								printf ("\t0x%x:", entry->r_offset);
								printf ("\t%s\t ->", strtab + syms[ELF32_R_SYM(entry->r_info)].st_name );
								printf ("   %s\n", strtab + syms[ELF32_R_SYM(entry->r_info)].st_name );
							}

						}
						break;

					// default:
						// wrong relocation
					}
				}
			}
		}
	}

	return result;
}


void *mapfile (char *filename) 
{
	int fd = open (filename, O_RDWR);
	if (fd < 0) {
		fprintf (stderr, "Bad file name %s\n", filename);
		return MAP_FAILED;
	}

	void *file = mmap (
		NULL,
		lseek (fd, 0, SEEK_END),
		PROT_READ | PROT_WRITE,
		MAP_SHARED,
		fd,
		0
	);

	if (file == MAP_FAILED) {
		perror ("mmap");
	}

	close (fd);

	return file;
}
