/* -*- Mode: C++; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* libe-book
 * Version: MPL 2.0 / LGPLv2.1+
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 * Alternatively, the contents of this file may be used under the terms
 * of the GNU Lesser General Public License Version 2.1 or later
 * (LGPLv2.1+), in which case the provisions of the LGPLv2.1+ are
 * applicable instead of those above.
 *
 * For further information visit http://libebook.sourceforge.net
 */

#ifndef PDXPARSER_H_INCLUDED
#define PDXPARSER_H_INCLUDED

#include <boost/scoped_ptr.hpp>

class WPXDocumentInterface;
class WPXInputStream;

#define PDX_CODE(s) ((s[0] << 24) | (s[1] << 16) | (s[2] << 8) | s[3])

namespace libebook
{

struct PDXParserImpl;

class PDXParser
{
// disable copying
  PDXParser(const PDXParser &other);
  PDXParser &operator=(const PDXParser &other);

public:
  virtual ~PDXParser() = 0;

  /** Test whether the input document is in supported format.
    *
    * This function will typically only read the Palm Database Format
    * header, but it might read the info block too.
    *
    * @return true if the document is supported
    */
  bool isSupported();

  /** Parse input and produce output to @e document.
    *
    * @return true if the input has been parsed successfully.
    */
  bool parse();

protected:
  /** Instantiate a parser for a document in Palm Database Format.
    *
    * @arg[in] input input stream
    * @arg[in] document output document generator
    */
  PDXParser(WPXInputStream *input, WPXDocumentInterface *document);

  /** Get the document generator used for this parsing run.
    *
    * @return the output document generator
    */
  WPXDocumentInterface *getDocument() const;

  const char *getName() const;

  /** Return a stream for the format's appInfo record, if it is present.
    *
    * A parser for a subformat might want to peek into the appInfo
    * record in isSupported().
    *
    * @return a newly allocated stream spanning the appInfo record or 0,
    *         if there is no appInfo record
    */
  WPXInputStream *getAppInfoRecord() const;

  /** Return a stream for the format's index record.
    *
    * A parser for a subformat might want to peek into the index record
    * in isSupported().
    *
    * @return a newly allocated stream spanning the index record
    */
  WPXInputStream *getIndexRecord() const;

  /** Return the number of data records in the document.
    *
    * @return the number of data records
    */
  unsigned getDataRecordCount() const;

  /** Return a stream for the n-th data record.
    *
    * @arg[in] n the stream number, 0-based
    * @return a newly allocated stream spanning one data record
    */
  WPXInputStream *getDataRecord(unsigned n) const;

  /** Return a stream for all data records.
    *
    * @return a newly allocated stream spanning all data records
    */
  WPXInputStream *getDataRecords() const;

  /** Return a stream for data records in a given range.
    *
    * @arg[in] first the first data record to include
    * @arg[in] last the first data record not to include
    *
    * @return a newly allocated stream spanning data records in a given range
    */
  WPXInputStream *getDataRecords(unsigned first, unsigned last) const;

private:
  /** Check whether a specific subformat is supported.
    *
    * @arg[in] type the subformat's type
    * @arg[in] creator the document's creator app.
    *
    * @return true if the subformat is supported
    */
  virtual bool isFormatSupported(unsigned type, unsigned creator) = 0;

  virtual void readAppInfoRecord(WPXInputStream *record) = 0;
  virtual void readSortInfoRecord(WPXInputStream *record) = 0;
  virtual void readIndexRecord(WPXInputStream *record) = 0;
  virtual void readDataRecord(WPXInputStream *record, bool last = false) = 0;

  virtual void readDataRecords();

  void readHeader();

  WPXInputStream *getRecordStream(unsigned n) const;

private:
  boost::scoped_ptr<PDXParserImpl> m_impl;
};

}

#endif // PDXPARSER_H_INCLUDED
/* vim:set shiftwidth=2 softtabstop=2 expandtab: */
