/*
 * // Copyright (c) Radzivon Bartoshyk 8/2025. All rights reserved.
 * //
 * // Redistribution and use in source and binary forms, with or without modification,
 * // are permitted provided that the following conditions are met:
 * //
 * // 1.  Redistributions of source code must retain the above copyright notice, this
 * // list of conditions and the following disclaimer.
 * //
 * // 2.  Redistributions in binary form must reproduce the above copyright notice,
 * // this list of conditions and the following disclaimer in the documentation
 * // and/or other materials provided with the distribution.
 * //
 * // 3.  Neither the name of the copyright holder nor the names of its
 * // contributors may be used to endorse or promote products derived from
 * // this software without specific prior written permission.
 * //
 * // THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * // AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * // IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * // DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * // FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * // DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * // SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * // CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * // OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * // OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/**
Generated Y0 Taylor expansion at zero and extremums.

Generated by SageMath:
```python
mp.prec = 115
terms = 28
print(f"pub(crate) static Y0_COEFFS_TAYLOR: [[(u64, u64); {terms}]; {len(y0_zeros)}] = [")

def get_constant_term(poly, y):
    for term in poly.operands():
        if term.is_constant():
            return term

def print_taylor_coeffs(poly, n):
    print("[")
    for i in range(0, n):
        coeff = poly[i]
        print_double_double("", RealField(300)(coeff))
        # print(f"{double_to_hex(coeff)},")
    print("],")

prev_zero = 0

for i in range(0, len(y0_zeros)):
    k_range = y0_zeros[i]
    range_diff = k_range - prev_zero

    x0 = mp.mpf(k_range)
    from mpmath import mp, bessely, taylor, chebyfit
    poly = taylor(lambda val: bessely(0, val), x0, terms)
    print_taylor_coeffs(poly, terms)
    prev_zero = y0_zeros[i]

print("];")
```
**/
pub(crate) static Y0_COEFFS_TAYLOR: [[(u64, u64); 28]; 47] = [
    [
        (0xbc749367c4c05aaa, 0x3fe0aa48442f014b),
        (0x0000000000000000, 0x0000000000000000),
        (0x3c649367c4c05aaa, 0xbfd0aa48442f014b),
        (0x3c4098c5b27f9542, 0x3fa439fac16525f6),
        (0xbc25adcf3e0230ee, 0x3f80d2af4e932386),
        (0xbbe4e773096516dd, 0x3f4f71646bea8111),
        (0x3bf3295907be08a9, 0xbf5444bda8b664a9),
        (0x3bcc92ed3e7dac0f, 0x3f384c220672ab82),
        (0xbbc2b56b5db9ac40, 0xbf217ab4afac0735),
        (0x3ba57929ce662323, 0x3f0dafb9b8983d86),
        (0xbb9a34080fad5051, 0xbef8bb962fa6c504),
        (0xbb70f97215a75a9f, 0x3ee48b6770c210bf),
        (0xbb6ac5099b6b7783, 0xbed1364ddbddf33b),
        (0x3b5c6a28d5d49339, 0x3ebd06aadd881cef),
        (0x3b360737a2bfac90, 0xbea8998eefe8caa1),
        (0x3b10ceb1fd02d154, 0x3e94f198aa7c73a8),
        (0x3b201bb903ea5d53, 0xbe81e742c1cf6acb),
        (0x3b0e3c97eb74692a, 0x3e6eb820636013a9),
        (0xbae9e5875024afeb, 0xbe5a709593e5d778),
        (0x3a9c3113327b82cf, 0x3e46d299b1a6ab0d),
        (0x3ad32a9aab166472, 0xbe33c08d3e0a3e4c),
        (0xbab2ec0cd7aefe76, 0x3e2122b2a831eedc),
        (0xba7fd1a1eaeba4e2, 0xbe0dcbb767b405d4),
        (0xba9f28de864d6d7d, 0x3df9f4d817711be0),
        (0xba84e5784f3ba466, 0xbde6a732ce7db295),
        (0x3a2ba83b1dceb593, 0x3dd3cdb1820544fc),
        (0x3a64cd3a569b8afb, 0xbdc156d9d7a9603b),
        (0xba4c3618e715b8ab, 0x3dae68388d1e7164),
    ],
    [
        (0x0000000000000000, 0x0000000000000000),
        (0xbc5a4026e436c4d3, 0xbfd9c34256a12a0c),
        (0xbc49ded6e8236fd3, 0x3faa09c9290367ef),
        (0xbc253b02d2a53d30, 0x3fadf6d59bf50ebd),
        (0xbc18dbf118b1a2cc, 0xbf7c116fdc598096),
        (0x3bf40db0aa3d7078, 0xbf61e32bc4ef8a41),
        (0xbbcbb6264c727077, 0x3f29982765166254),
        (0xbb86be82f4783a97, 0x3f0ab2c1fecdcfc4),
        (0x3b74540a4df5362c, 0xbed486371e6a663e),
        (0xbb38925afac5b2b8, 0xbe93b21684089b4c),
        (0x3af8953a9c0487f4, 0xbe5777058741bf97),
        (0xbad841f372dee964, 0x3e5452ea00945a70),
        (0x3aa79f6248f7f1b0, 0xbe2f6938829664f1),
        (0xba9880934623f972, 0x3e0a5a1f3a2a293e),
        (0xba74d07ce1ea2db9, 0xbde994e736de8ef8),
        (0x3a666975b24e41c6, 0x3dc88ae5c83467bc),
        (0x39e7564160acaf67, 0xbda75329361f19bb),
        (0x39f36d6ccdec6121, 0x3d8643c7a49b7f43),
        (0x39e06cdcecde2c8d, 0xbd655480c70f5929),
        (0xb9ee4b7548f46bdb, 0x3d447c1c44c1487e),
        (0x398be3f2b0a46b7a, 0xbd23b7d1fbf0fdd0),
        (0xb9aeefcda909f8cb, 0x3d0305531056d810),
        (0x398546902be387f3, 0xbce2626f5e5dfcac),
        (0x395bf1c6ad49bc60, 0x3cc1cd4c7460b8d8),
        (0xb91985693844d5c0, 0xbca1445851bfea5f),
        (0xb92b3bdde2175982, 0x3c80c63a86aa43b3),
        (0xb8ff7f8fe9d70fe2, 0xbc6051ca830eb872),
        (0xb8b0d0de6a7a8a72, 0x3c3fcc0fa90bfb3f),
    ],
    [
        (0x3c7b8d2a1c496808, 0xbfd5c7c556f0c19a),
        (0x0000000000000000, 0x0000000000000000),
        (0xbc6b8d2a1c496808, 0x3fc5c7c556f0c19a),
        (0xbc2b4d44aaadbbec, 0xbf8564d4b1ed0d7e),
        (0x3c2e4289c3acba27, 0xbf8a15d92dfe3e27),
        (0xbbedbea927b719eb, 0x3f4b438430469ff7),
        (0x3bdc1760512f76cc, 0x3f37a8924cc2f914),
        (0xbb70219aa771d161, 0xbef5f69b4a8a3c05),
        (0xbb77299907aec5c4, 0xbed85b940eb607f9),
        (0x3b3d5ae26ad627b8, 0x3e955ac0f5c3162c),
        (0xbb0f58c09c1fabd1, 0x3e6d563e18b47eb9),
        (0x3a99750b0b40769a, 0xbe23eac3a45e06ef),
        (0x3a9e3d09f9f42f26, 0xbe0007671d55a8be),
        (0xba495377f5d84985, 0x3dc04e13e8d2ceb5),
        (0x3981b2ad63946aa5, 0xbd46e2b306cbb1f3),
        (0xb9fbadc4f0bb048c, 0x3d54d18f7c90e926),
        (0xb9c8225b40e694f7, 0xbd34980895b5067a),
        (0xb998c3893e3c0087, 0x3d0a5b439cef90cb),
        (0x3977814f4b07cebf, 0xbce1c9c4c0d530e1),
        (0x3952fcdfd498e788, 0x3cb933cc6493bf1c),
        (0x393ffe3660f335f2, 0xbc91c67122e8e134),
        (0x3904bb25b4c5ee57, 0x3c6904e906c46283),
        (0x38cd1dbe18c0ee49, 0xbc41a5ff5486c2fe),
        (0xb8a09b2fbba2c634, 0x3c18f24433ae1991),
        (0x3898e91198e4959c, 0xbbf1a84c8250ebd5),
        (0xb856349e52eccab3, 0x3bc90826d8cc3eff),
        (0xb848f5b7e7d4ec81, 0xbba1c456da51d36d),
        (0x37f66d77762bebcf, 0x3b7940bb04835b31),
    ],
    [
        (0x0000000000000000, 0x0000000000000000),
        (0xbc78d4484b7cd2a6, 0x3fd334cca0697a5b),
        (0xbc341c66c86d3ebd, 0xbf95aef611fc4d57),
        (0xbc44929b43aab7fb, 0xbfa8969c64cbf452),
        (0x3bfab353487eb36f, 0x3f6b2f14a95527cb),
        (0x3c0aa67a62fad710, 0x3f61d35e85fde2a3),
        (0xbbb270ba54459cab, 0xbf226dd71e391c8b),
        (0xbb61863b08bc5950, 0xbf08177e4fe52324),
        (0x3b494954a2d7de21, 0x3ec6a9227352f96a),
        (0xbb44a5b1e0c3c3e1, 0x3ea34aa7573ef14e),
        (0xbb06984676a524bc, 0xbe60a2814d7ac448),
        (0xbad9251769eb31a9, 0xbe3442a3d3359ad9),
        (0x3a86fc915b18c175, 0x3defa6c509566c72),
        (0xba3b534d6e1095ea, 0x3dbf14e5584823db),
        (0x3a16e8b19c361860, 0xbd7714c292be3936),
        (0x39e500ccc143989e, 0xbd40679cd5855172),
        (0xb998a55fa2ea549b, 0x3cf3365d5a786a7b),
        (0x395c00dd62ac7e1c, 0x3cc38cf06b47bc6e),
        (0x38f0c8bd8691968e, 0xbc82bd5a83b64d59),
        (0x38bddc7ea088ddcd, 0x3c326e92cf91d829),
        (0x38b27643db6c3b27, 0xbc127116123c3d66),
        (0x388c47f57a6d55b0, 0x3be8df4f1bd4a3f7),
        (0x3853b5c4a12853b5, 0xbbb9b0c7fab8762b),
        (0x381d1068b57233fd, 0x3b8b3b04447f58f8),
        (0x37f6b2b5793b64aa, 0xbb5dbd231117d61d),
        (0xb7d3a5abcdba22eb, 0x3b3035d90c22fa4e),
        (0xb786f2af60849004, 0xbb01a58ba966a549),
        (0x376712c78c816e41, 0x3ad33cc7a0e946ec),
    ],
    [
        (0x3c61dc672a53c590, 0x3fd15f993fceab5c),
        (0x0000000000000000, 0x0000000000000000),
        (0xbc51dc672a53c590, 0xbfc15f993fceab5c),
        (0xbc0d1c390189cb30, 0x3f758ef6efbed797),
        (0xbc243ede64b782d9, 0x3f86395dfe49fcd4),
        (0xbbbb4484582c2699, 0xbf3fb15104a41c00),
        (0xbbd4410e752be535, 0xbf35f88a11d5564f),
        (0xbb8b2d9b83a11c89, 0x3eef37d226a824b7),
        (0xbb61fb17827843f5, 0x3ed6f7bab104f355),
        (0x3b2da2cb75d8087f, 0xbe8f0c45a3824d70),
        (0xbac1132bedbb6ebc, 0xbe6dfe11df12c715),
        (0xbac5290f391bbfe1, 0x3e2311adc2e753ee),
        (0xba92039061518817, 0x3dfad34e18504e16),
        (0xba4d6cd9e6f85fe9, 0xbdafdc8061ae8b15),
        (0xba189257c63f6446, 0xbd819498ca6cef9b),
        (0xb9c8f0f4bb78628c, 0x3d33a291d5991f74),
        (0x399127448ae5dbb8, 0x3d016f7ae80cad77),
        (0xb958d1b0f5eaa04b, 0xbcb1f8d80deccee6),
        (0x3910d158fa5b41a8, 0xbc7c27b5350a923c),
        (0x38c3559986f928ac, 0x3c2d08034e70dd4a),
        (0xb86c6bb218da7ccb, 0x3befac2006e277e3),
        (0x383c518f1fca847d, 0xbb93e926868daea2),
        (0x380ef79adc3898ef, 0xbb6dcdf980ddfdf4),
        (0x37cea115c120894b, 0x3b2c4665b1564c6a),
        (0xb783f5bf36c1dc64, 0xbaec0521ead39203),
        (0xb74144c7cd1de85b, 0x3abed64c1afbe47b),
        (0x3722cfb0d396394e, 0xba8e04ab2a193e84),
        (0xb6f088c0c22460a8, 0x3a5a77e09ed6a09d),
    ],
    [
        (0x0000000000000000, 0x0000000000000000),
        (0x3c61e8f568f8c6b9, 0xbfcff635cc72b9f1),
        (0x3c0fa6ebe8b75862, 0x3f89036451ff57c5),
        (0x3c46345bcf3c5b7d, 0x3fa4e667a71556af),
        (0x3c065e386829f03c, 0xbf60325ee41e910c),
        (0x3bfde6b6846da9d0, 0xbf5fe23914fb9128),
        (0xbbb9c28522c54ab2, 0x3f17f84d7c50e4c4),
        (0xbbaeab0290c99b85, 0x3f06afdd57be1e14),
        (0xbb2bbdfb6c4ce197, 0xbec04053abf4386a),
        (0x3b3787c4b6b6a0b6, 0xbea2aea9ec48d8fd),
        (0xbaf17a3632e2edc5, 0x3e593eb9f1ddb4e9),
        (0x3ad9f831e08fbb99, 0x3e3428a3a3e30a13),
        (0xba49cae682f11d51, 0xbde99d8c42f7a020),
        (0xba591eb785e59448, 0xbdbec9805045aa58),
        (0xba086b5bad70f23b, 0x3d72613659ce994f),
        (0xb9eb4013d9ba0032, 0x3d418f302bdba980),
        (0x38e42c05bf9582a5, 0xbcf3bc9ccb704253),
        (0x3952d598704c3b46, 0xbcbf0dc4067be8b2),
        (0x391246064faa45fa, 0x3c706e491f458956),
        (0xb88908e1a67ee56c, 0x3c36013bac769277),
        (0x388f595373e53a2e, 0xbbe616b4d62a47f6),
        (0x384c63a51904dc22, 0xbba93fb6fa556eab),
        (0xb7fb730412560441, 0x3b57963a88acf606),
        (0xb7aad1cf0b52dca3, 0x3b1937ef4e3db6a9),
        (0x37592af1b29a1e7c, 0xbac8387c0e875a84),
        (0xb6f7ae247b07d290, 0xba80a12689ec647c),
        (0x36beb9fdadfcf39a, 0x3a1bcc6e30896704),
        (0xb697aa9158bd1808, 0x39fd4db12fab6895),
    ],
    [
        (0x3c54d14c77bc1691, 0xbfcdc14ea14e89f9),
        (0x0000000000000000, 0x0000000000000000),
        (0xbc44d14c77bc1691, 0x3fbdc14ea14e89f9),
        (0x3c00ff03f97580f3, 0xbf6b037fe9cf2a52),
        (0xbc2f19a8f5b1379e, 0xbf8367d7d608e4ba),
        (0x3bdd892317f291c8, 0x3f34abef5636e4f7),
        (0x3bd5ba5b5b609683, 0x3f33d8a661229259),
        (0xbb700a3f00979ee5, 0xbee5cfe92a1a2c45),
        (0xbb6087e84c57d2ad, 0xbed571814a1aa301),
        (0x3b22b6740da6456f, 0x3e87414e33c9bacd),
        (0x3afa4836a36f8841, 0x3e6ca7069e73d1d9),
        (0x3abe65fd6bda59de, 0xbe1e0b3a705ff247),
        (0x3a95774cb23eb7b8, 0xbdfa15dd62cf9f10),
        (0xba399fbe055cb8be, 0x3daa33948eada12c),
        (0x3a2240f402a7b379, 0x3d81419b1fd428ef),
        (0x39cc415e74e0cbc8, 0xbd30882f60b8a775),
        (0xb991c41d97727556, 0xbd015e5c0af749d4),
        (0x392937d5bb210372, 0x3cafb1e115e08994),
        (0x38f58b3c9e456997, 0x3c7b838263708865),
        (0x38c5cd0199e8f790, 0xbc27e5d539870f02),
        (0xb89a3dc2e116c3ad, 0xbbf19dae392457a1),
        (0x383bc230f8650284, 0x3b9d2b72778fe832),
        (0x37f13768b14c0c18, 0x3b6298bb7e2277f2),
        (0xb7aa1239c7070276, 0xbb0d50baf60db64c),
        (0xb773592299e9674f, 0xbad08ad53b850ab4),
        (0x3717cc8a5ae6e9ab, 0x3a79141842fa569d),
        (0xb6b9bec849ea1c6d, 0x3a38ab0a361907e6),
        (0xb67a764f692d7a9b, 0xb9e14491ed3e919d),
    ],
    [
        (0x0000000000000000, 0x0000000000000000),
        (0x3c41398cacaa32d4, 0x3fcbf32a27594007),
        (0x3bfe5ce4af724b17, 0xbf80bc2d84e65214),
        (0x3c4acd5148eb7e9d, 0xbfa26cab38a8b368),
        (0x3bfab9a1103e2000, 0x3f55f03e47165d72),
        (0xbbc4598141a3f849, 0x3f5caaa76e34992d),
        (0xbbbb7ab55e886623, 0xbf10c5f18c46d030),
        (0xbb99f58bad5e7aa5, 0xbf04f0af7d46cd48),
        (0xbb3040f00afca09c, 0x3eb7d1e28094e21a),
        (0xbb3f12dedaa2ba65, 0x3ea1ad0731228479),
        (0xbae9fe69a3662740, 0xbe536021c8cb3c34),
        (0x3ad92bf3f184780b, 0xbe3371ae6c759181),
        (0x3a630cc7c21c0cfc, 0x3de46ce077a8bed7),
        (0xba5feb6d0e335e60, 0x3dbe235d3564b639),
        (0xb9d7ae1d94bd92ac, 0xbd6e408e259c72fb),
        (0xb9c8768c2460ab80, 0xbd415f07ac87e832),
        (0xb91b6919f885a208, 0x3cf0a44f77f6a8ef),
        (0x393312b132a8d347, 0x3cbefd8ab52f8eb9),
        (0x38f1bfe622c31b37, 0xbc6c54ebb5ad3869),
        (0xb8a61a375002a1c0, 0xbc360b43cbb5f5e9),
        (0xb88cc62cec929222, 0x3be33d74aacb57e2),
        (0x3842ffe5c04b22b9, 0x3ba99bfbb3fe627d),
        (0x37e1d4df5722a5b0, 0xbb555c0969d70a77),
        (0x37b44cfc1475dce9, 0xbb18c29d2318e9b5),
        (0xb7682246c98d07a1, 0x3ac3c0750d221b4f),
        (0xb717ffcd427c1eee, 0x3a843fa688c2eb51),
        (0x369fc06d11ba5fce, 0xba2ef4ae2931c6ca),
        (0xb674d5d2266c5526, 0xb9ec57869d64feb9),
    ],
    [
        (0x3c57ba12cd0fc91f, 0x3fca7022be084d99),
        (0x0000000000000000, 0x0000000000000000),
        (0xbc47ba12cd0fc91f, 0xbfba7022be084d99),
        (0xbc0b3327197c9da3, 0x3f62ee079d020b12),
        (0x3c2d0f8f36713120, 0x3f8163191c30aa62),
        (0xbbcc7f2dbb383204, 0xbf2d7806ea72fc76),
        (0x3bcf9ec581434d38, 0xbf320f95702b1d4e),
        (0x3b899e1a0f3c0857, 0x3ee00610882294aa),
        (0x3b61505ce3b29a78, 0x3ed3e398cbc472ea),
        (0xbae050c77864dd20, 0xbe81bbe181c65162),
        (0xbb0c3fb8339198a1, 0xbe6b0f89b7c61f3a),
        (0x3ab91b7c8e3c7c3c, 0x3e17c3f85882049b),
        (0xba976a7c841dba14, 0x3df90236614c84dc),
        (0xba43ab3b18e59a65, 0xbda564920d1387dc),
        (0x3a2b1754ca712321, 0xbd80be1811255cfe),
        (0x39b342bebee2071b, 0x3d2bb9712586bfac),
        (0xb99c92d876d25063, 0x3d0101c31e1df223),
        (0xb916e23fb5f67bb8, 0xbcab298cda76e77b),
        (0x3918a11b915d1274, 0xbc7b22acc441d872),
        (0x38cdcc71612b99d0, 0x3c24dca37a32e97c),
        (0x38855c0754c54ab3, 0x3bf176e94ee1fe21),
        (0x3836cf20a369d629, 0xbb99d4f0d74e9093),
        (0x3802940330562608, 0xbb6287585b150c7c),
        (0x37a0f4cbacc81873, 0x3b0a5c015f1f057c),
        (0x3779aaaf8d850193, 0x3ad07cc0082a44cf),
        (0xb70fd62f89ce46a9, 0xba76909f83007714),
        (0xb6d4e9d85e17b208, 0xba38f88423d4ad97),
        (0xb66d34e3378a5539, 0x39e072fc561ac2bd),
    ],
    [
        (0x0000000000000000, 0x0000000000000000),
        (0xbc65b9c39e42719e, 0xbfc925c35988ee29),
        (0xbbdda9c0588bc7cc, 0x3f7862549367591e),
        (0x3c4a142ed6a5521f, 0x3fa0a4512039d6a2),
        (0x3be94ce533c65cad, 0xbf5013b38cfb9292),
        (0xbbea2a14cf0f61af, 0xbf5a24a1215f6684),
        (0xbbae7b4c448b42eb, 0x3f08f91421377fad),
        (0x3baf72fd0f5209a4, 0x3f035d17cec0172f),
        (0x3b3cd242e4f5ce10, 0xbeb2283a93114096),
        (0x3b259dda3f9c3cd8, 0xbea099e71392f54e),
        (0xbad6dce1bfae1a53, 0x3e4e5de01e2e6566),
        (0xbab52eec76dd7e24, 0x3e32885854ea8b06),
        (0x3a6795ca81b0c1a2, 0xbde0730c2985fd22),
        (0x3a4e06d8277ffd55, 0xbdbd1743cbb2e11a),
        (0x3a013448997f1418, 0x3d68f8728745e1e6),
        (0x39dd9a182e4412f8, 0x3d40f166cb29a7d1),
        (0x398dd4bcf72e3139, 0xbcec10e67c004e3b),
        (0xb9542acf7b7def7d, 0xbcbe7aec1ba33e83),
        (0xb908d6ed91615a58, 0x3c68543e12c9bf9a),
        (0x38a13a9d572a9420, 0x3c35d3939f6d1ed2),
        (0x387a4284a63f1993, 0xbbe0c7ba11981a01),
        (0x38405b9a6bff4fca, 0xbba97d9177222a00),
        (0xb7f4c779b6c742c8, 0x3b52dfb95f4a8634),
        (0xb7aba81ffd3b5c41, 0x3b18c04f3759b6eb),
        (0x375fa028706c2252, 0xbac1a799c1562309),
        (0xb729c62dd7828d2a, 0xba844dc9c3e85489),
        (0xb6cf5aac2a393499, 0x3a2bea86b90f9faf),
        (0x36614b1df5aeed9d, 0x39ec85971210b79b),
    ],
    [
        (0xbc63db68c567283b, 0xbfc80781c32422e7),
        (0x0000000000000000, 0x0000000000000000),
        (0x3c53db68c567283b, 0x3fb80781c32422e7),
        (0xbbfab5094c039454, 0xbf5c6923374d561f),
        (0xbc032419a9d405db, 0xbf7fbe6df840847f),
        (0x3bc0f03bc2e75d6b, 0x3f264f4711a85f1a),
        (0x3bbd9972e6c197fd, 0x3f309ff42b0d7a68),
        (0xbb61aba5a7ce6455, 0xbed8a60685a597d9),
        (0xbafdb153b13f2cf0, 0xbed282d26a74c38e),
        (0xbb1d21d4a94b9e91, 0x3e7bdb57a889a4c1),
        (0xbb099a040a6939c6, 0x3e697d9c12a60cef),
        (0x3a35e114fc2a1213, 0xbe131aa8691738fb),
        (0x3a93c999393c858b, 0xbdf7d486eee39a85),
        (0x3a223c1c4424f0db, 0x3da197f258214b3d),
        (0xba1644a8cbcb4af5, 0x3d801dd7f92195ff),
        (0xb9ca37ef1c0ef3fa, 0xbd27480b9efaa860),
        (0xb98847e4ec1195cf, 0xbd00833ea1f83461),
        (0xb928c4b1729bab7e, 0x3ca73bf7964edc26),
        (0x3913ebe30cf0771f, 0x3c7a891d704fef9b),
        (0x3881294f9e950341, 0xbc22222dcf094b80),
        (0xb898aeadae639677, 0xbbf12db9d21e37ed),
        (0x37ea537285a44833, 0x3b96c4638e4cc003),
        (0xb7f3ee61741a27e7, 0x3b624ff2a82fc9c9),
        (0xb7987ce466329a95, 0xbb0782fd4fc7cb7b),
        (0xb75dca0980ad8630, 0xbad05bab97dc88c3),
        (0x36f1b4cbf5a58e0a, 0x3a74565e5f91bb35),
        (0xb6d553555a830f19, 0x3a38da3118f887bc),
        (0x36758225c016dcfa, 0xb9ddea7aa8da61e4),
    ],
    [
        (0x0000000000000000, 0x0000000000000000),
        (0x3c64fef53f4893e5, 0x3fc70c4f66cab47f),
        (0xbc1e032768318e4d, 0xbf72c6731071e936),
        (0x3c2f806b21bc9577, 0xbf9e924b85a17361),
        (0xbbe1434be87c6363, 0x3f48d6c364d92082),
        (0x3bf5cbe0638f4cef, 0x3f58247b02d6b0f6),
        (0xbbaade8b3660fd4e, 0xbf0376125938561d),
        (0x3bae81e47dd383cf, 0xbf0206da232a2b6a),
        (0x3b424b3c3173e20f, 0x3eacaf84db871510),
        (0xbb022e3bac70bc67, 0x3e9f3671177306ca),
        (0x3a9f8b17b052db72, 0xbe48654b001006cd),
        (0x3ad272b921d7360d, 0xbe3199ec88b18766),
        (0x3a6b9a3bf5083c1d, 0x3ddaeb1a849305fb),
        (0x3a57cadaf3f1d693, 0x3dbbe67c197c48c0),
        (0xb9c7cc3d8d1e5e49, 0xbd64cdfa0bd69898),
        (0xb9db4ff4c3ee8f12, 0xbd4064936f97922f),
        (0xb9777557537dd027, 0x3ce7c83b5186222f),
        (0x395de8595a9345f0, 0x3cbdb73ad7fa8bc8),
        (0x38f6136241cc3637, 0xbc64eecbf0b91d4f),
        (0xb8d2eab985ecf43c, 0xbc356ab697c0ab6d),
        (0x387327e90134d3b9, 0x3bdd4506fe436113),
        (0xb837e66ef2a0608f, 0x3ba925d81a8dc397),
        (0xb7f15df323201ec3, 0xbb50a89d96a14715),
        (0xb7bfc8950449ed4b, 0xbb18875d24cf9582),
        (0xb754b1be99b10be3, 0x3abf7e546e402e74),
        (0x372718433d137dec, 0x3a84326527427bb9),
        (0xb68468a05b2cb0da, 0xba2921298884b92b),
        (0x367c4fc25beb34a3, 0xb9ec75a8ffac9550),
    ],
    [
        (0xbc6d2f0105f3ce7c, 0x3fc62d94d97e859c),
        (0x0000000000000000, 0x0000000000000000),
        (0x3c5d2f0105f3ce7c, 0xbfb62d94d97e859c),
        (0xbbf6c13bde837960, 0x3f565481b55eaefb),
        (0x3c17013075a066f9, 0x3f7d5f857a2a6107),
        (0x3bc5f065aa6339f5, 0xbf21a015a2ccb78f),
        (0xbbc12f349174f010, 0xbf2eeb4eafd8614b),
        (0xbb53525d6141be28, 0x3ed3a7b79e4d80fc),
        (0xbb662d899f08a704, 0x3ed154ed4598d2f0),
        (0xbb0444753ebe791d, 0xbe767f762ea293a1),
        (0xbafe11bb6f1d4207, 0xbe680ec1e042ee62),
        (0x3aa725a667375b0d, 0x3e0f529652b9de0f),
        (0xba97b074118bab16, 0x3df6ad7bd2247f55),
        (0x39e3900c76e6a68e, 0xbd9d4e582fb43553),
        (0x3a1f5fd82381bd59, 0xbd7eeb6efcbb6228),
        (0x39c7389b8d7fe521, 0x3d23b1eabb5c1c2c),
        (0x3992974c386cdc2a, 0x3cffe92f0882b440),
        (0xb923c49452353597, 0xbca3f20671568bca),
        (0xb9002a4da3d3a816, 0xbc79ce30ac166eb5),
        (0x38b02f60dd891164, 0x3c1f8db6e94f1a96),
        (0x383459153c91e395, 0x3bf0cc90c0607e7d),
        (0x383003b333010a99, 0xbb940c6bb9f9a6c4),
        (0x37d6a317d4f5ae21, 0xbb61fe382b9ca927),
        (0xb76dfd62097696b0, 0x3b04ed907b4ff4b6),
        (0xb75abcbe1108a215, 0x3ad0234a19150d32),
        (0xb71105a3cea0d529, 0xba7246b54dbf759d),
        (0xb6cf812a75ac4069, 0xba3899f3eb5b7cf1),
        (0x36688dddcd012369, 0x39db1cda80386890),
    ],
    [
        (0x0000000000000000, 0x0000000000000000),
        (0x3c5fe2103f7148bb, 0xbfc5664d37c37d7b),
        (0xbc0a0fbbe8bcf9a6, 0x3f6e0ee8ec84659a),
        (0xbc383a77d074bd0a, 0x3f9c6c415c971b4b),
        (0xbbb06fe5233f06e8, 0xbf43ec49cb941f55),
        (0x3bfe14142e1727fe, 0xbf56853b2d047885),
        (0xbb98cea461592c41, 0x3eff632a9396fd4e),
        (0xbba049885559b2a5, 0x3f00e6afa01af561),
        (0x3b452a8549984c17, 0xbea7541247acdf59),
        (0x3b3a7242d1da1ee5, 0xbe9d735e8beb8fcc),
        (0xbae93d2eb398218a, 0x3e440e6a7cafc096),
        (0x3acae709ad00bd19, 0x3e30b9cfae2ab337),
        (0x3a72d6de94168a75, 0xbdd668087e67279b),
        (0x3a5eab8fef3b6421, 0xbdbab498e6f69c73),
        (0x3a0c8ec8094d97ed, 0x3d618b5674894afb),
        (0xb9db00611b51a32b, 0x3d3f99da023c7232),
        (0x3971efb1b8385b20, 0xbce450b8165c264c),
        (0xb934a2a8921ee27a, 0xbcbcd387688c508f),
        (0x390195709783e7c1, 0x3c621a103bd0a113),
        (0x38dbfdcea4f39167, 0x3c34e540faee9f3c),
        (0xb87ade2a66df50ab, 0xbbd999b3cf947860),
        (0x3847c3fb15ff4625, 0xbba8a8f1c18bd090),
        (0xb7c279575afc11de, 0x3b4d71c32bb0428b),
        (0x3797f3b19c608f42, 0x3b18289deefa7196),
        (0x3754153d9d2c7699, 0xbabc183d32d93a1f),
        (0x371c05c0e3242945, 0xba83f7ae467ae529),
        (0xb69ad19453e29618, 0x3a269b7adc6804ef),
        (0x367130005d91c163, 0x39ec3a580d0009a9),
    ],
    [
        (0xbc5e9088e9ff2518, 0xbfc4b2a38f1ab9b4),
        (0x0000000000000000, 0x0000000000000000),
        (0x3c4e9088e9ff2519, 0x3fb4b2a38f1ab9b4),
        (0xbbfcc8083276a193, 0xbf5225a5c73f2233),
        (0x3c1525fe797e2126, 0xbf7b750d89a9b35f),
        (0xbbb08db59499cb94, 0x3f1cbdb4f1d5dbbd),
        (0x3bc35c23938c810e, 0x3f2cfe933fc6d0b8),
        (0x3b52d453a3ea26cd, 0xbed01ef10d839bbc),
        (0x3b71cffb5ca00bc2, 0xbed05375a588a72b),
        (0xbb03f71b77aa7082, 0x3e729afe7ea7ba43),
        (0x3af336f1341f6d84, 0x3e66c8fe015e6610),
        (0x3aa95a6cddd6ccb3, 0xbe0a2a01e7af0241),
        (0xba7851970a0eac4f, 0xbdf59b33050d2092),
        (0xba24b1969b00bcf7, 0x3d98c0a50e7852d3),
        (0x3a16ee76e659ed58, 0x3d7da3426f355690),
        (0xb9cdfb48abef1f45, 0xbd20d3b48429ce84),
        (0x399e510926654c46, 0xbcfec46e14cd5d9e),
        (0xb94d8b3c4e2a8910, 0x3ca13c3da7405de9),
        (0xb9199ffbceedc2fd, 0x3c79043b4a0f03df),
        (0xb8b7be40ca4f6a7f, 0xbc1b902f4b26eb03),
        (0x38971ae8f878f8af, 0xbbf05dbe75075aa3),
        (0xb834ec73bf06d543, 0x3b91b0f5dc51ac20),
        (0x37f5aae0017b0662, 0x3b619b551575f49b),
        (0xb799e67f64663319, 0xbb02a3e0a7397c83),
        (0x376711f24f4773dc, 0xbacfb4a185898d2d),
        (0x371e5adab87fe639, 0x3a706acbac6f6bd1),
        (0xb694af13fd9e3bbb, 0x3a383fb94408a8c0),
        (0x367e1f0c9bfb4eba, 0xb9d88b17eec4fb7c),
    ],
    [
        (0x0000000000000000, 0x0000000000000000),
        (0x3c5f3474ffad3fd7, 0x3fc40f8ffdf09a5f),
        (0x3c047810b9792b13, 0xbf68c37a29c4586f),
        (0xbc348471cc77dfe2, 0xbf9aab099314b209),
        (0x3be6aa80fcd045e0, 0x3f406f735cc0f6a6),
        (0x3bebe02c3928856d, 0x3f552d29a06802e0),
        (0x3b89fdf10253d75f, 0xbef9fc04c675c0ed),
        (0xbb62f9306dab684a, 0xbeffe48825ed3c8e),
        (0x3b43251677276142, 0x3ea36bd2d58ff45b),
        (0xbb2af08b6d9ffd1a, 0x3e9be87e2cad2ce7),
        (0xbaec807c6c103803, 0xbe40d1fdedc6cb17),
        (0xba7064dd3b6b5724, 0xbe2fdbeb107ddccf),
        (0x3a75893a2f37955a, 0x3dd2f44d01321684),
        (0xba4232fa9e2160b9, 0x3db9921373ea0f26),
        (0xb9da39ce7fd7d9d8, 0xbd5df6b83ac56981),
        (0x39db340c9865a7cd, 0xbd3e6c005b62bffa),
        (0xb9708387feb998a6, 0x3ce1848c91de7162),
        (0x39532ac00e2fa8ff, 0x3cbbe5530d0fcf1e),
        (0xb8f7971717114a0f, 0xbc5f8540b8b06042),
        (0xb8d7787ecd6e687d, 0xbc3451eb9bf040e8),
        (0xb8696fbf3607765c, 0x3bd67f1edd448664),
        (0xb84ad321b52d14f9, 0x3ba8169212a3f8c0),
        (0x37d9b42cb89199fa, 0xbb4a1a0e01aa07e3),
        (0x37a4d1d59a34de84, 0xbb17b199c8e398ae),
        (0x37565ae964dca96e, 0x3ab91bd224db8139),
        (0xb71f2cb70464cd53, 0x3a83a756c81f58e6),
        (0x36a634d3a7864fac, 0xba245b460d9907d1),
        (0xb68d4c3a3fc04b15, 0xb9ebdf60ca34bcd7),
    ],
    [
        (0x3c4997782859a00d, 0x3fc37aaceac987b9),
        (0x0000000000000000, 0x0000000000000000),
        (0xbc3997782859a00d, 0xbfb37aaceac987b9),
        (0xbbd05a6eb7d72f47, 0x3f4e3fdbfd65014a),
        (0x3c1d91e80a0529b8, 0x3f79de7a33bc3a97),
        (0xbbb56ef10c1ebaad, 0xbf1801d911fbd06c),
        (0x3bcce92f388f20ea, 0xbf2b605a5ade3a80),
        (0xbb539bcd81dca55a, 0x3ecb0a2608144a3d),
        (0xbb6b3675770103a2, 0x3eceeceb341ad833),
        (0xbaf6cb1030653935, 0xbe6f623fc7be9f9f),
        (0xbb0412babffea489, 0xbe65a9ca94a0d7dd),
        (0x3a9025c6eebf2e96, 0x3e0638ba2f5f5e6b),
        (0xba745bdb1a5b34d9, 0x3df4a1b4217ef864),
        (0xba32b235688fed51, 0xbd9530c712e738d4),
        (0xba0b5b22087c168f, 0xbd7c6e5208e89146),
        (0x39aabbad8d2c58ef, 0x3d1d0eb5303a804a),
        (0x3988eb799dd68c33, 0x3cfda6ae03deea14),
        (0xb91737db06df4072, 0xbc9e05ef8aaeac6f),
        (0xb8ffe2eecdfe0f4b, 0xbc783796c23b134e),
        (0x38836225e752dac9, 0x3c183703be1e065f),
        (0xb87377273654d7a7, 0x3befd20ef0b9b32e),
        (0x381f98f1f5615157, 0xbb8f58105c7413a4),
        (0xb7f08f641ff4b998, 0xbb612ee75ce95b31),
        (0x37ac304030a2dba7, 0x3b00a4adc945e963),
        (0xb760c6584778a1df, 0x3acf0d914ecfdc7d),
        (0x36e07d1a751a4b9c, 0xba6d88af68997ed9),
        (0x36db1ad813072ac7, 0xba37d36ec5b8b8d2),
        (0x3672facc8520936e, 0x39d63a55dc5490c7),
    ],
    [
        (0x0000000000000000, 0x0000000000000000),
        (0xbc569479644686c1, 0xbfc2f206e49909c7),
        (0x3c0fab1b83d46057, 0x3f64dbf6a9fb80d7),
        (0xbc32d9141794e6dc, 0x3f99336443318ed1),
        (0xbbd0d543bfdf0729, 0xbf3bb6aa3d4e9e78),
        (0x3bebe3f68704392e, 0xbf540aaa5d94bd8d),
        (0x3b419d1c00b3671e, 0x3ef5f61b666129a6),
        (0x3b7d88846cdeeaef, 0x3efe4158391f2c2b),
        (0x3b46369cf5ad9126, 0xbea07a7a0745f74a),
        (0xbaff9b34051c76b6, 0xbe9a8ea97b670057),
        (0xbac6b3f27d760998, 0x3e3cb10b3affeaff),
        (0x3a82ba28da6d4ef2, 0x3e2e6d78879f98a1),
        (0xba64a30357ebbef6, 0xbdd044447904054e),
        (0xba47c38fca3d087e, 0xbdb8850c99b76d50),
        (0x39f0bc2d92a30cdc, 0x3d59e4fa8ff52f90),
        (0x39da7eb9dc7c3857, 0x3d3d4b2c1afd964f),
        (0x397e3e8ba2460cb7, 0xbcde807e2b0027a1),
        (0x394f139f76a41235, 0xbcbaf942a59e1d55),
        (0x38e1843405416d9e, 0x3c5ba64c4108b5da),
        (0x38d7063d60c6e606, 0x3c33ba614d4a4418),
        (0xb86c7ec67073bb42, 0xbbd3e1ce585ffa1f),
        (0x37e22b5fc552851b, 0xbba779ee6dc176e6),
        (0x37e956c614ab6059, 0x3b473c7bc63c3880),
        (0xb7b796c186b6db87, 0x3b172ca465caf995),
        (0x37588aab872e21fa, 0xbab681e73200f9aa),
        (0x37260574a8f5d812, 0xba83492c838e2b86),
        (0x36bb446923624df9, 0x3a225df78223620b),
        (0xb680e905ae135da7, 0x39eb6eaa1f93591f),
    ],
    [
        (0x3c6b7326e3fbaa70, 0xbfc2740819f1caaa),
        (0x0000000000000000, 0x0000000000000000),
        (0xbc5b7326e3fbaa70, 0x3fb2740819f1caaa),
        (0xbbe4b7a0f7878f6f, 0xbf49b6f37d0a00f9),
        (0xbc011249d3675d98, 0xbf78868d7401bf2e),
        (0x3bb6d7fdfc21dddd, 0x3f1470a7cbcb436a),
        (0xbbc5a6b64d367c4b, 0x3f29fe94ce3d3e83),
        (0xbb5352a68b13573e, 0xbec71660e61f0f0c),
        (0xbb68d34686b724d9, 0xbecd6dfcdb026028),
        (0x3af006603a027d68, 0x3e6ae79c35d01bca),
        (0xbb0ef1a3dfdf0941, 0x3e64ac50be6ca02e),
        (0x3a8def2f9646d0a2, 0xbe03258409210f45),
        (0x3a9043cd5dc1ca57, 0xbdf3c09685c05e25),
        (0x3a140fbcac25067c, 0x3d925d85992670e2),
        (0x3a184f1121e2c362, 0x3d7b509288f452b8),
        (0x39b4b4c230260136, 0xbd19586d0c8690d8),
        (0x39776fb98c811c8b, 0xbcfc973a175ffbb6),
        (0xb9337afd9ae7c0eb, 0x3c9a5d61717a7033),
        (0x3907444e5980ecc1, 0x3c776fa9ee1a429f),
        (0x38bdaeb8edb9cbf2, 0xbc156907cfeb69d8),
        (0xb8705ec9beaf1eac, 0xbbeee74de5fce6e5),
        (0x382fdb3ba7577f93, 0x3b8be6f4a0664634),
        (0xb80c67b2a3f6b8fa, 0x3b60be8c60df72fe),
        (0xb7980c2fe23edbd8, 0xbafdd3c10be941a0),
        (0x376107929e229c71, 0xbace5afc740d98d1),
        (0xb6ffc39670c4b338, 0x3a6aa1c12cf8bc0c),
        (0x36bf0bd26da1ffcc, 0x3a375bd104e008ce),
        (0x3671519a4046ccc5, 0xb9d429e651c05e27),
    ],
    [
        (0x0000000000000000, 0x0000000000000000),
        (0xbc6be2029a752b31, 0x3fc1ff5ebddd3c3a),
        (0xbc04bba52e05ef56, 0xbf61e2035324643c),
        (0xbc3a42d5bcad204f, 0xbf97f3506d4a1231),
        (0xbbd8ddfdbbecb379, 0x3f37c65c9302c53b),
        (0xbbff79e837d220e1, 0x3f53117816335151),
        (0x3b97f81238bbf061, 0xbef2df9afa521294),
        (0xbb9270fec55a19ce, 0xbefcd5d4a9d78a14),
        (0xbb3a3cd8b5652e44, 0x3e9c672d4d6f7766),
        (0xbb2736243a1eeac9, 0x3e995e4b102194f0),
        (0xbad74d87610e5d51, 0xbe38d22636140e12),
        (0x3acf7d99cf14d15d, 0xbe2d24fc9458bf7f),
        (0xba609b77f08fc036, 0x3dcc462d9da63971),
        (0x3a5ec10df895da79, 0x3db78eafb15964df),
        (0xb9d9e108200b834c, 0xbd569f5d498b660b),
        (0x39c6a15cc2204e9d, 0xbd3c3c4e5b0fab49),
        (0xb9732871d1bbe384, 0x3cdaccaf11834f8d),
        (0xb93cb3c7ca476321, 0x3cba161c63abb182),
        (0xb8e7d3af6ae5f8f7, 0xbc5870224af67eb4),
        (0xb8dc947a17c950f7, 0xbc3324870c0b9f1a),
        (0x3835837498a32b18, 0x3bd1adca41a58439),
        (0x384e7a933bc23443, 0x3ba6da83c46ff8a1),
        (0xb7e2d9939a6b62de, 0xbb44c90654b3ecef),
        (0x37b4d48249a8211b, 0xbb16a125665039e4),
        (0x3751c4fab8892f89, 0x3ab4403fbdd7d32b),
        (0xb723bebaa05fe118, 0x3a82e31c9f4289aa),
        (0x36be4670338eb907, 0xba209e5b516e3eb9),
        (0x36636814b89df314, 0xb9eaf0173952c775),
    ],
    [
        (0xbc4081c2a50ad27b, 0x3fc192f2627a74e3),
        (0x0000000000000000, 0x0000000000000000),
        (0x3c3081c2a50ad27b, 0xbfb192f2627a74e3),
        (0xbbed88bdcc1e7819, 0x3f4635b38affe698),
        (0x3bfbe434e30d7a64, 0x3f775eceaabf7f86),
        (0x3b9ca8671aa55df6, 0xbf11ac9e0164f7c1),
        (0x3bc6ba93b83783a0, 0xbf28cc464a35b0af),
        (0xbb6979a9d3cf377f, 0x3ec4014d9bf389c3),
        (0xbb64740ce7aaa9f5, 0x3ecc1f05a2d85165),
        (0x3b05322df74ff9fc, 0xbe6761d18ebb04af),
        (0xbb0eb4591c8e1e8d, 0xbe63cb9af103e0f5),
        (0x3aa091624db08269, 0x3e00b3cb55bbe5d2),
        (0xba9e3e54a390f0cd, 0x3df2f5c763b33667),
        (0x3a315b71b511ed17, 0xbd9017771db5a5e9),
        (0xba14beedd13cfb94, 0xbd7a4a5e79f39fb9),
        (0x39987da2b8e5b9fc, 0x3d165128cc4d99cc),
        (0xb995477d64dc86b2, 0x3cfb991575a9daff),
        (0x391a0c68aab7bf20, 0xbc9756473a57c932),
        (0x391f7919e47d81ac, 0xbc76b070fece822a),
        (0x38aea8dff65f4d33, 0x3c130e1f9f1562ad),
        (0xb885307e292801c8, 0x3bee01b29bf38fbe),
        (0x380c55e014e1fd44, 0xbb88f8681f03bf7f),
        (0xb7a97a5de24d257a, 0xbb604e16b9d18563),
        (0xb79be9ebe189653a, 0x3afad6e6ca7adff4),
        (0xb75781440aba92bc, 0x3acda3ed85696d1e),
        (0x36e87609f3833ad8, 0xba6817ad40ddad7a),
        (0xb6d43ee7c83c85cd, 0xba36de22ba99ddd2),
        (0x3644163dc4acb5e3, 0x39d255cee2696246),
    ],
    [
        (0x0000000000000000, 0x0000000000000000),
        (0xbc4e7aa4db2a788a, 0xbfc12dd55d4be2b3),
        (0xbbf6da5fa7c968b2, 0x3f5f1aee31818d19),
        (0xbc319e43d5f0f1d5, 0x3f96de64242a8310),
        (0xbbd5e5667369987f, 0xbf34afdf89fca61a),
        (0x3bd951e5980d1eeb, 0xbf5238cfc13ac771),
        (0x3b9632bd2bd2db1d, 0x3ef0719d13e00e52),
        (0x3b672b1f290673a0, 0x3efb974781a526b4),
        (0xbb1d4aa2a151d3bc, 0xbe98cc82a70d752f),
        (0xbb3bd18c494b86c6, 0xbe9850ae878c25bb),
        (0x3ad9b4f29c25b035, 0x3e35bba73e282ede),
        (0x3ac284476c98f69a, 0x3e2bfe1396e83644),
        (0x3a4529f3c858a5ed, 0xbdc8d7dbe4a241ab),
        (0xba562391a18ff729, 0xbdb6adfbd773748d),
        (0xb9fff61487742711, 0x3d53f51dcb398ad3),
        (0xb9d0d5d8e6a35200, 0x3d3b40d30be85f3d),
        (0xb97f65625ced3453, 0xbcd7bf544872d9cd),
        (0xb9476385a2a2b1f8, 0xbcb93f163a09379d),
        (0x38f06b1beee10cee, 0x3c55c1ca7d35fe43),
        (0x38b8eeb49079a75a, 0x3c3293b43a786b37),
        (0xb854b85c8c7227d3, 0xbbcfa1df0227f182),
        (0xb80a3744049cba08, 0xbba63d0b33ebfae1),
        (0xb7cca2e89550a07c, 0x3b42afdbba3b1d6f),
        (0xb799806c5d3ae5db, 0x3b161427fedb2614),
        (0x374631901a8ab5ef, 0xbab24bb69b617f29),
        (0x3706634bb0b1f240, 0xba827970d7fd5a93),
        (0x369d62ccd478840e, 0x3a1e2c0be398bf2f),
        (0xb676a35ad46d678c, 0x39ea69a77fc2c19d),
    ],
    [
        (0x3c60c06e2860e868, 0xbfc0cf3ee98f769b),
        (0x0000000000000000, 0x0000000000000000),
        (0xbc50c06e2860e868, 0x3fb0cf3ee98f769b),
        (0xbbb4697a99d5fadb, 0xbf436f451f6e14fb),
        (0x3c194225ed089995, 0xbf765d05948a946a),
        (0x3b91ed6777d1acf2, 0x3f0ef55c5a0d16cd),
        (0x3bcf058a2a3a8b0d, 0x3f27bfec9d15d038),
        (0x3b563856fcc3c915, 0xbec18c549f28e528),
        (0x3b59158b4542d2c4, 0xbecaf7544eeac766),
        (0x3af5145c85b8d71b, 0x3e648e81edf38b1d),
        (0x3afbd8241fda0445, 0x3e630341e78d1014),
        (0xba51c1f8c39d84cc, 0xbdfd73d2c942e970),
        (0xba8a04a03e6e4b0a, 0xbdf23ec2729d3a6d),
        (0x3a229664fb891b84, 0x3d8c78e44cdec8b3),
        (0x39dac0cbab30eb59, 0x3d795a6f6107b4e6),
        (0xb9ba116bfc433ee0, 0xbd13d1c77de8a03d),
        (0x399e9b0cd3f3a20f, 0xbcfaace944141c61),
        (0x393126e2b3ac5379, 0x3c94cfcb5e84963e),
        (0x390ce3384ec723cf, 0x3c75fbc3b0ec9599),
        (0xb8a8a1431d96f0b6, 0xbc11115c48c0688d),
        (0x385d6ff29b2308fa, 0xbbed24eb2fb8425b),
        (0xb825e6a6df55bdde, 0x3b8677909ec26485),
        (0xb7c21d072a134e9c, 0x3b5fbfed070231f3),
        (0xb797829f30055e39, 0xbaf8423b3340403e),
        (0xb75751d2373f6e38, 0xbacced4275822f9a),
        (0x370ed561d6ebcce7, 0x3a65dfcd7ab2578d),
        (0x36cf53856f82b22b, 0x3a365e43730aa685),
        (0xb61c3bb3f92eb84f, 0xb9d0b87da5a9b4de),
    ],
    [
        (0x0000000000000000, 0x0000000000000000),
        (0xbc64353fd6c42f1d, 0x3fc0768257dad56a),
        (0x3bf3a4186dcffc49, 0xbf5b602a7beaaa48),
        (0x3c16aaef36d826d8, 0xbf95ebc22efd092c),
        (0xbb76c80c9b070fe3, 0x3f3236a604142e61),
        (0x3beddcf66c5978ce, 0x3f517a482faa8d85),
        (0x3b66ad649af93743, 0xbeecfc00890787ae),
        (0x3b928f6dcddab568, 0xbefa7d7b128ac538),
        (0x3b25a6c98552063d, 0x3e95e419f1b57043),
        (0xbaf32b6c630ee4fa, 0x3e97603cea235244),
        (0xbad9cee30553553d, 0xbe333a0b436c8824),
        (0x3abe7890b4a4106e, 0xbe2af46417845f40),
        (0x3a4f7394a3298f16, 0x3dc609ac8cd8177a),
        (0xba59cd5f345dc226, 0x3db5e11b87ad3903),
        (0x39f73ed5597f6f80, 0xbd51c2756590391a),
        (0x399986e8c9348fca, 0xbd3a586483dd891d),
        (0xb97cc034606fd4e1, 0x3cd534d22c600964),
        (0xb957a2d70318a209, 0x3cb8755ad322c506),
        (0x38e4680effc80bb4, 0xbc53809e528c6189),
        (0x38d5dd4c01f7d4c2, 0xbc3209a0a78d682e),
        (0xb8644847850e1cab, 0x3bcc77842d1f731c),
        (0x384c2f3bd9e41480, 0x3ba5a4538092f65b),
        (0x37efb4bb53ec98f4, 0xbb40e2a86e6cb3f9),
        (0x377521ae7e540c78, 0xbb1588f3efb4ad64),
        (0x375fdbdc3816089b, 0x3ab099874d761b53),
        (0x37298a0b16897a8b, 0x3a820f23922028f2),
        (0x36b4686e1a4ffa47, 0xba1b7c94063b4326),
        (0xb682cf52edcbf7b1, 0xb9e9dfbe640c7d02),
    ],
    [
        (0x3c61166b7995967a, 0x3fc0230ba90f2871),
        (0x0000000000000000, 0x0000000000000000),
        (0xbc51166b7995967a, 0xbfb0230ba90f2871),
        (0x3be85d0df08ff853, 0x3f413164a0864cde),
        (0x3c175b99fcae0fa4, 0x3f7579c1bdbcfc99),
        (0xbbaa847d3261af11, 0xbf0b67e1913c668b),
        (0xbbbdc3b0bfb76e70, 0xbf26d26de4fd8c5d),
        (0x3b509cffcaa6fd75, 0x3ebf1b520b063853),
        (0x3b56e93d7c515f0c, 0x3ec9f01e7c1909a0),
        (0x3aeb43dee8b56e57, 0xbe624071b1796027),
        (0x3a99b4c0bd0ff4a5, 0xbe624f8e939ce43c),
        (0xba9ac0bb7d806efd, 0x3dfa35663595463f),
        (0xba625661d1f74770, 0x3df199120d49d507),
        (0x3a1bddfd9246953c, 0xbd896771e1b15e44),
        (0x3a1acb7638900b33, 0xbd787ede134ed7a0),
        (0x39a9e2e9004bba4f, 0x3d11bccdfbafa1fd),
        (0xb995262f16494ab9, 0x3cf9d22a6dc5a57c),
        (0x393e65a43ea6fde8, 0xbc92b00e367301bf),
        (0xb8e38ef38ce7f9e3, 0xbc755236722b3018),
        (0xb8a085f6fce94877, 0x3c0ec2ce49a94662),
        (0x38851022a9daf9df, 0x3bec52d2fb48c061),
        (0xb8260ec09f1c11bc, 0xbb84522e399fa43a),
        (0x37d6e8cbd63da93e, 0xbb5eeb3ee8f96a17),
        (0xb780414b4648c1d1, 0x3af60651f3414a8f),
        (0xb75a1e55435fe75b, 0x3acc3a2c8e72fa37),
        (0x37034c532c4c52e0, 0xba63ef7c46a9be7d),
        (0xb6d60fee2f339275, 0xba35deebe060dd5c),
        (0x366c4892cbf5f446, 0x39ce97bce469d7a5),
    ],
    [
        (0x0000000000000000, 0x0000000000000000),
        (0x3c10417847765c1a, 0xbfbfa8b3f9ae4375),
        (0xbbcf334b3619566d, 0x3f5856073b7fa2cd),
        (0x3c3664c698368b18, 0x3f9514e652eb2e96),
        (0xbbb93b53ec827565, 0xbf3032298718ea1a),
        (0x3bfdfbf103edc8e3, 0xbf50d1153fde431b),
        (0x3b7c365adad42a66, 0x3ee9cb455c29d6a7),
        (0xbb9fa9b04c310c2c, 0x3ef982167b1c52c3),
        (0x3b3466b707822fa7, 0xbe938191ef68eab3),
        (0x3b31f4e7061400c0, 0xbe968865345b3130),
        (0x3adaf50c9fd62c74, 0x3e312960010da387),
        (0xbace36911ce663a9, 0x3e2a0403679f6882),
        (0x3a4177820d3b2b22, 0xbdc3b67cc0b96c06),
        (0xba565f2ed3c28fcc, 0xbdb526038b956efd),
        (0xb9d5a1b33bcbb16b, 0x3d4fda06e840fa34),
        (0x39d1d67f1bd55539, 0x3d3981de640d0e00),
        (0x397c94419a95b1f3, 0xbcd311fc56c4b88a),
        (0x39532577b4bca3e2, 0xbcb7b8f62899fc83),
        (0xb8f552faebfd4704, 0x3c51977cb27410a2),
        (0x38d26b959afd2d20, 0x3c318706eb31dd46),
        (0x386ce90ca03d9538, 0xbbc9c2ecf35b9e14),
        (0x384b179344e2eb73, 0xbba511e7341e8837),
        (0xb7dba680290d429b, 0x3b3eaa190aa91ed0),
        (0xb7b49a491a85f98b, 0x3b1501910205c4a9),
        (0xb74ebc3f994c58ab, 0xbaae3fe1d152dac7),
        (0xb72316bcea29e953, 0xba81a632639f368e),
        (0xb698c03bf571184d, 0x3a192191e962f593),
        (0xb68c49d86016d888, 0x39e95576c2d368f1),
    ],
    [
        (0x3c50db2c50623ec0, 0xbfbf13fb0c0e6fcd),
        (0x0000000000000000, 0x0000000000000000),
        (0xbc40db2c50623ec0, 0x3faf13fb0c0e6fcd),
        (0xbbddaa7169324a5a, 0xbf3eb3e6fcc47c00),
        (0xbc1a7c91ef9a7da1, 0xbf74af74cbd77bef),
        (0xbb8ca4e36c0e29e8, 0x3f087bb1ebeaec75),
        (0x3bc0dc1e11076c62, 0x3f25fe629203150e),
        (0xbb5990b5b348875c, 0xbebbd0f2a6555e6b),
        (0x3b6af9c37c81d002, 0xbec9040de830649e),
        (0x3ae759c4b27cdf74, 0x3e6057f7a76993d7),
        (0xbaf1d65071810a7a, 0x3e61ad6dd5105c3e),
        (0x3a9d40eb62d35800, 0xbdf782a6f5738cdb),
        (0x3a8bf5df6e6d389e, 0xbdf1027dc06d4453),
        (0xba28e41b1a5587d6, 0x3d86d65b13b16f65),
        (0xb9fc75d3c8772d41, 0x3d77b59e9ed7367d),
        (0xb9af754c04eface9, 0xbd0ff800f2a49a00),
        (0xb991f98cd4e6ac14, 0xbcf907be43054aec),
        (0xb933d033de8a73d1, 0x3c90e2e1d8f5ad67),
        (0x38eed8fb6a3c50a9, 0x3c74b3a8b0fb5259),
        (0x388913f9676a0ac9, 0xbc0be06d96b6ddb7),
        (0xb88ea864297f3fc7, 0xbbeb8c227343f289),
        (0xb82ff7b2b57b5685, 0x3b8278dcce458371),
        (0xb7f78beca367a3df, 0x3b5e1fb52b874a03),
        (0x379b7478eaac89e3, 0xbaf41590cb708f7e),
        (0xb752fe7b149fbb7c, 0xbacb8ca63f91e577),
        (0xb70548503aed227a, 0x3a623cdbe3c71609),
        (0x36c12f6cf019ff77, 0x3a3561f0d31660bb),
        (0x365d1dc388e61bc9, 0xb9cc13f7faffaf26),
    ],
    [
        (0x0000000000000000, 0x0000000000000000),
        (0x3c56a7c2ed8fa844, 0x3fbe8727c572a2c2),
        (0x3bfdee3ce04b3d39, 0xbf55d1ef092ab395),
        (0x3c206dda20eba87a, 0xbf9454e7a7395636),
        (0x3bb99eb39ffae4f4, 0x3f2d0cc3a7fa6d3d),
        (0xbbc9d1e5b54f9af7, 0x3f50398d2cbd02df),
        (0x3b8aa1f4bf33efdb, 0xbee725b0909f9c66),
        (0xbb9b20d87a760067, 0xbef8a022f5fee447),
        (0xbb1fcb0c42b0e854, 0x3e9185b1ea97a54b),
        (0x3b3ea0aa1deb55ca, 0x3e95c571167401ae),
        (0x3acd2759cfccc15b, 0xbe2edff6514e1da9),
        (0x3ab2592f6b4b9153, 0xbe29298da3c4fa71),
        (0xba62158d0945cb91, 0x3dc1c34705eadfb1),
        (0xba424ede84d84b4b, 0x3db47ab8ec529c51),
        (0xb9e73e83248604d7, 0xbd4cc24a2f007c3e),
        (0xb9c2979e3d03632b, 0xbd38bbcc44e6c225),
        (0x39703e1bd29b296c, 0x3cd1420df9bb4359),
        (0x395d047d716a7a8d, 0x3cb7095f8b3266bd),
        (0xb8cd11469be08694, 0xbc4feb87e4e655cc),
        (0x38be1dd0526935fa, 0xbc310c0caf4ba7a9),
        (0x385a60dd5e84c9c5, 0x3bc76f4ca221cf39),
        (0xb83ecbabc0dd7d07, 0x3ba4868177e6f5ef),
        (0x37b8847e584a462d, 0xbb3bf93dc497bf28),
        (0xb7bb4e77466271a4, 0xbb147f2c424e92f5),
        (0x374a584169fadea4, 0x3aabace093fa5e7e),
        (0x370af91e2a607b95, 0x3a813fe3dbef7300),
        (0x36a10a818d4db7be, 0xba170f75c4345a45),
        (0xb6269b19813edb68, 0xb9e8ccf104c5a0ca),
    ],
    [
        (0xbc3b9f1d130797af, 0x3fbe018dac1c17e3),
        (0x0000000000000000, 0x0000000000000000),
        (0x3c2b9f1d130797af, 0xbfae018dac1c17e3),
        (0xbbc5edf62333beba, 0x3f3ba21bd15d02a4),
        (0x3c138b4f0d3a4110, 0x3f73f9e0db07e7ef),
        (0xbba6f23ddcae2db1, 0xbf060b77c5e27622),
        (0x3ba86d4e04faa212, 0xbf253f9b1a5d228c),
        (0xbb5fddc005e382a4, 0x3eb910b38812c253),
        (0xbb40935ad53575b8, 0x3ec82ee6dfdfedeb),
        (0x3afa8bf307790c80, 0xbe5d7cc2a9a602ee),
        (0x3b0b50784ff133f1, 0xbe611a57d8645358),
        (0x3a8673cb487ec434, 0x3df53d401519442d),
        (0xba8c03600ffb7a70, 0x3df07915ef87408d),
        (0x3a2022a5f15014b6, 0xbd84aabcb6141c76),
        (0xba1d37109cce2377, 0xbd76fcb8e2eada15),
        (0xb9a95525555a3484, 0x3d0cfcfdb2faeb02),
        (0x39997fd3cf0000fa, 0x3cf84c5369d2e33d),
        (0x38f88b42ac1a9330, 0xbc8eb11d7e6581b7),
        (0x39181a3f66a29421, 0xbc741f9ab4632e3b),
        (0xb8abf00ddaa40351, 0x3c0965159407b5ab),
        (0x3885289d8a9dc178, 0x3bead0e3cd065e84),
        (0xb8220f36807a6153, 0xbb80dee0e94d3838),
        (0xb7e15def3c2d10d9, 0xbb5d5e035bd44cab),
        (0x379e09d7e4e2053c, 0x3af26452e5cf7372),
        (0x375c2b8feb4af0ba, 0x3acae5d15fef7e8f),
        (0x370c46b7285b4063, 0xba60bf2894ab39b9),
        (0x36bc8ff0c23e41b6, 0xba34e87ee39a6f39),
        (0x366a289544183384, 0x39c9daa655367c6d),
    ],
    [
        (0x0000000000000000, 0x0000000000000000),
        (0x3c555d27e18add4d, 0xbfbd82939ab62339),
        (0xbbe5f2055a671318, 0x3f53b5a54845670f),
        (0xbc331ee0af44ab91, 0x3f93a7ff1622def8),
        (0xbbb2001fd8691606, 0xbf2a3ebc476a606d),
        (0xbbb7a5c728534a7d, 0xbf4f61adde3a8c61),
        (0xbb6721b027970bc9, 0x3ee4ec45da0478b0),
        (0x3b60295953d04e19, 0x3ef7d3b28159f23c),
        (0x3b21dd48380f5e9a, 0xbe8fb3f21bcd7fc4),
        (0x3b3047a478eaf94e, 0xbe95145a876d2273),
        (0xbac575cc2ab01d93, 0x3e2bf64cdf7c0558),
        (0x3a9c7ea365ef1c0c, 0x3e2862204ac427d2),
        (0x3a25523941e42ede, 0xbdc01c0d48888811),
        (0x3a3766e8ee24e58a, 0xbdb3dd6cb1cbcd8f),
        (0x39e11ac6552bc64c, 0x3d4a2020c8b3c20d),
        (0x39bbaa3793fd39c4, 0x3d3804a9d711292e),
        (0xb965d9ecddfcbd4d, 0xbccf6a1ef9182e2c),
        (0xb95ee6d1970be0a7, 0xbcb665c82dfd3111),
        (0x38e2ccf8afc7f5eb, 0x3c4d1cbc474ae672),
        (0x38baa3fe632e7af8, 0x3c309883728ed041),
        (0x386cc2812a2e5111, 0xbbc56bda19c4b049),
        (0x3825988783a1ff06, 0xbba4025c7e87cb33),
        (0x3798e8afd688bd54, 0x3b39a17a3b2afb39),
        (0x37b0dd0e73f34a5f, 0x3b14026125d71009),
        (0xb73504f75a230f17, 0xbaa96b8155611361),
        (0x36fcaf6a0d9e7292, 0xba80dcfdcb67c11b),
        (0x363354ed9306d4a3, 0x3a153c10e4500766),
        (0xb6880389022204ec, 0x39e84794b22bad24),
    ],
    [
        (0xbc56edd809f4ec43, 0xbfbd09b21e36c0bd),
        (0x0000000000000000, 0x0000000000000000),
        (0x3c46edd809f4ec43, 0x3fad09b21e36c0bd),
        (0x3bd1da06fbc186d3, 0xbf390b0e6a475e45),
        (0xbc140975d5184af6, 0xbf7355b904fbf7ee),
        (0x3b9e3b983f117cfc, 0x3f03fc459d1e25a4),
        (0xbbb494c77a177873, 0x3f2492cc61d19dfe),
        (0x3b46367d4d299b08, 0xbeb6bcf110a02ad1),
        (0x3b6b6ea46d6db74f, 0xbec76d44f6a83523),
        (0x3a8030056b81d3a7, 0x3e5ac61efcb3c36c),
        (0x3afeb8462b17799c, 0x3e609436fcaa3965),
        (0xba99e294aeb8cd7b, 0xbdf34eb6095f969b),
        (0xba88db8194acb808, 0xbdeff662945eb368),
        (0xba237c1915fc9c0d, 0x3d82d05bf8ff5181),
        (0xb9d3e0b801e8758e, 0x3d76525fbb563fe0),
        (0xb9a734f42937bddb, 0xbd0a6eb937c7f6c0),
        (0x396073f8f8b180ec, 0xbcf79e90468589ab),
        (0xb922e469e5eb93e5, 0x3c8c0998b46c2a6d),
        (0x38badc48095c4e55, 0x3c73955fe5fb130b),
        (0xb8949ebb69e71ffd, 0xbc073ec897598aef),
        (0xb88f7564544e1430, 0xbbea20bd7b4d506a),
        (0xb7fb326fec2fdde6, 0x3b7ef3962c3d7988),
        (0x37e82dd0be2c1482, 0x3b5ca6507646e165),
        (0xb7879646e68dbcf8, 0xbaf0e8ca73f2d68d),
        (0xb74018d4e9ae2637, 0xbaca463d4098cc93),
        (0x36f61d9992deaded, 0x3a5edd93261617c3),
        (0xb6dbe7fd3fa3deaf, 0x3a34734a62443d4b),
        (0x3623d3bbf119c872, 0xb9c7e1ec2479954a),
    ],
    [
        (0x0000000000000000, 0x0000000000000000),
        (0x3c4e1f19f10295a1, 0x3fbc96700031f601),
        (0x3bd8339df4a4a968, 0xbf51eb2a07d0f09e),
        (0xbc39bfe53ec12c8a, 0xbf930b36eddaa234),
        (0xbbb206af4ba8d313, 0x3f27dd2dde84b73c),
        (0xbbd1f3a6f0dd2e56, 0x3f4e696553e0b8a5),
        (0x3b70bebcc672e081, 0xbee3085aa35a1647),
        (0x3b964955e62a56d5, 0xbef7199f24bccae1),
        (0x3b29acbd0629609d, 0x3e8cdbab6a766176),
        (0x3b336b635edd4912, 0x3e9472a7cffbf91e),
        (0xbaa919ffba04358e, 0xbe2979f5211c4dae),
        (0xbacb3a83d64d312f, 0xbe27ab4b01a902db),
        (0xba3d20c26289bb46, 0x3dbd63904e4c487c),
        (0x3a3a2263834e8f85, 0x3db34c83fb1c2279),
        (0x39d0d1629893175f, 0xbd47dcfdf13e5ccc),
        (0x39b38e8b7b2086d0, 0xbd375b01ddacf988),
        (0xb96c94f8a51700d8, 0x3cccbd41ea6afd29),
        (0xb9596f4635f2eb43, 0x3cb5cd46eaba1e98),
        (0xb8ef331fa6111e01, 0xbc4aaded22d1c6ea),
        (0xb8c316503d17378d, 0xbc302c0feb2cc9ff),
        (0xb86b0bb9dacdd402, 0x3bc3ab0efb0f5a6a),
        (0xb83c8dfa9ab81f86, 0x3ba38564ae73cbd4),
        (0xb7ce2bd40d6dd04f, 0xbb379494ee2fe052),
        (0xb7a9f50c26ffe35d, 0xbb138b6c6bd628fc),
        (0xb704cc1a22fd0a79, 0x3aa76f817567d54d),
        (0xb72b4eeb7ce6fd0f, 0x3a807ded69c2252a),
        (0x36b142adaa59897c, 0xba139e9549f763d8),
        (0xb6561aae4c08fc78, 0xb9e7c644065d040a),
    ],
    [
        (0xbc5ca34ef67ceca5, 0x3fbc2861347b1b39),
        (0x0000000000000000, 0x0000000000000000),
        (0x3c4ca34ef67ceca5, 0xbfac2861347b1b39),
        (0x3bbf201ae2302561, 0x3f36d57bffb37fd5),
        (0xbc16b4ecac2fa1c2, 0x3f72c060ef553f18),
        (0xbb9df91b90d34cd8, 0xbf023a407b722d2e),
        (0xbbca3432d200d5db, 0xbf23f55581ec683d),
        (0xbb5e04bafba5dc5a, 0x3eb4bfb4daa2ff62),
        (0xbb42a29e463f2261, 0x3ec6bc69099af2f7),
        (0xbafb52717e21f4f1, 0xbe5873340987d276),
        (0xbadb9738396340d1, 0xbe601951f02d2b37),
        (0x3a8c2da917c5d2d4, 0x3df1a6174fe55fbe),
        (0xba7265752e07ea23, 0x3def0eca5bc0284a),
        (0xba1b8d994d0c183a, 0xbd8137cb1fcc55a3),
        (0x39c1b1fb4f7e04be, 0xbd75b4f7f2bb19ed),
        (0x39961a8be6115ecd, 0x3d083934ed797abc),
        (0xb998209f68cee7a8, 0x3cf6fd29fcbcf38b),
        (0x38da8f6db73066be, 0xbc89bbe4a386b1fc),
        (0x39119e7a973ba60e, 0xbc73143b8f6fa9b5),
        (0xb8a42d280d3b7044, 0x3c055f1ef8c6277c),
        (0xb8741bc4d5b892a4, 0x3be97b209a81cdd1),
        (0x381f56010f66d563, 0xbb7c822a3d227a0b),
        (0xb7d326c5dc6baa73, 0xbb5bf86da968d5d8),
        (0x37685ec534d88737, 0x3aef3590b082abca),
        (0x376f5328f688841e, 0x3ac9ae18938ed4dc),
        (0x369b410a6e56968b, 0xba5c8a82558390d1),
        (0xb6c5a55f28288a0c, 0xba3402b3aeae6a78),
        (0x365785f4c242b76d, 0x39c621268afb8484),
    ],
    [
        (0x0000000000000000, 0x0000000000000000),
        (0xbc2986f8b543f277, 0xbfbbbf246019c0d4),
        (0xbbf8243dc68faf05, 0x3f506224199140d8),
        (0xbc31c8ab4f070de9, 0x3f927c3416c09898),
        (0x3bc1400990dffd7c, 0xbf25d279dc87cf69),
        (0xbba49f5dbb72a054, 0xbf4d86a5f5adbdac),
        (0x3b8c1a134d478d15, 0x3ee169041634c248),
        (0x3b908bd4df68b455, 0x3ef66f5bd1bedae1),
        (0x3b2340bc492ebacd, 0xbe8a69b2c717d54a),
        (0x3b2b327658085637, 0xbe93de4eab456cbd),
        (0xba8a52518b85c3bb, 0x3e2755eabbde6915),
        (0xbaceee9ff82c0693, 0x3e2702fe7ba9f8c3),
        (0xba380d66035d4ef3, 0xbdbaf2007eeae256),
        (0x3a5fcfcb227b90a8, 0xbdb2c69746948e55),
        (0x39e9a9de8d952335, 0x3d45e77ad9a9daf0),
        (0xb9d9fd6d3c1b2712, 0x3d36bd7bc6045cd0),
        (0x392032afd924a7d3, 0xbcca6a2a200a32c2),
        (0xb959d59bb0713bcd, 0xbcb53ef0001a8fc4),
        (0xb8ed4be5f4212c88, 0x3c488eb37fc1d400),
        (0xb89d39eabce00ec1, 0x3c2f8c832f51b562),
        (0x3840e097e8a92261, 0xbbc22200a03da113),
        (0x384c3af44c4ec552, 0xbba30f596dc1d400),
        (0xb7dc9bccb090c4a1, 0x3b35c6dd14718296),
        (0xb7bce5b1aace9f68, 0x3b131a4eaddd4e23),
        (0x3744f32babed55e0, 0xbaa5ae9b0fa87ab5),
        (0x372de16b15f9f82a, 0xba8022e41d997f41),
        (0x36b5867aa0456426, 0x3a122f7b20539f5a),
        (0x368433fa75fa367b, 0x39e74982e085fb06),
    ],
    [
        (0x3c497d2b9281abc8, 0xbfbb5a622198a72c),
        (0x0000000000000000, 0x0000000000000000),
        (0xbc397d2b9281abc8, 0x3fab5a622198a72c),
        (0x3bd29fd72f6917e5, 0xbf34ee71cf67c243),
        (0x3bef69e3e89e3765, 0xbf7237c02b462f6f),
        (0xbbaa32f85df10bfa, 0x3f00b67b6fdfec62),
        (0xbbcb53a20d4b2f61, 0x3f2365167d8bc330),
        (0xbb5df27a7ce0e8f5, 0xbeb3086b7a296e69),
        (0xbb64ce30c488f398, 0xbec61a141425c2a6),
        (0xbaf5a586401bb54e, 0x3e56712b23991e80),
        (0x3afc0839aea10c7b, 0x3e5f50736839e481),
        (0x3a894d28c9a59269, 0xbdf0366d02b91b84),
        (0xba3b7dda45b64dc6, 0xbdee38f9bda5f6df),
        (0x3a0f464a8fd7c7bf, 0x3d7faa3d1cb01e65),
        (0xba1e45ca78c915d9, 0x3d752317a2a5b9a2),
        (0x39a7d004a800c3b5, 0xbd064cd06847bd1d),
        (0xb95179a5ef0f66b7, 0xbcf666eebfeb573d),
        (0xb9285e2b929e1519, 0x3c87b8b54f1785bc),
        (0xb919589128d0188e, 0x3c729b70ca32ef58),
        (0xb8a46a9746bfb343, 0xbc03ba848f791229),
        (0x386730f56682d5be, 0xbbe8df65522ba529),
        (0x37f2ab030e344d2f, 0x3b7a5b900787f393),
        (0xb7bb11ca75f0fcdb, 0x3b5b53fa2dd77bb4),
        (0x378eb255c1629783, 0xbaece6fc2aac5380),
        (0xb729982419793a6f, 0xbac91d5406064eb8),
        (0xb6ebe671ea8254cb, 0x3a5a7a1d77627f77),
        (0xb6cf1004e96b1aa1, 0x3a3396e1ce8aa477),
        (0xb62a0c13a37cd740, 0xb9c490dcdeea0de1),
    ],
    [
        (0x0000000000000000, 0x0000000000000000),
        (0x3c56e5b3d654a3de, 0x3fbaf9cb42cd08a7),
        (0x3bd4df22cc118423, 0xbf4e1c66d7616e37),
        (0x3c32068ccdde2ecb, 0xbf91f90fd1013589),
        (0xbbb9426a5fb5baf4, 0x3f240e3eb09b7d6d),
        (0x3bcf4131edb03e4e, 0x3f4cb682ff471274),
        (0x3b8c9cd4cf70b38d, 0xbee0016819a61dd8),
        (0x3b8325d69b159805, 0xbef5d2d1c420646d),
        (0xbb157a8a155b14f1, 0x3e884b0fe8a85355),
        (0x3b1d5bbe26d9661b, 0x3e93559d96bef598),
        (0x3ac3da55cae73bf1, 0xbe257a0be74997ad),
        (0xbaac817776602a6c, 0xbe26677c8f53b6a6),
        (0xba48a3adc3758fa2, 0x3db8d1ae21b756a1),
        (0xba55e41c080dae7e, 0x3db24a6e1ac65e87),
        (0x39d87ea029a22f95, 0xbd4432017034b94c),
        (0xb9b0c571c0172220, 0xbd362ae0447c7c88),
        (0x392c7ada7dc7ba07, 0x3cc86190e6a87084),
        (0xb8cce5846ffd630a, 0x3cb4b9e1622f9586),
        (0x38e2a29910f701c9, 0xbc46b1f3e51e7716),
        (0x389f461c8b8d14fd, 0xbc2ecd4131542af6),
        (0xb865a2f5fd7d4f74, 0x3bc0c7d6edd792f0),
        (0xb84d80cbe9f82337, 0x3ba29fe1dfb030da),
        (0xb7be1d7b0beadd6d, 0xbb342eb58314dd57),
        (0xb7bd13d73eb584cb, 0xbb12aee37e25148e),
        (0x3706be9737507cbd, 0x3aa420385af46d51),
        (0x37178c7e05bb4a09, 0x3a7f97d766899d96),
        (0x36b40aa8f19fb9a4, 0xba10e85b8b8bab6c),
        (0xb68622de5eece514, 0xb9e6d19365680b1c),
    ],
    [
        (0x3c49a6abbfd839f8, 0x3fba9d183bc04545),
        (0x0000000000000000, 0x0000000000000000),
        (0xbc39a6abbfd839f8, 0xbfaa9d183bc04545),
        (0x3b97025ee34cfaec, 0x3f334779874010dc),
        (0xbc01fefbfc4e4e9c, 0x3f71ba2299ab88a8),
        (0xbb55c0e2a19295ff, 0xbefecb19f5bdc649),
        (0xbbcce75d43258257, 0xbf22e052707cc859),
        (0xbb5a63ba682d6499, 0x3eb18a5da77edd83),
        (0xbb6dd00d54139579, 0x3ec5846b622f592b),
        (0xbae5e872e9945332, 0xbe54b17e97b5884a),
        (0x3ac6a221a04b77f7, 0xbe5e7f76674dfaab),
        (0xba84b342c0e05e8d, 0x3dedeb504ed353c6),
        (0x3a86775d5b3d25ee, 0x3ded72e25f190409),
        (0xba00d09a917a59f6, 0xbd7d3e076b0201d6),
        (0x39c46b750b88e7f8, 0xbd749b825302af88),
        (0x39abeb00dd42cc76, 0x3d049d35475fc5c4),
        (0x399639c0b0dc546c, 0x3cf5dac9623f7b3a),
        (0xb8c1489d884eb85a, 0xbc85f3d5130dc414),
        (0xb91861dc55985f01, 0xbc722a4aeb6977eb),
        (0x3890153f6d844191, 0x3c02479e3ad06b06),
        (0x3880093480726623, 0x3be84cdbd4996301),
        (0xb80c8f94083b0211, 0xbb78745ed629d641),
        (0x37e3f2458bb649a6, 0xbb5ab87a5deb4183),
        (0xb788f6fa139ccbd7, 0x3aeada86a4d62add),
        (0x3756d41ddc1f8d52, 0x3ac893b9d1f77835),
        (0xb6ee50fc22fbfa2f, 0xba58a3198036b128),
        (0x36c2de5121e2e61a, 0xba332fd57127bac6),
        (0xb63875952298668b, 0x39c32aa4779bf0be),
    ],
    [
        (0x0000000000000000, 0x0000000000000000),
        (0x3c5729f43724612e, 0xbfba4407dac72297),
        (0x3be5723d945cf385, 0x3f4bcba4dec1da44),
        (0xbc3fe775590d7dac, 0x3f91803c65cafdfb),
        (0x3bc90cfa57dbb628, 0xbf2283df2b3e4a6e),
        (0x3bc3e3e0391f1e19, 0xbf4bf695e89259cd),
        (0x3b630f86ed6a90e0, 0x3edd8f3759122d51),
        (0x3b9030fb5fd0d14f, 0x3ef54246c8e04218),
        (0xbb1b2983b9c451c8, 0xbe867111bc8e67a4),
        (0xbb3a3cef2b75d5c2, 0xbe92d72a1f62ec18),
        (0x3acbd165727f6821, 0x3e23d9ca3eb1e22f),
        (0xbacc01aecceaa6e4, 0x3e25d74a6e225515),
        (0xba59e0bf6ede94ee, 0xbdb6f4b420b2f957),
        (0xba5cdfc28e1823b4, 0xbdb1d6f985dc3ec9),
        (0x39e7d772f8fce3e1, 0x3d42b1d81101dea8),
        (0xb9d82ef548b3e361, 0x3d35a2197089fbfb),
        (0xb956841fa8612f5c, 0xbcc6974aff17c78f),
        (0x394b2b4bcdaabf0e, 0xbcb43d4892ea1bcf),
        (0xb8dd3073293bb4a5, 0x3c450d21b45ffbaf),
        (0xb8b6f6e75ed53665, 0x3c2e196b40872c65),
        (0x384e654070aa0a0c, 0xbbbf2abd4dc31a0d),
        (0x3847ef0270966133, 0xbba23699d6695800),
        (0xb7d774e0a1072a6c, 0x3b32c4314fb1b042),
        (0x3798dfd4a7af4ecb, 0x3b1248f0b0a01e5c),
        (0x374feaa8039e5d21, 0xbaa2bd2ced8351bd),
        (0x37162b8b059830f3, 0xba7ef1e8bc571d77),
        (0x36a5be76c75f587a, 0x3a0f87929eae0a41),
        (0x367e0d8209c727ce, 0x39e65e8aae87b358),
    ],
    [
        (0x3c4f5da9526c15aa, 0xbfb9ee5eee1a97c6),
        (0x0000000000000000, 0x0000000000000000),
        (0xbc3f5da9526c15aa, 0x3fa9ee5eee1a97c6),
        (0xbbc33100916421b3, 0xbf31d55d93e59bd7),
        (0xbbb6887e944ccfcf, 0xbf7146219394a99c),
        (0x3b8ca8e50931ec0e, 0x3efc7d32bb646e2a),
        (0x3b459561844c2171, 0x3f226599aed4f1ca),
        (0x3b2f06633c8a57af, 0xbeb03ba58ec85c12),
        (0xbb602686ed9f14f1, 0xbec4f9e3961bb9dd),
        (0x3aeea9ee2c50aae8, 0x3e5328d4dc334079),
        (0xbaef3a53aa36f2d4, 0x3e5dbda7ac70ff93),
        (0x3a77a84a53fa4382, 0xbdebb7c2f187144e),
        (0xba64d01b19ecf2c5, 0xbdecbac24c760567),
        (0x3a16eec036aa6895, 0x3d7b1c35b65eca4e),
        (0xba1d9ece3561eb27, 0x3d741d23f93de7a0),
        (0x39ab2aa54e1d2f75, 0xbd03208d806698c2),
        (0xb99acf6d8899800a, 0xbcf557c166250d9c),
        (0x391934ffa520f9d8, 0x3c84637130a9d7e6),
        (0x38f36ddde127bd83, 0x3c71c021934a5bf8),
        (0xb89d68e84d698c21, 0xbc00fed119c94fe7),
        (0x388e25551faa0fb9, 0xbbe7c2d64524cfa1),
        (0x381e8f2946a1a0bb, 0x3b76c3357c3f7adf),
        (0x37f7b13ebe2df289, 0x3b5a25665c8bb28a),
        (0xb78e1214cb5a34de, 0xbae906c5169421c4),
        (0x376f2d89008c8e68, 0xbac810fd94a57b7b),
        (0x36d90aeea93b07c4, 0x3a56fd9fc0a89c82),
        (0xb6dc9ebd96e6e684, 0x3a32cd7652121959),
        (0xb61332c9199d4380, 0xb9c1e8fff9203895),
    ],
    [
        (0x0000000000000000, 0x0000000000000000),
        (0x3c422d28ddb0663f, 0x3fb99be73fa3efcc),
        (0x3bc06623176e8323, 0xbf49c3248da75775),
        (0x3c24e67d5d6f01f8, 0xbf91107147eda800),
        (0xbbb9f85d6cd5a102, 0x3f212980be9d8520),
        (0xbbc6d7f4255305b6, 0x3f4b44e1221e6051),
        (0xbb75ffceb4092099, 0xbedb67b78c1dd9c5),
        (0xbb952647474dbc63, 0xbef4bc49e9b5bcfd),
        (0xbb07d96f2518a716, 0x3e84d02ffb8bcfd0),
        (0x3b3e32d27646b835, 0x3e9261c31456e43c),
        (0xbac76119a10d67c2, 0xbe226b400744def1),
        (0x3acd5778eebf7e73, 0xbe2551251a2b4223),
        (0x3a46fe94a695c8b2, 0x3db5500f9531112d),
        (0x3a083f83ad3b6fcc, 0x3db16b4e936f8a96),
        (0x39d92e892227b268, 0xbd415e70b28182e4),
        (0xb9d311286e6202d4, 0xbd352230c03cb29f),
        (0x39694a35e842f6b5, 0x3cc5019095879294),
        (0xb95c82d35e7d0a85, 0x3cb3c864dd8bd878),
        (0xb8e021ec04623c9c, 0xbc4397af75e324f3),
        (0xb8c1d4036b77bcc2, 0xbc2d701c9937cabc),
        (0x385d6fca0674878d, 0x3bbd0965d9326e17),
        (0xb84fa05c3c3c3a3f, 0x3ba1d319c4fce696),
        (0xb77cd5e86688f3b8, 0xbb3180c1f00da609),
        (0xb7ba86637aa4bf02, 0xbb11e8305b488b70),
        (0x3736c6f17e2a0301, 0x3aa17f7804b40b24),
        (0xb7084fe04399b71a, 0x3a7e53bca7936259),
        (0x369739312d3bebf0, 0xba0d7a5725978bd4),
        (0x3689e9b194e5ecb2, 0xb9e5f05f8d349ae8),
    ],
    [
        (0xbc55bbc298d062cd, 0x3fb94c6f5898708b),
        (0x0000000000000000, 0x0000000000000000),
        (0x3c45bbc298d062cd, 0xbfa94c6f5898708b),
        (0xbbc91fb79f224deb, 0x3f308f4f1b8fb0dc),
        (0x3c11274d5c28ac61, 0x3f70da931c776c71),
        (0x3b468ef3cfedd83e, 0xbefa7550d2940f2a),
        (0x3bbf74462dfd6402, 0xbf21f3b978a47cf4),
        (0xbb46a542136ebd9e, 0x3eae28ea02472df4),
        (0xbb6c80d173ca8c69, 0x3ec47930aec78cb8),
        (0x3adf057ade51b16d, 0xbe51ce32c4f34e51),
        (0x3afb45adc74dd76d, 0xbe5d0950e99c8b39),
        (0xba7a6618445563ec, 0x3de9c5a1603e79e8),
        (0xba8e2efa815aa6df, 0x3dec0f1778cf9887),
        (0xba168e5ec792cc80, 0xbd7938fe13097690),
        (0xba1edc43381b3d99, 0xbd73a70bdaedabef),
        (0x39ad72e528e9a77a, 0x3d01ceeed28a9bbc),
        (0x3981b5050358ac3a, 0x3cf4dcf95c5ca94a),
        (0x38e5c074caf95ef8, 0xbc82ff92717657b8),
        (0xb8c1517cb6203642, 0xbc715c5a3e02462c),
        (0x38848ac21ce9d63b, 0x3bffb3c968d1317c),
        (0xb868eaf9f67497e9, 0x3be740ae3a9480c4),
        (0xb7e7ce53355d1e7c, 0xbb754054fa0921f1),
        (0xb7db46466a3c45dd, 0xbb599a3335285172),
        (0x376a7830f0710c1c, 0x3ae763db68405483),
        (0x37496d5dfac38e6e, 0x3ac794c6d23aefa1),
        (0xb6d0259444fde6bc, 0xba5583129058029b),
        (0xb6b49a7445a09fb2, 0xba326f9c89204c85),
        (0x3648f301c94c285c, 0x39c0c73ede23db3f),
    ],
    [
        (0x0000000000000000, 0x0000000000000000),
        (0xbbb1d709b3e6aac3, 0xbfb8ffc9b9a131f6),
        (0x3bdb4dd57b3641ab, 0x3f47f724314bb99d),
        (0xbc3086cbc396cba9, 0x3f90a89c5d1074ba),
        (0x3bb8da379d94d744, 0xbf1feeaeb33465e8),
        (0xbbef433944c1a75a, 0xbf4a9fba36a7a08e),
        (0xbb7f256bdb98dded, 0x3ed980034770f3b2),
        (0x3b91b4c8faeaae10, 0x3ef43fa4bb8cba57),
        (0xbad948481046b8ce, 0xbe835f40f3a3902b),
        (0x3b3e2e9d940a603b, 0xbe91f465ba292c0d),
        (0xba98e1101ee20622, 0x3e212688f74f8106),
        (0x3ac17030074177fa, 0x3e24d3f7e3fdde95),
        (0xba567e7ce0ee9db6, 0xbdb3daee9c38c74a),
        (0x3a51aeae4597f6fa, 0xbdb106a132ee69a7),
        (0x39ebf6485dd79da8, 0x3d4030e7765d6c75),
        (0x39c15870eda63050, 0x3d34aa4c24070ac4),
        (0xb958792c38458a32, 0xbcc3987402712b60),
        (0xb95ac711e3d7df47, 0xbcb35a87994f9f22),
        (0xb8b987c84af86497, 0x3c424aa138f72769),
        (0x38cd1d7e8860d6ea, 0x3c2cd07eb7a4b74f),
        (0xb85adedac22a5607, 0xbbbb21f2d5c76559),
        (0xb8403dbefd803008, 0xbba174fb8123d957),
        (0xb7decb7c15886751, 0x3b305ef4d5737c95),
        (0xb78dcfe2c3d73d6b, 0x3b118c574d742d02),
        (0x3746542ad0f810ba, 0xbaa0621051dc9615),
        (0x371434d10d672431, 0xba7dbd00a70ea7c3),
        (0x36a3d2f4625f489c, 0x3a0ba13931894290),
        (0x367d34b5778ce244, 0x39e586f4fb44eb1c),
    ],
    [
        (0x3c5827414357db53, 0xbfb8b5ccb03d459b),
        (0x0000000000000000, 0x0000000000000000),
        (0xbc4827414357db53, 0x3fa8b5ccb03d459b),
        (0x3bc829bf4eaf884d, 0xbf2edc98b92fedf3),
        (0x3c0a89c9aabf2841, 0xbf70767d524cea88),
        (0xbb9b6aea43cf7c69, 0x3ef8a85a847c89c3),
        (0x3b9126e1254c1c3f, 0x3f2189afaa2bffcc),
        (0xbb239c5c72e2b643, 0xbeac1d231a69bb02),
        (0x3b6e89e6ffd32981, 0xbec40139bdbc9d36),
        (0xbac65998040abd88, 0x3e509a6859c8e374),
        (0x3aded4461012c62e, 0x3e5c60fc5f9b505f),
        (0xba80f4ba127543b5, 0xbde80ad391f7224c),
        (0xba8175d6b9c314bd, 0xbdeb6e954dd6c922),
        (0xba13a59996f17ca1, 0x3d778adc76ca1d1b),
        (0x39e11be5ff01d8f8, 0x3d733867c824f37c),
        (0xb99156aaeb019d3d, 0xbd00a1ec5f4dca90),
        (0xb98d52e6f6ab31a5, 0xbcf469ac872b68fd),
        (0xb92873e76419e2d3, 0x3c81c1b6937f11e6),
        (0x38ccce25f4bbd867, 0x3c70fe687033cfb9),
        (0x3891a388623b4d18, 0xbbfda774598ea00a),
        (0xb869e8b9411b1044, 0xbbe6c5c790ba8cb1),
        (0x38166a66767c0cb3, 0x3b73e54f8ca945a2),
        (0xb7f10320398ad824, 0x3b5916586436314c),
        (0x37824e29afdee9e2, 0xbae5eb34af321144),
        (0x376cca34b6192949, 0xbac71eb7d8b137e6),
        (0xb6e44bfb5b2e06d0, 0x3a542dd9e755ef71),
        (0x36d5af611735126d, 0x3a321616f456ec2b),
        (0xb6303bf3b5725f15, 0xb9bf82bf12709cef),
    ],
    [
        (0x0000000000000000, 0x0000000000000000),
        (0x3c116404635b6173, 0x3fb86e51bb2ee24d),
        (0x3be40f7dbe05e2df, 0xbf465e3bad214eb8),
        (0xbc3502bddec37518, 0xbf9047d6ed159c31),
        (0xbbb68a41509d990f, 0x3f1dce4a381ce24e),
        (0x3be0a2f7a61834bb, 0x3f4a05b93842da9a),
        (0x3b384a0ade44f859, 0xbed7ce3df60b0ea1),
        (0x3b8858267c36b09d, 0xbef3cb50138cce4b),
        (0x3b2cb60fbd510cee, 0x3e8216e84a5dec74),
        (0x3b2d7337b289cb04, 0x3e918e354c01533b),
        (0xbac53c3104a34351, 0xbe2005493039e5f0),
        (0x3ac1ef2cc08d8fcc, 0xbe245ed4a3733e9d),
        (0xba3577c4c664c7a0, 0x3db28e2ed388f175),
        (0xba513618783b7d45, 0x3db0a83fb8aae5cc),
        (0x39d2152affe981ca, 0xbd3e4745b1e410b2),
        (0xb9cc35474710f93a, 0xbd3439aae872dae3),
        (0xb90f376af336ee4d, 0x3cc2557a9d9a51a0),
        (0xb9418e01d92d0d8b, 0x3cb2f313647cb524),
        (0xb8ee555d5c581be4, 0xbc4120389f39a9c9),
        (0x38bceac00481a0f5, 0xbc2c39cb315eebe0),
        (0x38543f94ec753f1e, 0x3bb96c5a77539703),
        (0xb7f664a9d463aaf9, 0x3ba11bdcff0ddb62),
        (0xb7909879d65e235f, 0xbb2eb47c0f406536),
        (0x37b323a3eb4616f4, 0xbb1135192cf31d4c),
        (0x36df969385034399, 0x3a9ec17139b4fa1e),
        (0xb712728ddaed35bd, 0x3a7d2d581a06903c),
        (0xb68322eeb5683b7f, 0xba09f599e9b3fcee),
        (0xb6584681390e9457, 0xb9e522216f24e570),
    ],
    [
        (0xbc5aa58d824fcbf0, 0x3fb829356c2fb67c),
        (0x0000000000000000, 0x0000000000000000),
        (0x3c4aa58d824fcbf0, 0xbfa829356c2fb67c),
        (0x3bc03ee1c336b408, 0x3f2cd964e5e4caa0),
        (0x3c0adae94b604802, 0x3f70190ce66b97c3),
        (0xbb6093ba87cf3d05, 0xbef70d5fd8bad7b5),
        (0xbbae25e1759c428e, 0xbf2126a150469b05),
        (0x3b4f93e54fe01a02, 0x3eaa49ea85815004),
        (0xbb4a798f784213ff, 0x3ec3910f5c159f54),
        (0x3ab455b6aeed4cdd, 0xbe4f0f4bdbe20c90),
        (0xbaf74c179ce6757b, 0xbe5bc3697829b270),
        (0x3a883549e84165fc, 0x3de67f2959cd47c2),
        (0x3a6dc709a66c6637, 0x3dead81c085abf1e),
        (0xb9f0a2a8eed90e0b, 0xbd760a10ab0898ac),
        (0x3a0d4bed5455a7cd, 0xbd72d07fd9984920),
        (0x399e8c3991730925, 0x3cff2887ab34ba4b),
        (0xb99d15049de98d82, 0x3cf3fd2c46f43bfa),
        (0x3918217b1630091e, 0xbc80a48214b2d81b),
        (0xb90aebeb32b2ec56, 0xbc70a5cd24e7aff5),
        (0x387f8e2a60b26930, 0x3bfbd0274fe43cf0),
        (0xb87decdf6830646f, 0x3be65191993b3def),
        (0xb8169726f5849485, 0xbb72acc84c6c9c30),
        (0xb7ecea63d3d7881a, 0xbb5899530e0f85dd),
        (0x3783523fe1e47394, 0x3ae4974878ee147d),
        (0xb75b7329483fca7c, 0x3ac6ae722e577e49),
        (0x36f5f5ad175e3a10, 0xba52f937d6c2c894),
        (0x36da45ec2b162935, 0xba31c0af96ec39d1),
        (0xb63b5886d2aae802, 0x39bda7e9893ee0e5),
    ],
    [
        (0x0000000000000000, 0x0000000000000000),
        (0xbc4ea89f3a43207f, 0xbfb7e656ed57a0d1),
        (0x3bed3586339aa922, 0x3f44f0cfc62eb61c),
        (0x3c1736dd17549bef, 0x3f8fdaba63e9c655),
        (0x3bb8a2f82c46d7d1, 0xbf1be7bad1c50547),
        (0x3be9f603d64ca819, 0xbf4975ac0770c98d),
        (0xbb716dcdea217bd7, 0x3ed64a6af83ba860),
        (0x3b95937b8720b7f0, 0x3ef35e6b41b6d77b),
        (0x3b1fd0d91a2328f8, 0xbe80f12d050df219),
        (0x3b0b801799b9887e, 0xbe912e744222ec5c),
        (0x3a882bc4f09a4762, 0x3e1e04a8c0a89be6),
        (0x3ab7f6d6a00158e7, 0x3e23f0ed5e6a0d3b),
        (0x3a40220f85fad9d6, 0xbdb163fd7157593b),
        (0x3a410e6d9a99af97, 0xbdb04f8efe519654),
        (0xb9d5f5b1b377833f, 0x3d3c6417f7a11d41),
        (0xb9df5b1599ff1601, 0x3d33cfa2affaf9e4),
        (0xb953811266af2ff0, 0xbcc1334e857adc93),
        (0xb95d7d93dd11d118, 0xbcb2917ae18b1995),
        (0x38e71d9f8e4fabac, 0x3c4013b4389ffe8d),
        (0x38b8b37790d4d1d0, 0x3c2bab4c1141eb75),
        (0x3858ea4627e8efee, 0xbbb7e1ea8188735c),
        (0x381cd9d4b13e954d, 0xbba0c761c695703d),
        (0xb7c2ccb36474c65d, 0x3b2cdd9ab5adf543),
        (0xb7ad59a9bff1b6e1, 0x3b10e22b0412a059),
        (0x37298e06ba0a7eff, 0xba9cefb8354e799e),
        (0x371ce2c8c6a65fa1, 0xba7ca4621c89c3ea),
        (0x369bfbcde055522b, 0x3a0871d638e71562),
        (0x36702a74cab2e18d, 0x39e4c1b3f6f3c4fc),
    ],
    [
        (0xbc52d2ff041ff2f6, 0xbfb7a597eb76a5e3),
        (0x0000000000000000, 0x0000000000000000),
        (0x3c42d2ff041ff2f6, 0x3fa7a597eb76a5e3),
        (0x3bb9dc564feff08d, 0xbf2b0bd9eb615315),
        (0xbc03cbe9e9bac1ac, 0xbf6f831b9629acd2),
        (0xbb95cc1ecba38a9d, 0x3ef59d1c40c95ba7),
        (0x3bad7b9ad1a36974, 0x3f20c9d35d4be191),
        (0xbb39533f46784cb0, 0xbea8a711bec6920b),
        (0xbb4180f2a561ac4b, 0xbec327e42f39066e),
        (0x3adb23dbc633fe86, 0x3e4d225aa8ded6cd),
        (0xbaea1e7c5d2c24a0, 0x3e5b2f83a946aed5),
        (0x3a72070bf439257d, 0xbde51bef06eb5d9f),
        (0xba7f45169aef013e, 0xbdea4ab19859cce7),
        (0xba0912db1823cb9f, 0x3d74b03ce339d83a),
        (0x3a007d1a27a09f97, 0x3d726eb2bbaa63c7),
        (0xb991c095fa32b480, 0xbcfd433d6a1a8fc3),
        (0x3988ca74e8a71011, 0xbcf396dd9428b7d8),
        (0xb91f2420c8969cf6, 0x3c7f470853f5d87d),
        (0x39129d5b5b244d63, 0x3c705215ddbe7188),
        (0xb89ad8ca1c6e4ad7, 0xbbfa26d052836024),
        (0xb86057f50da94b85, 0xbbe5e38754f9398d),
        (0x381cce919f096c57, 0x3b71923ffdac92d4),
        (0x37f2bbe0806d82ac, 0x3b5822a7b984366d),
        (0x377a0e974a7e67bc, 0xbae3636b0de0af87),
        (0x376c2d3712cd37ea, 0xbac6439959854a9a),
        (0xb6f5653a15338763, 0x3a51e12428cdc675),
        (0x36bd7966e734f9d4, 0x3a316f2e8a6d63cb),
        (0xb64936e1f9cb0f09, 0xb9bbf81db6228424),
    ],
];
