/*******************************************************************************
 * Copyright (C) 2004-2007 Intel Corp. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 *   - Redistributions of source code must retain the above copyright notice,
 *     this list of conditions and the following disclaimer.
 * 
 *   - Redistributions in binary form must reproduce the above copyright notice,
 *     this list of conditions and the following disclaimer in the documentation
 *     and/or other materials provided with the distribution.
 * 
 *   - Neither the name of Intel Corp. nor the names of its
 *     contributors may be used to endorse or promote products derived from this
 *     software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL Intel Corp. OR THE CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *******************************************************************************/

//----------------------------------------------------------------------------
//
//  File:       LocalAgent.cpp
//
//----------------------------------------------------------------------------

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <ctype.h>

#include "ZTCLocalAgent.h"
#include "HECILinux.h"
#include "resource.h"
#include "version.h"
#include "Utils.h"
#include "Types.h"
#include "HECI_if.h"
#include "StringTableUtils.h"

static bool _verbose = false;


/*
* This is a main entry point
*/
int main(UINT32 argc, CHAR *argv[])
{

	AMT_STATUS 	status;
	LOCAL_AGENT_PARAMS param;
	StringTableUtils::StringTableInit("StatusStrings.dat",TRUE);
	StringTableUtils::StringTableInit("ZTCLocalAgent.dat",FALSE);
	status = Parse(argc, argv, &param);
	if (AMT_STATUS_SUCCESS != status )
	{
		PrintHelp();
		return status;
	}
	_verbose = param.Verbose;
	// Print Local Agent version
	DisplayLocalAgentVersion();
	PTHICommand command(_verbose);
	status = DiscoveryTest(&command,param.Activate);
	if(AMT_STATUS_SUCCESS == status && param.Activate)
	{
		status = Activate(&command, param);
	}
	if (AMT_STATUS_SUCCESS != status )
	{
		DisplayStatusString(status);
		return status;
	}
	return status;
}

/*
* Calls to DiscoveryTest for check AMT configuration.
* Arguments:
*	isActivate - if this test running for start amt configuration
* Return values:
*	AMT_STATUS_SUCCESS - on success for get amt configuration data
*	appropriate error value defined in StatusCodeDefinitions.h - on failure
*/
UINT32 DiscoveryTest(PTHICommand *command,bool isActivate)
{
	AMT_STATUS status =AMT_STATUS_SUCCESS ;
	CODE_VERSIONS CodeVersions;
	AMT_HASH_HANDLES HashHandles;
	AMT_PROVISIONING_STATE provstate;
	AMT_BOOLEAN legacy;
	CFG_PROVISIONING_MODE mode;
	bool isReinit = false;
	if(true != command->PTHIClient.Init())
	{

		if(INVALID_HANDLE_VALUE == command->PTHIClient.GetHandle())
		{
			DisplayErrorMessage(LA_HECI_NOT_INSTALLED_ERROR);
			status = AMT_STATUS_INTERNAL_ERROR;
		}
		else
		{
			command->PTHIClient.Deinit();
			DisplayErrorMessage( HECI_CONNECT_TO_PTHI_CLIENT_FAILURE );
			if(!isActivate)
			{
				return status;
			}
			if(true == ChangeToAMT())
			{
				DisplayMessage(WORD_CHANGE_TO_AMT);
				DisplayMessage(WORD_SUCCESS);
				isReinit = ReInitPTHI(command);
			}
			else
			{
				DisplayErrorMessage( CHANGE_TO_AMT_FAILURE );
				status = AMT_STATUS_INTERNAL_ERROR;
			}
		}
		if(!isReinit)
			return status;
	}
	status = command->GetCodeVersions(&CodeVersions);
	if (AMT_STATUS_SUCCESS  == status )
	{
		command->DisplayCodeVersions(&CodeVersions);
	}
	else
	{
		DisplayStatusString(status);
	}
	// provisioning state
	status = command->GetProvisioningState(&provstate);
	if (AMT_STATUS_SUCCESS  == status )
	{
		command->DisplayProvisioningState(&provstate);
	}
	else
	{
		DisplayStatusString(status);
	}
	// provisioning  mode
	status = command->GetProvisioningMode(&legacy,&mode);
	if (AMT_STATUS_SUCCESS == status )
	{
		if(PROVISIONING_STATE_PRE == provstate && PROVISIONING_MODE_SMB == mode)
		{
			command->PTHIClient.Deinit();
			if(!isActivate)
			{
				return status;
			}
			if(true == ChangeToAMT())
			{
				DisplayMessage(WORD_CHANGE_TO_AMT);
				DisplayMessage(WORD_SUCCESS);
				isReinit = ReInitPTHI(command);
			}
			else
			{
				DisplayErrorMessage( CHANGE_TO_AMT_FAILURE );
				status = AMT_STATUS_INTERNAL_ERROR;
			}
		}
		if(!isReinit)
			return status;
	}
	else
	{
		DisplayStatusString(status);
		return status;
	}

	if(true==isActivate && PROVISIONING_STATE_PRE != provstate)
	{
		status = AMT_STATUS_INVALID_PROVISIONING_STATE;
	}
	//enumerate hash handles
	status = command->EnumerateHashHandles(&HashHandles);
	if (AMT_STATUS_SUCCESS  == status )
	{
		// print all handles in here if any
		if(HashHandles.Length > 0)
		{
			DisplayMessage(FOUND);
			fprintf(stdout," %d ",HashHandles.Length);
			DisplayString(CERT_HASHES_IN_FW);
			fprintf(stdout,"\n");
			for(UINT32 i = 0;i<HashHandles.Length ;i++)
			{
				fprintf(stdout,"%d,",HashHandles.Handles[i]);
			}
			fprintf(stdout,"\n");

			//print certificate hashes
			for(UINT32 i = 0;i<HashHandles.Length ;i++)
			{
				CERTHASH_ENTRY HashEntry;
				status = command->GetCertificateHashEntry(HashHandles.Handles[i],&HashEntry);
				if (AMT_STATUS_SUCCESS  != status )
				{
					DisplayStatusString(status);
				}
				else
				{
					command->DisplayHashEntry(HashEntry);
					if(NULL != HashEntry.Name.Buffer)
						free(HashEntry.Name.Buffer);
				}
			}
		}
		else
		{
			DisplayMessage(NO_HANDLES_FOUND);
			fprintf(stdout,"\n");
		}
	}
	else
	{
		DisplayStatusString(status);
	}

	return status;
}

/*
* Calls to Activate AMT configuration.
* Arguments:
*	param - Local agent parameters structure
* Return values:
*	PT_STATUS_SUCCESS - on success for get amt configuration data
*	appropriate error value defined in StatusCodeDefinitions.h - on failure
*/
UINT32 Activate(PTHICommand *command,LOCAL_AGENT_PARAMS param)
{
	AMT_STATUS status;
	AMT_BOOLEAN legacy;
	AMT_BOOLEAN ztcEnabled;
	AMT_PROVISIONING_TLS_MODE provisioningTlsMode;
	AMT_RNG_STATUS rngStatus;
	AMT_ANSI_STRING dnsSuffix;
	AMT_ANSI_STRING tempDnsSuffix;
	AMT_ANSI_STRING otp;
	AMT_PROVISIONING_STATE provstate;
	CFG_PROVISIONING_MODE mode;
	do
	{
		// legacy mode provisioning
		status = command->GetProvisioningMode(&legacy,&mode);
		if (AMT_STATUS_SUCCESS == status )
		{
			command->DisplayAMTMode(&legacy);
		}
		else
		{
			DisplayStatusString(status);
			break;
		}
		if(legacy && 0!= param.OneTimePassword.Length)
		{
			status = AMT_STATUS_INVALID_PT_MODE;
			DisplayStatusString(status);
			break;
		}
		// ZTC enabled
		status = command->GetZeroTouchEnabled(&ztcEnabled);
		if (AMT_STATUS_SUCCESS == status )
		{
			command->DisplayZTCEnabled(&ztcEnabled);
		}
		else
		{
			DisplayStatusString(status);
			break;
		}
		if(!ztcEnabled && 0!= param.OneTimePassword.Length)
		{
			status = AMT_STATUS_INVALID_PT_MODE;
			DisplayStatusString(status);
			break;
		}
		// Get provisioning TLS mode
		status = command->GetProvisioningTlsMode(&provisioningTlsMode);
		if (AMT_STATUS_SUCCESS == status )
		{
			command->DisplayProvisioningTlsMode(&provisioningTlsMode);
		}
		else
		{
			DisplayStatusString(status);
			break;
		}
		if(PSK == provisioningTlsMode && 0!= param.OneTimePassword.Length)
		{
			status = PTSDK_STATUS_INVALID_PARAM;
			DisplayStatusString(status);
			break;
		}
		if(PSK != provisioningTlsMode && ztcEnabled )
		{

			//Generate Rng Key
			status = command->GenerateRngKey();
			if(AMT_STATUS_SUCCESS != status && AMT_STATUS_RNG_GENERATION_IN_PROGRESS != status)
			{
				DisplayStatusString(status);
				break;
			}
			else if(AMT_STATUS_RNG_GENERATION_IN_PROGRESS == status)
			{
				for(int i = 0; i < 12; i++)
				{
					sleep(5);//5 seconds

					if(true != command->PTHIClient.Init())
						continue;

					// Calling GetRngSeedStatus() too soon might fail.
					// We need to give the firmware time to finish its reset
					// and to generate the RNG seed.
					sleep(5);//seconds
					status = command->GetRngSeedStatus(&rngStatus);
					if(status == AMT_STATUS_SUCCESS)
					{
						break;
					}
				}
			}
			else
			{
				status = command->GetRngSeedStatus(&rngStatus);
			}

			if (AMT_STATUS_SUCCESS == status )
			{
				command->DisplayRngSeedStatus(&rngStatus);
			}
			else
			{
				DisplayStatusString(status);
				break;
			}
			if( RNG_STATUS_EXIST != rngStatus)
			{
				status = AMT_STATUS_PKI_MISSING_KEYS;
				DisplayStatusString(status);
				break;
			}
			if(0 != param.DnsSuffix.Length)
			{
				dnsSuffix.Length = param.DnsSuffix.Length;
				dnsSuffix.Buffer = (CHAR*)malloc(dnsSuffix.Length);
				if (NULL == dnsSuffix.Buffer)
				{
					status = AMT_STATUS_INTERNAL_ERROR;
					DisplayErrorMessage(ALLOCATE_MEMORY_ERROR);
					break;
				}
				memcpy(dnsSuffix.Buffer,param.DnsSuffix.String,dnsSuffix.Length);
				status = command->SetDnsSuffix(dnsSuffix);

				if (AMT_STATUS_SUCCESS != status )
				{
					DisplayStatusString(status);
					if(NULL != dnsSuffix.Buffer)
						free(dnsSuffix.Buffer);
					break;
				}
				status = command->GetDnsSuffix(&tempDnsSuffix);
				if (AMT_STATUS_SUCCESS != status )
				{
					DisplayStatusString(status);
					if(NULL != dnsSuffix.Buffer)
						free(dnsSuffix.Buffer);
					break;
				}
				if(tempDnsSuffix.Length!=dnsSuffix.Length ||0!=memcmp(dnsSuffix.Buffer,tempDnsSuffix.Buffer,tempDnsSuffix.Length))
				{
					status=AMT_STATUS_INTERNAL_ERROR;
					DisplayStatusString(status);
					if(NULL != dnsSuffix.Buffer)
						free(dnsSuffix.Buffer);

					if(NULL != tempDnsSuffix.Buffer)
						free(tempDnsSuffix.Buffer);
					break;
				}

				if(NULL != dnsSuffix.Buffer)
					free(dnsSuffix.Buffer);
				if(NULL != tempDnsSuffix.Buffer)
					free(tempDnsSuffix.Buffer);
			}
			if(0 != param.OneTimePassword.Length)
			{
				otp.Length = param.OneTimePassword.Length;
				otp.Buffer = (CHAR*)malloc(otp.Length);
				if (NULL == otp.Buffer)
				{
					status = AMT_STATUS_INTERNAL_ERROR;
					DisplayErrorMessage(ALLOCATE_MEMORY_ERROR);
					break;
				}
				memcpy(otp.Buffer,param.OneTimePassword.String,otp.Length);
				status = command->SetProvisioningServerOTP(otp);
				if(NULL != otp.Buffer)
					free(otp.Buffer);
				if (AMT_STATUS_SUCCESS != status )
				{
					DisplayStatusString(status);
					break;
				}
			}
		}

		//Start configuration
		status = command->StartConfiguration();
		if(AMT_STATUS_SUCCESS != status && AMT_STATUS_CERTIFICATE_NOT_READY != status )
		{
			DisplayStatusString(status);
			break;
		}
		else if(AMT_STATUS_CERTIFICATE_NOT_READY == status)
		{
			if(PSK == provisioningTlsMode)
			{
				DisplayStatusString(status);
				break;
			}
			else
			{
				for(int i = 0; i < 20; i++)
				{
					sleep(30);//30 seconds
					// provisioning state
					status = command->GetProvisioningState(&provstate);
					if (AMT_STATUS_SUCCESS  == status )
					{
						if( PROVISIONING_STATE_IN == provstate)
						{
							command->DisplayProvisioningState(&provstate);
							break;
						}
					}
					else
					{
						DisplayStatusString(status);
						break;
					}
				}
				if(AMT_STATUS_SUCCESS  == status && PROVISIONING_STATE_IN != provstate)
				{
					command->DisplayProvisioningState(&provstate);
					status = AMT_STATUS_INVALID_PROVISIONING_STATE;
					DisplayStatusString(status);
					break;
				}
			}
		}
	}while(0);
	DisplayMessage(WORD_AMT_CONFIG_ACTIVATE);
	if(AMT_STATUS_SUCCESS == status)
	{
		DisplayMessage(WORD_SUCCESS);
		fprintf(stdout,"\n");

	}
	else
	{
		DisplayMessage(WORD_FAILURE);
		fprintf(stdout,"\n");

	}
	return status;
}

// Define GUID used to connect to the Watchdog client (via the HECI device)
// {05B79A6F-4628-4D7F-899D-A91514CB32AB}
DEFINE_GUID(HECI_WATCHDOG_GUID,0x05B79A6F, 0x4628, 0x4D7F, 0x89, 0x9D, 0xA9, 0x15, 0x14, 0xCB, 0x32, 0xAB);

/*
* Change SCU to AMT
*/
bool ChangeToAMT()
{
	UCHAR *rxBuff = (UCHAR *)NULL;
	DWORD bytesRead, bytesWritten;
	unsigned long bufSize;
	HECILinux WDClient(HECI_WATCHDOG_GUID,_verbose);

	if(true != WDClient.Init())
	{
		if (_verbose)
		{
			DisplayErrorMessage(HECI_CONNECT_TO_WD_CLIENT_FAILURE);
			fprintf(stdout,"\n");
		}
		return false;
	}
	bufSize = WDClient.GetBufferSize();

	if (bufSize < sizeof(STATE_INDEPNDENCE_COMMAND))
	{
		if (_verbose)
		{
			DisplayErrorMessage(FW_BUFFER_IS_TO_SMALL);
		}
		goto err;
	}

	rxBuff = (UCHAR *)malloc(bufSize);
	if (NULL  == rxBuff)
	{
		if (_verbose)
		{
			DisplayErrorMessage(ALLOCATE_MEMORY_ERROR);
		}
		goto err;
	}
	STATE_INDEPNDENCE_COMMAND msg;
	msg.Cmd = STATE_INDEPNDENCE_CMD;
	msg.ByteCount = STATE_INDEPNDENCE_BYTE_COUNT;
	msg.SubCmd = STATE_INDEPNDENCE_ENABLED;
	msg.Version = STATE_INDEPNDENCE_VERSION;


	bytesWritten = WDClient.SendMessage((UCHAR *)&msg, sizeof(msg));
	if (bytesWritten != sizeof(msg))
	{
		if (_verbose)
		{
			DisplayErrorMessage(SEND_DATA_TO_FW_FAILURE);
		}
		goto err;
	}
	STATE_INDEPENDECE_IS_CHANGE_ENABLED_REPLY repEnabledMsg;
	bytesRead = WDClient.ReceiveMessage(rxBuff, bufSize);
	if (bytesRead != sizeof(STATE_INDEPENDECE_IS_CHANGE_ENABLED_REPLY))
	{
		if (_verbose)
		{
			DisplayErrorMessage(RECEIVE_DATA_FROM_FW_FAILURE);
		}
		goto err;
	}
	memcpy(&repEnabledMsg,rxBuff,sizeof(STATE_INDEPENDECE_IS_CHANGE_ENABLED_REPLY));
	if (!repEnabledMsg.Enabled)
	{
		goto err;
	}
	//Send change to AMT
	msg.SubCmd = STATE_INDEPNDENCE_CHANGE_TO_AMT;
	bytesWritten = WDClient.SendMessage((UCHAR *)&msg, sizeof(msg));
	if (bytesWritten != sizeof(msg))
	{
		if (_verbose)
		{
			DisplayErrorMessage(SEND_DATA_TO_FW_FAILURE);
		}
		goto err;
	}
	STATE_INDEPENDECE_CHANGE_TO_AMT_REPLY repChangeToAMTMsg;
	bytesRead = WDClient.ReceiveMessage(rxBuff, bufSize);
	if (bytesRead != sizeof(STATE_INDEPENDECE_CHANGE_TO_AMT_REPLY))
	{
		if (_verbose)
		{
			DisplayErrorMessage(RECEIVE_DATA_FROM_FW_FAILURE);
		}
		goto err;
	}
	memcpy(&repChangeToAMTMsg,rxBuff,sizeof(STATE_INDEPENDECE_CHANGE_TO_AMT_REPLY));
	if (HECI_STATUS_OK  != repChangeToAMTMsg.Status)
	{
		goto err;
	}
	free(rxBuff);
	WDClient.Deinit();
	return true;

err:
	if (rxBuff) free(rxBuff);
	WDClient.Deinit();

	return false;
}

void DisplayLocalAgentVersion()
{
	fprintf(stdout,"\n");
	DisplayMessage(VERSION_MESSAGE);
	fprintf(stdout,"%d.%d.%d.%d\n\n",
		MAJOR_VERSION, MINOR_VERSION,
		QUICK_FIX_NUMBER, VER_BUILD);
}

bool ReInitPTHI(PTHICommand *command)
{
	for(int i = 0; i < 12; i++)
	{
		sleep(5);

		if(true != command->PTHIClient.Init())
			continue;
		else
		{
			return true;

		}

	}
	return false;
}
