/*******************************************************************************
 * Copyright (C) 2004-2007 Intel Corp. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 *   - Redistributions of source code must retain the above copyright notice,
 *     this list of conditions and the following disclaimer.
 * 
 *   - Redistributions in binary form must reproduce the above copyright notice,
 *     this list of conditions and the following disclaimer in the documentation
 *     and/or other materials provided with the distribution.
 * 
 *   - Neither the name of Intel Corp. nor the names of its
 *     contributors may be used to endorse or promote products derived from this
 *     software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL Intel Corp. OR THE CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *******************************************************************************/

//----------------------------------------------------------------------------
//
//  File:       Options.h
//
//  Notes:      This file contains the definition of a generic class for command line options
//              parsing.
//
//----------------------------------------------------------------------------

#ifndef _OPTIONS_H
#define _OPTIONS_H

#include <map>

/*
Options is a generic class for parsing command line arguments. The command line is expected to
be of the form: progname.exe -opt1 val1 -opt2 val2... where opt1 is the option name, and val1
is the value for this option.
The option names are specified in advance, and the ones specified are the only valid ones:
any other option name is considered an error. The values of the parameters are optional, but it
must be specified in advance whether each parameter has a value.
Any other parameter in the command line, that doesn't start with '-' is ignored, and would not
cause errors.
*/
class Options
{
private:
	//internal function object for comparing strings in map (case insensitive)
	struct StrLess
	{
		bool operator()(const char* left, const char* right) const;
	};

public:
	//One node in the format list of the options.
	struct FormatNode
	{
		const char* option; //Name of option
		bool bHasParam; //Indication whether this option has a parameter
	};

	//class that represents the entire format of the options.
	class Format
	{
	public:
		//ctor to build Format from a list of FormatNode's
		Format(const FormatNode rawFormat[], int rawFormatSize);
		//default ctor
		Format();

		//Add one option to the format, with option name and parameter indication.
		void addOption(const char* option, bool bHasParam);

		//returns true iff a certain option exists in the format.
		bool optionExists(const char *option) const;

		//returns true iff a certain option has a prameter indicated by the format.
		bool optionHasParam(const char* option) const;

	private:
		typedef std::map<const char*, bool, StrLess> FormatMap;
		//Map that holds the association between option name and parameter indication.
		FormatMap m_formatMap;
	};

	//ctor to build Options by parsing a command line, with a given Format object.
	Options(int argc, const char* argv[], const Format &format);

	//ctor to build Options by parsing a command line, with a given array of FormatNode's
	Options(int argc, const char* argv[], const FormatNode rawFormat[], int rawFormatSize);

	//default ctor. parse() should be used when using it.
	Options();

	//return the value of a given option.
	//Returns NULL if option doesn't exist, "" if option has no value.
	const char* getOption(const char* option) const;

	//returns true iff a certain option exists in the Options object.
	bool optionExists(const char* option) const;

	//returns true iff parse was called - either directly or by a ctor.
	//Will return false if parsing wasn't done or if it failed.
	bool isParsed() const {return m_parsed;}

	//parse a command line according to Format object, and store results in this Options object.
	bool parse(int argc, const char* argv[], const Format &format);

	//parse a command line according to array of FormatNode's, and store results in this
	//Options object.
	bool parse(int argc, const char* argv[], const FormatNode rawFormat[], int rawFormatSize);

private:
	typedef std::map<const char*, const char*, StrLess> OptionsMap;

	OptionsMap m_optionsMap; //map that associates option names with their values.
	bool m_parsed; //was parse() called
};

#endif //#ifndef _OPTIONS_H

