/* X Letters - Catch falling words */
/* Copyright (C) 1998 by
 *   Peter Horvai (peter.horvai@ens.fr)
 *   David A. Madore (david.madore@ens.fr) 
 * 
 * May 2001: Felipe Bergo (bergo@seul.org),
 *           added automake/autoconf building, made
 *           word file be searched for across some
 *	     probable directories (WORDPATH defined in
 *           xletters.h
 */

/*
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See
 * the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/* ********** INCLUDE FILES ********** */

/* General include files */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include <errno.h>
#include <fcntl.h>
#include <pwd.h>

#include <sys/mman.h>
#include <sys/types.h>

/* Xlib include files */
#include <X11/Xlib.h>

/* Xt include files */
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>

/* Xaw include files */
#include <X11/Xaw/Box.h>
#include <X11/Xaw/Label.h>
#include <X11/Xaw/Command.h>
#include <X11/Xaw/Toggle.h>

/* Configuration file */
#include "xletters.h"

/* ********** TYPES AND VARIABLES ********** */

/* Game variables */
char deathmode; /* Send and receive words on standard input/output */
char lifemode; /* Make words fall periodically, as usual */
char trainingmode;
char scorecounts;
int level;
long score,vpoints,levpoints;
int lives;
char bonuslevel;

/* Data structure describing each falling word */
typedef struct word_state_s {
  char *s; /* The actual word - malloc()ated */
  int len; /* The length of that word */
  int typed; /* Number of characters correctly typed */
  int width,twidth; /* Width in pixels of whole string and typed part */
  long x; /* x position of left-most point, in pixels */
  long y; /* y position of baseline, in *millipixels* */
  long rate; /* Descent rate in millipixels per time unit */
} word_state_t;

/* The words that are falling */
word_state_t words[MAXWORDS];
/* The number of words that are falling */
int nb_words;

#define MAX_WORDSIZE 1023

/* Whether the game is paused */
char paused;

/* Times are measured in fundamental time units (normally 75ms) */
unsigned long current_time; /* Time since beginning of play */
unsigned long lastword_time; /* Time since last word started falling */
unsigned long level_time; /* Time since beginning of level */

/* The word being typed in deathmatch mode */
char deathword[MAX_WORDSIZE+1];
/* The length of that word */
int deathln;

/* The word file */
char *wordf; /* wordfile position after mmap()ed */
off_t wordf_len; /* The length of the word file in characters */

/* ---------- X related variables ---------- */

/* Fallback resources */
char *fallback_resources[] = {
  "*input: True",
  "*background: Grey51",
  "*foreground: White",
  "*Box.background: DarkSlateGrey",
  "*Label.foreground: White",
  "*label.foreground: Green",
  "*Command.foreground: Yellow",
  "*Toggle.foreground: Yellow",
  "*quitButton.label: Quit",
  "*pauseButton.label: Pause",
  "*modeButton.label: Type",
  "*sendButton.label: Send",
  "*nextButton.label: Next",
  "*quitButton.accelerators:"
  "<KeyPress>Escape: set() notify()",
  "*pauseButton.accelerators:"
  "<KeyPress>Pause: toggle() notify()\\n\\\n"
  "<KeyPress>Tab: toggle() notify()",
  "*modeButton.accelerators:"
  "<KeyPress>BackSpace: toggle() notify()",
  "*sendButton.accelerators:"
  "<KeyPress>Return: set() notify() unset()",
  "*nextButton.accelerators:"
  "<KeyPress>Next: set() notify() unset()",
  NULL
};

/* Application resources */
typedef struct app_res_val_s {
  XFontStruct *word_font;
  Pixel word_color,typed_color;
  char *death_mode;
  Boolean training_mode;
} app_res_val_t;

app_res_val_t app_res_val;

XtResource app_res_tab[] = {
  {"wordFont", XtCFont, XtRFontStruct, sizeof(XFontStruct *),
   XtOffsetOf(app_res_val_t,word_font),
   XtRString, FONTNAME
  },
  {"wordColor", XtCForeground, XtRPixel, sizeof(Pixel),
   XtOffsetOf(app_res_val_t,word_color),
   XtRString, WORD_COLOR
  },
  {"typedColor", "HighlightColor", XtRPixel, sizeof(Pixel),
   XtOffsetOf(app_res_val_t,typed_color),
   XtRString, TYPED_COLOR
  },
  {"deathMode", "DeathMode", XtRString, sizeof(char *),
   XtOffsetOf(app_res_val_t,death_mode),
   XtRImmediate, "normal"
  },
  {"trainingMode", "TrainingMode", XtRBoolean, sizeof(Boolean),
   XtOffsetOf(app_res_val_t,training_mode),
   XtRImmediate, False
  },
};

/* Recognized options */
XrmOptionDescRec options[] = {
  {"-wfn", "*wordFont", XrmoptionSepArg, NULL},
  {"-wc", "*wordColor", XrmoptionSepArg, NULL},
  {"-tc", "*typedColor", XrmoptionSepArg, NULL},
  {"-gbg", "*gameSpace.background", XrmoptionSepArg, NULL},
  {"-death", "*deathMode", XrmoptionNoArg, (XtPointer)"death"},
  {"-duel", "*deathMode", XrmoptionNoArg, (XtPointer)"duel"},
  {"-nodeath", "*deathMode", XrmoptionNoArg, (XtPointer)"normal"},
  {"-train", "*trainingMode", XrmoptionNoArg, (XtPointer)"True"},
  {"-notrain", "*trainingMode", XrmoptionNoArg, (XtPointer)"False"},
};

/* The timer interval reference */
XtIntervalId timeout;

/* The widgets */
Widget toplevel,ground_box,label;
Widget quit_button,pause_button,mode_button,send_button,next_button;
Widget lives_label,score_label,level_label;
Widget type_space;
Widget game_space;

/* The application context */
XtAppContext app_context;

/* Various Xlib related variables */
Display *dpy;
Window win;
GC gc0,gc1,gc2;
XFontStruct *fnt_info;
Font fnt;

/* ---------- High score table ---------- */

/* High score table */
typedef struct hstable_entry_s {
  char name[1024];
  uid_t uid;
  int level;
  long score;
  unsigned long dur;
} hstable_entry_t;

hstable_entry_t hstable[SCORELEN];

uid_t euid;
gid_t egid;

/* ********** HIGH SCORE TABLE MANAGEMENT ROUTINES ********** */

void
print_hstable_entry(hstable_entry_t *ent,int n)
{
  int i,f;
  if (n==-1) printf("You: ");
  else printf("%4d ",n+1);
  f=1;
  for (i=0;i<20;i++) {
    if (f&&(ent->name[i]==0)) f=0;
    if (f) {
      if (ent->name[i]=='_') printf(" ");
      else printf("%c",ent->name[i]);
    } else printf(" ");
  }
  printf("%4d%6ld%3lu'%02lu\n",ent->level,ent->score,
	 ent->dur/60000,(ent->dur/1000)%60);
}

void
show_hstable(void)
     /* Show high score table and quit */
{
  hstable_entry_t yours;
  FILE *hsf;
  int i,replace,place;
  struct passwd *pwent;
  char hsedit;
  if (trainingmode) exit(0);
  yours.uid = getuid();
  pwent=getpwuid(yours.uid);
  strncpy(yours.name,pwent->pw_gecos,1000);
  yours.name[1000]=0;
  for (i=0;i<1000;i++) {
    if (yours.name[i]==' ') yours.name[i]='_';
    if (yours.name[i]==',') yours.name[i]=0;
  }
  yours.level=level;
  yours.score=score;
  yours.dur=current_time*BASETIME;
  printf("     Name                Lev.Score Time\n");
  print_hstable_entry(&yours,-1);
  seteuid(euid);
  setegid(egid);
  hsedit=1;
  hsf = fopen(SCOREFILE,"r+");
  if (hsf==NULL) {
    hsf = fopen(SCOREFILE,"w+");
    if (hsf==NULL) {
      hsedit=0;
      hsf = fopen(SCOREFILE,"r");
      if (hsf==NULL) {
	perror("Can't open high score file");
	exit(1);
      }
    }
  }
  rewind(hsf);
  for (i=0;i<SCORELEN;i++) {
    unsigned long x;
    if (fscanf(hsf,"%s %lu %d %ld %lu\n",hstable[i].name,
	       &x,&hstable[i].level,&hstable[i].score,
	       &hstable[i].dur)==5) {
      /* Beware: the above lines contain a buffer overrun.  Don't mess
       * up with the high score tables!  In particular, if the program
       * is made suid or sgid, only people with the corresponding uid
       * or gid should be allowed to modify the table. */
      hstable[i].uid=x;
    } else {
      strcpy(hstable[i].name,"Joe_Player");
      hstable[i].uid=9999;
      hstable[i].level=0;
      hstable[i].score=0;
      hstable[i].dur=0;
    }
  }
#if MULTENT
  replace=SCORELEN-1;
#else
  for (replace=0;replace<SCORELEN-1;replace++) {
    if (hstable[replace].uid==yours.uid)
      break;
  }
#endif
  if (scorecounts&&(yours.score>hstable[replace].score)) {
    printf("You made it into the high score file!\n");
    for (place=0;place<SCORELEN;place++) {
      if (yours.score>hstable[place].score)
	break;
    }
    for (i=replace;i>place;i--) {
      hstable[i]=hstable[i-1];
    }
    hstable[place]=yours;
  }
  printf("HIGH SCORES:\n");
  for (i=0;i<SCORELEN;i++)
    print_hstable_entry(hstable+i,i);
  if (hsedit) {
    rewind(hsf);
    for (i=0;i<SCORELEN;i++) {
      fprintf(hsf,"%s %lu %d %ld %lu\n",hstable[i].name,
	      (long)hstable[i].uid,hstable[i].level,hstable[i].score,
	      hstable[i].dur);
    }
  }
  fclose(hsf);
  exit(0);
}

/* ********** SMALL GAME ROUTINES ********** */

long
level_speed(int lev)
     /* Return the base descent rate (in millipixels per time unit)
      * for the level. */
{
  return 900+80*lev;
}

long
level_rate(int lev)
     /* How long to wait before to drop a new word on the given
      * level */
{
  int l;
  l = ((lev<20)?lev:20);
  return 10+(50000-l*2500)/level_speed(lev);
}

void
show_lives(void)
     /* Update the lives label. */
{
  char s[256];
  sprintf(s,"Lives: %d",lives);
  XtVaSetValues(lives_label,
		XtNlabel,s,
		NULL);
}

void
show_score(void)
     /* Update the score label. */
{
  char s[256];
  if (vpoints>0)
    sprintf(s,"Score: %ld (+%ld)",score,vpoints);
  else
    sprintf(s,"Score: %ld",score);
  XtVaSetValues(score_label,
		XtNlabel,s,
		NULL);
}

void
show_level(void)
     /* Update the level label. */
{
  char s[256];
  sprintf(s,"Level: %d",level);
  XtVaSetValues(level_label,
		XtNlabel,s,
		NULL);
}

void
add_score(long n)
     /* Increase player's score. */
{
  long newscore;
  long k;
  k=n;
  if (vpoints<k) k=vpoints;
  newscore=score+n+k;
  vpoints-=k;
  levpoints+=n;
  if ((newscore/2500)>(score/2500)) {
    lives++;
    show_lives();
  }
  score=newscore;
  show_score();
}

void
lose_life(void)
     /* Remove a life from the player. */
{
  if (lives==1) {
    show_hstable();
  }
  lives--;
  show_lives();
}

void expose_proc(Widget w,XtPointer client_data,XEvent *evt,Boolean *cnt);

void
clear_all(void)
     /* Remove all falling words. */
{
  int i;
  Boolean garbage;
  for (i=0;i<nb_words;i++)
    free(words[i].s);
  nb_words=0;
  expose_proc(game_space,NULL,NULL,&garbage);
}

void
new_level(char nobonus)
     /* Switch to next level. */
{
  if (lifemode) clear_all();
  if (bonuslevel||nobonus||!lifemode) {
    level++;
    bonuslevel=0;
    levpoints=0;
    show_level();
  } else bonuslevel=1;
  level_time=current_time;
  lastword_time=-level_rate(level);
}

void
draw_word(word_state_t *w)
     /* Base routine for drawing a word - or erasing it. */
{
  XDrawString(dpy,win,gc1,w->x,w->y/1000,
	      w->s,w->typed);
  XDrawString(dpy,win,gc2,w->x+w->twidth,w->y/1000,
	      w->s+w->typed,w->len-w->typed);
}

void timer_proc(XtPointer client_data,XtIntervalId *id);

void
retime(void)
     /* Make sure the game is running. */
{
  paused = 0;
  timeout = XtAppAddTimeOut(app_context,BASETIME,timer_proc,NULL);
}

void
resume(void)
     /* Resume the game if it was paused. */
{
  if (paused)
    retime();
}

void
pauze(void)
     /* Pause the game if it wasn't. */
{
  if (!paused) {
    XtRemoveTimeOut(timeout);
    paused = 1;
  }
}

/* ********** MAIN GAME ROUTINES ********** */

void
quit_callback(Widget w,XtPointer client_data,XtPointer call_data)
     /* Callback called by the quit button */
{
  show_hstable();
}

void
pause_callback(Widget w,XtPointer client_data,XtPointer call_data)
     /* Callback called by the pause button */
{
  char b;
  XtVaGetValues(w,
		XtNstate,&b,
		NULL);
  if (b)
    pauze();
  else
    resume();
}

int in_dict(const char *s);

void
send_callback(Widget w,XtPointer client_data,XtPointer call_data)
     /* Callback called by the send button */
{
  if (in_dict(deathword) == 2)
    {
      printf("%s\n",deathword);
      fflush(stdout);
      /* this block may be inside or out the if.
       * inside = unfinished word not sent, stays
       * outside = unfinished word silently disapears (not sent)
       */
      deathword[0]=0;
      deathln=0;
      in_dict(deathword); /* Reset dictionary pointer */
      XtVaSetValues(type_space,
		    XtNlabel,deathword,
		    XtNwidth,WIDTH,
		    NULL);
      /* This is the end of the block */
    }
}

void
next_callback(Widget w,XtPointer client_data,XtPointer call_data)
     /* Callback called by the next button */
{
  int i;
  long n;
  if (!bonuslevel)
    for (i=0;i<nb_words;i++)
      if (words[i].y>=HEIGHT*200) {
	XBell(dpy,0);
	return;
      }
  n=350-levpoints*2;
  if (n<0) n=0;
  vpoints+=n;
  show_score();
  new_level(1);
}

void
x_resume(void)
     /* Resume the game, untoggling the Pause button if necessary. */
{
  if (!paused) return;
  XtVaSetValues(pause_button,
		XtNstate,False,
		NULL);
  resume();
}

/* returns 0 not existant, 1 possible, 2 match */
/* reset by *s = 0 */
int
in_dict(const char *s)
{
  static off_t curpos;

  if (*s == 0)
    {
      curpos = 0;
      return 1;
    }

  while (strncmp(wordf+curpos,s,strlen(s)) != 0)
    while (wordf[curpos++] != '\n')
      if (curpos >= wordf_len)
	{
	  curpos = 0;
	  return 0;
	}
  if (wordf[curpos+strlen(s)] == '\n') return 2;
  return 1;
}

char *
find_new_word(void)
     /* Return a newly malloc()ated word to let fall. */
{
  int i;
  int len;
  char *s;
  if ((!trainingmode)&&bonuslevel)
    {
      len = random()%5+5;
      s = malloc((len+1)*sizeof(char));
      if (s==NULL)
	{
	  perror("Out of memory");
	  exit(1);
	}
      for (i=0;i<len;i++) s[i] = 33+random()%94;
      s[len] = 0;
    }
  else
    {
      off_t curpos; /* position in wordfile */
      char tmp[MAX_WORDSIZE+1];
      
      do curpos = random()%wordf_len; while (wordf[curpos]!='\n');
      while ((curpos>=0)&&(wordf[curpos-1]!='\n')) curpos--;
      for (i=0;(i<MAX_WORDSIZE)&&((wordf[curpos+i])!='\n');i++)
	tmp[i] = wordf[curpos+i];
      tmp[i] = 0;
      len = strlen(tmp);
      s = malloc((len+1)*sizeof(char));
      if (s==NULL)
	{
	  perror("Out of memory");
	  exit(1);
	}
      memcpy(s,tmp,len);
      s[len] = 0;
    }
  return s;
}

void
enter_word(char *s)
     /* Get the given word falling.  The string s MUST have been
      * returned by malloc(). */
{
  if (nb_words>=MAXWORDS) { free(s); return; }
  words[nb_words].s=s;
  words[nb_words].len=strlen(words[nb_words].s);
  if (words[nb_words].len==0) { free(s); return; }
  words[nb_words].typed = 0;
  {
    int dir_ret,f_a_ret,f_d_ret;
    XCharStruct overall;
    XTextExtents(fnt_info,words[nb_words].s,words[nb_words].len,
		 &dir_ret,&f_a_ret,&f_d_ret,&overall);
    words[nb_words].width = overall.width;
  }
  words[nb_words].twidth = 0;
  if (words[nb_words].width<WIDTH)
    words[nb_words].x = random()%(WIDTH-words[nb_words].width);
  else words[nb_words].x = 0;
  if (trainingmode) {
    words[nb_words].y = (fnt_info->ascent
      +random()%(HEIGHT-(fnt_info->ascent+fnt_info->descent)))*1000;
  } else {
#if NOVCOLLIDE
    {
      int i;
      long y;
      long sep;
      sep=(fnt_info->ascent+fnt_info->descent)*1000;
      y=0;
      for (i=0;i<nb_words;i++) {
	if (y>words[i].y-sep)
	  y=words[i].y-sep;
      }
      words[nb_words].y = y;
    }
#else
    words[nb_words].y = 0;
#endif
  }
  if (trainingmode) {
    words[nb_words].rate = 0;
  } else {
    words[nb_words].rate = level_speed(level)*7/(2+words[nb_words].len);
#if NOCOLLIDE
    {
      int i;
      int left;
      int right;
      char redo;
      left=words[nb_words].x;
      right=words[nb_words].x+words[nb_words].width;
      do {
	redo=0;
	for (i=0;i<nb_words;i++) {
	  if ((words[i].x<right)&&(words[i].x+words[i].width>=left)) {
	    if (words[i].rate<words[nb_words].rate)
	      words[i].rate=words[nb_words].rate;
	    if (words[i].x<left) {
	      left=words[i].x;
	      redo=1;
	    }
	    if (words[i].x+words[i].width>right) {
	      right=words[i].x+words[i].width;
	      redo=1;
	    }
	  }
	}
      } while (redo);
    }
#endif
  }
  draw_word(words+nb_words);
  lastword_time=current_time;
  nb_words++;
}

void
new_word(void)
{
  enter_word(find_new_word());
}

void
calc_twidth(word_state_t *w)
     /* Calculate the twidth field of the word state structure. */
{
  int dir_ret,f_a_ret,f_d_ret;
  XCharStruct overall;
  XTextExtents(fnt_info,w->s,w->typed,
	       &dir_ret,&f_a_ret,&f_d_ret,&overall);
  w->twidth = overall.width;
}

char
key_typed(word_state_t *w)
     /* Mark a new correct key as typed in the given word. */
{
  w->typed++;
  calc_twidth(w);
  return (w->typed>=w->len);
}

void
key_untyped(word_state_t *w,char ch)
     /* Determine what happens when a wrong key has been pressed. */
{
#if CONTEXT
  int t0,i;
  t0=w->typed;
  for (;w->typed>=0;w->typed--) {
    if (w->s[w->typed]!=ch)
      continue;
    for (i=0;i<w->typed;i++)
      if (w->s[w->typed-1-i]!=w->s[t0-1-i])
	break;
    if (i>=w->typed)
      break;
    /* Sometimes I think I should participate in IOCCC's... */
  }
  w->typed++;
  calc_twidth(w);
#else
  w->typed = 0;
  w->twidth = 0;
#endif
}

void
input_proc(XtPointer client_data,int *source,XtInputId *id)
     /* Called when the standard input has something to say */
{
  char tmp[MAX_WORDSIZE+1];
  int len;
  char *s;

  fgets(tmp,MAX_WORDSIZE+1,stdin);
  for (len=0;(((unsigned char)tmp[len])>=33)
	 &&(((unsigned char)tmp[len])<=127);len++);
  tmp[len]=0;
  /* len = strlen(tmp); */
  s = malloc((len+1)*sizeof(char));
  if (s==NULL)
    {
      perror("Out of memory");
      exit(1);
    }
  memcpy(s,tmp,len);
  s[len] = 0;
  enter_word(s);
}

void
timer_proc(XtPointer client_data,XtIntervalId *id)
     /* Timeout handler called every time unit (when the game is
      * unpaused) */
{
  int i;
  current_time++;
  if (current_time>=level_time+LEVELDUR)
    new_level(0);
  for (i=0;i<nb_words;i++) {
    word_state_t w;
    w = words[i];
    words[i].y+=words[i].rate;
    if (words[i].y>HEIGHT*1000) {
      if (!bonuslevel) {
	int j;
	nb_words--;
	for (j=i;j<nb_words;j++)
	  words[j]=words[j+1];
	i--; /* YUCK ALERT */
	draw_word(&w);
	free(w.s);
	lose_life();
      } else {
	new_level(0);
	break;
      }
    } else {
      draw_word(words+i);
      draw_word(&w);
    }
  }
  if (lifemode&&(current_time>=lastword_time+level_rate(level))) {
    new_word();
  }
  retime();
}

#define KBD_1BUF_SIZE 256 /* Rather ridiculous: 1 should be enough */

void
key_proc(Widget w,XtPointer client_data,XEvent *evt,Boolean *cnt)
     /* Event handler called when a key is pressed */
{
  int i;
  char buf[KBD_1BUF_SIZE];
  int buflen;
  int p;
  KeySym keysym;
  Boolean mode;
  buflen=XLookupString(&evt->xkey,buf,KBD_1BUF_SIZE,&keysym,NULL);
  for (p=0;p<buflen;p++) {
    if ((buf[p]<=32)||(buf[p]>=127))
      continue;
    if (!trainingmode) x_resume();
    if (deathmode)
      XtVaGetValues(mode_button,XtNstate,&mode,NULL);
    else mode=0;
    if (mode) {
      if (deathln<MAX_WORDSIZE) {
	deathword[deathln]=buf[p];
	deathword[deathln+1]=0;
	deathln++;
	if (in_dict(deathword) == 0)
	  {
	    deathword[0] = 0;
	    deathln = 0;
	    in_dict(deathword);
	  }
	XtVaSetValues(type_space,
		      XtNlabel,deathword,
		      XtNwidth,WIDTH,
		      NULL);
      }
    } else {
      for (i=0;i<nb_words;i++) {
	word_state_t w;
	w = words[i];
	if (words[i].s[words[i].typed]==buf[p]) {
	  if (key_typed(words+i)) {
	    int j;
	    nb_words--;
	    for (j=i;j<nb_words;j++)
	      words[j]=words[j+1];
	    i--; /* YUCK ALERT */
	    draw_word(&w);
	    free(w.s);
	    if (!trainingmode)
	      add_score((w.y<HEIGHT*200)?8+w.len:5+w.len);
	    else while (nb_words==0) new_word();
	  } else {
	    draw_word(words+i);
	    draw_word(&w);
	  }
	} else {
	  key_untyped(words+i,buf[p]);
	  draw_word(words+i);
	  draw_word(&w);
	}
      }
    }
  }
  *cnt=True;
}

void
expose_proc(Widget w,XtPointer client_data,XEvent *evt,Boolean *cnt)
     /* Event handler called when an Expose event is received */
{
  int i;
  XFillRectangle(dpy,win,gc0,0,0,WIDTH,HEIGHT);
  for (i=0;i<nb_words;i++)
    draw_word(words+i);
  *cnt=False;
}

/* Open the word file. */
char *
wordopen(const char *wordfile, const char *wordpath)
{
  int word_fd;
  char *wordf;  
  char tmp1[512], tmp2[512], *p;
  
  strcpy(tmp1,wordpath);
  p=strtok(tmp1,":");
  word_fd=-1;
  errno = 0;

  while(p) {
    if (p[0]=='~')
      sprintf(tmp2,"%s/%s/%s",getenv("HOME"),&p[1],wordfile);
    else
      sprintf(tmp2,"%s/%s",p,wordfile);
	fprintf(stderr,"opening [%s]\n",tmp2);
    word_fd=open(tmp2,O_RDONLY,0);
    if (word_fd >= 0)
      break;
    p=strtok(0,":");
  }

  if (word_fd < 0)
    {
      fprintf(stderr,"no %s file in %s\n",wordfile,wordpath);
      perror(wordfile);
      exit(1);
    }
  wordf_len = lseek(word_fd,0,SEEK_END);
  if (wordf_len < 0)
    {
      perror(wordfile);
      exit(1);
    }
  wordf = (char *) mmap (NULL,wordf_len,PROT_READ,MAP_SHARED,word_fd,0);
  if (wordf == MAP_FAILED)
    {
      perror(wordfile);
      exit(1);
    }
  close(word_fd);

  return wordf;
}

/* The main function (yeah, this is too long) */

int
main(int argc,char **argv)
{
  /* Save uid and gid for later */
  euid = geteuid();
  egid = getegid();
  seteuid(getuid());
  setegid(getgid());
  /* Randomize. */
  srandom(time(NULL));
  wordf = wordopen(WORDFILE,WORDPATH);
  /* Read application resources. */
  toplevel = XtVaAppInitialize(&app_context,APP_CLASS,
			       options,XtNumber(options),
			       &argc,argv,fallback_resources,
			       NULL);
  if (argc>1) {
    /* We don't deal properly with GNU long options, but I don't
     * intend to link with GNU getopt simply so people are able to
     * type --vers instead of --version */
    if ((strcmp(argv[1],"-version")==0)||(strcmp(argv[1],"-v")==0)
	||(strcmp(argv[1],"-V")==0)
	||(strcmp(argv[1],"--version")==0)||(strcmp(argv[1],"--ver")==0)) {
      fprintf(stderr,
	      "xletters - catch falling words\n"
	      "Version 1.1.0\n"
	      "Copyright (C) 1998 by\n"
	      "  Peter Horvai (peter.horvai@ens.fr)\n"
	      "  David A. Madore (david.madore@ens.fr)\n"
	      "xletters comes with ABSOLUTELY NO WARRANTY:\n"
	      "see the GNU general public license for more details.\n"
	      );
      exit(0);
    } else if ((strcmp(argv[1],"-help")==0)||(strcmp(argv[1],"-h")==0)
	       ||(strcmp(argv[1],"--help")==0)) {
      fprintf(stderr,
	      "Usage is:\n"
	      "%s [-version] [-help] [-bg color] [-fg color] [-fn font]\n"
	      "[-name name] [-title string] [-geometry geom] [-display disp]\n"
	      "[-xrm resource] [-wfn font] [-wc color] [-tc color] [-gbg color]\n"
	      "[-notrain] [-train] [-nodeath] [-death] [-duel]\n",
	      argv[0]);
      exit(0);
    } else {
      fprintf(stderr,"Unrecognized option %s\n",argv[1]);
      fprintf(stderr,"Type %s -help for help\n",argv[0]);
      exit(1);
    }
  }
  XtGetApplicationResources(toplevel,&app_res_val,app_res_tab,
			    XtNumber(app_res_tab),NULL,0);
  if (strcmp(app_res_val.death_mode,"normal")==0) {
    deathmode=0;
    lifemode=1;
  } else if (strcmp(app_res_val.death_mode,"death")==0) {
    deathmode=1;
    lifemode=1;
  } else if (strcmp(app_res_val.death_mode,"duel")==0) {
    deathmode=1;
    lifemode=0;
  } else {
    fprintf(stderr,"Unrecognized deathmode %s\n",app_res_val.death_mode);
    exit(1);
  }
  trainingmode = app_res_val.training_mode;
  if (trainingmode) { deathmode=0; lifemode=0; }
  scorecounts = !(deathmode||trainingmode);
  /* Create widgets. */
  ground_box = XtVaCreateManagedWidget("groundBox",boxWidgetClass,toplevel,
				       NULL);
  label = XtVaCreateManagedWidget("label",labelWidgetClass,ground_box,
				  XtNlabel,"xletters",
				  NULL);
  quit_button = XtVaCreateManagedWidget("quitButton",commandWidgetClass,
					ground_box,
					NULL);
  if (!trainingmode)
    pause_button = XtVaCreateManagedWidget("pauseButton",toggleWidgetClass,
					   ground_box,
					   NULL);
  if (deathmode)
    mode_button = XtVaCreateManagedWidget("modeButton",toggleWidgetClass,
					  ground_box,
					  NULL);
  if (deathmode)
    send_button = XtVaCreateManagedWidget("sendButton",commandWidgetClass,
					  ground_box,
					  NULL);
  if (lifemode)
    next_button = XtVaCreateManagedWidget("nextButton",commandWidgetClass,
					  ground_box,
					  NULL);
  if (!trainingmode) {
    lives_label = XtVaCreateManagedWidget("livesLabel",labelWidgetClass,
					  ground_box,
					  NULL);
    score_label = XtVaCreateManagedWidget("scoreLabel",labelWidgetClass,
					  ground_box,
					  NULL);
    level_label = XtVaCreateManagedWidget("levelLabel",labelWidgetClass,
					  ground_box,
					  NULL);
  }
  if (deathmode)
    type_space = XtVaCreateManagedWidget("typeSpace",labelWidgetClass,
					 ground_box,
					 XtNwidth,WIDTH,
					 XtNlabel,"",
					 NULL);
  game_space = XtVaCreateManagedWidget("gameSpace",coreWidgetClass,ground_box,
				       XtNwidth,WIDTH,
				       XtNheight,HEIGHT,
				       NULL);
  XtRealizeWidget(toplevel);
  /* Xlib miscellanea */
  dpy = XtDisplay(game_space);
  win = XtWindow(game_space);
  gc0 = XCreateGC(dpy,win,0,NULL);
  {
    XGCValues val;
    val.function = GXxor;
    gc1 = XCreateGC(dpy,win,GCFunction,&val);
    gc2 = XCreateGC(dpy,win,GCFunction,&val);
  }
  {
    Pixel bgc,c1,c2;
    XtVaGetValues(game_space,
		  XtNbackground,&bgc,
		  NULL);
    XSetForeground(dpy,gc0,bgc);
    c1 = app_res_val.typed_color;
    c2 = app_res_val.word_color;
    XSetForeground(dpy,gc1,bgc^c1);
    XSetForeground(dpy,gc2,bgc^c2);
  }
  fnt_info = app_res_val.word_font;
  fnt = fnt_info->fid;
  XSetFont(dpy,gc0,fnt);
  XSetFont(dpy,gc1,fnt);
  XSetFont(dpy,gc2,fnt);
  /* Setup callbacks and handlers */
  if (!trainingmode)
    retime();
  if (deathmode)
    XtAppAddInput(app_context,0,(XtPointer)XtInputReadMask,input_proc,NULL);
  XtAddCallback(quit_button,XtNcallback,quit_callback,NULL);
  if (!trainingmode)
    XtAddCallback(pause_button,XtNcallback,pause_callback,NULL);
  if (deathmode)
    XtAddCallback(send_button,XtNcallback,send_callback,NULL);
  if (lifemode)
    XtAddCallback(next_button,XtNcallback,next_callback,NULL);
  XtAddEventHandler(game_space,KeyPressMask,False,key_proc,NULL);
  if (deathmode)
    XtAddEventHandler(type_space,KeyPressMask,False,key_proc,NULL);
  XtAddEventHandler(game_space,ExposureMask,False,expose_proc,NULL);
  XtInstallAllAccelerators(game_space,ground_box);
  if (deathmode)
    XtInstallAllAccelerators(type_space,ground_box);
  XtInstallAllAccelerators(ground_box,ground_box);
  /* Etcetera */
  if (!trainingmode) {
    lives = 5;
    score = 0;
    vpoints = 0;
    level = 1;
    bonuslevel = 0;
    lastword_time = -level_rate(level);
    show_lives();
    show_score();
    show_level();
  } else while (nb_words==0) new_word();
  XtAppMainLoop(app_context);
  return 0;
}
