/*
 * Copyright (C) 2000-2025 the xine project
 *
 * This file is part of xine, a unix video player.
 *
 * xine is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * xine is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110, USA
 *
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdarg.h>
#include <string.h>

#include "_xitk.h"
#include "menu.h"
#include "label.h"
#include "labelbutton.h"
#include "button.h"
#include "font.h"
#include "backend.h"

#define TITLE_BAR_HEIGHT 20

/*
 * Dialog window
 */

typedef struct xitk_dialog_s xitk_dialog_t;
typedef struct _xitk_dialog_menu_s _xitk_dialog_menu_t;

typedef enum {
  _W_w1 = 0,
  _W_w2,
  _W_w3,
  _W_checkbox,
  _W_checklabel,
  _W_default,
  _W_LAST
} _W_t;
  
struct xitk_dialog_s {
  xitk_t                 *xitk;
  xitk_window_t          *xwin;
  xitk_register_key_t     key;

  unsigned int            state;
  xitk_widget_t          *w[_W_LAST];

  void                  (*done3cb)(void *userdata, int state);
  void                   *done3data;

  _xitk_dialog_menu_t    *menu;
  unsigned int            mlen;
  char                    msg[1];
};

struct _xitk_dialog_menu_s {
  xitk_dialog_t *wd;
  int refs;
};

static void _xitk_dialog_menu_action (xitk_widget_t *w, xitk_menu_entry_t *me, void *data) {
  _xitk_dialog_menu_t *menu = data;

  (void)w;
  /* paranoia */
  if (!menu)
    return;

  if (menu->wd && (menu->wd->menu == menu)) {
    if (me) {
      if (me->user_id == 0) {
        xitk_clipboard_set_text (menu->wd->w[_W_default], menu->wd->msg, menu->wd->mlen);
      }
    }
  }
  if (!me && --menu->refs == 0)
    free (menu);
}

static int _xitk_dialog_menu_open (xitk_dialog_t *wd, int x, int y) {
  if (!wd->menu) {
    wd->menu = malloc (sizeof (*wd->menu));
    if (!wd->menu)
      return 0;
    wd->menu->wd = wd;
    wd->menu->refs = 1;
  }

  if (wd->menu) {
    char buf[1][16];
    const char *cstr = _("Ctrl");
    size_t clen = xitk_find_byte (cstr, 0);

    if (clen > 12)
      cstr = "Ctrl", clen = 4;
    memcpy (buf[0], cstr, clen); memcpy (buf[0] + clen, "-C", 3);

    {
      xitk_menu_entry_t menu_entries[] = {
        { XITK_MENU_ENTRY_PLAIN, 0, _("Copy"), buf[0] },
        { XITK_MENU_ENTRY_END,   0, NULL,      NULL }
      };
      xitk_menu_widget_t menu = {
        .nw = {
          .wl = wd->w[_W_default]->wl,
          .userdata = wd->menu
        },
        .cb = _xitk_dialog_menu_action,
        .menu_tree = &menu_entries[0]
      };
      xitk_widget_t *w = xitk_noskin_menu_create (&menu, wd->w[_W_default]->wl->win.x + x, wd->w[_W_default]->wl->win.y + y);

      if (!w)
        return 0;
      wd->menu->refs++;
      xitk_menu_show_menu (w);
      return 1;
    }
  }
  return 0;
}

static void _xitk_dialog_destr (void *data) {
  xitk_dialog_t *wd = data;

  xitk_window_destroy_window (wd->xwin);
  XITK_FREE (wd);
}

static void _xitk_dialog_done (xitk_widget_t *w, void *data, int state, unsigned int modifier) {
  xitk_dialog_t *wd = data;
  xitk_register_key_t key = wd->key;

  (void)state;
  (void)modifier;

  if (wd->done3cb)
    wd->done3cb (wd->done3data, xitk_widget_user_id (w) | wd->state);

  /* this will _xitk_dialog_destr (wd). */
  xitk_unregister_event_handler (wd->xitk, &key);
}

/*
 * Local XEvent handling.
 */
static int _xitk_dialog_event (void *data, const xitk_be_event_t *e) {
  xitk_dialog_t *wd = data;

  if (wd->w[_W_default]) {
    switch (e->type) {
      case XITK_EV_KEY_DOWN:
        if (e->utf8[0] == (0x1f & 'c')) {
          xitk_clipboard_set_text (wd->w[_W_default], wd->msg, wd->mlen);
          return 1;
        }
        /* fall through */
      case XITK_EV_KEY_UP:
        if (e->utf8[0] == XITK_CTRL_KEY_PREFIX) {
          if (e->utf8[1] == XITK_KEY_ESCAPE)
            return xitk_widget_key_event (wd->w[_W_default], e, 1);
          if (e->utf8[1] == XITK_KEY_MENU)
            return _xitk_dialog_menu_open (wd, 10, 10);
        }
        break;
      case XITK_EV_BUTTON_DOWN:
        if (e->code == 3)
          return _xitk_dialog_menu_open (wd, e->x, e->y);
        break;
      case XITK_EV_DEL_WIN:
        _xitk_dialog_done (wd->w[_W_default], wd, 0, e->qual);
        return 1;
      case XITK_EV_EXPOSE:
        xitk_set_focus_to_widget (wd->w[_W_default]);
        return 1;
      default: ;
    }
  }
  return 0;
}

static const char *_xitk_dialog_label (const char *label) {
  switch ((uintptr_t)label) {
    case (uintptr_t)XITK_LABEL_OK: return _("OK");
    case (uintptr_t)XITK_LABEL_NO: return _("No");
    case (uintptr_t)XITK_LABEL_YES: return _("Yes");
    case (uintptr_t)XITK_LABEL_CANCEL: return _("Cancel");
    default: return label;
  }
}

static void _xitk_dialog_check (xitk_widget_t *w, void *data, int state, unsigned int modifier) {
  xitk_dialog_t *wp = data;

  (void)w;
  (void)modifier;
  wp->state = state ? XITK_WINDOW_DIALOG_CHECKED : 0;
}

xitk_register_key_t xitk_window_dialog_3 (xitk_t *xitk, xitk_window_t *transient_for, int layer_above,
  int width, const char *title,
  void (*done_cb)(void *userdata, int state), void *userdata,
  const char *button1_label, const char *button2_label, const char *button3_label,
  const char *check_label, int checked,
  int text_align, const char *text_fmt, ...) {
  xitk_dialog_t *wd;
  xitk_widget_list_t *widget_list;
  int num_buttons = !!button1_label + !!button2_label +!!button3_label;
  int winw = width, winh = TITLE_BAR_HEIGHT + 40;

  if (!xitk)
    return 0;
  if (num_buttons)
    winh += 50;
  if (check_label)
    winh += 50;

  {
    char buf[4096];
    const char *text;
    xitk_image_t *ti, *bg;
    xitk_window_t *xwin;
    unsigned int mlen;

    if (!title)
      title = (num_buttons < 2) ? _("Notice") : _("Question?");

    if (text_fmt) {
      va_list args;
      va_start (args, text_fmt);
      if (!strcmp (text_fmt, "%s")) {
        text = va_arg (args, const char *);
        mlen = xitk_find_byte (text, 0);
      } else {
        mlen = vsnprintf (buf, sizeof (buf), text_fmt, args);
        text = buf;
      }
      va_end (args);
    } else {
      text = "??";
      mlen = 3;
    }
    ti = xitk_image_from_string (xitk, DEFAULT_FONT_12, winw - 40, text_align, text);
    if (!ti)
      return 0;
    winh += ti->height;

    xwin = xitk_window_create_window_ext (xitk, XITK_INT_KEEP, XITK_INT_KEEP, winw, winh,
      title, NULL, NULL, 0, 0, NULL, XITK_WINDOW_BG_FRAME);
    if (!xwin) {
      xitk_image_free_image (&ti);
      return 0;
    }

    bg = xitk_window_get_background_image (xwin);
    if (bg) {
      xitk_image_copy_rect (ti, bg, 0, 0, ti->width, ti->height, (winw - ti->width) >> 1, TITLE_BAR_HEIGHT + 20);
      xitk_window_set_background_image (xwin, bg);
    }
    xitk_image_free_image (&ti);

    wd = calloc (1, sizeof (*wd) + mlen);
    if (!wd) {
      xitk_window_destroy_window (xwin);
      return 0;
    }
    wd->xitk = xitk;
    wd->xwin = xwin;
    wd->mlen = mlen;
    memcpy (wd->msg, text, mlen);
  }

  if (xitk_init_NULL ()) {
    unsigned int u;
    for (u = 0; u < _W_default; u++)
      wd->w[u] = NULL;
  }

  widget_list = xitk_window_widget_list (wd->xwin);

  wd->done3cb = done_cb;
  wd->done3data = userdata;
  wd->state = checked ? XITK_WINDOW_DIALOG_CHECKED : 0;
  if (check_label) {
    int x = 25, y = winh - (num_buttons ? 50 : 0) - 50;
    xitk_button_widget_t b = {
      .nw = {
        .wl            = widget_list,
        .add_state     = checked ? (XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE | XITK_WIDGET_STATE_ON)
                                 : (XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE),
      },
      .symbol          = XITK_SYMBOL_CHECK,
      .state_callback  = _xitk_dialog_check
    };
    xitk_label_widget_t lbl = {
      .nw = {
        .wl        = widget_list,
        .add_state = XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE
      },
      .label    = check_label
    };

    wd->w[_W_checkbox] = xitk_noskin_button_create (&b, x, y + 5, 10, 10);
    wd->w[_W_checklabel] = xitk_noskin_label_create (&lbl, x + 15, y, winw - x - 40, 20, DEFAULT_FONT_12);
    xitk_widget_set_focus_redirect (wd->w[_W_checklabel], wd->w[_W_checkbox]);
  }

  if (num_buttons) {
    xitk_labelbutton_widget_t lb = {
      .nw = {
        .wl        = widget_list,
        .add_state = XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE,
        .userdata  = wd
      },
      .button_type = CLICK_BUTTON,
      .align       = ALIGN_CENTER,
      .callback    = _xitk_dialog_done
    };
    int bx, bdx, by, bwidth = 150;

    if (bwidth > (winw - 8) / num_buttons)
      bwidth = (winw - 8) / num_buttons;
    bx = (winw - bwidth * num_buttons) / (num_buttons + 1);
    bdx = bx + bwidth;
    by = winh - 50;

    if (button1_label) {
      lb.nw.user_id = 1;
      lb.label = _xitk_dialog_label (button1_label);
      wd->w[_W_w1] = xitk_noskin_labelbutton_create (&lb,
        bx, by, bwidth, 30, XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_INV, DEFAULT_BOLD_FONT_12);
      bx += bdx;
    }

    if (button2_label) {
      lb.nw.user_id = 2;
      lb.label = _xitk_dialog_label (button2_label);
      wd->w[_W_w2] = xitk_noskin_labelbutton_create (&lb,
        bx, by, bwidth, 30, XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_INV, DEFAULT_BOLD_FONT_12);
      bx += bdx;
    }

    if (button3_label) {
      lb.nw.user_id = 3;
      lb.label = _xitk_dialog_label (button3_label);
      wd->w[_W_w3] = xitk_noskin_labelbutton_create (&lb,
        bx, by, bwidth, 30, XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_INV, DEFAULT_BOLD_FONT_12);
    }
  }
  wd->w[_W_default] = wd->w[_W_w3];
  if (!wd->w[_W_default]) {
    wd->w[_W_default] = wd->w[_W_w1];
    if (!wd->w[_W_default])
      wd->w[_W_default] = wd->w[_W_w2];
  }

  xitk_window_flags (wd->xwin, XITK_WINF_VISIBLE | XITK_WINF_ICONIFIED, XITK_WINF_VISIBLE);
  xitk_window_raise_window (wd->xwin);

  xitk_widgets_state (wd->w, _W_LAST, XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE, ~0u);

  if (layer_above)
    xitk_window_set_window_layer (wd->xwin, layer_above);
  xitk_window_set_input_focus (wd->xwin);

  wd->key = xitk_be_register_event_handler ("xitk_dialog_3", wd->xwin, _xitk_dialog_event, wd, _xitk_dialog_destr, wd);
  if (transient_for)
    xitk_window_set_transient_for_win (wd->xwin, transient_for);

  xitk_widget_list_defferred_destroy (widget_list);
  return wd->key;
}
