//============================================================================
//  The contents of this file are covered by the Viskores license. See
//  LICENSE.txt for details.
//
//  By contributing to this file, all contributors agree to the Developer
//  Certificate of Origin Version 1.1 (DCO 1.1) as stated in DCO.txt.
//============================================================================

//============================================================================
//  Copyright (c) Kitware, Inc.
//  All rights reserved.
//  See LICENSE.txt for details.
//
//  This software is distributed WITHOUT ANY WARRANTY; without even
//  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
//  PURPOSE.  See the above copyright notice for more information.
//============================================================================
#ifndef viskores_rendering_testing_Testing_h
#define viskores_rendering_testing_Testing_h


#include <viskores/cont/DataSet.h>
#include <viskores/cont/Error.h>
#include <viskores/cont/Logging.h>
#include <viskores/cont/RuntimeDeviceTracker.h>
#include <viskores/cont/testing/Testing.h>

#include <viskores/filter/image_processing/ImageDifference.h>
#include <viskores/internal/Configure.h>
#include <viskores/io/FileUtils.h>
#include <viskores/io/ImageUtils.h>

#include <viskores/rendering/View.h>

// Because the testing directory is reserved for test executables and not
// libraries, the viskores_rendering_testing module has to put this file in
// viskores/rendering/testlib instead of viskores/rendering/testing where you normally
// would expect it.
#include <viskores/rendering/testlib/viskores_rendering_testing_export.h>

#include <fstream>
#include <sstream>
#include <vector>

/// \brief Tests multiple image files against a provided view for differences.
///
/// This function tests multiple files provided via fileNames against the rendered
/// canvas generated by the provided view using the ImageDifference filter. If one
/// of the provided images is within the error threshold for the image difference
/// this function will return true. Otherwise the view is too different from the images
/// and this will return false with corresponding error messages.
///
/// This function will generate an image if the provided file is missing. If a file is
/// missing the image will be generated for that file and the test will continue.
///
VISKORES_RENDERING_TESTING_EXPORT TestEqualResult
test_equal_images(viskores::rendering::View& view,
                  const std::vector<std::string>& fileNames,
                  const viskores::IdComponent& averageRadius = 0,
                  const viskores::IdComponent& pixelShiftRadius = 0,
                  const viskores::FloatDefault& allowedPixelErrorRatio = 0.00025f,
                  const viskores::FloatDefault& threshold = 0.05f,
                  const bool& writeDiff = true,
                  const bool& returnOnPass = true);

/// \brief Tests multiple image files against a provided canvas for differences.
///
/// This function tests multiple files provided via fileNames against the rendered
/// canvas using the ImageDifference filter. If one of the provided images is within
/// the error threshold for the image difference this function will return true.
/// Otherwise the view is too different from the images and this will return false
/// with corresponding error messages.
///
/// The canvas must already be rendered when this is called.
///
/// This function will generate an image if the provided file is missing. If a file is
/// missing the image will be generated for that file and the test will continue.
///
VISKORES_RENDERING_TESTING_EXPORT TestEqualResult
test_equal_images(const viskores::rendering::Canvas& canvas,
                  const std::vector<std::string>& fileNames,
                  const viskores::IdComponent& averageRadius = 0,
                  const viskores::IdComponent& pixelShiftRadius = 0,
                  const viskores::FloatDefault& allowedPixelErrorRatio = 0.00025f,
                  const viskores::FloatDefault& threshold = 0.05f,
                  const bool& writeDiff = true,
                  const bool& returnOnPass = true);

/// \brief Tests multiple image files against a provided image for differences.
///
/// This function tests multiple files provided via fileNames against the provided
/// rendered image using the ImageDifference filter. If one of the provided images
/// is within the error threshold for the image difference this function will return
/// true. Otherwise the view is too different from the images and this will return
/// false with corresponding error messages.
///
/// The provided `DataSet` must contain a `CellSetStructured<2>` and its first field
/// must be the colors to compare.
///
/// This function will generate an image if the provided file is missing. If a file is
/// missing the image will be generated for that file and the test will continue.
///
VISKORES_RENDERING_TESTING_EXPORT TestEqualResult
test_equal_images(const viskores::cont::DataSet& generatedImage,
                  const std::vector<std::string>& fileNames,
                  const viskores::IdComponent& averageRadius = 0,
                  const viskores::IdComponent& pixelShiftRadius = 0,
                  const viskores::FloatDefault& allowedPixelErrorRatio = 0.00025f,
                  const viskores::FloatDefault& threshold = 0.05f,
                  const bool& writeDiff = true,
                  const bool& returnOnPass = true);

/// \brief Tests multiple image files against a provided image for differences.
///
/// This form of `test_equal_images` takes a single filename and searches for a list
/// of files that match that name or that name with numbers appended to it. It then
/// calls the equivalent `test_equal_images` with the list of files found.
///
/// The `ImageType` can be any type of object that other forms of `test_equal_images`
/// accept such as a `View`, a `Canvas`, or a `DataSet` containing an image.
///
template <typename ImageType>
inline TestEqualResult test_equal_images(
  ImageType&& image,
  const std::string& fileName,
  const viskores::IdComponent& averageRadius = 0,
  const viskores::IdComponent& pixelShiftRadius = 0,
  const viskores::FloatDefault& allowedPixelErrorRatio = 0.00025f,
  const viskores::FloatDefault& threshold = 0.05f,
  const bool& writeDiff = true)
{
  std::vector<std::string> fileNames;

  // Check to see if there are multiple versions of the file of the format filenName#.png.
  auto found = fileName.rfind(".");
  auto prefix = fileName.substr(0, found);
  auto suffix = fileName.substr(found, fileName.length());

  for (int i = 0;; i++)
  {
    std::ostringstream fileNameStream;
    fileNameStream << prefix << i << suffix;
    std::ifstream check(
      viskores::cont::testing::Testing::RegressionImagePath(fileNameStream.str()));
    if (!check.good())
    {
      VISKORES_LOG_S(viskores::cont::LogLevel::Info,
                     "Stopped filename search at: " << fileNameStream.str()
                                                    << ", beginning testing");
      break;
    }
    fileNames.push_back(fileNameStream.str());
  }

  // Check to see the filename without a number exists.
  {
    std::ifstream check(viskores::cont::testing::Testing::RegressionImagePath(fileName));
    if (check.good())
    {
      fileNames.push_back(fileName);
    }
  }

  // Make sure at least one file exists
  if (fileNames.empty())
  {
    fileNames.push_back(fileName);
  }

  return test_equal_images(std::forward<ImageType>(image),
                           fileNames,
                           averageRadius,
                           pixelShiftRadius,
                           allowedPixelErrorRatio,
                           threshold,
                           writeDiff);
}

#endif // viskores_rendering_testing_Testing_h
