/************************************************************************/
/*                                                                      */
/*    vspline - a set of generic tools for creation and evaluation      */
/*              of uniform b-splines                                    */
/*                                                                      */
/*            Copyright 2015, 2016 by Kay F. Jahnke                     */
/*                                                                      */
/*    Permission is hereby granted, free of charge, to any person       */
/*    obtaining a copy of this software and associated documentation    */
/*    files (the "Software"), to deal in the Software without           */
/*    restriction, including without limitation the rights to use,      */
/*    copy, modify, merge, publish, distribute, sublicense, and/or      */
/*    sell copies of the Software, and to permit persons to whom the    */
/*    Software is furnished to do so, subject to the following          */
/*    conditions:                                                       */
/*                                                                      */
/*    The above copyright notice and this permission notice shall be    */
/*    included in all copies or substantial portions of the             */
/*    Software.                                                         */
/*                                                                      */
/*    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND    */
/*    EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES   */
/*    OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND          */
/*    NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT       */
/*    HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,      */
/*    WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING      */
/*    FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR     */
/*    OTHER DEALINGS IN THE SOFTWARE.                                   */
/*                                                                      */
/************************************************************************/

/// \file impulse_response.cc
///
/// \brief get the impulse response of a b-spline prefilter
/// 
/// filter a unit pulse with a b-spline prefilter of a given degree
/// and display the central section of the result
///
/// compile with:
/// g++ -std=c++11 -o impulse_response -O3 -pthread -DUSE_VC=1 impulse_response.cc -lVc
///
/// to get the central section with values beyond +/- 0.0042 of a degree 5 b-spline:
///
/// impulse_response 5 .0042
///
/// producing this output:
/// 
/// long double ir_5[] = {
/// -0.0084918610197410 ,
/// +0.0197222540252632 ,
/// -0.0458040841925519 ,
/// +0.1063780046433000 ,
/// -0.2470419274022756 ,
/// +0.5733258709616592 ,
/// -1.3217294729875093 ,
/// +2.8421709220216247 ,
/// -1.3217294729875098 ,
/// +0.5733258709616593 ,
/// -0.2470419274022757 ,
/// +0.1063780046433000 ,
/// -0.0458040841925519 ,
/// +0.0197222540252632 ,
/// -0.0084918610197410 ,
///  } ;
///
/// which, when used as a convolution kernel, will have the same effect on a signal
/// as applying the recursive filter itself, but with lessened precision due to windowing.
///
/// note how three different ways of getting the result are given, the variants
/// using lower-level access to the filter are commented out.

#include <iomanip>
#include <assert.h>
#include <vspline/multithread.h>
#include <vspline/vspline.h>

int main ( int argc , char * argv[] )
{
  if ( argc < 3 )
  {
    std::cerr << "please pass spline degree and cutoff on the command line"
              << std::endl ;
    exit ( -1 ) ;
  }

  int degree = std::atoi ( argv[1] ) ;
  long double cutoff = std::atof ( argv[2] ) ;
  
  std::cout << "calculating impulse resonse with spline degree "
            << degree << " and cutoff " << cutoff << std::endl ;
            
  assert ( degree >= 0 && degree < 25 ) ;
  
  int npoles = degree / 2 ;
  const long double * poles = vspline_constants::precomputed_poles [ degree ] ;

// using the highest-level access to prefiltering, we code:

  vspline::bspline < long double , 1 > bsp ( 1001 , degree , vspline::MIRROR ) ;
  auto v1 = bsp.core ;
  v1 [ 500 ] = 1.0 ;
  bsp.prefilter() ;

// using slightly lower-level access to the prefiltering code, we could achieve
// the same result with:
// 
//   typedef vigra::MultiArray < 1 , long double > array_t ;
//   vigra::TinyVector < vspline::bc_code , 1 > bcv ;
//   bcv[0] = vspline::MIRROR ;
//   array_t v1 ( 1001 ) ;
//   v1[500] = 1.0 ;
//   vspline::solve < array_t , array_t , long double >
//     ( v1 , v1 , bcv , degree , 0.000000000001 ) ;
  
// and, going yet one level lower, this code also produces the same result:
  
//   vigra::MultiArray < 1 , long double > v1 ( 1001 ) ;
//   v1[500] = 1.0 ;
//   typedef decltype ( v1.begin() ) iter_type ;
//   vspline::filter < iter_type , iter_type , long double >
//     f ( v1.size() ,
//         vspline::overall_gain ( npoles , poles ) ,
//         vspline::MIRROR ,
//         npoles ,
//         poles ,
//         0.000000000001 ) ;
//   f.solve ( v1.begin() ) ;
        
  std::cout << "long double ir_" << degree << "[] = {" << std::endl ;

  std::cout << std::fixed << std::showpos << std::showpoint
            << std::setprecision(std::numeric_limits<long double>::max_digits10) ;

  for ( int k = 0 ; k < 1001 ; k++ )
  {
    if ( std::abs ( v1[k] ) > cutoff )
    {
      std::cout << v1[k] << "L ," << std::endl ;
    }
  }
  std::cout << "} ;" << std::endl ;
}
