// SPDX-FileCopyrightText: 2006 Istituto Nazionale di Fisica Nucleare
//
// SPDX-License-Identifier: Apache-2.0

package org.italiangrid.voms.util;

import static java.nio.file.StandardOpenOption.CREATE_NEW;
import static java.nio.file.StandardOpenOption.WRITE;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.Files;
import java.nio.file.NoSuchFileException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.PosixFilePermission;
import java.nio.file.attribute.PosixFilePermissions;
import java.security.PrivateKey;
import java.security.cert.X509Certificate;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.Set;

import eu.emi.security.authn.x509.X509Credential;
import eu.emi.security.authn.x509.helpers.CertificateHelpers;
import eu.emi.security.authn.x509.impl.CertificateUtils;
import eu.emi.security.authn.x509.impl.CertificateUtils.Encoding;

/**
 * An utility class for handling credentials
 * 
 * @author Daniele Andreotti
 * @author Andrea Ceccanti
 * 
 */
public class CredentialsUtils {

  /**
   * 
   * The encoding used to serialize proxy credentials private key.
   *
   */
  public enum PrivateKeyEncoding {
    PKCS_1, PKCS_8
  }

  /**
   * The default encoding used when no encoding is specified by callers.
   */
  public static final PrivateKeyEncoding DEFAULT_ENCONDING = PrivateKeyEncoding.PKCS_1;

  /**
   * Serializes a private key to an output stream according to an encoding.
   * 
   * @param os
   *          the target output stream
   * @param key
   *          the key to be serialized
   * @param encoding
   *          the encoding
   *          
   * @throws IllegalArgumentException
   *          for unsupported private key encodings
   * @throws IOException
   *          if write fails for any reason on the output stream
   */
  public static void savePrivateKey(OutputStream os, PrivateKey key,
    PrivateKeyEncoding encoding) throws IOException {

    switch (encoding) {
    case PKCS_1:
      savePrivateKeyPKCS1(os, key);
      break;
    case PKCS_8:
      savePrivateKeyPKCS8(os, key);
      break;
    default:
      throw new IllegalArgumentException("Unsupported private key encoding: "
        + encoding.name());
    }
  }

  /**
   * Serializes a private key to an output stream following the pkcs8 encoding.
   * 
   * This method just delegates to canl, but provides a much more understandable
   * signature.
   * 
   * @param os
   * @param key
   * @throws IllegalArgumentException
   * @throws IOException
   */
  private static void savePrivateKeyPKCS8(OutputStream os, PrivateKey key)
    throws IllegalArgumentException, IOException {

    CertificateUtils.savePrivateKey(os, key, Encoding.PEM, null, null);

  }

  /**
   * Serializes a private key to an output stream following the pkcs1 encoding.
   * 
   * This method just delegates to canl, but provides a much more understandable
   * signature.
   * 
   * @param os
   * @param key
   * @throws IllegalArgumentException
   * @throws IOException
   */
  private static void savePrivateKeyPKCS1(OutputStream os, PrivateKey key)
    throws IllegalArgumentException, IOException {

    CertificateUtils.savePrivateKey(os, key, Encoding.PEM, null, new char[0],
      true);

  }

  /**
   * Saves user credentials as a plain text PEM data. <br>
   * Writes the user certificate chain first, then the user key.
   * 
   * @param os
   *          the output stream
   * @param uc
   *          the user credential that must be serialized
   * @param encoding
   *          the private key encoding
   *                    
   *                    
   * @throws IOException
   *          in case of errors writing on the output stream
   */
  public static void saveProxyCredentials(OutputStream os, X509Credential uc,
    PrivateKeyEncoding encoding) throws IOException {

    X509Certificate[] chain = CertificateHelpers.sortChain(Arrays.asList(uc
      .getCertificateChain()));

    PrivateKey key = uc.getKey();
    X509Certificate cert = uc.getCertificate();

    CertificateUtils.saveCertificate(os, cert, Encoding.PEM);

    if (key != null)
      savePrivateKey(os, key, encoding);

    X509Certificate c = null;
    for (int index = 1; index < chain.length; index++) {

      c = chain[index];

      int basicConstraints = c.getBasicConstraints();

      // Only save non-CA certs to proxy file
      if (basicConstraints < 0){
        CertificateUtils.saveCertificate(os, c, Encoding.PEM);
      }
      
    }

    os.flush();
  }

  /**
   * 
   * Saves user credentials as a plain text PEM data. <br>
   * Writes the user certificate chain first, then the user key, using the
   * default encoding specified in {@link #DEFAULT_ENCONDING}.
   *
   * @param os
   *          the output stream for the saved proxy
   *          
   * @param uc
   *          the user credential
   * 
   * @throws IOException
   *          in case of errors writing to the output stream
   * 
   */
  public static void saveProxyCredentials(OutputStream os, X509Credential uc)
    throws IOException {

    saveProxyCredentials(os, uc, DEFAULT_ENCONDING);
  }

  /**
   * Saves proxy credentials to a file. This method ensures that the stored
   * proxy is saved with the appropriate file permissions.
   * 
   * @param proxyFileName
   *          the file where the proxy will be saved
   * @param uc
   *          the credential to be saved
   * @param encoding
   *          the private key encoding
   * @throws IOException
   *          in case of errors writing to the proxy file
   */
  public static void saveProxyCredentials(String proxyFileName, X509Credential uc,
      PrivateKeyEncoding encoding) throws IOException {

    ByteArrayOutputStream baos = new ByteArrayOutputStream();
    saveProxyCredentials(baos, uc, encoding);
    baos.close();

    final ByteBuffer proxyBuffer = ByteBuffer.wrap(baos.toByteArray());
    final Path proxyFilePath = Paths.get(proxyFileName);
    final FileAttribute<Set<PosixFilePermission>> permissions =
        PosixFilePermissions.asFileAttribute(PosixFilePermissions.fromString("rw-------"));
    final Set<StandardOpenOption> options = new HashSet<>();
    Collections.addAll(options, CREATE_NEW, WRITE);

    try {
      Files.delete(proxyFilePath);
    } catch (NoSuchFileException e) {
      // ignore
    }
    try (FileChannel fc = FileChannel.open(proxyFilePath, options, permissions)) {
      fc.write(proxyBuffer);
    } catch (FileAlreadyExistsException e) {
      throw new IOException("Cannot create proxy file safely");
    }
  }

  /**
   * 
   * Saves proxy credentials to a file. This method ensures that the stored
   * proxy is saved with the appropriate file permissions, using the default
   * encoding specified in {@link #DEFAULT_ENCONDING}.
   * 
   * @param proxyFileName
   *         the file where the proxy will be saved
   * @param uc
   *         the credential to be saved
   *         
   * @throws IOException
   *          in case of errors writing the credential to the proxy file
   */
  public static void saveProxyCredentials(String proxyFileName,
    X509Credential uc) throws IOException {

    saveProxyCredentials(proxyFileName, uc, DEFAULT_ENCONDING);
  }
}
