/*******************************************************************************
 * Part of "Intel(R) Active Management Technology (Intel(R) AMT)
 *                   User Notification Service (UNS)"
 *
 * Copyright (c) 2007 Intel Corp.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification.
 * 2. Redistributions in binary form must reproduce at minimum a disclaimer
 *    substantially similar to the "NO WARRANTY" disclaimer below
 *    ("Disclaimer") and any redistribution must be conditioned upon
 *    including a substantially similar Disclaimer requirement for further
 *    binary redistribution.
 * 3. Neither the names of the above-listed copyright holders nor the names
 *    of any contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * Alternatively, this software may be distributed under the terms of the
 * GNU General Public License ("GPL") version 2 as published by the Free
 * Software Foundation.
 *
 * NO WARRANTY
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * HOLDERS OR CONTRIBUTORS BE LIABLE FOR SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGES.
 *******************************************************************************/

//----------------------------------------------------------------------------
//
//  File:       LinuxRegistry.cpp
//
//  Notes:      code to linux registry simulation
//
//----------------------------------------------------------------------------

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/file.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include "AMTPPDefs.h"
#include "registryWrapper.h"

#define MAX_PATH 100
#define MAX_DATA 4096

#define UNSREG_VAL_ACCESS_MODE_PUBLIC (S_IRUSR | S_IWUSR | S_IROTH | S_IRGRP)
#define UNSREG_VAL_ACCESS_MODE (S_IRUSR | S_IWUSR)
#define UNSREG_KEY_ACCESS_MODE (S_IRWXU | S_IXOTH | S_IXGRP)

bool DeleteRegistryKey(const char *base0, const char *key) {

	char path[MAX_PATH];

	if (MAX_PATH < (strnlen("/usr/", MAX_PATH) + 
			strnlen(base0, MAX_PATH) + 
			strnlen("/", MAX_PATH) + 
			strnlen(key, MAX_PATH) + 1))
	{
			return false;
	}

	strncpy(path,"/usr/", strnlen("/usr/", MAX_PATH - 1) + 1);
	strncat(path,base0, strnlen(base0, MAX_PATH - 1));
	strncat(path,("/"), strnlen("/", MAX_PATH - 1));
	strncat(path,key, strnlen(key, MAX_PATH - 1));

	if(-1 == unlink(path))
	{
		return false;
	}
	return true;
}

bool DeleteRegistryVal(const char *base0, const char *key, const char *valuename) {

	char path[MAX_PATH*2];

	// sanity check before string copying
	if (MAX_PATH*2 < (strnlen("/usr/", 2*MAX_PATH) + 
			  strnlen(base0, 2*MAX_PATH) + 
			  strnlen("/", 2*MAX_PATH) + 
			  strnlen(key, 2*MAX_PATH) + 
			  strnlen("/", 2*MAX_PATH) + 
			  strnlen(valuename, 2*MAX_PATH) + 1))
	{
			return false;
	}

	strncpy(path,"/usr/", strnlen("/usr/", 2*MAX_PATH - 1) + 1);
	strncat(path,base0, strnlen(base0, 2*MAX_PATH - 1));
	strncat(path,("/"), strnlen("/", 2*MAX_PATH - 1));
	strncat(path,key, strnlen(key, 2*MAX_PATH - 1));
	strncat(path,("/"), strnlen("/", 2*MAX_PATH - 1));
	strncat(path,valuename, strnlen(valuename, 2*MAX_PATH - 1));

	if(-1 == unlink(path))
	{
		return false;
	}
	return true;
}

bool GetRegistryData(void *value, UNSREG_DATATYPE *valsz, UNSREG_DATATYPE *type,
			 const char *base0, const char *key, const char *valuename) {
	int fd;
	bool ret = true;
	int size;

	char path[MAX_PATH*2];
	char tmpval[MAX_DATA];

	// sanity check before string copying
	if (MAX_PATH*2 < (strnlen("/usr/", 2*MAX_PATH) + 
			  strnlen(base0, 2*MAX_PATH) + 
			  strnlen("/", 2*MAX_PATH) + 
			  strnlen(key, 2*MAX_PATH) + 
			  strnlen("/", 2*MAX_PATH) + 
			  strnlen(valuename, 2*MAX_PATH) + 1))
	{
			return false;
	}

	strncpy(path,"/usr/", strnlen("/usr/", 2*MAX_PATH - 1) + 1);
	strncat(path,base0, strnlen(base0, 2*MAX_PATH - 1));
	strncat(path,("/"), strnlen("/", 2*MAX_PATH - 1));
	strncat(path,key, strnlen(key, 2*MAX_PATH - 1));
	strncat(path,("/"), strnlen("/", 2*MAX_PATH - 1));
	strncat(path,valuename, strnlen(valuename, 2*MAX_PATH - 1));

	//open file for read
	if(-1 == (fd = open(path, O_RDONLY, S_IRWXU)))
	{
		return  false;
	}
	if(-1 == (size = read(fd, type, sizeof(UNSREG_DATATYPE))))
	{
		ret = false;
	}
	//read value from file
	if(-1 == (size = read(fd, tmpval, sizeof(tmpval))))
	{
		ret = false;
	}
	if( (NULL == value) || ((int)*valsz < size))
	{
		*valsz = size;
	}
	else
	{
		memcpy(value, tmpval, (size_t)size);
	}
	//close file
	if(-1 == close(fd))
	{
		ret = false;
	}
	return ret;
}

bool SetRegistryData(const void* value, UNSREG_DATATYPE valsz, UNSREG_DATATYPE type,const char *base0,const char *key,
			 const char *valuename, bool bPublic) {


	int fd;
	int wrote;
	bool ret = true;
	char path[MAX_PATH*2];

	// sanity check before string copying
	if (MAX_PATH*2 < (strnlen("/usr/", 2*MAX_PATH) + 
			  strnlen(base0, 2*MAX_PATH) + 
			  strnlen("/", 2*MAX_PATH) + 
			  strnlen(key, 2*MAX_PATH) + 
			  strnlen("/", 2*MAX_PATH) + 
			  strnlen(valuename, 2*MAX_PATH) + 1))
	{
			return false;
	}

	strncpy(path,"/usr/", strnlen("/usr/", 2*MAX_PATH - 1) + 1);
	strncat(path,base0, strnlen(base0, 2*MAX_PATH - 1));
	//create path
	mkdir(path, UNSREG_KEY_ACCESS_MODE);

	strncat(path,("/"), strnlen("/", 2*MAX_PATH - 1));
	strncat(path,key, strnlen(key, 2*MAX_PATH - 1));

	//create path
	mkdir(path, UNSREG_KEY_ACCESS_MODE);

	strncat(path,("/"), strnlen("/", 2*MAX_PATH - 1));
	strncat(path,valuename, strnlen(valuename, 2*MAX_PATH - 1));

	//open file for writing
	if(-1 == (fd = open(path, O_RDWR|O_CREAT|O_TRUNC,
			    (bPublic == true)?UNSREG_VAL_ACCESS_MODE_PUBLIC:UNSREG_VAL_ACCESS_MODE)))
	{
		return false;
	}
	if(-1 == (wrote = write(fd, &type, sizeof(UNSREG_DATATYPE))))
	{
		ret = false;
	}
	// check write integrity
	if (wrote != sizeof(UNSREG_DATATYPE))
	{
		ret = false;
	} 
	if(-1 == (wrote = write(fd, value, (size_t)valsz)))
	{
		ret = false;
	}
	// check write integrity
	if (wrote != valsz)
	{
		ret = false;
	}
 	if(-1 == close(fd))
	{
		ret = false;
	}
	return ret;
}
