# -*- coding: utf-8 -*-
#
# Copyright 2012 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# In addition, as a special exception, the copyright holders give
# permission to link the code of portions of this program with the
# OpenSSL library under certain conditions as described in each
# individual source file, and distribute linked combinations
# including the two.
# You must obey the GNU General Public License in all respects
# for all of the code used other than OpenSSL.  If you modify
# file(s) with this exception, you may extend this exception to your
# version of the file(s), but you are not obligated to do so.  If you
# do not wish to do so, delete this exception statement from your
# version.  If you delete this exception statement from all source
# files in the program, then also delete it here.
"""Tests for the expander widget."""

from PyQt4.QtGui import QLabel
from twisted.internet import defer
from twisted.trial.unittest import TestCase

from ubuntu_kylin_sso.qt.arrow import QArrow
from ubuntu_kylin_sso.qt.expander import QExpanderLabel, QExpander

# pylint: disable=C0103, W0201, W0212


class QExpanderLabelTestCase(TestCase):
    """Test the expander labe."""

    @defer.inlineCallbacks
    def setUp(self):
        """Set tests."""
        yield super(QExpanderLabelTestCase, self).setUp()
        self.label_content = 'test'
        self.label = QExpanderLabel(self.label_content)

    def test_mouse_press_event_arrow_right(self):
        """Test the actions taken when we press the mouse."""
        self.label.arrow.direction = QArrow.RIGHT
        self.label.mousePressEvent(None)
        self.assertEqual(QArrow.DOWN, self.label.arrow.direction)

    def test_mouse_press_event_arrow_down(self):
        """Test the actions taken when we press the mouse."""
        self.label.arrow.direction = QArrow.DOWN
        self.label.mousePressEvent(None)
        self.assertEqual(QArrow.RIGHT, self.label.arrow.direction)

    def test_text(self):
        """Test getting the text."""
        self.assertEqual(self.label.label.text(), self.label.text())

    def test_set_text(self):
        """Test setting the text."""
        new_label = 'new'
        self.label.setText(new_label)
        self.assertEqual(self.label.label.text(), new_label)


class QExpanderTestCase(TestCase):
    """Test the QExpander widget."""

    @defer.inlineCallbacks
    def setUp(self):
        """Set up tests."""
        yield super(QExpanderTestCase, self).setUp()
        self.label_content = 'test'
        self.expander = QExpander(self.label_content)

    def test_init(self):
        """Test the constructor."""

    def test_add_widget(self):
        """Test adding a widget."""
        widget = QLabel('test')
        widget.setObjectName('test')
        self.expander.addWidget(widget)
        self.assertEqual(widget, self.expander.content)
        self.assertEqual(1, self.expander.layout.indexOf(widget))

    def test_add_widget_present(self):
        """Test adding a widge when one is already present."""
        old_widget = QLabel('old')
        widget = QLabel('test')
        self.expander.addWidget(old_widget)
        self.expander.addWidget(widget)
        self.assertEqual(-1, self.expander.layout.indexOf(old_widget))
        self.assertEqual(1, self.expander.layout.indexOf(widget))

    def test_add_widget_expanded(self):
        """Test adding a widget when we are expanded."""
        widget = QLabel('test')
        self.expander._expanded = True
        self.expander.addWidget(widget)
        self.assertEqual(1, self.expander.layout.indexOf(widget))

    def test_add_widget_collapsed(self):
        """Test adding a widget when we are collapsed."""
        widget = QLabel('test')
        self.expander._expanded = False
        self.expander.addWidget(widget)
        self.assertEqual(1, self.expander.layout.indexOf(widget))
        self.assertFalse(widget.isVisible())

    def test_text(self):
        """Test getting the text."""
        self.assertEqual(self.expander.text(), self.expander.label.text())

    def test_set_text(self):
        """Test setting the text."""
        label_content = 'content'
        self.expander.setText(label_content)
        self.assertEqual(label_content, self.expander.label.text())

    def test_expanded(self):
        """Test getting expanded."""
        self.expander._expanded = True
        self.assertEqual(self.expander.expanded(), self.expander._expanded)
