-- Topal: GPG/GnuPG and Alpine/Pine integration
-- Copyright (C) 2001--2012  Phillip J. Brooke
--
-- This program is free software: you can redistribute it and/or modify
-- it under the terms of the GNU General Public License version 3 as
-- published by the Free Software Foundation.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program.  If not, see <http://www.gnu.org/licenses/>.

with Ada.Sequential_IO;
with Ada.Text_IO;
with Globals;            use Globals;

package Misc is

   -- How to handle errors and debugging.
   Panic          : exception;
   User_Interrupt : exception;

   procedure Error (The_Error : in String);

   -- ErrorNE (error-no-exception) doesn't raise an exception.
   -- You need to do it.
   procedure ErrorNE (The_Error : in String);

   procedure Debug (Message : in String);

   -- Receiving.Split_Two_Parts and others needs to write out files without
   -- an end-of-line being appended (as happens with Text_IO).
   package Character_IO is new Ada.Sequential_IO (Element_Type => Character);

   procedure Character_IO_Put (F : in Character_IO.File_Type;
                               S : in String);

   procedure Character_IO_Put_Line (F : in Character_IO.File_Type;
                                    S : in String);

   -- Strings to integers.

   String_Not_Integer : exception;

   function String_To_Integer (S : String) return Integer;
   function String_To_Integer (S : UBS) return Integer;

   -- Throw away leading blanks from a string.
   function Trim_Leading_Spaces (S : String) return String;
   
   -- Case insensitive comparison of two strings.
   function EqualCI (A, B : UBS) return Boolean;
   function EqualCI (A : String; B : UBS) return Boolean;
   function EqualCI (A : UBS; B : String) return Boolean;
   function EqualCI (A, B : String) return Boolean;

   -- Create our own temporary file names.  A sequence number is also
   --  inserted unless Use_Sequence_Number is set to False.
   function Temp_File_Name (Tail                : String;
                            Use_Sequence_Number : Boolean := True) return String;

   -- An `unbounded' Get_Line.
   function Unbounded_Get_Line (File : in Ada.Text_IO.File_Type)
                                return UBS;

   function Unbounded_Get_Line return UBS;

   -- Eat and fold an entire file.
   -- Set Include_LF to true to add a LF each new line.
   function Read_Fold (File       : in String;
		       Include_LF : in Boolean := False) return UBS;

   -- Open and close the result file.
   procedure Open_Result_File (Resultfile : in String);

   procedure Close_Result_File;

   -- Disclaimer.
   procedure Disclaimer;

   -- Wrapper for reading unbounded strings out of the config record.
   -- If the string is empty, then barf.
   Need_Nonempty_String : exception;
   function Value_Nonempty (V : UBS) return UBS;

   function Value_Nonempty (V : UBS) return String;

   -- Split up a string into multiple tokens, using spaces as the
   -- delimiter, but also honouring quoting and stuffing using `"'.
   function Split_Arguments (A : UBS) return UBS_Array;

   -- Split up a GPG colon-separated line.
   function Split_GPG_Colons (AS : String) return UBS_Array;
   
   -- Construct a UP.
   function UPC (A, B : String) return UP;
   function UPC (A, B : UBS) return UP;
   
   -- Get the basename of a filename.
   function Basename (S : String) return String;

   -- Basename.
   function Command_Basename return String;

   -- Hexadecimal decoder.
   function Hex_Decode (S : in String) return Natural;
   
   -- Have we got base64 or binary?
   function Guess_SMIME_Encoding(Infile : String) return String;

   -- Return a string for --[no-]use-agent depending on config and the
   --  type of operation.
   function UGA_Str (Signing : Boolean) return String;

   -- Write terminal SGR string.
   function Do_SGR (S : String) return String;
   function Do_SGR (U : UBS) return String;

   -- Reset terminal SGR.
   function Reset_SGR return String;

   -- Rewrite menu prompts.
   function Rewrite_Menu_Prompt (S : in String) return String;

   -- We turn SIGINT into an exception so that we can clean up
   --  temporary files.
   -- Handle interrupts, SIGINT, etc.
   pragma Unreserve_All_Interrupts;
   -- because we want to be able to handle ctrl-C.
   protected Signal_Handlers is
      procedure Sigint_Handler;
      function Sigint_Pending return Boolean;
      pragma Interrupt_Handler(Sigint_Handler);
   private
      Sigint_Pending_Flag : Boolean := False;
   end Signal_Handlers;
   -- And a procedure for taking over sigint.
   procedure Set_Sigint_Handler;

end Misc;
