/*
 *  Copyright (C) 2008 Nicolas Vion <nico@picapo.net>
 *
 *   This file is part of swac-get.
 *
 *   Swac-get is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   Swac-get is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with Foobar; if not, write to the Free Software
 *   Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <string>
#include <list>
#include <iostream>

#include "http.hh"
#include "posix.hh"
#include "basic.hh"
#include "global.hh"

#include "swac_repository.hh"


bool Tswac_package::load(std::string table, std::string path, xmlXPathContextPtr xpath_ctxt) {
	xmlNodePtr info_node = xpath_get_node("package/information", xpath_ctxt);
	xmlXPathContextPtr xpath_info_ctxt;
	if (info_node != NULL) {
		xpath_info_ctxt = xmlXPathNewContext(doc);
		xpath_info_ctxt->node = info_node;
	}

	SqlQuery query(&db);
	query.prepare(
		"INSERT INTO `" + table + "` " + 
		"(path, packid, version, authors, license, format, count, name, section, url, organization, organization_url, description, readme) " + 
		"VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?);"
	);
 
	query.bind(path, 1);

	query.bind(xpath_get_value("package/id", xpath_ctxt), 2);
	query.bind(xpath_get_value("package/version", xpath_ctxt), 3);
	query.bind(xpath_get_value("package/author", xpath_info_ctxt), 4);
	query.bind(xpath_get_value("package/license", xpath_info_ctxt), 5);
	query.bind(xpath_get_value("package/audio/format", xpath_ctxt), 6);
	query.bind(xpath_get_value("package/content/count", xpath_ctxt), 7);

	query.bind(xpath_get_value("collection/name", xpath_info_ctxt), 8);
	query.bind(xpath_get_value("collection/section", xpath_info_ctxt), 9);
	query.bind(xpath_get_value("collection/url", xpath_info_ctxt), 10);
	query.bind(xpath_get_value("organization/name", xpath_info_ctxt), 11);
	query.bind(xpath_get_value("organization/url", xpath_info_ctxt), 12);
	query.bind(xpath_get_value("description", xpath_info_ctxt), 13);
	query.bind(xpath_get_value("readme", xpath_info_ctxt), 14);

	query.step();
	query.clearBindings();
	
	if (xpath_info_ctxt != NULL)
		xmlXPathFreeContext(xpath_info_ctxt); 

	return (db.changes() == 1);
}

bool Tswac_package::load(std::string table, std::string path) {
	std::string xml(path);
	xml += "package.xml";

	if (createFromFile(xml.c_str())) {
		xmlXPathContextPtr xpath_ctxt = xmlXPathNewContext(doc);
		xpath_ctxt->node = xmlDocGetRootElement(doc);
		xpath_ctxt->node = xpath_get_node("/", xpath_ctxt);
		load(table, path, xpath_ctxt);
		xmlXPathFreeContext(xpath_ctxt); 
		return true;
	}

	print_error(std::string(_("Can not read XML file")) + " \"" + path + "\"");
	return false;
}

bool Tswac_repository_http::read_dom(std::string root) {
	xmlXPathContextPtr xpath_ctxt = xmlXPathNewContext(doc);
	if (xpath_ctxt == NULL)
		return false;

	xmlXPathObjectPtr xpathObjPackages = xmlXPathEvalExpression(BAD_CAST "//repository/directory", xpath_ctxt);
	xmlNodeSetPtr packages;
	if (xpathObjPackages != NULL and (packages = xpathObjPackages->nodesetval) != NULL)
		for (int i = 0; i < packages->nodeNr; i++) {
			xpath_ctxt->node = packages->nodeTab[i];
			Tswac_package::load("packages_available", root + xpath_get_value("./@path", xpath_ctxt), xpath_ctxt);
		}

	xmlXPathFreeObject(xpathObjPackages);
	xmlXPathFreeContext(xpath_ctxt); 
	return true;
}

bool Tswac_repository_http::create_dom(std::string url) {
	xmlURIPtr uri = xmlParseURI(url.c_str());
	if (uri == NULL) 
		return false;

	FILE * f = http_fopen(uri);
	xmlFreeURI(uri);

	if (f == NULL)
		return false;

	if (!createFromStreamBz2(f, url.c_str())) {
		print_error(std::string(_("Unable to parse file")) + " \"" + url + "\"");
		fclose(f);
		return false;
	}
	else {
		fclose(f);
		return true;
	}
}

bool Tswac_repository_http::load(std::string url) {
	if (create_dom(url + "repository.xml.bz2")) {
		read_dom(url);
		clear_dom();
		return true;
	}
	return false;
}

bool Tswac_repository_local::load(std::string path) {
	std::list<std::string> list;
	if (!list_subdirs(path, &list))
		return false;

	for (std::list<std::string>::iterator it = list.begin(); it != list.end(); it++) {
		if (fileexists(*it + "package.xml") &&  fileexists(*it + "index.xml.bz2")) {
			Tswac_package::load("packages_available", "file://" + *it);
		}
	}
	return true;
}



