use core:lang;
use lang:bs;
use lang:bs:macro;

on Compiler:

/**
 * Decorator that adds the `toJson` member and a constructor that accepts a `JsonValue`.
 */
void jsonSerializable(Type type) {
	Value t = thisPtr(type);
	Value jsonValue = named{JsonValue};

	// What functions are available already?
	Function? oCtor = findFn(type, "__init", [t, jsonValue], Value());
	Function? oToJson = findFn(type, "toJson", [t], jsonValue);

	// Members to serialize.
	Member[] toSerialize;
	for (m in type.variables)
		toSerialize << Member(m.name, m);

	if (oCtor.empty) {
		addFromJson(type, toSerialize);
	}

	if (oToJson.empty) {
		addToJson(type, toSerialize);
	}
}

private:

// Find a function inside 'type', and make sure it has the proper return type.
Function? findFn(Type type, Str name, Value[] params, Value result) {
	if (fn = type.findHere(SimplePart(name, params), Scope(type)) as Function) {
		if (result.mayReferTo(fn.result)) {
			return fn;
		}
	}
	null;
}

// Add the 'toJson' member.
void addToJson(Type to, Member[] members) {
	BSTreeFn f(named{JsonValue}, SStr("toJson"), [thisParam(to)], null);
	to.add(f);

	Scope srcScope(f, BSLookup());
	Scope scope(named{json}, BSLookup());
	FnBody body(f, scope);
	f.body = body;

	LocalVarAccess object(SrcPos(), body.parameters[0]);

	Expr createResult = FnCall(SrcPos(), scope, named{JsonValue:emptyObject}, Actuals());
	if (parent = to.super) {
		if (fn = parent.find("toJson", [thisPtr(parent)], srcScope) as Function) {
			createResult = FnCall(SrcPos(), srcScope, fn, Actuals(object), false);
		} else if (parent is named{Object} | parent is named{TObject}) {
			// Fine.
		} else {
			throw SyntaxError(to.pos, "The parent type of ${to.identifier} is not serializable.");
		}
	}

	Var resultVar(body, SStr("out"), createResult);
	body.add(resultVar);
	LocalVarAccess result(SrcPos(), resultVar.var);

	for (member in members) {
		Expr e = memberToJson(object, member, body, srcScope);
		Actuals params(result);
		params.add(StrLiteral(SrcPos(), member.name));
		params.add(e);
		body.add(FnCall(SrcPos(), scope, named{JsonValue:put<JsonValue, Str, JsonValue>}, params));
	}

	body.add(result);
}

// Serialize a single member.
Expr memberToJson(LocalVarAccess object, Member member, Block block, Scope scope) {
	unless (type = member.member.type.type)
		throw InternalError("No result type of member: ${member.member.identifier}");

	exprToJson(type, MemberVarAccess(member.member.pos, object, member.member), block, scope);
}

// Serialize a value.
Expr exprToJson(Type type, Expr expr, Block block, Scope scope) {
	Type jsonValue = named{JsonValue};

	if (type.isA(named{JsonValue})) {
		// Allow inserting JSON data directly if desired.
		return expr;
	} else if (type as ArrayType) {
		// It is an array.
		ExprBlock sub(SrcPos(), block);

		Var arrayVar(sub, SStr("array"), FnCall(SrcPos(), scope, named{JsonValue:emptyArray}, Actuals()));
		sub.add(arrayVar);
		LocalVarAccess arrayAccess(SrcPos(), arrayVar.var);

		RangeFor loop(SrcPos(), sub, SStr("x"), expr);
		sub.add(loop);
		unless (loopVar = loop.variableHere(SimplePart("x")))
			throw InternalError("Failed to find the loop variable.");

		Actuals params;
		params.add(arrayAccess);
		params.add(exprToJson(type.containedType, LocalVarAccess(SrcPos(), loopVar), loop, scope));
		loop.body(FnCall(SrcPos(), scope, named{JsonValue:push<JsonValue, JsonValue>}, params));

		sub.add(arrayAccess);
		return sub;
	} else if (type as MapType) {
		// It is a map. We only support cases where the key is a string.
		unless (type.keyType is named{Str})
			throw SyntaxError(expr.pos, "It is only possible to serialize maps where the key is a string.");

		// It is an array.
		ExprBlock sub(SrcPos(), block);

		Var mapVar(sub, SStr("map"), FnCall(SrcPos(), scope, named{JsonValue:emptyObject}, Actuals()));
		sub.add(mapVar);
		LocalVarAccess mapAccess(SrcPos(), mapVar.var);

		RangeFor loop(SrcPos(), sub, SStr("k"), SStr("v"), expr);
		sub.add(loop);
		unless (keyVar = loop.variableHere(SimplePart("k")))
			throw InternalError("Failed to find the loop variable.");
		unless (valVar = loop.variableHere(SimplePart("v")))
			throw InternalError("Failed to find the loop variable.");

		Actuals params;
		params.add(mapAccess);
		params.add(LocalVarAccess(SrcPos(), keyVar));
		params.add(exprToJson(type.valueType, LocalVarAccess(SrcPos(), valVar), loop, scope));
		loop.body(FnCall(SrcPos(), scope, named{JsonValue:put<JsonValue, Str, JsonValue>}, params));

		sub.add(mapAccess);
		return sub;
	} else if (type as MaybeType) {
		// It is a maybe type.
		WeakMaybeCast c(expr);
		c.name(SStr("converted"));
		If cond(block, c);

		unless (condResult = c.result)
			throw InternalError("Expected a result variable from the maybe conversion.");
		var converted = exprToJson(type.containedType, LocalVarAccess(SrcPos(), condResult), cond.successBlock, scope);
		cond.successBlock.set(converted);

		cond.failStmt = CtorCall(SrcPos(), scope, named{JsonValue:__init<JsonValue>}, Actuals());

		return cond;
	} else if (fn = type.find("toJson", thisPtr(type), scope) as Function) {
		return FnCall(SrcPos(), scope, fn, Actuals(expr));
	} else if (ctor = jsonValue.find("__init", [thisPtr(jsonValue), type], scope) as Function) {
		return CtorCall(SrcPos(), scope, ctor, Actuals(expr));
	} else {
		throw SyntaxError(expr.pos, "Unable to serialize ${expr} into Json!");
	}
}

// Add the from JSON constructor
void addFromJson(Type to, Member[] members) {
	BSTreeCtor c([thisParam(to), ValParam(named{JsonValue}, "json")], SrcPos());
	to.add(c);

	Scope srcScope(c, BSLookup());
	Scope scope(named{json}, BSLookup());
	CtorBody body(c, scope);
	c.body = body;

	LocalVarAccess srcJson(SrcPos(), body.parameters[1]);

	Actuals? superParams;
	if (parent = to.super) {
		if (ctor = parent.find("__init", [thisPtr(parent), named{JsonValue}], srcScope) as Function) {
			superParams = Actuals(srcJson);
		} else if (parent is named{Object} | parent is named{TObject}) {
			// This is fine.
		} else {
			throw SyntaxError(to.pos, "The parent type of ${to.identifier} is not serializable.");
		}
	}

	Initializer[] initializers;
	for (m in members) {
		unless (type = m.member.type.type)
			throw InternalError("No result type of member: ${m.member.identifier}");

		var init = jsonToMember(m.member.pos, srcJson, m.name, type, body, srcScope, m.initializer);
		initializers << Initializer(SStr(m.member.name), init);
	}

	body.add(InitBlock(SrcPos(), body, superParams, initializers));
}

Expr jsonToMember(SrcPos pos, Expr object, Str name, Type type, Block block, Scope scope, Expr? defaultInit) {
	Actuals params;
	params.add(object);
	params.add(StrLiteral(pos, name));
	if (defaultInit) {
		// If we have a default value, call 'at' and use the relevant one.
		FnCall extracted(SrcPos(), scope, named{JsonValue:at<JsonValue, Str>}, params);

		WeakMaybeCast cast(extracted);
		cast.name(SStr("e"));
		If cond(block, cast);

		unless (condResult = cast.result)
			throw InternalError("Expected a result variable from the maybe conversion.");

		cond.failStmt = defaultInit;
		cond.successBlock.set(jsonToMember(LocalVarAccess(pos, condResult), type, block, scope));

		cond;
	} else {
		// Otherwise, call "[]" instead.
		FnCall extracted(pos, scope, named{JsonValue:"[]"<JsonValue, Str>}, params);
		jsonToMember(extracted, type, block, scope);
	}
}

Expr jsonToMember(Expr data, Type type, Block block, Scope scope) {
	if (type.isA(named{JsonValue})) {
		return data;
	} else if (fn = builtInType(type)) {
		return FnCall(SrcPos(), scope, fn, Actuals(data));
	} else if (type as ArrayType) {
		// Extract an array.
		ExprBlock sub(SrcPos(), block);
		Var arrayVar(sub, type, SStr("array"), Actuals());
		sub.add(arrayVar);
		LocalVarAccess arrayAccess(SrcPos(), arrayVar.var);

		sub.add(namedExpr(block, SrcPos(), "reserve", arrayAccess,
								Actuals(FnCall(SrcPos(), scope, named{JsonValue:count<JsonValue>}, Actuals(data)))));

		FnCall jsonArray(SrcPos(), scope, named{JsonValue:array<JsonValue>}, Actuals(data));
		RangeFor loop(SrcPos(), sub, SStr("x"), jsonArray);
		sub.add(loop);

		unless (loopVar = loop.variableHere(SimplePart("x")))
			throw InternalError("Failed to find the loop variable.");

		var item = jsonToMember(LocalVarAccess(SrcPos(), loopVar), type.containedType, loop, scope);
		loop.body(namedExpr(loop, SrcPos(), "push", arrayAccess, Actuals(item)));

		sub.add(arrayAccess);
		return sub;
	} else if (type as MapType) {
		// It is a map. We only support cases where the key is a string.
		unless (type.keyType is named{Str})
			throw SyntaxError(data.pos, "It is only possible to serialize maps where the key is a string.");

		// Extract a map.
		ExprBlock sub(SrcPos(), block);
		Var arrayVar(sub, type, SStr("map"), Actuals());
		sub.add(arrayVar);
		LocalVarAccess arrayAccess(SrcPos(), arrayVar.var);

		FnCall jsonArray(SrcPos(), scope, named{JsonValue:object<JsonValue>}, Actuals(data));
		RangeFor loop(SrcPos(), sub, SStr("k"), SStr("v"), jsonArray);
		sub.add(loop);

		unless (keyVar = loop.variableHere(SimplePart("k")))
			throw InternalError("Failed to find the loop variable.");
		unless (valVar = loop.variableHere(SimplePart("v")))
			throw InternalError("Failed to find the loop variable.");

		Actuals params;
		params.add(LocalVarAccess(SrcPos(), keyVar));
		params.add(jsonToMember(LocalVarAccess(SrcPos(), valVar), type.valueType, loop, scope));
		loop.body(namedExpr(loop, SrcPos(), "put", arrayAccess, params));

		sub.add(arrayAccess);
		return sub;
	} else if (type as MaybeType) {
		// Maybe.
		ExprBlock sub(SrcPos(), block);
		Var dataCopy(sub, SStr("data"), data);
		sub.add(dataCopy);
		LocalVarAccess dataAccess(SrcPos(), dataCopy.var);

		If cond(sub, FnCall(SrcPos(), scope, named{JsonValue:isNull<JsonValue>}, Actuals(dataAccess)));
		sub.add(cond);

		cond.successBlock.set(defaultCtor(SrcPos(), scope, type));

		var converted = jsonToMember(dataAccess, type.containedType, cond, scope);
		cond.failStmt = expectCastTo(converted, type, scope);

		return sub;
	} else if (ctor = type.findHere(SimplePart("__init", [type, named{JsonValue}]), scope) as Function) {
		return CtorCall(SrcPos(), scope, ctor, Actuals(data));
	} else {
		throw SyntaxError(data.pos, "The type ${type.identifier} is not serializable.");
	}
}

Function? builtInType(Type type) {
	if (type is named{Bool})
		return named{JsonValue:bool<JsonValue>};
	else if (type is named{Byte})
		return named{JsonValue:byte<JsonValue>};
	else if (type is named{Int})
		return named{JsonValue:int<JsonValue>};
	else if (type is named{Nat})
		return named{JsonValue:nat<JsonValue>};
	else if (type is named{Long})
		return named{JsonValue:long<JsonValue>};
	else if (type is named{Word})
		return named{JsonValue:word<JsonValue>};
	else if (type is named{Float})
		return named{JsonValue:float<JsonValue>};
	else if (type is named{Double})
		return named{JsonValue:double<JsonValue>};
	else if (type is named{Str})
		return named{JsonValue:str<JsonValue>};
	return null;
}

/**
 * Information about a member to serialize.
 */
class Member {
	// Serialized name (could differ from the actual member's name if we support such annotations).
	Str name;

	// Variable.
	MemberVar member;

	// Create.
	init(Str name, MemberVar member) {
		init {
			name = name;
			member = member;
		}
	}

	// Find the default initializer.
	Expr? initializer() {
		if (init = member.initializer) {
			return FnCall(SrcPos(), Scope(), init, Actuals());
		// Should we allow defaults like in the other serialization? It swallows some error messages for
		// primitives that might not be ideal.
		// } else if (type = member.type.type) {
		// 	if (ctor = type.defaultCtor)
		// 		return CtorCall(SrcPos(), Scope(), ctor, Actuals());
		}
		return null;
	}
}
