/**
 * A piece of formatted text.
 *
 * The text is represented as a list of pieces, each with their own formatting.
 */
class FormattedText {
	// Text pieces.
	TextSpan[] spans;

	// Create an empty text span.
	init() {
		init {}
	}

	// Create a text span from a string. It will contain a single span.
	init(Str text) {
		init {}
		add(text);
	}

	// Create from a single span piece.
	init(TextSpan span) {
		init {}
		add(span);
	}

	// Add a span.
	void add(TextSpan span) {
		if (span as PackedText)
			spans.append(span.text.spans);
		else
			spans << span;
	}

	// Add a string as a plain span.
	void add(Str span) {
		add(PlainTextSpan(span));
	}

	// Add another FormattedText object.
	void add(FormattedText other) {
		spans.append(other.spans);
	}

	// Add another FormattedText object as a line of text.
	void addLine(FormattedText other) {
		if (spans.any)
			add("\n");
		add(other);
	}

	// Visit the entire structure.
	FormattedText visit(Visitor v) {
		TextSpan[] newSpans;
		for (x in spans) {
			var newSpan = x.visit(v);
			if (newSpan as PackedText) {
				newSpans.append(newSpan.text.spans);
			} else {
				newSpans << newSpan;
			}
		}

		spans = newSpans;

		return v.visit(this);
	}

	// To string.
	void toS(StrBuf to) : override {
		for (span in spans)
			span.toS(to);
	}

	// To HTML.
	void toHtml(Html to) {
		for (span in spans)
			span.toHtml(to);
	}
}

/**
 * A piece of formatted text.
 */
class TextSpan {
	// Create an HTML representation of the text span.
	void toHtml(Html to) : abstract;

	// Visit the element recursively. The function is expected to run `v.visit(this)` as the last
	// step. The returned value is the element that replaces this span after the visiting
	// operation. If nothing is changed, then the object itself is returned.
	TextSpan visit(Visitor v) : abstract;
}

/**
 * A plain piece of text.
 */
class PlainTextSpan extends TextSpan {
	// The text.
	Str text;

	// Create.
	init(Str text) {
		init { text = text; }
	}

	// To string.
	void toS(StrBuf to) : override {
		to << text;
	}

	// To HTML.
	void toHtml(Html to) : override {
		to.text(text);
	}

	// Visit recursively.
	TextSpan visit(Visitor v) : override {
		v.visit(this);
	}
}

/**
 * Italics.
 */
class ItalicText extends TextSpan {
	// Text.
	FormattedText text;

	// Create.
	init(FormattedText text) {
		init { text = text; }
	}

	// To string.
	void toS(StrBuf to) : override {
		to << "*" << text << "*";
	}

	// To HTML.
	void toHtml(Html to) : override {
		to.html("<em>");
		text.toHtml(to);
		to.html("</em>");
	}

	// Visit recursively.
	TextSpan visit(Visitor v) : override {
		text.visit(v);
		v.visit(this);
	}
}

/**
 * Bold.
 */
class BoldText extends TextSpan {
	// Text.
	FormattedText text;

	// Create.
	init(FormattedText text) {
		init { text = text; }
	}

	// To string.
	void toS(StrBuf to) : override {
		to << "**" << text << "**";
	}

	// To HTML.
	void toHtml(Html to) : override {
		to.html("<strong>");
		text.toHtml(to);
		to.html("</strong>");
	}

	// Visit recursively.
	TextSpan visit(Visitor v) : override {
		text.visit(v);
		v.visit(this);
	}
}

/**
 * Inline code.
 */
class InlineCode extends TextSpan {
	// Text.
	Str text;

	// Create.
	init(Str text) {
		init { text = text; }
	}

	// Create from string buffer.
	init(StrBuf text) {
		init { text = text.toS; }
	}

	// To string.
	void toS(StrBuf to) : override {
		to << "`" << text << "`";
	}

	// To HTML.
	void toHtml(Html to) : override {
		to.html("<code>");
		to.text(text);
		to.html("</code>");
	}

	// Visit recursively.
	TextSpan visit(Visitor v) : override {
		v.visit(this);
	}
}

/**
 * Link.
 */
class Link extends TextSpan {
	// Link text.
	FormattedText text;

	// Link target (TODO: Make into a URL).
	Str target;

	// Create.
	init(FormattedText text, Str target) {
		init { text = text; target = target; }
	}

	// To string.
	void toS(StrBuf to) : override {
		to << "[" << text << "](" << target << ")";
	}

	// To HTML.
	void toHtml(Html to) : override {
		to.html("<a href=\"");
		to.text(target);
		to.html("\">");
		text.toHtml(to);
		to.html("</a>");
	}

	// Visit recursively.
	TextSpan visit(Visitor v) : override {
		text.visit(v);
		v.visit(this);
	}
}

/**
 * Custom text (text inside [], but no () afterwards).
 */
class CustomText extends TextSpan {
	// Text.
	Str text;

	// Create.
	init(Str text) {
		init { text = text; }
	}

	// To string.
	void toS(StrBuf to) : override {
		to << "[" << text << "]";
	}

	// To HTML.
	void toHtml(Html to) {
		to.text(text);
	}

	// Visit recursively.
	TextSpan visit(Visitor v) : override {
		v.visit(this);
	}
}


/**
 * Custom inline HTML. Not generated by the parser, but useful by themes to generate custom elements.
 */
class CustomInlineHtml extends TextSpan {
	// Text.
	Str text;

	// Create.
	init(Str text) {
		init { text = text; }
	}

	// To string.
	void toS(StrBuf to) : override {
		to << "!! custom html: " << text << " !!";
	}

	// To HTML.
	void toHtml(Html to) : override {
		to.html(text);
	}

	// Visit recursively.
	TextSpan visit(Visitor v) : override {
		v.visit(this);
	}
}


/**
 * Packed text. Useful to replace a single TextSpan with multiple ones in a transform.
 */
class PackedText extends TextSpan {
	// Formatted text.
	FormattedText text;

	// Create.
	init() {
		init {}
	}

	// Create.
	init(FormattedText text) {
		init { text = text; }
	}

	// Add.
	void add(TextSpan span) {
		text.add(span);
	}

	// To string.
	void toS(StrBuf to) : override {
		to << text;
	}

	// To HTML.
	void toHtml(Html to) : override {
		text.toHtml(to);
	}

	// Visit recursively.
	TextSpan visit(Visitor v) : override {
		text = text.visit(v);
		v.visit(this);
	}
}
