/*
 * ber.C: Implementation of basic encoding rules class
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later 
 * version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the 
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 *
 * See the AUTHORS file for a list of people who have hacked on 
 * this code. 
 * See the ChangeLog file for a list of changes.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <stdarg.h>
#include <limits.h>
#include <errno.h>
#include <ctype.h>
#include <sys/types.h>
#include <netinet/in.h>

#include "ber.h"

void start_data(Tags types,unsigned int len, ustring &dest){
  dest+=char(types);
  if(len<128){
    dest+=char(len);
    return;
  } 
  unsigned char buf[sizeof(unsigned long)];
  unsigned long nlen=htonl(len);
  memcpy(buf,&nlen,sizeof(unsigned long));
  
  //counts the number of leading 0's
  unsigned int headlen;
  for(headlen=0;buf[headlen]==0 && headlen<3;headlen++);
  //magic function to to turn this into header length
  headlen=sizeof(unsigned long)-headlen;

  dest+=char(0x80+headlen);
  
  for(char j=sizeof(unsigned long)-headlen;j<=3;j++)
    dest+=buf[j];
}

unsigned long unpack_len(unsigned char *start,
			 unsigned char &headerlen) 
  throw(BerLengthException){
  if(start[1]<0x80){
    headerlen=2;    
    return start[1];
  }
  unsigned long sizeofsize=start[1]&0x7f;
  if(sizeofsize>sizeof(unsigned long)) throw BerLengthException();
  unsigned long asize=0;
  memcpy(reinterpret_cast<unsigned char*>(&asize)+sizeof(unsigned long)
	 -sizeofsize,start+2,sizeofsize);
  headerlen=2+sizeofsize;
  return htonl(asize);
}

/* 
 *
 * berNull - is the string 0x05 0x00 there is really not much too 
 * it.
 *
 */
BerNull::BerNull(unsigned char *str)
  throw(BerNullTagException,BerNullLengthExecption){
  if(str[0]!=NULL_TAG) throw BerNullTagException();
  if(str[1]!=0) throw BerNullLengthExecption();
}

ustring &BerNull::encode(ustring &dest){
  return dest.append(reinterpret_cast<const unsigned char*>("\005\000"),2);
}


/* berInt is the encoding of an int in ASN.1
   The encoding is a 0x02 followed by the size of the data then 
   the bytes of the data. The data is stored in the big endian 
   2's complement using the fewest bytes possible. However a 0 
   still uses one byte. i.e. 0x20 0x01 0x00
 */

void BerInt::ascii_print(std::string &str){
  char buf[30];
  snprintf(buf,30,"%ld",val);
  str+=buf;
}

void BerCounter::ascii_print(std::string &str){
  char buf[30];
  snprintf(buf,30,"%lu",val);
  str+=buf;
}

BerInt::BerInt(unsigned char *str) throw(BerIntTagException,
					 BerIntLengthExecption):val(0){
  if(str[0]!=INT_TAG) throw BerIntTagException();
  if(str[1]>sizeof(long)) throw BerIntLengthExecption();
  memcpy(reinterpret_cast<unsigned char*>(&val)+sizeof(long)-str[1],
	 str+2,str[1]);
  //extend out the sign if necessary
  if(str[1]!=sizeof(long) && str[2]&0x80)
    memset(&val,0xff,sizeof(long)-str[1]);
  val=ntohl(val);
}

BerCounter::BerCounter(unsigned char *str) 
  throw(BerCounterTagException,BerCounterLengthExecption):val(0){
  if(str[0]!=COUNTER_TAG) throw BerCounterTagException();
  if(str[1]>sizeof(unsigned long)) throw BerCounterLengthExecption();
  memcpy(reinterpret_cast<unsigned char*>(&val)
	 +sizeof(unsigned long)-str[1],str+2,str[1]);
  val=ntohl(val);
}


ustring &BerInt::encode(ustring &dest){
  unsigned char buf[sizeof(long)];
  unsigned char i;
  long valu=htonl(val);
  memcpy(buf,&valu,sizeof(long));
  for(i=sizeof(long);!buf[sizeof(long)-i] && i>1;i--);

  /* zero pad the case when the most significant byte has a 1 in 
     the most significant bit -- otherwise the other end will 
     interpret it as a negative number */
  if(i!=sizeof(long) && buf[sizeof(long)-i]&0x7f)
    i++;

  start_data(INT_TAG,i,dest);
  dest.append(buf+sizeof(long)-i,i);
  return dest;
}

ustring &BerCounter::encode(ustring &dest){
  unsigned char buf[sizeof(unsigned long)];
  unsigned char i;
  long valu=htonl(val);
  memcpy(buf,&valu,sizeof(unsigned long));
  for(i=sizeof(unsigned long);!buf[sizeof(unsigned long)-i] && i>1;
      i--);

  start_data(COUNTER_TAG,i,dest);
  dest.append(buf+sizeof(unsigned long)-i,i);
  return dest;
}

/*
 * Number of timeticks since some epoch. In 1/100 seconds
 */

BerTimeTick::BerTimeTick(unsigned char *str)
  throw(BerTimeTickTagException,BerTimeTickLengthExecption):val(0){
  if(str[0]!=TIME_TICK_TAG) throw BerTimeTickTagException();
  if(str[1]>sizeof(unsigned long)) throw BerTimeTickLengthExecption();
  memcpy(reinterpret_cast<unsigned char*>(&val)+sizeof(unsigned long)-
	 str[1],str+2,str[1]);
  val=ntohl(val);
}

ustring &BerTimeTick::encode(ustring &str){
  unsigned char buf[sizeof(unsigned long)];
  unsigned char i;
  long valu=htonl(val);
  memcpy(buf,&valu,sizeof(unsigned long));

  //count the number of bytes that are actually used
  for(i=sizeof(unsigned long);!buf[sizeof(unsigned long)-i]&& i>1;i--);

  start_data(TIME_TICK_TAG,i,str);
  str.append(buf+sizeof(unsigned long)-i,i);
  return str;
}

void BerTimeTick::ascii_print(std::string &str){
  // hopefully this works by just assming the bits are right 
  // rather than coercing the value into a different format.
  char buf[200];
  unsigned long t1=val%8640000;
  unsigned long t2=t1%360000;
  unsigned long t3=t2%6000;
  snprintf(buf,200,"Time: %lud %luh %lum %lu.%lus (%lu)",val/8640000ul,
	   t1/360000ul,t2/6000ul,t3/100ul,t3%100ul,val);
  str+=buf;
}

/* berString is used to encode a string. It is simply the length 
 and the data. the only complication is the encoding of the 
 length. Also terminates string. */

void BerString::ascii_print(std::string &bufstr){
  char buf[10];
  /* strings are used for binary data as well as text. We need to 
     handle 8 bit data as well as normal printable strings */
  for(std::string::iterator cur=str.begin();cur!=str.end();cur++){
    snprintf(buf,10,isprint(*cur)?"%c":"\\0x%02x",*cur);/*ITS4: ignore */
    bufstr+=buf;
  }
}

// Wire protocol only
BerString::BerString(unsigned char *strn) 
  throw(BerStringTagException,BerLengthException){
  if(strn[0]!=STRING_TAG) throw BerStringTagException();
  unsigned char headerlen;
  unsigned long len=unpack_len(strn,headerlen);
  str=std::string(reinterpret_cast<char*>(strn+headerlen),len);
}

ustring &BerString::encode(ustring &dest){
  start_data(STRING_TAG,str.length(),dest);
  dest.append(reinterpret_cast<const unsigned char*>(str.c_str()));
  return dest;
}

/* 
 * BerIPAddr is used to encode an ipaddress. 
 */

void BerIPAddr::ascii_print(std::string &dest)
  throw(BerIPAddrLengthExecption){
  if(str.length()!=4) throw BerIPAddrLengthExecption();// no v6 yet
  char buf[20];
  snprintf(buf,20,"%u.%u.%u.%u;",static_cast<unsigned>(str[0]) & 0xff,
	   static_cast<unsigned>(str[1]) & 0xff,
	   static_cast<unsigned>(str[2]) & 0xff,
	   static_cast<unsigned>(str[3]) & 0xff);
  dest+=buf;
}

BerIPAddr::BerIPAddr(unsigned char *dat,unsigned int len)
  throw(BerIPAddrLengthExecption):str(dat,len){
  if(len!=4)
    throw BerIPAddrLengthExecption();
}

BerIPAddr::BerIPAddr(const ustring &strng)
  throw(BerIPAddrLengthExecption):str(strng){ 
  if(strng.length()!=4)
    throw BerIPAddrLengthExecption(); //only handles v4 addrs
}

// Wire protocol only
BerIPAddr::BerIPAddr(unsigned char *strn)
  throw(BerIPAddrTagException,BerIPAddrLengthExecption){ 
  if(strn[0]!=IPADDR_TAG) throw BerIPAddrTagException();
  if(strn[1]!=4) throw BerIPAddrLengthExecption(); //ipv4 only
  str=ustring(strn+2,4);
}

ustring &BerIPAddr::encode(ustring &dest){
  start_data(IPADDR_TAG,4,dest);
  dest+=str;
  return dest;
}
