#ifndef _RHEOLEF_FIELD_EXPR_V2_NONLINEAR_H
#define _RHEOLEF_FIELD_EXPR_V2_NONLINEAR_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================
//
// field-valued nonlinear expressions, as:
//
//	field wh = interpolate  (Xh, 1/uh - uh*(1-vh)):
//	Float Ih = integrate (omega, 1/uh - uh*(1-vh), qopt):
//
// author: Pierre.Saramito@imag.fr
//
// date: 15 september 2015
//
// Notes; use template expressions and SFINAE techiques
//   The interpolation operator is required, as in
//   1/uh and uh*vh that do not belong to Xh when uh, vh in Xh
//
// SUMMARY:
// 1. unary operations
//    1.1. unary node
//    1.2. unary calls
//    1.3. unary compose
// 2. binary operations
//    2.1. binary node
//    2.2. binary calls
//         2.2.1. plus and minus
//         2.2.2. times
//         2.2.3. divides
//         2.2.4. std::math and extensions
//    2.3. binary compose
// 
#include "rheolef/field_expr_v2_nonlinear_terminal.h"

namespace rheolef {

// -------------------------------------------
// 1. unary operations
// -------------------------------------------
// 1.1. unary node
// -------------------------------------------
namespace details {

template<class UnaryFunction, class Expr>
class field_expr_v2_nonlinear_node_unary {
public:
// typedefs:

  typedef geo_element::size_type                      size_type;
  typedef typename Expr::memory_type                  memory_type;
  typedef typename details::generic_unary_traits<UnaryFunction>::template result_hint<typename Expr::result_type>::type
						      result_type;
  typedef result_type                                 value_type;
  typedef typename scalar_traits<value_type>::type    scalar_type;
  typedef typename  float_traits<value_type>::type    float_type;
  typedef field_expr_v2_nonlinear_node_unary<UnaryFunction,Expr> self_type;

// alocators:

  field_expr_v2_nonlinear_node_unary (const UnaryFunction& f, const Expr& expr) 
    : _f(f), _expr(expr) {}

// accessors:

  static const space_constant::valued_type valued_hint = space_constant::valued_tag_traits<result_type>::value;

  space_constant::valued_type valued_tag() const {
    return details::generic_unary_traits<UnaryFunction>::valued_tag (_expr.valued_tag());
  }

// initializators:

  bool initialize (const geo_basic<float_type,memory_type>& omega, const quadrature<float_type>& hat_x) const {
    return _expr.initialize (omega, hat_x);
  }
  bool initialize (const space_basic<float_type,memory_type>& Xh) const {
    return _expr.initialize (Xh);
  }
// evaluator:

  template<class Result, class Arg, class Status>
  struct evaluate_call_check {
    void operator() (const self_type& obj, const geo_element& K, std::vector<Result>& value) const {
      fatal_macro ("invalid type resolution: Result="<<typename_macro(Result)
          << ", Arg="<<typename_macro(Arg)
          << ", UnaryFunction="<<typename_macro(UnaryFunction)
      );
    }
  };
  template<class Result, class Arg>
  struct evaluate_call_check<Result,Arg,std::true_type> {
    void operator() (const self_type& obj, const geo_element& K, std::vector<Result>& value) const {
      std::vector<Arg> tmp_value;
      obj._expr.evaluate (K, tmp_value);
      value.resize(tmp_value.size());
      typename std::vector<Arg>::const_iterator tmp = tmp_value.begin();
      for (typename std::vector<Result>::iterator
	iter = value.begin(),
	last = value.end(); iter != last; ++iter, ++tmp) {
          *iter = obj._f(*tmp);
      }
    }
  };
  template<class Result, class Arg>
  void evaluate_call (const geo_element& K, std::vector<Result>& value) const {
    typedef typename details::generic_unary_traits<UnaryFunction>::template hint<Arg,Result>::result_type result_type;
    typedef typename details::and_type<
		typename details::is_equal<Result,result_type>::type
	       ,typename details::not_type<typename details::is_error<Arg>::type>::type
	      >::type
      status_t;
    evaluate_call_check<Result,Arg,status_t> eval;
    eval (*this, K, value);
  }
  template<class This, class Result, class Arg, space_constant::valued_type ArgTag = space_constant::valued_tag_traits<Arg>::value>
  struct evaluate_switch {};

  // when arg is unknown at run-time:
  template<class This, class Result, class Arg>
  struct evaluate_switch<This, Result, Arg, space_constant::last_valued> {
    void evaluate (const This& obj, const geo_element& K, std::vector<Result>& value) const {
      typedef typename scalar_traits<Arg>::type T;
      space_constant::valued_type arg_valued_tag = obj._expr.valued_tag();
      switch (arg_valued_tag) {
        case space_constant::scalar:
	  obj.template evaluate_call<Result,T> (K, value); break;
        case space_constant::vector:
	  obj.template evaluate_call<Result, point_basic<T> > (K, value); break;
        case space_constant::tensor:
        case space_constant::unsymmetric_tensor:
	  obj.template evaluate_call<Result, tensor_basic<T> > (K, value); break;
        default: { error_macro ("unexpected valued tag="<<arg_valued_tag); }
      }
    }
  };
  // when arg is known at compile-time:
  template<class This, class Result, class Arg>
  struct evaluate_switch <This, Result, Arg, space_constant::scalar> {
    typedef typename scalar_traits<Arg>::type T;
    void evaluate (const This& obj, const geo_element& K, std::vector<Result>& value) const {
      obj.template evaluate_call<Result,T> (K, value);
    }
  };
  template<class This, class Result, class Arg>
  struct evaluate_switch <This, Result, Arg, space_constant::vector> {
    typedef typename scalar_traits<Arg>::type T;
    void evaluate (const This& obj, const geo_element& K, std::vector<Result>& value) const {
      obj.template evaluate_call<Result, point_basic<T> > (K, value);
    }
  };
  template<class This, class Result, class Arg>
  struct evaluate_switch <This, Result, Arg, space_constant::tensor> {
    typedef typename scalar_traits<Arg>::type T;
    void evaluate (const This& obj, const geo_element& K, std::vector<Result>& value) const {
      obj.template evaluate_call<Result, tensor_basic<T> > (K, value);
    }
  };
  template<class This, class Result, class Arg>
  struct evaluate_switch <This, Result, Arg, space_constant::tensor3> {
    typedef typename scalar_traits<Arg>::type T;
    void evaluate (const This& obj, const geo_element& K, std::vector<Result>& value) const {
      obj.template evaluate_call<Result, tensor3_basic<T> > (K, value);
    }
  };
  template<class This, class Result, class Arg>
  struct evaluate_switch <This, Result, Arg, space_constant::tensor4> {
    typedef typename scalar_traits<Arg>::type T;
    void evaluate (const This& obj, const geo_element& K, std::vector<Result>& value) const {
      obj.template evaluate_call<Result, tensor4_basic<T> > (K, value);
    }
  };
  template<class Result>
  bool evaluate (const geo_element& K, std::vector<Result>& value) const
  {
    typedef field_expr_v2_nonlinear_node_unary<UnaryFunction, Expr> This;
    typedef typename details::generic_unary_traits<UnaryFunction>::template hint<typename Expr::value_type,Result>::argument_type
            A1;
#ifdef TO_CLEAN
    static const space_constant::valued_type argument_tag = space_constant::valued_tag_traits<A1>::value;
    evaluate_switch <This, Result, A1, argument_tag> helper;
#endif // TO_CLEAN
    evaluate_switch <This, Result, A1> helper;
    helper.evaluate (*this, K, value);
    return true;
  }
  template<class Result>
  void evaluate_on_side (const geo_element& K, const side_information_type& sid, std::vector<Result>& value) const { 
    // TODO: group with evaluate
    fatal_macro ("uf::evaluate_on_side: not yet");
  }

  template<class Result>
  bool valued_check() const {
    typedef typename details::generic_unary_traits<UnaryFunction>::template hint<typename Expr::value_type,Result>::argument_type
                     A1;
    if (! is_undeterminated<A1>::value) return _expr.template valued_check<A1>();
    return true;
  }

protected:
// data:
  UnaryFunction   _f;
  Expr            _expr;
};
template<class F, class Expr> struct is_field_expr_v2_nonlinear_arg     <field_expr_v2_nonlinear_node_unary<F,Expr> > : std::true_type {};

} // namespace details
// -------------------------------------------
// 1.2. unary calls
// -------------------------------------------
// unary operators +- and std::math

namespace details {

// avoid -(lin_expr) that is a linear expression
template <class Expr>
struct is_field_expr_v2_nonlinear_unary_operator_plus_minus
: and_type<
    is_field_expr_v2_nonlinear_arg<Expr>
   ,not_type <
      is_field_expr_v2_linear_arg<Expr>
    >
  >
{};

} // namespace details

#define _RHEOLEF_make_field_expr_v2_nonlinear_unary_general(FUNCTION,FUNCTOR,IS_EXPR)	\
template<class Expr>								\
inline										\
typename									\
std::enable_if<									\
  IS_EXPR<Expr>::value								\
 ,details::field_expr_v2_nonlinear_node_unary<					\
    FUNCTOR									\
   ,typename details::field_expr_v2_nonlinear_terminal_wrapper_traits<Expr>::type \
  >										\
>::type										\
FUNCTION (const Expr& expr)							\
{										\
  typedef typename details::field_expr_v2_nonlinear_terminal_wrapper_traits<Expr>::type wrap_t;  \
  return details::field_expr_v2_nonlinear_node_unary <FUNCTOR,wrap_t> (FUNCTOR(), wrap_t(expr)); \
}

#define _RHEOLEF_make_field_expr_v2_nonlinear_unary_operator(FUNCTION,FUNCTOR)		\
        _RHEOLEF_make_field_expr_v2_nonlinear_unary_general (FUNCTION, FUNCTOR, details::is_field_expr_v2_nonlinear_unary_operator_plus_minus)

#define _RHEOLEF_make_field_expr_v2_nonlinear_unary_function(FUNCTION)		\
        _RHEOLEF_make_field_expr_v2_nonlinear_unary_general (FUNCTION, details::FUNCTION##_, details::is_field_expr_v2_nonlinear_arg)

// standard unary operators
_RHEOLEF_make_field_expr_v2_nonlinear_unary_operator (operator+, details::unary_plus)
_RHEOLEF_make_field_expr_v2_nonlinear_unary_operator (operator-, details::negate)
// std::cmath
_RHEOLEF_make_field_expr_v2_nonlinear_unary_function (cos)
_RHEOLEF_make_field_expr_v2_nonlinear_unary_function (sin)
_RHEOLEF_make_field_expr_v2_nonlinear_unary_function (tan)
_RHEOLEF_make_field_expr_v2_nonlinear_unary_function (acos)
_RHEOLEF_make_field_expr_v2_nonlinear_unary_function (asin)
_RHEOLEF_make_field_expr_v2_nonlinear_unary_function (atan)
_RHEOLEF_make_field_expr_v2_nonlinear_unary_function (cosh)
_RHEOLEF_make_field_expr_v2_nonlinear_unary_function (sinh)
_RHEOLEF_make_field_expr_v2_nonlinear_unary_function (tanh)
_RHEOLEF_make_field_expr_v2_nonlinear_unary_function (exp)
_RHEOLEF_make_field_expr_v2_nonlinear_unary_function (log)
_RHEOLEF_make_field_expr_v2_nonlinear_unary_function (log10)
_RHEOLEF_make_field_expr_v2_nonlinear_unary_function (sqrt)
_RHEOLEF_make_field_expr_v2_nonlinear_unary_function (abs)
_RHEOLEF_make_field_expr_v2_nonlinear_unary_function (fabs)
_RHEOLEF_make_field_expr_v2_nonlinear_unary_function (floor)
_RHEOLEF_make_field_expr_v2_nonlinear_unary_function (ceil)
// rheolef extensions
_RHEOLEF_make_field_expr_v2_nonlinear_unary_function (sqr)
_RHEOLEF_make_field_expr_v2_nonlinear_unary_function (norm)
_RHEOLEF_make_field_expr_v2_nonlinear_unary_function (norm2)
_RHEOLEF_make_field_expr_v2_nonlinear_unary_function (tr)
_RHEOLEF_make_field_expr_v2_nonlinear_unary_function (trans)

#undef _RHEOLEF_make_field_expr_v2_nonlinear_unary_function
#undef _RHEOLEF_make_field_expr_v2_nonlinear_unary_operator
#undef _RHEOLEF_make_field_expr_v2_nonlinear_unary_general

// -------------------------------------------
// 1.3. unary compose
// -------------------------------------------

template<class Function, class Expr>
inline
typename
std::enable_if<
  details::is_field_expr_v2_nonlinear_arg<Expr>::value
 ,details::field_expr_v2_nonlinear_node_unary<
    typename details::function_traits<Function>::functor_type
   ,typename details::field_expr_v2_nonlinear_terminal_wrapper_traits<Expr>::type
  >
>::type
compose (const Function& f, const Expr& expr)
{
  typedef typename details::function_traits<Function>::functor_type                     fun_wrap_t;
  typedef typename details::field_expr_v2_nonlinear_terminal_wrapper_traits<Expr>::type expr_wrap_t;
  return details::field_expr_v2_nonlinear_node_unary <fun_wrap_t, expr_wrap_t> (fun_wrap_t(f), expr_wrap_t(expr));
}
// ---------------------------------------------------------------------------
// 2. binary operations
// ---------------------------------------------------------------------------
// 2.1. binary node
// -------------------------------------------
namespace details {

template<class BinaryFunction, class Expr1, class Expr2>
class field_expr_v2_nonlinear_node_binary {
public:
// typedefs:

  typedef geo_element::size_type                   size_type;
  typedef typename details::generic_binary_traits<BinaryFunction>::template result_hint<typename Expr1::result_type,typename Expr2::result_type>::type result_type;
  typedef result_type                              value_type;
  typedef typename scalar_traits<value_type>::type scalar_type;
  typedef typename  float_traits<value_type>::type float_type;
  typedef typename Expr1::memory_type              memory_type;

// alocators:

  field_expr_v2_nonlinear_node_binary (const BinaryFunction&             f, 
		    const Expr1& expr1,
                    const Expr2& expr2)
    : _f(f), _expr1(expr1), _expr2(expr2)
  {
  }

// accessors:
 
  static const space_constant::valued_type valued_hint = space_constant::valued_tag_traits<result_type>::value;

  space_constant::valued_type valued_tag() const {
    return details::generic_binary_traits<BinaryFunction>::valued_tag(_expr1.valued_tag(), _expr2.valued_tag());
  }

// initializers:

  bool initialize (const geo_basic<float_type,memory_type>& omega, const quadrature<float_type>& hat_x) const
  {
    return _expr1.initialize (omega, hat_x) &&
           _expr2.initialize (omega, hat_x);
  }
  bool initialize (const space_basic<float_type,memory_type>& Xh) const {
    return _expr1.initialize (Xh) &&
           _expr2.initialize (Xh);
  }

// evaluators:

  template<class Result, class Arg1, class Arg2>
  void evaluate_internal2 (const geo_element& K, std::vector<Result>& value) const {
#ifdef TO_CLEAN
    _check<Result,Arg1,Arg2> ();
#endif // TO_CLEAN
    std::vector<Arg1> tmp1_value;
    std::vector<Arg2> tmp2_value;
    _expr1.evaluate (K, tmp1_value);
    _expr2.evaluate (K, tmp2_value);
    value.resize(tmp1_value.size());
    typename std::vector<Arg1>::const_iterator tmp1 = tmp1_value.begin();
    typename std::vector<Arg2>::const_iterator tmp2 = tmp2_value.begin();
    for (typename std::vector<Result>::iterator
        iter = value.begin(),
	last = value.end(); iter != last; ++iter, ++tmp1, ++tmp2) {
      *iter = _f (*tmp1, *tmp2);
    }
  }
  template<class This, class Result, class ReturnType, class Arg1, class Arg2>
  struct evaluate_internal {
    void operator() (const This& obj, const geo_element& K, std::vector<Result>& value) const {
      fatal_macro ("unexpected return type "	
	<< pretty_typename_macro(ReturnType) << ": "
	<< pretty_typename_macro(Result) << " was expected for function "
	<< pretty_typename_macro(BinaryFunction) << "("
	<< pretty_typename_macro(Arg1) << ","
	<< pretty_typename_macro(Arg2) << ")");
    }
  };
  template<class This, class Result, class Arg1, class Arg2>
  struct evaluate_internal<This,Result,Result,Arg1,Arg2> {
    void operator() (const This& obj, const geo_element& K, std::vector<Result>& value) const {
      obj.template evaluate_internal2 <Result,Arg1,Arg2> (K, value);
    }
  };
  template<class Result, class Arg1, class Arg2>
  void evaluate_call (const geo_element& K, std::vector<Result>& value) const {
    typedef typename details::generic_binary_traits<BinaryFunction>::template result_hint<Arg1,Arg2>::type ReturnType;
    typedef field_expr_v2_nonlinear_node_binary<BinaryFunction, Expr1, Expr2> This;
    evaluate_internal<This,Result,ReturnType,Arg1,Arg2> eval_int;
    eval_int (*this, K, value);
  }
  // when both args are defined at compile time:
  template<class This, class Result,
	class Arg1,        space_constant::valued_type Arg1Tag,
  	class Arg2,        space_constant::valued_type Arg2Tag>
  struct evaluate_switch {
    void operator() (const This& obj, const geo_element& K, std::vector<Result>& value) const {
      obj.template evaluate_call<Result, Arg1, Arg2> (K, value);
    }
  };
  // when both args are undefined at compile time:
  template<class This, class Result,
	class Arg1,
  	class Arg2>
  struct evaluate_switch<This, Result,
	Arg1,   space_constant::last_valued,
        Arg2,   space_constant::last_valued> {
    void operator() (const This& obj, const geo_element& K, std::vector<Result>& value) const {
      typedef typename scalar_traits<Arg1>::type T1;
      typedef typename scalar_traits<Arg2>::type T2;
      space_constant::valued_type arg1_valued_tag = obj._expr1.valued_tag();
      space_constant::valued_type arg2_valued_tag = obj._expr2.valued_tag();
      switch (arg1_valued_tag) {
        case space_constant::scalar: {
          switch (arg2_valued_tag) {
            case space_constant::scalar:
	      obj.template evaluate_call<Result, T1, T2>                (K, value); break;
            case space_constant::vector:
	      obj.template evaluate_call<Result, T1, point_basic<T2> >  (K, value); break;
            case space_constant::tensor:
            case space_constant::unsymmetric_tensor:
	      obj.template evaluate_call<Result, T1, tensor_basic<T2> > (K, value); break;
            case space_constant::tensor3:
	      obj.template evaluate_call<Result, T1, tensor3_basic<T2> > (K, value); break;
            default: error_macro ("unexpected second argument valued tag="<<arg2_valued_tag);
          }
          break;
        }
        case space_constant::vector: {
          switch (arg2_valued_tag) {
            case space_constant::scalar:
	      obj.template evaluate_call<Result, point_basic<T1>, T2>                (K, value); break;
            case space_constant::vector:
	      obj.template evaluate_call<Result, point_basic<T1>, point_basic<T2> >  (K, value); break;
            case space_constant::tensor:
            case space_constant::unsymmetric_tensor:
	      obj.template evaluate_call<Result, point_basic<T1>, tensor_basic<T2> > (K, value); break;
            case space_constant::tensor3:
	      obj.template evaluate_call<Result, point_basic<T1>, tensor3_basic<T2> > (K, value); break;
            default: error_macro ("unexpected second argument valued tag="<<arg2_valued_tag);
          }
          break;
        }
        case space_constant::tensor:
        case space_constant::unsymmetric_tensor: {
          switch (arg2_valued_tag) {
            case space_constant::scalar:
	      obj.template evaluate_call<Result, tensor_basic<T1>, T2>                (K, value); break;
            case space_constant::vector:
	      obj.template evaluate_call<Result, tensor_basic<T1>, point_basic<T2> >  (K, value); break;
            case space_constant::tensor:
            case space_constant::unsymmetric_tensor:
	      obj.template evaluate_call<Result, tensor_basic<T1>, tensor_basic<T2> > (K, value); break;
            case space_constant::tensor3:
	      obj.template evaluate_call<Result, tensor_basic<T1>, tensor3_basic<T2> > (K, value); break;
            default: error_macro ("unexpected second argument valued tag="<<arg2_valued_tag);
          }
          break;
        }
        case space_constant::tensor3: {
          switch (arg2_valued_tag) {
            case space_constant::scalar:
	      obj.template evaluate_call<Result, tensor3_basic<T1>, T2>                (K, value); break;
            case space_constant::vector:
	      obj.template evaluate_call<Result, tensor3_basic<T1>, point_basic<T2> >  (K, value); break;
            case space_constant::tensor:
            case space_constant::unsymmetric_tensor:
	      obj.template evaluate_call<Result, tensor3_basic<T1>, tensor_basic<T2> > (K, value); break;
            case space_constant::tensor3:
	      obj.template evaluate_call<Result, tensor3_basic<T1>, tensor3_basic<T2> > (K, value); break;
            default: error_macro ("unexpected second argument valued tag="<<arg2_valued_tag);
          }
          break;
        }
        default: error_macro ("unexpected first argument valued tag="<<arg1_valued_tag);
      }
    }
  };
  // when only first arg is defined at compile time:
  template<class This, class Result,
	class Arg1,        space_constant::valued_type Arg1Tag,
  	class Arg2>
  struct evaluate_switch<This, Result,
	Arg1,   Arg1Tag,
        Arg2,   space_constant::last_valued> {
    void operator() (const This& obj, const geo_element& K, std::vector<Result>& value) const {
      typedef typename scalar_traits<Arg2>::type T2;
      space_constant::valued_type arg2_valued_tag = obj._expr2.valued_tag();
      switch (arg2_valued_tag) {
        case space_constant::scalar:
	  obj.template evaluate_call<Result, Arg1, T2>                (K, value); break;
        case space_constant::vector:
	  obj.template evaluate_call<Result, Arg1, point_basic<T2> >  (K, value); break;
        case space_constant::tensor:
        case space_constant::unsymmetric_tensor:
	  obj.template evaluate_call<Result, Arg1, tensor_basic<T2> > (K, value); break;
        case space_constant::tensor3:
	  obj.template evaluate_call<Result, Arg1, tensor3_basic<T2> > (K, value); break;
        default: error_macro ("unexpected second argument valued tag="<<arg2_valued_tag);
      }
    }
  };
  // when only second arg is defined at compile time:
  template<class This, class Result,
	class Arg1,     
  	class Arg2,        space_constant::valued_type Arg2Tag>
  struct evaluate_switch<This, Result,
	Arg1,              space_constant::last_valued,
        Arg2,              Arg2Tag> {
    void operator() (const This& obj, const geo_element& K, std::vector<Result>& value) const {
      typedef typename scalar_traits<Arg1>::type T1;
      space_constant::valued_type arg1_valued_tag = obj._expr1.valued_tag();
      switch (arg1_valued_tag) {
        case space_constant::scalar:
	  obj.template evaluate_call<Result, T1, Arg2>               (K, value); break;
        case space_constant::vector:
	  obj.template evaluate_call<Result, point_basic<T1>, Arg2>  (K, value); break;
        case space_constant::tensor:
        case space_constant::unsymmetric_tensor:
	  obj.template evaluate_call<Result, tensor_basic<T1>, Arg2> (K, value); break;
        case space_constant::tensor3:
	  obj.template evaluate_call<Result, tensor3_basic<T1>, Arg2> (K, value); break;
        default: error_macro ("unexpected first argument valued tag="<<arg1_valued_tag);
      }
    }
  };
  template<class Result>
  bool evaluate (const geo_element& K, std::vector<Result>& value) const
  {
    typedef typename details::generic_binary_traits<BinaryFunction>::template hint<
          typename Expr1::value_type
         ,typename Expr2::value_type
         ,Result>::first_argument_type   A1;
    typedef typename details::generic_binary_traits<BinaryFunction>::template hint<
          typename Expr1::value_type
         ,typename Expr2::value_type
         ,Result>::second_argument_type A2;
    static const space_constant::valued_type  first_argument_tag = space_constant::valued_tag_traits<A1>::value;
    static const space_constant::valued_type second_argument_tag = space_constant::valued_tag_traits<A2>::value;
    typedef field_expr_v2_nonlinear_node_binary<BinaryFunction, Expr1, Expr2> This;
    evaluate_switch <This, Result, A1, first_argument_tag, A2, second_argument_tag>   eval;
    eval (*this, K, value);
    return true;
  }
  template<class Result>
  bool valued_check() const {
    typedef typename details::generic_binary_traits<BinaryFunction>::template hint<
          typename Expr1::value_type
         ,typename Expr2::value_type
         ,Result>::first_argument_type   A1;
    typedef typename details::generic_binary_traits<BinaryFunction>::template hint<
          typename Expr1::value_type
         ,typename Expr2::value_type
         ,Result>::second_argument_type A2;
    bool status = true;
    if (! is_undeterminated<A1>::value)  status &= _expr1.template valued_check<A1>();
    if (! is_undeterminated<A2>::value)  status &= _expr2.template valued_check<A2>();
    return status;
  }

protected:
// data:
  BinaryFunction  _f;
  Expr1           _expr1;
  Expr2           _expr2;
};
template<class F, class Expr1, class Expr2> struct is_field_expr_v2_nonlinear_arg     <field_expr_v2_nonlinear_node_binary<F,Expr1,Expr2> > : std::true_type {};


} // namespace details
// -------------------------------------------
// 2.2. binary calls
// -------------------------------------------
// 2.2.1. plus and minus
// -------------------------------------------
/*
    combination table:

    + -   c L N
        c c L .
        L L L .
        N . . .

    legend;
      c : constant, as scalar, point, tensor, ect
      L : linear: as field, field_indirect, or field_expr_v2 node (linear expr)
      N : function, functor or field_expr_v2_nonlinear node
      . : means that the operation may be implemented here
          => at least one of the two args is of type N (a wrapped nonlin expr)
             and the other is either c, L or N :
	     when c: c value is embeded in bind_first or bind_second
                     and the operation reduces to an  unary one
	     when L: L is embeded in field_expr_v2_nonlinear_terminal_field
	     when N: no wrapper is need
          The L and N cases are grouped, thanks to the wrapper_traits
          and it remains to cases :
	  1) one arg is a field_expr_v2_nonlinear or a function and the second is either
	     a field_expr_v2_nonlinear or a functioon, or a field_expr_v2 (linear) 
	  2) one arg is a field_expr_v2_nonlinear or a function and the second argument is a constant

*/
// ------------------------------------------------------------------------------------
// 2.2.1.1. any args are a constant
// ------------------------------------------------------------------------------------
namespace details {

template <class Expr1, class Expr2>
struct is_field_expr_v2_nonlinear_binary_operator_plus_minus_half
: and_type<
    is_field_expr_v2_nonlinear_arg<Expr1>
   ,not_type <
      is_field_expr_v2_linear_arg <Expr1>
    >
   ,is_field_expr_v2_nonlinear_arg<Expr2>
  >
{};

template <class Expr1, class Expr2>
struct is_field_expr_v2_nonlinear_binary_operator_plus_minus
: or_type<
    is_field_expr_v2_nonlinear_binary_operator_plus_minus_half<Expr1,Expr2>,
    is_field_expr_v2_nonlinear_binary_operator_plus_minus_half<Expr2,Expr1>
  >
{};

} // namespace details

#define _RHEOLEF_make_field_expr_v2_nonlinear_binary_general(FUNCTION,FUNCTOR,IS_EXPR) \
template<class Expr1, class Expr2>						\
inline										\
typename									\
std::enable_if<									\
  IS_EXPR <Expr1,Expr2>::value							\
 ,details::field_expr_v2_nonlinear_node_binary<					\
    FUNCTOR									\
   ,typename details::field_expr_v2_nonlinear_terminal_wrapper_traits<Expr1>::type	\
   ,typename details::field_expr_v2_nonlinear_terminal_wrapper_traits<Expr2>::type 	\
  >										\
>::type										\
FUNCTION (const Expr1& expr1, const Expr2& expr2)				\
{										\
  typedef typename details::field_expr_v2_nonlinear_terminal_wrapper_traits<Expr1>::type wrap1_t; \
  typedef typename details::field_expr_v2_nonlinear_terminal_wrapper_traits<Expr2>::type wrap2_t; \
  return details::field_expr_v2_nonlinear_node_binary <FUNCTOR,wrap1_t,wrap2_t> \
	(FUNCTOR(), wrap1_t(expr1), wrap2_t(expr2)); 	\
}

#define _RHEOLEF_make_field_expr_v2_nonlinear_binary_operator_plus_minus(FUNCTION,FUNCTOR) \
        _RHEOLEF_make_field_expr_v2_nonlinear_binary_general (FUNCTION, FUNCTOR, details::is_field_expr_v2_nonlinear_binary_operator_plus_minus)

_RHEOLEF_make_field_expr_v2_nonlinear_binary_operator_plus_minus (operator+, details::plus)
_RHEOLEF_make_field_expr_v2_nonlinear_binary_operator_plus_minus (operator-, details::minus)
#undef  _RHEOLEF_make_field_expr_v2_nonlinear_binary_operator_plus_minus

// ------------------------------------------------------------------------------------
// 2.2.1.2. one is a constant
// ------------------------------------------------------------------------------------
namespace details {

// avoid 2*(linear expr) that is a linear-expr
template <class Expr1, class Expr2>
struct is_field_expr_v2_nonlinear_binary_operator_constant_at_left
: and_type<
    is_field_expr_v2_constant     <Expr1>
   ,is_field_expr_v2_nonlinear_arg<Expr2>
   ,not_type <
      is_field_expr_v2_linear_arg <Expr2>
    >
  >
{};
template <class Expr1, class Expr2>
struct is_field_expr_v2_nonlinear_binary_operator_constant_at_right
:      is_field_expr_v2_nonlinear_binary_operator_constant_at_left<Expr2,Expr1>
{};

} // namespace details

// general version used by: plus, minus, multiplies and divides
#define _RHEOLEF_make_field_expr_v2_nonlinear_binary_general_constant(FUNCTION, FUNCTOR, IS_AT_LEFT, IS_AT_RIGHT) \
template<class Expr1, class Expr2>						\
inline										\
typename 									\
std::enable_if<									\
  IS_AT_LEFT <Expr1,Expr2>::value, 						\
  details::field_expr_v2_nonlinear_node_unary<					\
    details::binder_first<							\
      FUNCTOR									\
     ,typename details::field_promote_first_argument<				\
        Expr1									\
       ,typename details::field_expr_v2_nonlinear_terminal_wrapper_traits<Expr2>::type::value_type \
      >::type									\
    >										\
   ,typename details::field_expr_v2_nonlinear_terminal_wrapper_traits<Expr2>::type \
  >										\
>::type										\
FUNCTION (const Expr1& expr1, const Expr2& expr2)				\
{										\
  typedef typename details::field_promote_first_argument<			\
        Expr1									\
       ,typename details::field_expr_v2_nonlinear_terminal_wrapper_traits<Expr2>::type::value_type \
      >::type									\
      value_type;								\
  typedef details::binder_first<FUNCTOR,value_type> fun_t;			\
  typedef typename details::field_expr_v2_nonlinear_terminal_wrapper_traits<Expr2>::type wrap2_t; \
  return details::field_expr_v2_nonlinear_node_unary<fun_t,wrap2_t>(fun_t(FUNCTOR(), expr1), wrap2_t(expr2)); \
}										\
template<class Expr1, class Expr2>						\
inline										\
typename 									\
std::enable_if<									\
  IS_AT_RIGHT <Expr1,Expr2>::value, 						\
  details::field_expr_v2_nonlinear_node_unary<					\
    details::binder_second<							\
      FUNCTOR									\
     ,typename details::field_promote_second_argument<				\
        typename details::field_expr_v2_nonlinear_terminal_wrapper_traits<Expr1>::type::value_type \
       ,Expr2									\
      >::type									\
    >										\
   ,typename details::field_expr_v2_nonlinear_terminal_wrapper_traits<Expr1>::type \
  >										\
>::type										\
FUNCTION (const Expr1& expr1, const Expr2& expr2)				\
{										\
  typedef typename details::field_promote_second_argument<			\
        typename details::field_expr_v2_nonlinear_terminal_wrapper_traits<Expr1>::type::value_type \
       ,Expr2									\
      >::type									\
      value_type;								\
  typedef details::binder_second<FUNCTOR,value_type> fun_t;			\
  typedef typename details::field_expr_v2_nonlinear_terminal_wrapper_traits<Expr1>::type wrap1_t; \
  return details::field_expr_v2_nonlinear_node_unary<fun_t,wrap1_t>(fun_t(FUNCTOR(), expr2), wrap1_t(expr1)); \
}

// less general version, used by +-* but not by / which is unsymmetric
#define _RHEOLEF_make_field_expr_v2_nonlinear_binary_operator_constant(FUNCTION,FUNCTOR) 	\
        _RHEOLEF_make_field_expr_v2_nonlinear_binary_general_constant (FUNCTION,FUNCTOR,	\
		details::is_field_expr_v2_nonlinear_binary_operator_constant_at_left,		\
		details::is_field_expr_v2_nonlinear_binary_operator_constant_at_right)

_RHEOLEF_make_field_expr_v2_nonlinear_binary_operator_constant (operator+, details::plus)
_RHEOLEF_make_field_expr_v2_nonlinear_binary_operator_constant (operator-, details::minus)

// note: later undef _RHEOLEF_make_field_expr_v2_nonlinear_binary_operator_constant
//       at it is re-used below

// -------------------------------------------
// 2.2.2. multiplies
// -------------------------------------------
/*
          c L N
        c c L .
        L L . .  
        N . . .

    legend;
      c : constant, as scalar, point, tensor, ect
      L : linear: as field, field_indirect, or field_expr_v2 node (linear expr)
      N : function, functor or field_expr_v2_nonlinear node
      . : means that the operation may be implemented here
          => either : at least one of the two args is of type N 
                      and the other is either c, L or N :
             or     : both two are of type L
	     when c: c value is embeded in bind_first or bind_second
                     and the operation reduces to an  unary one
	     when L: L is embeded in field_expr_v2_nonlinear_terminal_field
	     when N: no wrapper is need
          The L and N cases are grouped, thanks to the wrapper_traits
          and it remains to cases :
	  1) one arg is a field_expr_v2_nonlinear or a function and the second is either
	     a field_expr_v2_nonlinear or a functioon, or a field_expr_v2 (linear) 
	  2) one arg is a field_expr_v2_nonlinear or a function and the second argument is a constant
*/
// ------------------------------------------------------------------------------------
// 2.2.2.1. any args are a constant
// ------------------------------------------------------------------------------------
namespace details {

template <class Expr1, class Expr2>
struct is_field_expr_v2_nonlinear_binary_operator_multiplies
: and_type<
    is_field_expr_v2_nonlinear_arg <Expr1>
   ,is_field_expr_v2_nonlinear_arg <Expr2>
  >
{};

} // namespace details

_RHEOLEF_make_field_expr_v2_nonlinear_binary_general (operator*, details::multiplies, details::is_field_expr_v2_nonlinear_binary_operator_multiplies)

// ------------------------------------------------------------------------------------
// 2.2.2.2. one is a constant
// ------------------------------------------------------------------------------------

_RHEOLEF_make_field_expr_v2_nonlinear_binary_operator_constant (operator*, details::multiplies)
#undef _RHEOLEF_make_field_expr_v2_nonlinear_binary_operator_constant

// -------------------------------------------
// 2.2.3. divides
// -------------------------------------------
/*
          c L N
        c c . .  
        L L . . 
        N . . .

    legend;
      c : constant, as scalar, point, tensor, ect
      L : linear: as field, field_indirect, or field_expr_v2 node (linear expr)
      N : function, functor or field_expr_v2_nonlinear node
      . : means that the operation may be implemented here
          => either : first arg is of type N
                      and the second is either c, L or N :
             or     : the first is L and the second is L or N
          => it remains to cases :
	  1) both args are L or N
	  2) first arg is N and the snd is c
             or first arg is c an the second is L or N
*/
// ------------------------------------------------------------------------------------
// 2.2.3.1. any arg are a constant
// ------------------------------------------------------------------------------------
namespace details {

template <class Expr1, class Expr2>
struct is_field_expr_v2_nonlinear_binary_operator_divides
: and_type<
    is_field_expr_v2_nonlinear_arg <Expr1>
   ,is_field_expr_v2_nonlinear_arg <Expr2>
  >
{};

} // namespace details

_RHEOLEF_make_field_expr_v2_nonlinear_binary_general (operator/, details::divides, details::is_field_expr_v2_nonlinear_binary_operator_divides)

// ------------------------------------------------------------------------------------
// 2.2.3.2. one arg is a constant
// ------------------------------------------------------------------------------------
namespace details {

template <class Expr1, class Expr2>
struct is_field_expr_v2_nonlinear_binary_operator_divides_constant_at_left
: and_type<
    is_field_expr_v2_constant     <Expr1>
   ,is_field_expr_v2_nonlinear_arg<Expr2>
  >
{};
// right version exclude all linear field_expr / constant
template <class Expr1, class Expr2>
struct is_field_expr_v2_nonlinear_binary_operator_divides_constant_at_right
: and_type<
    and_type<
      is_field_expr_v2_nonlinear_arg<Expr1>
     ,not_type <
        is_field_expr_v2_linear_arg <Expr1>
      >
    >
   ,is_field_expr_v2_constant       <Expr2>
  >
{};

} // namespace details

#define _RHEOLEF_make_field_expr_v2_nonlinear_binary_operator_constant_divides(FUNCTION,FUNCTOR) 	\
        _RHEOLEF_make_field_expr_v2_nonlinear_binary_general_constant (FUNCTION,FUNCTOR,	\
		details::is_field_expr_v2_nonlinear_binary_operator_divides_constant_at_left,		\
		details::is_field_expr_v2_nonlinear_binary_operator_divides_constant_at_right)

_RHEOLEF_make_field_expr_v2_nonlinear_binary_operator_constant_divides (operator/, details::divides)

#undef _RHEOLEF_make_field_expr_v2_nonlinear_binary_operator_constant_divides
// -------------------------------------------
// 2.2.4. std::maths
// -------------------------------------------
namespace details {

template <class Expr1, class Expr2>
struct is_field_expr_v2_nonlinear_binary_function
: and_type<
     is_field_expr_v2_nonlinear_arg<Expr1>
    ,is_field_expr_v2_nonlinear_arg<Expr2>
  >
{};

template <class Expr1, class Expr2>
struct is_field_expr_v2_nonlinear_binary_function_constant_at_left
: and_type<
    is_field_expr_v2_constant<Expr1>
   ,is_field_expr_v2_nonlinear_arg<Expr2>
  >
{};

template <class Expr1, class Expr2>
struct is_field_expr_v2_nonlinear_binary_function_constant_at_right
:      is_field_expr_v2_nonlinear_binary_function_constant_at_left<Expr2,Expr1>
{};

} // namespace details


#define _RHEOLEF_make_field_expr_v2_nonlinear_binary_function(FUNCTION)				\
        _RHEOLEF_make_field_expr_v2_nonlinear_binary_general (					\
		FUNCTION, 									\
		details::FUNCTION##_, 								\
		details::is_field_expr_v2_nonlinear_binary_function) 				\
        _RHEOLEF_make_field_expr_v2_nonlinear_binary_general_constant (		\
		FUNCTION,									\
		details::FUNCTION##_,								\
		details::is_field_expr_v2_nonlinear_binary_function_constant_at_left,		\
		details::is_field_expr_v2_nonlinear_binary_function_constant_at_right)

_RHEOLEF_make_field_expr_v2_nonlinear_binary_function (atan2)
_RHEOLEF_make_field_expr_v2_nonlinear_binary_function (pow)
_RHEOLEF_make_field_expr_v2_nonlinear_binary_function (fmod)
_RHEOLEF_make_field_expr_v2_nonlinear_binary_function (min)
_RHEOLEF_make_field_expr_v2_nonlinear_binary_function (max)
_RHEOLEF_make_field_expr_v2_nonlinear_binary_function (dot)
_RHEOLEF_make_field_expr_v2_nonlinear_binary_function (ddot)

#undef _RHEOLEF_make_field_expr_v2_nonlinear_binary_function
#undef _RHEOLEF_make_field_expr_v2_nonlinear_binary_general_constant
#undef _RHEOLEF_make_field_expr_v2_nonlinear_binary_general

// -------------------------------------------
// 2.3. binary compose
// -------------------------------------------
// note: compose 1 & 2 are not reductible to n-ary
//       as it uses deductible return types
// TODO: do not use deductible types => reduces to n-ary !!

// two args are field-expressions
template<class Function, class Expr1, class Expr2>
inline
typename
std::enable_if<
     details::is_field_expr_v2_nonlinear_arg<Expr1>::value
  && details::is_field_expr_v2_nonlinear_arg<Expr2>::value
 ,details::field_expr_v2_nonlinear_node_binary<
    typename details::function_traits<Function>::functor_type
   ,typename details::field_expr_v2_nonlinear_terminal_wrapper_traits<Expr1>::type
   ,typename details::field_expr_v2_nonlinear_terminal_wrapper_traits<Expr2>::type
  >
>::type
compose (const Function& f, const Expr1& expr1, const Expr2& expr2)
{
  typedef typename details::function_traits<Function>::functor_type                      fun_wrap_t;
  typedef typename details::field_expr_v2_nonlinear_terminal_wrapper_traits<Expr1>::type expr1_wrap_t;
  typedef typename details::field_expr_v2_nonlinear_terminal_wrapper_traits<Expr2>::type expr2_wrap_t;
  return details::field_expr_v2_nonlinear_node_binary 
	<fun_wrap_t,    expr1_wrap_t,        expr2_wrap_t> 
	(fun_wrap_t(f), expr1_wrap_t(expr1), expr2_wrap_t(expr2));
}
// left arg is a constant
template <class Function, class Expr1, class Expr2>
inline
typename
std::enable_if<
  details::is_field_expr_v2_nonlinear_binary_function_constant_at_left <Expr1,Expr2>::value,
  details::field_expr_v2_nonlinear_node_unary<
    details::binder_first<
      typename details::function_traits<Function>::functor_type
     ,typename promote<
        Expr1
       ,typename details::field_expr_v2_nonlinear_terminal_wrapper_traits<Expr2>::type::value_type
      >::type
    >
   ,typename details::field_expr_v2_nonlinear_terminal_wrapper_traits<Expr2>::type
  >
>::type
compose (const Function& f, const Expr1& expr1, const Expr2& expr2)
{
  typedef typename promote<
    Expr1
   ,typename details::field_expr_v2_nonlinear_terminal_wrapper_traits<Expr2>::type::value_type
  >::type value_type;
  typedef typename details::function_traits<Function>::functor_type wrap_fun_t;
  typedef details::binder_first<wrap_fun_t,value_type> binded_fun_t;
  typedef typename details::field_expr_v2_nonlinear_terminal_wrapper_traits<Expr2>::type wrap2_t;
  return details::field_expr_v2_nonlinear_node_unary
	   <binded_fun_t,                       wrap2_t>
           (binded_fun_t(wrap_fun_t(f), expr1), wrap2_t(expr2));
}
// right arg is a constant
template <class Function, class Expr1, class Expr2>
inline
typename
std::enable_if<
  details::is_field_expr_v2_nonlinear_binary_function_constant_at_right <Expr1,Expr2>::value,
  details::field_expr_v2_nonlinear_node_unary<
    details::binder_second<
      typename details::function_traits<Function>::functor_type
     ,typename promote<
        typename details::field_expr_v2_nonlinear_terminal_wrapper_traits<Expr1>::type::value_type
       ,Expr2
      >::type
    >
   ,typename details::field_expr_v2_nonlinear_terminal_wrapper_traits<Expr1>::type
  >
>::type
compose (const Function& f, const Expr1& expr1, const Expr2& expr2)
{
  typedef typename promote<
    typename details::field_expr_v2_nonlinear_terminal_wrapper_traits<Expr1>::type::value_type
   ,Expr2
  >::type value_type;
  typedef typename details::function_traits<Function>::functor_type wrap_fun_t;
  typedef details::binder_second<wrap_fun_t,value_type>             binded_fun_t;
  typedef typename details::field_expr_v2_nonlinear_terminal_wrapper_traits<Expr1>::type wrap1_t;
  return details::field_expr_v2_nonlinear_node_unary
	   <binded_fun_t,                       wrap1_t>
           (binded_fun_t(wrap_fun_t(f), expr2), wrap1_t(expr1));
}

} // namespace rheolef
#endif // _RHEOLEF_FIELD_EXPR_V2_NONLINEAR_H
