// Copyright 2015 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "platform/scheduler/renderer/render_widget_signals.h"

#include "base/logging.h"
#include "base/memory/ptr_util.h"
#include "public/platform/scheduler/renderer/render_widget_scheduling_state.h"

namespace blink {
namespace scheduler {

RenderWidgetSignals::RenderWidgetSignals(Observer* observer)
    : observer_(observer),
      num_visible_render_widgets_(0),
      num_visible_render_widgets_with_touch_handlers_(0) {}

std::unique_ptr<RenderWidgetSchedulingState>
RenderWidgetSignals::NewRenderWidgetSchedulingState() {
  return base::WrapUnique(new RenderWidgetSchedulingState(this));
}

void RenderWidgetSignals::IncNumVisibleRenderWidgets() {
  num_visible_render_widgets_++;

  if (num_visible_render_widgets_ == 1)
    observer_->SetAllRenderWidgetsHidden(false);
}

void RenderWidgetSignals::DecNumVisibleRenderWidgets() {
  num_visible_render_widgets_--;
  DCHECK_GE(num_visible_render_widgets_, 0);

  if (num_visible_render_widgets_ == 0)
    observer_->SetAllRenderWidgetsHidden(true);
}

void RenderWidgetSignals::IncNumVisibleRenderWidgetsWithTouchHandlers() {
  num_visible_render_widgets_with_touch_handlers_++;

  if (num_visible_render_widgets_with_touch_handlers_ == 1)
    observer_->SetHasVisibleRenderWidgetWithTouchHandler(true);
}

void RenderWidgetSignals::DecNumVisibleRenderWidgetsWithTouchHandlers() {
  num_visible_render_widgets_with_touch_handlers_--;
  DCHECK_GE(num_visible_render_widgets_with_touch_handlers_, 0);

  if (num_visible_render_widgets_with_touch_handlers_ == 0)
    observer_->SetHasVisibleRenderWidgetWithTouchHandler(false);
}

void RenderWidgetSignals::AsValueInto(
    base::trace_event::TracedValue* state) const {
  state->BeginDictionary("renderer_widget_signals");
  state->SetInteger("num_visible_render_widgets", num_visible_render_widgets_);
  state->SetInteger("num_visible_render_widgets_with_touch_handlers",
                    num_visible_render_widgets_with_touch_handlers_);
  state->EndDictionary();
}

}  // namespace scheduler
}  // namespace blink
