/*
 *@BEGIN LICENSE
 *
 * PSI4: an ab initio quantum chemistry software package
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 *@END LICENSE
 */

#include <libmints/vector.h>
#include "PW91_Cfunctional.h"
#include "utility.h"
#include <cmath>

using namespace psi;

namespace psi {

PW91_CFunctional::PW91_CFunctional()
{
    name_ = "PW91_C";
    description_ = "    PW91 Correlation\n";
    citation_ = "    J.P. Perdew, et. al., Phys. Rev. B., 46(11), 6671-6687, 1992\n";
    alpha_ = 1.0;
    omega_ = 0.0;
    lrc_ = false;
    gga_ = true;
    meta_ = false;
    parameters_["c"] =   6.2035049089939986E-01;
    parameters_["two_13"] =   1.2599210498948732E+00;
    parameters_["k"] =   3.0936677262801355E+00;
    parameters_["pi_m12"] =   5.6418958354775628E-01;
    parameters_["d2fz0"] =   1.7099210000000000E+00;
    parameters_["Aa"] =   1.6886999999999999E-02;
    parameters_["a1a"] =   1.1125000000000000E-01;
    parameters_["b1a"] =   1.0356999999999999E+01;
    parameters_["b2a"] =   3.6231000000000000E+00;
    parameters_["b3a"] =   8.8026000000000004E-01;
    parameters_["b4a"] =   4.9670999999999998E-01;
    parameters_["c0p"] =   3.1091000000000001E-02;
    parameters_["a1p"] =   2.1370000000000000E-01;
    parameters_["b1p"] =   7.5956999999999999E+00;
    parameters_["b2p"] =   3.5876000000000001E+00;
    parameters_["b3p"] =   1.6382000000000001E+00;
    parameters_["b4p"] =   4.9293999999999999E-01;
    parameters_["c0f"] =   1.5545000000000000E-02;
    parameters_["a1f"] =   2.0548000000000000E-01;
    parameters_["b1f"] =   1.4118900000000000E+01;
    parameters_["b2f"] =   6.1977000000000002E+00;
    parameters_["b3f"] =   3.3662000000000001E+00;
    parameters_["b4f"] =   6.2517000000000000E-01;
    parameters_["alph"] =   8.9999999999999997E-02;
    parameters_["bet"] =   6.6726322680061109E-02;
    parameters_["nu"] =   1.5755920349483143E+01;
    parameters_["Cc0"] =   4.2350000000000000E-03;
    parameters_["Cx"] =  -1.6670000000000001E-03;
    parameters_["Cc1"] =   2.5680000000000001E+00;
    parameters_["Cc2"] =   2.3265999999999998E+01;
    parameters_["Cc3"] =   7.3889999999999997E-03;
    parameters_["Cc4"] =   8.7230000000000008E+00;
    parameters_["Cc5"] =   4.7199999999999998E-01;
    parameters_["Cc6"] =   7.3889999999999997E-02;
}
PW91_CFunctional::~PW91_CFunctional()
{
}
void PW91_CFunctional::compute_functional(const std::map<std::string,SharedVector>& in, const std::map<std::string,SharedVector>& out, int npoints, int deriv, double alpha)
{
    double c = parameters_["c"];
    double two_13 = parameters_["two_13"];
    double k = parameters_["k"];
    double pi_m12 = parameters_["pi_m12"];
    double d2fz0 = parameters_["d2fz0"];
    double Aa = parameters_["Aa"];
    double a1a = parameters_["a1a"];
    double b1a = parameters_["b1a"];
    double b2a = parameters_["b2a"];
    double b3a = parameters_["b3a"];
    double b4a = parameters_["b4a"];
    double c0p = parameters_["c0p"];
    double a1p = parameters_["a1p"];
    double b1p = parameters_["b1p"];
    double b2p = parameters_["b2p"];
    double b3p = parameters_["b3p"];
    double b4p = parameters_["b4p"];
    double c0f = parameters_["c0f"];
    double a1f = parameters_["a1f"];
    double b1f = parameters_["b1f"];
    double b2f = parameters_["b2f"];
    double b3f = parameters_["b3f"];
    double b4f = parameters_["b4f"];
    double alph = parameters_["alph"];
    double bet = parameters_["bet"];
    double nu = parameters_["nu"];
    double Cc0 = parameters_["Cc0"];
    double Cx = parameters_["Cx"];
    double Cc1 = parameters_["Cc1"];
    double Cc2 = parameters_["Cc2"];
    double Cc3 = parameters_["Cc3"];
    double Cc4 = parameters_["Cc4"];
    double Cc5 = parameters_["Cc5"];
    double Cc6 = parameters_["Cc6"];

    // Overall scale factor
    double scale = alpha_ * alpha;

    // => Input variables <= //

    double* rho_ap = NULL;
    double* rho_bp = NULL;
    double* gamma_aap = NULL;
    double* gamma_abp = NULL;
    double* gamma_bbp = NULL;
    double* tau_ap = NULL;
    double* tau_bp = NULL;

    if (true) {
        rho_ap = in.find("RHO_A")->second->pointer();
        rho_bp = in.find("RHO_B")->second->pointer();
    }
    if (gga_) {  
        gamma_aap = in.find("GAMMA_AA")->second->pointer();
        gamma_abp = in.find("GAMMA_AB")->second->pointer();
        gamma_bbp = in.find("GAMMA_BB")->second->pointer();
    } 
    if (meta_)  {
        tau_ap = in.find("TAU_A")->second->pointer();
        tau_bp = in.find("TAU_B")->second->pointer();
    }

    // => Outut variables <= //

    double* v = NULL;

    double* v_rho_a = NULL;
    double* v_rho_b = NULL;
    double* v_gamma_aa = NULL;
    double* v_gamma_ab = NULL;
    double* v_gamma_bb = NULL;
    double* v_tau_a = NULL;
    double* v_tau_b = NULL;
     
    double* v_rho_a_rho_a = NULL;
    double* v_rho_a_rho_b = NULL;
    double* v_rho_b_rho_b = NULL;
    double* v_gamma_aa_gamma_aa = NULL;
    double* v_gamma_aa_gamma_ab = NULL;
    double* v_gamma_aa_gamma_bb = NULL;
    double* v_gamma_ab_gamma_ab = NULL;
    double* v_gamma_ab_gamma_bb = NULL;
    double* v_gamma_bb_gamma_bb = NULL;
    double* v_tau_a_tau_a = NULL;
    double* v_tau_a_tau_b = NULL;
    double* v_tau_b_tau_b = NULL;
    double* v_rho_a_gamma_aa = NULL;
    double* v_rho_a_gamma_ab = NULL;
    double* v_rho_a_gamma_bb = NULL;
    double* v_rho_b_gamma_aa = NULL;
    double* v_rho_b_gamma_ab = NULL;
    double* v_rho_b_gamma_bb = NULL;
    double* v_rho_a_tau_a = NULL;
    double* v_rho_a_tau_b = NULL;
    double* v_rho_b_tau_a = NULL;
    double* v_rho_b_tau_b = NULL;
    double* v_gamma_aa_tau_a = NULL;
    double* v_gamma_aa_tau_b = NULL;
    double* v_gamma_ab_tau_a = NULL;
    double* v_gamma_ab_tau_b = NULL;
    double* v_gamma_bb_tau_a = NULL;
    double* v_gamma_bb_tau_b = NULL;

    if (deriv >= 0) {
        v = out.find("V")->second->pointer();
    } 
    if (deriv >= 1) {
        if (true) {
            v_rho_a = out.find("V_RHO_A")->second->pointer();
            v_rho_b = out.find("V_RHO_B")->second->pointer();
        }
        if (gga_) {
            v_gamma_aa = out.find("V_GAMMA_AA")->second->pointer();
            v_gamma_ab = out.find("V_GAMMA_AB")->second->pointer();
            v_gamma_bb = out.find("V_GAMMA_BB")->second->pointer();
        }
        if (meta_) {    
            v_tau_a = out.find("V_TAU_A")->second->pointer();
            v_tau_b = out.find("V_TAU_B")->second->pointer();
        }
    }
    if (deriv >= 2) {
        if (true) {
            v_rho_a_rho_a = out.find("V_RHO_A_RHO_A")->second->pointer();
            v_rho_a_rho_b = out.find("V_RHO_A_RHO_B")->second->pointer();
            v_rho_b_rho_b = out.find("V_RHO_B_RHO_B")->second->pointer();
        }
        if (gga_) {
            v_gamma_aa_gamma_aa = out.find("V_GAMMA_AA_GAMMA_AA")->second->pointer();
            v_gamma_aa_gamma_ab = out.find("V_GAMMA_AA_GAMMA_AB")->second->pointer();
            v_gamma_aa_gamma_bb = out.find("V_GAMMA_AA_GAMMA_BB")->second->pointer();
            v_gamma_ab_gamma_ab = out.find("V_GAMMA_AB_GAMMA_AB")->second->pointer();
            v_gamma_ab_gamma_bb = out.find("V_GAMMA_AB_GAMMA_BB")->second->pointer();
            v_gamma_bb_gamma_bb = out.find("V_GAMMA_BB_GAMMA_BB")->second->pointer();
        }
        if (meta_) {
            v_tau_a_tau_a = out.find("V_TAU_A_TAU_A")->second->pointer();
            v_tau_a_tau_b = out.find("V_TAU_A_TAU_B")->second->pointer();
            v_tau_b_tau_b = out.find("V_TAU_B_TAU_B")->second->pointer();
        }
        if (gga_) {
            v_rho_a_gamma_aa = out.find("V_RHO_A_GAMMA_AA")->second->pointer();
            v_rho_a_gamma_ab = out.find("V_RHO_A_GAMMA_AB")->second->pointer();
            v_rho_a_gamma_bb = out.find("V_RHO_A_GAMMA_BB")->second->pointer();
            v_rho_b_gamma_aa = out.find("V_RHO_B_GAMMA_AA")->second->pointer();
            v_rho_b_gamma_ab = out.find("V_RHO_B_GAMMA_AB")->second->pointer();
            v_rho_b_gamma_bb = out.find("V_RHO_B_GAMMA_BB")->second->pointer();
        }
        if (meta_) {
            v_rho_a_tau_a = out.find("V_RHO_A_TAU_A")->second->pointer();
            v_rho_a_tau_b = out.find("V_RHO_A_TAU_B")->second->pointer();
            v_rho_b_tau_a = out.find("V_RHO_B_TAU_A")->second->pointer();
            v_rho_b_tau_b = out.find("V_RHO_B_TAU_B")->second->pointer();
        }
        if (gga_ && meta_) {
            v_gamma_aa_tau_a = out.find("V_GAMMA_AA_TAU_A")->second->pointer();
            v_gamma_aa_tau_b = out.find("V_GAMMA_AA_TAU_B")->second->pointer();
            v_gamma_ab_tau_a = out.find("V_GAMMA_AB_TAU_A")->second->pointer();
            v_gamma_ab_tau_b = out.find("V_GAMMA_AB_TAU_B")->second->pointer();
            v_gamma_bb_tau_a = out.find("V_GAMMA_BB_TAU_A")->second->pointer();
            v_gamma_bb_tau_b = out.find("V_GAMMA_BB_TAU_B")->second->pointer();
        }
    }

    // => Loop over points <= //

    for (int Q = 0; Q < npoints; Q++) {

        // Input variables 
        double rho_a;
        double rho_b;
        double gamma_aa;
        double gamma_ab;
        double gamma_bb;
        double tau_a;
        double tau_b;

        if (true) {
            rho_a = rho_ap[Q];
            rho_b = rho_bp[Q];
        }        
        if (gga_) {
            gamma_aa = gamma_aap[Q];
            gamma_ab = gamma_abp[Q];
            gamma_bb = gamma_bbp[Q];
        }        
        if (meta_) {
            tau_a = tau_ap[Q];
            tau_b = tau_bp[Q];
        }        

        // Definitions (asymptotics to prevent numerical problems)
        if (rho_a < lsda_cutoff_ && rho_b < lsda_cutoff_) {
            continue;
        } else if (rho_a < lsda_cutoff_) {
            // v
            if (deriv >= 0) {
                double t13151 = rho_a+rho_b;
                double t13152 = 1.0/bet;
                double t13153 = 1.0/pow(t13151,1.0/3.0);
                double t13154 = c*t13153;
                double t13155 = pow(2.0,1.0/3.0);
                double t13156 = sqrt(t13154);
                double t13157 = pow(t13154,3.0/2.0);
                double t13158 = c*c;
                double t13159 = 1.0/pow(t13151,2.0/3.0);
                double t13160 = 1.0/c0p;
                double t13161 = b1p*t13156;
                double t13162 = b3p*t13157;
                double t13163 = b4p*t13158*t13159;
                double t13164 = b2p*c*t13153;
                double t13165 = t13161+t13162+t13163+t13164;
                double t13166 = 1.0/t13165;
                double t13167 = t13160*t13166*(1.0/2.0);
                double t13168 = t13167+1.0;
                double t13169 = log(t13168);
                double t13170 = a1p*c*t13153;
                double t13171 = t13170+1.0;
                double t13172 = c0p*t13171*t13169*2.0;
                double t13173 = gamma_ab*2.0;
                double t13174 = gamma_aa+gamma_bb+t13173;
                double t13175 = 1.0/(bet*bet);
                double t13176 = 1.0/(k*k);
                double t13177 = 1.0/(pi_m12*pi_m12*pi_m12*pi_m12);
                double t13178 = 1.0/pow(t13151,1.4E1/3.0);
                double t13179 = t13155*2.0;
                double t13180 = t13179-2.0;
                double t13181 = two_13*2.0;
                double t13182 = t13181-2.0;
                double t13183 = 1.0/t13182;
                double t13184 = 1.0/c0f;
                double t13185 = b1f*t13156;
                double t13186 = b3f*t13157;
                double t13187 = b4f*t13158*t13159;
                double t13188 = b2f*c*t13153;
                double t13189 = t13185+t13186+t13187+t13188;
                double t13190 = 1.0/t13189;
                double t13191 = t13190*t13184*(1.0/2.0);
                double t13192 = t13191+1.0;
                double t13193 = log(t13192);
                double t13194 = a1f*c*t13153;
                double t13195 = t13194+1.0;
                double t13206 = c0f*t13193*t13195*2.0;
                double t13196 = -t13206+t13172;
                double t13197 = t13172-t13180*t13183*t13196;
                double t13198 = alph*t13175*t13197*4.0;
                double t13199 = exp(t13198);
                double t13200 = t13199-1.0;
                double t13201 = t13174*t13174;
                double t13202 = pow(2.0,2.0/3.0);
                double t13203 = 1.0/k;
                double t13204 = 1.0/(pi_m12*pi_m12);
                double t13205 = 1.0/pow(t13151,7.0/3.0);
                double t13207 = t13206-t13172;
                double t13208 = t13180*t13207*t13183;
                v[Q] += scale * (-t13151*(t13172+t13208-((bet*bet)*log((alph*t13152*(t13202*t13203*t13204*t13205*t13174*(1.0/1.6E1)+(alph*t13201*t13152*t13155*t13176*t13177*t13178*(1.0/6.4E1))/t13200)*2.0)/((alph*t13202*t13203*t13204*t13205*t13152*t13174*(1.0/8.0))/(exp(alph*t13175*(t13172+t13208)*4.0)-1.0)+(alph*alph)*1.0/(t13200*t13200)*t13201*t13155*t13175*t13176*t13177*t13178*(1.0/3.2E1)+1.0)+1.0)*(1.0/4.0))/alph+nu*t13202*t13203*t13204*t13205*t13174*exp(1.0/pow(t13151,8.0/3.0)*t13155*t13174*t13176*(-2.5E1/2.0))*(Cc0+Cx*(1.0E1/7.0)-(Cc1*(1.0/1.0E3)+Cc2*c*t13153*(1.0/1.0E3)+Cc3*t13158*t13159*(1.0/1.0E3))/(Cc4*c*t13153+Cc5*t13158*t13159+(Cc6*c*t13158)/t13151+1.0))*(1.0/3.2E1)));
            }
            
            // v_rho_a
            if (deriv >= 1) {
                double t13210 = rho_a+rho_b;
                double t13211 = 1.0/pow(t13210,1.0/3.0);
                double t13212 = c*t13211;
                double t13213 = sqrt(t13212);
                double t13214 = b1f*t13213;
                double t13215 = pow(t13212,3.0/2.0);
                double t13216 = b3f*t13215;
                double t13217 = c*c;
                double t13218 = 1.0/pow(t13210,2.0/3.0);
                double t13219 = b4f*t13217*t13218;
                double t13220 = b2f*c*t13211;
                double t13221 = t13220+t13214+t13216+t13219;
                double t13222 = 1.0/pow(t13210,4.0/3.0);
                double t13223 = b1p*t13213;
                double t13224 = b3p*t13215;
                double t13225 = b4p*t13217*t13218;
                double t13226 = b2p*c*t13211;
                double t13227 = t13223+t13224+t13225+t13226;
                double t13228 = 1.0/pow(t13210,5.0/3.0);
                double t13229 = 1.0/sqrt(t13212);
                double t13230 = 1.0/c0f;
                double t13231 = 1.0/t13221;
                double t13232 = t13230*t13231*(1.0/2.0);
                double t13233 = t13232+1.0;
                double t13234 = 1.0/c0p;
                double t13235 = 1.0/t13227;
                double t13236 = t13234*t13235*(1.0/2.0);
                double t13237 = t13236+1.0;
                double t13238 = pow(2.0,1.0/3.0);
                double t13239 = 1.0/bet;
                double t13240 = log(t13237);
                double t13241 = a1p*c*t13211;
                double t13242 = t13241+1.0;
                double t13243 = t13238*2.0;
                double t13244 = t13243-2.0;
                double t13245 = two_13*2.0;
                double t13246 = t13245-2.0;
                double t13247 = 1.0/t13246;
                double t13248 = log(t13233);
                double t13249 = a1f*c*t13211;
                double t13250 = t13249+1.0;
                double t13251 = c0p*t13240*t13242*2.0;
                double t13252 = gamma_ab*2.0;
                double t13253 = gamma_aa+gamma_bb+t13252;
                double t13254 = 1.0/(bet*bet);
                double t13255 = 1.0/(k*k);
                double t13256 = 1.0/(pi_m12*pi_m12*pi_m12*pi_m12);
                double t13257 = 1.0/pow(t13210,1.4E1/3.0);
                double t13268 = c0f*t13250*t13248*2.0;
                double t13258 = t13251-t13268;
                double t13269 = t13244*t13247*t13258;
                double t13259 = t13251-t13269;
                double t13260 = alph*t13254*t13259*4.0;
                double t13261 = exp(t13260);
                double t13262 = t13261-1.0;
                double t13263 = t13253*t13253;
                double t13264 = pow(2.0,2.0/3.0);
                double t13265 = 1.0/k;
                double t13266 = 1.0/(pi_m12*pi_m12);
                double t13267 = 1.0/pow(t13210,7.0/3.0);
                double t13270 = 1.0/t13262;
                double t13271 = alph*alph;
                double t13272 = 1.0/(t13262*t13262);
                double t13273 = 1.0/t13233;
                double t13274 = 1.0/(t13221*t13221);
                double t13275 = b4f*t13217*t13228*(2.0/3.0);
                double t13276 = b2f*c*t13222*(1.0/3.0);
                double t13277 = b1f*c*t13222*t13229*(1.0/6.0);
                double t13278 = b3f*c*t13213*t13222*(1.0/2.0);
                double t13279 = t13275+t13276+t13277+t13278;
                double t13280 = t13250*t13273*t13274*t13279;
                double t13281 = 1.0/t13237;
                double t13282 = 1.0/(t13227*t13227);
                double t13283 = b4p*t13217*t13228*(2.0/3.0);
                double t13284 = b2p*c*t13222*(1.0/3.0);
                double t13285 = b1p*c*t13222*t13229*(1.0/6.0);
                double t13286 = b3p*c*t13213*t13222*(1.0/2.0);
                double t13287 = t13283+t13284+t13285+t13286;
                double t13288 = a1p*c*c0p*t13222*t13240*(2.0/3.0);
                double t13291 = t13242*t13281*t13282*t13287;
                double t13302 = a1f*c*c0f*t13222*t13248*(2.0/3.0);
                double t13289 = -t13302+t13280-t13291+t13288;
                double t13290 = t13244*t13247*t13289;
                double t13292 = t13271*t13254*t13263*t13272*t13255*t13238*t13256*t13257*(1.0/3.2E1);
                double t13293 = alph*t13270*t13253*t13264*t13265*t13239*t13266*t13267*(1.0/8.0);
                double t13294 = t13292+t13293+1.0;
                double t13295 = 1.0/t13294;
                double t13296 = t13253*t13264*t13265*t13266*t13267*(1.0/1.6E1);
                double t13297 = alph*t13270*t13263*t13255*t13238*t13256*t13239*t13257*(1.0/6.4E1);
                double t13298 = t13296+t13297;
                double t13299 = 1.0/pow(t13210,1.7E1/3.0);
                double t13300 = 1.0/pow(t13210,1.0E1/3.0);
                double t13301 = 1.0/(bet*bet*bet);
                double t13303 = t13290+t13291-t13288;
                double t13304 = 1.0/pow(t13210,8.0/3.0);
                double t13318 = t13304*t13253*t13255*t13238*(2.5E1/2.0);
                double t13305 = exp(-t13318);
                double t13306 = Cx*(1.0E1/7.0);
                double t13307 = Cc1*(1.0/1.0E3);
                double t13308 = Cc3*t13217*t13218*(1.0/1.0E3);
                double t13309 = Cc2*c*t13211*(1.0/1.0E3);
                double t13310 = t13307+t13308+t13309;
                double t13311 = Cc5*t13217*t13218;
                double t13312 = 1.0/t13210;
                double t13313 = Cc6*c*t13312*t13217;
                double t13314 = Cc4*c*t13211;
                double t13315 = t13311+t13313+t13314+1.0;
                double t13316 = 1.0/t13315;
                double t13323 = t13310*t13316;
                double t13317 = Cc0-t13323+t13306;
                double t13319 = 1.0/alph;
                double t13320 = bet*bet;
                double t13321 = alph*t13239*t13295*t13298*2.0;
                double t13322 = t13321+1.0;
                v_rho_a[Q] += scale * (-t13251-t13210*(t13290+t13291-t13288+(t13320*t13319*(alph*t13239*t13295*(t13300*t13253*t13264*t13265*t13266*(7.0/4.8E1)+alph*t13270*t13263*t13255*t13238*t13256*t13239*t13299*(7.0/9.6E1)+t13301*t13303*t13261*t13271*t13263*t13272*t13255*t13238*t13256*t13257*(1.0/1.6E1))*2.0-alph*t13239*1.0/(t13294*t13294)*t13298*(t13271*t13254*t13263*t13272*t13255*t13238*t13256*t13299*(7.0/4.8E1)+alph*t13300*t13270*t13253*t13264*t13265*t13239*t13266*(7.0/2.4E1)+t13301*t13303*t13261*t13253*t13271*t13272*t13264*t13265*t13266*t13267*(1.0/2.0)+alph*1.0/(bet*bet*bet*bet)*t13303*t13261*1.0/(t13262*t13262*t13262)*t13271*t13263*t13255*t13238*t13256*t13257*(1.0/4.0))*2.0)*(1.0/4.0))/t13322+1.0/(k*k*k)*nu*1.0/(t13210*t13210*t13210*t13210*t13210*t13210)*t13305*t13263*t13317*t13266*(2.5E1/1.2E1)+nu*t13305*t13253*t13264*t13265*t13266*t13267*(t13316*(Cc2*c*t13222*(1.0/3.0E3)+Cc3*t13217*t13228*(1.0/1.5E3))-t13310*1.0/(t13315*t13315)*(Cc4*c*t13222*(1.0/3.0)+Cc5*t13217*t13228*(2.0/3.0)+Cc6*c*1.0/(t13210*t13210)*t13217))*(1.0/3.2E1)-nu*t13300*t13305*t13253*t13317*t13264*t13265*t13266*(7.0/9.6E1))+t13320*t13319*log(t13322)*(1.0/4.0)+t13244*t13247*(t13251-t13268)-nu*t13305*t13253*t13317*t13264*t13265*t13266*t13267*(1.0/3.2E1));
            }
            
            // v_rho_b
            if (deriv >= 1) {
                double t13325 = rho_a+rho_b;
                double t13326 = 1.0/pow(t13325,1.0/3.0);
                double t13327 = c*t13326;
                double t13328 = sqrt(t13327);
                double t13329 = b1f*t13328;
                double t13330 = pow(t13327,3.0/2.0);
                double t13331 = b3f*t13330;
                double t13332 = c*c;
                double t13333 = 1.0/pow(t13325,2.0/3.0);
                double t13334 = b4f*t13332*t13333;
                double t13335 = b2f*c*t13326;
                double t13336 = t13331+t13334+t13335+t13329;
                double t13337 = 1.0/pow(t13325,4.0/3.0);
                double t13338 = b1p*t13328;
                double t13339 = b3p*t13330;
                double t13340 = b4p*t13332*t13333;
                double t13341 = b2p*c*t13326;
                double t13342 = t13340+t13341+t13338+t13339;
                double t13343 = 1.0/pow(t13325,5.0/3.0);
                double t13344 = 1.0/sqrt(t13327);
                double t13345 = 1.0/c0f;
                double t13346 = 1.0/t13336;
                double t13347 = t13345*t13346*(1.0/2.0);
                double t13348 = t13347+1.0;
                double t13349 = 1.0/c0p;
                double t13350 = 1.0/t13342;
                double t13351 = t13350*t13349*(1.0/2.0);
                double t13352 = t13351+1.0;
                double t13353 = pow(2.0,1.0/3.0);
                double t13354 = 1.0/bet;
                double t13355 = log(t13352);
                double t13356 = a1p*c*t13326;
                double t13357 = t13356+1.0;
                double t13358 = t13353*2.0;
                double t13359 = t13358-2.0;
                double t13360 = two_13*2.0;
                double t13361 = t13360-2.0;
                double t13362 = 1.0/t13361;
                double t13363 = log(t13348);
                double t13364 = a1f*c*t13326;
                double t13365 = t13364+1.0;
                double t13366 = c0p*t13355*t13357*2.0;
                double t13367 = gamma_ab*2.0;
                double t13368 = gamma_aa+gamma_bb+t13367;
                double t13369 = 1.0/(bet*bet);
                double t13370 = 1.0/(k*k);
                double t13371 = 1.0/(pi_m12*pi_m12*pi_m12*pi_m12);
                double t13372 = 1.0/pow(t13325,1.4E1/3.0);
                double t13383 = c0f*t13363*t13365*2.0;
                double t13373 = -t13383+t13366;
                double t13374 = t13366-t13362*t13373*t13359;
                double t13375 = alph*t13374*t13369*4.0;
                double t13376 = exp(t13375);
                double t13377 = t13376-1.0;
                double t13378 = t13368*t13368;
                double t13379 = pow(2.0,2.0/3.0);
                double t13380 = 1.0/k;
                double t13381 = 1.0/(pi_m12*pi_m12);
                double t13382 = 1.0/pow(t13325,7.0/3.0);
                double t13384 = t13383-t13366;
                double t13385 = t13362*t13384*t13359;
                double t13386 = t13366+t13385;
                double t13387 = alph*t13386*t13369*4.0;
                double t13388 = exp(t13387);
                double t13389 = t13388-1.0;
                double t13390 = 1.0/t13389;
                double t13391 = alph*alph;
                double t13392 = 1.0/t13348;
                double t13393 = 1.0/(t13336*t13336);
                double t13394 = b4f*t13332*t13343*(2.0/3.0);
                double t13395 = b2f*c*t13337*(1.0/3.0);
                double t13396 = b1f*c*t13344*t13337*(1.0/6.0);
                double t13397 = b3f*c*t13328*t13337*(1.0/2.0);
                double t13398 = t13394+t13395+t13396+t13397;
                double t13399 = t13365*t13392*t13393*t13398;
                double t13400 = 1.0/t13352;
                double t13401 = 1.0/(t13342*t13342);
                double t13402 = b4p*t13332*t13343*(2.0/3.0);
                double t13403 = b2p*c*t13337*(1.0/3.0);
                double t13404 = b1p*c*t13344*t13337*(1.0/6.0);
                double t13405 = b3p*c*t13328*t13337*(1.0/2.0);
                double t13406 = t13402+t13403+t13404+t13405;
                double t13407 = a1p*c*c0p*t13337*t13355*(2.0/3.0);
                double t13410 = t13400*t13401*t13406*t13357;
                double t13419 = a1f*c*c0f*t13363*t13337*(2.0/3.0);
                double t13408 = -t13410+t13407-t13419+t13399;
                double t13409 = t13362*t13408*t13359;
                double t13411 = 1.0/(t13389*t13389);
                double t13412 = alph*t13380*t13354*t13381*t13390*t13382*t13368*t13379*(1.0/8.0);
                double t13413 = t13380*t13381*t13382*t13368*t13379*(1.0/1.6E1);
                double t13414 = t13411*t13370*t13353*t13371*t13372*t13391*t13369*t13378*(1.0/3.2E1);
                double t13415 = t13412+t13414+1.0;
                double t13416 = 1.0/pow(t13325,1.7E1/3.0);
                double t13417 = 1.0/pow(t13325,1.0E1/3.0);
                double t13418 = 1.0/(bet*bet*bet);
                double t13420 = t13410-t13407+t13409;
                double t13421 = 1.0/pow(t13325,8.0/3.0);
                double t13435 = t13421*t13370*t13353*t13368*(2.5E1/2.0);
                double t13422 = exp(-t13435);
                double t13423 = Cx*(1.0E1/7.0);
                double t13424 = Cc1*(1.0/1.0E3);
                double t13425 = Cc3*t13332*t13333*(1.0/1.0E3);
                double t13426 = Cc2*c*t13326*(1.0/1.0E3);
                double t13427 = t13424+t13425+t13426;
                double t13428 = Cc5*t13332*t13333;
                double t13429 = 1.0/t13325;
                double t13430 = Cc6*c*t13332*t13429;
                double t13431 = Cc4*c*t13326;
                double t13432 = t13430+t13431+t13428+1.0;
                double t13433 = 1.0/t13432;
                double t13441 = t13433*t13427;
                double t13434 = Cc0+t13423-t13441;
                double t13436 = 1.0/alph;
                double t13437 = bet*bet;
                double t13438 = alph*t13370*t13353*t13371*t13354*t13372*t13390*t13378*(1.0/6.4E1);
                double t13439 = t13413+t13438;
                double t13440 = 1.0/t13415;
                v_rho_b[Q] += scale * (-t13366-t13385-t13325*(t13410-t13407+t13409+(t13436*t13437*(alph*t13440*t13354*(t13380*t13381*t13417*t13368*t13379*(7.0/4.8E1)+alph*t13370*t13353*t13371*t13416*t13354*t13390*t13378*(7.0/9.6E1)+t13411*t13420*t13370*t13353*t13371*t13372*t13391*t13418*t13378*t13388*(1.0/1.6E1))*2.0-alph*1.0/(t13415*t13415)*t13354*t13439*(t13411*t13370*t13353*t13371*t13416*t13391*t13369*t13378*(7.0/4.8E1)+alph*t13380*t13354*t13381*t13390*t13417*t13368*t13379*(7.0/2.4E1)+t13411*t13420*t13380*t13381*t13382*t13391*t13418*t13368*t13379*t13388*(1.0/2.0)+alph*1.0/(bet*bet*bet*bet)*t13420*t13370*t13353*t13371*t13372*t13391*t13378*t13388*1.0/(t13389*t13389*t13389)*(1.0/4.0))*2.0)*(1.0/4.0))/((alph*t13354*(t13413+(alph*t13370*t13353*t13371*t13354*t13372*t13378*(1.0/6.4E1))/t13377)*2.0)/(t13412+t13370*t13353*t13371*t13372*t13391*1.0/(t13377*t13377)*t13369*t13378*(1.0/3.2E1)+1.0)+1.0)+1.0/(k*k*k)*nu*t13422*1.0/(t13325*t13325*t13325*t13325*t13325*t13325)*t13434*t13381*t13378*(2.5E1/1.2E1)+nu*t13422*t13380*t13381*t13382*t13368*t13379*(t13433*(Cc2*c*t13337*(1.0/3.0E3)+Cc3*t13332*t13343*(1.0/1.5E3))-1.0/(t13432*t13432)*t13427*(Cc4*c*t13337*(1.0/3.0)+Cc5*t13332*t13343*(2.0/3.0)+Cc6*c*t13332*1.0/(t13325*t13325)))*(1.0/3.2E1)-nu*t13422*t13380*t13434*t13381*t13417*t13368*t13379*(7.0/9.6E1))+t13436*t13437*log(alph*t13440*t13354*t13439*2.0+1.0)*(1.0/4.0)-nu*t13422*t13380*t13434*t13381*t13382*t13368*t13379*(1.0/3.2E1));
            }
            
            // v_gamma_aa
            if (deriv >= 1) {
                double t13443 = rho_a+rho_b;
                double t13444 = 1.0/bet;
                double t13445 = 1.0/pow(t13443,1.0/3.0);
                double t13446 = c*t13445;
                double t13447 = pow(2.0,1.0/3.0);
                double t13448 = sqrt(t13446);
                double t13449 = pow(t13446,3.0/2.0);
                double t13450 = c*c;
                double t13451 = 1.0/pow(t13443,2.0/3.0);
                double t13452 = 1.0/c0p;
                double t13453 = b1p*t13448;
                double t13454 = b3p*t13449;
                double t13455 = b4p*t13450*t13451;
                double t13456 = b2p*c*t13445;
                double t13457 = t13453+t13454+t13455+t13456;
                double t13458 = 1.0/t13457;
                double t13459 = t13452*t13458*(1.0/2.0);
                double t13460 = t13459+1.0;
                double t13461 = log(t13460);
                double t13462 = a1p*c*t13445;
                double t13463 = t13462+1.0;
                double t13464 = c0p*t13461*t13463*2.0;
                double t13465 = 1.0/(bet*bet);
                double t13466 = 1.0/(k*k);
                double t13467 = 1.0/(pi_m12*pi_m12*pi_m12*pi_m12);
                double t13468 = 1.0/pow(t13443,1.4E1/3.0);
                double t13469 = t13447*2.0;
                double t13470 = t13469-2.0;
                double t13471 = two_13*2.0;
                double t13472 = t13471-2.0;
                double t13473 = 1.0/t13472;
                double t13474 = 1.0/c0f;
                double t13475 = b1f*t13448;
                double t13476 = b3f*t13449;
                double t13477 = b4f*t13450*t13451;
                double t13478 = b2f*c*t13445;
                double t13479 = t13475+t13476+t13477+t13478;
                double t13480 = 1.0/t13479;
                double t13481 = t13480*t13474*(1.0/2.0);
                double t13482 = t13481+1.0;
                double t13483 = log(t13482);
                double t13484 = a1f*c*t13445;
                double t13485 = t13484+1.0;
                double t13496 = c0f*t13483*t13485*2.0;
                double t13486 = t13464-t13496;
                double t13497 = t13470*t13473*t13486;
                double t13487 = t13464-t13497;
                double t13488 = alph*t13465*t13487*4.0;
                double t13489 = exp(t13488);
                double t13490 = t13489-1.0;
                double t13499 = gamma_ab*2.0;
                double t13491 = gamma_aa+gamma_bb+t13499;
                double t13492 = pow(2.0,2.0/3.0);
                double t13493 = 1.0/k;
                double t13494 = 1.0/(pi_m12*pi_m12);
                double t13495 = 1.0/pow(t13443,7.0/3.0);
                double t13498 = 1.0/t13490;
                double t13500 = alph*alph;
                double t13501 = 1.0/(t13490*t13490);
                double t13502 = gamma_aa*2.0;
                double t13503 = gamma_ab*4.0;
                double t13504 = gamma_bb*2.0;
                double t13505 = t13502+t13503+t13504;
                double t13506 = t13491*t13491;
                double t13507 = t13500*t13501*t13506*t13447*t13465*t13466*t13467*t13468*(1.0/3.2E1);
                double t13508 = alph*t13444*t13491*t13492*t13493*t13494*t13495*t13498*(1.0/8.0);
                double t13509 = t13507+t13508+1.0;
                double t13510 = t13491*t13492*t13493*t13494*t13495*(1.0/1.6E1);
                double t13511 = alph*t13506*t13444*t13447*t13466*t13467*t13468*t13498*(1.0/6.4E1);
                double t13512 = t13510+t13511;
                double t13513 = 1.0/t13509;
                double t13514 = 1.0/pow(t13443,8.0/3.0);
                double t13515 = exp(t13514*t13491*t13447*t13466*(-2.5E1/2.0));
                double t13516 = Cx*(1.0E1/7.0);
                double t13517 = Cc1*(1.0/1.0E3);
                double t13518 = Cc3*t13450*t13451*(1.0/1.0E3);
                double t13519 = Cc2*c*t13445*(1.0/1.0E3);
                double t13520 = t13517+t13518+t13519;
                double t13521 = Cc5*t13450*t13451;
                double t13522 = 1.0/t13443;
                double t13523 = Cc6*c*t13450*t13522;
                double t13524 = Cc4*c*t13445;
                double t13525 = t13521+t13523+t13524+1.0;
                double t13526 = 1.0/t13525;
                double t13527 = Cc0+t13516-t13520*t13526;
                v_gamma_aa[Q] += scale * (t13443*(((bet*bet)*(alph*t13513*t13444*(t13492*t13493*t13494*t13495*(1.0/1.6E1)+alph*t13505*t13444*t13447*t13466*t13467*t13468*t13498*(1.0/6.4E1))*2.0-alph*t13512*t13444*1.0/(t13509*t13509)*(alph*t13444*t13492*t13493*t13494*t13495*t13498*(1.0/8.0)+t13500*t13501*t13505*t13447*t13465*t13466*t13467*t13468*(1.0/3.2E1))*2.0)*(1.0/4.0))/(alph*(alph*t13512*t13513*t13444*2.0+1.0))+1.0/(k*k*k)*nu*1.0/(t13443*t13443*t13443*t13443*t13443)*t13515*t13491*t13527*t13494*(2.5E1/3.2E1)-nu*t13515*t13527*t13492*t13493*t13494*t13495*(1.0/3.2E1)));
            }
            
            // v_gamma_ab
            if (deriv >= 1) {
                double t13529 = rho_a+rho_b;
                double t13530 = 1.0/bet;
                double t13531 = 1.0/pow(t13529,1.0/3.0);
                double t13532 = c*t13531;
                double t13533 = pow(2.0,1.0/3.0);
                double t13534 = sqrt(t13532);
                double t13535 = pow(t13532,3.0/2.0);
                double t13536 = c*c;
                double t13537 = 1.0/pow(t13529,2.0/3.0);
                double t13538 = 1.0/c0p;
                double t13539 = b1p*t13534;
                double t13540 = b3p*t13535;
                double t13541 = b4p*t13536*t13537;
                double t13542 = b2p*c*t13531;
                double t13543 = t13540+t13541+t13542+t13539;
                double t13544 = 1.0/t13543;
                double t13545 = t13544*t13538*(1.0/2.0);
                double t13546 = t13545+1.0;
                double t13547 = log(t13546);
                double t13548 = a1p*c*t13531;
                double t13549 = t13548+1.0;
                double t13550 = c0p*t13547*t13549*2.0;
                double t13551 = 1.0/(bet*bet);
                double t13552 = 1.0/(k*k);
                double t13553 = 1.0/(pi_m12*pi_m12*pi_m12*pi_m12);
                double t13554 = 1.0/pow(t13529,1.4E1/3.0);
                double t13555 = t13533*2.0;
                double t13556 = t13555-2.0;
                double t13557 = two_13*2.0;
                double t13558 = t13557-2.0;
                double t13559 = 1.0/t13558;
                double t13560 = 1.0/c0f;
                double t13561 = b1f*t13534;
                double t13562 = b3f*t13535;
                double t13563 = b4f*t13536*t13537;
                double t13564 = b2f*c*t13531;
                double t13565 = t13561+t13562+t13563+t13564;
                double t13566 = 1.0/t13565;
                double t13567 = t13560*t13566*(1.0/2.0);
                double t13568 = t13567+1.0;
                double t13569 = log(t13568);
                double t13570 = a1f*c*t13531;
                double t13571 = t13570+1.0;
                double t13582 = c0f*t13571*t13569*2.0;
                double t13572 = t13550-t13582;
                double t13583 = t13572*t13556*t13559;
                double t13573 = t13550-t13583;
                double t13574 = alph*t13551*t13573*4.0;
                double t13575 = exp(t13574);
                double t13576 = t13575-1.0;
                double t13585 = gamma_ab*2.0;
                double t13577 = gamma_aa+gamma_bb+t13585;
                double t13578 = pow(2.0,2.0/3.0);
                double t13579 = 1.0/k;
                double t13580 = 1.0/(pi_m12*pi_m12);
                double t13581 = 1.0/pow(t13529,7.0/3.0);
                double t13584 = 1.0/t13576;
                double t13586 = alph*alph;
                double t13587 = 1.0/(t13576*t13576);
                double t13588 = gamma_aa*4.0;
                double t13589 = gamma_ab*8.0;
                double t13590 = gamma_bb*4.0;
                double t13591 = t13590+t13588+t13589;
                double t13592 = t13577*t13577;
                double t13593 = t13533*t13551*t13552*t13553*t13554*t13592*t13586*t13587*(1.0/3.2E1);
                double t13594 = alph*t13530*t13580*t13581*t13584*t13577*t13578*t13579*(1.0/8.0);
                double t13595 = t13593+t13594+1.0;
                double t13596 = t13580*t13581*t13577*t13578*t13579*(1.0/1.6E1);
                double t13597 = alph*t13530*t13533*t13552*t13553*t13554*t13592*t13584*(1.0/6.4E1);
                double t13598 = t13596+t13597;
                double t13599 = 1.0/t13595;
                double t13600 = 1.0/pow(t13529,8.0/3.0);
                double t13601 = exp(t13600*t13533*t13552*t13577*(-2.5E1/2.0));
                double t13602 = Cx*(1.0E1/7.0);
                double t13603 = Cc1*(1.0/1.0E3);
                double t13604 = Cc3*t13536*t13537*(1.0/1.0E3);
                double t13605 = Cc2*c*t13531*(1.0/1.0E3);
                double t13606 = t13603+t13604+t13605;
                double t13607 = Cc5*t13536*t13537;
                double t13608 = 1.0/t13529;
                double t13609 = Cc6*c*t13536*t13608;
                double t13610 = Cc4*c*t13531;
                double t13611 = t13610+t13607+t13609+1.0;
                double t13612 = 1.0/t13611;
                double t13613 = Cc0+t13602-t13612*t13606;
                v_gamma_ab[Q] += scale * (t13529*(((bet*bet)*(alph*t13530*t13599*(t13580*t13581*t13578*t13579*(1.0/8.0)+alph*t13530*t13533*t13552*t13553*t13554*t13591*t13584*(1.0/6.4E1))*2.0-alph*t13530*1.0/(t13595*t13595)*t13598*(alph*t13530*t13580*t13581*t13584*t13578*t13579*(1.0/4.0)+t13533*t13551*t13552*t13553*t13554*t13591*t13586*t13587*(1.0/3.2E1))*2.0)*(1.0/4.0))/(alph*(alph*t13530*t13598*t13599*2.0+1.0))+1.0/(k*k*k)*nu*t13601*t13613*t13580*1.0/(t13529*t13529*t13529*t13529*t13529)*t13577*(2.5E1/1.6E1)-nu*t13601*t13613*t13580*t13581*t13578*t13579*(1.0/1.6E1)));
            }
            
            // v_gamma_bb
            if (deriv >= 1) {
                double t13615 = rho_a+rho_b;
                double t13616 = 1.0/bet;
                double t13617 = 1.0/pow(t13615,1.0/3.0);
                double t13618 = c*t13617;
                double t13619 = pow(2.0,1.0/3.0);
                double t13620 = sqrt(t13618);
                double t13621 = pow(t13618,3.0/2.0);
                double t13622 = c*c;
                double t13623 = 1.0/pow(t13615,2.0/3.0);
                double t13624 = 1.0/c0p;
                double t13625 = b1p*t13620;
                double t13626 = b3p*t13621;
                double t13627 = b4p*t13622*t13623;
                double t13628 = b2p*c*t13617;
                double t13629 = t13625+t13626+t13627+t13628;
                double t13630 = 1.0/t13629;
                double t13631 = t13630*t13624*(1.0/2.0);
                double t13632 = t13631+1.0;
                double t13633 = log(t13632);
                double t13634 = a1p*c*t13617;
                double t13635 = t13634+1.0;
                double t13636 = c0p*t13633*t13635*2.0;
                double t13637 = 1.0/(bet*bet);
                double t13638 = 1.0/(k*k);
                double t13639 = 1.0/(pi_m12*pi_m12*pi_m12*pi_m12);
                double t13640 = 1.0/pow(t13615,1.4E1/3.0);
                double t13641 = t13619*2.0;
                double t13642 = t13641-2.0;
                double t13643 = two_13*2.0;
                double t13644 = t13643-2.0;
                double t13645 = 1.0/t13644;
                double t13646 = 1.0/c0f;
                double t13647 = b1f*t13620;
                double t13648 = b3f*t13621;
                double t13649 = b4f*t13622*t13623;
                double t13650 = b2f*c*t13617;
                double t13651 = t13650+t13647+t13648+t13649;
                double t13652 = 1.0/t13651;
                double t13653 = t13652*t13646*(1.0/2.0);
                double t13654 = t13653+1.0;
                double t13655 = log(t13654);
                double t13656 = a1f*c*t13617;
                double t13657 = t13656+1.0;
                double t13668 = c0f*t13655*t13657*2.0;
                double t13658 = t13636-t13668;
                double t13669 = t13642*t13645*t13658;
                double t13659 = t13636-t13669;
                double t13660 = alph*t13637*t13659*4.0;
                double t13661 = exp(t13660);
                double t13662 = t13661-1.0;
                double t13671 = gamma_ab*2.0;
                double t13663 = gamma_aa+gamma_bb+t13671;
                double t13664 = pow(2.0,2.0/3.0);
                double t13665 = 1.0/k;
                double t13666 = 1.0/(pi_m12*pi_m12);
                double t13667 = 1.0/pow(t13615,7.0/3.0);
                double t13670 = 1.0/t13662;
                double t13672 = alph*alph;
                double t13673 = 1.0/(t13662*t13662);
                double t13674 = gamma_aa*2.0;
                double t13675 = gamma_ab*4.0;
                double t13676 = gamma_bb*2.0;
                double t13677 = t13674+t13675+t13676;
                double t13678 = t13663*t13663;
                double t13679 = t13640*t13672*t13619*t13637*t13673*t13638*t13639*t13678*(1.0/3.2E1);
                double t13680 = alph*t13616*t13670*t13663*t13664*t13665*t13666*t13667*(1.0/8.0);
                double t13681 = t13680+t13679+1.0;
                double t13682 = t13663*t13664*t13665*t13666*t13667*(1.0/1.6E1);
                double t13683 = alph*t13640*t13616*t13670*t13619*t13638*t13639*t13678*(1.0/6.4E1);
                double t13684 = t13682+t13683;
                double t13685 = 1.0/t13681;
                double t13686 = 1.0/pow(t13615,8.0/3.0);
                double t13687 = exp(t13663*t13619*t13638*t13686*(-2.5E1/2.0));
                double t13688 = Cx*(1.0E1/7.0);
                double t13689 = Cc1*(1.0/1.0E3);
                double t13690 = Cc3*t13622*t13623*(1.0/1.0E3);
                double t13691 = Cc2*c*t13617*(1.0/1.0E3);
                double t13692 = t13690+t13691+t13689;
                double t13693 = Cc5*t13622*t13623;
                double t13694 = 1.0/t13615;
                double t13695 = Cc6*c*t13622*t13694;
                double t13696 = Cc4*c*t13617;
                double t13697 = t13693+t13695+t13696+1.0;
                double t13698 = 1.0/t13697;
                double t13699 = Cc0+t13688-t13692*t13698;
                v_gamma_bb[Q] += scale * (t13615*(((bet*bet)*(alph*t13616*t13685*(t13664*t13665*t13666*t13667*(1.0/1.6E1)+alph*t13640*t13616*t13670*t13619*t13638*t13639*t13677*(1.0/6.4E1))*2.0-alph*t13616*1.0/(t13681*t13681)*t13684*(alph*t13616*t13670*t13664*t13665*t13666*t13667*(1.0/8.0)+t13640*t13672*t13619*t13637*t13673*t13638*t13639*t13677*(1.0/3.2E1))*2.0)*(1.0/4.0))/(alph*(alph*t13616*t13684*t13685*2.0+1.0))+1.0/(k*k*k)*nu*1.0/(t13615*t13615*t13615*t13615*t13615)*t13663*t13666*t13687*t13699*(2.5E1/3.2E1)-nu*t13664*t13665*t13666*t13667*t13687*t13699*(1.0/3.2E1)));
            }
            
        } else if (rho_b < lsda_cutoff_) {
            // v
            if (deriv >= 0) {
                double t13703 = rho_a+rho_b;
                double t13704 = 1.0/bet;
                double t13705 = 1.0/pow(t13703,1.0/3.0);
                double t13706 = c*t13705;
                double t13707 = pow(2.0,1.0/3.0);
                double t13708 = sqrt(t13706);
                double t13709 = pow(t13706,3.0/2.0);
                double t13710 = c*c;
                double t13711 = 1.0/pow(t13703,2.0/3.0);
                double t13712 = 1.0/c0p;
                double t13713 = b1p*t13708;
                double t13714 = b3p*t13709;
                double t13715 = b4p*t13710*t13711;
                double t13716 = b2p*c*t13705;
                double t13717 = t13713+t13714+t13715+t13716;
                double t13718 = 1.0/t13717;
                double t13719 = t13712*t13718*(1.0/2.0);
                double t13720 = t13719+1.0;
                double t13721 = log(t13720);
                double t13722 = a1p*c*t13705;
                double t13723 = t13722+1.0;
                double t13724 = c0p*t13721*t13723*2.0;
                double t13725 = gamma_ab*2.0;
                double t13726 = gamma_aa+gamma_bb+t13725;
                double t13727 = 1.0/(bet*bet);
                double t13728 = 1.0/(k*k);
                double t13729 = 1.0/(pi_m12*pi_m12*pi_m12*pi_m12);
                double t13730 = 1.0/pow(t13703,1.4E1/3.0);
                double t13731 = t13707*2.0;
                double t13732 = t13731-2.0;
                double t13733 = two_13*2.0;
                double t13734 = t13733-2.0;
                double t13735 = 1.0/t13734;
                double t13736 = 1.0/c0f;
                double t13737 = b1f*t13708;
                double t13738 = b3f*t13709;
                double t13739 = b4f*t13710*t13711;
                double t13740 = b2f*c*t13705;
                double t13741 = t13740+t13737+t13738+t13739;
                double t13742 = 1.0/t13741;
                double t13743 = t13742*t13736*(1.0/2.0);
                double t13744 = t13743+1.0;
                double t13745 = log(t13744);
                double t13746 = a1f*c*t13705;
                double t13747 = t13746+1.0;
                double t13758 = c0f*t13745*t13747*2.0;
                double t13748 = t13724-t13758;
                double t13759 = t13732*t13735*t13748;
                double t13749 = t13724-t13759;
                double t13750 = alph*t13727*t13749*4.0;
                double t13751 = exp(t13750);
                double t13752 = t13751-1.0;
                double t13753 = t13726*t13726;
                double t13754 = pow(2.0,2.0/3.0);
                double t13755 = 1.0/k;
                double t13756 = 1.0/(pi_m12*pi_m12);
                double t13757 = 1.0/pow(t13703,7.0/3.0);
                double t13760 = 1.0/t13752;
                v[Q] += scale * (-t13703*(t13724-t13759-((bet*bet)*log((alph*t13704*(t13726*t13754*t13755*t13756*t13757*(1.0/1.6E1)+alph*t13730*t13704*t13760*t13707*t13753*t13728*t13729*(1.0/6.4E1))*2.0)/((alph*alph)*t13730*t13707*1.0/(t13752*t13752)*t13753*t13727*t13728*t13729*(1.0/3.2E1)+alph*t13704*t13760*t13726*t13754*t13755*t13756*t13757*(1.0/8.0)+1.0)+1.0)*(1.0/4.0))/alph+nu*t13726*t13754*t13755*t13756*t13757*exp(1.0/pow(t13703,8.0/3.0)*t13707*t13726*t13728*(-2.5E1/2.0))*(Cc0+Cx*(1.0E1/7.0)-(Cc1*(1.0/1.0E3)+Cc2*c*t13705*(1.0/1.0E3)+Cc3*t13710*t13711*(1.0/1.0E3))/(Cc4*c*t13705+Cc5*t13710*t13711+(Cc6*c*t13710)/t13703+1.0))*(1.0/3.2E1)));
            }
            
            // v_rho_a
            if (deriv >= 1) {
                double t13762 = rho_a+rho_b;
                double t13763 = 1.0/pow(t13762,1.0/3.0);
                double t13764 = c*t13763;
                double t13765 = sqrt(t13764);
                double t13766 = b1f*t13765;
                double t13767 = pow(t13764,3.0/2.0);
                double t13768 = b3f*t13767;
                double t13769 = c*c;
                double t13770 = 1.0/pow(t13762,2.0/3.0);
                double t13771 = b4f*t13770*t13769;
                double t13772 = b2f*c*t13763;
                double t13773 = t13771+t13772+t13766+t13768;
                double t13774 = 1.0/pow(t13762,4.0/3.0);
                double t13775 = b1p*t13765;
                double t13776 = b3p*t13767;
                double t13777 = b4p*t13770*t13769;
                double t13778 = b2p*c*t13763;
                double t13779 = t13775+t13776+t13777+t13778;
                double t13780 = 1.0/pow(t13762,5.0/3.0);
                double t13781 = 1.0/sqrt(t13764);
                double t13782 = 1.0/c0f;
                double t13783 = 1.0/t13773;
                double t13784 = t13782*t13783*(1.0/2.0);
                double t13785 = t13784+1.0;
                double t13786 = 1.0/c0p;
                double t13787 = 1.0/t13779;
                double t13788 = t13786*t13787*(1.0/2.0);
                double t13789 = t13788+1.0;
                double t13790 = pow(2.0,1.0/3.0);
                double t13791 = 1.0/bet;
                double t13792 = log(t13789);
                double t13793 = a1p*c*t13763;
                double t13794 = t13793+1.0;
                double t13795 = t13790*2.0;
                double t13796 = t13795-2.0;
                double t13797 = two_13*2.0;
                double t13798 = t13797-2.0;
                double t13799 = 1.0/t13798;
                double t13800 = log(t13785);
                double t13801 = a1f*c*t13763;
                double t13802 = t13801+1.0;
                double t13803 = c0p*t13792*t13794*2.0;
                double t13804 = gamma_ab*2.0;
                double t13805 = gamma_aa+gamma_bb+t13804;
                double t13806 = 1.0/(bet*bet);
                double t13807 = 1.0/(k*k);
                double t13808 = 1.0/(pi_m12*pi_m12*pi_m12*pi_m12);
                double t13809 = 1.0/pow(t13762,1.4E1/3.0);
                double t13820 = c0f*t13800*t13802*2.0;
                double t13810 = -t13820+t13803;
                double t13811 = t13803-t13810*t13796*t13799;
                double t13812 = alph*t13811*t13806*4.0;
                double t13813 = exp(t13812);
                double t13814 = t13813-1.0;
                double t13815 = t13805*t13805;
                double t13816 = pow(2.0,2.0/3.0);
                double t13817 = 1.0/k;
                double t13818 = 1.0/(pi_m12*pi_m12);
                double t13819 = 1.0/pow(t13762,7.0/3.0);
                double t13821 = t13820-t13803;
                double t13822 = t13821*t13796*t13799;
                double t13823 = t13803+t13822;
                double t13824 = alph*t13823*t13806*4.0;
                double t13825 = exp(t13824);
                double t13826 = t13825-1.0;
                double t13827 = 1.0/t13826;
                double t13828 = alph*alph;
                double t13829 = 1.0/t13785;
                double t13830 = 1.0/(t13773*t13773);
                double t13831 = b4f*t13780*t13769*(2.0/3.0);
                double t13832 = b2f*c*t13774*(1.0/3.0);
                double t13833 = b1f*c*t13781*t13774*(1.0/6.0);
                double t13834 = b3f*c*t13765*t13774*(1.0/2.0);
                double t13835 = t13831+t13832+t13833+t13834;
                double t13836 = t13802*t13830*t13835*t13829;
                double t13837 = 1.0/t13789;
                double t13838 = 1.0/(t13779*t13779);
                double t13839 = b4p*t13780*t13769*(2.0/3.0);
                double t13840 = b2p*c*t13774*(1.0/3.0);
                double t13841 = b1p*c*t13781*t13774*(1.0/6.0);
                double t13842 = b3p*c*t13765*t13774*(1.0/2.0);
                double t13843 = t13840+t13841+t13842+t13839;
                double t13844 = a1p*c*c0p*t13774*t13792*(2.0/3.0);
                double t13847 = t13843*t13837*t13838*t13794;
                double t13856 = a1f*c*c0f*t13800*t13774*(2.0/3.0);
                double t13845 = t13844+t13836-t13847-t13856;
                double t13846 = t13845*t13796*t13799;
                double t13848 = 1.0/(t13826*t13826);
                double t13849 = alph*t13805*t13816*t13817*t13791*t13818*t13827*t13819*(1.0/8.0);
                double t13850 = t13805*t13816*t13817*t13818*t13819*(1.0/1.6E1);
                double t13851 = t13806*t13815*t13807*t13790*t13808*t13809*t13828*t13848*(1.0/3.2E1);
                double t13852 = t13851+t13849+1.0;
                double t13853 = 1.0/pow(t13762,1.7E1/3.0);
                double t13854 = 1.0/pow(t13762,1.0E1/3.0);
                double t13855 = 1.0/(bet*bet*bet);
                double t13857 = -t13844+t13846+t13847;
                double t13858 = 1.0/pow(t13762,8.0/3.0);
                double t13872 = t13805*t13807*t13790*t13858*(2.5E1/2.0);
                double t13859 = exp(-t13872);
                double t13860 = Cx*(1.0E1/7.0);
                double t13861 = Cc1*(1.0/1.0E3);
                double t13862 = Cc3*t13770*t13769*(1.0/1.0E3);
                double t13863 = Cc2*c*t13763*(1.0/1.0E3);
                double t13864 = t13861+t13862+t13863;
                double t13865 = Cc5*t13770*t13769;
                double t13866 = 1.0/t13762;
                double t13867 = Cc6*c*t13866*t13769;
                double t13868 = Cc4*c*t13763;
                double t13869 = t13865+t13867+t13868+1.0;
                double t13870 = 1.0/t13869;
                double t13878 = t13870*t13864;
                double t13871 = Cc0+t13860-t13878;
                double t13873 = 1.0/alph;
                double t13874 = bet*bet;
                double t13875 = alph*t13815*t13807*t13790*t13808*t13791*t13809*t13827*(1.0/6.4E1);
                double t13876 = t13850+t13875;
                double t13877 = 1.0/t13852;
                v_rho_a[Q] += scale * (-t13803-t13822-t13762*(-t13844+t13846+t13847+(t13873*t13874*(alph*t13791*t13877*(t13805*t13816*t13817*t13818*t13854*(7.0/4.8E1)+alph*t13815*t13807*t13790*t13808*t13853*t13791*t13827*(7.0/9.6E1)+t13815*t13807*t13825*t13790*t13808*t13809*t13828*t13855*t13848*t13857*(1.0/1.6E1))*2.0-alph*1.0/(t13852*t13852)*t13791*t13876*(t13806*t13815*t13807*t13790*t13808*t13853*t13828*t13848*(7.0/4.8E1)+alph*t13805*t13816*t13817*t13791*t13818*t13827*t13854*(7.0/2.4E1)+t13805*t13816*t13825*t13817*t13818*t13819*t13828*t13855*t13848*t13857*(1.0/2.0)+alph*1.0/(bet*bet*bet*bet)*t13815*t13807*t13825*t13790*t13808*1.0/(t13826*t13826*t13826)*t13809*t13828*t13857*(1.0/4.0))*2.0)*(1.0/4.0))/((alph*t13791*(t13850+(alph*t13815*t13807*t13790*t13808*t13791*t13809*(1.0/6.4E1))/t13814)*2.0)/(t13849+1.0/(t13814*t13814)*t13806*t13815*t13807*t13790*t13808*t13809*t13828*(1.0/3.2E1)+1.0)+1.0)+1.0/(k*k*k)*nu*t13815*1.0/(t13762*t13762*t13762*t13762*t13762*t13762)*t13871*t13818*t13859*(2.5E1/1.2E1)+nu*t13805*t13816*t13817*t13818*t13819*t13859*(t13870*(Cc2*c*t13774*(1.0/3.0E3)+Cc3*t13780*t13769*(1.0/1.5E3))-t13864*1.0/(t13869*t13869)*(Cc4*c*t13774*(1.0/3.0)+Cc5*t13780*t13769*(2.0/3.0)+Cc6*c*1.0/(t13762*t13762)*t13769))*(1.0/3.2E1)-nu*t13805*t13816*t13817*t13871*t13818*t13854*t13859*(7.0/9.6E1))+t13873*t13874*log(alph*t13791*t13876*t13877*2.0+1.0)*(1.0/4.0)-nu*t13805*t13816*t13817*t13871*t13818*t13819*t13859*(1.0/3.2E1));
            }
            
            // v_rho_b
            if (deriv >= 1) {
                double t13880 = rho_a+rho_b;
                double t13881 = 1.0/pow(t13880,1.0/3.0);
                double t13882 = c*t13881;
                double t13883 = sqrt(t13882);
                double t13884 = b1f*t13883;
                double t13885 = pow(t13882,3.0/2.0);
                double t13886 = b3f*t13885;
                double t13887 = c*c;
                double t13888 = 1.0/pow(t13880,2.0/3.0);
                double t13889 = b4f*t13887*t13888;
                double t13890 = b2f*c*t13881;
                double t13891 = t13890+t13884+t13886+t13889;
                double t13892 = 1.0/pow(t13880,4.0/3.0);
                double t13893 = b1p*t13883;
                double t13894 = b3p*t13885;
                double t13895 = b4p*t13887*t13888;
                double t13896 = b2p*c*t13881;
                double t13897 = t13893+t13894+t13895+t13896;
                double t13898 = 1.0/pow(t13880,5.0/3.0);
                double t13899 = 1.0/sqrt(t13882);
                double t13900 = 1.0/c0f;
                double t13901 = 1.0/t13891;
                double t13902 = t13900*t13901*(1.0/2.0);
                double t13903 = t13902+1.0;
                double t13904 = 1.0/c0p;
                double t13905 = 1.0/t13897;
                double t13906 = t13904*t13905*(1.0/2.0);
                double t13907 = t13906+1.0;
                double t13908 = pow(2.0,1.0/3.0);
                double t13909 = 1.0/bet;
                double t13910 = log(t13907);
                double t13911 = a1p*c*t13881;
                double t13912 = t13911+1.0;
                double t13913 = t13908*2.0;
                double t13914 = t13913-2.0;
                double t13915 = two_13*2.0;
                double t13916 = t13915-2.0;
                double t13917 = 1.0/t13916;
                double t13918 = log(t13903);
                double t13919 = a1f*c*t13881;
                double t13920 = t13919+1.0;
                double t13921 = c0p*t13910*t13912*2.0;
                double t13922 = gamma_ab*2.0;
                double t13923 = gamma_aa+gamma_bb+t13922;
                double t13924 = 1.0/(bet*bet);
                double t13925 = 1.0/(k*k);
                double t13926 = 1.0/(pi_m12*pi_m12*pi_m12*pi_m12);
                double t13927 = 1.0/pow(t13880,1.4E1/3.0);
                double t13938 = c0f*t13920*t13918*2.0;
                double t13928 = t13921-t13938;
                double t13939 = t13914*t13917*t13928;
                double t13929 = t13921-t13939;
                double t13930 = alph*t13924*t13929*4.0;
                double t13931 = exp(t13930);
                double t13932 = t13931-1.0;
                double t13933 = t13923*t13923;
                double t13934 = pow(2.0,2.0/3.0);
                double t13935 = 1.0/k;
                double t13936 = 1.0/(pi_m12*pi_m12);
                double t13937 = 1.0/pow(t13880,7.0/3.0);
                double t13940 = 1.0/t13932;
                double t13941 = alph*alph;
                double t13942 = 1.0/(t13932*t13932);
                double t13943 = 1.0/t13903;
                double t13944 = 1.0/(t13891*t13891);
                double t13945 = b4f*t13887*t13898*(2.0/3.0);
                double t13946 = b2f*c*t13892*(1.0/3.0);
                double t13947 = b1f*c*t13892*t13899*(1.0/6.0);
                double t13948 = b3f*c*t13883*t13892*(1.0/2.0);
                double t13949 = t13945+t13946+t13947+t13948;
                double t13950 = t13920*t13943*t13944*t13949;
                double t13951 = 1.0/t13907;
                double t13952 = 1.0/(t13897*t13897);
                double t13953 = b4p*t13887*t13898*(2.0/3.0);
                double t13954 = b2p*c*t13892*(1.0/3.0);
                double t13955 = b1p*c*t13892*t13899*(1.0/6.0);
                double t13956 = b3p*c*t13883*t13892*(1.0/2.0);
                double t13957 = t13953+t13954+t13955+t13956;
                double t13958 = a1p*c*c0p*t13910*t13892*(2.0/3.0);
                double t13961 = t13912*t13951*t13952*t13957;
                double t13972 = a1f*c*c0f*t13918*t13892*(2.0/3.0);
                double t13959 = t13950-t13961-t13972+t13958;
                double t13960 = t13914*t13917*t13959;
                double t13962 = t13941*t13924*t13933*t13942*t13925*t13908*t13926*t13927*(1.0/3.2E1);
                double t13963 = alph*t13940*t13923*t13934*t13935*t13909*t13936*t13937*(1.0/8.0);
                double t13964 = t13962+t13963+1.0;
                double t13965 = 1.0/t13964;
                double t13966 = t13923*t13934*t13935*t13936*t13937*(1.0/1.6E1);
                double t13967 = alph*t13940*t13933*t13925*t13908*t13926*t13909*t13927*(1.0/6.4E1);
                double t13968 = t13966+t13967;
                double t13969 = 1.0/pow(t13880,1.7E1/3.0);
                double t13970 = 1.0/pow(t13880,1.0E1/3.0);
                double t13971 = 1.0/(bet*bet*bet);
                double t13973 = t13960+t13961-t13958;
                double t13974 = 1.0/pow(t13880,8.0/3.0);
                double t13988 = t13923*t13925*t13908*t13974*(2.5E1/2.0);
                double t13975 = exp(-t13988);
                double t13976 = Cx*(1.0E1/7.0);
                double t13977 = Cc1*(1.0/1.0E3);
                double t13978 = Cc3*t13887*t13888*(1.0/1.0E3);
                double t13979 = Cc2*c*t13881*(1.0/1.0E3);
                double t13980 = t13977+t13978+t13979;
                double t13981 = Cc5*t13887*t13888;
                double t13982 = 1.0/t13880;
                double t13983 = Cc6*c*t13982*t13887;
                double t13984 = Cc4*c*t13881;
                double t13985 = t13981+t13983+t13984+1.0;
                double t13986 = 1.0/t13985;
                double t13993 = t13980*t13986;
                double t13987 = Cc0-t13993+t13976;
                double t13989 = 1.0/alph;
                double t13990 = bet*bet;
                double t13991 = alph*t13909*t13965*t13968*2.0;
                double t13992 = t13991+1.0;
                v_rho_b[Q] += scale * (-t13921-t13880*(t13960+t13961-t13958+(t13990*t13989*(alph*t13909*t13965*(t13923*t13934*t13970*t13935*t13936*(7.0/4.8E1)+alph*t13940*t13933*t13925*t13908*t13926*t13909*t13969*(7.0/9.6E1)+t13931*t13941*t13933*t13942*t13925*t13908*t13926*t13971*t13927*t13973*(1.0/1.6E1))*2.0-alph*t13909*1.0/(t13964*t13964)*t13968*(t13941*t13924*t13933*t13942*t13925*t13908*t13926*t13969*(7.0/4.8E1)+alph*t13940*t13923*t13934*t13970*t13935*t13909*t13936*(7.0/2.4E1)+t13931*t13923*t13941*t13942*t13934*t13935*t13971*t13936*t13937*t13973*(1.0/2.0)+alph*1.0/(bet*bet*bet*bet)*t13931*1.0/(t13932*t13932*t13932)*t13941*t13933*t13925*t13908*t13926*t13927*t13973*(1.0/4.0))*2.0)*(1.0/4.0))/t13992+1.0/(k*k*k)*nu*t13933*1.0/(t13880*t13880*t13880*t13880*t13880*t13880)*t13936*t13975*t13987*(2.5E1/1.2E1)+nu*t13923*t13934*t13935*t13936*t13937*t13975*(t13986*(Cc2*c*t13892*(1.0/3.0E3)+Cc3*t13887*t13898*(1.0/1.5E3))-t13980*1.0/(t13985*t13985)*(Cc4*c*t13892*(1.0/3.0)+Cc5*t13887*t13898*(2.0/3.0)+Cc6*c*1.0/(t13880*t13880)*t13887))*(1.0/3.2E1)-nu*t13923*t13934*t13970*t13935*t13936*t13975*t13987*(7.0/9.6E1))+t13990*t13989*log(t13992)*(1.0/4.0)+t13914*t13917*(t13921-t13938)-nu*t13923*t13934*t13935*t13936*t13937*t13975*t13987*(1.0/3.2E1));
            }
            
            // v_gamma_aa
            if (deriv >= 1) {
                double t13995 = rho_a+rho_b;
                double t13996 = 1.0/bet;
                double t13997 = 1.0/pow(t13995,1.0/3.0);
                double t13998 = c*t13997;
                double t13999 = pow(2.0,1.0/3.0);
                double t14000 = sqrt(t13998);
                double t14001 = pow(t13998,3.0/2.0);
                double t14002 = c*c;
                double t14003 = 1.0/pow(t13995,2.0/3.0);
                double t14004 = 1.0/c0p;
                double t14005 = b1p*t14000;
                double t14006 = b3p*t14001;
                double t14007 = b4p*t14002*t14003;
                double t14008 = b2p*c*t13997;
                double t14009 = t14005+t14006+t14007+t14008;
                double t14010 = 1.0/t14009;
                double t14011 = t14010*t14004*(1.0/2.0);
                double t14012 = t14011+1.0;
                double t14013 = log(t14012);
                double t14014 = a1p*c*t13997;
                double t14015 = t14014+1.0;
                double t14016 = c0p*t14013*t14015*2.0;
                double t14017 = 1.0/(bet*bet);
                double t14018 = 1.0/(k*k);
                double t14019 = 1.0/(pi_m12*pi_m12*pi_m12*pi_m12);
                double t14020 = 1.0/pow(t13995,1.4E1/3.0);
                double t14021 = t13999*2.0;
                double t14022 = t14021-2.0;
                double t14023 = two_13*2.0;
                double t14024 = t14023-2.0;
                double t14025 = 1.0/t14024;
                double t14026 = 1.0/c0f;
                double t14027 = b1f*t14000;
                double t14028 = b3f*t14001;
                double t14029 = b4f*t14002*t14003;
                double t14030 = b2f*c*t13997;
                double t14031 = t14030+t14027+t14028+t14029;
                double t14032 = 1.0/t14031;
                double t14033 = t14032*t14026*(1.0/2.0);
                double t14034 = t14033+1.0;
                double t14035 = log(t14034);
                double t14036 = a1f*c*t13997;
                double t14037 = t14036+1.0;
                double t14048 = c0f*t14035*t14037*2.0;
                double t14038 = t14016-t14048;
                double t14049 = t14022*t14025*t14038;
                double t14039 = t14016-t14049;
                double t14040 = alph*t14017*t14039*4.0;
                double t14041 = exp(t14040);
                double t14042 = t14041-1.0;
                double t14051 = gamma_ab*2.0;
                double t14043 = gamma_aa+gamma_bb+t14051;
                double t14044 = pow(2.0,2.0/3.0);
                double t14045 = 1.0/k;
                double t14046 = 1.0/(pi_m12*pi_m12);
                double t14047 = 1.0/pow(t13995,7.0/3.0);
                double t14050 = 1.0/t14042;
                double t14052 = alph*alph;
                double t14053 = 1.0/(t14042*t14042);
                double t14054 = gamma_aa*2.0;
                double t14055 = gamma_ab*4.0;
                double t14056 = gamma_bb*2.0;
                double t14057 = t14054+t14055+t14056;
                double t14058 = t14043*t14043;
                double t14059 = t13999*t14020*t14052*t14017*t14053*t14018*t14019*t14058*(1.0/3.2E1);
                double t14060 = alph*t13996*t14050*t14043*t14044*t14045*t14046*t14047*(1.0/8.0);
                double t14061 = t14060+t14059+1.0;
                double t14062 = t14043*t14044*t14045*t14046*t14047*(1.0/1.6E1);
                double t14063 = alph*t13996*t13999*t14020*t14050*t14018*t14019*t14058*(1.0/6.4E1);
                double t14064 = t14062+t14063;
                double t14065 = 1.0/t14061;
                double t14066 = 1.0/pow(t13995,8.0/3.0);
                double t14067 = exp(t13999*t14043*t14018*t14066*(-2.5E1/2.0));
                double t14068 = Cx*(1.0E1/7.0);
                double t14069 = Cc1*(1.0/1.0E3);
                double t14070 = Cc3*t14002*t14003*(1.0/1.0E3);
                double t14071 = Cc2*c*t13997*(1.0/1.0E3);
                double t14072 = t14070+t14071+t14069;
                double t14073 = Cc5*t14002*t14003;
                double t14074 = 1.0/t13995;
                double t14075 = Cc6*c*t14002*t14074;
                double t14076 = Cc4*c*t13997;
                double t14077 = t14073+t14075+t14076+1.0;
                double t14078 = 1.0/t14077;
                double t14079 = Cc0+t14068-t14072*t14078;
                v_gamma_aa[Q] += scale * (t13995*(((bet*bet)*(alph*t13996*t14065*(t14044*t14045*t14046*t14047*(1.0/1.6E1)+alph*t13996*t13999*t14020*t14050*t14018*t14019*t14057*(1.0/6.4E1))*2.0-alph*t13996*1.0/(t14061*t14061)*t14064*(alph*t13996*t14050*t14044*t14045*t14046*t14047*(1.0/8.0)+t13999*t14020*t14052*t14017*t14053*t14018*t14019*t14057*(1.0/3.2E1))*2.0)*(1.0/4.0))/(alph*(alph*t13996*t14064*t14065*2.0+1.0))+1.0/(k*k*k)*nu*1.0/(t13995*t13995*t13995*t13995*t13995)*t14043*t14046*t14067*t14079*(2.5E1/3.2E1)-nu*t14044*t14045*t14046*t14047*t14067*t14079*(1.0/3.2E1)));
            }
            
            // v_gamma_ab
            if (deriv >= 1) {
                double t14081 = rho_a+rho_b;
                double t14082 = 1.0/bet;
                double t14083 = 1.0/pow(t14081,1.0/3.0);
                double t14084 = c*t14083;
                double t14085 = pow(2.0,1.0/3.0);
                double t14086 = sqrt(t14084);
                double t14087 = pow(t14084,3.0/2.0);
                double t14088 = c*c;
                double t14089 = 1.0/pow(t14081,2.0/3.0);
                double t14090 = 1.0/c0p;
                double t14091 = b1p*t14086;
                double t14092 = b3p*t14087;
                double t14093 = b4p*t14088*t14089;
                double t14094 = b2p*c*t14083;
                double t14095 = t14091+t14092+t14093+t14094;
                double t14096 = 1.0/t14095;
                double t14097 = t14090*t14096*(1.0/2.0);
                double t14098 = t14097+1.0;
                double t14099 = log(t14098);
                double t14100 = a1p*c*t14083;
                double t14101 = t14100+1.0;
                double t14102 = c0p*t14101*t14099*2.0;
                double t14103 = 1.0/(bet*bet);
                double t14104 = 1.0/(k*k);
                double t14105 = 1.0/(pi_m12*pi_m12*pi_m12*pi_m12);
                double t14106 = 1.0/pow(t14081,1.4E1/3.0);
                double t14107 = t14085*2.0;
                double t14108 = t14107-2.0;
                double t14109 = two_13*2.0;
                double t14110 = t14109-2.0;
                double t14111 = 1.0/t14110;
                double t14112 = 1.0/c0f;
                double t14113 = b1f*t14086;
                double t14114 = b3f*t14087;
                double t14115 = b4f*t14088*t14089;
                double t14116 = b2f*c*t14083;
                double t14117 = t14113+t14114+t14115+t14116;
                double t14118 = 1.0/t14117;
                double t14119 = t14112*t14118*(1.0/2.0);
                double t14120 = t14119+1.0;
                double t14121 = log(t14120);
                double t14122 = a1f*c*t14083;
                double t14123 = t14122+1.0;
                double t14134 = c0f*t14121*t14123*2.0;
                double t14124 = t14102-t14134;
                double t14135 = t14111*t14124*t14108;
                double t14125 = t14102-t14135;
                double t14126 = alph*t14103*t14125*4.0;
                double t14127 = exp(t14126);
                double t14128 = t14127-1.0;
                double t14137 = gamma_ab*2.0;
                double t14129 = gamma_aa+gamma_bb+t14137;
                double t14130 = pow(2.0,2.0/3.0);
                double t14131 = 1.0/k;
                double t14132 = 1.0/(pi_m12*pi_m12);
                double t14133 = 1.0/pow(t14081,7.0/3.0);
                double t14136 = 1.0/t14128;
                double t14138 = alph*alph;
                double t14139 = 1.0/(t14128*t14128);
                double t14140 = gamma_aa*4.0;
                double t14141 = gamma_ab*8.0;
                double t14142 = gamma_bb*4.0;
                double t14143 = t14140+t14141+t14142;
                double t14144 = t14129*t14129;
                double t14145 = t14103*t14104*t14105*t14106*t14144*t14138*t14085*t14139*(1.0/3.2E1);
                double t14146 = alph*t14130*t14131*t14132*t14133*t14082*t14136*t14129*(1.0/8.0);
                double t14147 = t14145+t14146+1.0;
                double t14148 = t14130*t14131*t14132*t14133*t14129*(1.0/1.6E1);
                double t14149 = alph*t14104*t14105*t14106*t14144*t14082*t14136*t14085*(1.0/6.4E1);
                double t14150 = t14148+t14149;
                double t14151 = 1.0/t14147;
                double t14152 = 1.0/pow(t14081,8.0/3.0);
                double t14153 = exp(t14104*t14152*t14129*t14085*(-2.5E1/2.0));
                double t14154 = Cx*(1.0E1/7.0);
                double t14155 = Cc1*(1.0/1.0E3);
                double t14156 = Cc3*t14088*t14089*(1.0/1.0E3);
                double t14157 = Cc2*c*t14083*(1.0/1.0E3);
                double t14158 = t14155+t14156+t14157;
                double t14159 = Cc5*t14088*t14089;
                double t14160 = 1.0/t14081;
                double t14161 = Cc6*c*t14160*t14088;
                double t14162 = Cc4*c*t14083;
                double t14163 = t14161+t14162+t14159+1.0;
                double t14164 = 1.0/t14163;
                double t14165 = Cc0+t14154-t14164*t14158;
                v_gamma_ab[Q] += scale * (t14081*(((bet*bet)*(alph*t14151*t14082*(t14130*t14131*t14132*t14133*(1.0/8.0)+alph*t14104*t14105*t14106*t14143*t14082*t14136*t14085*(1.0/6.4E1))*2.0-alph*t14150*t14082*1.0/(t14147*t14147)*(alph*t14130*t14131*t14132*t14133*t14082*t14136*(1.0/4.0)+t14103*t14104*t14105*t14106*t14143*t14138*t14085*t14139*(1.0/3.2E1))*2.0)*(1.0/4.0))/(alph*(alph*t14150*t14151*t14082*2.0+1.0))+1.0/(k*k*k)*nu*t14132*1.0/(t14081*t14081*t14081*t14081*t14081)*t14153*t14129*t14165*(2.5E1/1.6E1)-nu*t14130*t14131*t14132*t14133*t14153*t14165*(1.0/1.6E1)));
            }
            
            // v_gamma_bb
            if (deriv >= 1) {
                double t14167 = rho_a+rho_b;
                double t14168 = 1.0/bet;
                double t14169 = 1.0/pow(t14167,1.0/3.0);
                double t14170 = c*t14169;
                double t14171 = pow(2.0,1.0/3.0);
                double t14172 = sqrt(t14170);
                double t14173 = pow(t14170,3.0/2.0);
                double t14174 = c*c;
                double t14175 = 1.0/pow(t14167,2.0/3.0);
                double t14176 = 1.0/c0p;
                double t14177 = b1p*t14172;
                double t14178 = b3p*t14173;
                double t14179 = b4p*t14174*t14175;
                double t14180 = b2p*c*t14169;
                double t14181 = t14180+t14177+t14178+t14179;
                double t14182 = 1.0/t14181;
                double t14183 = t14182*t14176*(1.0/2.0);
                double t14184 = t14183+1.0;
                double t14185 = log(t14184);
                double t14186 = a1p*c*t14169;
                double t14187 = t14186+1.0;
                double t14188 = c0p*t14185*t14187*2.0;
                double t14189 = 1.0/(bet*bet);
                double t14190 = 1.0/(k*k);
                double t14191 = 1.0/(pi_m12*pi_m12*pi_m12*pi_m12);
                double t14192 = 1.0/pow(t14167,1.4E1/3.0);
                double t14193 = t14171*2.0;
                double t14194 = t14193-2.0;
                double t14195 = two_13*2.0;
                double t14196 = t14195-2.0;
                double t14197 = 1.0/t14196;
                double t14198 = 1.0/c0f;
                double t14199 = b1f*t14172;
                double t14200 = b3f*t14173;
                double t14201 = b4f*t14174*t14175;
                double t14202 = b2f*c*t14169;
                double t14203 = t14200+t14201+t14202+t14199;
                double t14204 = 1.0/t14203;
                double t14205 = t14204*t14198*(1.0/2.0);
                double t14206 = t14205+1.0;
                double t14207 = log(t14206);
                double t14208 = a1f*c*t14169;
                double t14209 = t14208+1.0;
                double t14220 = c0f*t14207*t14209*2.0;
                double t14210 = -t14220+t14188;
                double t14211 = t14188-t14210*t14194*t14197;
                double t14212 = alph*t14211*t14189*4.0;
                double t14213 = exp(t14212);
                double t14214 = t14213-1.0;
                double t14221 = gamma_ab*2.0;
                double t14215 = gamma_aa+gamma_bb+t14221;
                double t14216 = pow(2.0,2.0/3.0);
                double t14217 = 1.0/k;
                double t14218 = 1.0/(pi_m12*pi_m12);
                double t14219 = 1.0/pow(t14167,7.0/3.0);
                double t14222 = t14220-t14188;
                double t14223 = t14222*t14194*t14197;
                double t14224 = t14223+t14188;
                double t14225 = alph*t14224*t14189*4.0;
                double t14226 = exp(t14225);
                double t14227 = t14226-1.0;
                double t14228 = 1.0/t14227;
                double t14229 = alph*alph;
                double t14230 = gamma_aa*2.0;
                double t14231 = gamma_ab*4.0;
                double t14232 = gamma_bb*2.0;
                double t14233 = t14230+t14231+t14232;
                double t14234 = t14215*t14215;
                double t14235 = 1.0/(t14227*t14227);
                double t14236 = alph*t14215*t14216*t14217*t14218*t14219*t14228*t14168*(1.0/8.0);
                double t14237 = t14215*t14216*t14217*t14218*t14219*(1.0/1.6E1);
                double t14238 = alph*t14171*t14234*t14190*t14191*t14192*t14228*t14168*(1.0/6.4E1);
                double t14239 = t14237+t14238;
                double t14240 = t14171*t14234*t14190*t14235*t14191*t14192*t14229*t14189*(1.0/3.2E1);
                double t14241 = t14240+t14236+1.0;
                double t14242 = 1.0/pow(t14167,8.0/3.0);
                double t14243 = exp(t14215*t14242*t14171*t14190*(-2.5E1/2.0));
                double t14244 = Cx*(1.0E1/7.0);
                double t14245 = Cc1*(1.0/1.0E3);
                double t14246 = Cc3*t14174*t14175*(1.0/1.0E3);
                double t14247 = Cc2*c*t14169*(1.0/1.0E3);
                double t14248 = t14245+t14246+t14247;
                double t14249 = Cc5*t14174*t14175;
                double t14250 = 1.0/t14167;
                double t14251 = Cc6*c*t14250*t14174;
                double t14252 = Cc4*c*t14169;
                double t14253 = t14251+t14252+t14249+1.0;
                double t14254 = 1.0/t14253;
                double t14255 = Cc0+t14244-t14254*t14248;
                v_gamma_bb[Q] += scale * (t14167*(((bet*bet)*((alph*t14168*(t14216*t14217*t14218*t14219*(1.0/1.6E1)+(alph*t14233*t14171*t14190*t14191*t14192*t14168*(1.0/6.4E1))/t14214)*2.0)/(t14236+1.0/(t14214*t14214)*t14171*t14234*t14190*t14191*t14192*t14229*t14189*(1.0/3.2E1)+1.0)-alph*1.0/(t14241*t14241)*t14239*t14168*(alph*t14216*t14217*t14218*t14219*t14228*t14168*(1.0/8.0)+t14233*t14171*t14190*t14235*t14191*t14192*t14229*t14189*(1.0/3.2E1))*2.0)*(1.0/4.0))/(alph*((alph*t14239*t14168*2.0)/t14241+1.0))+1.0/(k*k*k)*nu*t14215*t14243*t14218*t14255*1.0/(t14167*t14167*t14167*t14167*t14167)*(2.5E1/3.2E1)-nu*t14216*t14243*t14217*t14218*t14219*t14255*(1.0/3.2E1)));
            }
            
        } else {
            // v
            if (deriv >= 0) {
                double t12343 = rho_a+rho_b;
                double t12344 = 1.0/pow(t12343,1.0/3.0);
                double t12345 = c*t12344;
                double t12346 = 1.0/t12343;
                double t12347 = rho_a-rho_b;
                double t12348 = t12346*t12347;
                double t12349 = 1.0/bet;
                double t12350 = t12348+1.0;
                double t12351 = pow(t12350,2.0/3.0);
                double t12352 = t12351*(1.0/2.0);
                double t12353 = -t12348+1.0;
                double t12354 = pow(t12353,2.0/3.0);
                double t12355 = t12354*(1.0/2.0);
                double t12356 = t12352+t12355;
                double t12357 = 1.0/c0p;
                double t12358 = sqrt(t12345);
                double t12359 = b1p*t12358;
                double t12360 = pow(t12345,3.0/2.0);
                double t12361 = b3p*t12360;
                double t12362 = c*c;
                double t12363 = 1.0/pow(t12343,2.0/3.0);
                double t12364 = b4p*t12362*t12363;
                double t12365 = b2p*c*t12344;
                double t12366 = t12361+t12364+t12365+t12359;
                double t12367 = 1.0/t12366;
                double t12368 = t12357*t12367*(1.0/2.0);
                double t12369 = t12368+1.0;
                double t12370 = log(t12369);
                double t12371 = a1p*c*t12344;
                double t12372 = t12371+1.0;
                double t12373 = c0p*t12370*t12372*2.0;
                double t12374 = t12347*t12347;
                double t12375 = two_13*2.0;
                double t12376 = t12375-2.0;
                double t12377 = 1.0/t12376;
                double t12378 = 1.0/(t12343*t12343*t12343*t12343);
                double t12379 = t12374*t12374;
                double t12380 = pow(t12350,4.0/3.0);
                double t12381 = pow(t12353,4.0/3.0);
                double t12382 = t12380+t12381-2.0;
                double t12383 = gamma_ab*2.0;
                double t12384 = gamma_aa+gamma_bb+t12383;
                double t12385 = 1.0/(bet*bet);
                double t12386 = 1.0/(k*k);
                double t12387 = 1.0/(pi_m12*pi_m12*pi_m12*pi_m12);
                double t12388 = 1.0/(t12356*t12356*t12356);
                double t12389 = 1.0/c0f;
                double t12390 = b1f*t12358;
                double t12391 = b3f*t12360;
                double t12392 = b4f*t12362*t12363;
                double t12393 = b2f*c*t12344;
                double t12394 = t12390+t12391+t12392+t12393;
                double t12395 = 1.0/t12394;
                double t12396 = t12395*t12389*(1.0/2.0);
                double t12397 = t12396+1.0;
                double t12398 = log(t12397);
                double t12399 = a1f*c*t12344;
                double t12400 = t12399+1.0;
                double t12427 = c0f*t12400*t12398*2.0;
                double t12401 = t12373-t12427;
                double t12402 = 1.0/d2fz0;
                double t12403 = 1.0/Aa;
                double t12404 = b1a*t12358;
                double t12405 = b3a*t12360;
                double t12406 = b4a*t12362*t12363;
                double t12407 = b2a*c*t12344;
                double t12408 = t12404+t12405+t12406+t12407;
                double t12409 = 1.0/t12408;
                double t12410 = t12403*t12409*(1.0/2.0);
                double t12411 = t12410+1.0;
                double t12412 = log(t12411);
                double t12413 = a1a*c*t12344;
                double t12414 = t12413+1.0;
                double t12415 = t12378*t12379;
                double t12416 = t12415-1.0;
                double t12417 = Aa*t12402*t12412*t12414*t12416*t12382*t12377*2.0;
                double t12428 = t12401*t12382*t12377*t12378*t12379;
                double t12418 = t12417+t12373-t12428;
                double t12419 = alph*t12418*t12385*t12388*2.0;
                double t12420 = exp(t12419);
                double t12421 = t12420-1.0;
                double t12422 = 1.0/pow(t12343,1.4E1/3.0);
                double t12423 = 1.0/(t12356*t12356*t12356*t12356);
                double t12424 = t12384*t12384;
                double t12425 = 1.0/k;
                double t12426 = 1.0/(pi_m12*pi_m12);
                double t12429 = 1.0/t12421;
                double t12430 = 1.0/pow(t12343,7.0/3.0);
                double t12431 = 1.0/(t12356*t12356);
                double t12432 = t12356*t12356;
                v[Q] += scale * (-t12343*(t12417+t12373-t12428-((bet*bet)*t12432*t12356*log((alph*t12349*(t12430*t12431*t12425*t12426*t12384*(1.0/1.6E1)+alph*t12422*t12423*t12424*t12429*t12349*t12386*t12387*(1.0/1.28E2))*2.0)/((alph*alph)*1.0/(t12421*t12421)*t12422*t12423*t12424*t12385*t12386*t12387*(1.0/6.4E1)+alph*t12430*t12431*t12425*t12426*t12384*t12429*t12349*(1.0/8.0)+1.0)+1.0)*(1.0/2.0))/alph+nu*t12430*t12425*t12426*t12356*t12384*exp(t12432*1.0/pow(t12343,8.0/3.0)*t12384*t12386*-2.5E1)*(Cc0+Cx*(1.0E1/7.0)-(Cc1*(1.0/1.0E3)+Cc2*c*t12344*(1.0/1.0E3)+Cc3*t12362*t12363*(1.0/1.0E3))/(Cc4*c*t12344+Cc5*t12362*t12363+Cc6*c*t12362*t12346+1.0))*(1.0/1.6E1)));
            }
            
            // v_rho_a
            if (deriv >= 1) {
                double t12434 = rho_a+rho_b;
                double t12435 = 1.0/pow(t12434,1.0/3.0);
                double t12436 = c*t12435;
                double t12437 = sqrt(t12436);
                double t12438 = b1p*t12437;
                double t12439 = pow(t12436,3.0/2.0);
                double t12440 = b3p*t12439;
                double t12441 = c*c;
                double t12442 = 1.0/pow(t12434,2.0/3.0);
                double t12443 = b4p*t12441*t12442;
                double t12444 = b2p*c*t12435;
                double t12445 = t12440+t12443+t12444+t12438;
                double t12446 = 1.0/pow(t12434,4.0/3.0);
                double t12447 = 1.0/t12434;
                double t12448 = rho_a-rho_b;
                double t12449 = t12447*t12448;
                double t12450 = 1.0/bet;
                double t12451 = t12449+1.0;
                double t12452 = pow(t12451,2.0/3.0);
                double t12453 = t12452*(1.0/2.0);
                double t12454 = -t12449+1.0;
                double t12455 = pow(t12454,2.0/3.0);
                double t12456 = t12455*(1.0/2.0);
                double t12457 = t12453+t12456;
                double t12458 = 1.0/c0p;
                double t12459 = 1.0/t12445;
                double t12460 = t12458*t12459*(1.0/2.0);
                double t12461 = t12460+1.0;
                double t12462 = a1p*c*t12435;
                double t12463 = t12462+1.0;
                double t12464 = log(t12461);
                double t12465 = c0p*t12463*t12464*2.0;
                double t12466 = t12448*t12448;
                double t12467 = two_13*2.0;
                double t12468 = t12467-2.0;
                double t12469 = 1.0/t12468;
                double t12470 = 1.0/(t12434*t12434*t12434*t12434);
                double t12471 = t12466*t12466;
                double t12472 = pow(t12451,4.0/3.0);
                double t12473 = pow(t12454,4.0/3.0);
                double t12474 = t12472+t12473-2.0;
                double t12475 = gamma_ab*2.0;
                double t12476 = gamma_aa+gamma_bb+t12475;
                double t12477 = 1.0/(bet*bet);
                double t12478 = 1.0/(k*k);
                double t12479 = 1.0/(pi_m12*pi_m12*pi_m12*pi_m12);
                double t12480 = 1.0/(t12457*t12457*t12457);
                double t12481 = 1.0/c0f;
                double t12482 = b1f*t12437;
                double t12483 = b3f*t12439;
                double t12484 = b4f*t12441*t12442;
                double t12485 = b2f*c*t12435;
                double t12486 = t12482+t12483+t12484+t12485;
                double t12487 = 1.0/t12486;
                double t12488 = t12481*t12487*(1.0/2.0);
                double t12489 = t12488+1.0;
                double t12490 = log(t12489);
                double t12491 = a1f*c*t12435;
                double t12492 = t12491+1.0;
                double t12519 = c0f*t12490*t12492*2.0;
                double t12493 = t12465-t12519;
                double t12494 = 1.0/d2fz0;
                double t12495 = 1.0/Aa;
                double t12496 = b1a*t12437;
                double t12497 = b3a*t12439;
                double t12498 = b4a*t12441*t12442;
                double t12499 = b2a*c*t12435;
                double t12500 = t12496+t12497+t12498+t12499;
                double t12501 = 1.0/t12500;
                double t12502 = t12501*t12495*(1.0/2.0);
                double t12503 = t12502+1.0;
                double t12504 = log(t12503);
                double t12505 = a1a*c*t12435;
                double t12506 = t12505+1.0;
                double t12507 = t12470*t12471;
                double t12508 = t12507-1.0;
                double t12509 = Aa*t12504*t12506*t12508*t12474*t12494*t12469*2.0;
                double t12520 = t12470*t12471*t12474*t12493*t12469;
                double t12510 = -t12520+t12509+t12465;
                double t12511 = alph*t12510*t12480*t12477*2.0;
                double t12512 = exp(t12511);
                double t12513 = t12512-1.0;
                double t12514 = 1.0/pow(t12434,1.4E1/3.0);
                double t12515 = 1.0/(t12457*t12457*t12457*t12457);
                double t12516 = t12476*t12476;
                double t12517 = 1.0/k;
                double t12518 = 1.0/(pi_m12*pi_m12);
                double t12521 = 1.0/t12513;
                double t12522 = 1.0/pow(t12434,7.0/3.0);
                double t12523 = 1.0/(t12457*t12457);
                double t12524 = 1.0/(t12434*t12434);
                double t12540 = t12524*t12448;
                double t12525 = -t12540+t12447;
                double t12526 = 1.0/alph;
                double t12527 = bet*bet;
                double t12528 = t12522*t12523*t12517*t12518*t12476*(1.0/1.6E1);
                double t12529 = alph*t12521*t12450*t12514*t12515*t12516*t12478*t12479*(1.0/1.28E2);
                double t12530 = t12528+t12529;
                double t12531 = alph*alph;
                double t12532 = 1.0/(t12513*t12513);
                double t12533 = t12531*t12514*t12532*t12515*t12516*t12477*t12478*t12479*(1.0/6.4E1);
                double t12534 = alph*t12521*t12450*t12522*t12523*t12517*t12518*t12476*(1.0/8.0);
                double t12535 = t12533+t12534+1.0;
                double t12536 = 1.0/t12535;
                double t12537 = alph*t12530*t12450*t12536*2.0;
                double t12538 = t12537+1.0;
                double t12539 = 1.0/pow(t12451,1.0/3.0);
                double t12541 = t12525*t12539*(1.0/3.0);
                double t12542 = 1.0/pow(t12454,1.0/3.0);
                double t12544 = t12542*t12525*(1.0/3.0);
                double t12543 = t12541-t12544;
                double t12545 = 1.0/t12461;
                double t12546 = 1.0/(t12445*t12445);
                double t12547 = 1.0/pow(t12434,5.0/3.0);
                double t12548 = b4p*t12441*t12547*(2.0/3.0);
                double t12549 = b2p*c*t12446*(1.0/3.0);
                double t12550 = 1.0/sqrt(t12436);
                double t12551 = b1p*c*t12550*t12446*(1.0/6.0);
                double t12552 = b3p*c*t12437*t12446*(1.0/2.0);
                double t12553 = t12551+t12552+t12548+t12549;
                double t12554 = t12463*t12553*t12545*t12546;
                double t12555 = 1.0/(t12434*t12434*t12434*t12434*t12434);
                double t12556 = pow(t12451,1.0/3.0);
                double t12557 = t12525*t12556*(4.0/3.0);
                double t12558 = pow(t12454,1.0/3.0);
                double t12564 = t12525*t12558*(4.0/3.0);
                double t12559 = -t12564+t12557;
                double t12560 = 1.0/pow(t12434,1.7E1/3.0);
                double t12561 = 1.0/pow(t12434,1.0E1/3.0);
                double t12562 = 1.0/(t12457*t12457*t12457*t12457*t12457);
                double t12563 = t12470*t12474*t12448*t12466*t12493*t12469*4.0;
                double t12565 = t12470*t12471*t12493*t12469*t12559;
                double t12566 = 1.0/t12489;
                double t12567 = 1.0/(t12486*t12486);
                double t12568 = b4f*t12441*t12547*(2.0/3.0);
                double t12569 = b2f*c*t12446*(1.0/3.0);
                double t12570 = b1f*c*t12550*t12446*(1.0/6.0);
                double t12571 = b3f*c*t12437*t12446*(1.0/2.0);
                double t12572 = t12570+t12571+t12568+t12569;
                double t12573 = a1f*c*c0f*t12490*t12446*(2.0/3.0);
                double t12574 = a1p*c*c0p*t12446*t12464*(2.0/3.0);
                double t12575 = t12471*t12555*4.0;
                double t12589 = t12470*t12448*t12466*4.0;
                double t12576 = t12575-t12589;
                double t12577 = Aa*t12504*t12506*t12474*t12494*t12576*t12469*2.0;
                double t12578 = 1.0/t12503;
                double t12579 = 1.0/(t12500*t12500);
                double t12580 = b4a*t12441*t12547*(2.0/3.0);
                double t12581 = b2a*c*t12446*(1.0/3.0);
                double t12582 = b1a*c*t12550*t12446*(1.0/6.0);
                double t12583 = b3a*c*t12437*t12446*(1.0/2.0);
                double t12584 = t12580+t12581+t12582+t12583;
                double t12585 = Aa*a1a*c*t12504*t12508*t12446*t12474*t12494*t12469*(2.0/3.0);
                double t12586 = alph*t12510*t12515*t12543*t12477*6.0;
                double t12594 = t12572*t12492*t12566*t12567;
                double t12587 = t12554+t12573-t12574-t12594;
                double t12588 = t12470*t12471*t12474*t12469*t12587;
                double t12595 = Aa*t12504*t12506*t12508*t12494*t12469*t12559*2.0;
                double t12596 = t12506*t12508*t12474*t12494*t12584*t12469*t12578*t12579;
                double t12612 = t12471*t12474*t12555*t12493*t12469*4.0;
                double t12590 = -t12612-t12554+t12563+t12565+t12574+t12585+t12577-t12595-t12596+t12588;
                double t12591 = alph*t12480*t12590*t12477*2.0;
                double t12592 = t12591+t12586;
                double t12593 = t12457*t12457;
                double t12597 = 1.0/pow(t12434,8.0/3.0);
                double t12608 = t12476*t12593*t12478*t12597*2.5E1;
                double t12598 = exp(-t12608);
                double t12599 = Cc5*t12441*t12442;
                double t12600 = Cc6*c*t12441*t12447;
                double t12601 = Cc4*c*t12435;
                double t12602 = t12600+t12601+t12599+1.0;
                double t12603 = 1.0/t12602;
                double t12604 = Cc1*(1.0/1.0E3);
                double t12605 = Cc3*t12441*t12442*(1.0/1.0E3);
                double t12606 = Cc2*c*t12435*(1.0/1.0E3);
                double t12607 = t12604+t12605+t12606;
                double t12609 = Cx*(1.0E1/7.0);
                double t12611 = t12603*t12607;
                double t12610 = Cc0-t12611+t12609;
                double t12613 = log(t12538);
                v_rho_a[Q] += scale * (-t12509-t12465-t12434*(t12612+t12554-t12563-t12565-t12574-t12585-t12577+t12595+t12596-t12588-t12613*t12543*t12526*t12527*t12593*(3.0/2.0)+(t12526*t12527*t12457*t12593*(alph*t12450*t12536*(t12523*t12561*t12517*t12518*t12476*(7.0/4.8E1)+t12522*t12480*t12543*t12517*t12518*t12476*(1.0/8.0)+alph*t12521*t12450*t12515*t12560*t12516*t12478*t12479*(7.0/1.92E2)+alph*t12521*t12450*t12514*t12516*t12543*t12562*t12478*t12479*(1.0/3.2E1)-alph*t12512*t12450*t12514*t12532*t12515*t12516*t12592*t12478*t12479*(1.0/1.28E2))*2.0-alph*t12530*t12450*1.0/(t12535*t12535)*(t12531*t12532*t12515*t12560*t12516*t12477*t12478*t12479*(7.0/9.6E1)+alph*t12521*t12450*t12523*t12561*t12517*t12518*t12476*(7.0/2.4E1)+alph*t12521*t12450*t12522*t12480*t12543*t12517*t12518*t12476*(1.0/4.0)+t12531*t12514*t12532*t12516*t12543*t12562*t12477*t12478*t12479*(1.0/1.6E1)-t12512*1.0/(t12513*t12513*t12513)*t12531*t12514*t12515*t12516*t12592*t12477*t12478*t12479*(1.0/3.2E1)-alph*t12512*t12450*t12522*t12523*t12532*t12517*t12518*t12592*t12476*(1.0/8.0))*2.0)*(1.0/2.0))/t12538+nu*t12522*t12517*t12518*t12457*t12476*t12598*(t12603*(Cc2*c*t12446*(1.0/3.0E3)+Cc3*t12441*t12547*(1.0/1.5E3))-1.0/(t12602*t12602)*t12607*(Cc4*c*t12446*(1.0/3.0)+Cc5*t12441*t12547*(2.0/3.0)+Cc6*c*t12441*t12524))*(1.0/1.6E1)+nu*t12610*t12522*t12543*t12517*t12518*t12476*t12598*(1.0/1.6E1)-nu*t12610*t12561*t12517*t12518*t12457*t12476*t12598*(7.0/4.8E1)+nu*t12610*t12522*t12517*t12518*t12457*t12476*t12598*(1.0/pow(t12434,1.1E1/3.0)*t12476*t12593*t12478*(2.0E2/3.0)-t12543*t12457*t12476*t12478*t12597*5.0E1)*(1.0/1.6E1))+t12613*t12526*t12527*t12457*t12593*(1.0/2.0)+t12470*t12471*t12474*t12469*(t12465-t12519)-nu*t12610*t12522*t12517*t12518*t12457*t12476*t12598*(1.0/1.6E1));
            }
            
            // v_rho_b
            if (deriv >= 1) {
                double t12615 = rho_a+rho_b;
                double t12616 = 1.0/pow(t12615,1.0/3.0);
                double t12617 = c*t12616;
                double t12618 = sqrt(t12617);
                double t12619 = b1p*t12618;
                double t12620 = pow(t12617,3.0/2.0);
                double t12621 = b3p*t12620;
                double t12622 = c*c;
                double t12623 = 1.0/pow(t12615,2.0/3.0);
                double t12624 = b4p*t12622*t12623;
                double t12625 = b2p*c*t12616;
                double t12626 = t12621+t12624+t12625+t12619;
                double t12627 = 1.0/pow(t12615,4.0/3.0);
                double t12628 = 1.0/t12615;
                double t12629 = rho_a-rho_b;
                double t12630 = t12628*t12629;
                double t12631 = 1.0/bet;
                double t12632 = t12630+1.0;
                double t12633 = pow(t12632,2.0/3.0);
                double t12634 = t12633*(1.0/2.0);
                double t12635 = -t12630+1.0;
                double t12636 = pow(t12635,2.0/3.0);
                double t12637 = t12636*(1.0/2.0);
                double t12638 = t12634+t12637;
                double t12639 = 1.0/c0p;
                double t12640 = 1.0/t12626;
                double t12641 = t12640*t12639*(1.0/2.0);
                double t12642 = t12641+1.0;
                double t12643 = a1p*c*t12616;
                double t12644 = t12643+1.0;
                double t12645 = log(t12642);
                double t12646 = c0p*t12644*t12645*2.0;
                double t12647 = t12629*t12629;
                double t12648 = two_13*2.0;
                double t12649 = t12648-2.0;
                double t12650 = 1.0/t12649;
                double t12651 = 1.0/(t12615*t12615*t12615*t12615);
                double t12652 = t12647*t12647;
                double t12653 = pow(t12632,4.0/3.0);
                double t12654 = pow(t12635,4.0/3.0);
                double t12655 = t12653+t12654-2.0;
                double t12656 = gamma_ab*2.0;
                double t12657 = gamma_aa+gamma_bb+t12656;
                double t12658 = 1.0/(bet*bet);
                double t12659 = 1.0/(k*k);
                double t12660 = 1.0/(pi_m12*pi_m12*pi_m12*pi_m12);
                double t12661 = 1.0/(t12638*t12638*t12638);
                double t12662 = 1.0/c0f;
                double t12663 = b1f*t12618;
                double t12664 = b3f*t12620;
                double t12665 = b4f*t12622*t12623;
                double t12666 = b2f*c*t12616;
                double t12667 = t12663+t12664+t12665+t12666;
                double t12668 = 1.0/t12667;
                double t12669 = t12662*t12668*(1.0/2.0);
                double t12670 = t12669+1.0;
                double t12671 = log(t12670);
                double t12672 = a1f*c*t12616;
                double t12673 = t12672+1.0;
                double t12700 = c0f*t12671*t12673*2.0;
                double t12674 = -t12700+t12646;
                double t12675 = 1.0/d2fz0;
                double t12676 = 1.0/Aa;
                double t12677 = b1a*t12618;
                double t12678 = b3a*t12620;
                double t12679 = b4a*t12622*t12623;
                double t12680 = b2a*c*t12616;
                double t12681 = t12680+t12677+t12678+t12679;
                double t12682 = 1.0/t12681;
                double t12683 = t12682*t12676*(1.0/2.0);
                double t12684 = t12683+1.0;
                double t12685 = log(t12684);
                double t12686 = a1a*c*t12616;
                double t12687 = t12686+1.0;
                double t12688 = t12651*t12652;
                double t12689 = t12688-1.0;
                double t12690 = Aa*t12650*t12655*t12675*t12685*t12687*t12689*2.0;
                double t12691 = t12690+t12646-t12650*t12651*t12652*t12655*t12674;
                double t12692 = alph*t12661*t12691*t12658*2.0;
                double t12693 = exp(t12692);
                double t12694 = t12693-1.0;
                double t12695 = 1.0/pow(t12615,1.4E1/3.0);
                double t12696 = 1.0/(t12638*t12638*t12638*t12638);
                double t12697 = t12657*t12657;
                double t12698 = 1.0/k;
                double t12699 = 1.0/(pi_m12*pi_m12);
                double t12701 = 1.0/pow(t12615,7.0/3.0);
                double t12702 = 1.0/(t12638*t12638);
                double t12703 = 1.0/(t12615*t12615);
                double t12704 = t12703*t12629;
                double t12705 = t12704+t12628;
                double t12706 = 1.0/alph;
                double t12707 = bet*bet;
                double t12708 = t12701*t12702*t12657*t12698*t12699*(1.0/1.6E1);
                double t12709 = t12700-t12646;
                double t12710 = t12650*t12651*t12652*t12655*t12709;
                double t12711 = t12710+t12690+t12646;
                double t12712 = alph*t12711*t12661*t12658*2.0;
                double t12713 = exp(t12712);
                double t12714 = t12713-1.0;
                double t12715 = 1.0/t12714;
                double t12716 = alph*alph;
                double t12717 = alph*t12701*t12702*t12631*t12715*t12657*t12698*t12699*(1.0/8.0);
                double t12718 = 1.0/(t12714*t12714);
                double t12719 = t12660*t12716*t12718*t12658*t12659*t12695*t12696*t12697*(1.0/6.4E1);
                double t12720 = t12717+t12719+1.0;
                double t12721 = 1.0/t12720;
                double t12722 = 1.0/pow(t12632,1.0/3.0);
                double t12723 = t12722*t12705*(1.0/3.0);
                double t12724 = 1.0/pow(t12635,1.0/3.0);
                double t12726 = t12705*t12724*(1.0/3.0);
                double t12725 = t12723-t12726;
                double t12727 = 1.0/t12642;
                double t12728 = 1.0/(t12626*t12626);
                double t12729 = 1.0/pow(t12615,5.0/3.0);
                double t12730 = b4p*t12622*t12729*(2.0/3.0);
                double t12731 = b2p*c*t12627*(1.0/3.0);
                double t12732 = 1.0/sqrt(t12617);
                double t12733 = b1p*c*t12732*t12627*(1.0/6.0);
                double t12734 = b3p*c*t12618*t12627*(1.0/2.0);
                double t12735 = t12730+t12731+t12733+t12734;
                double t12736 = 1.0/(t12615*t12615*t12615*t12615*t12615);
                double t12737 = pow(t12632,1.0/3.0);
                double t12738 = t12705*t12737*(4.0/3.0);
                double t12739 = pow(t12635,1.0/3.0);
                double t12749 = t12705*t12739*(4.0/3.0);
                double t12740 = t12738-t12749;
                double t12741 = alph*t12631*t12660*t12715*t12659*t12695*t12696*t12697*(1.0/1.28E2);
                double t12742 = t12741+t12708;
                double t12743 = 1.0/pow(t12615,1.7E1/3.0);
                double t12744 = 1.0/pow(t12615,1.0E1/3.0);
                double t12745 = 1.0/(t12638*t12638*t12638*t12638*t12638);
                double t12746 = t12644*t12735*t12727*t12728;
                double t12747 = t12650*t12651*t12655*t12709*t12629*t12647*4.0;
                double t12748 = t12650*t12652*t12655*t12709*t12736*4.0;
                double t12750 = t12650*t12740*t12651*t12652*t12709;
                double t12751 = 1.0/t12670;
                double t12752 = 1.0/(t12667*t12667);
                double t12753 = b4f*t12622*t12729*(2.0/3.0);
                double t12754 = b2f*c*t12627*(1.0/3.0);
                double t12755 = b1f*c*t12732*t12627*(1.0/6.0);
                double t12756 = b3f*c*t12618*t12627*(1.0/2.0);
                double t12757 = t12753+t12754+t12755+t12756;
                double t12758 = a1f*c*c0f*t12671*t12627*(2.0/3.0);
                double t12759 = a1p*c*c0p*t12627*t12645*(2.0/3.0);
                double t12760 = t12651*t12629*t12647*4.0;
                double t12761 = t12652*t12736*4.0;
                double t12762 = t12760+t12761;
                double t12763 = Aa*t12650*t12762*t12655*t12675*t12685*t12687*2.0;
                double t12764 = Aa*t12650*t12740*t12675*t12685*t12687*t12689*2.0;
                double t12765 = 1.0/t12684;
                double t12766 = 1.0/(t12681*t12681);
                double t12767 = b4a*t12622*t12729*(2.0/3.0);
                double t12768 = b2a*c*t12627*(1.0/3.0);
                double t12769 = b1a*c*t12732*t12627*(1.0/6.0);
                double t12770 = b3a*c*t12618*t12627*(1.0/2.0);
                double t12771 = t12770+t12767+t12768+t12769;
                double t12772 = Aa*a1a*c*t12650*t12627*t12655*t12675*t12685*t12689*(2.0/3.0);
                double t12778 = t12751*t12752*t12673*t12757;
                double t12773 = t12650*t12651*t12652*t12655*(t12746+t12758-t12759-t12778);
                double t12779 = t12650*t12771*t12655*t12675*t12765*t12766*t12687*t12689;
                double t12774 = t12750+t12763+t12772-t12746+t12764+t12773+t12747+t12748+t12759-t12779;
                double t12775 = alph*t12661*t12774*t12658*2.0;
                double t12776 = t12775-alph*t12711*t12725*t12658*t12696*6.0;
                double t12777 = t12638*t12638;
                double t12780 = 1.0/pow(t12615,8.0/3.0);
                double t12791 = t12780*t12657*t12659*t12777*2.5E1;
                double t12781 = exp(-t12791);
                double t12782 = Cc5*t12622*t12623;
                double t12783 = Cc6*c*t12622*t12628;
                double t12784 = Cc4*c*t12616;
                double t12785 = t12782+t12783+t12784+1.0;
                double t12786 = 1.0/t12785;
                double t12787 = Cc1*(1.0/1.0E3);
                double t12788 = Cc3*t12622*t12623*(1.0/1.0E3);
                double t12789 = Cc2*c*t12616*(1.0/1.0E3);
                double t12790 = t12787+t12788+t12789;
                double t12792 = Cx*(1.0E1/7.0);
                double t12794 = t12790*t12786;
                double t12793 = Cc0+t12792-t12794;
                double t12795 = alph*t12631*t12721*t12742*2.0;
                double t12796 = t12795+1.0;
                v_rho_b[Q] += scale * (-t12710-t12690-t12646+t12615*(t12750+t12763+t12772+t12764+t12773+t12747+t12748+t12759-t12779-t12644*t12735*t12727*t12728-t12706*t12707*t12725*t12777*log((alph*t12631*(t12708+(alph*t12631*t12660*t12659*t12695*t12696*t12697*(1.0/1.28E2))/t12694)*2.0)/(t12717+t12660*t12716*t12658*1.0/(t12694*t12694)*t12659*t12695*t12696*t12697*(1.0/6.4E1)+1.0)+1.0)*(3.0/2.0)+(t12706*t12707*t12638*t12777*(alph*t12631*t12721*(t12702*t12744*t12657*t12698*t12699*(-7.0/4.8E1)+t12701*t12661*t12725*t12657*t12698*t12699*(1.0/8.0)-alph*t12631*t12660*t12715*t12743*t12659*t12696*t12697*(7.0/1.92E2)+alph*t12631*t12660*t12715*t12725*t12745*t12659*t12695*t12697*(1.0/3.2E1)+alph*t12631*t12713*t12660*t12718*t12659*t12695*t12696*t12697*(alph*t12661*t12658*(t12750+t12763+t12772+t12764+t12747+t12748+t12759-t12644*t12735*t12727*t12728+t12650*t12651*t12652*t12655*(t12746+t12758-t12751*t12752*t12673*t12757-a1p*c*c0p*t12627*t12645*(2.0/3.0))-t12650*t12771*t12655*t12675*t12765*t12766*t12687*t12689)*2.0-alph*t12711*t12725*t12658*t12696*6.0)*(1.0/1.28E2))*2.0-alph*1.0/(t12720*t12720)*t12631*t12742*(t12660*t12716*t12743*t12718*t12658*t12659*t12696*t12697*(-7.0/9.6E1)-alph*t12702*t12631*t12715*t12744*t12657*t12698*t12699*(7.0/2.4E1)+alph*t12701*t12631*t12661*t12715*t12725*t12657*t12698*t12699*(1.0/4.0)+t12660*t12716*t12725*t12718*t12745*t12658*t12659*t12695*t12697*(1.0/1.6E1)+t12713*t12660*1.0/(t12714*t12714*t12714)*t12716*t12658*t12659*t12695*t12776*t12696*t12697*(1.0/3.2E1)+alph*t12701*t12702*t12631*t12713*t12718*t12657*t12776*t12698*t12699*(1.0/8.0))*2.0)*(1.0/2.0))/t12796-nu*t12701*t12781*t12638*t12657*t12698*t12699*(t12786*(Cc2*c*t12627*(1.0/3.0E3)+Cc3*t12622*t12729*(1.0/1.5E3))-t12790*1.0/(t12785*t12785)*(Cc4*c*t12627*(1.0/3.0)+Cc5*t12622*t12729*(2.0/3.0)+Cc6*c*t12622*t12703))*(1.0/1.6E1)+nu*t12701*t12725*t12781*t12657*t12793*t12698*t12699*(1.0/1.6E1)+nu*t12744*t12781*t12638*t12657*t12793*t12698*t12699*(7.0/4.8E1)-nu*t12701*t12781*t12638*t12657*t12793*t12698*t12699*(1.0/pow(t12615,1.1E1/3.0)*t12657*t12659*t12777*(2.0E2/3.0)+t12725*t12780*t12638*t12657*t12659*5.0E1)*(1.0/1.6E1))+t12706*t12707*t12638*t12777*log(t12796)*(1.0/2.0)-nu*t12701*t12781*t12638*t12657*t12793*t12698*t12699*(1.0/1.6E1));
            }
            
            // v_gamma_aa
            if (deriv >= 1) {
                double t12798 = rho_a+rho_b;
                double t12799 = 1.0/t12798;
                double t12800 = rho_a-rho_b;
                double t12801 = t12800*t12799;
                double t12802 = 1.0/bet;
                double t12803 = t12801+1.0;
                double t12804 = pow(t12803,2.0/3.0);
                double t12805 = t12804*(1.0/2.0);
                double t12806 = -t12801+1.0;
                double t12807 = pow(t12806,2.0/3.0);
                double t12808 = t12807*(1.0/2.0);
                double t12809 = t12805+t12808;
                double t12810 = 1.0/pow(t12798,1.0/3.0);
                double t12811 = c*t12810;
                double t12812 = sqrt(t12811);
                double t12813 = pow(t12811,3.0/2.0);
                double t12814 = c*c;
                double t12815 = 1.0/pow(t12798,2.0/3.0);
                double t12816 = 1.0/c0p;
                double t12817 = b1p*t12812;
                double t12818 = b3p*t12813;
                double t12819 = b4p*t12814*t12815;
                double t12820 = b2p*c*t12810;
                double t12821 = t12820+t12817+t12818+t12819;
                double t12822 = 1.0/t12821;
                double t12823 = t12822*t12816*(1.0/2.0);
                double t12824 = t12823+1.0;
                double t12825 = log(t12824);
                double t12826 = a1p*c*t12810;
                double t12827 = t12826+1.0;
                double t12828 = c0p*t12825*t12827*2.0;
                double t12829 = t12800*t12800;
                double t12830 = two_13*2.0;
                double t12831 = t12830-2.0;
                double t12832 = 1.0/t12831;
                double t12833 = 1.0/(t12798*t12798*t12798*t12798);
                double t12834 = t12829*t12829;
                double t12835 = pow(t12803,4.0/3.0);
                double t12836 = pow(t12806,4.0/3.0);
                double t12837 = t12835+t12836-2.0;
                double t12838 = gamma_ab*2.0;
                double t12839 = gamma_aa+gamma_bb+t12838;
                double t12840 = 1.0/(bet*bet);
                double t12841 = 1.0/(k*k);
                double t12842 = 1.0/(pi_m12*pi_m12*pi_m12*pi_m12);
                double t12843 = 1.0/(t12809*t12809*t12809);
                double t12844 = 1.0/c0f;
                double t12845 = b1f*t12812;
                double t12846 = b3f*t12813;
                double t12847 = b4f*t12814*t12815;
                double t12848 = b2f*c*t12810;
                double t12849 = t12845+t12846+t12847+t12848;
                double t12850 = 1.0/t12849;
                double t12851 = t12850*t12844*(1.0/2.0);
                double t12852 = t12851+1.0;
                double t12853 = log(t12852);
                double t12854 = a1f*c*t12810;
                double t12855 = t12854+1.0;
                double t12882 = c0f*t12853*t12855*2.0;
                double t12856 = t12828-t12882;
                double t12857 = 1.0/d2fz0;
                double t12858 = 1.0/Aa;
                double t12859 = b1a*t12812;
                double t12860 = b3a*t12813;
                double t12861 = b4a*t12814*t12815;
                double t12862 = b2a*c*t12810;
                double t12863 = t12860+t12861+t12862+t12859;
                double t12864 = 1.0/t12863;
                double t12865 = t12864*t12858*(1.0/2.0);
                double t12866 = t12865+1.0;
                double t12867 = log(t12866);
                double t12868 = a1a*c*t12810;
                double t12869 = t12868+1.0;
                double t12870 = t12833*t12834;
                double t12871 = t12870-1.0;
                double t12872 = Aa*t12832*t12871*t12837*t12857*t12867*t12869*2.0;
                double t12883 = t12832*t12833*t12834*t12837*t12856;
                double t12873 = t12872+t12828-t12883;
                double t12874 = alph*t12840*t12843*t12873*2.0;
                double t12875 = exp(t12874);
                double t12876 = t12875-1.0;
                double t12877 = 1.0/pow(t12798,1.4E1/3.0);
                double t12878 = 1.0/(t12809*t12809*t12809*t12809);
                double t12879 = t12839*t12839;
                double t12880 = 1.0/k;
                double t12881 = 1.0/(pi_m12*pi_m12);
                double t12884 = 1.0/t12876;
                double t12885 = 1.0/pow(t12798,7.0/3.0);
                double t12886 = 1.0/(t12809*t12809);
                double t12887 = alph*alph;
                double t12888 = 1.0/(t12876*t12876);
                double t12889 = t12840*t12841*t12842*t12877*t12878*t12887*t12879*t12888*(1.0/6.4E1);
                double t12890 = alph*t12802*t12880*t12881*t12839*t12884*t12885*t12886*(1.0/8.0);
                double t12891 = t12890+t12889+1.0;
                double t12892 = 1.0/t12891;
                double t12893 = t12880*t12881*t12839*t12885*t12886*(1.0/1.6E1);
                double t12894 = alph*t12802*t12841*t12842*t12884*t12877*t12878*t12879*(1.0/1.28E2);
                double t12895 = t12893+t12894;
                double t12896 = gamma_aa*2.0;
                double t12897 = gamma_ab*4.0;
                double t12898 = gamma_bb*2.0;
                double t12899 = t12896+t12897+t12898;
                double t12900 = t12809*t12809;
                double t12901 = 1.0/pow(t12798,8.0/3.0);
                double t12902 = exp(t12900*t12901*t12841*t12839*-2.5E1);
                double t12903 = Cx*(1.0E1/7.0);
                double t12904 = Cc1*(1.0/1.0E3);
                double t12905 = Cc3*t12814*t12815*(1.0/1.0E3);
                double t12906 = Cc2*c*t12810*(1.0/1.0E3);
                double t12907 = t12904+t12905+t12906;
                double t12908 = Cc5*t12814*t12815;
                double t12909 = Cc6*c*t12814*t12799;
                double t12910 = Cc4*c*t12810;
                double t12911 = t12910+t12908+t12909+1.0;
                double t12912 = 1.0/t12911;
                double t12913 = Cc0+t12903-t12912*t12907;
                v_gamma_aa[Q] += scale * (t12798*(((bet*bet)*t12900*t12809*(alph*t12802*t12892*(t12880*t12881*t12885*t12886*(1.0/1.6E1)+alph*t12802*t12841*t12842*t12884*t12877*t12878*t12899*(1.0/1.28E2))*2.0-alph*t12802*1.0/(t12891*t12891)*t12895*(alph*t12802*t12880*t12881*t12884*t12885*t12886*(1.0/8.0)+t12840*t12841*t12842*t12877*t12878*t12887*t12888*t12899*(1.0/6.4E1))*2.0)*(1.0/2.0))/(alph*(alph*t12802*t12892*t12895*2.0+1.0))-nu*t12902*t12913*t12880*t12809*t12881*t12885*(1.0/1.6E1)+1.0/(k*k*k)*nu*t12900*t12902*t12913*t12809*t12881*t12839*1.0/(t12798*t12798*t12798*t12798*t12798)*(2.5E1/1.6E1)));
            }
            
            // v_gamma_ab
            if (deriv >= 1) {
                double t12915 = rho_a+rho_b;
                double t12916 = 1.0/t12915;
                double t12917 = rho_a-rho_b;
                double t12918 = t12916*t12917;
                double t12919 = 1.0/bet;
                double t12920 = t12918+1.0;
                double t12921 = pow(t12920,2.0/3.0);
                double t12922 = t12921*(1.0/2.0);
                double t12923 = -t12918+1.0;
                double t12924 = pow(t12923,2.0/3.0);
                double t12925 = t12924*(1.0/2.0);
                double t12926 = t12922+t12925;
                double t12927 = 1.0/pow(t12915,1.0/3.0);
                double t12928 = c*t12927;
                double t12929 = sqrt(t12928);
                double t12930 = pow(t12928,3.0/2.0);
                double t12931 = c*c;
                double t12932 = 1.0/pow(t12915,2.0/3.0);
                double t12933 = 1.0/c0p;
                double t12934 = b1p*t12929;
                double t12935 = b3p*t12930;
                double t12936 = b4p*t12931*t12932;
                double t12937 = b2p*c*t12927;
                double t12938 = t12934+t12935+t12936+t12937;
                double t12939 = 1.0/t12938;
                double t12940 = t12933*t12939*(1.0/2.0);
                double t12941 = t12940+1.0;
                double t12942 = log(t12941);
                double t12943 = a1p*c*t12927;
                double t12944 = t12943+1.0;
                double t12945 = c0p*t12942*t12944*2.0;
                double t12946 = t12917*t12917;
                double t12947 = two_13*2.0;
                double t12948 = t12947-2.0;
                double t12949 = 1.0/t12948;
                double t12950 = 1.0/(t12915*t12915*t12915*t12915);
                double t12951 = t12946*t12946;
                double t12952 = pow(t12920,4.0/3.0);
                double t12953 = pow(t12923,4.0/3.0);
                double t12954 = t12952+t12953-2.0;
                double t12955 = gamma_ab*2.0;
                double t12956 = gamma_aa+gamma_bb+t12955;
                double t12957 = 1.0/(bet*bet);
                double t12958 = 1.0/(k*k);
                double t12959 = 1.0/(pi_m12*pi_m12*pi_m12*pi_m12);
                double t12960 = 1.0/(t12926*t12926*t12926);
                double t12961 = 1.0/c0f;
                double t12962 = b1f*t12929;
                double t12963 = b3f*t12930;
                double t12964 = b4f*t12931*t12932;
                double t12965 = b2f*c*t12927;
                double t12966 = t12962+t12963+t12964+t12965;
                double t12967 = 1.0/t12966;
                double t12968 = t12961*t12967*(1.0/2.0);
                double t12969 = t12968+1.0;
                double t12970 = log(t12969);
                double t12971 = a1f*c*t12927;
                double t12972 = t12971+1.0;
                double t12999 = c0f*t12970*t12972*2.0;
                double t12973 = t12945-t12999;
                double t12974 = 1.0/d2fz0;
                double t12975 = 1.0/Aa;
                double t12976 = b1a*t12929;
                double t12977 = b3a*t12930;
                double t12978 = b4a*t12931*t12932;
                double t12979 = b2a*c*t12927;
                double t12980 = t12976+t12977+t12978+t12979;
                double t12981 = 1.0/t12980;
                double t12982 = t12981*t12975*(1.0/2.0);
                double t12983 = t12982+1.0;
                double t12984 = log(t12983);
                double t12985 = a1a*c*t12927;
                double t12986 = t12985+1.0;
                double t12987 = t12950*t12951;
                double t12988 = t12987-1.0;
                double t12989 = Aa*t12954*t12974*t12984*t12949*t12986*t12988*2.0;
                double t13000 = t12950*t12951*t12954*t12973*t12949;
                double t12990 = t12945+t12989-t13000;
                double t12991 = alph*t12960*t12990*t12957*2.0;
                double t12992 = exp(t12991);
                double t12993 = t12992-1.0;
                double t12994 = 1.0/pow(t12915,1.4E1/3.0);
                double t12995 = 1.0/(t12926*t12926*t12926*t12926);
                double t12996 = t12956*t12956;
                double t12997 = 1.0/k;
                double t12998 = 1.0/(pi_m12*pi_m12);
                double t13001 = 1.0/t12993;
                double t13002 = 1.0/pow(t12915,7.0/3.0);
                double t13003 = 1.0/(t12926*t12926);
                double t13004 = alph*alph;
                double t13005 = 1.0/(t12993*t12993);
                double t13006 = t12957*t12958*t12994*t12959*t12995*t12996*t13004*t13005*(1.0/6.4E1);
                double t13007 = alph*t12919*t12956*t12997*t12998*t13001*t13002*t13003*(1.0/8.0);
                double t13008 = t13006+t13007+1.0;
                double t13009 = 1.0/t13008;
                double t13010 = t12956*t12997*t12998*t13002*t13003*(1.0/1.6E1);
                double t13011 = alph*t12919*t12958*t12994*t12959*t12995*t12996*t13001*(1.0/1.28E2);
                double t13012 = t13010+t13011;
                double t13013 = gamma_aa*4.0;
                double t13014 = gamma_ab*8.0;
                double t13015 = gamma_bb*4.0;
                double t13016 = t13013+t13014+t13015;
                double t13017 = t12926*t12926;
                double t13018 = 1.0/pow(t12915,8.0/3.0);
                double t13019 = exp(t12956*t12958*t13017*t13018*-2.5E1);
                double t13020 = Cx*(1.0E1/7.0);
                double t13021 = Cc1*(1.0/1.0E3);
                double t13022 = Cc3*t12931*t12932*(1.0/1.0E3);
                double t13023 = Cc2*c*t12927*(1.0/1.0E3);
                double t13024 = t13021+t13022+t13023;
                double t13025 = Cc5*t12931*t12932;
                double t13026 = Cc6*c*t12931*t12916;
                double t13027 = Cc4*c*t12927;
                double t13028 = t13025+t13026+t13027+1.0;
                double t13029 = 1.0/t13028;
                double t13030 = Cc0+t13020-t13024*t13029;
                v_gamma_ab[Q] += scale * (t12915*(((bet*bet)*t12926*t13017*(alph*t12919*t13009*(t12997*t12998*t13002*t13003*(1.0/8.0)+alph*t12919*t12958*t12994*t12959*t12995*t13001*t13016*(1.0/1.28E2))*2.0-alph*t12919*t13012*1.0/(t13008*t13008)*(alph*t12919*t12997*t12998*t13001*t13002*t13003*(1.0/4.0)+t12957*t12958*t12994*t12959*t12995*t13004*t13005*t13016*(1.0/6.4E1))*2.0)*(1.0/2.0))/(alph*(alph*t12919*t13012*t13009*2.0+1.0))-nu*t12926*t12997*t12998*t13002*t13030*t13019*(1.0/8.0)+1.0/(k*k*k)*nu*1.0/(t12915*t12915*t12915*t12915*t12915)*t12926*t12956*t12998*t13030*t13017*t13019*(2.5E1/8.0)));
            }
            
            // v_gamma_bb
            if (deriv >= 1) {
                double t13032 = rho_a+rho_b;
                double t13033 = 1.0/t13032;
                double t13034 = rho_a-rho_b;
                double t13035 = t13033*t13034;
                double t13036 = 1.0/bet;
                double t13037 = t13035+1.0;
                double t13038 = pow(t13037,2.0/3.0);
                double t13039 = t13038*(1.0/2.0);
                double t13040 = -t13035+1.0;
                double t13041 = pow(t13040,2.0/3.0);
                double t13042 = t13041*(1.0/2.0);
                double t13043 = t13042+t13039;
                double t13044 = 1.0/pow(t13032,1.0/3.0);
                double t13045 = c*t13044;
                double t13046 = sqrt(t13045);
                double t13047 = pow(t13045,3.0/2.0);
                double t13048 = c*c;
                double t13049 = 1.0/pow(t13032,2.0/3.0);
                double t13050 = 1.0/c0p;
                double t13051 = b1p*t13046;
                double t13052 = b3p*t13047;
                double t13053 = b4p*t13048*t13049;
                double t13054 = b2p*c*t13044;
                double t13055 = t13051+t13052+t13053+t13054;
                double t13056 = 1.0/t13055;
                double t13057 = t13050*t13056*(1.0/2.0);
                double t13058 = t13057+1.0;
                double t13059 = log(t13058);
                double t13060 = a1p*c*t13044;
                double t13061 = t13060+1.0;
                double t13062 = c0p*t13061*t13059*2.0;
                double t13063 = t13034*t13034;
                double t13064 = two_13*2.0;
                double t13065 = t13064-2.0;
                double t13066 = 1.0/t13065;
                double t13067 = 1.0/(t13032*t13032*t13032*t13032);
                double t13068 = t13063*t13063;
                double t13069 = pow(t13037,4.0/3.0);
                double t13070 = pow(t13040,4.0/3.0);
                double t13071 = t13070+t13069-2.0;
                double t13072 = gamma_ab*2.0;
                double t13073 = gamma_aa+gamma_bb+t13072;
                double t13074 = 1.0/(bet*bet);
                double t13075 = 1.0/(k*k);
                double t13076 = 1.0/(pi_m12*pi_m12*pi_m12*pi_m12);
                double t13077 = 1.0/(t13043*t13043*t13043);
                double t13078 = 1.0/c0f;
                double t13079 = b1f*t13046;
                double t13080 = b3f*t13047;
                double t13081 = b4f*t13048*t13049;
                double t13082 = b2f*c*t13044;
                double t13083 = t13080+t13081+t13082+t13079;
                double t13084 = 1.0/t13083;
                double t13085 = t13084*t13078*(1.0/2.0);
                double t13086 = t13085+1.0;
                double t13087 = log(t13086);
                double t13088 = a1f*c*t13044;
                double t13089 = t13088+1.0;
                double t13116 = c0f*t13087*t13089*2.0;
                double t13090 = t13062-t13116;
                double t13091 = 1.0/d2fz0;
                double t13092 = 1.0/Aa;
                double t13093 = b1a*t13046;
                double t13094 = b3a*t13047;
                double t13095 = b4a*t13048*t13049;
                double t13096 = b2a*c*t13044;
                double t13097 = t13093+t13094+t13095+t13096;
                double t13098 = 1.0/t13097;
                double t13099 = t13092*t13098*(1.0/2.0);
                double t13100 = t13099+1.0;
                double t13101 = log(t13100);
                double t13102 = a1a*c*t13044;
                double t13103 = t13102+1.0;
                double t13104 = t13067*t13068;
                double t13105 = t13104-1.0;
                double t13106 = Aa*t13101*t13103*t13105*t13071*t13091*t13066*2.0;
                double t13117 = t13071*t13090*t13066*t13067*t13068;
                double t13107 = t13106+t13062-t13117;
                double t13108 = alph*t13107*t13074*t13077*2.0;
                double t13109 = exp(t13108);
                double t13110 = t13109-1.0;
                double t13111 = 1.0/pow(t13032,1.4E1/3.0);
                double t13112 = 1.0/(t13043*t13043*t13043*t13043);
                double t13113 = t13073*t13073;
                double t13114 = 1.0/k;
                double t13115 = 1.0/(pi_m12*pi_m12);
                double t13118 = 1.0/t13110;
                double t13119 = 1.0/pow(t13032,7.0/3.0);
                double t13120 = 1.0/(t13043*t13043);
                double t13121 = alph*alph;
                double t13122 = 1.0/(t13110*t13110);
                double t13123 = t13111*t13112*t13121*t13113*t13122*t13074*t13075*t13076*(1.0/6.4E1);
                double t13124 = alph*t13120*t13114*t13115*t13036*t13073*t13118*t13119*(1.0/8.0);
                double t13125 = t13123+t13124+1.0;
                double t13126 = 1.0/t13125;
                double t13127 = t13120*t13114*t13115*t13073*t13119*(1.0/1.6E1);
                double t13128 = alph*t13111*t13112*t13113*t13036*t13118*t13075*t13076*(1.0/1.28E2);
                double t13129 = t13127+t13128;
                double t13130 = gamma_aa*2.0;
                double t13131 = gamma_ab*4.0;
                double t13132 = gamma_bb*2.0;
                double t13133 = t13130+t13131+t13132;
                double t13134 = t13043*t13043;
                double t13135 = 1.0/pow(t13032,8.0/3.0);
                double t13136 = exp(t13134*t13135*t13073*t13075*-2.5E1);
                double t13137 = Cx*(1.0E1/7.0);
                double t13138 = Cc1*(1.0/1.0E3);
                double t13139 = Cc3*t13048*t13049*(1.0/1.0E3);
                double t13140 = Cc2*c*t13044*(1.0/1.0E3);
                double t13141 = t13140+t13138+t13139;
                double t13142 = Cc5*t13048*t13049;
                double t13143 = Cc6*c*t13033*t13048;
                double t13144 = Cc4*c*t13044;
                double t13145 = t13142+t13143+t13144+1.0;
                double t13146 = 1.0/t13145;
                double t13147 = Cc0+t13137-t13141*t13146;
                v_gamma_bb[Q] += scale * (t13032*(((bet*bet)*t13043*t13134*(alph*t13036*t13126*(t13120*t13114*t13115*t13119*(1.0/1.6E1)+alph*t13111*t13112*t13133*t13036*t13118*t13075*t13076*(1.0/1.28E2))*2.0-alph*1.0/(t13125*t13125)*t13036*t13129*(alph*t13120*t13114*t13115*t13036*t13118*t13119*(1.0/8.0)+t13111*t13112*t13121*t13122*t13133*t13074*t13075*t13076*(1.0/6.4E1))*2.0)*(1.0/2.0))/(alph*(alph*t13036*t13126*t13129*2.0+1.0))-nu*t13114*t13043*t13115*t13136*t13119*t13147*(1.0/1.6E1)+1.0/(k*k*k)*nu*1.0/(t13032*t13032*t13032*t13032*t13032)*t13043*t13115*t13134*t13073*t13136*t13147*(2.5E1/1.6E1)));
            }
            
        }
    }
}

}
