/***********************************************************************************
* Smooth Tasks
* Copyright (C) 2009 Marcin Baszczewski <marcin.baszczewski@gmail.com>
* Copyright (C) 2009 Mathias Panzenböck <grosser.meister.morti@gmx.net>
* Copyright (C) 2009-2012 Toni Dietze <smooth-tasks@derflupp.e4ward.com>
*
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU General Public License
* as published by the Free Software Foundation; either version 2
* of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*
***********************************************************************************/

// Smooth Tasks
#include "TaskbuttonFrameSvg.h"
#include "SmoothTasks/TaskItem.h"
#include "SmoothTasks/Applet.h"
#include "SmoothTasks/TaskIcon.h"
#include "SmoothTasks/Light.h"
#include "SmoothTasks/Global.h"
#include "SmoothTasks/SmoothToolTip.h"
#include "SmoothTasks/TaskbarLayout.h"

// Qt
#include <QtGlobal>
#include <QApplication>
#include <QPainter>
#include <QFontMetrics>
#include <QSizeF>
#include <QGraphicsLinearLayout>
#include <QGraphicsSceneHoverEvent>
#include <QGraphicsSceneContextMenuEvent>
#include <QStyleOptionGraphicsItem>
#include <QGraphicsSceneMouseEvent>
#include <QTextLayout>
#include <QGraphicsScene>
#include <QGraphicsView>
#include <QMimeData>
#include <QTextDocument>
#include <QTimer>
#include <QString>

// KDE
#include <KIcon>
#include <KIconEffect>
#include <KIconLoader>
#include <KGlobalSettings>
#include <KColorUtils>
#include <KWindowSystem>
#include <KRun>
#include <KServiceTypeTrader>
#include <KStandardDirs>
#include <KUrl>

// Plasma
#include <Plasma/FrameSvg>
#include <Plasma/Theme>
#include <Plasma/PaintUtils>

#include <cmath>
#include <cstring>

namespace SmoothTasks {

const qreal   TaskItem::MINIMIZED_TEXT_ALPHA  = 0.85;
const QString TaskItem::GROUP_EXPANDER_TOP    = QString::fromLatin1("group-expander-top");
const QString TaskItem::GROUP_EXPANDER_RIGHT  = QString::fromLatin1("group-expander-right");
const QString TaskItem::GROUP_EXPANDER_LEFT   = QString::fromLatin1("group-expander-left");
const QString TaskItem::GROUP_EXPANDER_BOTTOM = QString::fromLatin1("group-expander-bottom");

TaskItem::TaskItem(AbstractGroupableItem *abstractItem, Applet *applet)
		: QGraphicsWidget(applet),
		  m_applet(applet),
		  m_icon(new TaskIcon(this)),
		  m_task(new Task(abstractItem, this)),
		  m_light(new Light(this)),
		  m_abstractItem(abstractItem),
		  m_activateTimer(NULL),
		  m_updateTimer(new QTimer()),
		  m_mouseIn(false),
		  m_delayedMouseIn(false),
		  m_stateAnimation(),
		  m_orientation(Qt::Horizontal),
		  m_updateScheduled(false),
		  m_staticallyExpanded(false),
		  m_dynamicallyExpanded(false) {
	
	qDebug("TaskItem::TaskItem");
	
	connect(applet, SIGNAL(settingsChanged()), this, SLOT(settingsChanged()));

	m_icon->setIcon(m_task->icon());

	m_updateTimer->setInterval(1000 / m_applet->fps());
	connect(m_updateTimer, SIGNAL(timeout()), this, SLOT(updateTimerTick()));

	setAcceptsHoverEvents(true);
	setAcceptDrops(true);

	// task signals
	connect(m_task, SIGNAL(update()),        this, SLOT(update()));
	connect(m_task, SIGNAL(updateState()),   this, SLOT(updateState()));
	connect(m_task, SIGNAL(textChanged()),   this, SLOT(updateText()));
	connect(m_task, SIGNAL(updateToolTip()), this, SLOT(updateToolTip()));
	connect(m_task, SIGNAL(gotTask()),       this, SLOT(publishIconGeometry()));

	// icon
	connect(m_icon, SIGNAL(update()),                 this,   SLOT(update()));
	connect(m_task, SIGNAL(updateIcon(const QIcon&)), m_icon, SLOT(setIcon(const QIcon&)));
	
	// desktop changes
	connect(KWindowSystem::self(), SIGNAL(currentDesktopChanged(int)), this, SLOT(updateExpansion()));
	
	// light
	connect(m_light, SIGNAL(update()), this, SLOT(update()));
	
	m_preferredTextLayoutSize = ::SmoothTasks::preferredTextLayoutSize(m_task->text(), KGlobalSettings::taskbarFont());
	
	updateState();
	
	if (m_task->type() == Task::StartupItem) {
		m_icon->startStartupAnimation(500);
		m_light->startAnimation(Light::StartupAnimation, 500, -1);
	}
		
	if (abstractItem->itemType() == TaskManager::GroupItemType) {
		TaskManager::TaskGroup* group = static_cast<TaskManager::TaskGroup*>(abstractItem);
		connect(
			group, SIGNAL(itemAdded(AbstractGroupableItem*)),
			this, SLOT(updateToolTip()));
		connect(
			group, SIGNAL(itemRemoved(AbstractGroupableItem*)),
			this, SLOT(updateToolTip()));
	}

	connect(
		&m_stateAnimation, SIGNAL(update()),
		this, SLOT(update()));

}

TaskItem::~TaskItem() {
	m_applet->toolTip()->itemDelete(this);
	m_updateTimer->deleteLater();
	if (m_activateTimer) {
		delete m_activateTimer;
		m_activateTimer = NULL;
	}
}

void TaskItem::setOrientation(Qt::Orientation orientation) {
	m_orientation = orientation;
}

QSizeF TaskItem::preferredRdSizeStatic(const TaskManager::AbstractGroupableItem& abstractGroupableItem, const Applet& applet, TaskbuttonFrameSvg& taskbuttonFrameSvg, const Qt::Orientation& orientation, const qreal maxRdHeight, const TaskIcon* taskIcon, const QSizeF* preferredTextLayoutSize) {
	qreal l, t, r, b;
	taskbuttonFrameSvg.getMaximalMargins(l, t, r, b);
	qreal frameRdWidth, frameRdHeight;
	if (orientation == Qt::Vertical && !applet.dontRotateFrame()) {
		frameRdWidth  = t + b;
		frameRdHeight = l + r;
	} else {
		frameRdWidth  = l + r;
		frameRdHeight = t + b;
	}
	
	const QSizeF iconSize =
		taskIcon
		? taskIcon->preferredRdSize(maxRdHeight - frameRdHeight, maxRdHeight)
		: TaskIcon::preferredRdSize(abstractGroupableItem.icon(), applet, orientation, maxRdHeight - frameRdHeight, maxRdHeight);
	qreal rdWidth  = iconSize.width();
	qreal rdHeight = iconSize.height();
	
	const bool expandedStatic = isExpandedStatic(abstractGroupableItem, applet);
	if (expandedStatic) {
		QSizeF prefTextSize =
			preferredTextLayoutSize
			? *preferredTextLayoutSize
			: ::SmoothTasks::preferredTextLayoutSize(abstractGroupableItem.name() /* TODO: is this sufficient her? see Task::text() */, KGlobalSettings::taskbarFont(), applet.maxButtonWidth());
		rdWidth += 5 + prefTextSize.width();  // TODO: make text seperation variable
		rdHeight = qMax(rdHeight, prefTextSize.height());
	}
	
	if (abstractGroupableItem.itemType() == TaskManager::GroupItemType) {
		QRectF expanderRec = expanderRect(abstractGroupableItem, applet, orientation, QRectF(0, 0, 0, 0));
		rdWidth += expanderRec.width() + 5;  // TODO: make text seperation variable
		rdHeight = qMax(rdHeight, expanderRec.height());
	}
	
	rdWidth  += frameRdWidth;
	rdHeight += frameRdHeight;
	
	if (rdWidth > applet.maxButtonWidth())
		rdWidth = applet.maxButtonWidth();
	if (rdWidth < applet.minButtonWidth() && expandedStatic)
		rdWidth = applet.minButtonWidth();
	
	return QSizeF(rdWidth, rdHeight);
}

QSizeF TaskItem::preferredRdSizeStatic(const qreal maxRdHeight) const {
	return preferredRdSizeStatic(*m_abstractItem, *m_applet, *m_applet->frame(), m_orientation, maxRdHeight, m_icon, &m_preferredTextLayoutSize);
}

qreal TaskItem::preferredRdWidthDynamic(const qreal& rdHeight) const {
	if (isExpandedByHover())
		return m_applet->expandedWidth();
	else
		return preferredRdSizeStatic(rdHeight).width();
}

bool TaskItem::isExpandedByHover() const {
	return m_dynamicallyExpanded && !m_staticallyExpanded;
}


bool TaskItem::isExpandedStatic() const {
	return m_staticallyExpanded;
}

bool TaskItem::isExpanded() const {
	return m_staticallyExpanded || m_dynamicallyExpanded;
}

void TaskItem::settingsChanged() {
	qDebug("TaskItem::settingsChanged");
	m_updateTimer->setInterval(1000 / m_applet->fps());
	updateText();
	updateExpansion();
}

bool TaskItem::isExpandedStatic(const TaskManager::AbstractGroupableItem& abstractGroupableItem, const Applet& applet) {
	if (applet.expandTasks() && abstractGroupableItem.itemType() != TaskManager::LauncherItemType) {
		switch (applet.keepExpanded()) {
			case Applet::ExpandNone          : return false;
			case Applet::ExpandAll           : return true;
			case Applet::ExpandCurrentDesktop: return abstractGroupableItem.isOnCurrentDesktop() || abstractGroupableItem.isOnAllDesktops();
			case Applet::ExpandActive        : return abstractGroupableItem.isActive();
		}
	}
	return false;
}

void TaskItem::updateExpansion() {
	bool staticallyExpanded = false;
	bool dynamicallyExpanded = false;

	if (m_applet->expandTasks() && m_task->type() != Task::LauncherItem) {
		switch (m_applet->keepExpanded()) {
		case Applet::ExpandNone:
			staticallyExpanded = false;
			break;
		case Applet::ExpandAll:
			staticallyExpanded = true;
			break;
		case Applet::ExpandCurrentDesktop:
			staticallyExpanded = m_task->isOnCurrentDesktop() || m_task->isOnAllDesktops();
			break;
		case Applet::ExpandActive:
			staticallyExpanded = m_task->isActive();
			break;
		}
		
		dynamicallyExpanded = staticallyExpanded;
		if (!staticallyExpanded) {
			if (m_applet->expandOnHover() && m_delayedMouseIn) {
				dynamicallyExpanded = true;
			}
			if (m_applet->expandOnAttention() && m_task->demandsAttention()) {
				dynamicallyExpanded = true;
			}
		}
	}
	
	qDebug() << "TaskItem::updateExpansion" << " static:" << m_staticallyExpanded << "->" << staticallyExpanded << " dynamic:" << m_dynamicallyExpanded << "->" << dynamicallyExpanded;
	
	if (m_staticallyExpanded != staticallyExpanded) {
		m_staticallyExpanded = staticallyExpanded;
		m_dynamicallyExpanded = dynamicallyExpanded;
		emit preferredSizeStaticChange(this);
	} else if (m_dynamicallyExpanded != dynamicallyExpanded) {
		m_dynamicallyExpanded = dynamicallyExpanded;
		emit preferredSizeDynamicChange(this);
	}
}

void TaskItem::updateTimerTick() {
	if (m_updateScheduled) {
		QGraphicsWidget::update();
		m_updateScheduled = false;
	} else {
		m_updateTimer->stop();
	}
}

void TaskItem::update() {
	// limit framerate to configured value
	if (m_updateTimer->isActive()) {
		m_updateScheduled = true;
	}
	else {
		m_updateTimer->start();
		QGraphicsWidget::update();
	}
}

void TaskItem::updateToolTip() {
	m_applet->toolTip()->itemUpdate(this);
}

void TaskItem::setAnimationTargetGeometry(const QRectF& rect) {
	m_animationDestinationGeometry = rect;
}

QPointF TaskItem::pointAnimationStep(const QPointF& current, const QPointF& target, const qreal v, const qreal dt) {
	qreal dx = -target.x() + current.x();
	qreal dy = -target.y() + current.y();
	qreal l = std::sqrt(dx * dx + dy * dy);
	
	if (l <= 1.0)
		return target;
	
	
	// Wurf-Bewegung
	const qreal t0 = 0.0;
	qreal t = -std::sqrt((l + t0 * t0) / v) + t0  +  dt;
	if (t >= 0.0)
		return target;
	
	qreal lNew = v * (t - t0) * (t - t0) - t0 * t0;
	
	lNew = lNew / l;  // eine Division sparen
	dx = dx * lNew;
	dy = dy * lNew;
	
	return QPointF(target.x() + dx, target.y() + dy);
	
	
	// exponentielle Annäherung
	/*const qreal c = std::pow(0.5, 30);
	const qreal f = std::pow(c, dt);
	dx = f * dx;
	dy = f * dy;
	return QPointF(target.x() + dx, target.y() + dy);*/
}

bool TaskItem::animateStep(const qreal v, qreal dt) {
	const QRectF geometryCurrent = geometry();
	const QRectF& geometryTarget = m_animationDestinationGeometry;
	if (geometryCurrent != geometryTarget) {
		QRectF geometryNew(
				pointAnimationStep(geometryCurrent.topLeft(), geometryTarget.topLeft(), v, dt),
				pointAnimationStep(geometryCurrent.bottomRight(), geometryTarget.bottomRight(), v, dt)
			);
		
		setGeometry(geometryNew);
		
		if (geometryNew != geometryTarget)
			return false;
	}
	return true;
}

void TaskItem::skipAnimation() {
	setGeometry(m_animationDestinationGeometry);
}

// XXX: for some reason sometimes this cannot find a parent view when
//      TaskItem width > heigth and shown in the plasmoidviewer
QRect TaskItem::iconGeometry() const {
	if (!scene() || !boundingRect().isValid()) {
		return QRect();
	}

	QRectF  sceneBoundingRect(this->sceneBoundingRect());
	QPointF scenePos(this->scenePos());
	QGraphicsView *parentView = NULL;
	QGraphicsView *possibleParentView = NULL;
	// The following was taken from Plasma::Applet,
	// it doesn't make sense to make the item an applet,
	// and this was the easiest way around it.
	
	foreach (QGraphicsView *view, scene()->views()) {
		if (view->sceneRect().intersects(sceneBoundingRect) ||
			view->sceneRect().contains(scenePos)) {
			if (view->isActiveWindow()) {
				parentView = view;
				break;
			} else {
				possibleParentView = view;
			}
		}
	}

	if (!parentView) {
		parentView = possibleParentView;

		if (!parentView) {
			return QRect();
		}
	}

	QRect rect(parentView->mapFromScene(
		mapToScene(boundingRect())).boundingRect().adjusted(0, 0, 1, 1));
	rect.moveTopLeft(parentView->mapToGlobal(rect.topLeft()));
	return rect;
}

void TaskItem::publishIconGeometry() {
	QRect iconRect(iconGeometry());
	TaskManager::Task* task;
	TaskManager::GroupPtr group;
	
	switch (m_task->type()) {
		case Task::TaskItem:
			task = m_task->task();
			
			if (task) {
				task->publishIconGeometry(iconRect);
			}
			break;
		case Task::GroupItem:
			group = m_task->group();
			
			if (group) {
				foreach (TaskManager::AbstractGroupableItem *item, group->members()) {
					TaskManager::TaskItem *task = qobject_cast<TaskManager::TaskItem*>(item);
					if (task) {
						task->task()->publishIconGeometry(iconRect);
					}
				}
			}
		default:
			break;
	}
}

// XXX: this gets called to often! (still?)
void TaskItem::updateState() {
	qDebug("TaskItem::updateState");
	int newState = m_mouseIn ? TaskStateAnimation::Hover : TaskStateAnimation::Normal;
	//m_applet->sorting(this);
	publishIconGeometry();
	m_icon->stopStartupAnimation();
	m_light->stopAnimation();

	if (m_task->demandsAttention()) {
		newState |= TaskStateAnimation::Attention;
		m_light->startAnimation(Light::AttentionAnimation, 900, 3);
	} 
	else if (m_task->type() == Task::LauncherItem) {
		newState |= TaskStateAnimation::Launcher;
	}
	else if (m_task->isMinimized()) {
		newState |= TaskStateAnimation::Minimized;
	}
	else if (m_task->isActive()) {
		emit itemActive(this);
		newState |= TaskStateAnimation::Focus;
	}
	
	updateExpansion();
	
	m_stateAnimation.setState(newState, m_applet->fps(), m_applet->animationDuration());
}

void TaskItem::updateText() {
	qDebug("TaskItem::updateText");
	QSizeF preferredTextLayoutSize(::SmoothTasks::preferredTextLayoutSize(m_task->text(), KGlobalSettings::taskbarFont(), m_applet->maxButtonWidth()));
	if (preferredTextLayoutSize.width() < m_preferredTextLayoutSize.width() - 10.0  // add a little toleranze (e.g. for play time counters in titles)
	 || preferredTextLayoutSize.width() > m_preferredTextLayoutSize.width()) {
		m_preferredTextLayoutSize = preferredTextLayoutSize;
		qDebug() << "m_preferredTextLayoutSize changed to" << m_preferredTextLayoutSize;
		if (m_staticallyExpanded)
			emit preferredSizeStaticChange(this);
		else if (m_dynamicallyExpanded)
			emit preferredSizeDynamicChange(this);
	}
	if (isExpanded()) update();
}

void TaskItem::contextMenuEvent(QGraphicsSceneContextMenuEvent *event) {
	switch (m_task->type()) {
		case Task::TaskItem:
		case Task::GroupItem:
			m_applet->toolTip()->hide();
			m_applet->popup(this);
			event->accept();
			break;
		case Task::LauncherItem:
			m_applet->toolTip()->hide();
			m_applet->popup(this);
			event->accept();
			break;
		default:
			event->ignore();
	}
}

void TaskItem::mousePressEvent(QGraphicsSceneMouseEvent *event) {
	update();
	event->accept();
//	event->ignore();
}

void TaskItem::mouseMoveEvent(QGraphicsSceneMouseEvent *event) {
	if (
			QPoint(event->screenPos() - event->buttonDownScreenPos(Qt::LeftButton))
			.manhattanLength() < QApplication::startDragDistance()) {
		return;
	}
	
	m_applet->dragItem(this, event);
}

void TaskItem::mouseReleaseEvent(QGraphicsSceneMouseEvent *event) {
	TaskManager::Task* task;
	
	switch (event->button()) {
	case Qt::LeftButton:
		m_applet->toolTip()->hide();
		
		publishIconGeometry();
		
		switch (m_task->type()) {
		case Task::TaskItem:
			task = m_task->task();
			
			if (task) {
				if (event->modifiers() == Qt::ControlModifier) {
					KUrl url = launcherUrl(m_task->abstractItem());
					if (m_applet->groupManager()->launcherExists(url)) {
						new KRun(url, 0);
					}
				} else {
					task->activateRaiseOrIconify();
				}
			}
			break;
		case Task::GroupItem:
			if (event->modifiers() == Qt::ControlModifier) {
				KUrl url = launcherUrl(m_task->abstractItem());
				if (m_applet->groupManager()->launcherExists(url)) {
					new KRun(url, 0);
				}
			} else {
				activateOrIconifyGroup();
			}
//		{
//			TaskManager::GroupPopupMenu *groupMenu = new TaskManager::GroupPopupMenu(
//				NULL,
//				m_task->group(),
//				m_applet->groupManager());
//			groupMenu->exec(QCursor::pos());
//
//			delete groupMenu;
//		}
			break;
		case Task::LauncherItem:
			m_task->launcherItem()->launch();
			break;
 		default:
			break;
		}
		break;
	case Qt::MidButton:
		m_applet->middleClickTask(m_task->abstractItem());
	default:
		break;
	}
}

KUrl TaskItem::launcherUrl(AbstractGroupableItem* item)
{
	if (!item) return KUrl();
	if (item->itemType() == TaskManager::TaskItemType) {
		TaskManager::TaskItem * t = qobject_cast<TaskManager::TaskItem*>(item);
		// Search for applications which are executable and case-insensitively match the windowclass of the task and
		// See http://techbase.kde.org/Development/Tutorials/Services/Traders#The_KTrader_Query_Language
		QString query = QString("exist Exec and ('%1' =~ Name)").arg(t->task()->classClass());
		KService::List services = KServiceTypeTrader::self()->query("Application", query);
		if (!services.empty()) {
			return KUrl::fromPath((services[0]->entryPath()));
		} else {
			// No desktop-file was found, so try to find at least the executable
			// usually it's the lower cased window class class, but if that fails let's trust it
			QString path = KStandardDirs::findExe(t->task()->classClass().toLower());
			if (path.isEmpty()) {
				path = KStandardDirs::findExe(t->task()->classClass());
			}

			if (!path.isEmpty()) {
				return KUrl::fromPath(path);
			}
		}
	} else if (item->itemType() == TaskManager::GroupItemType) {
		TaskManager::TaskGroup * t = qobject_cast<TaskManager::TaskGroup*>(item);
		// Strategy: try to return the first non-group item's launcherUrl,
		// failing that, try to return the  launcherUrl of the first group
		// if any
		foreach (TaskManager::AbstractGroupableItem *i, t->members()) {
			if (i->itemType() != TaskManager::GroupItemType) {
				return launcherUrl(i);
			}
		}

		if (t->members().isEmpty()) {
			return KUrl();
		}

		return launcherUrl(t->members().first());
	} else {
		return KUrl();
	}
	return KUrl();
}

void TaskItem::activateOrIconifyGroup() {
	TaskManager::GroupPtr group = m_task->group();
	
	if (group == NULL) {
		return;
	}
	
	bool includesActive = false;
	TaskManager::ItemList items(group->members());
	int iconified = 0;
	foreach (TaskManager::AbstractGroupableItem *item, items) {
		TaskManager::TaskItem *task = qobject_cast<TaskManager::TaskItem*>(item);
		if (task) {
			if (task->task()->isIconified()) {
				++ iconified;
			}
			
			if (task->task()->isActive()) {
				includesActive = true;
			}
		}
	}
	
	if (includesActive && items.size() - iconified > iconified) {
		// iconify
		foreach (TaskManager::AbstractGroupableItem *item, items) {
			TaskManager::TaskItem *task = qobject_cast<TaskManager::TaskItem*>(item);
			if (task) {
				task->task()->setIconified(true);
			}
		}
	}
	else {
		// activate
		QList<WId> winOrder(KWindowSystem::stackingOrder());
		const int winCount = winOrder.size();
		TaskManager::TaskItem* sortedItems[winCount];
		
		std::memset(sortedItems, 0, sizeof(TaskManager::TaskItem*) * winCount);
		
		foreach (TaskManager::AbstractGroupableItem *item, items) {
			TaskManager::TaskItem *task = qobject_cast<TaskManager::TaskItem*>(item);
			if (task) {
				int index = winOrder.indexOf(task->task()->window());
				if (index != -1) {
					sortedItems[index] = task;
				}
			}
		}
		
		for (int index = 0; index < winCount; ++ index) {
			TaskManager::TaskItem* task = sortedItems[index];
			if (task) {
				task->task()->activate();
			}
		}
	}
}

void TaskItem::activate() {
	TaskManager::Task* task;
	
	switch (m_task->type()) {
	case Task::TaskItem:
		task = m_task->task();
		
		if (task) {
			task->activate();
		}
		break;
	case Task::GroupItem:
		m_applet->toolTip()->quickShow(this);
	default:
		break;
	}
}

void TaskItem::dragEnterEvent(QGraphicsSceneDragDropEvent *event) {
	qDebug("TaskItem::dragEnterEvent");
	if (event->mimeData()->hasFormat(TASK_ITEM)) {
		//event->ignore(); //ignore it so the taskbar gets the event
		event->acceptProposedAction();
		return;
	}
	event->accept();
	if (m_task->type() == Task::GroupItem) {
		m_stateAnimation.setState(
			m_stateAnimation.toState() | TaskStateAnimation::Hover,
			m_applet->fps(), m_applet->animationDuration());
	
		/*if (m_applet->expandTasks()) {
			expandTask();
		}*/ // TODO: check necessarity
		m_applet->toolTip()->quickShow(this);
	}
	else {
		if (!m_activateTimer) {
			m_activateTimer = new QTimer(this);
			m_activateTimer->setSingleShot(true);
			m_activateTimer->setInterval(DRAG_HOVER_DELAY);
			connect(m_activateTimer, SIGNAL(timeout()), this, SLOT(activate()));
		}
		m_activateTimer->start();
		hoverEnterEvent();
	}
}

void TaskItem::dragMoveEvent(QGraphicsSceneDragDropEvent *event) {
	Q_UNUSED(event);

	if (m_activateTimer) {
		m_activateTimer->start();
	}
	update();

	m_applet->dragMoveEvent(pos() + event->pos());
}

void TaskItem::dragLeaveEvent(QGraphicsSceneDragDropEvent *event) {
	Q_UNUSED(event);
	if (m_activateTimer) {
		delete m_activateTimer;
		m_activateTimer = NULL;
	}
	hoverLeaveEvent();
}

void TaskItem::dropEvent(QGraphicsSceneDragDropEvent *event) {
	if (m_activateTimer) {
		delete m_activateTimer;
		m_activateTimer = NULL;
	}
	m_applet->dropEvent(event);
}

void TaskItem::hoverMoveEvent(QGraphicsSceneHoverEvent* e) {
	update();
	QGraphicsWidget::hoverMoveEvent(e);
}

void TaskItem::hoverEnterEvent(QGraphicsSceneHoverEvent *event) {
	Q_UNUSED(event);
	hoverEnterEvent();
}

void TaskItem::hoverEnterEvent() {
	m_mouseIn = true;
	m_stateAnimation.setState(
		m_stateAnimation.toState() | TaskStateAnimation::Hover,
		m_applet->fps(), m_applet->animationDuration());
}

void TaskItem::confirmEnter() {
	m_delayedMouseIn = true;
	updateExpansion();
}

void TaskItem::hoverLeaveEvent(QGraphicsSceneHoverEvent *event) {
	Q_UNUSED(event);
	hoverLeaveEvent();
}

void TaskItem::hoverLeaveEvent() {
	m_mouseIn = false;
	m_stateAnimation.setState(
		m_stateAnimation.toState() & ~TaskStateAnimation::Hover,
		m_applet->fps(), m_applet->animationDuration());

	if (m_applet->toolTip()->hoverItem() != this) {
		updateExpansion();
	}
}

void TaskItem::confirmLeave() {
	m_delayedMouseIn = false;
	updateExpansion();
}

QPointF TaskItem::mapFromGlobal(const QPoint& point, bool *contained) const {
	QGraphicsScene *scene = this->scene();
	
	if (scene == NULL) {
		if (contained) {
			*contained = false;
		}
		return QPointF(-1, -1);
	}
	
	foreach (QGraphicsView *view, scene->views()) {
		QPointF mapped(mapFromScene(view->mapToScene(view->mapFromGlobal(point))));
		
		if (contains(mapped)) {
			if (contained) {
				*contained = true;
			}
			return mapped;
		}
	}
	
	if (contained) {
		*contained = false;
	}
	return QPointF(-1, -1);
}

void TaskItem::paint(QPainter *p, const QStyleOptionGraphicsItem *option, QWidget *widget) {
	Q_UNUSED(option);
	Q_UNUSED(widget);

	const QRectF bounds = boundingRect();
	
	if (!bounds.isValid())
		return;
	
	const bool isVertical = m_orientation == Qt::Vertical;
	const bool showFrame = m_task->type() != Task::LauncherItem;
	QRectF contentsRdRect;
	QRectF lightsRdRect;
	QRectF boundsRdRect;
	p->setRenderHint(QPainter::Antialiasing);

	// draw frame
	// TODO: cache frame in TaskItem?
	TaskbuttonFrameSvg *frame = m_applet->frame();
	
	if (isVertical) {
		boundsRdRect.setRect(bounds.y(), bounds.x(), bounds.height(), bounds.width());
		if (m_applet->dontRotateFrame()) {
			frame->resizeFrame(bounds.size());
			if(showFrame) {
				drawFrame(p, frame);
			}
			
			contentsRdRect = frame->minimalContentsRect();
			contentsRdRect.setRect(contentsRdRect.y(), bounds.right() - contentsRdRect.right(), contentsRdRect.height(), contentsRdRect.width());
			lightsRdRect = frame->minimalContentsRectCustom();
			lightsRdRect.setRect(lightsRdRect.y(), bounds.right() - lightsRdRect.right(), lightsRdRect.height(), lightsRdRect.width());
			
			p->save();
			p->rotate(-90);
			p->translate(-bounds.height(), 0);
		}
		else {
			p->save();
			p->rotate(-90);
			p->translate(-bounds.height(), 0);
		
			frame->resizeFrame(QSizeF(bounds.height(), bounds.width()));
			if(showFrame) {
				drawFrame(p, frame);
			}
			contentsRdRect = frame->minimalContentsRect();  // frame->contentsRect();
			lightsRdRect = frame->minimalContentsRectCustom();
		}
	}
	else {
		boundsRdRect = bounds;
		frame->resizeFrame(bounds.size());
		if(showFrame) {
			drawFrame(p, frame);
		}
		contentsRdRect = frame->minimalContentsRect();  // frame->contentsRect();
		lightsRdRect = frame->minimalContentsRectCustom();
	}
	
	if (!contentsRdRect.isValid())
		return;
	
	// TODO: It would be nice if I could use p->setClipPath(), but it doesn't
	//       seem to be possible to get a bounding path of a FrameSvg.
	// p->setClipRegion(frame->mask());
	// draw light
	if (m_applet->lights() && lightsRdRect.isValid() && m_task->type() != Task::LauncherItem) {
		bool mouseIn = false;
		QPointF pos(mapFromGlobal(QCursor::pos(), &mouseIn));
		
		m_light->paint(p, lightsRdRect, pos, mouseIn, isVertical);
	}

	QRectF expRect = expanderRect(contentsRdRect);

	// hack to leave space vor expander
	// does not work in vertical mode
	const bool rtl = QApplication::layoutDirection() == Qt::RightToLeft;
	m_icon->setRects
		( contentsRdRect.adjusted(rtl ? expRect.width() : 0, 0, rtl ? 0 : -expRect.width(), 0)
		,   boundsRdRect.adjusted(rtl ? expRect.width() : 0, 0, rtl ? 0 : -expRect.width(), 0)
		);

	// draw text
	if (m_applet->expandTasks()) {
		const qreal textSep = 5;
		QRectF textRect(contentsRdRect);
		if (QApplication::isRightToLeft()) {
			textRect.setRight(contentsRdRect.right() - m_icon->rdSize().width() - textSep);
			textRect.setLeft(expRect.right());
		} else {
			textRect.setLeft(contentsRdRect.left() + m_icon->rdSize().width() + textSep);
			textRect.setRight(expRect.left());
		}
		drawText(p, textRect);
	}
	drawExpander(p, expRect.intersect(contentsRdRect));
	
	if (isVertical) {
		p->restore();
	}
	
	// draw icon
	m_icon->paint(p, m_stateAnimation.hover(), m_task->type() == Task::GroupItem);
}

void TaskItem::drawFrame(QPainter *p, Plasma::FrameSvg *frame) {
	// draw "layers":
	//    hover
	//    attention
	//    focus
	//    minimized
	//    normal
	int animatedState  = m_stateAnimation.animatedState();
	int reachedUpState = m_stateAnimation.reachedUpState();

	if (animatedState) {
		QPixmap pixmap;
		bool didPaint   = false;
		int  shownState = m_stateAnimation.shownState();
		
		if (!reachedUpState) {
			frame->setElementPrefix(NORMAL);
			pixmap   = frame->framePixmap();
			didPaint = true;
		}

		if (shownState & TaskStateAnimation::Minimized &&
				!(reachedUpState & (
				TaskStateAnimation::Hover |
				TaskStateAnimation::Attention |
				TaskStateAnimation::Focus))) {
			frame->setElementPrefix(MINIMIZED);

			if (didPaint) {
				pixmap = Plasma::PaintUtils::transition(
					pixmap, frame->framePixmap(),
					m_stateAnimation.minimized());
			}
			else {
				pixmap   = frame->framePixmap();
				didPaint = true;
			}
		}

		if (shownState & TaskStateAnimation::Focus &&
				!(reachedUpState & (
				TaskStateAnimation::Hover |
				TaskStateAnimation::Attention))) {
			frame->setElementPrefix(FOCUS);

			if (didPaint) {
				pixmap = Plasma::PaintUtils::transition(
					pixmap, frame->framePixmap(),
					m_stateAnimation.focus());
			}
			else {
				pixmap   = frame->framePixmap();
				didPaint = true;
			}
		}

		if (shownState & TaskStateAnimation::Attention &&
				!(reachedUpState & TaskStateAnimation::Hover)) {
			frame->setElementPrefix(ATTENTION);

			if (didPaint) {
				pixmap = Plasma::PaintUtils::transition(
					pixmap, frame->framePixmap(),
					m_stateAnimation.attention());
			}
			else {
				pixmap   = frame->framePixmap();
				didPaint = true;
			}
		}

		if (shownState & TaskStateAnimation::Hover && !(m_applet->lights() && m_applet->onlyLights())) {
			frame->setElementPrefix(HOVER);

			if (didPaint) {
				pixmap = Plasma::PaintUtils::transition(
					pixmap, frame->framePixmap(),
					m_stateAnimation.hover());
			}
			else {
				pixmap   = frame->framePixmap();
				didPaint = true;
			}
		}

		p->drawPixmap(QPoint(0, Plasma::TopMargin), pixmap);
	}
	else {
		if (reachedUpState & TaskStateAnimation::Hover && !(m_applet->lights() && m_applet->onlyLights())) {
			frame->setElementPrefix(HOVER);
		}
		else if (reachedUpState & TaskStateAnimation::Attention) {
			frame->setElementPrefix(ATTENTION);
		}
		else if (reachedUpState & TaskStateAnimation::Focus) {
			frame->setElementPrefix(FOCUS);
		}
		else if (reachedUpState & TaskStateAnimation::Minimized) {
			frame->setElementPrefix(MINIMIZED);
		}
// 		else if (reachedUpState & TaskStateAnimation::Launcher) {
// 			frame->setElementPrefix(MINIMIZED);
// 		}
		else {
			frame->setElementPrefix(NORMAL);
		}

		frame->paintFrame(p);
	}
}

QColor TaskItem::textColor() const {
	QColor color;
	
	Plasma::Theme *theme = Plasma::Theme::defaultTheme();
	int animatedState  = m_stateAnimation.animatedState();
	int reachedUpState = m_stateAnimation.reachedUpState();

	if (animatedState) {
		// TODO: think about whether this makes sense
		//       I think this can be simplified at least
		bool didPaint   = false;
		int  shownState = m_stateAnimation.shownState();
		
		if (!reachedUpState) {
			color = theme->color(Plasma::Theme::TextColor);
			didPaint = true;
		}

		if (shownState & TaskStateAnimation::Minimized &&
				!(reachedUpState & (
				TaskStateAnimation::Hover |
				TaskStateAnimation::Attention |
				TaskStateAnimation::Focus))) {
			if (didPaint) {
				color.setAlphaF(1.0 -
					(1.0 - MINIMIZED_TEXT_ALPHA) *
					m_stateAnimation.minimized());
			}
			else {
				color = theme->color(Plasma::Theme::TextColor);
				color.setAlphaF(MINIMIZED_TEXT_ALPHA);
				didPaint = true;
			}
		}

		if (shownState & TaskStateAnimation::Focus &&
				!(reachedUpState & (
				TaskStateAnimation::Hover |
				TaskStateAnimation::Attention))) {
			if (didPaint) {
				color = KColorUtils::mix(
					color,
					theme->color(Plasma::Theme::TextColor),
					m_stateAnimation.focus());
			}
			else {
				color    = theme->color(Plasma::Theme::TextColor);
				didPaint = true;
			}
		}

		if (shownState & TaskStateAnimation::Attention &&
				!(reachedUpState & TaskStateAnimation::Hover)) {
			if (didPaint) {
				color = KColorUtils::mix(
					color,
					theme->color(Plasma::Theme::ButtonTextColor),
					m_stateAnimation.focus());
			}
			else {
				color    = theme->color(Plasma::Theme::ButtonTextColor);
				didPaint = true;
			}
		}

		if (shownState & TaskStateAnimation::Hover) {
			if (didPaint) {
				color = KColorUtils::mix(
					color,
					theme->color(Plasma::Theme::TextColor),
					m_stateAnimation.focus());
			}
			else {
				color    = theme->color(Plasma::Theme::TextColor);
				didPaint = true;
			}
		}
	}
	else if (reachedUpState & TaskStateAnimation::Hover) {
		color = theme->color(Plasma::Theme::TextColor);
	}
	else if (reachedUpState & TaskStateAnimation::Attention) {
		color = theme->color(Plasma::Theme::ButtonTextColor);
	}
	else if (reachedUpState & TaskStateAnimation::Focus) {
		color = theme->color(Plasma::Theme::TextColor);
	}
	else if (reachedUpState & TaskStateAnimation::Minimized) {
		color = theme->color(Plasma::Theme::TextColor);
		color.setAlphaF(MINIMIZED_TEXT_ALPHA);
	}
	else {
		color = theme->color(Plasma::Theme::TextColor);
	}
	
	return color;
}

void TaskItem::drawText(QPainter *p, const QRectF& contentsRect) {
	QColor color(textColor());
	p->setPen(QPen(color, 1.0));

	//const bool rtl = QApplication::isRightToLeft();
	QTextLayout layout(m_task->text(), KGlobalSettings::taskbarFont());

	QTextOption textOption(layout.textOption());
	textOption.setTextDirection(QApplication::layoutDirection());
	layout.setTextOption(textOption);

	QRectF textRect(contentsRect);
	QSizeF textSize(::SmoothTasks::layoutText(layout, contentsRect.size()));

	// Hack for plamsa layouts that define a to large margin:
	// This causes the text to be drawn over the painted button border
	// in case the text does not fit the available space.
	// However, the layouting function still uses the space where the
	// margins are subtracted so this should only make a difference
	// when the button is really thin and not even one line fits the
	// available space.
	if (contentsRect.height() <= textSize.height()) {
		qreal hackTextSize = textSize.height() + 1.0;
		textRect.setY(contentsRect.y() - hackTextSize * 0.5 + contentsRect.height() * 0.5);
		textRect.setHeight(hackTextSize);
	}

	drawTextLayout(p, layout, textRect, textSize, color);
}

void TaskItem::drawTextLayout(QPainter *painter,
		const QTextLayout& layout,
		const QRectF&      rect,
		const QSizeF&      textSize,
		const QColor&      color) const {

	if (!rect.isValid())
		return;

	QPixmap pixmap(std::ceil(rect.width()), std::ceil(rect.height()));
	pixmap.fill(Qt::transparent);

	QPainter p(&pixmap);
	p.setPen(painter->pen());

	QFontMetrics fm(layout.font());
	QPointF position(0, (rect.height() - textSize.height()) * 0.5 +
		(fm.tightBoundingRect(M).height() - fm.xHeight()) * 0.5);

	QLinearGradient alphaGradient(0, 0, 1, 0);
	alphaGradient.setCoordinateMode(QGradient::ObjectBoundingMode);
	const bool rtl = layout.textOption().textDirection() == Qt::RightToLeft;
	if (rtl) {
		alphaGradient.setColorAt(0, QColor(0, 0, 0, 0));
		alphaGradient.setColorAt(1, QColor(0, 0, 0, 255));
	}
	else {
		alphaGradient.setColorAt(0, QColor(0, 0, 0, 255));
		alphaGradient.setColorAt(1, QColor(0, 0, 0, 0));
	}

	QList<QRectF> fadeRects;
	const qreal fadeWidth = 30;
	// Draw each line in the layout
	for (int i = 0; i < layout.lineCount(); ++ i) {
		QTextLine line(layout.lineAt(i));
		QPointF linePos(position);
		qreal textWidth = line.naturalTextWidth();

		if (rtl)
			linePos.setX(rect.width() - textWidth);

		line.draw(&p, linePos);

		// Add a fade out rect to the list if the line is too long
		if (textWidth > rect.width())
			fadeRects.append
				( QRectF
					( rtl ? 0 : rect.width() - fadeWidth
					, line.position().y() + linePos.y()
					, fadeWidth
					, line.height()
					)
				);
	}
	// Reduce the alpha in each fade out rect using the alpha gradient
	if (!fadeRects.isEmpty()) {
		p.setCompositionMode(QPainter::CompositionMode_DestinationIn);
		foreach (const QRectF &r, fadeRects)
			p.fillRect(r, alphaGradient);
		p.setCompositionMode(QPainter::CompositionMode_SourceOver);
	}
	p.end();
	
	if (m_applet->textShadow()) {
		QImage shadow(pixmap.toImage());
		Plasma::PaintUtils::shadowBlur(shadow, 2, (color.value() < 128 ? Qt::white : Qt::black));
		painter->drawImage(rect.topLeft() + QPointF(1, 2), shadow);
	}
	painter->drawPixmap(rect.topLeft(), pixmap);
}

/** initially copied from KDEs Task plasmoid but substantially altered */
void TaskItem::drawExpander(QPainter *painter, const QRectF& expRect) const {
	if (m_task->type() == Task::GroupItem && expRect.isValid()) {
		QFont font(KGlobalSettings::smallestReadableFont());
		QFontMetrics fm(font);
		
		TaskbuttonFrameSvg *itemBackground = m_applet->frame();
		const QString& expElem(expanderElement());
		QRectF textRect(expRect.left(), expRect.top(), expRect.width(), fm.height());
		
		if (itemBackground->hasElement(expElem)) {
			QSizeF arrowSize(itemBackground->elementSize(expElem));
			QRectF arrowRect(
				expRect.left() + expRect.width() * 0.5 - arrowSize.width() * 0.5,
				expRect.top(),
				arrowSize.width(), arrowSize.height());
			
			switch (m_applet->location()) {
				case Plasma::TopEdge:
				case Plasma::LeftEdge:
					arrowRect.moveTop(arrowRect.top() + fm.height() + fm.leading());
					break;
				case Plasma::BottomEdge:
				case Plasma::RightEdge:
					textRect.moveTop(textRect.top() + arrowSize.height());
					break;
				default:
					if (m_orientation == Qt::Vertical) {
						arrowRect.moveTop(arrowRect.top() + fm.height() + fm.leading());
					}
					else {
						textRect.moveTop(textRect.top() + arrowSize.height());
					}
			}
			itemBackground->paint(painter, arrowRect, expElem);
		}
		else {
			textRect.moveTop(expRect.top() + expRect.height() * 0.5 - textRect.height() * 0.5);
		}
		
		painter->setFont(font);
		painter->drawText(textRect, Qt::AlignHCenter | Qt::AlignTop, QString::number(m_task->taskCount()));
	}
}

/** initially copied from KDEs Task plasmoid but substantially altered */
QRectF TaskItem::expanderRect(const TaskManager::AbstractGroupableItem& abstractGroupableItem, const Applet& applet, const Qt::Orientation& orientation, const QRectF& bounds) {
	if (!abstractGroupableItem.itemType() == TaskManager::GroupItemType) {
		if (QApplication::isLeftToRight()) {
			return QRectF(bounds.right(), bounds.top(), 0, 0);
		}
		else {
			return QRectF(bounds.left(), bounds.top(), 0, 0);
		}
	}
	
	QFontMetrics fm(KGlobalSettings::smallestReadableFont());
	const TaskbuttonFrameSvg *itemBackground = applet.frame();
	
	QSize arrowSize(itemBackground->elementSize(expanderElement(applet.location(), orientation)));
	QString text(QString::number(static_cast<const TaskManager::TaskGroup&>(abstractGroupableItem).members().size()));
	
	int   width  = qMax(fm.width(text), arrowSize.width());
	int   height = fm.height() + fm.leading() + arrowSize.height();
	qreal top    = bounds.top() + bounds.height() * 0.5 - height * 0.5;
	
	if (QApplication::isRightToLeft()) {
		return QRectF(bounds.left(), top, width, height);
	}
	else {
		return QRectF(bounds.right() - width, top, width, height);
	}
}

QRectF TaskItem::expanderRect(const QRectF &bounds) const {
	return expanderRect(*m_abstractItem, *m_applet, m_orientation, bounds);
}

const QString& TaskItem::expanderElement(const Plasma::Location& location, const Qt::Orientation& orientation) {
	// because of the rotation we never need _LEFT or _RIGHT
	switch (location) {
	case Plasma::TopEdge:
	case Plasma::LeftEdge:
		return GROUP_EXPANDER_TOP;
	case Plasma::BottomEdge:
	case Plasma::RightEdge:
		return GROUP_EXPANDER_BOTTOM;
	default:
		if (orientation) {
			return GROUP_EXPANDER_TOP;
		}
		else {
			return GROUP_EXPANDER_BOTTOM;
		}
	}
}

const QString& TaskItem::expanderElement() const {
	return expanderElement(m_applet->location(), m_orientation);
}

} // namespace SmoothTasks
#include "TaskItem.moc"
